/*
* This file is part of OpenModelica.
*
* Copyright (c) 1998-2021, Open Source Modelica Consortium (OSMC),
* c/o Linköpings universitet, Department of Computer and Information Science,
* SE-58183 Linköping, Sweden.
*
* All rights reserved.
*
* THIS PROGRAM IS PROVIDED UNDER THE TERMS OF GPL VERSION 3 LICENSE OR
* THIS OSMC PUBLIC LICENSE (OSMC-PL) VERSION 1.2.
* ANY USE, REPRODUCTION OR DISTRIBUTION OF THIS PROGRAM CONSTITUTES
* RECIPIENT'S ACCEPTANCE OF THE OSMC PUBLIC LICENSE OR THE GPL VERSION 3,
* ACCORDING TO RECIPIENTS CHOICE.
*
* The OpenModelica software and the Open Source Modelica
* Consortium (OSMC) Public License (OSMC-PL) are obtained
* from OSMC, either from the above address,
* from the URLs: http://www.ida.liu.se/projects/OpenModelica or
* http://www.openmodelica.org, and in the OpenModelica distribution.
* GNU version 3 is obtained from: http://www.gnu.org/copyleft/gpl.html.
*
* This program is distributed WITHOUT ANY WARRANTY; without
* even the implied warranty of  MERCHANTABILITY or FITNESS
* FOR A PARTICULAR PURPOSE, EXCEPT AS EXPRESSLY SET FORTH
* IN THE BY RECIPIENT SELECTED SUBSIDIARY LICENSE CONDITIONS OF OSMC-PL.
*
* See the full OSMC Public License conditions for more details.
*
*/
encapsulated uniontype NBStrongComponent
"file:        NBStrongComponent.mo
 package:     NBStrongComponent
 description: This file contains the data-types used save the strong Component
              data after causalization.
"
public
  import NBResizable.EvalOrder;

protected
  // selfimport
  import StrongComponent = NBStrongComponent;

  // NF imports
  import ComponentRef = NFComponentRef;
  import Dimension = NFDimension;
  import Expression = NFExpression;
  import Subscript = NFSubscript;
  import Type = NFType;
  import Variable = NFVariable;

  // Backend imports
  import Adjacency = NBAdjacency;
  import NBAdjacency.Mapping;
  import BackendDAE = NBackendDAE;
  import Causalize = NBCausalize;
  import BVariable = NBVariable;
  import NBEquation.{Equation, EquationPointer, EquationPointers, EquationAttributes, Iterator};
  import Initialization = NBInitialization;
  import Inline = NBInline;
  import NBJacobian.JacobianType;
  import Matching = NBMatching;
  import Resizable = NBResizable;
  import Solve = NBSolve;
  import Sorting = NBSorting;
  import NBSorting.SuperNode;
  import BPartition = NBPartition;
  import NBPartition.{Partition};
  import Tearing = NBTearing;
  import NBVariable.{VariablePointer, VariablePointers};

  // Util imports
  import Pointer;
  import Slice = NBSlice;
  import StringUtil;
  import UnorderedMap;
  import UnorderedSet;

public
  uniontype AliasInfo
    record ALIAS_INFO
      BPartition.Kind kind      "The partition kind";
      Integer partitionIndex    "the partition index";
      Integer componentIndex    "The index in that strong component array";
    end ALIAS_INFO;

    function toString
      input AliasInfo info;
      output String str = Partition.kindToString(info.kind) + "[" + intString(info.partitionIndex) + " | " + intString(info.componentIndex) + "]";
    end toString;

    function hash
      input AliasInfo info;
      output Integer i = stringHashDjb2(toString(info));
    end hash;

    function isEqual
      input AliasInfo info1;
      input AliasInfo info2;
      output Boolean b = (info1.componentIndex == info2.componentIndex) and (info1.partitionIndex == info2.partitionIndex) and (info1.kind == info2.kind);
    end isEqual;
  end AliasInfo;

  record SINGLE_COMPONENT
    "component for all equations that solve for a single (possibly multidimensional) variable
    SCALAR_EQUATION, ARRAY_EQUATION, RECORD_EQUATION."
    Pointer<Variable> var;
    Pointer<Equation> eqn;
    Solve.Status status;
  end SINGLE_COMPONENT;

  record MULTI_COMPONENT
    "component for all equations that can solve for more than one variable instance
    ALGORITHM, WHEN_EQUATION, IF_EQUATION"
    list<Slice<VariablePointer>> vars;
    Slice<EquationPointer> eqn;
    Solve.Status status;
  end MULTI_COMPONENT;

  record SLICED_COMPONENT
    "component for all equations AND/OR variables that need to be sliced (zero based indices)"
    ComponentRef var_cref       "cref to solve for";
    Slice<VariablePointer> var  "sliced variable";
    Slice<EquationPointer> eqn  "sliced equation";
    Solve.Status status;
  end SLICED_COMPONENT;

  record RESIZABLE_COMPONENT
    "component for for-equations with trivial evaluation order"
    ComponentRef var_cref                       "cref to solve for";
    Slice<VariablePointer> var                  "sliced variable";
    Slice<EquationPointer> eqn                  "sliced equation";
    UnorderedMap<ComponentRef, EvalOrder> order "independent, forward, backward";
    Solve.Status status;
  end RESIZABLE_COMPONENT;

  record GENERIC_COMPONENT
    "component for all equations that need to be sliced but where no for-loop could be recovered
    has no status since this is generated by the Solve module and is always status=EXPLICIT."
    ComponentRef var_cref       "cref to solve for";
    Slice<VariablePointer> var  "sliced variable";
    Slice<EquationPointer> eqn  "sliced equation";
  end GENERIC_COMPONENT;

  record ENTWINED_COMPONENT
    "component for entwined SLICED_COMPONENT or GENERIC_COMPONENT
    the body equations have to be called in a specific pattern but do not form an algebraic loop"
    list<StrongComponent> entwined_slices                     "has to be SLICED_COMPONENT()";
    list<tuple<Pointer<Equation>, Integer>> entwined_tpl_lst  "equation with scalar idx (0 based) - fallback scalarization";
  end ENTWINED_COMPONENT;

  record ALGEBRAIC_LOOP
    "component for equations that have to be solved as a system."
    Integer idx;
    Tearing strict;
    Option<Tearing> casual;
    Boolean linear              "true if the loop is linear";
    Boolean mixed               "true for systems that have discrete variables";
    Boolean homotopy            "true if contains homotopy()";
    Solve.Status status;
  end ALGEBRAIC_LOOP;

  record ALIAS
    "Component representing equal strong components in ODE<->INIT<->DAE
    has no status since this is generated by the Solve module and is always status=EXPLICIT."
    AliasInfo aliasInfo       "The strong component array and index it refers to";
    StrongComponent original  "The original strong component for analysis";
  end ALIAS;

  function toString
    input StrongComponent comp;
    input Integer index = -1    "negative indices will not be printed";
    output String str;
  protected
    Integer s = StrongComponent.size(comp, true);
    String indexStr = if index > 0 then " " + intString(index) else "";
  algorithm
    str := match comp

      case SINGLE_COMPONENT() algorithm
        str := StringUtil.headline_3("BLOCK" + indexStr + ": Single Strong Component (status = " + Solve.statusString(comp.status) + ", size = " + intString(s) + ")");
        str := str + "### Variable:\n" + Variable.toString(Pointer.access(comp.var), "\t") + "\n";
        str := str + "### Equation:\n" + Equation.toString(Pointer.access(comp.eqn), "\t") + "\n";
      then str;

      case MULTI_COMPONENT() algorithm
        str := StringUtil.headline_3("BLOCK" + indexStr + ": Multi Strong Component (status = " + Solve.statusString(comp.status) + ", size = " + intString(s) + ")");
        str := str + "### Variables:\n";
        str := str + List.toString(comp.vars, function Slice.toString(func = BVariable.pointerToString, maxLength = 10), "", "\t", "\n\t", "");
        str := str + "\n### Equation:\n" + Slice.toString(comp.eqn, function Equation.pointerToString(str = "\t")) + "\n";
      then str;

      case SLICED_COMPONENT() algorithm
        str := if index == -2 then "" else StringUtil.headline_3("BLOCK" + indexStr + ": Sliced Component (status = " + Solve.statusString(comp.status) + ", size = " + intString(s) + ")");
        str := str + "### Variable:\n\t" + ComponentRef.toString(comp.var_cref) + "\n";
        str := str + "### Equation:\n" + Slice.toString(comp.eqn, function Equation.pointerToString(str = "\t")) + "\n";
      then str;

      case RESIZABLE_COMPONENT() algorithm
        str := StringUtil.headline_3("BLOCK" + indexStr + ": Resizable Component (status = " + Solve.statusString(comp.status) + ", size = " + intString(s) + ")");
        str := str + "### Variable:\n\t" + ComponentRef.toString(comp.var_cref) + "\n";
        str := str + "### Equation:\n\t" + Equation.pointerToString(Slice.getT(comp.eqn)) + "\n";
      then str;

      case ENTWINED_COMPONENT() algorithm
        str := StringUtil.headline_3("BLOCK" + indexStr + ": Entwined Component (status = Solve.EXPLICIT, size = " + intString(s) + ")");
        str := str + "call order: " + List.toString(list(Equation.getEqnName(Util.tuple21(e)) for e in comp.entwined_tpl_lst), ComponentRef.toString, "", "{", ", ", "}", true, 10) + "\n";
        str := str + List.toString(comp.entwined_slices, function toString(index = -2), "", "", "", "");
      then str;

      case GENERIC_COMPONENT() algorithm
        str := StringUtil.headline_3("BLOCK" + indexStr + ": Generic Component (status = Solve.EXPLICIT, size = " + intString(s) + ")");
        str := str + "### Variable:\n\t" + ComponentRef.toString(comp.var_cref) + "\n";
        str := str + "### Equation:\n" + Slice.toString(comp.eqn, function Equation.pointerToString(str = "\t")) + "\n";
      then str;

      case ALGEBRAIC_LOOP() algorithm
        str := StringUtil.headline_3("BLOCK" + indexStr + ": Algebraic Loop (Linear = " + boolString(comp.linear) + ", Mixed = " + boolString(comp.mixed) + ", Homotopy = " + boolString(comp.homotopy) + ", size = " + intString(s) + ")");
        str := str + Tearing.toString(comp.strict, "Strict Tearing Set");
        if isSome(comp.casual) then
          str := str + Tearing.toString(Util.getOption(comp.casual), "Casual Tearing Set");
        end if;
      then str;

      case ALIAS() algorithm
        str := "--- Alias of " + AliasInfo.toString(comp.aliasInfo) + " ---\n" + toString(comp.original, index);
      then str;

      else algorithm
        Error.addMessage(Error.INTERNAL_ERROR, {getInstanceName() + " failed."});
      then fail();
    end match;
  end toString;

  uniontype CountCollector
    record COUNT_COLLECTOR
      Integer single_scalar;
      Integer single_array;
      Integer single_record;
      Integer multi_algorithm;
      Integer multi_when;
      Integer multi_if;
      Integer multi_tpl;
      Integer resizable_for;
      Integer generic_for;
      Integer entwined_for;
      Integer loop_lin;
      Integer loop_nlin;
    end COUNT_COLLECTOR;
  end CountCollector;

  function strongComponentInfo
    input output StrongComponent comp;
    input Pointer<CountCollector> collector_ptr;
  protected
    CountCollector collector = Pointer.access(collector_ptr);
  algorithm
    _ := match comp
      case SINGLE_COMPONENT() algorithm
        _ := match Pointer.access(comp.eqn)
          case Equation.SCALAR_EQUATION() algorithm collector.single_scalar := collector.single_scalar + 1; Pointer.update(collector_ptr, collector); then ();
          case Equation.ARRAY_EQUATION()  algorithm collector.single_array := collector.single_array + 1; Pointer.update(collector_ptr, collector);   then ();
          case Equation.RECORD_EQUATION() algorithm collector.single_record := collector.single_record + 1; Pointer.update(collector_ptr, collector); then ();
          else                            algorithm Error.addCompilerWarning("Cannot classify strong component:\n" + toString(comp) + "\n");          then ();
        end match;
      then ();

      case MULTI_COMPONENT() algorithm
        _ := match Pointer.access(Slice.getT(comp.eqn))
          case Equation.ALGORITHM()       algorithm collector.multi_algorithm := collector.multi_algorithm + 1; Pointer.update(collector_ptr, collector); then ();
          case Equation.WHEN_EQUATION()   algorithm collector.multi_when := collector.multi_when + 1; Pointer.update(collector_ptr, collector);           then ();
          case Equation.IF_EQUATION()     algorithm collector.multi_if := collector.multi_if + 1; Pointer.update(collector_ptr, collector);               then ();
          case Equation.RECORD_EQUATION() algorithm collector.multi_tpl := collector.multi_tpl + 1; Pointer.update(collector_ptr, collector);             then ();
          else                            algorithm Error.addCompilerWarning("Cannot classify strong component:\n" + toString(comp) + "\n");              then ();
        end match;
      then ();

      case SLICED_COMPONENT() algorithm
        _ := match Pointer.access(Slice.getT(comp.eqn))
          case Equation.SCALAR_EQUATION() algorithm collector.single_scalar := collector.single_scalar + 1; Pointer.update(collector_ptr, collector); then ();
          case Equation.ARRAY_EQUATION()  algorithm collector.single_array := collector.single_array + 1; Pointer.update(collector_ptr, collector);   then ();
          case Equation.RECORD_EQUATION() algorithm collector.single_record := collector.single_record + 1; Pointer.update(collector_ptr, collector); then ();
          else                            algorithm Error.addCompilerWarning("Cannot classify strong component:\n" + toString(comp) + "\n");          then ();
        end match;
      then ();

      case RESIZABLE_COMPONENT()                algorithm collector.resizable_for := collector.resizable_for +1; Pointer.update(collector_ptr, collector);  then ();
      case GENERIC_COMPONENT()                  algorithm collector.generic_for := collector.generic_for + 1; Pointer.update(collector_ptr, collector);     then ();
      case ENTWINED_COMPONENT()                 algorithm collector.entwined_for := collector.entwined_for + 1; Pointer.update(collector_ptr, collector);   then ();
      case ALGEBRAIC_LOOP() guard(comp.linear)  algorithm collector.loop_lin := collector.loop_lin + 1; Pointer.update(collector_ptr, collector);           then ();
      case ALGEBRAIC_LOOP()                     algorithm collector.loop_nlin := collector.loop_nlin + 1; Pointer.update(collector_ptr, collector);         then ();
      case ALIAS()                              algorithm strongComponentInfo(comp.original, collector_ptr);                                                then ();
      else                                      algorithm Error.addCompilerWarning("Cannot classify strong component:\n" + toString(comp) + "\n");          then ();
    end match;
  end strongComponentInfo;

  function hash
    "only hashes basic types, isEqual is used to differ between sliced/entwined loops"
    input StrongComponent comp;
    output Integer i;
  algorithm
    i := match comp
      case SINGLE_COMPONENT()   then BVariable.hash(comp.var) + Equation.hash(comp.eqn);
      case MULTI_COMPONENT()    then Equation.hash(Slice.getT(comp.eqn));
      case SLICED_COMPONENT()   then ComponentRef.hash(comp.var_cref) + Equation.hash(Slice.getT(comp.eqn));
      case RESIZABLE_COMPONENT()then ComponentRef.hash(comp.var_cref) + Equation.hash(Slice.getT(comp.eqn));
      case GENERIC_COMPONENT()  then Equation.hash(Slice.getT(comp.eqn));
      case ENTWINED_COMPONENT() then sum(hash(sub_comp) for sub_comp in comp.entwined_slices);
      case ALGEBRAIC_LOOP()     then Tearing.hash(comp.strict);
      case ALIAS()              then AliasInfo.hash(comp.aliasInfo);
    end match;
  end hash;

  function isEqual
    input StrongComponent comp1;
    input StrongComponent comp2;
    output Boolean b;
  algorithm
    b := match(comp1, comp2)
      case (SINGLE_COMPONENT(), SINGLE_COMPONENT())       then BVariable.equalName(comp1.var, comp2.var) and Equation.isEqualPtr(comp1.eqn, comp2.eqn);
      case (MULTI_COMPONENT(), MULTI_COMPONENT())         then Equation.isEqualPtr(Slice.getT(comp1.eqn), Slice.getT(comp2.eqn));
      case (SLICED_COMPONENT(), SLICED_COMPONENT())       then ComponentRef.isEqual(comp1.var_cref, comp2.var_cref) and Slice.isEqual(comp1.eqn, comp2.eqn, Equation.isEqualPtr);
      case (RESIZABLE_COMPONENT(), RESIZABLE_COMPONENT()) then ComponentRef.isEqual(comp1.var_cref, comp2.var_cref) and Slice.isEqual(comp1.eqn, comp2.eqn, Equation.isEqualPtr);
      case (GENERIC_COMPONENT(), GENERIC_COMPONENT())     then Slice.isEqual(comp1.eqn, comp2.eqn, Equation.isEqualPtr);
      case (ENTWINED_COMPONENT(), ENTWINED_COMPONENT())   then List.isEqualOnTrue(comp1.entwined_slices, comp2.entwined_slices, isEqual);
      case (ALGEBRAIC_LOOP(), ALGEBRAIC_LOOP())           then Tearing.isEqual(comp1.strict, comp2.strict);
      case (ALIAS(), ALIAS())                             then AliasInfo.isEqual(comp1.aliasInfo, comp2.aliasInfo);
      else false;
    end match;
  end isEqual;

  function size
    input StrongComponent comp;
    input Boolean resize;
    output Integer s;
  algorithm
    s := match comp
      case SINGLE_COMPONENT()     then Equation.size(comp.eqn, resize);
      case MULTI_COMPONENT()      then Slice.size(comp.eqn, function Equation.size(resize = resize));
      case SLICED_COMPONENT()     then Slice.size(comp.eqn, function Equation.size(resize = resize));
      case RESIZABLE_COMPONENT()  then Slice.size(comp.eqn, function Equation.size(resize = resize));
      case GENERIC_COMPONENT()    then Slice.size(comp.eqn, function Equation.size(resize = resize));
      case ENTWINED_COMPONENT()   then sum(StrongComponent.size(c, resize) for c in comp.entwined_slices);
      case ALGEBRAIC_LOOP()       then Tearing.size(comp.strict, resize);
      case ALIAS()                then StrongComponent.size(comp.original, resize);
      else algorithm
         Error.addMessage(Error.INTERNAL_ERROR, {getInstanceName() + " failed. Cannot determine size of strong component:\n" + toString(comp) + "\n"});
      then fail();
    end match;
  end size;

  function removeAlias
    input output StrongComponent comp;
  algorithm
    comp := match comp
      case ALIAS() then comp.original;
      else comp;
    end match;
  end removeAlias;

  function createPseudoSlice
    input Integer var_arr_idx;
    input Integer eqn_arr_idx;
    input ComponentRef cref_to_solve;
    input list<Integer> eqn_scal_indices;
    input array<Integer> eqn_to_var;
    input EquationPointers eqns;
    input Adjacency.Mapping mapping;
    input Boolean independent = false   "true if scalar equations can be solved in any order";
    output StrongComponent comp;
  protected
    Pointer<Variable> var_ptr;
    Pointer<Equation> eqn_ptr;
    Integer first_var, var_size, first_eqn, eqn_size;
    Slice<VariablePointer>var_slice;
    Slice<EquationPointer>eqn_slice;
    list<Integer> var_scal_indices;
    UnorderedMap<ComponentRef, EvalOrder> order;
  algorithm
    // get and save sliced variable and equation
    var_ptr := BVariable.getVarPointer(cref_to_solve, sourceInfo());
    eqn_ptr := EquationPointers.getEqnAt(eqns, eqn_arr_idx);
    (first_var, var_size) := mapping.var_AtS[var_arr_idx];
    (first_eqn, eqn_size) := mapping.eqn_AtS[eqn_arr_idx];
    var_scal_indices := list(eqn_to_var[e] for e in eqn_scal_indices);

    // check if the full variable occurs and its independent
    if independent and Equation.isArrayEquation(eqn_ptr) and listLength(eqn_scal_indices) == eqn_size and listLength(var_scal_indices) == var_size then
      var_slice := Slice.SLICE(var_ptr, {});
      eqn_slice := Slice.SLICE(eqn_ptr, {});
    else
      var_slice := Slice.SLICE(var_ptr, list(idx - first_var for idx in var_scal_indices));
      eqn_slice := Slice.SLICE(eqn_ptr, list(idx - first_eqn for idx in eqn_scal_indices));
    end if;

    // check if it is a resizable component
    order := Resizable.detect(Pointer.access(eqn_ptr), cref_to_solve);
    if not List.any(UnorderedMap.valueList(order), Resizable.orderFailed) and listLength(eqn_scal_indices) == eqn_size then
      comp := RESIZABLE_COMPONENT(
        var_cref  = cref_to_solve,
        var       = var_slice,
        eqn       = eqn_slice,
        order     = order,
        status    = NBSolve.Status.UNPROCESSED);
    else
      comp := createSliceOrSingle(cref_to_solve, var_slice, eqn_slice);
    end if;
  end createPseudoSlice;

  function createPseudoEntwined
    input list<Integer> eqn_indices;
    input array<Integer> eqn_to_var;
    input Mapping mapping;
    input VariablePointers vars;
    input EquationPointers eqns;
    input list<SuperNode> nodes;
    output StrongComponent entwined;
  protected
    UnorderedMap<Integer, Slice.IntLst> elem_map = UnorderedMap.new<Slice.IntLst>(Util.id, intEq);
    UnorderedMap<Integer, ComponentRef> cref_map = UnorderedMap.new<ComponentRef>(Util.id, intEq);
    list<tuple<Integer, Slice.IntLst>> flat_map;
    Integer eqn_arr_idx, var_arr_idx;
    Slice.IntLst scal_indices;
    list<StrongComponent> entwined_slices = {};
    list<tuple<Pointer<Equation>, Integer>> entwined_tpl_lst;
  algorithm
    // collect individual buckets again
    for idx in eqn_indices loop
      UnorderedMap.add(mapping.eqn_StA[idx], idx :: UnorderedMap.getOrDefault(mapping.eqn_StA[idx], elem_map, {}), elem_map);
    end for;
    // collect crefs to solve for
    for node in nodes loop
      () := match node
        case SuperNode.ARRAY_BUCKET() algorithm
          UnorderedMap.add(node.arr_idx, node.cref_to_solve, cref_map);
        then ();
        else ();
      end match;
    end for;

    // create individual slices
    for tpl in UnorderedMap.toList(elem_map) loop
      (eqn_arr_idx, scal_indices) := tpl;
      var_arr_idx := mapping.var_StA[eqn_to_var[Util.tuple21(mapping.eqn_AtS[eqn_arr_idx])]];
      entwined_slices := createPseudoSlice(var_arr_idx, eqn_arr_idx, UnorderedMap.getSafe(eqn_arr_idx, cref_map, sourceInfo()), scal_indices, eqn_to_var, eqns, mapping) :: entwined_slices;
    end for;

    // create scalar list for fallback
    entwined_tpl_lst := list((EquationPointers.getEqnAt(eqns, mapping.eqn_StA[idx]), idx) for idx in eqn_indices);

    entwined := ENTWINED_COMPONENT(entwined_slices, entwined_tpl_lst);
  end createPseudoEntwined;

  function createAlias
    input BPartition.Kind kind;
    input Integer partitionIndex;
    input Pointer<Integer> index_ptr;
    input StrongComponent orig_comp;
    output StrongComponent alias_comp;
  algorithm
    alias_comp := ALIAS(ALIAS_INFO(kind, partitionIndex, Pointer.access(index_ptr)), orig_comp);
    Pointer.update(index_ptr, Pointer.access(index_ptr) + 1);
  end createAlias;

  function createPseudoEntwinedIndices
    input array<list<Integer>> entwined_indices;
    input EquationPointers eqns;
    input Adjacency.Mapping mapping;
    output list<tuple<Pointer<Equation>, Integer>> flat_tpl_indices = {};
  protected
    Integer arr_idx, first_idx;
    array<Integer> eqn_StA        "safe access with iterated integer (void pointer)";
  algorithm
    for tmp in entwined_indices loop
      for scal_idx in tmp loop
        eqn_StA := mapping.eqn_StA;
        arr_idx := eqn_StA[scal_idx];
        (first_idx, _) := mapping.eqn_AtS[arr_idx];
        flat_tpl_indices := (EquationPointers.getEqnAt(eqns, arr_idx), scal_idx-first_idx) :: flat_tpl_indices;
      end for;
    end for;
    flat_tpl_indices := listReverse(flat_tpl_indices);
  end createPseudoEntwinedIndices;

  type DAEType = enumeration(UNPROCESSED, REMOVED, INNER, RESIDUAL);

  function sortDAEModeComponents
    input output Option<array<StrongComponent>> comps;
    input VariablePointers variables;
    input Pointer<Integer> uniqueIndex;
  protected
    list<StrongComponent> residuals = {}, inners = {};
    // used to determine if a sliced equation will be handled as residual or as inner
    UnorderedSet<ComponentRef> slice_set = UnorderedSet.new(ComponentRef.hash, ComponentRef.isEqual);
  algorithm
    comps := match comps
        local
          array<StrongComponent> original;

      case SOME(original) algorithm
        for comp in original loop
          (residuals, inners) := sortDAEModeComponent(comp, residuals, inners, variables, uniqueIndex, slice_set);
        end for;

        /* order of inners matters */
        comps := SOME(listArray(listAppend(listReverse(inners), residuals)));
      then comps;

      else comps;
    end match;
  end sortDAEModeComponents;

  function sortDAEModeComponent
    input StrongComponent comp;
    input output list<StrongComponent> residuals;
    input output list<StrongComponent> inners;
    input VariablePointers variables;
    input Pointer<Integer> uniqueIndex;
    input UnorderedSet<ComponentRef> slice_set;
  protected
    list<StrongComponent> new_residuals;
    DAEType dae_type;
  algorithm
    (new_residuals, dae_type) := match comp
      // single equation fully solved for single variable (not neccessarily scalar)
      case SINGLE_COMPONENT() algorithm
        (new_residuals, dae_type) := singleDAEModeComponent(comp.eqn, variables, uniqueIndex);
      then (new_residuals, dae_type);

      case MULTI_COMPONENT() algorithm
        (new_residuals, dae_type) := slicedDAEModeComponent(comp.vars, {comp.eqn}, variables, uniqueIndex, slice_set);
      then (new_residuals, dae_type);

      case SLICED_COMPONENT() algorithm
        // this will always result in inner equation for now as either eqn or var are sliced
        // -> in the future improve this
        (new_residuals, dae_type) := slicedDAEModeComponent({comp.var}, {comp.eqn}, variables, uniqueIndex, slice_set);
      then (new_residuals, dae_type);

      case RESIZABLE_COMPONENT() algorithm
        (new_residuals, dae_type) := slicedDAEModeComponent({comp.var}, {comp.eqn}, variables, uniqueIndex, slice_set);
      then (new_residuals, dae_type);

      case GENERIC_COMPONENT() algorithm
        (new_residuals, dae_type) := slicedDAEModeComponent({comp.var}, {comp.eqn}, variables, uniqueIndex, slice_set);
      then (new_residuals, dae_type);

      case ALGEBRAIC_LOOP() algorithm
        (new_residuals, dae_type) := slicedDAEModeComponent(comp.strict.iteration_vars, comp.strict.residual_eqns, variables, uniqueIndex, slice_set);
      then (new_residuals, dae_type);

      else ({}, if StrongComponent.isDiscrete(comp) then DAEType.REMOVED else DAEType.INNER);
    end match;

    if dae_type == DAEType.RESIDUAL then
      // add residuals
      residuals := listAppend(new_residuals, residuals);
    elseif dae_type == DAEType.INNER then
      // add original to inners
      inners := comp :: inners;
    end if;
  end sortDAEModeComponent;

  function slicedDAEModeComponent
    input list<Slice<Pointer<Variable>>> var_slices;
    input list<Slice<Pointer<Equation>>> eqn_slices;
    input VariablePointers variables;
    input Pointer<Integer> uniqueIndex;
    input UnorderedSet<ComponentRef> slice_set;
    output list<StrongComponent> new_residuals;
    output DAEType dae_type;
  protected
    Pointer<Equation> eqn;
    ComponentRef eqn_name;
    list<list<StrongComponent>> acc_new_residuals = {};
  algorithm
    if List.all(list(v.indices for v in var_slices), listEmpty) then
      for eqn_slice in eqn_slices loop
        eqn       := Slice.getT(eqn_slice);
        eqn_name  := Equation.getEqnName(eqn);
        if listEmpty(eqn_slice.indices) and not UnorderedSet.contains(eqn_name, slice_set) then
          // unsliced equation in multi component / equation not found in map
          (new_residuals, dae_type) := singleDAEModeComponent(eqn, variables, uniqueIndex);
          if dae_type == DAEType.RESIDUAL then
            acc_new_residuals := new_residuals :: acc_new_residuals;
          elseif dae_type == DAEType.INNER then
            break;
          end if;
        else
          // this sliced equation cannot be made residual, add it to the map
          dae_type := DAEType.INNER;
          break;
        end if;
      end for;
    else
      dae_type := DAEType.INNER;
    end if;

    if dae_type == DAEType.INNER then
      for eqn_slice in eqn_slices loop
        eqn       := Slice.getT(eqn_slice);
        eqn_name  := Equation.getEqnName(eqn);
        UnorderedSet.add(eqn_name, slice_set);
      end for;
      new_residuals := {};
    else
      new_residuals := List.flatten(acc_new_residuals);
    end if;
  end slicedDAEModeComponent;

  function singleDAEModeComponent
    input Pointer<Equation> eqn_ptr;
    input VariablePointers variables;
    input Pointer<Integer> uniqueIndex;
    output list<StrongComponent> new_residuals;
    output DAEType dae_type = DAEType.RESIDUAL;
  protected
    Pointer<list<Pointer<Equation>>> new_eqns;
    UnorderedSet<VariablePointer> dummy_set;
    Equation eqn;
    list<Pointer<Equation>> eqns;
  algorithm
    new_eqns  := Pointer.create({});
    dummy_set := UnorderedSet.new(BVariable.hash, BVariable.equalName);
    eqn       := Inline.inlineRecordTupleArrayEquation(Pointer.access(eqn_ptr), Iterator.EMPTY(), variables, new_eqns, dummy_set, uniqueIndex, true);
    eqns      := Pointer.access(new_eqns);
    // create equation, deliberately use new pointer. allow creating residual to fail and add original strong component to inners
    eqns := if listEmpty(eqns) then {Pointer.create(eqn)} else eqns;
    (new_residuals, dae_type) := inlinedDAEModeComponent(eqns);
  end singleDAEModeComponent;

  function inlinedDAEModeComponent
    input list<Pointer<Equation>> eqns;
    output list<StrongComponent> comps = {};
    output DAEType dae_type = DAEType.UNPROCESSED;
  protected
    Pointer<Equation> new_eqn;
    StrongComponent new_comp;
  algorithm
    for eqn in eqns loop
      if Equation.isDiscrete(eqn) then
        // all sub equations are discrete, remove whole equation
        if dae_type < DAEType.INNER then
          dae_type := DAEType.REMOVED;
        end if;
      else
        new_eqn := Equation.createResidual(eqn, false, true);
        if Equation.isResidual(new_eqn) then
          // add to residuals
          new_comp := SINGLE_COMPONENT(Equation.getResidualVar(new_eqn), new_eqn, NBSolve.Status.UNPROCESSED);
          comps := new_comp :: comps;
          dae_type := DAEType.RESIDUAL;
        else
          // cannot make residuals for all, make whole equation inner for now
          // might not be neccessary --> needs further solving to get proper sub strong component
          dae_type := DAEType.INNER; break;
        end if;
      end if;
    end for;
  end inlinedDAEModeComponent;

  function fromSolvedEquationSlice
    "creates a strong component assuming the equation is already solved
    todo: if and when equations"
    input Slice<EquationPointer> eqn_slice;
    output StrongComponent comp;
  protected
    Pointer<Equation> eqn = Slice.getT(eqn_slice);
  algorithm
    comp := match Pointer.access(eqn)
      case Equation.SCALAR_EQUATION() then SINGLE_COMPONENT(BVariable.getVarPointer(Expression.toCref(Equation.getLHS(Pointer.access(eqn))), sourceInfo()), eqn, NBSolve.Status.EXPLICIT);
      case Equation.ARRAY_EQUATION()  then SINGLE_COMPONENT(BVariable.getVarPointer(Expression.toCref(Equation.getLHS(Pointer.access(eqn))), sourceInfo()), eqn, NBSolve.Status.EXPLICIT);
      case Equation.RECORD_EQUATION() then SINGLE_COMPONENT(BVariable.getVarPointer(Expression.toCref(Equation.getLHS(Pointer.access(eqn))), sourceInfo()), eqn, NBSolve.Status.EXPLICIT);
      case Equation.IF_EQUATION()     then SINGLE_COMPONENT(BVariable.getVarPointer(Expression.toCref(Equation.getLHS(Pointer.access(eqn))), sourceInfo()), eqn, NBSolve.Status.EXPLICIT);
      case Equation.FOR_EQUATION()    then SLICED_COMPONENT(ComponentRef.EMPTY(), Slice.SLICE(Pointer.create(NBVariable.DUMMY_VARIABLE), {}), eqn_slice, NBSolve.Status.EXPLICIT);
      // ToDo: the other types
      else algorithm
        Error.addMessage(Error.INTERNAL_ERROR,{getInstanceName() + " failed for:\n" + Slice.toString(eqn_slice, function Equation.pointerToString(str = ""))});
      then fail();
    end match;
  end fromSolvedEquationSlice;

  function toSolvedEquation
    "creates a solved equation for an explicitly solved strong component.
    fails if it is not solved explicitly."
    input StrongComponent comp;
    output Pointer<Equation> eqn;
  algorithm
    eqn := match comp
      case SINGLE_COMPONENT(status = NBSolve.Status.EXPLICIT) then comp.eqn;
      case MULTI_COMPONENT(status = NBSolve.Status.EXPLICIT)  then Slice.getT(comp.eqn);
      case SLICED_COMPONENT(status = NBSolve.Status.EXPLICIT) then Slice.getT(comp.eqn);
      case GENERIC_COMPONENT()                                then Slice.getT(comp.eqn);
      else algorithm
        Error.addMessage(Error.INTERNAL_ERROR,{getInstanceName() + " failed because strong component could not be
        solved explicitly:\n" + toString(comp)});
      then fail();
    end match;
  end toSolvedEquation;

  function collectCrefs
    "Collects dependent crefs in current comp and saves them in the
     unordered map. Saves both directions."
    input StrongComponent comp                                "strong component to be analyzed";
    input VariablePointers var_rep                            "scalarized variable representatives";
    input VariablePointers eqn_rep                            "scalarized equation representatives";
    input Mapping var_rep_mapping                             "index mapping for variable representatives";
    input Mapping eqn_rep_mapping                             "index mapping for equation representatives";
    input UnorderedMap<ComponentRef, list<ComponentRef>> map  "unordered map to save the dependencies";
    input UnorderedSet<ComponentRef> set                      "unordered set of array crefs to check for relevance (index lookup)";
    input JacobianType jacType                                "sets the context";
  algorithm
    () := match comp
      local
        Pointer<Equation> eqn_ptr;
        ComponentRef cref;
        list<ComponentRef> dependencies, loop_vars, tmp;
        list<tuple<ComponentRef, list<ComponentRef>>> scalarized_dependencies;
        Tearing strict;
        Equation eqn, body;
        Iterator iter;
        list<ComponentRef> names;
        list<Expression> ranges;
        UnorderedSet<ComponentRef> deps_set;

      // sliced array equations - create all the single entries
      case SINGLE_COMPONENT() guard(Equation.isArrayEquation(comp.eqn)) algorithm
        dependencies := Equation.collectCrefs(Pointer.access(comp.eqn), function Slice.getDependentCrefCausalized(set = set), Expression.fakeMap);
        scalarized_dependencies := Slice.getDependentCrefsPseudoArrayCausalized(BVariable.getVarName(comp.var), dependencies);
        addScalarizedDependencies(scalarized_dependencies, map, jacType);
      then ();

      case SINGLE_COMPONENT() algorithm
        dependencies := Equation.collectCrefs(Pointer.access(comp.eqn), function Slice.getDependentCrefCausalized(set = set), Expression.fakeMap);
        dependencies := List.flatten(list(ComponentRef.scalarizeAll(dep, true) for dep in dependencies));
        deps_set := prepareDependencies(UnorderedSet.fromList(dependencies, ComponentRef.hash, ComponentRef.isEqual), map, jacType);
        updateDependencyMap(BVariable.getVarName(comp.var), deps_set, map);
      then ();

      case MULTI_COMPONENT() algorithm
        dependencies := Equation.collectCrefs(Pointer.access(Slice.getT(comp.eqn)), function Slice.getDependentCrefCausalized(set = set), Expression.fakeMap);
        dependencies := list(ComponentRef.stripIteratorSubscripts(dep) for dep in dependencies);
        dependencies := List.flatten(list(ComponentRef.scalarizeAll(dep, true) for dep in dependencies));
        deps_set := prepareDependencies(UnorderedSet.fromList(dependencies, ComponentRef.hash, ComponentRef.isEqual), map, jacType);
        for var in comp.vars loop
          for cref in ComponentRef.scalarizeAll(BVariable.getVarName(Slice.getT(var)), true) loop
            updateDependencyMap(cref, deps_set, map);
          end for;
        end for;
      then ();

      // resizable for equations - create all the single entries
      case RESIZABLE_COMPONENT() guard(Equation.isForEquation(Slice.getT(comp.eqn))) algorithm
        addForLoopDependencies(Pointer.access(Slice.getT(comp.eqn)), comp.eqn.indices, comp.var_cref, var_rep, eqn_rep, var_rep_mapping, eqn_rep_mapping, map, set, jacType);
      then ();

      // sliced for equations - create all the single entries
      case SLICED_COMPONENT() guard(Equation.isForEquation(Slice.getT(comp.eqn))) algorithm
        addForLoopDependencies(Pointer.access(Slice.getT(comp.eqn)), comp.eqn.indices, comp.var_cref, var_rep, eqn_rep, var_rep_mapping, eqn_rep_mapping, map, set, jacType);
      then ();

      // sliced array equations - create all the single entries
      case SLICED_COMPONENT() guard(Equation.isArrayEquation(Slice.getT(comp.eqn))) algorithm
        eqn := Pointer.access(Slice.getT(comp.eqn));
        dependencies := Equation.collectCrefs(eqn, function Slice.getDependentCrefCausalized(set = set), Expression.fakeMap);
        scalarized_dependencies := Slice.getDependentCrefsPseudoArrayCausalized(comp.var_cref, dependencies, comp.eqn.indices);
        addScalarizedDependencies(scalarized_dependencies, map, jacType);
      then ();

      // sliced regular equation.
      case SLICED_COMPONENT() algorithm
        eqn := Pointer.access(Slice.getT(comp.eqn));
        dependencies := Equation.collectCrefs(eqn, function Slice.getDependentCrefCausalized(set = set), Expression.fakeMap);
        dependencies := List.flatten(list(ComponentRef.scalarizeAll(dep, true) for dep in dependencies));
        deps_set := prepareDependencies(UnorderedSet.fromList(dependencies, ComponentRef.hash, ComponentRef.isEqual), map, jacType);
        updateDependencyMap(comp.var_cref, deps_set, map);
      then ();

      // sliced for equations - create all the single entries
      case GENERIC_COMPONENT() guard(Equation.isForEquation(Slice.getT(comp.eqn))) algorithm
        addForLoopDependencies(Pointer.access(Slice.getT(comp.eqn)), comp.eqn.indices, comp.var_cref, var_rep, eqn_rep, var_rep_mapping, eqn_rep_mapping, map, set, jacType);
      then ();

      case ALGEBRAIC_LOOP(strict = strict) algorithm
        // traverse residual equations and collect dependencies
        deps_set := UnorderedSet.new(ComponentRef.hash, ComponentRef.isEqual);
        for slice in strict.residual_eqns loop
          // ToDo: does this work properly for arrays?
          tmp := Equation.collectCrefs(Pointer.access(Slice.getT(slice)), function Slice.getDependentCrefCausalized(set = set), Expression.fakeMap);
          eqn_ptr := Slice.getT(slice);
          if Equation.isForEquation(eqn_ptr) then
            // if its a for equation get all dependencies corresponding to their residual.
            // we do not really care for order and assume full dependency anyway
            eqn as Equation.FOR_EQUATION(iter = iter, body = {body}) := Pointer.access(eqn_ptr);
            cref := Equation.getEqnName(eqn_ptr);
            scalarized_dependencies := Slice.getDependentCrefsPseudoForCausalized(
              cref, tmp, var_rep, eqn_rep, var_rep_mapping, eqn_rep_mapping,
              iter, Equation.size(eqn_ptr), slice.indices, true);
            tmp := List.flatten(list(Util.tuple22(tpl) for tpl in scalarized_dependencies));
          end if;
          for dep in tmp loop
            for scal in ComponentRef.scalarizeAll(dep, true) loop
              UnorderedSet.add(scal, deps_set);
            end for;
          end for;
        end for;
        deps_set := prepareDependencies(deps_set, map, jacType);

        // collect iteration loop vars
        loop_vars := list(BVariable.getVarName(Slice.getT(var)) for var in strict.iteration_vars);

        // traverse inner equations and collect loop vars and dependencies
        for i in 1:arrayLength(strict.innerEquations) loop
          // collect inner equation dependencies
          collectCrefs(strict.innerEquations[i], var_rep, eqn_rep, var_rep_mapping, eqn_rep_mapping, map, set, jacType);

          // collect inner loop variables
          loop_vars := listAppend(list(BVariable.getVarName(var) for var in getVariables(strict.innerEquations[i])), loop_vars);
        end for;

        // add all dependencies
        for cref in loop_vars loop
          updateDependencyMap(cref, deps_set, map);
        end for;

      then ();

      case ALIAS() algorithm
        collectCrefs(comp.original, var_rep, eqn_rep, var_rep_mapping, eqn_rep_mapping, map, set, jacType);
      then ();

      /* ToDo add the others and let else case fail! */

      else ();
    end match;
  end collectCrefs;

  function addScalarizedDependencies
    input list<tuple<ComponentRef, list<ComponentRef>>> scalarized_dependencies;
    input UnorderedMap<ComponentRef, list<ComponentRef>> map  "unordered map to save the dependencies";
    input JacobianType jacType                                "sets the context";
  protected
    ComponentRef cref;
    list<ComponentRef> dependencies;
    UnorderedSet<ComponentRef> deps_set;
  algorithm
    for tpl in listReverse(scalarized_dependencies) loop
      (cref, dependencies) := tpl;
      deps_set := prepareDependencies(UnorderedSet.fromList(dependencies, ComponentRef.hash, ComponentRef.isEqual), map, jacType);
      updateDependencyMap(cref, deps_set, map);
    end for;
  end addScalarizedDependencies;

  function addForLoopDependencies
    input Equation eqn;
    input list<Integer> indices;
    input ComponentRef var_cref;
    input VariablePointers var_rep                            "scalarized variable representatives";
    input VariablePointers eqn_rep                            "scalarized equation representatives";
    input Mapping var_rep_mapping                             "index mapping for variable representatives";
    input Mapping eqn_rep_mapping                             "index mapping for equation representatives";
    input UnorderedMap<ComponentRef, list<ComponentRef>> map  "unordered map to save the dependencies";
    input UnorderedSet<ComponentRef> set                      "unordered set of array crefs to check for relevance (index lookup)";
    input JacobianType jacType                                "sets the context";
  protected
    Iterator iter;
    Equation body;
    list<ComponentRef> dependencies;
    ComponentRef cref;
    list<tuple<ComponentRef, list<ComponentRef>>> scalarized_dependencies;
  algorithm
    Equation.FOR_EQUATION(iter = iter, body = {body}) := eqn;
    dependencies := Equation.collectCrefs(eqn, function Slice.getDependentCrefCausalized(set = set), Expression.fakeMap);
    if ComponentRef.isEmpty(var_cref) then
      Expression.CREF(cref = cref) := Equation.getLHS(body);
    else
      cref := var_cref;
    end if;
    scalarized_dependencies := Slice.getDependentCrefsPseudoForCausalized(
      cref, dependencies, var_rep, eqn_rep, var_rep_mapping, eqn_rep_mapping,
      iter, Equation.size(Pointer.create(eqn)), indices, false);
    addScalarizedDependencies(scalarized_dependencies, map, jacType);
  end addForLoopDependencies;

  function addLoopJacobian
    input output StrongComponent comp;
    input Option<BackendDAE> jac;
  algorithm
    comp := match comp
      local
        Tearing strict;

      case ALGEBRAIC_LOOP(strict = strict) algorithm
        // ToDo: update linearity here
        strict.jac := jac;
        comp.strict := strict;
      then comp;

      else algorithm
        Error.addMessage(Error.INTERNAL_ERROR,{getInstanceName() + " failed because of wrong component: " + toString(comp)});
      then fail();
    end match;
  end addLoopJacobian;

  function getLoopResiduals
    input StrongComponent comp;
    output list<Pointer<Variable>> residuals;
  algorithm
    residuals := match comp
      case ALGEBRAIC_LOOP()  then Tearing.getResidualVars(comp.strict);
                        else {};
    end match;
  end getLoopResiduals;

  function getVariables
    input StrongComponent comp;
    output list<Pointer<Variable>> vars;
  algorithm
    vars := match comp
      case SINGLE_COMPONENT()   then {comp.var};
      case MULTI_COMPONENT()    then list(Slice.getT(v) for v in comp.vars);
      case SLICED_COMPONENT()   then {Slice.getT(comp.var)};
      case RESIZABLE_COMPONENT()then {Slice.getT(comp.var)};
      case GENERIC_COMPONENT()  then {Slice.getT(comp.var)};
      case ENTWINED_COMPONENT() then List.flatten(list(getVariables(slice) for slice in comp.entwined_slices));
      case ALGEBRAIC_LOOP()     then Tearing.getResidualVars(comp.strict); // + inner?
      case ALIAS()              then getVariables(comp.original);
      else algorithm
        Error.addMessage(Error.INTERNAL_ERROR,{getInstanceName() + " failed because of wrong component: " + toString(comp)});
      then fail();
    end match;
  end getVariables;

  function getEquations
    input StrongComponent comp;
    output list<Pointer<Equation>> eqns;
  algorithm
    eqns := match comp
      case SINGLE_COMPONENT()   then {comp.eqn};
      case MULTI_COMPONENT()    then {Slice.getT(comp.eqn)};
      case SLICED_COMPONENT()   then {Slice.getT(comp.eqn)};
      case RESIZABLE_COMPONENT()then {Slice.getT(comp.eqn)};
      case GENERIC_COMPONENT()  then {Slice.getT(comp.eqn)};
      case ENTWINED_COMPONENT() then List.flatten(list(getEquations(slice) for slice in comp.entwined_slices));
      case ALGEBRAIC_LOOP()     then Tearing.getResidualEqns(comp.strict); // + inner?
      case ALIAS()              then getEquations(comp.original);
      else algorithm
        Error.addMessage(Error.INTERNAL_ERROR,{getInstanceName() + " failed because of wrong component: " + toString(comp)});
      then fail();
    end match;
  end getEquations;

  function isDiscrete
    "checks if all equations are discrete"
    input StrongComponent comp;
    output Boolean b;
  algorithm
    b := match comp
      case SINGLE_COMPONENT()   then Equation.isDiscrete(comp.eqn);
      case MULTI_COMPONENT()    then Equation.isDiscrete(Slice.getT(comp.eqn));
      case SLICED_COMPONENT()   then Equation.isDiscrete(Slice.getT(comp.eqn));
      case RESIZABLE_COMPONENT()then Equation.isDiscrete(Slice.getT(comp.eqn));
      case ENTWINED_COMPONENT() then List.all(comp.entwined_slices, isDiscrete);
      case GENERIC_COMPONENT()  then Equation.isDiscrete(Slice.getT(comp.eqn));
      case ALGEBRAIC_LOOP()     then comp.mixed;
      case ALIAS()              then isDiscrete(comp.original);
      else algorithm
        Error.addMessage(Error.INTERNAL_ERROR,{getInstanceName() + " failed because of wrong component: " + toString(comp)});
      then fail();
    end match;
  end isDiscrete;

  function isDummy
    input StrongComponent comp;
    output Boolean b;
  algorithm
    b := match comp
      case SINGLE_COMPONENT() then Equation.isDummy(Pointer.access(comp.eqn));
      case MULTI_COMPONENT()  then Equation.isDummy(Pointer.access(Slice.getT(comp.eqn)));
      else false;
    end match;
  end isDummy;

  function isAlias
    input StrongComponent comp;
    output Boolean b;
  algorithm
    b := match comp
      case ALIAS() then true;
      else false;
    end match;
  end isAlias;

  function createPseudoScalar
    input list<Integer> comp_indices;
    input array<Integer> eqn_to_var;
    input Adjacency.Mapping mapping;
    input VariablePointers vars;
    input EquationPointers eqns;
    output StrongComponent comp;
  algorithm
    comp := match comp_indices
      local
        Integer i, var_scal_idx, var_arr_idx, size;
        Pointer<Variable> var;
        Pointer<Equation> eqn;
        list<Slice<VariablePointer>> comp_vars;
        list<Slice<EquationPointer>> comp_eqns;
        Tearing tearingSet;
        Slice<VariablePointer> var_slice;
        Slice<EquationPointer> eqn_slice;
        Pointer<Boolean> homotopy = Pointer.create(false);

      // Size 1 strong component
      // - case 1: sliced equation because of for-equation
      // - case 2: multi components for when/if and algorithm although its size 1
      // - case 3: single or sliced strong component
      case {i} algorithm
        var_scal_idx := eqn_to_var[i];
        var_arr_idx := mapping.var_StA[var_scal_idx];
        var := VariablePointers.getVarAt(vars, var_arr_idx);
        eqn := EquationPointers.getEqnAt(eqns, mapping.eqn_StA[i]);
        (_, size) := mapping.var_AtS[var_arr_idx];

        comp := match Pointer.access(eqn)
          // - case 1: sliced equation because of for-equation
          case _ guard(Equation.isForEquation(eqn)) algorithm
            try
              ({var_slice}, {eqn_slice}) := getLoopVarsAndEqns(comp_indices, eqn_to_var, mapping, vars, eqns);
            else
              Error.addMessage(Error.INTERNAL_ERROR,{getInstanceName() + " failed because single indices did not turn out to be single components."});
              fail();
            end try;
          then SLICED_COMPONENT(VariablePointers.varSlice(vars, var_scal_idx, mapping), var_slice, eqn_slice, NBSolve.Status.UNPROCESSED);

          // - case 2: multi components for when/if and algorithm although its size 1
          case Equation.WHEN_EQUATION()   then MULTI_COMPONENT({Slice.SLICE(var, {})}, Slice.SLICE(eqn, {}), NBSolve.Status.UNPROCESSED);
          case Equation.IF_EQUATION()     then MULTI_COMPONENT({Slice.SLICE(var, {})}, Slice.SLICE(eqn, {}), NBSolve.Status.UNPROCESSED);
          case Equation.ALGORITHM()       then MULTI_COMPONENT({Slice.SLICE(var, {})}, Slice.SLICE(eqn, {}), NBSolve.Status.UNPROCESSED);

          // - case 3: single or sliced strong component
          else algorithm
            try
              ({var_slice}, {eqn_slice}) := getLoopVarsAndEqns(comp_indices, eqn_to_var, mapping, vars, eqns);
            else
              Error.addMessage(Error.INTERNAL_ERROR,{getInstanceName() + " failed because single indices did not turn out to be single components."});
              fail();
            end try;
          then createSliceOrSingle(VariablePointers.varSlice(vars, var_scal_idx, mapping), var_slice, eqn_slice);
        end match;
      then comp;

      // Size > 1 strong component
      case _ algorithm
        (comp_vars, comp_eqns) := getLoopVarsAndEqns(comp_indices, eqn_to_var, mapping, vars, eqns);
        comp := match (comp_vars, comp_eqns)
          case ({var_slice}, {eqn_slice}) guard(not (Equation.isForEquation(Slice.getT(eqn_slice)) or Equation.isAlgorithm(Slice.getT(eqn_slice))))
          then createSliceOrSingle(BVariable.getVarName(Slice.getT(var_slice)), var_slice, eqn_slice);

          // for equations that are not algebraic loops are caught earlier! Any for equation
          // getting to this point is an actual algebraic loop
          case (_, {eqn_slice}) guard(not Equation.isForEquation(Slice.getT(eqn_slice)))
          then MULTI_COMPONENT(
            vars    = comp_vars,
            eqn     = eqn_slice,
            status  = NBSolve.Status.UNPROCESSED
          );

          else algorithm
            tearingSet := Tearing.TEARING_SET(
              iteration_vars  = comp_vars,
              residual_eqns   = comp_eqns,
              innerEquations  = listArray({}),
              jac             = NONE());
            for eqn in comp_eqns loop
              Equation.map(Pointer.access(Slice.getT(eqn)), function Initialization.containsHomotopyCall(b = homotopy));
            end for;
          then ALGEBRAIC_LOOP(
            idx     = -1,
            strict  = tearingSet,
            casual  = NONE(),
            linear  = false,
            mixed   = false,
            homotopy = Pointer.access(homotopy),
            status  = NBSolve.Status.IMPLICIT);
        end match;
      then comp;

      else algorithm
        Error.addMessage(Error.INTERNAL_ERROR, {getInstanceName() + " failed."});
      then fail();
    end match;
  end createPseudoScalar;

  function createSliceOrSingle
    input ComponentRef cref;
    input Slice<VariablePointer> var_slice;
    input Slice<EquationPointer> eqn_slice;
    output StrongComponent comp;
  algorithm
    if Slice.isFull(var_slice) and Slice.isFull(eqn_slice) then
      comp := SINGLE_COMPONENT(
        var       = Slice.getT(var_slice),
        eqn       = Slice.getT(eqn_slice),
        status    = NBSolve.Status.UNPROCESSED);
    else
      comp := SLICED_COMPONENT(
        var_cref  = cref,
        var       = var_slice,
        eqn       = eqn_slice,
        status    = NBSolve.Status.UNPROCESSED);
    end if;
  end createSliceOrSingle;

  // ############################################################
  //                Protected Functions and Types
  // ############################################################

protected
  function getLoopVarsAndEqns
    "adds the equation and matched variable to accumulated lists.
    used to collect algebraic loops.
    ToDo: currently assumes full dependency - update with Slice structures!"
    input list<Integer> comp_indices;
    input array<Integer> eqn_to_var;
    input Adjacency.Mapping mapping;
    input VariablePointers vars;
    input EquationPointers eqns;
    output list<Slice<VariablePointer>> acc_vars = {};
    output list<Slice<EquationPointer>> acc_eqns = {};
  protected
    Integer var_idx, var_arr_idx, var_scal_idx, eqn_arr_idx, eqn_scal_idx;
    list<Integer> idx_lst;
    Pointer<Variable> var;
    Pointer<Equation> eqn;
    Integer len_comps = listLength(comp_indices);
    UnorderedMap<Integer, Slice.IntLst> var_map = UnorderedMap.new<Slice.IntLst>(Util.id, intEq, len_comps);
    UnorderedMap<Integer, Slice.IntLst> eqn_map = UnorderedMap.new<Slice.IntLst>(Util.id, intEq, len_comps);
  algorithm
    // store all component var and eqn indices in maps
    for eqn_idx in comp_indices loop
      var_idx           := eqn_to_var[eqn_idx];
      var_arr_idx       := mapping.var_StA[var_idx];
      eqn_arr_idx       := mapping.eqn_StA[eqn_idx];

      // collect variable and equation slices
      idx_lst := UnorderedMap.getOrDefault(var_arr_idx, var_map, {});
      UnorderedMap.add(var_arr_idx, var_idx :: idx_lst, var_map);
      idx_lst := UnorderedMap.getOrDefault(eqn_arr_idx, eqn_map, {});
      UnorderedMap.add(eqn_arr_idx, eqn_idx :: idx_lst, eqn_map);
    end for;

    // extract variables and equations from maps
    // check if slices are full and reduce them to base 0 indexing
    for tpl in UnorderedMap.toList(var_map) loop
      (var_arr_idx, idx_lst)  := tpl;
      (var_scal_idx, _)       := mapping.var_AtS[var_arr_idx];
      var                     := VariablePointers.getVarAt(vars, var_arr_idx);
      idx_lst                 := if listLength(idx_lst) == BVariable.size(var) then {} else list(i - var_scal_idx for i in idx_lst);
      acc_vars                := Slice.SLICE(var, idx_lst) :: acc_vars;
    end for;
    for tpl in UnorderedMap.toList(eqn_map) loop
      (eqn_arr_idx, idx_lst)  := tpl;
      (eqn_scal_idx, _)       := mapping.eqn_AtS[eqn_arr_idx];
      eqn                     := EquationPointers.getEqnAt(eqns, eqn_arr_idx);
      idx_lst                 := if listLength(idx_lst) == Equation.size(eqn) then {} else list(i - eqn_scal_idx for i in idx_lst);
      acc_eqns                := Slice.SLICE(eqn, idx_lst) :: acc_eqns;
    end for;
  end getLoopVarsAndEqns;

  function updateDependencyMap
    input ComponentRef cref                                   "cref representing current equation";
    input UnorderedSet<ComponentRef> dependencies             "the dependency crefs";
    input UnorderedMap<ComponentRef, list<ComponentRef>> map  "unordered map to save the dependencies";
  protected
    Boolean removed;
  algorithm
    removed := UnorderedSet.remove(cref, dependencies)              "remove self dependency";
    UnorderedMap.add(cref, UnorderedSet.toList(dependencies), map)  "update the current value (res/tmp) --> {independent vars}";
    if removed then UnorderedSet.addNew(cref, dependencies); end if "restore dependencies";
  end updateDependencyMap;

  function prepareDependencies
    input output UnorderedSet<ComponentRef> dependencies;
    input UnorderedMap<ComponentRef, list<ComponentRef>> map;
    input JacobianType jacType;
  protected
    function addStateDependencies
      input ComponentRef dep;
      input UnorderedMap<ComponentRef, list<ComponentRef>> map;
      input output UnorderedSet<ComponentRef> set;
    algorithm
      // if the dependency is a state add itself, otherwise add the dependencies already saved
      // (those are known to be states). ToDo: avoid this check by adding state self dependency beforehand?
      if BVariable.checkCref(dep, BVariable.isState, sourceInfo()) then
        UnorderedSet.add(dep, set);
      else
        for tmp in UnorderedMap.getSafe(dep, map, sourceInfo()) loop
          UnorderedSet.add(tmp, set);
        end for;
      end if;
    end addStateDependencies;
  algorithm
    UnorderedSet.apply(dependencies, function ComponentRef.mapExp(func = Expression.replaceResizableParameter));
    UnorderedSet.apply(dependencies, function ComponentRef.simplifySubscripts(trim = false));
    // replace non derivative dependencies with their previous dependencies
    // (be careful with algebraic loops. this here assumes that cyclic dependencies have already been resolved)
    if jacType == NBJacobian.JacobianType.ODE then
      dependencies := UnorderedSet.fold(dependencies, function addStateDependencies(map = map), UnorderedSet.new(ComponentRef.hash, ComponentRef.isEqual));
    end if;
  end prepareDependencies;

  annotation(__OpenModelica_Interface="backend");
end NBStrongComponent;
