/*
 * This file is part of OpenModelica.
 *
 * Copyright (c) 1998-2014, Open Source Modelica Consortium (OSMC),
 * c/o Linköpings universitet, Department of Computer and Information Science,
 * SE-58183 Linköping, Sweden.
 *
 * All rights reserved.
 *
 * THIS PROGRAM IS PROVIDED UNDER THE TERMS OF GPL VERSION 3 LICENSE OR
 * THIS OSMC PUBLIC LICENSE (OSMC-PL) VERSION 1.2.
 * ANY USE, REPRODUCTION OR DISTRIBUTION OF THIS PROGRAM CONSTITUTES
 * RECIPIENT'S ACCEPTANCE OF THE OSMC PUBLIC LICENSE OR THE GPL VERSION 3,
 * ACCORDING TO RECIPIENTS CHOICE.
 *
 * The OpenModelica software and the Open Source Modelica
 * Consortium (OSMC) Public License (OSMC-PL) are obtained
 * from OSMC, either from the above address,
 * from the URLs: http://www.ida.liu.se/projects/OpenModelica or
 * http://www.openmodelica.org, and in the OpenModelica distribution.
 * GNU version 3 is obtained from: http://www.gnu.org/copyleft/gpl.html.
 *
 * This program is distributed WITHOUT ANY WARRANTY; without
 * even the implied warranty of  MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE, EXCEPT AS EXPRESSLY SET FORTH
 * IN THE BY RECIPIENT SELECTED SUBSIDIARY LICENSE CONDITIONS OF OSMC-PL.
 *
 * See the full OSMC Public License conditions for more details.
 *
 */

encapsulated package SerializeInitXML

import SimCode;
import SimCode.ModelInfo;
import SimCode.SimCode.SIMCODE;
import File;

protected
import BackendDAE.VarKind;
import CR=ComponentReference;
import DAE.{Exp,Type};
import Dump;
import ExpressionDump.printExpStr;
import File.Escape.XML;
import Settings;
import SimCode.{SimulationSettings,VarInfo};
import SimCodeVar.{AliasVariable,Causality,SimVar};
import SimCodeUtil;
import Types;
import Util;

public

function simulationInitFile
 "Generates the contents of the init.xml file for the simulation case."
  input SimCode.SimCode simCode;
  input String guid;
algorithm
  true := simulationInitFileReturnBool(simCode, guid);
end simulationInitFile;

function simulationInitFileReturnBool
 "Generates the contents of the init.xml file for the simulation case."
  input SimCode.SimCode simCode;
  input String guid;
  output Boolean success = false;
protected
  VarInfo vi;
  SimulationSettings s;
  File.File file = File.File();
  String fileName, FMUType;
algorithm
  try
    fileName := match Config.simCodeTarget()
      case "omsic" then simCode.fullPathPrefix+"/"+simCode.fileNamePrefix + "_init.xml";
      /*Temporary disabled omsicpp
      case "omsicpp" then simCode.fullPathPrefix+"/"+simCode.fileNamePrefix + "_init.xml";*/
      else simCode.fileNamePrefix + "_init.xml";
    end match;
    File.open(file, fileName, File.Mode.Write);

    vi := simCode.modelInfo.varInfo;
    SOME(s) := simCode.simulationSettingsOpt;
    FMUType := match Config.simCodeTarget()
      case "omsic" then "2.0";
      case "omsicpp" then "2.0";
      else "1.0";
    end match;


    File.write(file, "<?xml version = \"1.0\" encoding=\"UTF-8\"?>\n\n");
    File.write(file, "<!-- description of the model interface using an extention of the FMI standard -->\n");
    File.write(file, "<fmiModelDescription\n");
    File.write(file, "  fmiVersion                          = \""+FMUType+"\"\n\n");

    File.write(file, "  modelName                           = \"");
    Dump.writePath(file, simCode.modelInfo.name, initialDot=false);
    File.write(file, "\"\n");

    File.write(file, "  modelIdentifier                     = \"");
    Dump.writePath(file, simCode.modelInfo.name, delimiter="_", initialDot=false);
    File.write(file, "\"\n\n");

    File.write(file, "  OPENMODELICAHOME                    = \"");
    File.write(file, simCode.makefileParams.omhome);
    File.write(file, "\"\n\n");


    File.write(file, "  guid                                = \"{");
    File.write(file, guid);
    File.write(file, "}\"\n\n");


    File.write(file, "  description                         = \"");
    File.writeEscape(file, simCode.modelInfo.description, XML);
    File.write(file, "\"\n");

    File.write(file, "  generationTool                      = \"OpenModelica Compiler ");
    File.write(file, Settings.getVersionNr());
    File.write(file, "\"\n");

    File.write(file, "  generationDateAndTime               = \"");
    xsdateTime(file, Util.getCurrentDateTime());
    File.write(file, "\"\n\n");


    File.write(file, "  variableNamingConvention            = \"structured\"\n\n");

    File.write(file, "  numberOfEventIndicators             = \"");
    File.writeInt(file, vi.numZeroCrossings);
    File.write(file, "\"  cmt_numberOfEventIndicators             = \"NG:       number of zero crossings,                           FMI\"\n");

    File.write(file, "  numberOfTimeEvents                  = \"");
    File.writeInt(file, vi.numTimeEvents);
    File.write(file, "\"  cmt_numberOfTimeEvents                  = \"NG_SAM:   number of zero crossings that are samples,          OMC\"\n\n");


    File.write(file, "  numberOfInputVariables              = \"");
    File.writeInt(file, vi.numInVars);
    File.write(file, "\"  cmt_numberOfInputVariables              = \"NI:       number of inputvar on topmodel,                     OMC\"\n");

    File.write(file, "  numberOfOutputVariables             = \"");
    File.writeInt(file, vi.numOutVars);
    File.write(file, "\"  cmt_numberOfOutputVariables             = \"NO:       number of outputvar on topmodel,                    OMC\"\n\n");

    File.write(file, "  numberOfExternalObjects             = \"");
    File.writeInt(file, vi.numExternalObjects);

    File.write(file, "\"  cmt_numberOfExternalObjects             = \"NEXT:     number of external objects,                         OMC\"\n");

    File.write(file, "  numberOfFunctions                   = \"");
    File.writeInt(file, listLength(simCode.modelInfo.functions));
    File.write(file, "\"  cmt_numberOfFunctions                   = \"NFUNC:    number of functions used by the simulation,         OMC\"\n\n");


    File.write(file, "  numberOfContinuousStates            = \"");
    File.writeInt(file, vi.numStateVars);
    File.write(file, "\"  cmt_numberOfContinuousStates            = \"NX:       number of states,                                   FMI\"\n");

    File.write(file, "  numberOfRealAlgebraicVariables      = \"");
    File.writeInt(file, vi.numAlgVars+vi.numDiscreteReal+vi.numOptimizeConstraints+vi.numOptimizeFinalConstraints);
    File.write(file, "\"  cmt_numberOfRealAlgebraicVariables      = \"NY:       number of real variables,                           OMC\"\n");

    File.write(file, "  numberOfRealAlgebraicAliasVariables = \"");
    File.writeInt(file, vi.numAlgAliasVars);
    File.write(file, "\"  cmt_numberOfRealAlgebraicAliasVariables = \"NA:       number of alias variables,                          OMC\"\n");

    File.write(file, "  numberOfRealParameters              = \"");
    File.writeInt(file, vi.numParams);
    File.write(file, "\"  cmt_numberOfRealParameters              = \"NP:       number of parameters,                               OMC\"\n\n");


    File.write(file, "  numberOfIntegerAlgebraicVariables   = \"");
    File.writeInt(file, vi.numIntAlgVars);
    File.write(file, "\"  cmt_numberOfIntegerAlgebraicVariables   = \"NYINT:    number of alg. int variables,                       OMC\"\n");

    File.write(file, "  numberOfIntegerAliasVariables       = \"");
    File.writeInt(file, vi.numIntAliasVars);
    File.write(file, "\"  cmt_numberOfIntegerAliasVariables       = \"NAINT:    number of alias int variables,                      OMC\"\n");

    File.write(file, "  numberOfIntegerParameters           = \"");
    File.writeInt(file, vi.numIntParams);
    File.write(file, "\"  cmt_numberOfIntegerParameters           = \"NPINT:    number of int parameters,                           OMC\"\n\n");

    File.write(file, "  numberOfStringAlgebraicVariables    = \"");
    File.writeInt(file, vi.numStringAlgVars);
    File.write(file, "\"  cmt_numberOfStringAlgebraicVariables    = \"NYSTR:    number of alg. string variables,                    OMC\"\n");

    File.write(file, "  numberOfStringAliasVariables        = \"");
    File.writeInt(file, vi.numStringAliasVars);
    File.write(file, "\"  cmt_numberOfStringAliasVariables        = \"NASTR:    number of alias string variables,                   OMC\"\n");

    File.write(file, "  numberOfStringParameters            = \"");
    File.writeInt(file, vi.numStringParamVars);
    File.write(file, "\"  cmt_numberOfStringParameters            = \"NPSTR:    number of string parameters,                        OMC\"\n\n");


    File.write(file, "  numberOfBooleanAlgebraicVariables   = \"");
    File.writeInt(file, vi.numBoolAlgVars);
    File.write(file, "\"  cmt_numberOfBooleanAlgebraicVariables   = \"NYBOOL:   number of alg. bool variables,                      OMC\"\n");

    File.write(file, "  numberOfBooleanAliasVariables       = \"");
    File.writeInt(file, vi.numBoolAliasVars);
    File.write(file, "\"  cmt_numberOfBooleanAliasVariables       = \"NABOOL:   number of alias bool variables,                     OMC\"\n");

    File.write(file, "  numberOfBooleanParameters           = \"");
    File.writeInt(file, vi.numBoolParams);
    File.write(file, "\"  cmt_numberOfBooleanParameters           = \"NPBOOL:   number of bool parameters,                          OMC\" >\n\n\n");


    File.write(file, "  <!-- startTime, stopTime, tolerance are FMI specific, all others are OMC specific -->\n");

    File.write(file, "  <DefaultExperiment\n");

    File.write(file, "    startTime      = \"");
    File.writeReal(file, s.startTime);
    File.write(file, "\"\n");

    File.write(file, "    stopTime       = \"");
    File.writeReal(file, s.stopTime);
    File.write(file, "\"\n");

    File.write(file, "    stepSize       = \"");
    File.writeReal(file, s.stepSize);
    File.write(file, "\"\n");

    File.write(file, "    tolerance      = \"");
    File.writeReal(file, s.tolerance);
    File.write(file, "\"\n");

    File.write(file, "    solver         = \"");
    File.write(file, s.method);
    File.write(file, "\"\n");

    File.write(file, "    outputFormat   = \"");
    File.write(file, s.outputFormat);
    File.write(file, "\"\n");

    File.write(file, "    variableFilter = \"");
    File.write(file, s.variableFilter);
    File.write(file, "\" />\n\n");

    File.write(file, "  <!-- variables in the model -->\n");
    File.write(file, "  <ModelVariables>\n\n");
    modelVariables(file, simCode.modelInfo.vars);
    File.write(file, "\n\n\n  </ModelVariables>\n\n");

    File.write(file, "\n</fmiModelDescription>\n\n");
    success := true;
  else
  end try;
end simulationInitFileReturnBool;

protected

function modelVariables "Generates code for ModelVariables file for FMU target."
  input File.File file;
  input SimCodeVar.SimVars vars;
protected
  Integer vr, ix=0;
algorithm
  // set starting index
  vr := match Config.simCodeTarget()
    case "omsic" then 0;
    case "omsicpp" then 0;
    else 1000;
  end match;

  vr := scalarVariables(file, vars.stateVars, "rSta", vr);
  vr := scalarVariables(file, vars.derivativeVars, "rDer", vr);
  (vr,ix) := scalarVariables(file, vars.algVars, "rAlg", vr, ix);
  (vr,ix) := scalarVariables(file, vars.discreteAlgVars, "rAlg", vr, ix);
  (vr,ix) := scalarVariables(file, vars.realOptimizeConstraintsVars, "rAlg", vr, ix);
  (vr,ix) := scalarVariables(file, vars.realOptimizeFinalConstraintsVars, "rAlg", vr, ix);
  vr := scalarVariables(file, vars.paramVars, "rPar", vr);
  vr := scalarVariables(file, vars.aliasVars, "rAli", vr);

  vr := scalarVariables(file, vars.intAlgVars, "iAlg", vr);
  vr := scalarVariables(file, vars.intParamVars, "iPar", vr);
  vr := scalarVariables(file, vars.intAliasVars, "iAli", vr);

  vr := scalarVariables(file, vars.boolAlgVars, "bAlg", vr);
  vr := scalarVariables(file, vars.boolParamVars, "bPar", vr);
  vr := scalarVariables(file, vars.boolAliasVars, "bAli", vr);

  vr := scalarVariables(file, vars.stringAlgVars, "sAlg", vr);
  vr := scalarVariables(file, vars.stringParamVars, "sPar", vr);
  vr := scalarVariables(file, vars.stringAliasVars, "sAli", vr);

  // sensitivity variables
  vr := scalarVariables(file, vars.sensitivityVars, "rSen", vr);
end modelVariables;

function scalarVariables
  input File.File file;
  input list<SimVar> vars;
  input String classType;
  input output Integer valueReference;
  input output Integer index=0;
algorithm
  for var in vars loop
    scalarVariable(file, var, classType, valueReference, index);
    index := index + 1;
    valueReference := valueReference + 1;
  end for;
end scalarVariables;

function scalarVariable
  input File.File file;
  input SimVar var;
  input String classType;
  input Integer valueReference;
  input Integer classIndex;
protected
  String type_name = if DAEUtil.expTypeArray(var.type_) then "ArrayVariable" else "ScalarVariable";
algorithm
  File.write(file, "  <" + type_name + "\n");
  scalarVariableAttribute(file, var, classType, valueReference, classIndex);
  File.write(file, "    ");
  scalarVariableType(file, var);
  File.write(file, "\n  </" + type_name + ">\n");
end scalarVariable;

function scalarVariableAttribute "Generates code for ScalarVariable Attribute file for FMU target."
  input File.File file;
  input SimVar simVar;
  input String classType;
  input Integer valueReference;
  input Integer classIndex;
protected
  Integer inputIndex = SimCodeUtil.getInputIndex(simVar);
  SourceInfo info = simVar.source.info;
algorithm

  File.write(file, "    name = \"");
  CR.writeCref(file, simVar.name, XML);
  File.write(file, "\"\n");

  File.write(file, "    valueReference = \"");
  File.writeInt(file, valueReference);
  File.write(file, "\"\n");

  if simVar.comment <> "" then
    File.write(file, "    description = \"");
    File.writeEscape(file, simVar.comment, XML);
    File.write(file, "\"\n");
  end if;

  File.write(file, "    variability = \"");
  File.write(file, getVariablity(simVar.varKind));
  File.write(file, "\" isDiscrete = \"");
  File.write(file, String(simVar.isDiscrete));
  File.write(file, "\"\n");

  File.write(file, "    causality = \"");
  File.write(file, getCausality(simVar.causality));
  File.write(file, "\" isValueChangeable = \"");
  File.write(file, String(simVar.isValueChangeable));
  File.write(file, "\"\n");

  if inputIndex <> -1 then
    File.write(file, "    inputIndex = \"");
    File.writeInt(file, inputIndex);
    File.write(file, "\"\n");
  end if;

  File.write(file, "    alias = ");
  getAliasVar(file, simVar);
  File.write(file, "\n");

  File.write(file, "    classIndex = \"");
  File.writeInt(file, classIndex);
  File.write(file, "\" classType = \"");
  File.write(file, classType);
  File.write(file, "\"\n");

  File.write(file, "    isProtected = \"");
  File.write(file, String(simVar.isProtected));
  File.write(file, "\" hideResult = \"");
  File.write(file, Util.applyOptionOrDefault(simVar.hideResult, boolString, ""));
  File.write(file, "\" isEncrypted = \"");
  File.write(file, boolString(simVar.isEncrypted));
  File.write(file, "\" initNonlinear = \"");
  File.write(file, boolString(simVar.initNonlinear));
  File.write(file, "\"\n");

  File.write(file, "    fileName = \"");
  File.writeEscape(file, info.fileName, XML);
  File.write(file, "\" startLine = \"");
  File.writeInt(file, info.lineNumberStart);
  File.write(file, "\" startColumn = \"");
  File.writeInt(file, info.columnNumberStart);
  File.write(file, "\" endLine = \"");
  File.writeInt(file, info.lineNumberEnd);
  File.write(file, "\" endColumn = \"");
  File.writeInt(file, info.columnNumberEnd);
  File.write(file, "\" fileWritable = \"");
  File.write(file, String(not info.isReadOnly));
  File.write(file, "\">\n");

  for dim in Expression.arrayDimension(simVar.type_) loop
    File.write(file, "    <Dimension start=\"" + intString(Expression.dimensionSize(dim)) + "\"/>\n");
  end for;
end scalarVariableAttribute;

function scalarVariableType "Generates code for ScalarVariable Type file for FMU target."
  input File.File file;
  input SimVar v;
protected
  Absyn.Path path;
algorithm
  () := match Types.arrayElementType(v.type_)
  case Type.T_INTEGER()
    algorithm
      File.write(file, "<Integer");
      scalarVariableTypeAttribute(file, v.initialValue, "start");
      scalarVariableTypeFixedAttribute(file, v.isFixed);
      scalarVariableTypeAttribute(file, v.minValue, "min");
      scalarVariableTypeAttribute(file, v.maxValue, "max");
      scalarVariableTypeStringAttribute(file, v.unit, "unit");
      scalarVariableTypeStringAttribute(file, v.displayUnit, "displayUnit");
      File.write(file, " />");
    then ();
  case Type.T_REAL()
    algorithm
      File.write(file, "<Real");
      scalarVariableTypeAttribute(file, v.initialValue, "start");
      scalarVariableTypeFixedAttribute(file, v.isFixed);
      scalarVariableTypeUseAttribute(file, v.nominalValue, "useNominal", "nominal");
      scalarVariableTypeAttribute(file, v.minValue, "min");
      scalarVariableTypeAttribute(file, v.maxValue, "max");
      scalarVariableTypeStringAttribute(file, v.unit, "unit");
      scalarVariableTypeStringAttribute(file, v.displayUnit, "displayUnit");
      File.write(file, " />");
    then ();
  case Type.T_BOOL()
    algorithm
      File.write(file, "<Boolean");
      scalarVariableTypeAttribute(file, v.initialValue, "start");
      scalarVariableTypeFixedAttribute(file, v.isFixed);
      scalarVariableTypeStringAttribute(file, v.unit, "unit");
      scalarVariableTypeStringAttribute(file, v.displayUnit, "displayUnit");
      File.write(file, " />");
    then ();
  case Type.T_STRING()
    algorithm
      File.write(file, "<String");
      scalarVariableTypeAttribute(file, v.initialValue, "start");
      scalarVariableTypeFixedAttribute(file, v.isFixed);
      scalarVariableTypeStringAttribute(file, v.unit, "unit");
      scalarVariableTypeStringAttribute(file, v.displayUnit, "displayUnit");
      File.write(file, " />");
    then ();
  case Type.T_ENUMERATION()
    algorithm
      File.write(file, "<Integer");
      scalarVariableTypeAttribute(file, v.initialValue, "start");
      scalarVariableTypeFixedAttribute(file, v.isFixed);
      scalarVariableTypeStringAttribute(file, v.unit, "unit");
      scalarVariableTypeStringAttribute(file, v.displayUnit, "displayUnit");
      File.write(file, " />");
    then ();
  case Type.T_COMPLEX(complexClassType = ClassInf.EXTERNAL_OBJ(path=path))
    algorithm
      File.write(file, "<ExternalObject path=\"");
      Dump.writePath(file, path, XML);
      File.write(file, "\" />");
    then ();
  else
    algorithm
      Error.addInternalError(getInstanceName() + ": " + Types.unparseType(v.type_), sourceInfo());
    then fail();
  end match;
end scalarVariableType;

function scalarVariableTypeUseAttribute
  input File.File file;
  input Option<Exp> attr;
  input String use, name;
protected
  String expStr;
algorithm
  File.write(file, " ");
  File.write(file, use);
  File.write(file, "=\"");
  File.write(file, String(isSome(attr)));
  File.write(file, "\"");
  scalarVariableTypeAttribute(file, attr, name);
end scalarVariableTypeUseAttribute;

function scalarVariableTypeFixedAttribute
  input File.File file;
  input Boolean isFixed;
algorithm
  File.write(file, " fixed=\"");
  File.write(file, String(isFixed));
  File.write(file, "\"");
end scalarVariableTypeFixedAttribute;

function scalarVariableTypeAttribute
  input File.File file;
  input Option<Exp> attr;
  input String name;
protected
  String expStr;
algorithm
  try
    expStr := expString(Util.getOption(attr));
    File.write(file, " ");
    File.write(file, name);
    File.write(file, "=\"");
    File.write(file, expStr);
    File.write(file, "\"");
  else
  end try;
end scalarVariableTypeAttribute;

function scalarVariableTypeStringAttribute
  input File.File file;
  input String attr;
  input String name;
algorithm
  if attr == "" then
    return;
  end if;
  File.write(file, " ");
  File.write(file, name);
  File.write(file, "=\"");
  File.writeEscape(file, attr, XML);
  File.write(file, "\"");
end scalarVariableTypeStringAttribute;

function getCausality "Returns the Causality Attribute of ScalarVariable."
  input Option<Causality> c;
  output String str;
algorithm
  str := match c
    case SOME(Causality.NONECAUS()) then "none";
    case SOME(Causality.OUTPUT()) then "output";
    case SOME(Causality.INPUT()) then "input";
    case SOME(Causality.LOCAL()) then "local"; // replacement for INTERNAL(), From now Use LOCAL according to FMI-2.0 specification
    case SOME(Causality.PARAMETER()) then "parameter";
    case SOME(Causality.CALCULATED_PARAMETER()) then "calculatedParameter";
    else "local";
  end match;
end getCausality;

function getVariablity "Returns the variablity Attribute of ScalarVariable."
  input VarKind varKind;
  output String str;
algorithm
  str := match varKind
  case VarKind.DISCRETE() then "discrete";
  case VarKind.PARAM() then "parameter";
  case VarKind.CONST() then "constant";
  else "continuous";
  end match;
end getVariablity;

function getAliasVar "Returns the alias Attribute of ScalarVariable."
  input File.File file;
  input SimCodeVar.SimVar simVar;
algorithm
  () := match simVar
  local SimCodeVar.AliasVariable aliasvar;
  case SimCodeVar.SIMVAR(aliasvar = aliasvar as AliasVariable.ALIAS())
    algorithm
      File.write(file, "\"alias\" aliasVariable=\"");
      CR.writeCref(file, aliasvar.varName, XML);
      File.write(file, "\" aliasVariableId=\"");
      File.write(file, SimCodeUtil.getValueReference(simVar, SimCodeUtil.getSimCode(), true));
      File.write(file, "\"");
    then ();
  case SimCodeVar.SIMVAR(aliasvar = aliasvar as AliasVariable.NEGATEDALIAS())
    algorithm
      File.write(file, "\"negatedAlias\" aliasVariable=\"");
      CR.writeCref(file, aliasvar.varName, XML);
      File.write(file, "\" aliasVariableId=\"");
      File.write(file, SimCodeUtil.getValueReference(simVar, SimCodeUtil.getSimCode(), true));
      File.write(file, "\"");
    then ();
  else
    algorithm File.write(file, "\"noAlias\""); then ();
  end match;
end getAliasVar;

function xsdateTime "YYYY-MM-DDThh:mm:ssZ"
  input File.File file;
  input Util.DateTime dt;
algorithm
  File.writeInt(file, dt.year);
  File.writeInt(file, dt.mon, "-%02d");
  File.writeInt(file, dt.mday, "-%02d");
  File.writeInt(file, dt.hour, "T%02d");
  File.writeInt(file, dt.min, ":%02d");
  File.writeInt(file, dt.sec, ":%02dZ");
end xsdateTime;

function expString
  input Exp exp;
  output String str;
algorithm
  str := match exp
    case Exp.ICONST() then intString(exp.integer);
    case Exp.RCONST() then realString(exp.real);
    case Exp.SCONST() then Util.escapeModelicaStringToXmlString(exp.string);
    case Exp.BCONST() then boolString(exp.bool);
    case Exp.ENUM_LITERAL() then intString(exp.index);
    case Exp.ARRAY() guard Expression.isSimpleLiteralValue(exp, true) then stringDelimitList(list(expString(e) for e in exp.array), " ");
    case Exp.REDUCTION() then expString(exp.expr);
    else fail();
    //else algorithm Error.addInternalError("initial value of unknown type: " + printExpStr(exp), sourceInfo()); then fail();
  end match;
end expString;

annotation(__OpenModelica_Interface="backend");
end SerializeInitXML;
