package SystemDynamics "System Dynamics Library (Version 2.1)"
  package UsersGuide "User's Guide"
    extends Modelica.Icons.Info;
    package Overview "Overview of Library"
      extends Modelica.Icons.Info;
      class Introduction "An Introduction to System Dynamics"
        extends Modelica.Icons.Info;
        annotation(Documentation(info = "<html>
<p><span style=\"color: #008000; font-size: large;\"><strong>System Dynamics</strong></span> is a modeling methodology designed by <a href=\"http://en.wikipedia.org/wiki/Jay_Wright_Forrester\">Jay Forrester</a> in the 1960s and 1970s for modeling and simulating mass and information flows of continuous-time systems in the soft sciences.</p>
<p>It is a low-level modeling methodology.  The software behind this library is not more complex than that behind the Block library.  Thus, the value of this library is not so much in the software modules themselves as it is in the application examples.</p>
<p><span style=\"color: #008000; font-size: large;\"><strong>System Dynamics</strong></span> is important, because this modeling methodology is widely used by researchers in the life and social sciences.</p>
<p>The methodology is built around the concepts of Levels and Rates.  <strong>Levels</strong> are quantities that can be accumulated, whereas <strong>Rates</strong> are the quantities that control the increase and/or decrease of the Levels.</p>
<p>Typical examples of Levels and Rates are given in the graph below:</p>
<p><img src=\"modelica://SystemDynamics/Resources/Images/SD_Fig8.png\" alt=\"\" /></p>
<p>System Dynamics modelers are generally quite \"generous\" (sloppy) in the use of concepts.  This becomes clear when looking at the entries in the above table.</p>
<p>The Levels can be computed from the Rates by a simple differential equation:</p>
<p><img src=\"modelica://SystemDynamics/Resources/Images/SD_Fig10.png\" alt=\"\" /></p>
<p>The Rates are being controlled by a so-called laundry list, a set of influencing factors that control the rates.  Below is a table that provides a possible laundry list for the human birth rate in a population dynamics study:</p>
<p><img src=\"modelica://SystemDynamics/Resources/Images/SD_Fig9.png\" alt=\"\" /></p>
<p>There are usually many ways to model systems from soft sciences, i.e., different sets of variables constituting a laundry list may lead to quite similar simulation results in the end.</p>
<p>The Rates are usually determined by static non-linear functions of several variables, namely those listed in the laundry list:</p>
<p><img src=\"modelica://SystemDynamics/Resources/Images/SD_Fig11.png\" alt=\"\" /></p>
<p>Since it is too complicated to identify such a complicated function of multiple arguments, the System Dynamics methodology usually simplifies the functional relationship by assuming a multiplicative relationship between the different input variables:</p>
<p><img src=\"modelica://SystemDynamics/Resources/Images/SD_Fig12.png\" alt=\"\" /></p>
<p>The individual non-linear functions are now functions of a single argument.  They are usually identified from statistical information and are specified in the model as tabular functions.</p>
<p>A simple example of a System Dynamics model is shown below:</p>
<p><img src=\"modelica://SystemDynamics/Resources/Images/SD_Fig7.png\" alt=\"\" /></p>
<p>This example describes the savings of a person.  The savings (a variable that can be accumulated, i.e., a Level) is controlled by income and expenses (Rate variables).  There are several categories of income (salary and interest), whereby the interest is proportional to the savings.  There are also several categories of expenses, namely bank fees, taxes, utility bills, and cost-of-living expenses.  The bank fees are also proportional to the savings.  If the parameters are set correctly, the modeler can observe whether his or her savings will grow over time, or whether they will get depleted.  Mass flows are represented by lilac connections, whereas information flows are depicted in blue.  The two lilac clouds represent sources and sinks of masses.  These are dummy models in our library.  They were only provided to maintain the look-and-feel of the System Dynamics methodology.</p>
<p>Originally, System Dynamics models were usually encoded in <span style=\"color: red;\"><strong>Dynamo</strong></span>, a very simple alphanumeric modeling/simulation language that was in fact already outdated (in comparison with other modeling/simulation environments) at the time it was created.</p>
<p>With the advent of the Macintosh computer in the mid-eighties, <span style=\"color: red;\"><strong>Dynamo</strong></span> was rapidly replaced by <span style=\"color: red;\"><strong>STELLA</strong></span>, a graphical tool designed for representing System Dynamics models.</p>
<p>Although <span style=\"color: red;\"><strong>STELLA</strong></span> offers a graphical user interface, the tool has its drawbacks.  It is still a very simple environment.  It is not truly hierarchical, and worst of all, it doesn't offer means for <em>documenting</em> models.  Yet documentation is very important, especially in soft sciences.  What precisely does a particular variable mean?  How is it measured?  What measurement units are being used?  For an electrical resistor or a mechanical inertia, the answer to these questions may be obvious, but this is not the case for many of the soft-science models.</p>
<p>The <span style=\"color: red;\"><strong>SystemDynamics</strong></span> Modelica library offers some of the touch and feel of the <span style=\"color: red;\"><strong>STELLA</strong></span> software, while providing the modeler with the full power and flexibility of the <span style=\"color: red;\"><strong>Modelica</strong></span> environment.  Integration of the <span style=\"color: red;\"><strong>SystemDynamics</strong></span> Modelica library with other available <span style=\"color: red;\"><strong>Modelica</strong></span> libraries allows broadening the applications of the System Dynamics modeling paradigm to engineering domains. The multi-formalism and multi-domain capabilities of <span style=\"color: red;\"><strong>Modelica</strong></span> can be exploited to create models of complex and large socio-technical systems that have not been feasible using earlier implementations of the System Dynamics modeling methodology.
</p></html>", revisions = ""));
      end Introduction;
      class Industry "Industrial Dynamics"
        extends Modelica.Icons.Info;
        annotation(Documentation(info = "<html>
This is the oldest application area of System Dynamics.  In fact, the System Dynamics methodology had been originally invented by <a href=\"http://en.wikipedia.org/wiki/Jay_Wright_Forrester\">Jay Forrester</a> to describe processes of factory automation. <p>

Although this is the original application of the methodology, it is not commonly used today for such purposes.  Most people nowadays prefer a different world view for the description of factory automation processes, namely that of a <font color=red><b>process-oriented discrete event simulation</b></font>, as this world view allows us to study better phenomena such as <font color=red><b>deadlock</b></font> and <font color=red><b>starvation</b></font> in a system. <p>

Typical tools for such purposes would include <font color=red><b>ARENA</b></font>, and there exists also a <font color=red><b>Modelica</b></font> implementation of that tool and modeling methodology, called the <font color=red><b>ARENALib</b></font>. <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Kelton, W.D., R.P. Sadowski, and D.T. Sturrock (2003), <a href=\"http://www.amazon.com/Simulation-Arena-CD-Rom-David-Kelton/dp/0072919817\">Simulation with Arena</a>, 3<sup>rd</sup> Edition, McGraw-Hill, 672 p.
<li> Prat, V.S., A.Urqu�a, and S. Dormido (2006), <a href=\"http://www.modelica.org/events/modelica2006/Proceedings/sessions/Session5c2.pdf\">ARENALib: A Modelica Library for Discrete-Event System
Simulation</a>, <i>Proc. 6<sup>th</sup> Modelica Conference</i>, Vienna, Austria, pp.539-548.
</ol> <p>
</html>", revisions = ""));
      end Industry;
      class Population "Population Dynamics"
        extends Modelica.Icons.Info;
        annotation(Documentation(info = "<html>
The System Dynamics methodology is frequently being used for the description of complex interactions between different biological species.  This application area, which forms part of the ecological sciences, is usually referred to as <font color=red><b>population dynamics</b></font>. <p>

This is still one of the most popular application areas of the System Dynamics methodology with thousands of publications describing different aspects of the dynamics of ecosystems.
</html>", revisions = ""));
      end Population;
      class City "Urban Dynamics"
        extends Modelica.Icons.Info;
        annotation(Documentation(info = "<html>
<font color=red><b>Urban Dynamics</b></font> describes an application area that is located somewhere in between <font color=red><b>World Dynamics</b></font> and <font color=red><b>Industrial Dynamics</b></font>.  It includes application areas such as the studying of traffic flows in an inner city or along a freeway. <p>

Most people today prefer a different world view for the description of traffic flows in the inner city, namely that of a <font color=red><b>process-oriented discrete event simulation</b></font>, as this world view allows us to study better phenomena such as <font color=red><b>deadlock</b></font> and <font color=red><b>starvation</b></font> in a system. <p>

Typical tools for such purposes would include <font color=red><b>ARENA</b></font>, and there exists also a <font color=red><b>Modelica</b></font> implementation of that tool and modeling methodology, called the <font color=red><b>ARENALib</b></font>. <p>

Yet, for the study of traffic flow along a freeway, the System Dynamics approach might still be the preferred way of dealing with such problems. <p>

Whereas the discrete event world view follows an individual car through the system from one traffic light and intersection to the next, the System Dynamics world view discusses the problem at a slightly more aggregated level using traffic volume as its levels and traffic densities as its rates. <p>

The <font color=red><b>Modelica</b></font> implementations of the two methodologies would allow us to intersect discrete-event traffic models of an urban area with System-Dynamics traffic models of its surroundings.  This application represents current research and has not much been tackled so far. <p>

<hr> <p>

<b>References:</b> <p>

<ol>
<li> Kelton, W.D., R.P. Sadowski, and D.T. Sturrock (2003), <a href=\"http://www.amazon.com/Simulation-Arena-CD-Rom-David-Kelton/dp/0072919817\">Simulation with Arena</a>, 3<sup>rd</sup> Edition, McGraw-Hill, 672 p.
<li> Prat, V.S., A.Urqu�a, and S. Dormido (2006), <a href=\"http://www.modelica.org/events/modelica2006/Proceedings/sessions/Session5c2.pdf\">ARENALib: A Modelica Library for Discrete-Event System
Simulation</a>, <i>Proc. 6<sup>th</sup> Modelica Conference</i>, Vienna, Austria, pp.539-548.
</ol> <p>
</html>"));
      end City;
      class World "World Dynamics"
        extends Modelica.Icons.Info;
        annotation(Documentation(info = "<html>
One of the most important applications of the System Dynamics methodology are the set of world models that were developed primarily by the M.I.T. group around Profs. <a href=\"http://en.wikipedia.org/wiki/Jay_Wright_Forrester\">Jay Forrester</a> and <a href=\"http://en.wikipedia.org/wiki/Dennis_Meadows\">Dennis Meadows</a>. <p>

Beside from the (meanwhile classical) world models, this view point would include models of climate dynamics, such as models of the emission of greenhouse gases into the atmosphere; models of the melting of our glaciers accompanied by a rise of the waters of our seas; models of the spreading of pollutants through the atmosphere; models of the salination of our soil; and finally, models of the depletion of resources, such as sweet water for drinking and irrigation. <p>

This work, which started in the early 1970s, has recently gained renewed importance with the highly publicized discussion of <font color=red><b>Peak Oil</b></font> and <font color=red><b>Global Warming</b></font>.
</html>", revisions = ""));
      end World;
      class Literature "Literature"
        extends Modelica.Icons.Info;
        annotation(Documentation(info = "<html>
<b>References:</b> <p>

<ol>
<li> Cellier, F.E. (1991), <a href=\"http://www.amazon.com/Continuous-System-Modeling-Fran%C3%A7ois-Cellier/dp/0387975020\">Continuous System Modeling</a>, Springer-Verlag, New York, ISBN: 0-387-97502-0, 755p.
<li> Fisher, D.M. (2007), <a href=\"http://www.iseesystems.com/store/ModelingBook/default.aspx\">Modeling Dynamic Systems: Lessons for a First Course</a>, 2<sup>nd</sup> Edition, ISEE Systems.
<li> Forrester, J.W. (1961), <a href=\"http://www.amazon.com/Forrester-Industrial-Dynamics-Jay-W/dp/0262060035\">Industrial Dynamics</a>, M.I.T. Press, 479p.
<li> Forrester, J.W. (1969), <a href=\"http://www.amazon.com/Urban-Dynamics-Jay-Wright-Forrester/dp/0262060264\">Urban Dynamics</a>, M.I.T. Press, 285p.
<li> Forrester, J.W. (1971), <a href=\"http://www.amazon.com/Principles-Systems-Jay-Wright-Forrester/dp/1883823412/ref=pd_sim_b_1_img/103-2487145-1208659\">Principles of Systems</a>, Pegasus Communications, 387p.
<li> Forrester, J.W. (1971), <a href=\"http://www.amazon.com/World-Dynamics-Jay-W-Forrester/dp/1883823382/ref=ed_oe_h/103-2487145-1208659\">World Dynamics</a>, Pegasus Communications, 160p.
<li> Hammer, M, and J. Champy (1994), <a href=\"http://www.amazon.com/Reengineering-Corporation-Manifesto-Business-Revolution/dp/0066621127\">Reengineering the Corporation: A Manifesto for Business Revolution</a>, Harper Business, New York, 257p.
<li> Kirkwood, C.W. (1998), <a href=\"http://www.public.asu.edu/~kirkwood/sysdyn/SDIntro/SDIntro.htm\">System Dynamics Methods: A Quick Introduction</a>, College of Business Administration, Arizona State University, Tempe, AZ, 117p.
<li> Lebel, J.D. (1982), \"System Dynamics,\" in: <a href=\"http://www.amazon.com/Progress-Modelling-Simulation-F-Cellier/dp/0121647803\">Progress in Modelling and Simulation</a> (F.E. Cellier, Ed.), Academic Press, London, pp. 119-158.
<li> Meadows, D.H., D.L. Meadows, J. Randers, and W.W. Behrens III (1972), <i>Limits to Growth: A Report for the Club of Rome's Project on the Predicament of Mankind</i>, Universe Books, New York, 205p.
<li> Meadows, D.H., D.L. Meadows, and J. Randers (1992), <i>Beyond the Limits</i>, Chelsea Green, 300p.
<li> Meadows, D.H., J. Randers, and D.L. Meadows (2004), <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a>, Chelsea Green, 368p.
<li> Odum, E.P. and G.W. Barrett (2004), <a href=\"http://www.amazon.com/Fundamentals-Ecology-Eugene-Odum/dp/0534420664\">Fundamentals of Ecology</a>, 5<sup>th</sup> Edition, Brooks Cole Publishing, 624 p.
<li> Odum, E.P. (1997), <a href=\"http://www.amazon.com/Ecology-Bridge-Between-Science-Society/dp/0878936300\">Ecology: A Bridge between Science and Society</a>, 3<sup>rd</sup> Revised Edition, Sinauer Associates, 330 p.
<li> Odum, H.T. (1971), <a href=\"http://www.amazon.com/Environment-Society-Environmental-Science-Technology/dp/047165275X\">Environment, Power, and Society</a>, John Wiley & Sons, 331 p.
<li> Odum, H.T. (1995), <a href=\"http://www.amazon.com/Environmental-Accounting-Emergy-Decision-Making/dp/0471114421\">Environmental Accounting: Emergy and Environmental Decision Making</a> John Wiley, 384 p.
<li> Odum, H.T. and E.C. Odum (2000), <a href=\"http://www.amazon.com/Modeling-all-Scales-Introduction-Simulation/dp/0125241704\">Modeling for all Scales: An Introduction to System Simulation</a>, Academic Press, 480 p.
<li> Richmond, B. (1997), <a href=\"http://www.iseesystems.com/store/ProductDescription.aspx?Type=PRODUCT&ID=693\">An Introduction to Systems Thinking</a>, STELLA Edition, ISEE Systems.
<li> Richmond, B., S. Peterson, and P. Vescuso (1987), <a href=\"http://www.amazon.com/ACADEMIC-USERS-GUIDE-STELLA/dp/B000RRMJHS/ref=sr_1_6/103-2487145-1208659?ie=UTF8&s=books&qid=1188791651&sr=1-6\">An Academic User's Guide to STELLA</a>, High Performance Systems, Inc., Lyme, N.H.
</ol> <p>
</html>"));
      end Literature;
      annotation(Documentation(info = "<html>
This overview contains the following sections: <p>

<ol>
<li><a href=\"Modelica://SystemDynamics.UsersGuide.Overview.Introduction\">An Introduction to System Dynamics</a>
<li><a href=\"Modelica://SystemDynamics.UsersGuide.Overview.Industry\">Industrial Dynamics</a>
<li><a href=\"Modelica://SystemDynamics.UsersGuide.Overview.Population\">Population Dynamics</a>
<li><a href=\"Modelica://SystemDynamics.UsersGuide.Overview.City\">Urban Dynamics</a>
<li><a href=\"Modelica://SystemDynamics.UsersGuide.Overview.World\">World Dynamics</a>
<li><a href=\"Modelica://SystemDynamics.UsersGuide.Overview.Literature\">Literature</a>
</ol>
</html>"));
    end Overview;
    package Versions "Release Notes"
      extends Modelica.Icons.Info;
      class Version_1_0 "old -> Version 1.0 (April 9, 2002)"
        extends Modelica.Icons.Info;
        annotation(Documentation(info = "<html>
<font color=red><b>SystemDynamics 1.0</b></font> constituted the first official release
of the <font color=red><b>SystemDynamics</b></font> library. That version was created by Stefan Fabricius in 2002, at a time when Stefan worked as a Ph.D. student at ETH Zurich.  After his graduation in 2003, Stefan decided to work in industry.  His current address is: <p>

<ul>
<li>Dr. Stefan Fabricius<br>
    PROMASIM GmbH<br>
    Dorfstr. 34<br>
    CH-8835 Feusisberg/SZ<br>
    Switzerland<br>
    email: <a href=\"mailto:stefan.fabricius@promasim.ch\">stefan.fabricius@promasim.ch</a>
</ul>

</html>"));
      end Version_1_0;
      class Version_2_0 "Version 1.0 -> Version 2.0 (September 13, 2007)"
        extends Modelica.Icons.Info;
        annotation(Documentation(info = "<html>
<font color=red><b>SystemDynamics 2.0</b></font> constitutes a completely reworked release
of the <font color=red><b>SystemDynamics</b></font> library.  This version is based in part
on Version 1.0, and in part on a different and independently created version of a System Dynamics
library for <font color=red><b>Dymola/Modelica</b></font>, a version that had been
developed by Prof. Cellier and his students at the University of Arizona, and had
been around in a much reduced form since 2001.  That version of SystemDynamics was
coded in <font color=red><b>Dymola 4.0</b></font> and was without documentation.
The sample programs were stored in separate files.  That version was strictly meant to
be used by Dr. Cellier's students. <p>

<ul>
<li><font color=red><b>Conversion from Dymola 4.0:</b></font> The library has been
    upgraded to the current graphical interface of Dymola 6.1.</li>

<li><font color=red><b>Documentation:</b></font> Appropriate documentation has been
    added to all modules.</li>

<li><font color=red><b>Examples:</b></font> An enhanced set of examples has been added to the
    library for improved documentation.</li>

</ol>
</html>", revisions = ""));
      end Version_2_0;
      class Version_2_1 "Version 2.0 -> Version 2.1 (October 16, 2013)"
        extends Modelica.Icons.Info;
        annotation(Documentation(info = "<html>
<p><span style=\"color: red;\"><strong>SystemDynamics 2.1</strong></span> is a reworked release of the <span style=\"color: red;\"><strong>SystemDynamics 2.0</strong></span> library.  This version is based on Version 2.0 that was developed by Prof. Cellier and his students, and have been primarily updated in order for it to work with MSL 3.2.1. The diagrams has been corrected from the old graphical definitions and some icons have been added.</p>
<ul>
<li><font color=red><b>Conversion from MSL 2.2.1:</b></font> The library has been upgraded to MSL 3.2.1, including redefined connectors</li>
<li>Added a unique package icon</li>
<li>Added a pseudo number generator component</li>
</ol>
</ul>
</html>", revisions = ""));
      end Version_2_1;
      annotation(Documentation(info = "<html><p>This section explains the major improvements made to the library from one version to the next:</p>
<p>&nbsp;</p>
<ol>
<li><a href=\"Modelica://SystemDynamics.UsersGuide.Versions.Version_1_0\">old -&gt; Version 1.0</a> (April 9, 2002) </li>
<li><a href=\"Modelica://SystemDynamics.UsersGuide.Versions.Version_2_0\">Version 1.0 -&gt; Version 2.0</a> (September 13, 2007) </li>
<li><a href=\"Modelica://SystemDynamics.UsersGuide.Versions.Version_2_1\">Version 2.0 -&gt; Version 2.1</a> (June 5, 2013) </li>
</ol> </html>", revisions = ""));
    end Versions;
    annotation(DocumentationClass = true, Documentation(info = "<html>
<h4><font color=\"#008000\" size=5>Users Guide of Package SystemDynamics</font></h4>
<p>
Library <b>SystemDynamics</b> is a <b>free</b> Modelica package providing components to model mass and information flows in continuous-time systems using the System Dynamics methodology. This package contains the <b>user’s  guide</b> to the library, and has the following content:
</p>
<ol>
<li><a href=\"Modelica://SystemDynamics.UsersGuide.Overview\">Overview of library</a>
       gives an overview of the library.
<li><a href=\"Modelica://SystemDynamics.UsersGuide.Versions\">Release Notes</a>
       offers a historic perspective on the development of this library.
</ol>
</html>", revisions = ""));
  end UsersGuide;
  package Interfaces "Connectors and partial models of the System Dynamics methodology"
    extends Modelica.Icons.Library;
    connector MassInPort = Modelica.Blocks.Interfaces.RealInput "Mass flow input signal" annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Polygon(visible = true, lineColor = {95,0,191}, fillColor = {95,0,191}, fillPattern = FillPattern.Solid, points = {{0.0,50.0},{100.0,0.0},{0.0,-50.0},{0.0,50.0}})}), Icon(coordinateSystem(extent = {{-100,-100},{100,100}}), graphics = {Polygon(points = {{-100,100},{100,0},{-100,-100},{-100,100}}, fillPattern = FillPattern.Solid, lineColor = {95,0,191}, fillColor = {95,0,191})}));
    connector MassOutPort = Modelica.Blocks.Interfaces.RealOutput "Mass flow output signal" annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Polygon(visible = true, lineColor = {95,0,191}, fillColor = {255,255,255}, fillPattern = FillPattern.Solid, points = {{-100.0,50.0},{0.0,0.0},{-100.0,-50.0},{-100.0,50.0}})}), Icon(coordinateSystem(extent = {{-100,-100},{100,100}}), graphics = {Polygon(points = {{-100,100},{100,0},{-100,-100},{-100,100}}, fillPattern = FillPattern.Solid, lineColor = {95,0,191}, fillColor = {255,255,255})}));
    partial block Nonlin_0 "Non-linear function with zero inputs"
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Line(visible = true, points = {{-80.0,80.0},{-80.0,-80.0},{20.0,-80.0},{80.0,0.0},{20.0,80.0},{-80.0,80.0}}, color = {0,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-60.0,-40.0},{60.0,40.0}}, textString = "y = const", fontName = "Arial")}), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Line(visible = true, points = {{-80.0,80.0},{-80.0,-80.0},{20.0,-80.0},{80.0,0.0},{20.0,80.0},{-80.0,80.0}}, color = {0,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-60.0,-40.0},{60.0,40.0}}, textString = "y = const", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,80.0},{100.0,118.0}}, textString = "%name", fontName = "Arial")}), Documentation(info = "<html>
General non-linear function without explicit inputs.  It does not have to be a constant function.  It could be a function of <i>time</i>.
</html>"));
    end Nonlin_0;
    partial block Nonlin_1 "Non-linear function with one input"
      Modelica.Blocks.Interfaces.RealInput u "Input variable" annotation(Placement(visible = true, transformation(origin = {-90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Line(visible = true, points = {{-80.0,80.0},{-80.0,-80.0},{20.0,-80.0},{80.0,0.0},{20.0,80.0},{-80.0,80.0}}, color = {0,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-60.0,-40.0},{60.0,40.0}}, textString = "y = f(u)", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,80.0},{100.0,118.0}}, textString = "%name", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Line(visible = true, points = {{-80.0,80.0},{-80.0,-80.0},{20.0,-80.0},{80.0,0.0},{20.0,80.0},{-80.0,80.0}}, color = {0,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-60.0,-40.0},{60.0,40.0}}, textString = "y = f(u)", fontName = "Arial")}), Documentation(info = "<html>
General non-linear function with one explicit input.
</html>"));
    end Nonlin_1;
    partial block Nonlin_2 "Non-linear function with two inputs"
      Modelica.Blocks.Interfaces.RealInput u1 "First input variable" annotation(Placement(visible = true, transformation(origin = {-90.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-90.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u2 "Second input variable" annotation(Placement(visible = true, transformation(origin = {-90.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-90.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Line(visible = true, points = {{-80.0,80.0},{-80.0,-80.0},{20.0,-80.0},{80.0,0.0},{20.0,80.0},{-80.0,80.0}}, color = {0,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-40.0},{80.0,40.0}}, textString = "y = f(u1,u2)", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-76.0,30.0},{-54.0,50.0}}, textString = "u1", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-76.0,-50.0},{-54.0,-30.0}}, textString = "u2", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,80.0},{100.0,118.0}}, textString = "%name", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Line(visible = true, points = {{-80.0,80.0},{-80.0,-80.0},{20.0,-80.0},{80.0,0.0},{20.0,80.0},{-80.0,80.0}}, color = {0,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-40.0},{80.0,40.0}}, textString = "y = f(u1,u2)", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-76.0,30.0},{-54.0,50.0}}, textString = "u1", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-76.0,-50.0},{-54.0,-30.0}}, textString = "u2", fontName = "Arial")}), Documentation(info = "<html>
General non-linear function with two explicit inputs.
</html>"));
    end Nonlin_2;
    partial block Nonlin_3 "Non-linear function with three inputs"
      Modelica.Blocks.Interfaces.RealInput u1 "First input variable" annotation(Placement(visible = true, transformation(origin = {-90.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-90.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u2 "Second input variable" annotation(Placement(visible = true, transformation(origin = {-90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u3 "Third input variable" annotation(Placement(visible = true, transformation(origin = {-90.0,-60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-90.0,-60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Line(visible = true, points = {{-80.0,80.0},{-80.0,-80.0},{20.0,-80.0},{80.0,0.0},{20.0,80.0},{-80.0,80.0}}, color = {0,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,80.0},{100.0,118.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-40.0},{80.0,40.0}}, textString = "y = f(u1,u2,u3)", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-76.0,50.0},{-54.0,70.0}}, textString = "u1", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-76.0,-70.0},{-54.0,-50.0}}, textString = "u3", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Line(visible = true, points = {{-80.0,80.0},{-80.0,-80.0},{20.0,-80.0},{80.0,0.0},{20.0,80.0},{-80.0,80.0}}, color = {0,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-40.0},{80.0,40.0}}, textString = "y = f(u1,u2,u3)", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-76.0,50.0},{-54.0,70.0}}, textString = "u1", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-76.0,-70.0},{-54.0,-50.0}}, textString = "u3", fontName = "Arial")}), Documentation(info = "<html>
General non-linear function with three explicit inputs.
</html>"));
    end Nonlin_3;
    partial block Nonlin_4 "Non-linear function with four inputs"
      Modelica.Blocks.Interfaces.RealInput u1 "First input variable" annotation(Placement(visible = true, transformation(origin = {-90.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-90.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u2 "Second input variable" annotation(Placement(visible = true, transformation(origin = {-90.0,20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-90.0,20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u3 "Third input variable" annotation(Placement(visible = true, transformation(origin = {-90.0,-20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-90.0,-20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u4 "Fourth input variable" annotation(Placement(visible = true, transformation(origin = {-90.0,-60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-90.0,-60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Line(visible = true, points = {{-80.0,80.0},{-80.0,-80.0},{20.0,-80.0},{80.0,0.0},{20.0,80.0},{-80.0,80.0}}, color = {0,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-40.0},{80.0,40.0}}, textString = "y = f(u1,u2,u3,u4)", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-76.0,50.0},{-54.0,70.0}}, textString = "u1", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-76.0,-70.0},{-54.0,-50.0}}, textString = "u4", fontName = "Arial")}), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Line(visible = true, points = {{-80.0,80.0},{-80.0,-80.0},{20.0,-80.0},{80.0,0.0},{20.0,80.0},{-80.0,80.0}}, color = {0,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-40.0},{80.0,40.0}}, textString = "y = f(u1,u2,u3,u4)", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-76.0,50.0},{-54.0,70.0}}, textString = "u1", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-76.0,-70.0},{-54.0,-50.0}}, textString = "u4", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,80.0},{100.0,118.0}}, textString = "%name", fontName = "Arial")}), Documentation(info = "<html>
General non-linear function with four explicit inputs.
</html>"));
    end Nonlin_4;
    annotation(preferedView = "info", Documentation(info = "<html>
This package contains the mass flow connectors of the System Dynamics methodology and some partial models that are used elsewhere.
</html>"));
  end Interfaces;
  package Auxiliary "Auxiliary elements of the System Dynamics methodology"
    extends Modelica.Icons.Library;
    block Const "A constant factor"
      parameter Real k = 0 "Constant additive term";
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {0.0,70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270), iconTransformation(origin = {0.0,70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
    equation
      y = k;
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,255}, lineThickness = 0.5, extent = {{-60.0,-60.0},{60.0,60.0}})}), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,255}, lineThickness = 0.5, extent = {{-60.0,-60.0},{60.0,60.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-60.0,-30.0},{60.0,34.0}}, textString = "k=%k", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,-112.0},{100.0,-70.0}}, textString = "%name", fontName = "Arial")}), Documentation(info = "<html>
System Dynamics constant factor model.
</html>"));
    end Const;
    block Gain "Gain factor"
      parameter Real k = 1 "Constant multiplicative factor";
      Modelica.Blocks.Interfaces.RealInput u "Input variable" annotation(Placement(visible = true, transformation(origin = {-70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
    equation
      y = k * u;
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, extent = {{-60.0,-60.0},{60.0,60.0}})}), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, extent = {{-60.0,-60.0},{60.0,60.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-60.0,-20.0},{60.0,20.0}}, textString = "k = %k", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-98.0,-118.0},{100.0,-80.0}}, textString = "%name", fontName = "Arial")}), Documentation(info = "<html>
Amplification gain factor.
</html>"));
    end Gain;
    block Prod_2 "Product of two influencing factors"
      Modelica.Blocks.Interfaces.RealInput u1 "First input variable" annotation(Placement(visible = true, transformation(origin = {-70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u2 "Second input variable" annotation(Placement(visible = true, transformation(origin = {70.0,0.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0), iconTransformation(origin = {70.0,0.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {0.0,70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270), iconTransformation(origin = {0.0,70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
    equation
      y = u1 * u2;
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,255}, lineThickness = 0.5, extent = {{-60.0,-60.0},{60.0,60.0}}),Line(visible = true, points = {{-30.0,0.0},{30.0,0.0}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-15.0,-25.99},{15.0,25.99}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-15.0,25.99},{15.0,-25.99}}, color = {0,0,255}, thickness = 0.5)}), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,255}, lineThickness = 0.5, extent = {{-60.0,-60.0},{60.0,60.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-38.0,-44.0},{40.0,44.0}}, textString = "", fontName = "Arial"),Line(visible = true, points = {{-30.0,0.0},{30.0,0.0}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-15.0,-25.99},{15.0,25.99}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-15.0,25.99},{15.0,-25.99}}, color = {0,0,255}, thickness = 0.5)}), Documentation(info = "<html>
Product of two influencing factors.
</html>"));
    end Prod_2;
    block Prod_3 "Product of three influencing factors"
      Modelica.Blocks.Interfaces.RealInput u1 "First input variable" annotation(Placement(visible = true, transformation(origin = {-70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u2 "Second input variable" annotation(Placement(visible = true, transformation(origin = {0.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270), iconTransformation(origin = {0.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u3 "Third input variable" annotation(Placement(visible = true, transformation(origin = {70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -180), iconTransformation(origin = {70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -180)));
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {0.0,70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270), iconTransformation(origin = {0.0,70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
    equation
      y = u1 * u2 * u3;
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Line(visible = true, points = {{-30.0,0.0},{30.0,0.0}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-15.0,-25.99},{15.0,25.99}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-15.0,25.99},{15.0,-25.99}}, color = {0,0,255}, thickness = 0.5),Ellipse(visible = true, lineColor = {0,0,255}, lineThickness = 0.5, extent = {{-60.0,-60.0},{60.0,60.0}})}), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,255}, lineThickness = 0.5, extent = {{-60.0,-60.0},{60.0,60.0}}),Line(visible = true, points = {{-30.0,0.0},{30.0,0.0}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-15.0,-25.99},{15.0,25.99}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-15.0,25.99},{15.0,-25.99}}, color = {0,0,255}, thickness = 0.5)}), Documentation(info = "<html>
Product of three influencing factors.
</html>"));
    end Prod_3;
    block Prod_4 "Product of four influencing factors"
      Modelica.Blocks.Interfaces.RealInput u1 "First input variable" annotation(Placement(visible = true, transformation(origin = {-70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u2 "Second input variable" annotation(Placement(visible = true, transformation(origin = {-40.0,-56.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270), iconTransformation(origin = {-40.0,-56.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u3 "Third input variable" annotation(Placement(visible = true, transformation(origin = {40.0,-56.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270), iconTransformation(origin = {40.0,-56.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u4 "Fourth input variable" annotation(Placement(visible = true, transformation(origin = {70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -180), iconTransformation(origin = {70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -180)));
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {0.0,70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270), iconTransformation(origin = {0.0,70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
    equation
      y = u1 * u2 * u3 * u4;
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Line(visible = true, points = {{-15.0,25.99},{15.0,-25.99}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-15.0,-25.99},{15.0,25.99}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-30.0,0.0},{30.0,0.0}}, color = {0,0,255}, thickness = 0.5),Ellipse(visible = true, lineColor = {0,0,255}, lineThickness = 0.5, extent = {{-60.0,-60.0},{60.0,60.0}})}), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,255}, lineThickness = 0.5, extent = {{-60.0,-60.0},{60.0,60.0}}),Line(visible = true, points = {{-30.0,0.0},{30.0,0.0}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-15.0,-25.99},{15.0,25.99}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-15.0,25.99},{15.0,-25.99}}, color = {0,0,255}, thickness = 0.5)}), Documentation(info = "<html>
Product of four influencing factors.
</html>"));
    end Prod_4;
    block Prod_5 "Product of five influencing factors"
      Modelica.Blocks.Interfaces.RealInput u1 "First input variable" annotation(Placement(visible = true, transformation(origin = {-70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u2 "Second input variable" annotation(Placement(visible = true, transformation(origin = {-46.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270), iconTransformation(origin = {-46.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u3 "Third input variable" annotation(Placement(visible = true, transformation(origin = {0.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270), iconTransformation(origin = {0.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u4 "Fourth input variable" annotation(Placement(visible = true, transformation(origin = {46.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270), iconTransformation(origin = {46.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u5 "Fifth input variable" annotation(Placement(visible = true, transformation(origin = {70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -180), iconTransformation(origin = {70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -180)));
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {0.0,70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270), iconTransformation(origin = {0.0,70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
    equation
      y = u1 * u2 * u3 * u4 * u5;
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Line(visible = true, points = {{-15.0,25.99},{15.0,-25.99}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-15.0,-25.99},{15.0,25.99}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-30.0,0.0},{30.0,0.0}}, color = {0,0,255}, thickness = 0.5),Ellipse(visible = true, lineColor = {0,0,255}, lineThickness = 0.5, extent = {{-60.0,-60.0},{60.0,60.0}})}), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,255}, lineThickness = 0.5, extent = {{-60.0,-60.0},{60.0,60.0}}),Line(visible = true, points = {{-30.0,0.0},{30.0,0.0}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-15.0,-25.99},{15.0,25.99}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-15.0,25.99},{15.0,-25.99}}, color = {0,0,255}, thickness = 0.5)}), Documentation(info = "<html>
Product of five influencing factors.
</html>"));
    end Prod_5;
    annotation(preferedView = "info", Documentation(info = "<html>
This package contains a set of simple static relationships that are frequently used in the System Dynamics approach to modeling.
</html>"));
  end Auxiliary;
  package Functions "Functions of the System Dynamics methodology"
    extends Modelica.Icons.Library annotation(preferedView = "info", Documentation(info = "<html>
This package contains a number of standard functional relationships used in the System Dynamics methodology.
</html>"));
    block Dead_Time "Delay function"
      parameter Modelica.SIunits.Time delayTime = 1 "Delay time of output with respect to input signal";
      Modelica.Blocks.Interfaces.RealInput u "Input variable" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Nonlinear.FixedDelay FixedDelay(delayTime = delayTime) annotation(Placement(visible = true, transformation(origin = {0.0,0.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
    equation
      connect(u,FixedDelay.u) annotation(Line(points = {{-110.0,0.0},{-24.0,0.0}}, color = {0,0,191}, visible = true));
      connect(FixedDelay.y,y) annotation(Line(points = {{22.0,0.0},{110.0,0.0}}, color = {0,0,191}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,191}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-100.0,-100.0},{100.0,100.0}}),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,191}, extent = {{-64.0,-44.0},{68.0,42.0}}, textString = "Delay", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics));
    end Dead_Time;
    block DELAY1 "First-order exponential delaying"
      parameter Real averaging_time = 1 "Averaging time";
      parameter Real delay_init = 0 "Initial value of delayed variable";
      SystemDynamics.Interfaces.MassInPort massInPort "Mass input signal" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort massOutPort "Mass output signal" annotation(Placement(visible = true, transformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Levels.Level Level1(x0 = delay_init) annotation(Placement(visible = true, transformation(origin = {-40.0,0.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
      SystemDynamics.Rates.Rate_1 Rate1 annotation(Placement(visible = true, transformation(origin = {40.0,0.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
      SystemDynamics.Auxiliary.Gain Delaying1(k = 1 / averaging_time) annotation(Placement(visible = true, transformation(origin = {9.0,-40.0}, extent = {{-29.0,20.0},{29.0,-20.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput state "State signal" annotation(Placement(visible = true, transformation(origin = {0.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270), iconTransformation(origin = {0.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
    equation
      connect(Level1.u1,massInPort) annotation(Line(points = {{-62.0,0.0},{-110.0,0.0}}, color = {95,0,191}, visible = true));
      connect(Level1.u2,Rate1.y) annotation(Line(points = {{-18.0,0.0},{30.0,0.0}}, color = {95,0,191}, visible = true));
      connect(Rate1.y1,massOutPort) annotation(Line(points = {{50.0,0.0},{110.0,0.0}}, color = {95,0,191}, visible = true));
      connect(Delaying1.y,Rate1.u) annotation(Line(points = {{29.3,-40.0},{40.0,-40.0},{40.0,-21.0}}, color = {0,0,191}, visible = true));
      connect(Delaying1.u,Level1.y2) annotation(Line(points = {{-11.3,-40.0},{-25.0,-40.0},{-25.0,-13.0}}, color = {0,0,191}, visible = true));
      connect(state,Level1.y1) annotation(Line(points = {{7.6667,66.0},{7.6667,-33.0},{-15.3333,-33.0}}, color = {0,0,191}, visible = true, origin = {-7.6667,44.0}));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,191}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, extent = {{-100.0,-100.0},{100.0,100.0}}),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,191}, extent = {{-64.0,-44.0},{68.0,42.0}}, textString = "DELAY1", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Documentation(info = "<html>
This is a first-order exponential delaying function.  It describes the delaying of mass flows. <p>
 
Incoming goods are delayed temporarily in an internal state.  The <b>state</b> output represents the amount of goods currently stored in the DELAY block. <p>
 
The initial number of items in the DELAY block can be set.  By default, it assumes a value of zero.
</html>"));
    end DELAY1;
    block DELAY3 "Third-order exponential delaying"
      parameter Real averaging_time = 1 "Averaging time";
      parameter Real delay_init = 0 "Initial value of delayed variable";
      SystemDynamics.Interfaces.MassInPort massInPort "Mass input signal" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort massOutPort "Mass output signal" annotation(Placement(visible = true, transformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput state "State signal" annotation(Placement(visible = true, transformation(origin = {0.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270), iconTransformation(origin = {0.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
      SystemDynamics.Levels.Level Level1(x0 = delay_init / 3) annotation(Placement(visible = true, transformation(origin = {-60.0,70.0}, extent = {{-20.0,-15.0},{20.0,15.0}}, rotation = 0)));
      SystemDynamics.Rates.Rate_1 Rate1 annotation(Placement(visible = true, transformation(origin = {10.0,70.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
      SystemDynamics.Auxiliary.Gain Delaying1(k = 3 / averaging_time) annotation(Placement(visible = true, transformation(origin = {-17.0,41.0}, extent = {{-29.0,14.0},{29.0,-14.0}}, rotation = 0)));
      SystemDynamics.Levels.Level Level2(x0 = delay_init / 3) annotation(Placement(visible = true, transformation(origin = {-40.0,10.0}, extent = {{-20.0,-15.0},{20.0,15.0}}, rotation = 0)));
      SystemDynamics.Rates.Rate_1 Rate2 annotation(Placement(visible = true, transformation(origin = {30.0,10.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
      SystemDynamics.Auxiliary.Gain Delaying2(k = 3 / averaging_time) annotation(Placement(visible = true, transformation(origin = {3.0,-19.0}, extent = {{-29.0,14.0},{29.0,-14.0}}, rotation = 0)));
      SystemDynamics.Levels.Level Level3(x0 = delay_init / 3) annotation(Placement(visible = true, transformation(origin = {-20.0,-50.0}, extent = {{-20.0,-15.0},{20.0,15.0}}, rotation = 0)));
      SystemDynamics.Rates.Rate_1 Rate3 annotation(Placement(visible = true, transformation(origin = {50.0,-50.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
      SystemDynamics.Auxiliary.Gain Delaying3(k = 3 / averaging_time) annotation(Placement(visible = true, transformation(origin = {23.0,-79.0}, extent = {{-29.0,14.0},{29.0,-14.0}}, rotation = 0)));
      Modelica.Blocks.Math.Add3 Add3 annotation(Placement(visible = true, transformation(origin = {70.0,70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
    equation
      connect(Rate1.y,Level1.u2) annotation(Line(points = {{2.0,70.0},{-38.0,70.0}}, color = {95,0,191}, visible = true));
      connect(Delaying1.u,Level1.y2) annotation(Line(points = {{-37.3,41.0},{-45.0,41.0},{-45.0,60.25}}, color = {0,0,191}, visible = true));
      connect(Delaying1.y,Rate1.u) annotation(Line(points = {{3.3,41.0},{10.0,41.0},{10.0,54.25}}, color = {0,0,191}, visible = true));
      connect(Rate2.y,Level2.u2) annotation(Line(points = {{22.0,10.0},{-18.0,10.0}}, color = {95,0,191}, visible = true));
      connect(Delaying2.u,Level2.y2) annotation(Line(points = {{-17.3,-19.0},{-25.0,-19.0},{-25.0,0.25}}, color = {0,0,191}, visible = true));
      connect(Delaying2.y,Rate2.u) annotation(Line(points = {{23.3,-19.0},{30.0,-19.0},{30.0,-5.75}}, color = {0,0,191}, visible = true));
      connect(Rate3.y,Level3.u2) annotation(Line(points = {{42.0,-50.0},{2.0,-50.0}}, color = {95,0,191}, visible = true));
      connect(Delaying3.u,Level3.y2) annotation(Line(points = {{2.7,-79.0},{-5.0,-79.0},{-5.0,-59.75}}, color = {0,0,191}, visible = true));
      connect(Delaying3.y,Rate3.u) annotation(Line(points = {{43.3,-79.0},{50.0,-79.0},{50.0,-65.75}}, color = {0,0,191}, visible = true));
      connect(Level1.u1,massInPort) annotation(Line(points = {{-82.0,70.0},{-90.0,70.0},{-90.0,0.0},{-110.0,0.0}}, color = {95,0,191}, visible = true));
      connect(Rate1.y1,Level2.u1) annotation(Line(points = {{18.0,70.0},{30.0,70.0},{30.0,28.0},{-70.0,28.0},{-70.0,10.0},{-62.0,10.0}}, color = {95,0,191}, visible = true));
      connect(Rate2.y1,Level3.u1) annotation(Line(points = {{38.0,10.0},{50.0,10.0},{50.0,-32.0},{-50.0,-32.0},{-50.0,-50.0},{-42.0,-50.0}}, color = {95,0,191}, visible = true));
      connect(Rate3.y1,massOutPort) annotation(Line(points = {{58.0,-50.0},{90.0,-50.0},{90.0,0.0},{110.0,0.0}}, color = {95,0,191}, visible = true));
      connect(Add3.u1,Level1.y1) annotation(Line(points = {{62.0,58.0},{62.0,50.0},{50.0,50.0},{50.0,84.0},{-30.0,84.0},{-30.0,78.25},{-43.0,78.25}}, color = {0,0,191}, visible = true));
      connect(Add3.u2,Level2.y1) annotation(Line(points = {{70.0,58.0},{70.0,24.0},{-10.0,24.0},{-10.0,18.25},{-23.0,18.25}}, color = {0,0,191}, visible = true));
      connect(Add3.u3,Level3.y1) annotation(Line(points = {{78.0,58.0},{78.0,-36.0},{10.0,-36.0},{10.0,-41.75},{-3.0,-41.75}}, color = {0,0,191}, visible = true));
      connect(Add3.y,state) annotation(Line(points = {{70.0,81.0},{70.0,90.0},{0.0,90.0},{0.0,110.0}}, color = {0,0,191}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,191}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, extent = {{-100.0,-100.0},{100.0,100.0}}),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,191}, extent = {{-64.0,-44.0},{68.0,42.0}}, textString = "DELAY3", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Documentation(info = "<html>
This is a third-order exponential delaying function.  It describes the delaying of mass flows. <p>
 
Incoming goods are delayed temporarily in three consecutive internal states.  The <b>state</b> output represents the amount of goods currently stored in the DELAY block. <p>
 
The initial number of items in the DELAY block can be set.  By default, it assumes a value of zero.
</html>"));
    end DELAY3;
    block Linear "Linear function"
      parameter Real m = 1 "Gradient";
      parameter Real b = 0 "Offset";
      Modelica.Blocks.Interfaces.RealInput u "Input variable" annotation(Placement(visible = true, transformation(origin = {-90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
    equation
      y = m * u + b;
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, lineThickness = 0.5, extent = {{-80.0,-80.0},{80.0,80.0}}),Line(visible = true, points = {{-40.0,-40.0},{-40.0,60.0},{-44.0,48.0},{-36.0,48.0},{-40.0,58.0}}, color = {0,0,255}),Line(visible = true, points = {{-40.0,-40.0},{60.0,-40.0},{48.0,-34.0},{48.0,-44.0},{60.0,-40.0}}, color = {0,0,255}),Text(visible = true, lineColor = {0,0,255}, extent = {{-64.0,38.0},{-46.0,62.0}}, textString = "y", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{40.0,-70.0},{62.0,-50.0}}, textString = "u", fontName = "Arial"),Line(visible = true, points = {{-40.0,-10.0},{48.0,30.0}}, color = {0,0,255})}), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, lineThickness = 0.5, extent = {{-80.0,-80.0},{80.0,80.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-98.0,-118.0},{100.0,-80.0}}, textString = "%name", fontName = "Arial"),Line(visible = true, points = {{-40.0,-40.0},{-40.0,60.0},{-44.0,48.0},{-36.0,48.0},{-40.0,58.0}}, color = {0,0,255}),Line(visible = true, points = {{-40.0,-40.0},{60.0,-40.0},{48.0,-34.0},{48.0,-44.0},{60.0,-40.0}}, color = {0,0,255}),Text(visible = true, lineColor = {0,0,255}, extent = {{-64.0,38.0},{-46.0,62.0}}, textString = "y", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{40.0,-70.0},{62.0,-50.0}}, textString = "u", fontName = "Arial"),Line(visible = true, points = {{-40.0,-10.0},{48.0,30.0}}, color = {0,0,255})}), Documentation(info = "<html>
This is a linear function, as it is frequently used in System Dynamics to represent linear regression models.
</html>"));
    end Linear;
    block SMTH1 "First-order exponential smoothing"
      parameter Real averaging_time = 1 "Averaging time";
      parameter Real smooth_init = 0 "Initial value of smoothed variable";
      SystemDynamics.Levels.Level1a Smooth_of_Input(x0 = smooth_init) annotation(Placement(visible = true, transformation(origin = {50.0,0.0}, extent = {{-16.0,-14.0},{16.0,14.0}}, rotation = 0)));
      SystemDynamics.Rates.Rate_1 Change_in_Smooth annotation(Placement(visible = true, transformation(origin = {0.0,0.0}, extent = {{-16.0,-14.0},{16.0,14.0}}, rotation = 0)));
      SystemDynamics.Sources.Source source annotation(Placement(visible = true, transformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Functions.Utilities.Change_In_Smooth change_in_smooth(averaging_time = averaging_time) annotation(Placement(visible = true, transformation(origin = {-0.0,-40.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)), Dialog(group = "Variables"));
      Modelica.Blocks.Interfaces.RealInput u "Input variable" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
    equation
      connect(Change_in_Smooth.y1,Smooth_of_Input.u1) annotation(Line(points = {{8.0,0.0},{20.0,-0.0},{20.0,0.0},{32.4,0.0}}, color = {95,0,191}, visible = true));
      connect(source.MassInPort1,Change_in_Smooth.y) annotation(Line(points = {{-39.0,0.0},{-23.5,0.0},{-23.5,0.0},{-8.0,0.0}}, color = {95,0,191}, visible = true));
      connect(change_in_smooth.y,Change_in_Smooth.u) annotation(Line(points = {{-0.0,-26.5},{0.0,-18.8},{0.0,-18.8},{0.0,-14.7}}, color = {0,0,191}, visible = true));
      connect(change_in_smooth.u2,Smooth_of_Input.y4) annotation(Line(points = {{6.0,-53.5},{6.0,-60.0},{38.0,-60.0},{38.0,-9.1}}, color = {0,0,191}, visible = true));
      connect(change_in_smooth.u1,u) annotation(Line(points = {{-6.0,-53.5},{-6.0,-60.0},{-80.0,-60.0},{-80.0,0.0},{-110.0,0.0}}, color = {0,0,191}, visible = true));
      connect(y,Smooth_of_Input.y1) annotation(Line(points = {{110.0,0.0},{80.0,0.0},{80.0,7.7},{63.6,7.7}}, color = {0,0,191}, visible = true));
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,191}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-100.0,-100.0},{100.0,100.0}}),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,191}, extent = {{-64.0,-44.0},{68.0,42.0}}, textString = "SMTH1", fontName = "Arial")}), Documentation(info = "<html>
This is a first-order exponential smoothing function.  It describes the smoothing of information flows. <p>
 
In engineering terms, this would be called a first-order low-pass filter with the transfer function: <p>
<font color=red><b>G(s) = 1/(tau*s + 1)</b></font> <p>
and the time constant <p>
<font color=red><b>tau = averaging_time</b></font>. <p>
 
The initial value of the output can be set.  By default, it assumes a value of zero.
</html>"));
    end SMTH1;
    block SMTH3 "Third-order exponential smoothing"
      parameter Real averaging_time = 1 "Averaging time";
      parameter Real smooth_init = 0 "Initial value of smoothed variable";
      SystemDynamics.Levels.Level1a Smooth1(x0 = smooth_init) annotation(Placement(visible = true, transformation(origin = {10.0,80.0}, extent = {{-16.0,-14.0},{16.0,14.0}}, rotation = 0)));
      SystemDynamics.Rates.Rate_1 Change_in_Smooth1 annotation(Placement(visible = true, transformation(origin = {-40.0,80.0}, extent = {{-16.0,-14.0},{16.0,14.0}}, rotation = 0)));
      SystemDynamics.Sources.Source source annotation(Placement(visible = true, transformation(origin = {-90.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Functions.Utilities.Change_In_Smooth ch_in_smooth1(averaging_time = averaging_time / 3) annotation(Placement(visible = true, transformation(origin = {-40.0,50.0}, extent = {{-10.0,-12.0},{10.0,12.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u "Input variable" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Levels.Level1a Smooth2(x0 = smooth_init) annotation(Placement(visible = true, transformation(origin = {40.0,10.0}, extent = {{-16.0,-14.0},{16.0,14.0}}, rotation = 0)));
      SystemDynamics.Rates.Rate_1 Change_in_Smooth2 annotation(Placement(visible = true, transformation(origin = {-10.0,10.0}, extent = {{-16.0,-14.0},{16.0,14.0}}, rotation = 0)));
      SystemDynamics.Sources.Source source1 annotation(Placement(visible = true, transformation(origin = {-60.0,10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Functions.Utilities.Change_In_Smooth ch_in_smooth2(averaging_time = averaging_time / 3) annotation(Placement(visible = true, transformation(origin = {-10.0,-20.0}, extent = {{-10.0,-12.0},{10.0,12.0}}, rotation = -270)));
      SystemDynamics.Levels.Level1a Smooth3(x0 = smooth_init) annotation(Placement(visible = true, transformation(origin = {70.0,-54.0}, extent = {{-16.0,-14.0},{16.0,14.0}}, rotation = 0)));
      SystemDynamics.Rates.Rate_1 Change_in_Smooth3 annotation(Placement(visible = true, transformation(origin = {20.0,-54.0}, extent = {{-16.0,-14.0},{16.0,14.0}}, rotation = 0)));
      SystemDynamics.Sources.Source source2 annotation(Placement(visible = true, transformation(origin = {-30.0,-54.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Functions.Utilities.Change_In_Smooth ch_in_smooth3(averaging_time = averaging_time / 3) annotation(Placement(visible = true, transformation(origin = {20.0,-84.0}, extent = {{-10.0,-12.0},{10.0,12.0}}, rotation = -270)));
    equation
      connect(Change_in_Smooth1.y1,Smooth1.u1) annotation(Line(points = {{-32.0,80.0},{-7.6,80.0}}, color = {95,0,191}, visible = true));
      connect(source.MassInPort1,Change_in_Smooth1.y) annotation(Line(points = {{-79.0,80.0},{-48.0,80.0}}, color = {95,0,191}, visible = true));
      connect(ch_in_smooth1.y,Change_in_Smooth1.u) annotation(Line(points = {{-40.0,59.0},{-40.0,65.3}}, color = {0,0,191}, visible = true));
      connect(ch_in_smooth1.u2,Smooth1.y4) annotation(Line(points = {{-35.2,41.0},{-35.2,34.0},{-2.0,34.0},{-2.0,70.9}}, color = {0,0,191}, visible = true));
      connect(Change_in_Smooth2.y1,Smooth2.u1) annotation(Line(points = {{-2.0,10.0},{4.1,10.0},{4.1,10.0},{10.2,10.0},{10.2,10.0},{22.4,10.0}}, color = {95,0,191}, visible = true));
      connect(source1.MassInPort1,Change_in_Smooth2.y) annotation(Line(points = {{-49.0,10.0},{-33.5,10.0},{-33.5,10.0},{-18.0,10.0}}, color = {95,0,191}, visible = true));
      connect(ch_in_smooth2.y,Change_in_Smooth2.u) annotation(Line(points = {{-10.0,-11.0},{-10.0,-4.7}}, color = {0,0,191}, visible = true));
      connect(ch_in_smooth2.u2,Smooth2.y4) annotation(Line(points = {{-5.2,-29.0},{-5.2,-36.0},{28.0,-36.0},{28.0,0.9}}, color = {0,0,191}, visible = true));
      connect(Change_in_Smooth3.y1,Smooth3.u1) annotation(Line(points = {{28.0,-54.0},{52.4,-54.0}}, color = {95,0,191}, visible = true));
      connect(source2.MassInPort1,Change_in_Smooth3.y) annotation(Line(points = {{-19.0,-54.0},{12.0,-54.0}}, color = {95,0,191}, visible = true));
      connect(ch_in_smooth3.y,Change_in_Smooth3.u) annotation(Line(points = {{20.0,-75.0},{20.0,-68.7}}, color = {0,0,191}, visible = true));
      connect(ch_in_smooth3.u2,Smooth3.y4) annotation(Line(points = {{24.8,-93.0},{24.8,-100.0},{58.0,-100.0},{58.0,-63.1}}, color = {0,0,191}, visible = true));
      connect(u,ch_in_smooth1.u1) annotation(Line(points = {{-110.0,0.0},{-90.0,0.0},{-90.0,34.0},{-44.8,34.0},{-44.8,41.0}}, color = {0,0,191}, visible = true));
      connect(ch_in_smooth2.u1,Smooth1.y3) annotation(Line(points = {{-14.8,-29.0},{-14.8,-36.0},{-40.0,-36.0},{-40.0,28.0},{10.0,28.0},{10.0,70.9}}, color = {0,0,191}, visible = true));
      connect(ch_in_smooth3.u1,Smooth2.y3) annotation(Line(points = {{15.2,-93.0},{15.2,-100.0},{-10.0,-100.0},{-10.0,-40.0},{40.0,-40.0},{40.0,0.9}}, color = {0,0,191}, visible = true));
      connect(y,Smooth3.y1) annotation(Line(points = {{110.0,0.0},{90.0,0.0},{90.0,-46.3},{83.6,-46.3}}, color = {0,0,191}, visible = true));
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,191}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-100.0,-100.0},{100.0,100.0}}),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,191}, extent = {{-64.0,-44.0},{68.0,42.0}}, textString = "SMTH3", fontName = "Arial")}), Documentation(info = "<html>
This is a third-order exponential smoothing function. <p>
 
In engineering terms, this would be called a third-order low-pass filter consisting of three first-order low-pass filters placed in series.  The third-order filter is characterized by the transfer function: <p>
<font color=red><b>G(s) = (1/(tau*s + 1))^3</b></font>. <p>
Each first-order filter has a time constants of: <p>
<font color=red><b>tau = averaging_time/3</b></font>. <p>
 
The initial values of the outputs of each of the three first-order filters can be set.  However, they can only be set together, i.e., only one parameter is provided for all three of them.  By default, they assume values of zero.
</html>"));
    end SMTH3;
    block Tabular "Tabular function"
      parameter Real x_vals[:] = {0} "Independent variable data points";
      parameter Real y_vals[:] = {0} "Dependent variable data points";
      Modelica.Blocks.Interfaces.RealInput u "Input variable" annotation(Placement(visible = true, transformation(origin = {-80.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-80.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealOutput y "Output variable" annotation(Placement(visible = true, transformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
    equation
      // New: (pass time as time_)
      y = Functions.Utilities.Piecewise(x = u, x_grid = x_vals, y_grid = y_vals, time_=time);
      // New ^
      // ORIGINAL:
      //y = Functions.Utilities.Piecewise(x = u, x_grid = x_vals, y_grid = y_vals);
      // ORIGINAL ^
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Line(visible = true, points = {{-70.0,0.0},{-44.0,0.0}}, color = {160,160,160}),Line(visible = true, points = {{28.0,0.0},{100.0,0.0}}, color = {160,160,160}),Text(visible = true, lineColor = {0,0,255}, fillColor = {160,160,160}, extent = {{-72.0,2.0},{-46.0,20.0}}, textString = "u", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {160,160,160}, extent = {{52.0,2.0},{78.0,20.0}}, textString = "y", fontName = "Arial"),Rectangle(visible = true, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-44.0,40.0},{-8.0,60.0}}),Rectangle(visible = true, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-44.0,20.0},{-8.0,40.0}}),Rectangle(visible = true, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-44.0,-2.0},{-8.0,20.0}}),Rectangle(visible = true, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-44.0,-22.0},{-8.0,0.0}}),Rectangle(visible = true, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-44.0,-42.0},{-8.0,-20.0}}),Rectangle(visible = true, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-44.0,-60.0},{-8.0,-40.0}}),Rectangle(visible = true, fillColor = {255,255,255}, fillPattern = FillPattern.Solid, extent = {{-8.0,40.0},{28.0,60.0}}),Rectangle(visible = true, fillColor = {255,255,255}, fillPattern = FillPattern.Solid, extent = {{-8.0,20.0},{28.0,40.0}}),Rectangle(visible = true, fillColor = {255,255,255}, fillPattern = FillPattern.Solid, extent = {{-8.0,-2.0},{28.0,20.0}}),Rectangle(visible = true, fillColor = {255,255,255}, fillPattern = FillPattern.Solid, extent = {{-8.0,-20.0},{28.0,0.0}}),Rectangle(visible = true, fillColor = {255,255,255}, fillPattern = FillPattern.Solid, extent = {{-8.0,-40.0},{28.0,-20.0}}),Rectangle(visible = true, fillColor = {255,255,255}, fillPattern = FillPattern.Solid, extent = {{-8.0,-60.0},{28.0,-40.0}})}), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-44.0,40.0},{-8.0,60.0}}),Rectangle(visible = true, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-44.0,20.0},{-8.0,40.0}}),Rectangle(visible = true, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-44.0,-2.0},{-8.0,20.0}}),Rectangle(visible = true, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-44.0,-22.0},{-8.0,0.0}}),Rectangle(visible = true, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-44.0,-42.0},{-8.0,-20.0}}),Rectangle(visible = true, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-44.0,-60.0},{-8.0,-40.0}}),Rectangle(visible = true, fillColor = {255,255,255}, fillPattern = FillPattern.Solid, extent = {{-8.0,40.0},{28.0,60.0}}),Rectangle(visible = true, fillColor = {255,255,255}, fillPattern = FillPattern.Solid, extent = {{-8.0,20.0},{28.0,40.0}}),Rectangle(visible = true, fillColor = {255,255,255}, fillPattern = FillPattern.Solid, extent = {{-8.0,-2.0},{28.0,20.0}}),Rectangle(visible = true, fillColor = {255,255,255}, fillPattern = FillPattern.Solid, extent = {{-8.0,-20.0},{28.0,0.0}}),Rectangle(visible = true, fillColor = {255,255,255}, fillPattern = FillPattern.Solid, extent = {{-8.0,-40.0},{28.0,-20.0}}),Rectangle(visible = true, fillColor = {255,255,255}, fillPattern = FillPattern.Solid, extent = {{-8.0,-60.0},{28.0,-40.0}}),Line(visible = true, points = {{-70.0,80.0},{-70.0,-80.0},{40.0,-80.0},{100.0,0.0},{40.0,80.0},{-70.0,80.0}}, color = {0,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,80.0},{100.0,122.0}}, textString = "%name", fontName = "Arial")}), Documentation(info = "<html>
This is a tabular function, as it is frequently used in System Dynamics to represent measured or estimated non-linear relationships.
</html>"));
    end Tabular;
    package Utilities "Utility modules of the set of functions"
      extends Modelica.Icons.Library annotation(preferedView = "info", Documentation(info = "<html>
Utility models of the set of functions.
</html>"));
      block Change_In_Smooth "Smoothing rate"
        extends Interfaces.Nonlin_2;
        parameter Real averaging_time = 1 "Averaging time";
        output Real input_variable "Variable to be smoothed";
        output Real smooth_of_input "Smoothed variable";
        output Real smoothing_rate "Smoothing rate";
      equation
        input_variable = u1;
        smooth_of_input = u2;
        smoothing_rate = (input_variable - smooth_of_input) / averaging_time;
        y = smoothing_rate;
        annotation(Diagram);
      end Change_In_Smooth;
      function InitRandomNormal
        extends Modelica.Icons.Function;

        external "C" ext_initRandomNormal()         annotation(Include = "#include <ext_initRandomNormal.c>"); 
        annotation(Diagram(coordinateSystem(extent = {{-148.5,-105.0},{148.5,105.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {5,5})));
      end InitRandomNormal;
      function RandomNormal
        extends Modelica.Icons.Function;
        output Real y;
        input Real u;

        external "C" y = ext_RandomNormal(u)         annotation(Include = "#include <ext_RandomNormal.c>");
        annotation(Diagram(coordinateSystem(extent = {{-148.5,-105.0},{148.5,105.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {5,5})));
      end RandomNormal;
      function Piecewise "Piecewise linear function"
        input Real x "Independent variable";
        input Real x_grid[:] "Independent variable data points";
        input Real y_grid[:] "Dependent variable data points";
        //New: time_ is an input so we can use it inside
        input Real time_;
        output Real y "Interpolated result";
        //Nuevo: para no repetir tanto codigo:
        protected String info_debug_base;
      protected
        Integer n;
      algorithm
        n:=size(x_grid, 1);
        // Adaptation to allow out of range values
        assert(size(x_grid, 1) == size(y_grid, 1), "Size mismatch");
        //assert(x >= x_grid[1] and x <= x_grid[n], "Out of range.");
        assert(x >= x_grid[1] and x <= x_grid[n], "Out of range. Accuracy low.  time:"+String(time_)+" x: "+String(x)+" x_grid[1]: "+String(x_grid[1])+" x_grid[n]: "+String(x_grid[n]), AssertionLevel.warning);
        info_debug_base := " time:"+String(time_)+" x: "+String(x)+" x_grid[1]: "+String(x_grid[1])+" x_grid[n]: "+String(x_grid[n]);
        if x<x_grid[1] then
            //y:=y_grid[1]; // Uses constant value for values outside of range
            y := y_grid[1] + (y_grid[2] - y_grid[1])  *((x - x_grid[1]) / (x_grid[2] - x_grid[1]));   // Extrapolates backwards using the first interval
            // Uncomment the following line to log to file (SLOWS DOWN EXECUTION CONSIDERABLY!)
            //Modelica.Utilities.Streams.print("x<x_grid[1] " +info_debug_base + " y: " + String(y) , "out_of_range_cases.txt");
        elseif x>x_grid[n] then
            //y:=y_grid[n]; // Uses constant value for values outside of range
            y := y_grid[n] + (y_grid[n] - y_grid[n-1])*((x - x_grid[n]) / (x_grid[n] - x_grid[n-1])); // Extrapolates forward using the last interval
            // Uncomment the following line to log to file (SLOWS DOWN EXECUTION CONSIDERABLY!)
            //Modelica.Utilities.Streams.print("x>x_grid[n] " +info_debug_base + " y: " + String(y) , "out_of_range_cases.txt");
        else
            for i in 1:n - 1 loop
              if x >= x_grid[i] and x <= x_grid[i + 1] then 
                y:=y_grid[i] + (y_grid[i + 1] - y_grid[i]) * (x - x_grid[i]) / (x_grid[i + 1] - x_grid[i]);
              else
              end if;
            end for;
        end if;

        // ORIGINAL:
        //assert(size(x_grid, 1) == size(y_grid, 1), "Size mismatch");
        //assert(x >= x_grid[1] and x <= x_grid[n], "Out of range");
        //for i in 1:n - 1 loop
        //          if x >= x_grid[i] and x <= x_grid[i + 1] then 
        //    y:=y_grid[i] + (y_grid[i + 1] - y_grid[i]) * (x - x_grid[i]) / (x_grid[i + 1] - x_grid[i]);
        //  else

        //  end if;
        //end for;
        annotation(Diagram(coordinateSystem(extent = {{-100,-100},{100,100}}), graphics = {Rectangle(extent = {{-44,60},{-8,40}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,0}),Rectangle(extent = {{-44,40},{-8,20}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,0}),Rectangle(extent = {{-44,20},{-8,-2}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,0}),Rectangle(extent = {{-44,0},{-8,-22}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,0}),Rectangle(extent = {{-44,-20},{-8,-42}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,0}),Rectangle(extent = {{-44,-40},{-8,-60}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,0}),Rectangle(extent = {{-8,60},{28,40}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,255}),Rectangle(extent = {{-8,40},{28,20}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,255}),Rectangle(extent = {{-8,20},{28,-2}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,255}),Rectangle(extent = {{-8,0},{28,-20}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,255}),Rectangle(extent = {{-8,-20},{28,-40}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,255}),Rectangle(extent = {{-8,-40},{28,-60}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,255}),Text(lineColor = {0,0,255}, extent = {{-96,94},{98,70}}, textString = "Linear interpolation")}), Icon(coordinateSystem(extent = {{-100,-100},{100,100}}), graphics = {Rectangle(extent = {{-44,60},{-8,40}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,0}),Rectangle(extent = {{-44,40},{-8,20}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,0}),Rectangle(extent = {{-44,20},{-8,-2}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,0}),Rectangle(extent = {{-44,0},{-8,-22}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,0}),Rectangle(extent = {{-44,-20},{-8,-42}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,0}),Rectangle(extent = {{-44,-40},{-8,-60}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,0}),Rectangle(extent = {{-8,60},{28,40}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,255}),Rectangle(extent = {{-8,40},{28,20}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,255}),Rectangle(extent = {{-8,20},{28,-2}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,255}),Rectangle(extent = {{-8,0},{28,-20}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,255}),Rectangle(extent = {{-8,-20},{28,-40}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,255}),Rectangle(extent = {{-8,-40},{28,-60}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, fillColor = {255,255,255})}));
      end Piecewise;
    end Utilities;
    block NoiseNormal "Normally distributed random noise"
      parameter Real mean = 0 "Mean value of random noise";
      parameter Real stdev = 100 "Standard deviation of random noise";
      parameter Real tSample = 1 "Noise sample time";
      Modelica.Blocks.Interfaces.RealOutput y annotation(Placement(visible = true, transformation(origin = {90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {90.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
    equation
      when initial() then
              SystemDynamics.Functions.Utilities.InitRandomNormal();
      
      end when;
      when sample(0, tSample) then
              y = mean + stdev * SystemDynamics.Functions.Utilities.RandomNormal(time);
      
      end when;
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10})), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-80.0},{80.0,80.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-0.0,-124.0},{0.0,-80.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-6.0,36.0},{-6.0,68.0}}, textString = "mean=%mean", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-4.0,-6.0},{-4.0,24.0}}, textString = "stdev=%stdev", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-2.0,-54.0},{-2.0,-24.0}}, textString = "tSample=%tSample", fontName = "Arial")}));
    end NoiseNormal;
  end Functions;
  package Levels "Levels of the System Dynamics methodology"
    extends Modelica.Icons.Library annotation(preferedView = "info", Documentation(info = "<html>
This package contains a set of different <b>Levels</b> (integrators of state variables) frequently used in the System Dynamics methodology.
</html>"));
    block Level "General System Dynamics level"
      parameter Real x0 = 0 "Initial condition";
      output Real level "Continuous state variable";
      Modelica.Blocks.Math.Add Add1(k2 = -1) annotation(Placement(visible = true, transformation(origin = {-40.0,0.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
      Modelica.Blocks.Continuous.Integrator Integrator1(y_start = x0) annotation(Placement(visible = true, transformation(origin = {20.0,0.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u1 "Inflow variable" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u2 "Outflow variable" annotation(Placement(visible = true, transformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y "State variable" annotation(Placement(visible = true, transformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0), iconTransformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "State variable" annotation(Placement(visible = true, transformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y2 "State variable" annotation(Placement(visible = true, transformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y3 "State variable" annotation(Placement(visible = true, transformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y4 "State variable" annotation(Placement(visible = true, transformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
    equation
      level = y;
      connect(Add1.y,Integrator1.u) annotation(Line(points = {{-18.0,0.0},{-4.0,0.0}}, color = {0,0,255}, visible = true));
      connect(y,Integrator1.y) annotation(Line(points = {{-85.0,55.0},{60.0,55.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(u1,Add1.u1) annotation(Line(points = {{-110.0,0.0},{-80.0,0.0},{-80.0,12.0},{-64.0,12.0}}, color = {95,0,191}, visible = true));
      connect(u2,Add1.u2) annotation(Line(points = {{110.0,0.0},{80.0,0.0},{80.0,-40.0},{-80.0,-40.0},{-80.0,-12.0},{-64.0,-12.0}}, color = {95,0,191}, visible = true));
      connect(y1,Integrator1.y) annotation(Line(points = {{85.0,55.0},{60.0,55.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y2,Integrator1.y) annotation(Line(points = {{75.0,-65.0},{75.0,-50.0},{60.0,-50.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y3,Integrator1.y) annotation(Line(points = {{0.0,-65.0},{0.0,-50.0},{60.0,-50.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y4,Integrator1.y) annotation(Line(points = {{-75.0,-65.0},{-75.0,-50.0},{60.0,-50.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-60.0},{80.0,20.0}}),Rectangle(visible = true, lineColor = {127,0,255}, lineThickness = 0.5, extent = {{-80.0,-60.0},{80.0,60.0}}),Line(visible = true, points = {{-100.0,0.0},{-80.0,0.0}}, color = {127,0,255}, thickness = 0.5),Line(visible = true, points = {{80.0,0.0},{100.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,70.0},{100.0,100.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-6.0},{80.0,20.0}}, textString = "x0 = %x0", fontName = "Arial")}), Documentation(info = "<html>
This is the general continuous <b>Level</b> model of the System Dynamics methodology with a single inflow and a single outflow.  It computes the level by integrating over the difference between inflow and outflow rates.
</html>"), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics));
    end Level;
    block CLevel "General System Dynamics controlling level"
      constant Real inf = Modelica.Constants.inf;
      parameter Real x0 = 0 "Initial condition";
      parameter Real minLevel = 0 "Smallest allowed value of stock";
      parameter Real maxLevel = inf "Largest allowed value of stock";
      output Real level "Continuous state variable";
      Modelica.Blocks.Continuous.Integrator Integrator1(y_start = x0) annotation(Placement(visible = true, transformation(origin = {-30.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u1 "Inflow variable" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u2 "Outflow variable" annotation(Placement(visible = true, transformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y "State variable" annotation(Placement(visible = true, transformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0), iconTransformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "State variable" annotation(Placement(visible = true, transformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y2 "State variable" annotation(Placement(visible = true, transformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      Modelica.Blocks.Logical.LessThreshold LessThreshold1(threshold = maxLevel) annotation(Placement(visible = true, transformation(origin = {-50.0,-55.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.BooleanOutput s1 "Inflow control signal" annotation(Placement(visible = true, transformation(origin = {-85.0,-55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0), iconTransformation(origin = {-85.0,-55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.BooleanOutput s2 "Outflow control signal" annotation(Placement(visible = true, transformation(origin = {85.0,-55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {85.0,-55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      Modelica.Blocks.Math.Add Add1(k1 = -1) annotation(Placement(visible = true, transformation(origin = {-70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Logical.GreaterThreshold GreaterThreshold1(threshold = minLevel) annotation(Placement(visible = true, transformation(origin = {50.0,-55.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
    equation
      level = y;
      connect(y1,Integrator1.y) annotation(Line(points = {{85.0,55.0},{0.0,55.0},{0.0,0.0},{-19.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y1,y) annotation(Line(points = {{85.0,55.0},{12.5,55.0},{12.5,55.0},{-85.0,55.0}}, color = {95,0,191}, visible = true));
      connect(LessThreshold1.y,s1) annotation(Line(points = {{-61.0,-55.0},{-85.0,-55.0}}, color = {255,0,255}, visible = true));
      connect(Add1.y,Integrator1.u) annotation(Line(points = {{-59.0,0.0},{-42.0,0.0}}, color = {0,0,191}, visible = true));
      connect(u1,Add1.u2) annotation(Line(points = {{-110.0,0.0},{-90.0,0.0},{-90.0,-6.0},{-82.0,-6.0}}, color = {95,0,191}, visible = true));
      connect(u2,Add1.u1) annotation(Line(points = {{110.0,0.0},{90.0,0.0},{90.0,70.0},{-90.0,70.0},{-90.0,6.0},{-82.0,6.0}}, color = {95,0,191}, visible = true));
      connect(GreaterThreshold1.y,s2) annotation(Line(points = {{61.0,-55.0},{69.5,-55.0},{69.5,-55.0},{85.0,-55.0}}, color = {255,0,255}, visible = true));
      connect(LessThreshold1.u,Integrator1.y) annotation(Line(points = {{-38.0,-55.0},{-10.0,-55.0},{-10.0,0.0},{-19.0,0.0}}, color = {0,0,191}, visible = true));
      connect(GreaterThreshold1.u,Integrator1.y) annotation(Line(points = {{38.0,-55.0},{10.0,-55.0},{10.0,0.0},{-19.0,0.0}}, color = {0,0,191}, visible = true));
      connect(y2,Integrator1.y) annotation(Line(points = {{6.3333,-43.3333},{6.3333,21.6667},{-12.6667,21.6667}}, color = {95,0,191}, visible = true, origin = {-6.3333,-21.6667}));
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-60.0},{80.0,20.0}}),Rectangle(visible = true, lineColor = {127,0,255}, lineThickness = 0.5, extent = {{-80.0,-60.0},{80.0,60.0}}),Line(visible = true, points = {{-100.0,0.0},{-80.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,70.0},{100.0,100.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-6.0},{80.0,20.0}}, textString = "x0 = %x0", fontName = "Arial"),Line(visible = true, points = {{80.0,0.0},{100.0,0.0}}, color = {127,0,255}, thickness = 0.5)}), Documentation(info = "<html>
This is the continuous controlling <b>Level</b> model of the System Dynamics methodology with a single inflow and a single outflow.  It computes the level by integrating over the difference between inflow and outflow rates. <p>
 
Once the level reaches its maximum allowed value, it sends a Boolean signal to its inflow rate element to tell it to stop sending more stuff.  Once it reaches its minimum allowed value, it sends a Boolean signal to its outflow rate element to tell it to stop demanding more stuff.
</html>"));
    end CLevel;
    block DiscreteLevel "Discrete level of the System Dynamics methodology"
      parameter Real x0 = 0 "Initial condition";
      parameter Real h = 1.0 "Sampling rate";
      output Real level(start = x0) "Discrete state variable";
      output Real rate_in "Inflow";
      output Real rate_out "Outflow";
      SystemDynamics.Interfaces.MassInPort u1 "Inflow variable" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u2 "Outflow variable" annotation(Placement(visible = true, transformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "Discrete state variable" annotation(Placement(visible = true, transformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0), iconTransformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y2 "Discrete state variable" annotation(Placement(visible = true, transformation(origin = {-75.0,-65.0}, extent = {{5.0,5.0},{-5.0,-5.0}}, rotation = -270), iconTransformation(origin = {-75.0,-65.0}, extent = {{5.0,5.0},{-5.0,-5.0}}, rotation = -270)));
      SystemDynamics.Interfaces.MassOutPort y3 "Discrete state variable" annotation(Placement(visible = true, transformation(origin = {0.0,-65.0}, extent = {{5.0,5.0},{-5.0,-5.0}}, rotation = -270), iconTransformation(origin = {0.0,-65.0}, extent = {{5.0,5.0},{-5.0,-5.0}}, rotation = -270)));
      SystemDynamics.Interfaces.MassOutPort y4 "Discrete state variable" annotation(Placement(visible = true, transformation(origin = {75.0,-65.0}, extent = {{5.0,5.0},{-5.0,-5.0}}, rotation = -270), iconTransformation(origin = {75.0,-65.0}, extent = {{5.0,5.0},{-5.0,-5.0}}, rotation = -270)));
      SystemDynamics.Interfaces.MassOutPort y5 "Discrete state variable" annotation(Placement(visible = true, transformation(origin = {85.0,55.0}, extent = {{5.0,5.0},{-5.0,-5.0}}, rotation = -180), iconTransformation(origin = {85.0,55.0}, extent = {{5.0,5.0},{-5.0,-5.0}}, rotation = -180)));
    equation
      rate_in = u1;
      rate_out = u2;
      y1 = pre(level);
      y2 = pre(level);
      y3 = pre(level);
      y4 = pre(level);
      y5 = pre(level);
      when sample(1, 1) then
              level = pre(level) + h * (rate_in - rate_out);
      
      end when;
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {0,255,127}, fillPattern = FillPattern.Solid, extent = {{-80.0,-60.0},{80.0,20.0}}),Rectangle(visible = true, lineColor = {127,0,255}, lineThickness = 0.5, extent = {{-80.0,-60.0},{80.0,60.0}}),Line(visible = true, points = {{-100.0,0.0},{-80.0,0.0}}, color = {95,0,191}, thickness = 0.5),Line(visible = true, points = {{80.0,0.0},{110.0,0.0}}, color = {95,0,191}, thickness = 0.5)}), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {0,255,127}, fillPattern = FillPattern.Solid, extent = {{-80.0,-60.0},{80.0,20.0}}),Rectangle(visible = true, lineColor = {127,0,255}, lineThickness = 0.5, extent = {{-80.0,-60.0},{80.0,60.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-6.0},{80.0,20.0}}, textString = "x0 = %x0", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,70.0},{100.0,100.0}}, textString = "%name", fontName = "Arial"),Line(visible = true, points = {{-100.0,0.0},{-80.0,0.0}}, color = {127,0,255}, thickness = 0.5),Line(visible = true, points = {{80.0,0.0},{100.0,0.0}}, color = {127,0,255}, thickness = 0.5)}), Documentation(info = "<html>
This is the general discrete <b>Level</b> model of the System Dynamics methodology.  It computes the level by a Forward Euler algorithm, i.e., by solving a difference equation with a fixed step size of 1 time unit.  The classical System Dynamics methodology operates on discrete levels only.
</html>"));
    end DiscreteLevel;
    block Reverse_Level "Level of the System Dynamics methodology with time reversal"
      parameter Real x0 = 0 "Initial condition";
      parameter Real xm = 0.0 "Minimum level";
      output Real level(start = x0) "Reversible state variable";
      output Real rate_in "Inflow";
      output Real rate_out "Outflow";
      input Real dir "Direction of time flow";
      SystemDynamics.Interfaces.MassInPort u1 "Inflow variable" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u2 "Outflow variable" annotation(Placement(visible = true, transformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y "Reversible state variable" annotation(Placement(visible = true, transformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0), iconTransformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "Reversible state variable" annotation(Placement(visible = true, transformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y2 "Reversible state variable" annotation(Placement(visible = true, transformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y3 "Reversible state variable" annotation(Placement(visible = true, transformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y4 "Reversible state variable" annotation(Placement(visible = true, transformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
    equation
      rate_in = u1;
      rate_out = u2;
      y = level;
      y1 = level;
      y2 = level;
      y3 = level;
      y4 = level;
      der(level) = if level > xm then dir * (rate_in - rate_out) else 0;
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,127,0}, fillPattern = FillPattern.Solid, extent = {{-80.0,-60.0},{80.0,20.0}}),Rectangle(visible = true, lineColor = {127,0,255}, lineThickness = 0.5, extent = {{-80.0,-60.0},{80.0,60.0}}),Line(visible = true, points = {{-100.0,0.0},{-80.0,0.0}}, color = {127,0,255}, thickness = 0.5),Line(visible = true, points = {{80.0,0.0},{110.0,0.0}}, color = {127,0,255}, thickness = 0.5)}), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,127,0}, fillPattern = FillPattern.Solid, extent = {{-80.0,-60.0},{80.0,20.0}}),Rectangle(visible = true, lineColor = {127,0,255}, lineThickness = 0.5, extent = {{-80.0,-60.0},{80.0,60.0}}),Line(visible = true, points = {{-100.0,0.0},{-80.0,0.0}}, color = {127,0,255}, thickness = 0.5),Line(visible = true, points = {{80.0,0.0},{100.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,70.0},{100.0,100.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-6.0},{80.0,20.0}}, textString = "x0 = %x0", fontName = "Arial")}), Documentation(info = "<html>
This is a time-reversible continuous <b>Level</b> model of the System Dynamics methodology. <p>
 
Given the following state-space model: <p>
 
<font color=red>der</font>(<b>x</b>) = <b>f</b>(<b>x</b>,t); <p>
 
Time reversal can be achieved by placing a minus sign in front of each state equation: <p>
 
<font color=red>der</font>(<b>x</b>) = -<b>f</b>(<b>x</b>,t); <p>
 
The <i>Reverse_Level</i> model contains a modulating variable <b>dir</b> that determines the direction of time flow: <p>
 
<font color=red>der</font>(<b>x</b>) = dir*<b>f</b>(<b>x</b>,t);
</html>"));
    end Reverse_Level;
    block Level1a "General System Dynamics level with no outflow"
      parameter Real x0 = 0 "Initial condition";
      output Real level "Continuous state variable";
      Modelica.Blocks.Continuous.Integrator Integrator1(y_start = x0) annotation(Placement(visible = true, transformation(origin = {-10.0,0.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u1 "Inflow variable" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y "State variable" annotation(Placement(visible = true, transformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0), iconTransformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "State variable" annotation(Placement(visible = true, transformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y2 "State variable" annotation(Placement(visible = true, transformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y3 "State variable" annotation(Placement(visible = true, transformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y4 "State variable" annotation(Placement(visible = true, transformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
    equation
      level = y;
      connect(y1,Integrator1.y) annotation(Line(points = {{85.0,55.0},{60.0,55.0},{60.0,0.0},{12.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y2,Integrator1.y) annotation(Line(points = {{75.0,-65.0},{75.0,-50.0},{60.0,-50.0},{60.0,0.0},{12.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y3,Integrator1.y) annotation(Line(points = {{0.0,-65.0},{0.0,-50.0},{60.0,-50.0},{60.0,0.0},{12.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y4,Integrator1.y) annotation(Line(points = {{-75.0,-65.0},{-75.0,-50.0},{60.0,-50.0},{60.0,0.0},{12.0,0.0}}, color = {95,0,191}, visible = true));
      connect(u1,Integrator1.u) annotation(Line(points = {{-110.0,0.0},{-34.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y1,y) annotation(Line(points = {{85.0,55.0},{12.5,55.0},{12.5,55.0},{-85.0,55.0}}, color = {95,0,191}, visible = true));
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-60.0},{80.0,20.0}}),Rectangle(visible = true, lineColor = {127,0,255}, lineThickness = 0.5, extent = {{-80.0,-60.0},{80.0,60.0}}),Line(visible = true, points = {{-100.0,0.0},{-80.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,70.0},{100.0,100.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-6.0},{80.0,20.0}}, textString = "x0 = %x0", fontName = "Arial")}), Documentation(info = "<html>
This is the general continuous <b>Level</b> model of the System Dynamics methodology with a single inflow and no outflow.  It computes the level by integrating over the inflow rate.
</html>"));
    end Level1a;
    block CLevel1a "General System Dynamics controlling level with no outflow"
      constant Real inf = Modelica.Constants.inf;
      parameter Real x0 = 0 "Initial condition";
      parameter Real maxLevel = inf "Largest allowed value of stock";
      output Real level "Continuous state variable";
      Modelica.Blocks.Continuous.Integrator Integrator1(y_start = x0) annotation(Placement(visible = true, transformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u1 "Inflow variable" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y "State variable" annotation(Placement(visible = true, transformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0), iconTransformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "State variable" annotation(Placement(visible = true, transformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y2 "State variable" annotation(Placement(visible = true, transformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y3 "State variable" annotation(Placement(visible = true, transformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      Modelica.Blocks.Logical.LessThreshold LessThreshold1(threshold = maxLevel) annotation(Placement(visible = true, transformation(origin = {-50.0,-55.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.BooleanOutput s "Control signal" annotation(Placement(visible = true, transformation(origin = {-85.0,-55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0), iconTransformation(origin = {-85.0,-55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0)));
    equation
      level = y;
      connect(y1,Integrator1.y) annotation(Line(points = {{85.0,55.0},{60.0,55.0},{60.0,0.0},{-39.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y2,Integrator1.y) annotation(Line(points = {{75.0,-65.0},{75.0,-50.0},{60.0,-50.0},{60.0,0.0},{-39.0,0.0}}, color = {95,0,191}, visible = true));
      connect(u1,Integrator1.u) annotation(Line(points = {{-110.0,0.0},{-62.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y1,y) annotation(Line(points = {{85.0,55.0},{12.5,55.0},{12.5,55.0},{-85.0,55.0}}, color = {95,0,191}, visible = true));
      connect(LessThreshold1.y,s) annotation(Line(points = {{-61.0,-55.0},{-85.0,-55.0}}, color = {255,0,255}, visible = true));
      connect(LessThreshold1.u,Integrator1.y) annotation(Line(points = {{-38.0,-55.0},{-20.0,-55.0},{-20.0,0.0},{-39.0,0.0}}, color = {0,0,191}, visible = true));
      connect(y3,Integrator1.y) annotation(Line(points = {{0.0,-65.0},{0.0,-50.0},{60.0,-50.0},{60.0,0.0},{-39.0,0.0}}, color = {95,0,191}, visible = true));
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-60.0},{80.0,20.0}}),Rectangle(visible = true, lineColor = {127,0,255}, lineThickness = 0.5, extent = {{-80.0,-60.0},{80.0,60.0}}),Line(visible = true, points = {{-100.0,0.0},{-80.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,70.0},{100.0,100.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-6.0},{80.0,20.0}}, textString = "x0 = %x0", fontName = "Arial")}), Documentation(info = "<html>
This is the continuous controlling <b>Level</b> model of the System Dynamics methodology with a single inflow and no outflow.  It computes the level by integrating over the inflow rate. <p>
 
Once the level reaches its maximum allowed value, it sends a Boolean signal to its inflow rate element to tell it to stop sending more stuff.
</html>"));
    end CLevel1a;
    block Level1b "General System Dynamics level with no inflow"
      parameter Real x0 = 0 "Initial condition";
      output Real level "Continuous state variable";
      Modelica.Blocks.Continuous.Integrator Integrator1(y_start = x0) annotation(Placement(visible = true, transformation(origin = {20.0,0.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u1 "Outflow variable" annotation(Placement(visible = true, transformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y "State variable" annotation(Placement(visible = true, transformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0), iconTransformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "State variable" annotation(Placement(visible = true, transformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y2 "State variable" annotation(Placement(visible = true, transformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y3 "State variable" annotation(Placement(visible = true, transformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y4 "State variable" annotation(Placement(visible = true, transformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      Modelica.Blocks.Math.Gain Gain1(k = -1) annotation(Placement(visible = true, transformation(origin = {-40.0,0.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
    equation
      level = y;
      connect(y,Integrator1.y) annotation(Line(points = {{-85.0,55.0},{60.0,55.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y2,Integrator1.y) annotation(Line(points = {{75.0,-65.0},{75.0,-50.0},{60.0,-50.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y3,Integrator1.y) annotation(Line(points = {{0.0,-65.0},{0.0,-50.0},{60.0,-50.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y4,Integrator1.y) annotation(Line(points = {{-75.0,-65.0},{-75.0,-50.0},{60.0,-50.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y,y1) annotation(Line(points = {{-85.0,55.0},{-2.5,55.0},{-2.5,55.0},{85.0,55.0}}, color = {95,0,191}, visible = true));
      connect(u1,Gain1.u) annotation(Line(points = {{110.0,0.0},{90.0,0.0},{90.0,-80.0},{-90.0,-80.0},{-90.0,0.0},{-64.0,0.0}}, color = {95,0,191}, visible = true));
      connect(Gain1.y,Integrator1.u) annotation(Line(points = {{-18.0,0.0},{-4.0,0.0}}, color = {0,0,191}, visible = true));
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-60.0},{80.0,20.0}}),Rectangle(visible = true, lineColor = {127,0,255}, lineThickness = 0.5, extent = {{-80.0,-60.0},{80.0,60.0}}),Line(visible = true, points = {{80.0,0.0},{100.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,70.0},{100.0,100.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-6.0},{80.0,20.0}}, textString = "x0 = %x0", fontName = "Arial")}), Documentation(info = "<html>
This is the general continuous <b>Level</b> model of the System Dynamics methodology with a single outflow and no inflow.  It computes the level by integrating over the negative outflow rate.
</html>"));
    end Level1b;
    block CLevel1b "General System Dynamics controlling level with no inflow"
      parameter Real x0 = 0 "Initial condition";
      parameter Real minLevel = 0 "Smallest allowed value of stock";
      output Real level "Continuous state variable";
      Modelica.Blocks.Continuous.Integrator Integrator1(y_start = x0) annotation(Placement(visible = true, transformation(origin = {-30.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u1 "Outflow variable" annotation(Placement(visible = true, transformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y "State variable" annotation(Placement(visible = true, transformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0), iconTransformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "State variable" annotation(Placement(visible = true, transformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y2 "State variable" annotation(Placement(visible = true, transformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y3 "State variable" annotation(Placement(visible = true, transformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      Modelica.Blocks.Math.Gain Gain1(k = -1) annotation(Placement(visible = true, transformation(origin = {-70.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.BooleanOutput s "Control signal" annotation(Placement(visible = true, transformation(origin = {85.0,-55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {85.0,-55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      Modelica.Blocks.Logical.GreaterThreshold GreaterThreshold1(threshold = minLevel) annotation(Placement(visible = true, transformation(origin = {50.0,-55.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
    equation
      level = y;
      connect(y,Integrator1.y) annotation(Line(points = {{-85.0,55.0},{0.0,55.0},{0.0,0.0},{-19.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y2,Integrator1.y) annotation(Line(points = {{6.3333,-43.3333},{6.3333,21.6667},{-12.6667,21.6667}}, color = {95,0,191}, visible = true, origin = {-6.3333,-21.6667}));
      connect(y,y1) annotation(Line(points = {{-85.0,55.0},{-2.5,55.0},{-2.5,55.0},{85.0,55.0}}, color = {95,0,191}, visible = true));
      connect(u1,Gain1.u) annotation(Line(points = {{110.0,0.0},{90.0,0.0},{90.0,70.0},{-90.0,70.0},{-90.0,0.0},{-82.0,0.0}}, color = {95,0,191}, visible = true));
      connect(Gain1.y,Integrator1.u) annotation(Line(points = {{-59.0,0.0},{-42.0,0.0}}, color = {0,0,191}, visible = true));
      connect(GreaterThreshold1.y,s) annotation(Line(points = {{61.0,-55.0},{85.0,-55.0}}, color = {255,0,255}, visible = true));
      connect(GreaterThreshold1.u,Integrator1.y) annotation(Line(points = {{38.0,-55.0},{20.0,-55.0},{20.0,0.0},{-19.0,0.0}}, color = {0,0,191}, visible = true));
      connect(y3,Integrator1.y) annotation(Line(points = {{-75.0,-65.0},{-75.0,-50.0},{0.0,-50.0},{0.0,0.0},{-19.0,0.0}}, color = {95,0,191}, visible = true));
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-60.0},{80.0,20.0}}),Rectangle(visible = true, lineColor = {127,0,255}, lineThickness = 0.5, extent = {{-80.0,-60.0},{80.0,60.0}}),Line(visible = true, points = {{80.0,0.0},{100.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,70.0},{100.0,100.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-6.0},{80.0,20.0}}, textString = "x0 = %x0", fontName = "Arial")}), Documentation(info = "<html>
This is the continuous controlling <b>Level</b> model of the System Dynamics methodology with a single outflow and no inflow.  It computes the level by integrating over the negative outflow rate. <p>
 
Once the level reaches its minimum allowed value, it sends a Boolean signal to its outflow rate element to tell it to stop demanding more stuff.
</html>"));
    end CLevel1b;
    block Level2a "General System Dynamics level with two inflows"
      parameter Real x0 = 0 "Initial condition";
      output Real level "Continuous state variable";
      Modelica.Blocks.Continuous.Integrator Integrator1(y_start = x0) annotation(Placement(visible = true, transformation(origin = {20.0,0.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u1 "First inflow variable" annotation(Placement(visible = true, transformation(origin = {-110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u2 "Second inflow variable" annotation(Placement(visible = true, transformation(origin = {-110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u3 "Outflow variable" annotation(Placement(visible = true, transformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y "State variable" annotation(Placement(visible = true, transformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {85.0,55.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "State variable" annotation(Placement(visible = true, transformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y2 "State variable" annotation(Placement(visible = true, transformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y3 "State variable" annotation(Placement(visible = true, transformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      Modelica.Blocks.Math.Add3 Add1(k3 = -1) annotation(Placement(visible = true, transformation(origin = {-40.0,0.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
    equation
      level = y;
      connect(y,Integrator1.y) annotation(Line(points = {{85.0,55.0},{60.0,55.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y1,Integrator1.y) annotation(Line(points = {{75.0,-65.0},{75.0,-50.0},{60.0,-50.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y2,Integrator1.y) annotation(Line(points = {{0.0,-65.0},{0.0,-50.0},{60.0,-50.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y3,Integrator1.y) annotation(Line(points = {{-75.0,-65.0},{-75.0,-50.0},{60.0,-50.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(Add1.y,Integrator1.u) annotation(Line(points = {{-18.0,0.0},{-4.0,0.0}}, color = {0,0,191}, visible = true));
      connect(u1,Add1.u1) annotation(Line(points = {{-110.0,40.0},{-90.0,40.0},{-90.0,16.0},{-64.0,16.0}}, color = {191,0,191}, visible = true));
      connect(u2,Add1.u2) annotation(Line(points = {{-110.0,-40.0},{-90.0,-40.0},{-90.0,0.0},{-64.0,0.0}}, color = {191,0,191}, visible = true));
      connect(u3,Add1.u3) annotation(Line(points = {{110.0,0.0},{80.0,0.0},{80.0,-34.0},{-80.0,-34.0},{-80.0,-16.0},{-64.0,-16.0}}, color = {191,0,191}, visible = true));
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-60.0},{80.0,20.0}}),Rectangle(visible = true, lineColor = {127,0,255}, lineThickness = 0.5, extent = {{-80.0,-60.0},{80.0,60.0}}),Line(visible = true, points = {{-100.0,40.0},{-80.0,40.0}}, color = {127,0,255}, thickness = 0.5),Line(visible = true, points = {{-100.0,-40.0},{-80.0,-40.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,70.0},{100.0,100.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-6.0},{80.0,20.0}}, textString = "x0 = %x0", fontName = "Arial"),Line(visible = true, points = {{80.0,0.0},{100.0,0.0}}, color = {127,0,255}, thickness = 0.5)}), Documentation(info = "<html>
This is the general continuous <b>Level</b> model of the System Dynamics methodology with two inflows and a single outflow.  It computes the level by integrating over the difference between inflow and outflow rates.
</html>"));
    end Level2a;
    block Level2b "General System Dynamics level with two outflows"
      parameter Real x0 = 0 "Initial condition";
      output Real level "Continuous state variable";
      Modelica.Blocks.Continuous.Integrator Integrator1(y_start = x0) annotation(Placement(visible = true, transformation(origin = {20.0,0.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u1 "Inflow variable" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u2 "First outflow variable" annotation(Placement(visible = true, transformation(origin = {110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u3 "Second outflow variable" annotation(Placement(visible = true, transformation(origin = {110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y "State variable" annotation(Placement(visible = true, transformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0), iconTransformation(origin = {-85.0,55.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "State variable" annotation(Placement(visible = true, transformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y2 "State variable" annotation(Placement(visible = true, transformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y3 "State variable" annotation(Placement(visible = true, transformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      Modelica.Blocks.Math.Add3 Add1(k3 = -1, k1 = -1) annotation(Placement(visible = true, transformation(origin = {-40.0,0.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
    equation
      level = y;
      connect(y,Integrator1.y) annotation(Line(points = {{-85.0,55.0},{60.0,55.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y1,Integrator1.y) annotation(Line(points = {{75.0,-65.0},{75.0,-50.0},{60.0,-50.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y2,Integrator1.y) annotation(Line(points = {{0.0,-65.0},{0.0,-50.0},{60.0,-50.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y3,Integrator1.y) annotation(Line(points = {{-75.0,-65.0},{-75.0,-50.0},{60.0,-50.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(Add1.y,Integrator1.u) annotation(Line(points = {{-18.0,0.0},{-4.0,0.0}}, color = {0,0,191}, visible = true));
      connect(u1,Add1.u2) annotation(Line(points = {{-110.0,0.0},{-64.0,0.0}}, color = {191,0,191}, visible = true));
      connect(u2,Add1.u1) annotation(Line(points = {{110.0,40.0},{90.0,40.0},{90.0,80.0},{-96.0,80.0},{-96.0,16.0},{-64.0,16.0}}, color = {191,0,191}, visible = true));
      connect(u3,Add1.u3) annotation(Line(points = {{110.0,-40.0},{90.0,-40.0},{90.0,-80.0},{-96.0,-80.0},{-96.0,-16.0},{-64.0,-16.0}}, color = {191,0,191}, visible = true));
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-60.0},{80.0,20.0}}),Rectangle(visible = true, lineColor = {127,0,255}, lineThickness = 0.5, extent = {{-80.0,-60.0},{80.0,60.0}}),Line(visible = true, points = {{-100.0,0.0},{-80.0,0.0}}, color = {127,0,255}, thickness = 0.5),Line(visible = true, points = {{80.0,40.0},{100.0,40.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,70.0},{100.0,100.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-6.0},{80.0,20.0}}, textString = "x0 = %x0", fontName = "Arial"),Line(visible = true, points = {{80.0,-40.0},{100.0,-40.0}}, color = {127,0,255}, thickness = 0.5)}), Documentation(info = "<html>
This is the general continuous <b>Level</b> model of the System Dynamics methodology with a single inflow and two outflows.  It computes the level by integrating over the difference between inflow and outflow rates.
</html>"));
    end Level2b;
    block Level3 "General System Dynamics level with two inflows and two outflows"
      parameter Real x0 = 0 "Initial condition";
      output Real level "Continuous state variable";
      Modelica.Blocks.Continuous.Integrator Integrator1(y_start = x0) annotation(Placement(visible = true, transformation(origin = {20.0,0.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u1 "First inflow variable" annotation(Placement(visible = true, transformation(origin = {-110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u2 "Second inflow variable" annotation(Placement(visible = true, transformation(origin = {-110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u3 "First outflow variable" annotation(Placement(visible = true, transformation(origin = {110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassInPort u4 "Second outflow variable" annotation(Placement(visible = true, transformation(origin = {110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y "State variable" annotation(Placement(visible = true, transformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y1 "State variable" annotation(Placement(visible = true, transformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {0.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      SystemDynamics.Interfaces.MassOutPort y2 "State variable" annotation(Placement(visible = true, transformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90), iconTransformation(origin = {-75.0,-65.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -90)));
      Modelica.Blocks.Math.Add3 Add1(k1 = -1) annotation(Placement(visible = true, transformation(origin = {-40.0,0.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
      Modelica.Blocks.Math.Add Add2 annotation(Placement(visible = true, transformation(origin = {-40.0,60.0}, extent = {{20.0,-20.0},{-20.0,20.0}}, rotation = 0)));
    equation
      level = y;
      connect(y,Integrator1.y) annotation(Line(points = {{75.0,-65.0},{75.0,-50.0},{60.0,-50.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y1,Integrator1.y) annotation(Line(points = {{0.0,-65.0},{0.0,-50.0},{60.0,-50.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y2,Integrator1.y) annotation(Line(points = {{-75.0,-65.0},{-75.0,-50.0},{60.0,-50.0},{60.0,0.0},{42.0,0.0}}, color = {95,0,191}, visible = true));
      connect(Add1.y,Integrator1.u) annotation(Line(points = {{-18.0,0.0},{-4.0,0.0}}, color = {0,0,191}, visible = true));
      connect(u2,Add1.u3) annotation(Line(points = {{-110.0,-40.0},{-90.0,-40.0},{-90.0,-16.0},{-64.0,-16.0}}, color = {191,0,191}, visible = true));
      connect(u1,Add1.u2) annotation(Line(points = {{-110.0,40.0},{-90.0,40.0},{-90.0,0.0},{-64.0,0.0}}, color = {191,0,191}, visible = true));
      connect(u3,Add2.u1) annotation(Line(points = {{110.0,40.0},{90.0,40.0},{90.0,72.0},{-16.0,72.0}}, color = {191,0,191}, visible = true));
      connect(u4,Add2.u2) annotation(Line(points = {{110.0,-40.0},{80.0,-40.0},{80.0,48.0},{-16.0,48.0}}, color = {191,0,191}, visible = true));
      connect(Add2.y,Add1.u1) annotation(Line(points = {{-62.0,60.0},{-80.0,60.0},{-80.0,16.0},{-64.0,16.0}}, color = {0,0,191}, visible = true));
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-60.0},{80.0,20.0}}),Rectangle(visible = true, lineColor = {127,0,255}, lineThickness = 0.5, extent = {{-80.0,-60.0},{80.0,60.0}}),Line(visible = true, points = {{-100.0,40.0},{-80.0,40.0}}, color = {127,0,255}, thickness = 0.5),Line(visible = true, points = {{-100.0,-40.0},{-80.0,-40.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,70.0},{100.0,100.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,-6.0},{80.0,20.0}}, textString = "x0 = %x0", fontName = "Arial"),Line(visible = true, points = {{80.0,40.0},{100.0,40.0}}, color = {127,0,255}, thickness = 0.5),Line(visible = true, points = {{80.0,-40.0},{100.0,-40.0}}, color = {127,0,255}, thickness = 0.5)}), Documentation(info = "<html>
This is the general continuous <b>Level</b> model of the System Dynamics methodology with two inflows and two outflows.  It computes the level by integrating over the difference between inflow and outflow rates.
</html>"));
    end Level3;
  end Levels;
  package Rates "Rates of the System Dynamics methodology"
    extends Modelica.Icons.Library annotation(preferedView = "info", Documentation(info = "<html>
This package contains a set of different <b>Rates</b> (state derivative variables) frequently used in the System Dynamics methodology.
</html>"));
    block Rate_1 "Unrestricted rate element with one influencing variable"
      output Real rate;
      Modelica.Blocks.Interfaces.RealInput u "Signal inflow variable" annotation(Placement(visible = true, transformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      SystemDynamics.Interfaces.MassOutPort y "Mass inflow variable" annotation(Placement(visible = true, transformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "Mass outflow variable" annotation(Placement(visible = true, transformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
    equation
      rate = y;
      connect(y,y1) annotation(Line(points = {{-50.0,0.0},{50.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y1,u) annotation(Line(points = {{50.0,0.0},{0.0,0.0},{0.0,-105.0}}, color = {95,0,191}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Polygon(visible = true, lineColor = {127,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, lineThickness = 0.5, points = {{-60.0,-100.0},{-60.0,-40.0},{40.0,26.0},{-40.0,26.0},{60.0,-40.0},{60.0,-100.0},{-60.0,-100.0}}),Line(visible = true, points = {{-40.0,0.0},{40.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,50.0},{100.0,88.0}}, textString = "%name", fontName = "Arial")}), Documentation(info = "<html>
This is the general System Dynamics unrestricted <b>Rate</b> element, whereby the rate itself is determined by a single variable in its laundry list.  The indicated direction of mass flow simply denotes the direction of positive mass flow.  However if the control signal of the rate assumes a negative value, mass will flow in the opposite direction.
</html>"), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics));
    end Rate_1;
    block RRate "Restricted rate element"
      constant Real inf = Modelica.Constants.inf;
      parameter Real minFlow = 0 "Smallest allowed flow";
      parameter Real maxFlow = inf "Largest allowed flow";
      output Real rate;
      Modelica.Blocks.Interfaces.RealInput u "Signal inflow variable" annotation(Placement(visible = true, transformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      SystemDynamics.Interfaces.MassOutPort y "Mass inflow variable" annotation(Placement(visible = true, transformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "Mass outflow variable" annotation(Placement(visible = true, transformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Nonlinear.Limiter Limiter1(uMax = maxFlow, uMin = minFlow) annotation(Placement(visible = true, transformation(origin = {0.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
    equation
      rate = y;
      connect(y,y1) annotation(Line(points = {{-50.0,0.0},{50.0,0.0}}, color = {95,0,191}, visible = true));
      connect(u,Limiter1.u) annotation(Line(points = {{0.0,-105.0},{0.0,-62.0},{0.0,-62.0}}, color = {0,0,191}, visible = true));
      connect(y1,Limiter1.y) annotation(Line(points = {{50.0,0.0},{0.0,0.0},{0.0,-39.0}}, color = {95,0,191}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Polygon(visible = true, lineColor = {127,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, lineThickness = 0.5, points = {{-60.0,-100.0},{-60.0,-40.0},{40.0,26.0},{-40.0,26.0},{60.0,-40.0},{60.0,-100.0},{-60.0,-100.0}}),Line(visible = true, points = {{-40.0,0.0},{40.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,50.0},{100.0,88.0}}, textString = "%name", fontName = "Arial"),Line(visible = true, points = {{-40.0,-74.0},{-12.0,-74.0}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{8.0,-46.0},{36.0,-46.0}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-12.0,-74.0},{8.0,-46.0}}, color = {0,0,255}, thickness = 0.5)}), Documentation(info = "<html>
This is the System Dynamics restricted <b>Rate</b> element, whereby the rate itself is determined by a single variable in its laundry list.  The indicated direction of mass flow denotes the direction of true mass flow.  If the control signal of the rate assumes a negative value, no mass will flow.  If the control signal of the rate assumes a too large positive value, the mass flow will be limited to its allowed maximum value. <p>
 
The restricted rate element should e.g. be used to model a kitchen faucet.  The valve can be opened or closed, but there is a limit to how much water per time unit you can get out of the faucet, and no manipulation of the faucet will ever make water flow from the sink back into the faucet.
</html>"), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics));
    end RRate;
    block CInflow "Controlled inflow element"
      constant Real inf = Modelica.Constants.inf;
      parameter Real minFlow = 0 "Smallest allowed flow";
      parameter Real maxFlow = inf "Largest allowed flow";
      output Real rate;
      Modelica.Blocks.Interfaces.RealInput u "Signal inflow variable" annotation(Placement(visible = true, transformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      SystemDynamics.Interfaces.MassOutPort y "Mass inflow variable" annotation(Placement(visible = true, transformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "Mass outflow variable" annotation(Placement(visible = true, transformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Nonlinear.Limiter Limiter1(uMax = maxFlow, uMin = minFlow) annotation(Placement(visible = true, transformation(origin = {0.0,-30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.BooleanInput s "Control signal" annotation(Placement(visible = true, transformation(origin = {65.0,-70.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0), iconTransformation(origin = {65.0,-70.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0)));
      Modelica.Blocks.Logical.Switch Switch1 annotation(Placement(visible = true, transformation(origin = {0.0,-60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
      Modelica.Blocks.Sources.Constant Const1(k = 0) annotation(Placement(visible = true, transformation(origin = {40.0,-50.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
    equation
      rate = y;
      connect(y,y1) annotation(Line(points = {{-50.0,0.0},{50.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y1,Limiter1.y) annotation(Line(points = {{33.3333,6.3333},{-16.6667,6.3333},{-16.6667,-12.6667}}, color = {95,0,191}, visible = true, origin = {16.6667,-6.3333}));
      connect(u,Switch1.u1) annotation(Line(points = {{0.0,-105.0},{0.0,-94.0},{-8.0,-94.0},{-8.0,-72.0}}, color = {0,0,191}, visible = true));
      connect(Switch1.y,Limiter1.u) annotation(Line(points = {{0.0,-49.0},{0.0,-50.5},{0.0,-50.5},{-0.0,-42.0}}, color = {0,0,191}, visible = true));
      connect(s,Switch1.u2) annotation(Line(points = {{65.0,-70.0},{50.0,-70.0},{50.0,-88.0},{0.0,-88.0},{0.0,-72.0},{0.0,-72.0}}, color = {255,0,255}, visible = true));
      connect(Switch1.u3,Const1.y) annotation(Line(points = {{8.0,-72.0},{8.0,-80.0},{20.0,-80.0},{20.0,-50.0},{29.0,-50.0}}, color = {0,0,191}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Polygon(visible = true, lineColor = {127,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, lineThickness = 0.5, points = {{-60.0,-100.0},{-60.0,-40.0},{40.0,26.0},{-40.0,26.0},{60.0,-40.0},{60.0,-100.0},{-60.0,-100.0}}),Line(visible = true, points = {{-40.0,0.0},{40.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,50.0},{100.0,88.0}}, textString = "%name", fontName = "Arial"),Line(visible = true, points = {{-40.0,-74.0},{-12.0,-74.0}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{8.0,-46.0},{36.0,-46.0}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-12.0,-74.0},{8.0,-46.0}}, color = {0,0,255}, thickness = 0.5)}), Documentation(info = "<html>
This is the System Dynamics controlled restricted <b>Inflow</b> element, whereby the rate itself is determined by a single variable in its laundry list.  This element is a restricted rate element with the additional feature that the downstream level can control the flow by telling it that it is full.  This is accomplished by a Boolean control signal from the downstream level to the rate element. <p>
 
The controlled inflow element should e.g. be used to model the access to a parking garage.  Additional cars are only admitted as long as the garage is not full.
</html>"), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics));
    end CInflow;
    block COutflow "Controlled outflow element"
      constant Real inf = Modelica.Constants.inf;
      parameter Real minFlow = 0 "Smallest allowed flow";
      parameter Real maxFlow = inf "Largest allowed flow";
      output Real rate;
      Modelica.Blocks.Interfaces.RealInput u "Signal inflow variable" annotation(Placement(visible = true, transformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      SystemDynamics.Interfaces.MassOutPort y "Mass inflow variable" annotation(Placement(visible = true, transformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "Mass outflow variable" annotation(Placement(visible = true, transformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Nonlinear.Limiter Limiter1(uMax = maxFlow, uMin = minFlow) annotation(Placement(visible = true, transformation(origin = {0.0,-30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.BooleanInput s "Control signal" annotation(Placement(visible = true, transformation(origin = {-65.0,-70.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {-65.0,-70.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      Modelica.Blocks.Logical.Switch Switch1 annotation(Placement(visible = true, transformation(origin = {0.0,-60.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = -90)));
      Modelica.Blocks.Sources.Constant Const1(k = 0) annotation(Placement(visible = true, transformation(origin = {-40.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
    equation
      rate = y;
      connect(y,y1) annotation(Line(points = {{-50.0,0.0},{50.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y1,Limiter1.y) annotation(Line(points = {{50.0,0.0},{0.0,0.0},{0.0,-19.0}}, color = {95,0,191}, visible = true));
      connect(Switch1.y,Limiter1.u) annotation(Line(points = {{0.0,-49.0},{0.0,-50.5},{0.0,-50.5},{0.0,-42.0}}, color = {0,0,191}, visible = true));
      connect(u,Switch1.u1) annotation(Line(points = {{0.0,-105.0},{0.0,-94.0},{8.0,-94.0},{8.0,-72.0}}, color = {0,0,191}, visible = true));
      connect(s,Switch1.u2) annotation(Line(points = {{-65.0,-70.0},{-50.0,-70.0},{-50.0,-86.0},{0.0,-86.0},{0.0,-72.0}}, color = {255,0,255}, visible = true));
      connect(Const1.y,Switch1.u3) annotation(Line(points = {{-29.0,-50.0},{-20.0,-50.0},{-20.0,-80.0},{-8.0,-80.0},{-8.0,-72.0}}, color = {0,0,191}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Polygon(visible = true, lineColor = {127,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, lineThickness = 0.5, points = {{-60.0,-100.0},{-60.0,-40.0},{40.0,26.0},{-40.0,26.0},{60.0,-40.0},{60.0,-100.0},{-60.0,-100.0}}),Line(visible = true, points = {{-40.0,0.0},{40.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,50.0},{100.0,88.0}}, textString = "%name", fontName = "Arial"),Line(visible = true, points = {{-40.0,-74.0},{-12.0,-74.0}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{8.0,-46.0},{36.0,-46.0}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-12.0,-74.0},{8.0,-46.0}}, color = {0,0,255}, thickness = 0.5)}), Documentation(info = "<html>
This is the System Dynamics controlled restricted <b>Outflow</b> element, whereby the rate itself is determined by a single variable in its laundry list.  This element is a restricted rate element with the additional feature that the upstream level can control the flow by telling it that it is empty.  This is accomplished by a Boolean control signal from the upstream level to the rate element. <p>
 
The controlled outflow element should e.g. be used to model the drainage of a kitchen sink.  Water can flow out of the kitchen sink only as long as there is any water left.
</html>"), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics));
    end COutflow;
    block CRate "Controlled rate element"
      constant Real inf = Modelica.Constants.inf;
      parameter Real minFlow = 0 "Smallest allowed flow";
      parameter Real maxFlow = inf "Largest allowed flow";
      output Real rate;
      Modelica.Blocks.Interfaces.RealInput u "Signal inflow variable" annotation(Placement(visible = true, transformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      SystemDynamics.Interfaces.MassOutPort y "Mass inflow variable" annotation(Placement(visible = true, transformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "Mass outflow variable" annotation(Placement(visible = true, transformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Nonlinear.Limiter Limiter1(uMax = maxFlow, uMin = minFlow) annotation(Placement(visible = true, transformation(origin = {0.0,-30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.BooleanInput s1 "Control signal" annotation(Placement(visible = true, transformation(origin = {-65.0,-70.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {-65.0,-70.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      Modelica.Blocks.Logical.Switch Switch1 annotation(Placement(visible = true, transformation(origin = {0.0,-60.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = -90)));
      Modelica.Blocks.Sources.Constant Const1(k = 0) annotation(Placement(visible = true, transformation(origin = {-40.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.BooleanInput s2 "Control signal" annotation(Placement(visible = true, transformation(origin = {65.0,-70.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0), iconTransformation(origin = {65.0,-70.0}, extent = {{5.0,-5.0},{-5.0,5.0}}, rotation = 0)));
      Modelica.Blocks.Logical.And and1 annotation(Placement(visible = true, transformation(origin = {-40.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
    equation
      rate = y;
      connect(y,y1) annotation(Line(points = {{-50.0,0.0},{50.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y1,Limiter1.y) annotation(Line(points = {{50.0,0.0},{0.0,0.0},{0.0,-19.0}}, color = {95,0,191}, visible = true));
      connect(Switch1.y,Limiter1.u) annotation(Line(points = {{0.0,-49.0},{0.0,-50.5},{0.0,-50.5},{0.0,-42.0}}, color = {0,0,191}, visible = true));
      connect(u,Switch1.u1) annotation(Line(points = {{0.0,-105.0},{0.0,-94.0},{8.0,-94.0},{8.0,-72.0}}, color = {0,0,191}, visible = true));
      connect(Const1.y,Switch1.u3) annotation(Line(points = {{-29.0,-40.0},{-18.0,-40.0},{-18.0,-80.0},{-8.0,-80.0},{-8.0,-72.0}}, color = {0,0,191}, visible = true));
      connect(and1.u1,s1) annotation(Line(points = {{-52.0,-70.0},{-65.0,-70.0}}, color = {255,0,255}, visible = true));
      connect(and1.y,Switch1.u2) annotation(Line(points = {{-29.0,-70.0},{-24.0,-70.0},{-24.0,-86.0},{0.0,-86.0},{0.0,-72.0}}, color = {255,0,255}, visible = true));
      connect(s2,and1.u2) annotation(Line(points = {{65.0,-70.0},{50.0,-70.0},{50.0,-20.0},{70.0,-20.0},{70.0,20.0},{-80.0,20.0},{-80.0,-78.0},{-52.0,-78.0}}, color = {255,0,255}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Polygon(visible = true, lineColor = {127,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, lineThickness = 0.5, points = {{-60.0,-100.0},{-60.0,-40.0},{40.0,26.0},{-40.0,26.0},{60.0,-40.0},{60.0,-100.0},{-60.0,-100.0}}),Line(visible = true, points = {{-40.0,0.0},{40.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,50.0},{100.0,88.0}}, textString = "%name", fontName = "Arial"),Line(visible = true, points = {{-40.0,-74.0},{-12.0,-74.0}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{8.0,-46.0},{36.0,-46.0}}, color = {0,0,255}, thickness = 0.5),Line(visible = true, points = {{-12.0,-74.0},{8.0,-46.0}}, color = {0,0,255}, thickness = 0.5)}), Documentation(info = "<html>
This is the System Dynamics controlled restricted <b>Rate</b> element, whereby the rate itself is determined by a single variable in its laundry list.  This element is a restricted rate element with the additional feature that the upstream level can control the flow by telling it that it is empty, and that the downstream level can control the flow by telling it that it is full.  This is accomplished by two Boolean control signals from the upstream and downstream levels to the rate element. <p>
 
The controlled rate element must be placed between two controlling levels.
</html>"), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics));
    end CRate;
    block Rate_2 "Unrestricted additive rate element with two influencing variables"
      output Real rate;
      Modelica.Blocks.Math.Add Add1 annotation(Placement(visible = true, transformation(origin = {0.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u "First signal inflow variable" annotation(Placement(visible = true, transformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u1 "Second signal inflow variable" annotation(Placement(visible = true, transformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      SystemDynamics.Interfaces.MassOutPort y "Mass inflow variable" annotation(Placement(visible = true, transformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "Mass outflow variable" annotation(Placement(visible = true, transformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
    equation
      rate = y;
      connect(u,Add1.u1) annotation(Line(points = {{-40.0,-105.0},{-40.0,-80.0},{-12.0,-80.0},{-12.0,-64.0}}, color = {0,0,191}, visible = true));
      connect(u1,Add1.u2) annotation(Line(points = {{40.0,-105.0},{40.0,-80.0},{12.0,-80.0},{12.0,-64.0}}, color = {0,0,191}, visible = true));
      connect(Add1.y,y1) annotation(Line(points = {{0.0,-18.0},{0.0,0.0},{50.0,0.0}}, color = {0,0,191}, visible = true));
      connect(y,y1) annotation(Line(points = {{-50.0,0.0},{50.0,0.0}}, color = {95,0,191}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Polygon(visible = true, lineColor = {127,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, lineThickness = 0.5, points = {{-60.0,-100.0},{-60.0,-40.0},{40.0,26.0},{-40.0,26.0},{60.0,-40.0},{60.0,-100.0},{-60.0,-100.0}}),Line(visible = true, points = {{-40.0,0.0},{40.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,50.0},{100.0,88.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-26.0,-80.0},{26.0,-42.0}}, textString = "+", fontName = "Arial")}), Documentation(info = "<html>
This is the general System Dynamics unrestricted additive <b>Rate</b> element, whereby the rate itself is determined by two variables in its laundry list.  The indicated direction of mass flow simply denotes the direction of positive mass flow.  However if the control signal of the rate assumes a negative value, mass will flow in the opposite direction.
</html>"), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics));
    end Rate_2;
    block MRate_2 "Unrestricted multiplicative rate element with two influencing variables"
      output Real rate;
      Modelica.Blocks.Interfaces.RealInput u "First signal inflow variable" annotation(Placement(visible = true, transformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u1 "Second signal inflow variable" annotation(Placement(visible = true, transformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      SystemDynamics.Interfaces.MassOutPort y "Mass inflow variable" annotation(Placement(visible = true, transformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "Mass outflow variable" annotation(Placement(visible = true, transformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Auxiliary.Prod_2 prod_2_1 annotation(Placement(visible = true, transformation(origin = {0.0,-50.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
    equation
      rate = y;
      connect(y,y1) annotation(Line(points = {{-50.0,0.0},{50.0,0.0}}, color = {95,0,191}, visible = true));
      connect(prod_2_1.u1,u) annotation(Line(points = {{17.3333,18.3333},{-8.6667,18.3333},{-8.6667,-36.6667}}, color = {0,0,191}, visible = true, origin = {-31.3333,-68.3333}));
      connect(prod_2_1.u2,u1) annotation(Line(points = {{-17.3333,18.3333},{8.6667,18.3333},{8.6667,-36.6667}}, color = {0,0,191}, visible = true, origin = {31.3333,-68.3333}));
      connect(y1,prod_2_1.y) annotation(Line(points = {{50.0,0.0},{0.0,0.0},{0.0,-36.0}}, color = {95,0,191}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Polygon(visible = true, lineColor = {127,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, lineThickness = 0.5, points = {{-60.0,-100.0},{-60.0,-40.0},{40.0,26.0},{-40.0,26.0},{60.0,-40.0},{60.0,-100.0},{-60.0,-100.0}}),Line(visible = true, points = {{-40.0,0.0},{40.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,50.0},{100.0,88.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-29.0,-98.0},{31.0,-42.0}}, textString = "*", fontName = "Arial")}), Documentation(info = "<html>
This is the general System Dynamics unrestricted multiplicative <b>Rate</b> element, whereby the rate itself is determined by two variables in its laundry list.  The indicated direction of mass flow simply denotes the direction of positive mass flow.  However if the control signal of the rate assumes a negative value, mass will flow in the opposite direction.
</html>"), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics));
    end MRate_2;
    block Rate_3 "Unrestricted additive rate element with three influencing variables"
      output Real rate;
      Modelica.Blocks.Math.Add3 Add3_1 annotation(Placement(visible = true, transformation(origin = {0.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u "First signal inflow variable" annotation(Placement(visible = true, transformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u1 "Second signal inflow variable" annotation(Placement(visible = true, transformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u2 "Third signal inflow variable" annotation(Placement(visible = true, transformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      SystemDynamics.Interfaces.MassOutPort y "Mass inflow variable" annotation(Placement(visible = true, transformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "Mass outflow variable" annotation(Placement(visible = true, transformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
    equation
      rate = y;
      connect(u,Add3_1.u1) annotation(Line(points = {{-40.0,-105.0},{-40.0,-80.0},{-16.0,-80.0},{-16.0,-64.0}}, color = {0,0,191}, visible = true));
      connect(u1,Add3_1.u2) annotation(Line(points = {{0.0,-105.0},{0.0,-64.0},{0.0,-64.0}}, color = {0,0,191}, visible = true));
      connect(u2,Add3_1.u3) annotation(Line(points = {{40.0,-105.0},{40.0,-80.0},{16.0,-80.0},{16.0,-64.0}}, color = {0,0,191}, visible = true));
      connect(Add3_1.y,y1) annotation(Line(points = {{0.0,-18.0},{0.0,0.0},{50.0,0.0}}, color = {0,0,191}, visible = true));
      connect(y,y1) annotation(Line(points = {{-50.0,0.0},{50.0,0.0}}, color = {95,0,191}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Polygon(visible = true, lineColor = {127,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, lineThickness = 0.5, points = {{-60.0,-100.0},{-60.0,-40.0},{40.0,26.0},{-40.0,26.0},{60.0,-40.0},{60.0,-100.0},{-60.0,-100.0}}),Line(visible = true, points = {{-40.0,0.0},{40.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,50.0},{100.0,88.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-26.0,-80.0},{26.0,-42.0}}, textString = "+", fontName = "Arial")}), Documentation(info = "<html>
This is the general System Dynamics unrestricted additive <b>Rate</b> element, whereby the rate itself is determined by three variables in its laundry list.  The indicated direction of mass flow simply denotes the direction of positive mass flow.  However if the control signal of the rate assumes a negative value, mass will flow in the opposite direction.
</html>"), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics));
    end Rate_3;
    block MRate_3 "Unrestricted multiplicative rate element with three influencing variables"
      output Real rate;
      Modelica.Blocks.Interfaces.RealInput u "First signal inflow variable" annotation(Placement(visible = true, transformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u1 "Second signal inflow variable" annotation(Placement(visible = true, transformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u2 "Third signal inflow variable" annotation(Placement(visible = true, transformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      SystemDynamics.Interfaces.MassOutPort y "Mass inflow variable" annotation(Placement(visible = true, transformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "Mass outflow variable" annotation(Placement(visible = true, transformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Auxiliary.Prod_3 prod_3_1 annotation(Placement(visible = true, transformation(origin = {0.0,-50.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
    equation
      rate = y;
      connect(y,y1) annotation(Line(points = {{-50.0,0.0},{50.0,0.0}}, color = {95,0,191}, visible = true));
      connect(prod_3_1.u1,u) annotation(Line(points = {{17.3333,18.3333},{-8.6667,18.3333},{-8.6667,-36.6667}}, color = {0,0,191}, visible = true, origin = {-31.3333,-68.3333}));
      connect(prod_3_1.u2,u1) annotation(Line(points = {{0.0,20.5},{0.0,-20.5}}, color = {0,0,191}, visible = true, origin = {0.0,-84.5}));
      connect(prod_3_1.u3,u2) annotation(Line(points = {{-17.3333,18.3333},{8.6667,18.3333},{8.6667,-36.6667}}, color = {0,0,191}, visible = true, origin = {31.3333,-68.3333}));
      connect(y1,prod_3_1.y) annotation(Line(points = {{50.0,0.0},{0.0,0.0},{0.0,-36.0}}, color = {95,0,191}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Polygon(visible = true, lineColor = {127,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, lineThickness = 0.5, points = {{-60.0,-100.0},{-60.0,-40.0},{40.0,26.0},{-40.0,26.0},{60.0,-40.0},{60.0,-100.0},{-60.0,-100.0}}),Line(visible = true, points = {{-40.0,0.0},{40.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,50.0},{100.0,88.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-30.0,-98.0},{30.0,-42.0}}, textString = "*", fontName = "Arial")}), Documentation(info = "<html>
This is the general System Dynamics unrestricted multiplicative <b>Rate</b> element, whereby the rate itself is determined by three variables in its laundry list.  The indicated direction of mass flow simply denotes the direction of positive mass flow.  However if the control signal of the rate assumes a negative value, mass will flow in the opposite direction.
</html>"), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics));
    end MRate_3;
    block Rate_4 "Unrestricted additive rate element with four influencing variables"
      output Real rate;
      Modelica.Blocks.Math.Add Add1 annotation(Placement(visible = true, transformation(origin = {-30.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      Modelica.Blocks.Math.Add3 Add3_1 annotation(Placement(visible = true, transformation(origin = {0.0,-30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u "First signal inflow variable" annotation(Placement(visible = true, transformation(origin = {-65.0,-69.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {-65.0,-69.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u1 "Second signal inflow variable" annotation(Placement(visible = true, transformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u2 "Third signal inflow variable" annotation(Placement(visible = true, transformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u3 "Fourth signal inflow variable" annotation(Placement(visible = true, transformation(origin = {65.0,-69.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -180), iconTransformation(origin = {65.0,-69.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -180)));
      SystemDynamics.Interfaces.MassOutPort y "Mass inflow variable" annotation(Placement(visible = true, transformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "Mass outflow variable" annotation(Placement(visible = true, transformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
    equation
      rate = y;
      connect(u,Add1.u1) annotation(Line(points = {{-65.0,-69.0},{-52.0,-69.0},{-52.0,-64.0},{-42.0,-64.0}}, color = {0,0,191}, visible = true));
      connect(u1,Add1.u2) annotation(Line(points = {{-40.0,-105.0},{-40.0,-90.0},{-52.0,-90.0},{-52.0,-76.0},{-42.0,-76.0}}, color = {0,0,191}, visible = true));
      connect(u2,Add3_1.u2) annotation(Line(points = {{40.0,-105.0},{40.0,-80.0},{0.0,-80.0},{0.0,-42.0}}, color = {0,0,191}, visible = true));
      connect(Add3_1.u3,u3) annotation(Line(points = {{8.0,-42.0},{8.0,-69.0},{65.0,-69.0}}, color = {0,0,191}, visible = true));
      connect(Add1.y,Add3_1.u1) annotation(Line(points = {{-19.0,-70.0},{-8.0,-70.0},{-8.0,-42.0}}, color = {0,0,191}, visible = true));
      connect(Add3_1.y,y1) annotation(Line(points = {{0.0,-19.0},{0.0,0.0},{50.0,0.0}}, color = {0,0,191}, visible = true));
      connect(y,y1) annotation(Line(points = {{-50.0,0.0},{50.0,0.0}}, color = {95,0,191}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Polygon(visible = true, lineColor = {127,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, lineThickness = 0.5, points = {{-60.0,-100.0},{-60.0,-40.0},{40.0,26.0},{-40.0,26.0},{60.0,-40.0},{60.0,-100.0},{-60.0,-100.0}}),Line(visible = true, points = {{-40.0,0.0},{40.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,50.0},{100.0,88.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-26.0,-80.0},{26.0,-42.0}}, textString = "+", fontName = "Arial")}), Documentation(info = "<html>
This is the general System Dynamics unrestricted additive <b>Rate</b> element, whereby the rate itself is determined by four variables in its laundry list.  The indicated direction of mass flow simply denotes the direction of positive mass flow.  However if the control signal of the rate assumes a negative value, mass will flow in the opposite direction.
</html>"), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics));
    end Rate_4;
    block MRate_4 "Unrestricted multiplicative rate element with four influencing variables"
      output Real rate;
      Modelica.Blocks.Interfaces.RealInput u "First signal inflow variable" annotation(Placement(visible = true, transformation(origin = {-65.0,-70.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {-65.0,-70.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u1 "Second signal inflow variable" annotation(Placement(visible = true, transformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u2 "Third signal inflow variable" annotation(Placement(visible = true, transformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u3 "Fourth signal inflow variable" annotation(Placement(visible = true, transformation(origin = {65.0,-70.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -180), iconTransformation(origin = {65.0,-70.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -180)));
      SystemDynamics.Interfaces.MassOutPort y "Mass inflow variable" annotation(Placement(visible = true, transformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "Mass outflow variable" annotation(Placement(visible = true, transformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Auxiliary.Prod_4 prod_4_1 annotation(Placement(visible = true, transformation(origin = {0.0,-50.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
    equation
      rate = y;
      connect(y,y1) annotation(Line(points = {{-50.0,0.0},{50.0,0.0}}, color = {95,0,191}, visible = true));
      connect(u,prod_4_1.u1) annotation(Line(points = {{-65.0,-70.0},{-50.0,-70.0},{-50.0,-50.0},{-14.0,-50.0}}, color = {0,0,191}, visible = true));
      connect(u1,prod_4_1.u2) annotation(Line(points = {{-40.0,-105.0},{-40.0,-80.0},{-8.0,-80.0},{-8.0,-61.2}}, color = {0,0,191}, visible = true));
      connect(u2,prod_4_1.u3) annotation(Line(points = {{40.0,-105.0},{40.0,-80.0},{8.0,-80.0},{8.0,-61.2}}, color = {0,0,191}, visible = true));
      connect(u3,prod_4_1.u4) annotation(Line(points = {{65.0,-70.0},{50.0,-70.0},{50.0,-50.0},{14.0,-50.0}}, color = {0,0,191}, visible = true));
      connect(y1,prod_4_1.y) annotation(Line(points = {{50.0,0.0},{0.0,0.0},{0.0,-36.0}}, color = {95,0,191}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Polygon(visible = true, lineColor = {127,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, lineThickness = 0.5, points = {{-60.0,-100.0},{-60.0,-40.0},{40.0,26.0},{-40.0,26.0},{60.0,-40.0},{60.0,-100.0},{-60.0,-100.0}}),Line(visible = true, points = {{-40.0,0.0},{40.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,50.0},{100.0,88.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-30.0,-98.0},{30.0,-42.0}}, textString = "*", fontName = "Arial")}), Documentation(info = "<html>
This is the general System Dynamics unrestricted multiplicative <b>Rate</b> element, whereby the rate itself is determined by four variables in its laundry list.  The indicated direction of mass flow simply denotes the direction of positive mass flow.  However if the control signal of the rate assumes a negative value, mass will flow in the opposite direction.
</html>"), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics));
    end MRate_4;
    block Rate_5 "Unrestricted additive rate element with five influencing variables"
      output Real rate;
      Modelica.Blocks.Math.Add3 Add3_1 annotation(Placement(visible = true, transformation(origin = {-30.0,-60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
      Modelica.Blocks.Math.Add3 Add3_2 annotation(Placement(visible = true, transformation(origin = {0.0,-20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u "First signal inflow variable" annotation(Placement(visible = true, transformation(origin = {-65.0,-69.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {-65.0,-69.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u1 "Second signal inflow variable" annotation(Placement(visible = true, transformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u2 "Third signal inflow variable" annotation(Placement(visible = true, transformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u3 "Fourth signal inflow variable" annotation(Placement(visible = true, transformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u4 "Fifth signal inflow variable" annotation(Placement(visible = true, transformation(origin = {65.0,-69.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -180), iconTransformation(origin = {65.0,-69.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -180)));
      SystemDynamics.Interfaces.MassOutPort y "Mass inflow variable" annotation(Placement(visible = true, transformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "Mass outflow variable" annotation(Placement(visible = true, transformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
    equation
      rate = y;
      connect(u,Add3_1.u1) annotation(Line(points = {{-65.0,-69.0},{-50.0,-69.0},{-50.0,-80.0},{-38.0,-80.0},{-38.0,-72.0}}, color = {0,0,191}, visible = true));
      connect(u1,Add3_1.u2) annotation(Line(points = {{-40.0,-105.0},{-40.0,-86.0},{-30.0,-86.0},{-30.0,-72.0}}, color = {0,0,191}, visible = true));
      connect(u2,Add3_1.u3) annotation(Line(points = {{0.0,-105.0},{0.0,-86.0},{-22.0,-86.0},{-22.0,-72.0}}, color = {0,0,191}, visible = true));
      connect(u3,Add3_2.u2) annotation(Line(points = {{40.0,-105.0},{40.0,-80.0},{0.0,-80.0},{0.0,-32.0}}, color = {0,0,191}, visible = true));
      connect(Add3_1.y,Add3_2.u1) annotation(Line(points = {{-30.0,-49.0},{-30.0,-40.0},{-8.0,-40.0},{-8.0,-32.0}}, color = {0,0,191}, visible = true));
      connect(u4,Add3_2.u3) annotation(Line(points = {{65.0,-69.0},{8.0,-69.0},{8.0,-32.0}}, color = {0,0,191}, visible = true));
      connect(Add3_2.y,y1) annotation(Line(points = {{0.0,-9.0},{0.0,0.0},{50.0,0.0}}, color = {0,0,191}, visible = true));
      connect(y,y1) annotation(Line(points = {{-50.0,0.0},{50.0,0.0}}, color = {95,0,191}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Polygon(visible = true, lineColor = {127,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, lineThickness = 0.5, points = {{-60.0,-100.0},{-60.0,-40.0},{40.0,26.0},{-40.0,26.0},{60.0,-40.0},{60.0,-100.0},{-60.0,-100.0}}),Line(visible = true, points = {{-40.0,0.0},{40.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,50.0},{100.0,88.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-26.0,-80.0},{26.0,-42.0}}, textString = "+", fontName = "Arial")}), Documentation(info = "<html>
This is the general System Dynamics unrestricted additive <b>Rate</b> element, whereby the rate itself is determined by five variables in its laundry list.  The indicated direction of mass flow simply denotes the direction of positive mass flow.  However if the control signal of the rate assumes a negative value, mass will flow in the opposite direction.
</html>"), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics));
    end Rate_5;
    block MRate_5 "Unrestricted multiplicative rate element with five influencing variables"
      output Real rate;
      Modelica.Blocks.Interfaces.RealInput u "First signal inflow variable" annotation(Placement(visible = true, transformation(origin = {-65.0,-70.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0), iconTransformation(origin = {-65.0,-70.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = 0)));
      Modelica.Blocks.Interfaces.RealInput u1 "Second signal inflow variable" annotation(Placement(visible = true, transformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {-40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u2 "Third signal inflow variable" annotation(Placement(visible = true, transformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {0.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u3 "Fourth signal inflow variable" annotation(Placement(visible = true, transformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270), iconTransformation(origin = {40.0,-105.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -270)));
      Modelica.Blocks.Interfaces.RealInput u4 "Fifth signal inflow variable" annotation(Placement(visible = true, transformation(origin = {65.0,-70.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -180), iconTransformation(origin = {65.0,-70.0}, extent = {{-5.0,-5.0},{5.0,5.0}}, rotation = -180)));
      SystemDynamics.Interfaces.MassOutPort y "Mass inflow variable" annotation(Placement(visible = true, transformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Interfaces.MassOutPort y1 "Mass outflow variable" annotation(Placement(visible = true, transformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {50.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      SystemDynamics.Auxiliary.Prod_5 prod_5_1 annotation(Placement(visible = true, transformation(origin = {0.0,-50.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
    equation
      rate = y;
      connect(y,y1) annotation(Line(points = {{-50.0,0.0},{50.0,0.0}}, color = {95,0,191}, visible = true));
      connect(y1,prod_5_1.y) annotation(Line(points = {{50.0,0.0},{0.0,0.0},{0.0,-36.0}}, color = {95,0,191}, visible = true));
      connect(u,prod_5_1.u1) annotation(Line(points = {{-65.0,-70.0},{-50.0,-70.0},{-50.0,-50.0},{-14.0,-50.0}}, color = {0,0,191}, visible = true));
      connect(u1,prod_5_1.u2) annotation(Line(points = {{-40.0,-105.0},{-40.0,-80.0},{-9.2,-80.0},{-9.2,-60.0}}, color = {0,0,191}, visible = true));
      connect(u2,prod_5_1.u3) annotation(Line(points = {{0.0,-20.5},{0.0,20.5}}, color = {0,0,191}, visible = true, origin = {0.0,-84.5}));
      connect(u3,prod_5_1.u4) annotation(Line(points = {{40.0,-105.0},{40.0,-80.0},{9.2,-80.0},{9.2,-60.0}}, color = {0,0,191}, visible = true));
      connect(u4,prod_5_1.u5) annotation(Line(points = {{65.0,-70.0},{50.0,-70.0},{50.0,-50.0},{14.0,-50.0}}, color = {0,0,191}, visible = true));
      annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Polygon(visible = true, lineColor = {127,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, lineThickness = 0.5, points = {{-60.0,-100.0},{-60.0,-40.0},{40.0,26.0},{-40.0,26.0},{60.0,-40.0},{60.0,-100.0},{-60.0,-100.0}}),Line(visible = true, points = {{-40.0,0.0},{40.0,0.0}}, color = {127,0,255}, thickness = 0.5),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,50.0},{100.0,88.0}}, textString = "%name", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-30.0,-98.0},{30.0,-42.0}}, textString = "*", fontName = "Arial")}), Documentation(info = "<html>
This is the general System Dynamics unrestricted multiplicative <b>Rate</b> element, whereby the rate itself is determined by five variables in its laundry list.  The indicated direction of mass flow simply denotes the direction of positive mass flow.  However if the control signal of the rate assumes a negative value, mass will flow in the opposite direction.
</html>"), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics));
    end MRate_5;
  end Rates;
  package Sources "Sources and sinks of the System Dynamics methodology"
    extends Modelica.Icons.Library;
    block Source "This is the (dummy) source model of System Dynamics"
      SystemDynamics.Interfaces.MassInPort MassInPort1 "Outflow variable" annotation(Placement(visible = true, transformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-8.0},{8.0,80.0}}),Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-10.0,-10.0},{80.0,80.0}}),Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-80.0},{10.0,10.0}}),Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-10.0,-80.0},{80.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {191,127,255}, pattern = LinePattern.None, fillPattern = FillPattern.Solid, extent = {{-62.0,-64.0},{64.0,60.0}}),Line(visible = true, points = {{64.0,0.0},{110.0,0.0}}, color = {127,0,255}, thickness = 0.5)}), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-8.0},{8.0,80.0}}),Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-10.0,-10.0},{80.0,80.0}}),Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-80.0},{10.0,10.0}}),Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-10.0,-80.0},{80.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {191,127,255}, pattern = LinePattern.None, fillPattern = FillPattern.Solid, extent = {{-62.0,-64.0},{64.0,60.0}}),Line(visible = true, points = {{64.0,0.0},{100.0,0.0}}, color = {127,0,255}, thickness = 0.5)}), Documentation(info = "<html>
This is the general <b>Source</b> model of the System Dynamics methodology.  In Modelica, this is a dummy model.  It is only provided to maintain the familiarity with the commonly used System Dynamics graphical symbols.
</html>"));
    end Source;
    block Sink "This is the (dummy) sink model of System Dynamics"
      SystemDynamics.Interfaces.MassInPort MassInPort1 "Inflow variable" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-8.0},{8.0,80.0}}),Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-10.0,-10.0},{80.0,80.0}}),Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-80.0},{10.0,10.0}}),Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-10.0,-80.0},{80.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {191,127,255}, pattern = LinePattern.None, fillPattern = FillPattern.Solid, extent = {{-62.0,-64.0},{64.0,60.0}}),Line(visible = true, points = {{-100.0,0.0},{-64.0,0.0}}, color = {127,0,255}, thickness = 0.5)}), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-8.0},{8.0,80.0}}),Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-10.0,-10.0},{80.0,80.0}}),Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-80.0,-80.0},{10.0,10.0}}),Ellipse(visible = true, lineColor = {127,0,255}, fillColor = {191,127,255}, fillPattern = FillPattern.Solid, extent = {{-10.0,-80.0},{80.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {191,127,255}, pattern = LinePattern.None, fillPattern = FillPattern.Solid, extent = {{-62.0,-64.0},{64.0,60.0}}),Line(visible = true, points = {{-100.0,0.0},{-64.0,0.0}}, color = {127,0,255}, thickness = 0.5)}), Documentation(info = "<html>
This is the general <b>Sink</b> model of the System Dynamics methodology.  In Modelica, this is a dummy model.  It is only provided to maintain the familiarity with the commonly used System Dynamics graphical symbols.
</html>", revisions = ""));
    end Sink;
    annotation(preferedView = "info", Documentation(info = "<html>
  This package contains the (dummy) source and sink elements of the System Dynamics methodology.
  </html>"));
  end Sources;
  package IndustrialDynamics "Factory models"
    extends Modelica.Icons.Example;
    package Inventory "Inventory model"
      extends Modelica.Icons.Example;
      model Factory "Factory model"
        parameter Real AIF = 4 "Gain factor for inventory in factory";
        parameter Real ALF(unit = "1/wk") = 1000000.0 "Manufacturing capacity limit at factory";
        parameter Real DCF(unit = "wk") = 1 "Delay clerical";
        parameter Real DHF(unit = "wk") = 1 "Delay in handling";
        parameter Real DIF(unit = "wk") = 4 "Delay in inventory (and pipeline) adjustment";
        parameter Real DPF(unit = "wk") = 6 "Delay in production lead time at factory";
        parameter Real DRF(unit = "wk") = 8 "Delay in smoothing requisitions";
        parameter Real DUF(unit = "wk") = 1 "Delay due to unfillable orders";
        parameter Real RRFini(unit = "1/wk") = 1000 "initial requests at retail";
        final parameter Real RSFini = RRFini;
        final parameter Real IAFini = AIF * RSFini;
        final parameter Real UOFini = RSFini * (DHF + DUF);
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {-100.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 MDF "New orders arriving" annotation(Placement(visible = true, transformation(origin = {-75.0,-40.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.MDF_d MDF_d(ALF = ALF) "Manufacturing rate at factory" annotation(Placement(visible = true, transformation(origin = {-75.095,-75.095}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Levels.Level IAF(x0 = IAFini) "Produced goods in factory" annotation(Placement(visible = true, transformation(origin = {30.5,-40.0}, extent = {{-15.5,-16.0},{15.5,16.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 SSF "Goods leaving factory" annotation(Placement(visible = true, transformation(origin = {75.0,-40.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.SSF_d SSF_d "Delivery rate at factory" annotation(Placement(visible = true, transformation(origin = {75.095,-73.652}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Interfaces.MassOutPort ssf(unit = "1/wk") "Manufactured goods leaving factory" annotation(Placement(visible = true, transformation(origin = {110.0,-30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort rrf(unit = "1/wk") "New order demands" annotation(Placement(visible = true, transformation(origin = {110.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Levels.Level UOF(x0 = UOFini) "Unfilled orders" annotation(Placement(visible = true, transformation(origin = {65.0,90.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 OFD "Orders filled due to delivery" annotation(Placement(visible = true, transformation(origin = {25.0,90.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink1 annotation(Placement(visible = true, transformation(origin = {-10.0,90.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.MWF MWF(DIF = DIF) "Manufacturing demand at factory" annotation(Placement(visible = true, transformation(origin = {-100.0,-10.0}, extent = {{-10.0,10.0},{10.0,-10.0}}, rotation = 180)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.MWF1 MWF1 "Partial manufacturing adjustment for factory" annotation(Placement(visible = true, transformation(origin = {-80.0,20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
        SystemDynamics.Auxiliary.Gain UNF(k = DHF + DUF) "Desired unfilled orders" annotation(Placement(visible = true, transformation(origin = {16.0,-4.0}, extent = {{27.0,13.0},{-27.0,-13.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain IDF(k = AIF) "Desired produced goods in factory" annotation(Placement(visible = true, transformation(origin = {-1.0,50.0}, extent = {{-15.0,-10.0},{15.0,10.0}}, rotation = -180)));
        Modelica.Blocks.Math.Add LAF "Actual delays in order processing" annotation(Placement(visible = true, transformation(origin = {-40.25,18.75}, extent = {{-8.75,-7.75},{8.75,7.75}}, rotation = -270)));
        SystemDynamics.Auxiliary.Gain LDF(k = DCF + DPF) "Desired delays in order processing" annotation(Placement(visible = true, transformation(origin = {16.0,25.0}, extent = {{27.0,13.0},{-27.0,-13.0}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.DFF DFF(DHF = DHF, DUF = DUF) "Desired delay in filling orders" annotation(Placement(visible = true, transformation(origin = {44.3566,-110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division STF "Desired order filling rate" annotation(Placement(visible = true, transformation(origin = {70.5,-105.5}, extent = {{-7.5,-7.5},{7.5,7.5}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort uof "Unfilled orders" annotation(Placement(visible = true, transformation(origin = {110.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort dff(unit = "wk") "Desired delay in filling orders" annotation(Placement(visible = true, transformation(origin = {110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.DELAY1 SRF_delayed(averaging_time(unit = "wk") = DRF, delay_init = RSFini * DRF) "Delayed new orders" annotation(Placement(visible = true, transformation(origin = {70.0,25.0}, extent = {{14.0,-10.0},{-14.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.DELAY3 Order_Delay(averaging_time(unit = "wk") = DCF, delay_init = RRFini * DCF) "Delay of processing order due to capacity constraints in factory" annotation(Placement(visible = true, transformation(origin = {-45.5,-40.0}, extent = {{-14.5,-10.0},{14.5,10.0}}, rotation = 0)));
        SystemDynamics.Functions.DELAY3 Production_Delay(averaging_time(unit = "wk") = DPF, delay_init = RRFini * DPF) "Delay in processing order due to time needed for production" annotation(Placement(visible = true, transformation(origin = {-8.5,-40.0}, extent = {{-14.5,-10.0},{14.5,10.0}}, rotation = 0)));
      equation
        connect(Production_Delay.state,LAF.u2) annotation(Line(visible = true, points = {{-8.5,-29.0},{-8.0,-29.0},{-8.0,-25.0},{-35.6,-25.0},{-35.6,8.25}}, color = {0,0,191}));
        connect(STF.y,SSF_d.u2) annotation(Line(visible = true, points = {{78.75,-105.5},{85.0,-105.5},{85.0,-90.0},{79.0,-90.0},{79.095,-82.652}}, color = {0,0,191}));
        connect(SSF_d.u1,IAF.y2) annotation(Line(visible = true, points = {{71.095,-82.652},{71.0,-90.0},{42.0,-90.0},{42.0,-50.4},{42.125,-50.4}}, color = {0,0,191}));
        connect(SSF_d.y,OFD.u) annotation(Line(visible = true, points = {{75.095,-64.652},{75.0,-60.0},{120.0,-60.0},{120.0,110.0},{-25.0,110.0},{-25.0,70.0},{25.0,70.0},{25.0,74.25}}, color = {0,0,191}));
        connect(SSF_d.y,SSF.u) annotation(Line(visible = true, points = {{75.095,-64.652},{75.0,-55.75}}, color = {0,0,191}));
        connect(DFF.y,dff) annotation(Line(visible = true, points = {{53.0,-110.0},{57.0,-110.0},{57.0,-120.0},{90.0,-120.0},{90.0,-80.0},{110.0,-80.0}}, color = {0,0,191}));
        connect(STF.u2,DFF.y) annotation(Line(visible = true, points = {{61.5,-110.0},{53.0,-110.0}}, color = {0,0,191}));
        connect(IDF.y,DFF.u2) annotation(Line(visible = true, points = {{-11.5,50.0},{-125.0,50.0},{-125.0,-114.0},{35.0,-114.0}}, color = {0,0,191}));
        connect(DFF.u1,IAF.y3) annotation(Line(visible = true, points = {{35.0,-106.0},{30.5,-106.0},{30.5,-50.4}}, color = {0,0,191}));
        connect(MWF.y,MDF_d.u) annotation(Line(visible = true, points = {{-109.0,-10.0},{-115.0,-10.0},{-115.0,-90.0},{-75.0,-90.0},{-75.095,-84.095}}, color = {0,0,191}));
        connect(MDF_d.y,MDF.u) annotation(Line(visible = true, points = {{-75.095,-66.095},{-75.0,-55.75}}, color = {0,0,191}));
        connect(LDF.y,MWF1.u1) annotation(Line(visible = true, points = {{-2.9,25.0},{-20.0,25.0},{-20.0,35.0},{-74.0,35.0},{-74.0,29.0}}, color = {0,0,191}));
        connect(MWF1.u3,IDF.y) annotation(Line(visible = true, origin = {-58.5,43.0}, points = {{-23.5,-14.0},{-23.5,7.0},{47.0,7.0}}, color = {0,0,191}));
        connect(LAF.y,MWF1.u2) annotation(Line(visible = true, points = {{-40.25,28.375},{-40.25,45.0},{-78.0,45.0},{-78.0,29.0}}, color = {0,0,191}));
        connect(MWF1.u4,IAF.y4) annotation(Line(visible = true, points = {{-86.0,29.0},{-86.0,35.0},{-120.0,35.0},{-120.0,-95.0},{18.875,-95.0},{18.875,-50.4}}, color = {0,0,191}));
        connect(MWF1.y,MWF.u1) annotation(Line(visible = true, origin = {-83.6667,1.0}, points = {{3.6667,10.0},{3.6667,-5.0},{-7.3333,-5.0}}, color = {0,0,191}));
        connect(MWF.u4,SRF_delayed.massInPort) annotation(Line(visible = true, points = {{-91.0,-16.0},{90.0,-16.0},{90.0,25.0},{85.4,25.0}}, color = {0,0,191}));
        connect(UNF.y,MWF.u3) annotation(Line(visible = true, points = {{-2.9,-4.0},{-55.0,-4.0},{-55.0,-12.0},{-91.0,-12.0}}, color = {0,0,191}));
        connect(MWF.u2,UOF.y2) annotation(Line(visible = true, points = {{-91.0,-8.0},{-60.0,-8.0},{-60.0,65.0},{54.0,65.0},{54.0,79.6},{53.75,79.6}}, color = {0,0,191}));
        connect(MWF.u4,rrf) annotation(Line(visible = true, points = {{-91.0,-16.0},{90.0,-16.0},{90.0,80.0},{110.0,80.0}}, color = {0,0,191}));
        connect(Order_Delay.state,LAF.u1) annotation(Line(visible = true, points = {{-45.5,-29.0},{-44.9,8.25}}, color = {0,0,191}));
        connect(MDF.y1,Order_Delay.massInPort) annotation(Line(visible = true, points = {{-67.5,-40.0},{-61.45,-40.0}}, color = {95,0,191}));
        connect(Order_Delay.massOutPort,Production_Delay.massInPort) annotation(Line(visible = true, points = {{-29.55,-40.0},{-24.45,-40.0}}, color = {95,0,191}));
        connect(Production_Delay.massOutPort,IAF.u1) annotation(Line(visible = true, points = {{7.45,-40.0},{13.45,-40.0}}, color = {95,0,191}));
        connect(IDF.u,SRF_delayed.massOutPort) annotation(Line(visible = true, points = {{9.5,50.0},{45.0,50.0},{45.0,25.0},{54.6,25.0}}, color = {0,0,191}));
        connect(UNF.u,SRF_delayed.massOutPort) annotation(Line(visible = true, points = {{34.9,-4.0},{45.0,-4.0},{45.0,25.0},{54.6,25.0}}, color = {0,0,191}));
        connect(LDF.u,SRF_delayed.massOutPort) annotation(Line(visible = true, points = {{34.9,25.0},{54.6,25.0}}, color = {0,0,191}));
        connect(UOF.u1,rrf) annotation(Line(visible = true, points = {{81.5,90.0},{90.0,90.0},{90.0,80.0},{110.0,80.0}}, color = {95,0,191}));
        connect(uof,UOF.y4) annotation(Line(visible = true, points = {{110.0,30.0},{95.0,30.0},{95.0,60.0},{76.25,60.0},{76.25,79.6}}, color = {0,0,191}));
        connect(SSF.y1,ssf) annotation(Line(visible = true, points = {{82.5,-40.0},{90.0,-40.0},{90.0,-30.0},{110.0,-30.0}}, color = {95,0,191}));
        connect(STF.u1,UOF.y3) annotation(Line(visible = true, points = {{61.5,-101.0},{50.0,-101.0},{50.0,60.0},{65.0,60.0},{65.0,79.6}}, color = {0,0,191}));
        connect(Sink1.MassInPort1,OFD.y1) annotation(Line(visible = true, points = {{1.0,90.0},{17.5,90.0}}, color = {95,0,191}));
        connect(OFD.y,UOF.u2) annotation(Line(visible = true, points = {{32.5,90.0},{48.5,90.0}}, color = {95,0,191}));
        connect(IAF.u2,SSF.y) annotation(Line(visible = true, points = {{47.55,-40.0},{67.5,-40.0}}, color = {95,0,191}));
        connect(Source1.MassInPort1,MDF.y) annotation(Line(visible = true, points = {{-89.0,-40.0},{-82.5,-40.0}}, color = {95,0,191}));
        annotation(Icon(coordinateSystem(extent = {{-125.0,-125.0},{125.0,125.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, extent = {{-100.0,-100.0},{100.0,100.0}}),Polygon(visible = true, lineColor = {0,0,255}, fillColor = {159,223,159}, fillPattern = FillPattern.Solid, points = {{-70.0,0.0},{-70.0,-60.0},{80.0,-60.0},{80.0,0.0},{60.0,20.0},{50.0,0.0},{30.0,20.0},{20.0,0.0},{0.0,20.0},{-10.0,0.0},{-30.0,20.0},{-40.0,0.0},{-60.0,20.0},{-70.0,0.0},{-70.0,0.0}}),Polygon(visible = true, lineColor = {0,0,255}, fillColor = {159,223,159}, fillPattern = FillPattern.Solid, points = {{-50.0,10.0},{-40.0,0.0},{-34.0,12.0},{-34.0,70.0},{-50.0,70.0},{-50.0,10.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-41.0,-60.0},{-20.0,-29.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-6.0,-20.0},{14.0,-9.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{35.0,-20.0},{55.0,-9.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-5.0,-48.0},{15.0,-37.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-46.0,-19.0},{-26.0,-8.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{35.0,-48.0},{55.0,-37.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-0.0,65.0},{0.0,101.0}}, textString = "%name", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-125.0,-125.0},{125.0,125.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{74.0,-61.0},{76.0,-59.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{89.0,79.0},{91.0,81.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{89.0,24.0},{91.0,26.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{44.0,24.0},{46.0,26.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-83.0,49.0},{-81.0,51.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{56.0,-111.0},{58.0,-109.0}})}), Documentation(info = "<html>
This model describes the factory floor.  Goods under production are represented by the mass-flow model in the lower part of the graph, whereas incoming orders for these goods are represented by the mass-flow model in the upper-right part of the graph.  The signal-flow model represents the control of the production of new items based on the information obtained from the incoming orders.
</html>"));
      end Factory;
      model Distribution "Distribution model"
        parameter Real AID = 6 "Gain factor for inventory in distribution";
        parameter Real DCD(unit = "wk") = 2 "Delay clerical";
        parameter Real DHD(unit = "wk") = 1 "Delay in handling";
        parameter Real DID(unit = "wk") = 4 "Delay in inventory (and pipeline) adjustment";
        parameter Real DMD(unit = "wk") = 0.5 "Mail delay";
        parameter Real DRD(unit = "wk") = 8 "Delay in smoothing requisitions";
        parameter Real DUD(unit = "wk") = 0.6 "Delay due to unfillable orders";
        parameter Real DTD(unit = "wk") = 2 "Transportation delay";
        parameter Real RRDini(unit = "1/wk") = 1000 "Initial requests at retail";
        final parameter Real RSDini = RRDini;
        final parameter Real IADini = AID * RSDini;
        final parameter Real UODini = RSDini * (DHD + DUD);
        SystemDynamics.Interfaces.MassOutPort rrf(unit = "1/wk") "New order demands" annotation(Placement(visible = true, transformation(origin = {-110.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort ssf(unit = "1/wk") "Manufactured goods leaving factory" annotation(Placement(visible = true, transformation(origin = {-110.0,-30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort uof "Unfilled orders" annotation(Placement(visible = true, transformation(origin = {-110.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort dff(unit = "wk") "Desired delay in filling orders" annotation(Placement(visible = true, transformation(origin = {-110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Levels.Level IAD(x0 = IADini) "Produced goods in distribution" annotation(Placement(visible = true, transformation(origin = {30.0,-30.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 SSD "Goods leaving distribution" annotation(Placement(visible = true, transformation(origin = {70.0,-30.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort ssd(unit = "1/wk") "Goods leaving distribution" annotation(Placement(visible = true, transformation(origin = {110.0,-30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.SSD_d SSD_d "Delivery rate at distribution" annotation(Placement(visible = true, transformation(origin = {70.0,-63.0331}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Rates.Rate_1 PPD "Orders leaving distribution" annotation(Placement(visible = true, transformation(origin = {-20.0,100.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {10.0,100.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort rrd(unit = "1/wk") "New order demands" annotation(Placement(visible = true, transformation(origin = {110.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Levels.Level UOD(x0 = UODini) "Unfilled orders" annotation(Placement(visible = true, transformation(origin = {75.0,80.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 ODD "Orders filled due to delivery" annotation(Placement(visible = true, transformation(origin = {40.0,80.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink1 annotation(Placement(visible = true, transformation(origin = {10.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.PPD1 PPD1(DID = DID) "Manufacturing demand at distribution" annotation(Placement(visible = true, transformation(origin = {-20.0,70.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 270)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.PPD2 PPD2 "Partial manufacturing adjustment for distribution" annotation(Placement(visible = true, transformation(origin = {-0.0,15.1746}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain UND(k = DHD + DUD) "Desired unfilled orders" annotation(Placement(visible = true, transformation(origin = {34.0,37.0}, extent = {{27.0,13.0},{-27.0,-13.0}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.LAD LAD "Actual delays in order processing" annotation(Placement(visible = true, transformation(origin = {-70.0,31.9455}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.LDD LDD(DCD = DCD, DMD = DMD, DTD = DTD) "Desired delay in order processing" annotation(Placement(visible = true, transformation(origin = {40.0,14.0074}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain IDD(k = AID) "Desired produced goods in distribution" annotation(Placement(visible = true, transformation(origin = {34.0,-5.0}, extent = {{-15.0,-10.0},{15.0,10.0}}, rotation = -180)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.DFD DFD(DHD = DHD, DUD = DUD) "Desired delay in filling orders" annotation(Placement(visible = true, transformation(origin = {-14.3597,-95.3278}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division STD "Desired order filling rate" annotation(Placement(visible = true, transformation(origin = {14.5,-90.5}, extent = {{-7.5,-7.5},{7.5,7.5}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort uod "Unfilled orders" annotation(Placement(visible = true, transformation(origin = {110.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort dfd(unit = "wk") "Desired delay in filling orders" annotation(Placement(visible = true, transformation(origin = {110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.DELAY1 RSD_delayed(averaging_time(unit = "wk") = DRD, delay_init = RSDini * DRD) "Delayed new orders" annotation(Placement(visible = true, transformation(origin = {79.0,10.0}, extent = {{14.0,-10.0},{-14.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.DELAY3 Transporation_Delay(averaging_time(unit = "wk") = DTD, delay_init = RRDini * DTD) "Transporation_Delay" annotation(Placement(visible = true, transformation(origin = {-65.0,-30.0}, extent = {{-15.0,-10.0},{15.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.DELAY3 Order_Delay(averaging_time(unit = "wk") = DCD, delay_init = RRDini * DCD) "Order_Delay" annotation(Placement(visible = true, transformation(origin = {-52.0,100.0}, extent = {{15.0,10.0},{-15.0,-10.0}}, rotation = 0)));
        SystemDynamics.Functions.DELAY3 Mail_Delay(averaging_time(unit = "wk") = DMD, delay_init = RRDini * DMD) "Mail_Delay" annotation(Placement(visible = true, transformation(origin = {-92.0,100.0}, extent = {{15.0,10.0},{-15.0,-10.0}}, rotation = 0)));
      equation
        connect(DFD.y,dfd) annotation(Line(visible = true, points = {{-5.3597,-95.3278},{0.0,-95.0},{0.0,-115.0},{95.0,-115.0},{95.0,-80.0},{110.0,-80.0}}, color = {0,0,191}));
        connect(STD.u2,DFD.y) annotation(Line(visible = true, points = {{5.5,-95.0},{-5.3597,-95.3278}}, color = {0,0,191}));
        connect(IDD.y,DFD.u2) annotation(Line(visible = true, points = {{23.5,-5.0},{-35.0,-5.0},{-35.0,-99.0},{-23.3597,-99.3278}}, color = {0,0,191}));
        connect(DFD.u1,IAD.y4) annotation(Line(visible = true, points = {{-23.3597,-91.3278},{-30.0,-91.0},{-30.0,-50.0},{18.0,-50.0},{18.0,-40.4}}, color = {0,0,191}));
        connect(SSD_d.u2,STD.y) annotation(Line(visible = true, points = {{74.0,-72.0331},{74.0,-90.5},{22.75,-90.5}}, color = {0,0,191}));
        connect(SSD_d.y,ODD.u) annotation(Line(visible = true, points = {{70.0,-54.0331},{70.0,-50.0},{120.0,-50.0},{120.0,60.0},{40.0,60.0},{40.0,64.25}}, color = {0,0,191}));
        connect(SSD_d.u1,IAD.y2) annotation(Line(visible = true, points = {{66.0,-72.0331},{66.0,-80.0},{42.0,-80.0},{42.0,-40.4}}, color = {0,0,191}));
        connect(SSD.u,SSD_d.y) annotation(Line(visible = true, points = {{70.0,-45.75},{70.0,-54.0331}}, color = {0,0,191}));
        connect(LDD.u1,RSD_delayed.massOutPort) annotation(Line(visible = true, points = {{49.0,18.0074},{60.0,18.0},{60.0,10.0},{63.6,10.0}}, color = {0,0,191}));
        connect(dff,LDD.u2) annotation(Line(visible = true, points = {{-110.0,-80.0},{-90.0,-80.0},{-90.0,-110.0},{90.0,-110.0},{90.0,-10.0},{55.0,-10.0},{55.0,10.0},{49.0,10.0074}}, color = {0,0,191}));
        connect(LDD.y,PPD2.u1) annotation(Line(visible = true, points = {{31.0,14.0074},{25.0,14.0},{25.0,21.0},{9.0,21.1746}}, color = {0,0,191}));
        connect(PPD2.y,PPD1.u1) annotation(Line(visible = true, points = {{-9.0,15.1746},{-14.0,15.0},{-14.0,61.0}}, color = {0,0,191}));
        connect(IDD.y,PPD2.u3) annotation(Line(visible = true, points = {{23.5,-5.0},{20.0,-5.0},{20.0,13.0},{9.0,13.1746}}, color = {0,0,191}));
        connect(PPD2.u2,LAD.y) annotation(Line(visible = true, points = {{9.0,17.1746},{13.0,17.0},{13.0,32.0},{-61.0,31.9455}}, color = {0,0,191}));
        connect(PPD2.u4,IAD.y) annotation(Line(visible = true, points = {{9.0,9.1746},{13.0,9.0},{13.0,-21.2},{16.4,-21.2}}, color = {0,0,191}));
        connect(PPD1.u4,RSD_delayed.massInPort) annotation(Line(visible = true, points = {{-26.0,61.0},{-26.0,55.0},{-120.0,55.0},{-120.0,116.0},{97.0,116.0},{97.0,10.0},{94.4,10.0}}, color = {0,0,191}));
        connect(UND.y,PPD1.u3) annotation(Line(visible = true, points = {{15.1,37.0},{-22.0,37.0},{-22.0,61.0}}, color = {0,0,191}));
        connect(PPD1.u2,UOD.y2) annotation(Line(visible = true, points = {{-18.0,61.0},{-18.0,55.0},{63.75,55.0},{63.75,69.6}}, color = {0,0,191}));
        connect(PPD1.u4,rrd) annotation(Line(visible = true, points = {{-26.0,61.0},{-26.0,55.0},{-120.0,55.0},{-120.0,116.0},{97.0,116.0},{97.0,80.0},{110.0,80.0}}, color = {0,0,191}));
        connect(PPD1.y,PPD.u) annotation(Line(visible = true, origin = {-20.0,81.625}, points = {{0.0,-2.625},{0.0,2.625}}, color = {0,0,191}));
        connect(LAD.u2,Mail_Delay.state) annotation(Line(visible = true, points = {{-79.0,33.9455},{-92.0,34.0},{-92.0,89.0}}, color = {0,0,191}));
        connect(LAD.u1,Order_Delay.state) annotation(Line(visible = true, points = {{-79.0,37.9455},{-85.0,38.0},{-85.0,50.0},{-52.0,50.0},{-52.0,89.0}}, color = {0,0,191}));
        connect(Transporation_Delay.state,LAD.u4) annotation(Line(visible = true, points = {{-65.0,-19.0},{-65.0,-6.0},{-85.0,-6.0},{-85.0,26.0},{-79.0,25.9455}}, color = {0,0,191}));
        connect(uof,LAD.u3) annotation(Line(visible = true, points = {{-110.0,30.0},{-79.0,29.9455}}, color = {0,0,191}));
        connect(Mail_Delay.massOutPort,rrf) annotation(Line(visible = true, points = {{-108.5,100.0},{-118.0,100.0},{-118.0,70.0},{-95.0,70.0},{-95.0,80.0},{-110.0,80.0}}, color = {95,0,191}));
        connect(Mail_Delay.massInPort,Order_Delay.massOutPort) annotation(Line(visible = true, points = {{-75.5,100.0},{-68.5,100.0}}, color = {95,0,191}));
        connect(PPD.y1,Order_Delay.massInPort) annotation(Line(visible = true, points = {{-27.5,100.0},{-35.5,100.0}}, color = {95,0,191}));
        connect(Transporation_Delay.massOutPort,IAD.u1) annotation(Line(visible = true, points = {{-48.5,-30.0},{12.4,-30.0}}, color = {95,0,191}));
        connect(Transporation_Delay.massInPort,ssf) annotation(Line(visible = true, points = {{-81.5,-30.0},{-110.0,-30.0}}, color = {95,0,191}));
        connect(UND.u,RSD_delayed.massOutPort) annotation(Line(visible = true, points = {{52.9,37.0},{60.0,37.0},{60.0,10.0},{63.6,10.0}}, color = {0,0,191}));
        connect(IDD.u,RSD_delayed.massOutPort) annotation(Line(visible = true, points = {{44.5,-5.0},{60.0,-5.0},{60.0,10.0},{63.6,10.0}}, color = {0,0,191}));
        connect(uod,UOD.y4) annotation(Line(visible = true, points = {{110.0,30.0},{86.25,30.0},{86.25,69.6}}, color = {0,0,191}));
        connect(STD.u1,UOD.y1) annotation(Line(visible = true, points = {{5.5,-86.0},{0.0,-86.0},{0.0,-60.0},{-125.0,-60.0},{-125.0,120.0},{55.0,120.0},{55.0,88.8},{62.25,88.8}}, color = {0,0,191}));
        connect(UOD.u1,rrd) annotation(Line(visible = true, points = {{91.5,80.0},{110.0,80.0}}, color = {95,0,191}));
        connect(ODD.y1,Sink1.MassInPort1) annotation(Line(visible = true, points = {{32.5,80.0},{21.0,80.0}}, color = {95,0,191}));
        connect(UOD.u2,ODD.y) annotation(Line(visible = true, points = {{58.5,80.0},{47.5,80.0}}, color = {95,0,191}));
        connect(Source1.MassInPort1,PPD.y) annotation(Line(visible = true, points = {{-1.0,100.0},{-12.5,100.0}}, color = {95,0,191}));
        connect(SSD.y1,ssd) annotation(Line(visible = true, points = {{77.5,-30.0},{110.0,-30.0}}, color = {95,0,191}));
        connect(SSD.y,IAD.u2) annotation(Line(visible = true, points = {{62.5,-30.0},{47.6,-30.0}}, color = {95,0,191}));
        annotation(Icon(coordinateSystem(extent = {{-125.0,-125.0},{125.0,125.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, extent = {{-100.0,-100.0},{100.0,100.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-0.0,65.0},{0.0,101.0}}, textString = "%name", fontName = "Arial"),Polygon(visible = true, lineColor = {0,0,255}, fillColor = {191,223,159}, fillPattern = FillPattern.Solid, points = {{-70.0,-60.0},{70.0,-60.0},{70.0,0.0},{0.0,20.0},{-70.0,0.0},{-70.0,-60.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-48.0,-60.0},{-26.0,-32.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-6.0,-2.0},{8.0,11.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{0.0,-80.0},{10.0,-70.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {192,192,192}, fillPattern = FillPattern.Solid, extent = {{3.0,-77.0},{8.0,-73.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{40.0,-80.0},{50.0,-70.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {192,192,192}, fillPattern = FillPattern.Solid, extent = {{43.0,-77.0},{48.0,-73.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{11.0,-80.0},{21.0,-70.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {192,192,192}, fillPattern = FillPattern.Solid, extent = {{14.0,-77.0},{19.0,-73.0}}),Polygon(visible = true, lineColor = {0,0,255}, fillColor = {160,160,160}, fillPattern = FillPattern.Solid, points = {{5.0,-63.0},{-5.0,-63.0},{-5.0,-68.0},{20.0,-68.0},{24.0,-74.0},{38.0,-74.0},{41.0,-69.0},{49.0,-69.0},{52.0,-74.0},{55.0,-74.0},{55.0,-63.0},{42.0,-61.0},{40.0,-54.0},{32.0,-54.0},{32.0,-63.0},{5.0,-63.0}})}), Diagram(coordinateSystem(extent = {{-125.0,-125.0},{125.0,125.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{69.0,-51.0},{71.0,-49.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{96.0,79.0},{98.0,81.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{59.0,17.0},{61.0,19.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{59.0,9.0},{61.0,11.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{19.0,-6.0},{21.0,-4.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-1.0,-96.0},{1.0,-94.0}})}), Documentation(info = "<html>
This model describes the distribution network of produced goods.  Goods that are being distributed are represented by the mass-flow model in the lower part of the graph, whereas the orders for these goods are represented by the mass-flow models in the upper part of the graph.  The signal-flow model represents the control of the flow of goods based on the information obtained from their order status.
</html>"));
      end Distribution;
      model Retail "Retail model"
        parameter Real AIR = 8 "Gain factor for inventory in retail";
        parameter Real DCR(unit = "wk") = 3 "Delay clerical";
        parameter Real DHR(unit = "wk") = 1 "Delay in handling at retail";
        parameter Real DIR(unit = "wk") = 4 "Delay in inventory (and pipeline) adjustment";
        parameter Real DMR(unit = "wk") = 0.5 "Mail delay";
        parameter Real DRR(unit = "wk") = 8 "Delay in smoothing requisitions";
        parameter Real DTR(unit = "wk") = 1 "Transportation delay";
        parameter Real DUR(unit = "wk") = 0.4 "Delay due to unfillable orders";
        parameter Real RRRini(unit = "1/wk") = 1000 "Initial requests at retail";
        final parameter Real RSRini = RRRini;
        final parameter Real IARini = AIR * RSRini;
        final parameter Real UORini = RSRini * (DHR + DUR);
        SystemDynamics.Levels.Level IAR(x0 = IARini) "Produced goods in retail" annotation(Placement(visible = true, transformation(origin = {30.0,-30.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 SSR "Goods leaving retail" annotation(Placement(visible = true, transformation(origin = {70.0,-30.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.SSD_d SSR_d "Delivery rate at retail" annotation(Placement(visible = true, transformation(origin = {70.0,-63.0331}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Rates.Rate_1 PDR "Orders leaving distribution" annotation(Placement(visible = true, transformation(origin = {-20.0,100.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {10.0,100.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Levels.Level UOR(x0 = UORini) "Unfilled orders" annotation(Placement(visible = true, transformation(origin = {75.0,80.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 ODR "Orders filled due to delivery" annotation(Placement(visible = true, transformation(origin = {40.0,80.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink1 annotation(Placement(visible = true, transformation(origin = {10.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.PPD1 PDR1(DID = DIR) "Manufacturing demand at retail" annotation(Placement(visible = true, transformation(origin = {-20.0,70.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 270)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.PPD2 PDR2 "Partial manufacturing adjustment for distribution" annotation(Placement(visible = true, transformation(origin = {0.0,14.7855}, extent = {{-10.0,10.0},{10.0,-10.0}}, rotation = 180)));
        SystemDynamics.Auxiliary.Gain UNR(k = DHR + DUR) "Desired unfilled orders" annotation(Placement(visible = true, transformation(origin = {34.0,37.0}, extent = {{27.0,13.0},{-27.0,-13.0}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.LAD LAR "Actual delays in order processing" annotation(Placement(visible = true, transformation(origin = {-40.0,31.5165}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.LDD LDR(DCD = DCR, DMD = DMR, DTD = DTR) "Desired delay in order processing" annotation(Placement(visible = true, transformation(origin = {42.4112,13.6183}, extent = {{-10.0,10.0},{10.0,-10.0}}, rotation = 180)));
        SystemDynamics.Auxiliary.Gain IDR(k = AIR) "Desired produced goods in retail" annotation(Placement(visible = true, transformation(origin = {34.0,-5.0}, extent = {{-15.0,-10.0},{15.0,10.0}}, rotation = -180)));
        SystemDynamics.IndustrialDynamics.Inventory.Utilities.DFD DFR(DHD = DHR, DUD = DUR) "Desired delay in filling orders" annotation(Placement(visible = true, transformation(origin = {-12.451,-94.9387}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division STR "Desired order filling rate" annotation(Placement(visible = true, transformation(origin = {14.5,-90.5}, extent = {{-7.5,-7.5},{7.5,7.5}}, rotation = 0)));
        SystemDynamics.Functions.DELAY1 RSR_delayed(averaging_time(unit = "wk") = DRR, delay_init = RSRini * DRR) "Delayed new orders" annotation(Placement(visible = true, transformation(origin = {79.0,10.0}, extent = {{14.0,-10.0},{-14.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.DELAY3 Transporation_Delay(averaging_time(unit = "wk") = DTR, delay_init = RRRini * DTR) "Transporation_Delay" annotation(Placement(visible = true, transformation(origin = {-65.0,-30.0}, extent = {{-15.0,-10.0},{15.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.DELAY3 Order_Delay(averaging_time(unit = "wk") = DCR, delay_init = RRRini * DCR) "Order_Delay" annotation(Placement(visible = true, transformation(origin = {-52.0,100.0}, extent = {{15.0,10.0},{-15.0,-10.0}}, rotation = 0)));
        SystemDynamics.Functions.DELAY3 Mail_Delay(averaging_time(unit = "wk") = DMR, delay_init = RRRini * DMR) "Mail_Delay" annotation(Placement(visible = true, transformation(origin = {-92.0,100.0}, extent = {{15.0,10.0},{-15.0,-10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort rrd(unit = "1/wk") "New order demands" annotation(Placement(visible = true, transformation(origin = {-110.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort ssd(unit = "1/wk") "Manufactured goods arriving at retail" annotation(Placement(visible = true, transformation(origin = {-110.0,-30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort uod "Unfilled orders" annotation(Placement(visible = true, transformation(origin = {-110.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort dfd(unit = "wk") "Desired delay in filling orders" annotation(Placement(visible = true, transformation(origin = {-110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort rrr(unit = "1/wk") "New order demands" annotation(Placement(visible = true, transformation(origin = {110.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort ssr(unit = "1/wk") "Goods leaving retail" annotation(Placement(visible = true, transformation(origin = {110.0,-30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      equation
        connect(SSR_d.u2,STR.y) annotation(Line(visible = true, points = {{74.0,-72.0331},{74.0,-90.5},{22.75,-90.5}}, color = {0,0,191}));
        connect(SSR_d.y,ODR.u) annotation(Line(visible = true, points = {{70.0,-54.0331},{70.0,-50.0},{120.0,-50.0},{120.0,60.0},{40.0,60.0},{40.0,64.25}}, color = {0,0,191}));
        connect(SSR_d.u1,IAR.y2) annotation(Line(visible = true, points = {{66.0,-72.0331},{66.0,-80.0},{42.0,-80.0},{42.0,-40.4}}, color = {0,0,191}));
        connect(SSR.u,SSR_d.y) annotation(Line(visible = true, points = {{70.0,-45.75},{70.0,-54.0331}}, color = {0,0,191}));
        connect(PDR2.y,PDR1.u1) annotation(Line(visible = true, points = {{-9.0,14.7855},{-14.0,15.0},{-14.0,61.0}}, color = {0,0,191}));
        connect(IDR.y,PDR2.u3) annotation(Line(visible = true, points = {{23.5,-5.0},{20.0,-5.0},{20.0,13.0},{9.0,12.7855}}, color = {0,0,191}));
        connect(LDR.y,PDR2.u1) annotation(Line(visible = true, points = {{33.4112,13.6183},{25.0,14.0},{25.0,21.0},{9.0,20.7855}}, color = {0,0,191}));
        connect(PDR2.u2,LAR.y) annotation(Line(visible = true, points = {{9.0,16.7855},{13.0,17.0},{13.0,32.0},{-31.0,31.5165}}, color = {0,0,191}));
        connect(PDR2.u4,IAR.y) annotation(Line(visible = true, points = {{9.0,8.7855},{13.0,9.0},{13.0,-21.2},{16.4,-21.2}}, color = {0,0,191}));
        connect(LDR.u1,RSR_delayed.massOutPort) annotation(Line(visible = true, points = {{51.4112,17.6183},{60.0,18.0},{60.0,10.0},{63.6,10.0}}, color = {0,0,191}));
        connect(dfd,LDR.u2) annotation(Line(visible = true, points = {{-110.0,-80.0},{-90.0,-80.0},{-90.0,-110.0},{90.0,-110.0},{90.0,-10.0},{55.0,-10.0},{55.0,10.0},{51.4112,9.6183}}, color = {0,0,191}));
        connect(LAR.u2,Mail_Delay.state) annotation(Line(visible = true, points = {{-49.0,33.5165},{-92.0,34.0},{-92.0,89.0}}, color = {0,0,191}));
        connect(LAR.u1,Order_Delay.state) annotation(Line(visible = true, points = {{-49.0,37.5165},{-85.0,38.0},{-85.0,50.0},{-52.0,50.0},{-52.0,89.0}}, color = {0,0,191}));
        connect(Transporation_Delay.state,LAR.u4) annotation(Line(visible = true, points = {{-65.0,-19.0},{-65.0,-6.0},{-85.0,-6.0},{-85.0,26.0},{-49.0,25.5165}}, color = {0,0,191}));
        connect(uod,LAR.u3) annotation(Line(visible = true, points = {{-110.0,30.0},{-49.0,29.5165}}, color = {0,0,191}));
        connect(PDR1.u4,rrr) annotation(Line(visible = true, points = {{-26.0,61.0},{-26.0,55.0},{-120.0,55.0},{-120.0,116.0},{97.0,116.0},{97.0,80.0},{110.0,80.0}}, color = {0,0,191}));
        connect(PDR1.u4,RSR_delayed.massInPort) annotation(Line(visible = true, points = {{-26.0,61.0},{-26.0,55.0},{-120.0,55.0},{-120.0,116.0},{97.0,116.0},{97.0,10.0},{94.4,10.0}}, color = {0,0,191}));
        connect(UNR.y,PDR1.u3) annotation(Line(visible = true, origin = {-9.6333,45.0}, points = {{24.7333,-8.0},{-12.3667,-8.0},{-12.3667,16.0}}, color = {0,0,191}));
        connect(PDR1.u2,UOR.y2) annotation(Line(visible = true, points = {{-18.0,61.0},{-18.0,55.0},{63.75,55.0},{63.75,69.6}}, color = {0,0,191}));
        connect(PDR1.y,PDR.u) annotation(Line(visible = true, origin = {-20.0,81.625}, points = {{0.0,-2.625},{0.0,2.625}}, color = {0,0,191}));
        connect(STR.u2,DFR.y) annotation(Line(visible = true, points = {{5.5,-95.0},{-3.451,-94.9387}}, color = {0,0,191}));
        connect(IDR.y,DFR.u2) annotation(Line(visible = true, points = {{23.5,-5.0},{-35.0,-5.0},{-35.0,-99.0},{-21.451,-98.9387}}, color = {0,0,191}));
        connect(DFR.u1,IAR.y4) annotation(Line(visible = true, points = {{-21.451,-90.9387},{-30.0,-91.0},{-30.0,-50.0},{18.0,-50.0},{18.0,-40.4}}, color = {0,0,191}));
        connect(ssr,SSR.y1) annotation(Line(visible = true, points = {{110.0,-30.0},{77.5,-30.0}}, color = {95,0,191}));
        connect(rrr,UOR.u1) annotation(Line(visible = true, points = {{110.0,80.0},{91.5,80.0}}, color = {95,0,191}));
        connect(Mail_Delay.massOutPort,rrd) annotation(Line(visible = true, points = {{-108.5,100.0},{-118.0,100.0},{-118.0,70.0},{-95.0,70.0},{-95.0,80.0},{-110.0,80.0}}, color = {95,0,191}));
        connect(Mail_Delay.massInPort,Order_Delay.massOutPort) annotation(Line(visible = true, points = {{-75.5,100.0},{-68.5,100.0}}, color = {95,0,191}));
        connect(PDR.y1,Order_Delay.massInPort) annotation(Line(visible = true, points = {{-27.5,100.0},{-35.5,100.0}}, color = {95,0,191}));
        connect(Transporation_Delay.massOutPort,IAR.u1) annotation(Line(visible = true, points = {{-48.5,-30.0},{12.4,-30.0}}, color = {95,0,191}));
        connect(Transporation_Delay.massInPort,ssd) annotation(Line(visible = true, points = {{-81.5,-30.0},{-110.0,-30.0}}, color = {95,0,191}));
        connect(UNR.u,RSR_delayed.massOutPort) annotation(Line(visible = true, points = {{52.9,37.0},{60.0,37.0},{60.0,10.0},{63.6,10.0}}, color = {0,0,191}));
        connect(IDR.u,RSR_delayed.massOutPort) annotation(Line(visible = true, points = {{44.5,-5.0},{60.0,-5.0},{60.0,10.0},{63.6,10.0}}, color = {0,0,191}));
        connect(STR.u1,UOR.y1) annotation(Line(visible = true, points = {{5.5,-86.0},{0.0,-86.0},{0.0,-60.0},{-125.0,-60.0},{-125.0,120.0},{55.0,120.0},{55.0,88.8},{62.25,88.8}}, color = {0,0,191}));
        connect(ODR.y1,Sink1.MassInPort1) annotation(Line(visible = true, points = {{32.5,80.0},{21.0,80.0}}, color = {95,0,191}));
        connect(UOR.u2,ODR.y) annotation(Line(visible = true, points = {{58.5,80.0},{47.5,80.0}}, color = {95,0,191}));
        connect(Source1.MassInPort1,PDR.y) annotation(Line(visible = true, points = {{-1.0,100.0},{-12.5,100.0}}, color = {95,0,191}));
        connect(SSR.y,IAR.u2) annotation(Line(visible = true, points = {{62.5,-30.0},{47.6,-30.0}}, color = {95,0,191}));
        annotation(Icon(coordinateSystem(extent = {{-125.0,-125.0},{125.0,125.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Text(visible = true, lineColor = {0,0,255}, extent = {{-0.0,65.0},{0.0,101.0}}, textString = "%name", fontName = "Arial"),Rectangle(visible = true, lineColor = {0,0,255}, extent = {{-100.0,-100.0},{100.0,100.0}}),Polygon(visible = true, lineColor = {0,0,255}, fillColor = {191,223,159}, fillPattern = FillPattern.Solid, points = {{-49.0,20.0},{-10.0,20.0},{10.0,10.0},{-34.0,10.0},{-34.0,-24.0},{0.0,-24.0},{0.0,-60.0},{-50.0,-60.0},{-50.0,20.0},{-49.0,20.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-22.0,-7.0},{-13.0,2.0}}),Polygon(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, points = {{-24.0,-24.0},{-11.0,-24.0},{-11.0,-11.0},{-18.0,-8.0},{-23.0,-11.0},{-23.0,-24.0},{-24.0,-24.0}}),Line(visible = true, points = {{-18.0,-12.0},{-9.0,-19.0},{-3.0,-16.0}}, color = {0,0,255}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {127,191,255}, fillPattern = FillPattern.Solid, extent = {{14.0,-15.0},{23.0,-6.0}}),Polygon(visible = true, lineColor = {0,0,255}, fillColor = {127,191,255}, fillPattern = FillPattern.Solid, points = {{14.0,-32.0},{25.0,-32.0},{25.0,-19.0},{18.0,-16.0},{13.0,-19.0},{13.0,-32.0},{14.0,-32.0}}),Line(visible = true, points = {{18.0,-20.0},{10.0,-24.0},{4.0,-20.0}}, color = {0,0,255}),Polygon(visible = true, lineColor = {0,0,255}, fillColor = {0,255,255}, fillPattern = FillPattern.Solid, points = {{13.0,-33.0},{25.0,-33.0},{25.0,-60.0},{21.0,-60.0},{18.0,-39.0},{12.0,-60.0},{9.0,-60.0},{13.0,-33.0},{13.0,-33.0}})}), Diagram(coordinateSystem(extent = {{-125.0,-125.0},{125.0,125.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{69.0,-51.0},{71.0,-49.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{96.0,79.0},{98.0,81.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{59.0,17.0},{61.0,19.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{59.0,9.0},{61.0,11.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{19.0,-6.0},{21.0,-4.0}})}), Documentation(info = "<html>
This model describes the retail of produced goods.  Goods that are being sold are represented by the mass-flow model in the lower part of the graph, whereas the orders for these goods are represented by the mass-flow models in the upper part of the graph.  The signal-flow model represents the control of the flow of goods based on the information obtained from their order status. <p>
 
The retail model is the same model as the distribution model except for the parameter values used.
</html>"));
      end Retail;
      model InventoryForresterNormalNoise "Inventory simulation with random orders"
        parameter Real RRRiniTop(unit = "1/wk") = 1000 "Inital value of customer requests at retail";
        parameter Real RRDiniTop(unit = "1/wk") = RRRiniTop "Inital value of requisitions received at distribution";
        parameter Real RRFiniTop(unit = "1/wk") = RRRiniTop "Inital value of requisitions received at factory";
        output Real randomNoise(unit = "1/wk") "Random noise signal";
        output Real factoryFlow(unit = "1/wk") "Manufacturing flow at factory";
        output Real retailStock "Stock of goods in retail";
        output Real distributionStock "Stock of goods in distribution";
        output Real factoryStock "Stock of goods in factory";
        SystemDynamics.IndustrialDynamics.Inventory.Factory Factory1(ALF = 1000 * RRFiniTop, RRFini = RRFiniTop) annotation(Placement(visible = true, transformation(origin = {-34.75,0.25}, extent = {{-12.75,-12.75},{12.75,12.75}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Distribution Distribution1(RRDini = RRDiniTop) annotation(Placement(visible = true, transformation(origin = {-4.75,0.25}, extent = {{-12.75,-12.75},{12.75,12.75}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Retail Retail1(RRRini = RRRiniTop) annotation(Placement(visible = true, transformation(origin = {25.25,0.25}, extent = {{-12.75,-12.75},{12.75,12.75}}, rotation = 0)));
        SystemDynamics.Functions.NoiseNormal NoiseNormal1(mean = 1000) annotation(Placement(visible = true, transformation(origin = {5.7239,30.0}, extent = {{-14.2761,-14.2761},{14.2761,14.2761}}, rotation = 0)));
      equation
        connect(Distribution1.rrf,Factory1.rrf) annotation(Line(visible = true, points = {{-15.97,8.41},{-18.985,8.41},{-18.985,8.41},{-23.53,8.41}}, color = {95,0,191}));
        connect(NoiseNormal1.y,Retail1.rrr) annotation(Line(visible = true, points = {{18.5724,30.0},{45.0,30.0},{45.0,8.41},{36.47,8.41}}, color = {0,0,191}));
        connect(Distribution1.ssd,Retail1.ssd) annotation(Line(visible = true, points = {{6.47,-2.81},{10.735,-2.81},{10.735,-2.81},{14.03,-2.81}}, color = {95,0,191}));
        connect(Distribution1.uod,Retail1.uod) annotation(Line(visible = true, points = {{6.47,3.31},{10.235,3.31},{10.235,3.31},{14.03,3.31}}, color = {0,0,191}));
        connect(Factory1.uof,Distribution1.uof) annotation(Line(visible = true, points = {{-23.53,3.31},{-19.765,3.31},{-19.765,3.31},{-15.97,3.31}}, color = {0,0,191}));
        connect(Factory1.ssf,Distribution1.ssf) annotation(Line(visible = true, points = {{-23.53,-2.81},{-19.765,-2.81},{-19.765,-2.81},{-15.97,-2.81}}, color = {95,0,191}));
        connect(Retail1.dfd,Distribution1.dfd) annotation(Line(visible = true, points = {{14.03,-7.91},{6.47,-7.91}}, color = {0,0,191}));
        connect(Retail1.rrd,Distribution1.rrd) annotation(Line(visible = true, points = {{14.03,8.41},{6.47,8.41}}, color = {95,0,191}));
        connect(Distribution1.dff,Factory1.dff) annotation(Line(visible = true, points = {{-15.97,-7.91},{-23.53,-7.91}}, color = {0,0,191}));
        randomNoise = NoiseNormal1.y;
        factoryFlow = Factory1.IAF.u1;
        retailStock = Retail1.IAR.y;
        distributionStock = Distribution1.IAD.y;
        factoryStock = Factory1.IAF.y;
        annotation(Icon(coordinateSystem(extent = {{-50.0,-50.0},{50.0,50.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Text(visible = true, lineColor = {0,0,255}, extent = {{1.0,-52.0},{1.0,-39.0}}, textString = "%name", fontName = "Arial"),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-40.0,-40.0},{40.0,50.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {127,255,0}, fillPattern = FillPattern.Solid, extent = {{-30.0,23.0},{30.0,38.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {127,255,0}, fillPattern = FillPattern.Solid, extent = {{-30.0,-2.0},{30.0,13.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {127,255,0}, fillPattern = FillPattern.Solid, extent = {{-30.0,-27.0},{30.0,-12.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-25.0,25.0},{27.0,35.0}}, textString = "Factory", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-25.0,1.0},{27.0,11.0}}, textString = "Distribution", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-25.0,-24.0},{27.0,-14.0}}, textString = "Retail", fontName = "Arial"),Line(visible = true, points = {{0.0,13.0},{0.0,23.0}}, color = {0,0,255}),Line(visible = true, points = {{0.0,-2.0},{0.0,-12.0}}, color = {0,0,255})}), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Documentation(info = "<html>
<b>Customer Demand Modeled as Noise</b> <p>
 
Customer demand usually fluctuates in a random fashion.  Therefore, it is modeled in this simulation as normally distributed random noise with a mean value of <b>mean=1000</b> and a standard deviation of <b>stdev=100</b>. The noise is sampled once per week and kept constant for the corresponding week.  The order flow is modeled using the equation:<p>
<font color=red><b>RRR(t) = RRRini + normal(1000,100);</b></font> <p>
 
<hr> <p>
 
Simulate the model across 10 years (520 weeks), and plot on a single graph the incoming orders, the production flow in the factory, and the levels of goods in retail, distribution, and the factory as functions of time: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/Inventory_1.png\"> <p>
 
Choose <font color=red><b>Radau-IIa</b></font> as your integration algorithm.  It handles noise input better than <font color=red><b>DASSL</b></font>. <p>
 
<hr> <p>
</html>", revisions = ""), experiment(StopTime = 520), experimentSetupOutput);
      end InventoryForresterNormalNoise;
      model InventoryForresterSinOneYear "Inventory simulation with sinusoidal orders"
        parameter Real RRRiniTop(unit = "1/wk") = 1000 "Inital value of customer requests at retail";
        parameter Real RRDiniTop(unit = "1/wk") = RRRiniTop "Inital value of requisitions received at distribution";
        parameter Real RRFiniTop(unit = "1/wk") = RRRiniTop "Inital value of requisitions received at factory";
        output Real orders(unit = "1/wk") "Sinusoidal orders";
        output Real factoryFlow(unit = "1/wk") "Manufacturing flow at factory";
        output Real retailStock "Stock of goods in retail";
        output Real distributionStock "Stock of goods in distribution";
        output Real factoryStock "Stock of goods in factory";
        SystemDynamics.IndustrialDynamics.Inventory.Factory Factory1(ALF = 1000 * RRFiniTop, RRFini = RRFiniTop) annotation(Placement(visible = true, transformation(origin = {-34.75,0.25}, extent = {{-12.75,-12.75},{12.75,12.75}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Distribution Distribution1(RRDini = RRDiniTop) annotation(Placement(visible = true, transformation(origin = {-4.75,0.25}, extent = {{-12.75,-12.75},{12.75,12.75}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Retail Retail1(RRRini = RRRiniTop) annotation(Placement(visible = true, transformation(origin = {25.25,0.25}, extent = {{-12.75,-12.75},{12.75,12.75}}, rotation = 0)));
        Modelica.Blocks.Sources.Sine Orders(amplitude = 100, freqHz = 1 / 52, offset = 1000) annotation(Placement(visible = true, transformation(origin = {0.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      equation
        connect(Orders.y,Retail1.rrr) annotation(Line(visible = true, points = {{11.0,30.0},{45.0,30.0},{45.0,8.41},{36.47,8.41}}, color = {0,0,191}));
        connect(Retail1.ssd,Distribution1.ssd) annotation(Line(visible = true, points = {{14.03,-2.81},{10.515,-2.81},{10.515,-2.81},{6.47,-2.81}}, color = {95,0,191}));
        connect(Retail1.uod,Distribution1.uod) annotation(Line(visible = true, points = {{14.03,3.31},{10.015,3.31},{10.015,3.31},{6.47,3.31}}, color = {0,0,191}));
        connect(Distribution1.ssf,Factory1.ssf) annotation(Line(visible = true, points = {{-15.97,-2.81},{-19.485,-2.81},{-19.485,-2.81},{-23.53,-2.81}}, color = {95,0,191}));
        connect(Distribution1.uof,Factory1.uof) annotation(Line(visible = true, points = {{-15.97,3.31},{-19.485,3.31},{-19.485,3.31},{-23.53,3.31}}, color = {0,0,191}));
        connect(Distribution1.rrf,Factory1.rrf) annotation(Line(visible = true, points = {{-15.97,8.41},{-23.53,8.41}}, color = {95,0,191}));
        connect(Retail1.dfd,Distribution1.dfd) annotation(Line(visible = true, points = {{14.03,-7.91},{6.47,-7.91}}, color = {0,0,191}));
        connect(Retail1.rrd,Distribution1.rrd) annotation(Line(visible = true, points = {{14.03,8.41},{6.47,8.41}}, color = {95,0,191}));
        connect(Distribution1.dff,Factory1.dff) annotation(Line(visible = true, points = {{-15.97,-7.91},{-23.53,-7.91}}, color = {0,0,191}));
        orders = Orders.y;
        factoryFlow = Factory1.IAF.u1;
        retailStock = Retail1.IAR.y;
        distributionStock = Distribution1.IAD.y;
        factoryStock = Factory1.IAF.y;
        annotation(Icon(coordinateSystem(extent = {{-50.0,-50.0},{50.0,50.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Text(visible = true, lineColor = {0,0,255}, extent = {{1.0,-52.0},{1.0,-39.0}}, textString = "%name", fontName = "Arial"),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-40.0,-40.0},{40.0,50.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {127,255,0}, fillPattern = FillPattern.Solid, extent = {{-30.0,23.0},{30.0,38.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {127,255,0}, fillPattern = FillPattern.Solid, extent = {{-30.0,-2.0},{30.0,13.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {127,255,0}, fillPattern = FillPattern.Solid, extent = {{-30.0,-27.0},{30.0,-12.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-25.0,25.0},{27.0,35.0}}, textString = "Factory", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-25.0,1.0},{27.0,11.0}}, textString = "Distribution", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-25.0,-24.0},{27.0,-14.0}}, textString = "Retail", fontName = "Arial"),Line(visible = true, points = {{0.0,13.0},{0.0,23.0}}, color = {0,0,255}),Line(visible = true, points = {{0.0,-2.0},{0.0,-12.0}}, color = {0,0,255})}), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), experiment(StopTime = 120, Algorithm = "dassl"), experimentSetupOutput, Documentation(info = "<html>
<b>Seasonally Fluctuating Customer Demand</b> <p>
 
The second simulation involves seasonal fluctuation of customers:<p>
<font color=red><b>RRR(t) = RRRini + 100*sin(2*pi*time/52);</b></font> <p>
 
<hr> <p>
 
Simulate the model across 120 weeks, and plot on one graph the incoming orders and the production flow in the factory and on a second graph the levels of goods in retail, distribution, and the factory as functions of time: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/Inventory_2.png\"/> <p>
 
<hr> <p>
</html>", revisions = ""));
      end InventoryForresterSinOneYear;
      model InventoryForresterSinOneYearCapacity "Inventory simulation with sinusoidal orders"
        parameter Real RRRiniTop(unit = "1/wk") = 1000 "Inital value of customer requests at retail";
        parameter Real RRDiniTop(unit = "1/wk") = RRRiniTop "Inital value of requisitions received at distribution";
        parameter Real RRFiniTop(unit = "1/wk") = RRRiniTop "Inital value of requisitions received at factory";
        output Real orders(unit = "1/wk") "Sinusoidal orders";
        output Real factoryFlow(unit = "1/wk") "Manufacturing flow at factory";
        output Real retailStock "Stock of goods in retail";
        output Real distributionStock "Stock of goods in distribution";
        output Real factoryStock "Stock of goods in factory";
        SystemDynamics.IndustrialDynamics.Inventory.Factory Factory1(RRFini = RRFiniTop, ALF = 1200) annotation(Placement(visible = true, transformation(origin = {-34.75,0.25}, extent = {{-12.75,-12.75},{12.75,12.75}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Distribution Distribution1(RRDini = RRDiniTop) annotation(Placement(visible = true, transformation(origin = {-4.75,0.25}, extent = {{-12.75,-12.75},{12.75,12.75}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Retail Retail1(RRRini = RRRiniTop) annotation(Placement(visible = true, transformation(origin = {25.25,0.25}, extent = {{-12.75,-12.75},{12.75,12.75}}, rotation = 0)));
        Modelica.Blocks.Sources.Sine Orders(amplitude = 100, freqHz = 1 / 52, offset = 1000) annotation(Placement(visible = true, transformation(origin = {0.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      equation
        connect(Orders.y,Retail1.rrr) annotation(Line(visible = true, points = {{11.0,30.0},{45.0,30.0},{45.0,8.41},{36.47,8.41}}, color = {0,0,191}));
        connect(Retail1.ssd,Distribution1.ssd) annotation(Line(visible = true, points = {{14.03,-2.81},{10.515,-2.81},{10.515,-2.81},{6.47,-2.81}}, color = {95,0,191}));
        connect(Retail1.uod,Distribution1.uod) annotation(Line(visible = true, points = {{14.03,3.31},{10.015,3.31},{10.015,3.31},{6.47,3.31}}, color = {0,0,191}));
        connect(Distribution1.ssf,Factory1.ssf) annotation(Line(visible = true, points = {{-15.97,-2.81},{-17.86,-2.81},{-17.86,-2.81},{-19.75,-2.81},{-19.75,-2.81},{-23.53,-2.81}}, color = {95,0,191}));
        connect(Distribution1.uof,Factory1.uof) annotation(Line(visible = true, points = {{-15.97,3.31},{-17.86,3.31},{-17.86,3.31},{-19.75,3.31},{-19.75,3.31},{-23.53,3.31}}, color = {0,0,191}));
        connect(Distribution1.rrf,Factory1.rrf) annotation(Line(visible = true, points = {{-15.97,8.41},{-23.53,8.41}}, color = {95,0,191}));
        connect(Retail1.dfd,Distribution1.dfd) annotation(Line(visible = true, points = {{14.03,-7.91},{6.47,-7.91}}, color = {0,0,191}));
        connect(Retail1.rrd,Distribution1.rrd) annotation(Line(visible = true, points = {{14.03,8.41},{6.47,8.41}}, color = {95,0,191}));
        connect(Distribution1.dff,Factory1.dff) annotation(Line(visible = true, points = {{-15.97,-7.91},{-23.53,-7.91}}, color = {0,0,191}));
        orders = Orders.y;
        factoryFlow = Factory1.IAF.u1;
        retailStock = Retail1.IAR.y;
        distributionStock = Distribution1.IAD.y;
        factoryStock = Factory1.IAF.y;
        annotation(Icon(coordinateSystem(extent = {{-50.0,-50.0},{50.0,50.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Text(visible = true, lineColor = {0,0,255}, extent = {{1.0,-52.0},{1.0,-39.0}}, textString = "%name", fontName = "Arial"),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-40.0,-40.0},{40.0,50.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {127,255,0}, fillPattern = FillPattern.Solid, extent = {{-30.0,23.0},{30.0,38.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {127,255,0}, fillPattern = FillPattern.Solid, extent = {{-30.0,-2.0},{30.0,13.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {127,255,0}, fillPattern = FillPattern.Solid, extent = {{-30.0,-27.0},{30.0,-12.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-25.0,25.0},{27.0,35.0}}, textString = "Factory", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-25.0,1.0},{27.0,11.0}}, textString = "Distribution", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-25.0,-24.0},{27.0,-14.0}}, textString = "Retail", fontName = "Arial"),Line(visible = true, points = {{0.0,13.0},{0.0,23.0}}, color = {0,0,255}),Line(visible = true, points = {{0.0,-2.0},{0.0,-12.0}}, color = {0,0,255})}), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), experiment(StopTime = 120, Algorithm = "dassl"), experimentSetupOutput, Documentation(info = "<html>
<b>Seasonally Fluctuating Customer Demand</b> <p>
 
The third simulation involves seasonal fluctuation of customers with a capacity limit of 1200 items/week in the factory.  The customer demand remains always below the manufacturing capacity limit; nevertheless, the manufacturing site must operate at its capacity limit for certain time periods, while at other times, it runs at 20% of its capacity only. <p>
 
<hr> <p>
 
Simulate the model across 120 weeks, and plot on one graph the incoming orders and the production flow in the factory and on a second graph the levels of goods in retail, distribution, and the factory as functions of time: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/Inventory_3.png\"> <p>
 
<hr> <p>
</html>", revisions = ""));
      end InventoryForresterSinOneYearCapacity;
      model InventoryForresterStep "Inventory simulation with sinusoidal orders"
        parameter Real RRRiniTop(unit = "1/wk") = 1000 "Inital value of customer requests at retail";
        parameter Real RRDiniTop(unit = "1/wk") = RRRiniTop "Inital value of requisitions received at distribution";
        parameter Real RRFiniTop(unit = "1/wk") = RRRiniTop "Inital value of requisitions received at factory";
        output Real orders(unit = "1/wk") "Sinusoidal orders";
        output Real factoryFlow(unit = "1/wk") "Manufacturing flow at factory";
        output Real retailStock "Stock of goods in retail";
        output Real distributionStock "Stock of goods in distribution";
        output Real factoryStock "Stock of goods in factory";
        SystemDynamics.IndustrialDynamics.Inventory.Factory Factory1(RRFini = RRFiniTop, ALF = 1000 * RRFiniTop) annotation(Placement(visible = true, transformation(origin = {-34.75,0.25}, extent = {{-12.75,-12.75},{12.75,12.75}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Distribution Distribution1(RRDini = RRDiniTop) annotation(Placement(visible = true, transformation(origin = {-4.75,0.25}, extent = {{-12.75,-12.75},{12.75,12.75}}, rotation = 0)));
        SystemDynamics.IndustrialDynamics.Inventory.Retail Retail1(RRRini = RRRiniTop) annotation(Placement(visible = true, transformation(origin = {25.25,0.25}, extent = {{-12.75,-12.75},{12.75,12.75}}, rotation = 0)));
        Modelica.Blocks.Sources.Step Orders(height = 100, offset = RRRiniTop, startTime = 10) annotation(Placement(visible = true, transformation(origin = {0.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      equation
        connect(Orders.y,Retail1.rrr) annotation(Line(visible = true, points = {{11.0,30.0},{45.0,30.0},{45.0,8.41},{36.47,8.41}}, color = {0,0,191}));
        connect(Retail1.ssd,Distribution1.ssd) annotation(Line(visible = true, points = {{14.03,-2.81},{10.515,-2.81},{10.515,-2.81},{6.47,-2.81}}, color = {95,0,191}));
        connect(Retail1.uod,Distribution1.uod) annotation(Line(visible = true, points = {{14.03,3.31},{10.015,3.31},{10.015,3.31},{6.47,3.31}}, color = {0,0,191}));
        connect(Distribution1.ssf,Factory1.ssf) annotation(Line(visible = true, points = {{-15.97,-2.81},{-17.86,-2.81},{-17.86,-2.81},{-19.75,-2.81},{-19.75,-2.81},{-23.53,-2.81}}, color = {95,0,191}));
        connect(Distribution1.uof,Factory1.uof) annotation(Line(visible = true, points = {{-15.97,3.31},{-17.86,3.31},{-17.86,3.31},{-19.75,3.31},{-19.75,3.31},{-23.53,3.31}}, color = {0,0,191}));
        connect(Distribution1.rrf,Factory1.rrf) annotation(Line(visible = true, points = {{-15.97,8.41},{-23.53,8.41}}, color = {95,0,191}));
        connect(Retail1.dfd,Distribution1.dfd) annotation(Line(visible = true, points = {{14.03,-7.91},{6.47,-7.91}}, color = {0,0,191}));
        connect(Retail1.rrd,Distribution1.rrd) annotation(Line(visible = true, points = {{14.03,8.41},{6.47,8.41}}, color = {95,0,191}));
        connect(Distribution1.dff,Factory1.dff) annotation(Line(visible = true, points = {{-15.97,-7.91},{-23.53,-7.91}}, color = {0,0,191}));
        orders = Orders.y;
        factoryFlow = Factory1.IAF.u1;
        retailStock = Retail1.IAR.y;
        distributionStock = Distribution1.IAD.y;
        factoryStock = Factory1.IAF.y;
        annotation(experiment(StopTime = 90, Algorithm = "dassl"), Icon(coordinateSystem(extent = {{-50.0,-50.0},{50.0,50.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Text(visible = true, lineColor = {0,0,255}, extent = {{1.0,-52.0},{1.0,-39.0}}, textString = "%name", fontName = "Arial"),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-40.0,-40.0},{40.0,50.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {127,255,0}, fillPattern = FillPattern.Solid, extent = {{-30.0,23.0},{30.0,38.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {127,255,0}, fillPattern = FillPattern.Solid, extent = {{-30.0,-2.0},{30.0,13.0}}),Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {127,255,0}, fillPattern = FillPattern.Solid, extent = {{-30.0,-27.0},{30.0,-12.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-25.0,25.0},{27.0,35.0}}, textString = "Factory", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-25.0,1.0},{27.0,11.0}}, textString = "Distribution", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-25.0,-24.0},{27.0,-14.0}}, textString = "Retail", fontName = "Arial"),Line(visible = true, points = {{0.0,13.0},{0.0,23.0}}, color = {0,0,255}),Line(visible = true, points = {{0.0,-2.0},{0.0,-12.0}}, color = {0,0,255})}), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), experimentSetupOutput, Documentation(info = "<html>
<b>Customer Demand Step</b> <p>
 
The fourth simulation involves a step in the incoming orders: <p>
<font color=red><b>RRR(t) = RRRini + if time > 10 then RRRoffset else 0;</b></font> <p>
 
<hr> <p>
 
Simulate the model across 90 weeks, and plot on one graph the incoming orders and the production flow in the factory and on a second graph the levels of goods in retail, distribution, and the factory as functions of time: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/Inventory_4.png\"> <p>
 
<hr> <p>
</html>", revisions = ""));
      end InventoryForresterStep;
      package Utilities "Utility models of Forrester's inventory model"
        extends Modelica.Icons.Library;
        block DFD "Desired delay in filling orders"
          extends Interfaces.Nonlin_2;
          parameter Real DHD(unit = "wk") = 1 "Delay in handling";
          parameter Real DUD(unit = "wk") = 1 "Delay due to unfillable orders";
          output Real IAD "Actual produced goods in distribution";
          output Real IDD "Desired produced goods in distribution";
          output Real dfd(unit = "wk") "Desired delay in filling orders";
        equation
          IAD = u1;
          IDD = u2;
          dfd = DHD + DUD * IDD / IAD;
          y = dfd;
        end DFD;
        block DFF "Desired delay in filling orders"
          extends Interfaces.Nonlin_2;
          parameter Real DHF(unit = "wk") = 1 "Delay in handling";
          parameter Real DUF(unit = "wk") = 1 "Delay due to unfillable orders";
          output Real IAF "Actual produced goods in factory";
          output Real IDF "Desired produced goods in factory";
          output Real dff(unit = "wk") "Desired delay in filling orders";
        equation
          IAF = u1;
          IDF = u2;
          dff = DHF + DUF * IDF / IAF;
          y = dff;
        end DFF;
        block LAD "Actual delay in order processing"
          extends Interfaces.Nonlin_4;
          output Real CPD "Items delayed in order";
          output Real PMD "Items delayed in mail";
          output Real UOF "Unfilled orders";
          output Real MTD "Items delayed in transportation";
          output Real lad "Items delayed in order processing";
        equation
          CPD = u1;
          PMD = u2;
          UOF = u3;
          MTD = u4;
          lad = CPD + PMD + UOF + MTD;
          y = lad;
        end LAD;
        block LDD "Desired delay in order processing"
          extends Interfaces.Nonlin_2;
          parameter Real DCD(unit = "wk") = 2 "Clerical delay";
          parameter Real DMD(unit = "wk") = 0.5 "Mail delay";
          parameter Real DTD(unit = "wk") = 2 "Transportation delay";
          output Real RSD(unit = "1/wk") "Delayed new orders";
          output Real DFF(unit = "wk") "Desired delay in filling orders";
          output Real ldd "Desired delay in order processing";
        equation
          RSD = u1;
          DFF = u2;
          ldd = RSD * (DCD + DMD + DFF + DTD);
          y = ldd;
        end LDD;
        block MDF_d "Manufacturing rate at factory"
          extends Interfaces.Nonlin_1;
          parameter Real ALF(unit = "1/wk") = 1000000.0 "Manufacturing capacity limit at factory";
          output Real MWF(unit = "1/wk") "Manufacturing demand at factory";
          output Real MDF(unit = "1/wk") "Manufacturing rate at factory";
        equation
          MWF = u;
          MDF = if ALF >= MWF then MWF else ALF;
          y = MDF;
        end MDF_d;
        block MWF "Manufacturing demand at factory"
          extends Interfaces.Nonlin_4;
          parameter Real DIF(unit = "wk") = 4 "Delay in inventory (and pipeline) adjustment";
          output Real mwf1 "Partial manufacturing adjustment for factory";
          output Real UOF "Actual unfilled orders";
          output Real UNF "Desired unfilled orders";
          output Real RRF(unit = "1/wk") "New orders arriving";
          output Real mwf(unit = "1/wk") "Manufacturing demand at factory";
        equation
          mwf1 = u1;
          UOF = u2;
          UNF = u3;
          RRF = u4;
          mwf = RRF + 1 / DIF * (mwf1 + UOF - UNF);
          y = mwf;
        end MWF;
        block MWF1 "Partial manufacturing adjustment for factory"
          extends Interfaces.Nonlin_4;
          output Real LDF "Desired delays in order processing";
          output Real LAF "Actual delays in order processing";
          output Real IDF "Desired produced goods in factory";
          output Real IAF "Actual produced goods in factory";
          output Real mwf1 "Partial manufacturing adjustment for factory";
        equation
          LDF = u1;
          LAF = u2;
          IDF = u3;
          IAF = u4;
          mwf1 = LDF - LAF + IDF - IAF;
          y = mwf1;
        end MWF1;
        block PPD1 "Manufacturing demand at distribution"
          extends Interfaces.Nonlin_4;
          parameter Real DID(unit = "wk") = 4 "Delay in inventory (and pipeline) adjustment";
          output Real ppd2 "Partial manufacturing adjustment for distribution";
          output Real UOD "Actual unfilled orders";
          output Real UND "Desired unfilled orders";
          output Real RRD(unit = "1/wk") "New orders arriving";
          output Real ppd1(unit = "1/wk") "Manufacturing demand at distribution";
        equation
          ppd2 = u1;
          UOD = u2;
          UND = u3;
          RRD = u4;
          ppd1 = RRD + 1 / DID * (ppd2 + UOD - UND);
          y = ppd1;
        end PPD1;
        block PPD2 "Partial manufacturing adjustment for distribution"
          extends Interfaces.Nonlin_4;
          output Real LDD "Desired delays in order processing";
          output Real LAD "Actual delays in order processing";
          output Real IDD "Desired produced goods in distribution";
          output Real IAD "Actual produced goods in distribution";
          output Real ppd2 "Partial manufacturing adjustment for distribution";
        equation
          LDD = u1;
          LAD = u2;
          IDD = u3;
          IAD = u4;
          ppd2 = LDD - LAD + IDD - IAD;
          y = ppd2;
        end PPD2;
        block SSD_d "Delivery rate at distribution"
          extends Interfaces.Nonlin_2;
          output Real IAD(unit = "1/wk") "Available distribution capacity at distribution";
          output Real STD(unit = "1/wk") "Desired order filling rate";
          output Real SSD(unit = "1/wk") "Delivery rate at distribution";
        equation
          IAD = u1;
          STD = u2;
          SSD = if IAD > STD then STD else IAD;
          y = SSD;
        end SSD_d;
        block SSF_d "Delivery rate at factory"
          extends Interfaces.Nonlin_2;
          output Real IAF(unit = "1/wk") "Available distribution capacity at factory";
          output Real STF(unit = "1/wk") "Desired order filling rate";
          output Real SSF(unit = "1/wk") "Delivery rate at factory";
        equation
          IAF = u1;
          STF = u2;
          SSF = if IAF > STF then STF else IAF;
          y = SSF;
        end SSF_d;
        annotation(preferedView = "info", Documentation(info = "<html>
Utility models of Forrester's inventory model.
</html>"));
      end Utilities;
      annotation(preferedView = "info", Documentation(info = "<html>
In Forrester (1961), a factory-distribution-retail inventory system is discussed in depth, modeled with the System Dynamics notation. At the time, the models had to be constructed using textual representations (<font color=red><b>Dynamo</b></font>). This very system is re-implemented here graphically using the <font color=red><b>SystemDynamics</b></font> library of <font color=red><b>Modelica</b></font>, and the same simulations are carried out as are given in the book by Forrester. Effects of amplification, delays, and information feedback are demonstrated in the simulations. Among other things, it is shown, how small changes in retail sales can lead to large swings in factory production. <p>
 
The model is not introduced in detail at this place.  The interested reader is referred to Forrester (1961). Roughly, the system is divided into a retail, a distribution, and a factory part, each containing inventories for the goods (consumer goods, such as refrigerators, for example) and a corresponding logic for ordering or producing items to be sold to customers. The only input to the system is customer orders RRR (Requisitions Received at Retail), and the internal variables of the system (levels and rates) are to be investigated when customer demand fluctuates. <p>
 
<b>References:</b> <p>
 
<ol>
<li> Forrester, J.W. (1961), <a href=\"http://www.amazon.com/Forrester-Industrial-Dynamics-Jay-W/dp/0262060035\">Industrial Dynamics</a>, M.I.T. Press, 479p.
</ol> <p>
 
<hr> <p>
</html>"));
    end Inventory;
    annotation(preferedView = "info", Documentation(info = "<html>
The oldest research area, where the System Dynamics methodology has been employed, is in the description of <font color=red><b>industrial dynamics</b></font>.  In fact, this is the application area for which the methodology had originally been developed by <a href=\"http://en.wikipedia.org/wiki/Jay_Wright_Forrester\">Jay Forrester</a>. <p>
 
This package is designed to contain examples of such <font color=red><b>industrial dynamics</b></font> models. <p>
 
<b>References:</b> <p>
 
<ol>
<li> Forrester, J.W. (1961), <a href=\"http://www.amazon.com/Forrester-Industrial-Dynamics-Jay-W/dp/0262060035\">Industrial Dynamics</a>, M.I.T. Press, 479p.
<li> Hammer, M, and J. Champy (1994), <a href=\"http://www.amazon.com/Reengineering-Corporation-Manifesto-Business-Revolution/dp/0066621127\">Reengineering the Corporation: A Manifesto for Business Revolution</a>, Harper Business, New York, 257p.
</ol> <p>
 
<hr> <p>
</html>"));
  end IndustrialDynamics;
  package PopulationDynamics "Ecological models"
    extends Modelica.Icons.Example;
    package LarchBudMoth "Model of larch bud moth population dynamics"
      extends Modelica.Icons.Example annotation(Documentation(info = "<html>
The larch bud moth model describes the population dynamics of the larch bud moth insect in the upper Engiadina valley of Switzerland. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Cellier, F.E. (1991), <a href=\"http://www.amazon.com/Continuous-System-Modeling-Fran%C3%A7ois-Cellier/dp/0387975020\">Continuous System Modeling</a>, Springer-Verlag, New York, ISBN: 0-387-97502-0, 755p.
<li> Fischlin, A. and W. Baltensweiler (1979), \"Systems Analysis of the Larch Bud Moth System.  Part I: The Larch - Larch Bud Moth Relationship,\" <i>Mitteilungen der Schweizerischen Entomologischen Gesellschaft</i>, <b>52</b>, pp.273-289.
</ol> <p>
 
<hr> <p>
</html>"), preferedView = "info");
      block Starvation "Starvation function"
        extends Interfaces.Nonlin_2;
        output Real foliage;
        output Real food_demand;
        output Real starvation;
      equation
        foliage = u1;
        food_demand = u2;
        starvation = exp(-foliage / food_demand);
        y = starvation;
        annotation(Documentation(info = "<html>
Model of starvation function.
</html>"));
      end Starvation;
      block Defoliation "Defoliation function"
        extends Interfaces.Nonlin_3;
        output Real foliage;
        output Real food_demand;
        output Real starvation;
        output Real defoliation;
      equation
        foliage = u1;
        food_demand = u2;
        starvation = u3;
        defoliation = (1.0 - starvation) * food_demand / foliage;
        y = defoliation;
        annotation(Documentation(info = "<html>
Model of defoliation function.
</html>"));
      end Defoliation;
      model Grecr "Raw fiber content function"
        extends Interfaces.Nonlin_2;
        output Real defoliation "First input";
        output Real rawfiber "Second input";
        output Real zraw "Auxiliary variable";
        output Real grecr1 "Auxiliary variable";
        output Real grecr2 "Auxiliary variable";
        output Real grecr3 "Auxiliary variable";
        output Real grecr4 "Auxiliary variable";
        output Real grecr "Output variable";
      equation
        defoliation = u1;
        rawfiber = u2;
        zraw = 0.425 + sqrt((18.0 - rawfiber) / max([(rawfiber - 11.99) ^ 2,0.01]));
        grecr1 = if zraw > rawfiber - 11.99 then 11.99 / rawfiber else 1.0 - zraw / rawfiber;
        grecr2 = if rawfiber > 11.99 then grecr1 else 1.0;
        grecr3 = 1.0 + (defoliation - 0.4) * (18.0 - rawfiber) / (0.4 * rawfiber);
        grecr4 = 18.0 / rawfiber;
        grecr = if defoliation < 0.4 then grecr2 else if defoliation < 0.8 then grecr3 else grecr4;
        y = grecr;
        annotation(Documentation(info = "<html>
This function models the change in raw fiber content as a function of defoliation.
</html>"));
      end Grecr;
      block Logarithm "Logarithmic transformation"
        extends Interfaces.Nonlin_1;
      equation
        y = Modelica.Math.log10(u);
        annotation(Documentation(info = "<html>
Logarithmic transformation.
</html>"));
      end Logarithm;
      model LBM "Larch bud moth population dynamics model"
        parameter Real wintermortality = 0.5728 "Percentage of eggs dying during the winter";
        parameter Modelica.SIunits.Mass biomass = 91.3 "Dry needle biomass per tree";
        parameter Integer nbr_trees = 511147 "Number of trees";
        parameter Integer egg0 = 4765975 "Initial number of eggs";
        parameter Real rawfiber0 = 15.0 "Initial rawfiber";
        output Real LogDensitySim "Simulated logarithmic larvae density";
        output Real LogDensityMeas "Measured logarithmic larvae density";
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {-160.0,160.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 NewEggs annotation(Placement(visible = true, transformation(origin = {-110.0,160.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.DiscreteLevel Eggs(x0 = egg0) annotation(Placement(visible = true, transformation(origin = {-50.0,160.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 OldEggs annotation(Placement(visible = true, transformation(origin = {10.0,160.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink1 annotation(Placement(visible = true, transformation(origin = {60.0,160.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain Hatching(k = 1.0 - wintermortality) annotation(Placement(visible = true, transformation(origin = {-10.0,90.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.DiscreteLevel Rawfiber(x0 = rawfiber0) annotation(Placement(visible = true, transformation(origin = {-10.0,-50.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 NewFiber annotation(Placement(visible = true, transformation(origin = {-70.0,-50.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source2 annotation(Placement(visible = true, transformation(origin = {-110.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 OldFiber annotation(Placement(visible = true, transformation(origin = {52.0,-50.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink2 annotation(Placement(visible = true, transformation(origin = {90.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Linear Incoincidence(m = 0.05112, b = -0.17932) annotation(Placement(visible = true, transformation(origin = {-15.0,25.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Linear Coincidence(m = -1.0, b = 1.0) annotation(Placement(visible = true, transformation(origin = {25.0,25.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_1 annotation(Placement(visible = true, transformation(origin = {34.0,66.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = -90)));
        SystemDynamics.Functions.Linear Weakening(m = 0.124017, b = -1.435284) annotation(Placement(visible = true, transformation(origin = {35.0,-11.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Linear Physiology(m = -1.0, b = 1.0) annotation(Placement(visible = true, transformation(origin = {77.0,-11.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Linear TreeFoliage(m = -2.25933, b = 67.38939) annotation(Placement(visible = true, transformation(origin = {-15.0,-113.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_2 annotation(Placement(visible = true, transformation(origin = {20.0,-135.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = -90)));
        SystemDynamics.Auxiliary.Const NbrTrees(k = nbr_trees) annotation(Placement(visible = true, transformation(origin = {20.0,-170.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.PopulationDynamics.LarchBudMoth.Starvation Starvation1 annotation(Placement(visible = true, transformation(origin = {70.0,-130.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain FoodDemand(k = 0.005472) annotation(Placement(visible = true, transformation(origin = {94.0,25.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Linear FoodSupply(m = -1.0, b = 1.0) annotation(Placement(visible = true, transformation(origin = {115.0,-130.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_3 Prod_3_1 annotation(Placement(visible = true, transformation(origin = {145.0,-11.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -450)));
        SystemDynamics.Auxiliary.Gain SexRatio(k = 0.44) annotation(Placement(visible = true, transformation(origin = {114.0,120.0}, extent = {{20.0,-20.0},{-20.0,20.0}}, rotation = 0)));
        SystemDynamics.Functions.Linear Fecundity(m = -18.475457, b = 356.72636) annotation(Placement(visible = true, transformation(origin = {-85.0,-113.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_3 annotation(Placement(visible = true, transformation(origin = {-146.0,85.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = -90)));
        SystemDynamics.PopulationDynamics.LarchBudMoth.Defoliation Defoliation1 annotation(Placement(visible = true, transformation(origin = {90.0,-190.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.PopulationDynamics.LarchBudMoth.Grecr Grecr1 annotation(Placement(visible = true, transformation(origin = {150.0,-190.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_4 annotation(Placement(visible = true, transformation(origin = {-130.0,-191.0}, extent = {{16.0,-15.0},{-16.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain LarvaeDensity(k = 1.0 / (biomass * nbr_trees)) annotation(Placement(visible = true, transformation(origin = {192.0,66.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.PopulationDynamics.LarchBudMoth.Logarithm Logarithm1 annotation(Placement(visible = true, transformation(origin = {233.0,66.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Measurement(x_vals = 1949:1978, y_vals = {0.018,0.082,0.444,4.174,68.797,331.76,126.541,21.28,2.246,0.085,0.08,0.371,1.638,22.878,248.817,184.272,3.116,0.019,0.002,0.059,0.197,1.068,10.569,173.932,249.612,176.023,4.749,0.014,0.008,0.056}) annotation(Placement(visible = true, transformation(origin = {192.0265,-50.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        Logarithm Logarithm2 annotation(Placement(visible = true, transformation(origin = {241.0,-50.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
      equation
        connect(Measurement.y,Logarithm2.u) annotation(Line(visible = true, origin = {220.7633,-50.0}, points = {{-6.7367,0.0},{6.7367,0.0}}, color = {0,0,191}));
        connect(NbrTrees.y,Prod_2_2.u2) annotation(Line(visible = true, points = {{20.0,-156.0},{20.0,-146.2}}, color = {0,0,191}));
        connect(Starvation1.u1,Prod_2_2.y) annotation(Line(visible = true, points = {{52.0,-122.0},{40.0,-122.0},{40.0,-135.0},{30.5,-135.0}}, color = {0,0,191}));
        connect(Defoliation1.u1,Prod_2_2.y) annotation(Line(visible = true, points = {{72.0,-178.0},{40.0,-178.0},{40.0,-135.0},{30.5,-135.0}}, color = {0,0,191}));
        connect(Prod_2_2.u1,TreeFoliage.y) annotation(Line(visible = true, points = {{20.0,-123.8},{20.0,-113.0},{-1.5,-113.0}}, color = {0,0,191}));
        connect(Prod_3_1.y,SexRatio.u) annotation(Line(visible = true, points = {{155.5,-11.0},{164.0,-11.0},{164.0,120.0},{128.0,120.0}}, color = {0,0,191}));
        connect(LarvaeDensity.u,Prod_3_1.u1) annotation(Line(visible = true, points = {{178.0,66.0},{145.0,66.0},{145.0,-0.5}}, color = {0,0,191}));
        connect(Physiology.y,Prod_3_1.u2) annotation(Line(visible = true, points = {{90.5,-11.0},{111.25,-11.0},{111.25,-11.0},{134.5,-11.0}}, color = {0,0,191}));
        connect(FoodSupply.y,Prod_3_1.u3) annotation(Line(visible = true, points = {{128.5,-130.0},{145.0,-130.0},{145.0,-21.5}}, color = {0,0,191}));
        connect(Prod_2_1.u1,Hatching.y) annotation(Line(visible = true, points = {{34.0,77.2},{34.0,90.0},{4.0,90.0}}, color = {0,0,191}));
        connect(LarvaeDensity.u,Prod_2_1.y) annotation(Line(visible = true, points = {{178.0,66.0},{45.2,66.0}}, color = {0,0,191}));
        connect(FoodDemand.u,Prod_2_1.y) annotation(Line(visible = true, points = {{82.8,25.0},{60.0,25.0},{60.0,66.0},{45.2,66.0}}, color = {0,0,191}));
        connect(Coincidence.y,Prod_2_1.u2) annotation(Line(visible = true, points = {{38.5,25.0},{50.0,25.0},{50.0,46.0},{34.0,46.0},{34.0,54.8}}, color = {0,0,191}));
        connect(Fecundity.y,Prod_2_3.u2) annotation(Line(visible = true, points = {{-71.5,-113.0},{-60.0,-113.0},{-60.0,-160.0},{-146.0,-160.0},{-146.0,73.8}}, color = {0,0,191}));
        connect(Prod_2_3.u1,SexRatio.y) annotation(Line(visible = true, points = {{-146.0,96.2},{-146.0,120.0},{100.0,120.0}}, color = {0,0,191}));
        connect(Prod_2_3.y,NewEggs.u) annotation(Line(visible = true, points = {{-135.5,85.0},{-110.0,85.0},{-110.0,139.0}}, color = {0,0,191}));
        connect(LarvaeDensity.y,Logarithm1.u) annotation(Line(visible = true, points = {{206.0,66.0},{219.5,66.0}}, color = {0,0,191}));
        connect(Incoincidence.y,Coincidence.u) annotation(Line(visible = true, points = {{-1.5,25.0},{4.25,25.0},{4.25,25.0},{11.5,25.0}}, color = {0,0,191}));
        connect(Incoincidence.u,Rawfiber.y1) annotation(Line(visible = true, points = {{-28.5,25.0},{-40.0,25.0},{-40.0,-39.0},{-27.0,-39.0}}, color = {0,0,191}));
        connect(Weakening.u,Rawfiber.y5) annotation(Line(visible = true, points = {{21.5,-11.0},{12.0,-11.0},{12.0,-39.0},{7.0,-39.0}}, color = {0,0,191}));
        connect(Weakening.y,Physiology.u) annotation(Line(visible = true, points = {{48.5,-11.0},{55.25,-11.0},{55.25,-11.0},{63.5,-11.0}}, color = {0,0,191}));
        connect(FoodDemand.y,Starvation1.u2) annotation(Line(visible = true, points = {{105.2,25.0},{120.0,25.0},{120.0,-100.0},{46.0,-100.0},{46.0,-138.0},{52.0,-138.0}}, color = {0,0,191}));
        connect(Defoliation1.u2,Starvation1.u2) annotation(Line(visible = true, points = {{72.0,-190.0},{46.0,-190.0},{46.0,-138.0},{52.0,-138.0}}, color = {0,0,191}));
        connect(Defoliation1.u3,Starvation1.y) annotation(Line(visible = true, points = {{72.0,-202.0},{60.0,-202.0},{60.0,-160.0},{94.0,-160.0},{94.0,-130.0},{88.0,-130.0}}, color = {0,0,191}));
        connect(Starvation1.y,FoodSupply.u) annotation(Line(visible = true, points = {{88.0,-130.0},{101.5,-130.0}}, color = {0,0,191}));
        connect(Defoliation1.y,Grecr1.u1) annotation(Line(visible = true, points = {{108.0,-190.0},{120.0,-190.0},{120.0,-182.0},{132.0,-182.0}}, color = {0,0,191}));
        connect(Grecr1.y,Prod_2_4.u2) annotation(Line(visible = true, points = {{168.0,-190.0},{190.0,-190.0},{190.0,-222.0},{-160.0,-222.0},{-160.0,-191.0},{-141.2,-191.0}}, color = {0,0,191}));
        connect(Grecr1.u2,Rawfiber.y2) annotation(Line(visible = true, points = {{132.0,-198.0},{120.0,-198.0},{120.0,-216.0},{-50.0,-216.0},{-50.0,-84.0},{-25.0,-84.0},{-25.0,-63.0}}, color = {0,0,191}));
        connect(Prod_2_4.u1,Grecr1.u2) annotation(Line(visible = true, points = {{-118.8,-191.0},{-50.0,-191.0},{-50.0,-216.0},{120.0,-216.0},{120.0,-198.0},{132.0,-198.0}}, color = {0,0,191}));
        connect(Fecundity.u,Rawfiber.y2) annotation(Line(visible = true, points = {{-98.5,-113.0},{-110.0,-113.0},{-110.0,-84.0},{-25.0,-84.0},{-25.0,-63.0}}, color = {0,0,191}));
        connect(TreeFoliage.u,Rawfiber.y3) annotation(Line(visible = true, points = {{-28.5,-113.0},{-40.0,-113.0},{-40.0,-90.0},{-10.0,-90.0},{-10.0,-63.0}}, color = {0,0,191}));
        connect(Prod_2_4.y,NewFiber.u) annotation(Line(visible = true, points = {{-130.0,-180.5},{-130.0,-78.0},{-70.0,-78.0},{-70.0,-71.0}}, color = {0,0,191}));
        connect(OldFiber.u,Rawfiber.y4) annotation(Line(visible = true, points = {{52.0,-71.0},{52.0,-78.0},{5.0,-78.0},{5.0,-63.0}}, color = {0,0,191}));
        connect(OldFiber.y1,Sink2.MassInPort1) annotation(Line(visible = true, points = {{62.0,-50.0},{70.0,-50.0},{70.0,-50.0},{79.0,-50.0}}, color = {191,0,191}));
        connect(Rawfiber.u2,OldFiber.y) annotation(Line(visible = true, points = {{12.0,-50.0},{42.0,-50.0}}, color = {191,0,191}));
        connect(NewFiber.y1,Rawfiber.u1) annotation(Line(visible = true, points = {{-60.0,-50.0},{-32.0,-50.0}}, color = {191,0,191}));
        connect(Source2.MassInPort1,NewFiber.y) annotation(Line(visible = true, points = {{-99.0,-50.0},{-80.0,-50.0}}, color = {191,0,191}));
        connect(OldEggs.y1,Sink1.MassInPort1) annotation(Line(visible = true, points = {{20.0,160.0},{49.0,160.0}}, color = {191,0,191}));
        connect(Eggs.u2,OldEggs.y) annotation(Line(visible = true, points = {{-28.0,160.0},{0.0,160.0}}, color = {191,0,191}));
        connect(Source1.MassInPort1,NewEggs.y) annotation(Line(visible = true, points = {{-149.0,160.0},{-120.0,160.0}}, color = {191,0,191}));
        connect(NewEggs.y1,Eggs.u1) annotation(Line(visible = true, points = {{-100.0,160.0},{-72.0,160.0}}, color = {191,0,191}));
        connect(OldEggs.u,Eggs.y4) annotation(Line(visible = true, points = {{10.0,139.0},{10.0,130.0},{-35.0,130.0},{-35.0,147.0}}, color = {0,0,191}));
        connect(Hatching.u,Eggs.y3) annotation(Line(visible = true, points = {{-24.0,90.0},{-50.0,90.0},{-50.0,147.0}}, color = {0,0,191}));
        LogDensitySim = Logarithm1.y;
        Measurement.u = time;
        LogDensityMeas = Logarithm2.y;
        annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{140.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, fillColor = {160,160,160}, fillPattern = FillPattern.Solid, extent = {{-100.0,-100.0},{140.0,100.0}}),Text(visible = true, lineColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-82.0,16.0},{124.0,76.0}}, textString = "Larch bud moth", fontName = "Arial"),Text(visible = true, origin = {-0.51,-14.0}, lineColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-53.6054,-16.0},{100.51,8.0}}, textString = "dynamics of the larch bud moth", fontName = "Arial"),Text(visible = true, origin = {-9.8174,-11.9208}, lineColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-60.1826,0.0},{124.664,30.0}}, textString = "The model describes the population", fontName = "Arial"),Text(visible = true, origin = {14.4661,-16.0}, lineColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-21.1707,-34.0},{40.8293,-14.0}}, textString = "in Switzerland", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-200.0,-240.0},{280.0,200.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{58.0,64.0},{62.0,68.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{38.0,-137.0},{42.0,-133.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{44.0,-140.0},{48.0,-136.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{92.0,-132.0},{96.0,-128.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-52.0,-86.0},{-48.0,-82.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-52.0,-193.0},{-48.0,-189.0}}),Text(visible = true, lineColor = {0,0,255}, fillColor = {128,128,128}, extent = {{12.0,92.0},{60.0,102.0}}, textString = "small Larvae", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {128,128,128}, extent = {{80.0,68.0},{128.0,78.0}}, textString = "large Larvae", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {128,128,128}, extent = {{126.0,124.0},{174.0,134.0}}, textString = "Insects", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {128,128,128}, extent = {{24.0,122.0},{72.0,132.0}}, textString = "Females", fontName = "Arial"),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{143.0,64.0},{147.0,68.0}})}), experiment(StartTime = 1949, StopTime = 1978), experimentSetupOutput, Documentation(info = "<html>
The larch bud moth model describes the population dynamics of the larch bud moth insect in the upper Engiadina valley of Switzerland. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Cellier, F.E. (1991), <a href=\"http://www.amazon.com/Continuous-System-Modeling-Fran%C3%A7ois-Cellier/dp/0387975020\">Continuous System Modeling</a>, Springer-Verlag, New York, ISBN: 0-387-97502-0, 755p.
<li> Fischlin, A. and W. Baltensweiler (1979), \"Systems Analysis of the Larch Bud Moth System.  Part I: The Larch - Larch Bud Moth Relationship,\" <i>Mitteilungen der Schweizerischen Entomologischen Gesellschaft</i>, <b>52</b>, pp.273-289.
</ol> <p>
 
<hr> <p>
 
Simulate the model across 29 time units (years) and plot the simulated and the measured data together against the calendar years. <p>
<img src=\"modelica://SystemDynamics/Resources/Images/LarchBudMoth_1.png\"> <p>
 
<hr> <p>
</html>", revisions = ""));
      end LBM;
    end LarchBudMoth;
    annotation(preferedView = "info", Documentation(info = "<html>
One research area, where the System Dynamics methodology has been used intensively and with much success, is in the description of the <font color=red><b>population dynamics</b></font> of interacting species. <p>
 
This package is designed to contain examples of such <font color=red><b>population dynamics</b></font> models. <p>
 
<b>References:</b> <p>
 
<ol>
<li> Cellier, F.E. (1991), <a href=\"http://www.amazon.com/Continuous-System-Modeling-Fran%C3%A7ois-Cellier/dp/0387975020\">Continuous System Modeling</a>, Springer-Verlag, New York, ISBN: 0-387-97502-0, 755p.
<li> Odum, E.P. and G.W. Barrett (2004), <a href=\"http://www.amazon.com/Fundamentals-Ecology-Eugene-Odum/dp/0534420664\">Fundamentals of Ecology</a>, 5<sup>th</sup> Edition, Brooks Cole Publishing, 624 p.
<li> Odum, E.P. (1997), <a href=\"http://www.amazon.com/Ecology-Bridge-Between-Science-Society/dp/0878936300\">Ecology: A Bridge between Science and Society</a>, 3<sup>rd</sup> Revised Edition, Sinauer Associates, 330 p.
</ol> <p>
 
<hr> <p>
</html>"));
  end PopulationDynamics;
  package WorldDynamics "World models"
    extends Modelica.Icons.Example;
    package World2 "Forrester's World Model"
      extends Modelica.Icons.Example;
      model Scenario_1 "1st Scenario"
        parameter Real Population_0 = 1650000000.0 "World population in 1900";
        parameter Real Pollution_0 = 200000000.0 "Pollution in 1900";
        parameter Real Nat_Resources_0(unit = "ton") = 900000000000.0 "Unrecoverable natural resources in 1900";
        parameter Real Cap_Invest_0(unit = "dollar") = 400000000.0 "Capital investment in 1900";
        parameter Real CIAF_0 = 0.2 "Proportion of capital investment in agriculture in 1900";
        parameter Real BRN(unit = "1/yr") = 0.04 "Normal birth rate";
        parameter Real CIAFN(unit = "dollar") = 0.3 "CIAF normalization";
        parameter Real CIAFT(unit = "yr") = 15.0 "CIAF time constant";
        parameter Real CIDN(unit = "dollar/yr") = 0.025 "Normal capital discard";
        parameter Real CIGN(unit = "dollar/yr") = 0.05 "Normal capital generation";
        parameter Real DRN(unit = "1/yr") = 0.028 "Normal death rate";
        parameter Real ECIRN(unit = "dollar") = 1.0 "Capital normalization";
        parameter Real FC(unit = "kg/yr") = 1.0 "Food coefficient";
        parameter Real FN(unit = "kg/yr") = 1.0 "Food normalization";
        parameter Real Land_Area(unit = "hectare") = 135000000.0 "Area of arable land";
        parameter Real NRI(unit = "ton") = 900000000000.0 "Initial natural resources";
        parameter Real NRUN(unit = "1/yr") = 1.0 "Normal resource utilization";
        parameter Real POLN(unit = "1/yr") = 1.0 "Normal pollution";
        parameter Real POLS = 3599900000.0 "Standard pollution";
        parameter Real Pop_dens_norm(unit = "1/hectare") = 26.5 "Normal population density";
        parameter Real QLS = 1.0 "Standard quality of life";
        output Real Pop "World population";
        output Real Pol "Pollution";
        output Real Pol_rat "Pollution ratio";
        output Real Cap_inv(unit = "dollar") "Capital investment";
        output Real Qual_life "Quality of life";
        output Real Nat_res(unit = "ton") "Natural unrecoverable resources";
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {-150.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Birth_Rate annotation(Placement(visible = true, transformation(origin = {-120.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Population(x0 = Population_0) annotation(Placement(visible = true, transformation(origin = {-70.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Death_Rate annotation(Placement(visible = true, transformation(origin = {-20.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink1 annotation(Placement(visible = true, transformation(origin = {10.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source2 annotation(Placement(visible = true, transformation(origin = {50.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Generation annotation(Placement(visible = true, transformation(origin = {80.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Natural_Resources(x0 = Nat_Resources_0) annotation(Placement(visible = true, transformation(origin = {130.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Depletion annotation(Placement(visible = true, transformation(origin = {180.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink2 annotation(Placement(visible = true, transformation(origin = {210.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Const Gen_Const(k = 0) annotation(Placement(visible = true, transformation(origin = {80.0,101.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular BRMM(x_vals = 0:5, y_vals = {1.2,1.0,0.85,0.75,0.7,0.7}) annotation(Placement(visible = true, transformation(origin = {-105.0,185.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular DRMM(x_vals = 0:0.5:5, y_vals = {3.0,1.8,1.0,0.8,0.7,0.6,0.53,0.5,0.5,0.5,0.5}) annotation(Placement(visible = true, transformation(origin = {51.0,171.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain BR_norm(k = BRN) annotation(Placement(visible = true, transformation(origin = {-135.0,103.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_5 Prod_5_1 annotation(Placement(visible = true, transformation(origin = {-160.0,80.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain DR_norm(k = DRN) annotation(Placement(visible = true, transformation(origin = {-3.0,103.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_5 Prod_5_2 annotation(Placement(visible = true, transformation(origin = {20.0,79.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular BRFM(x_vals = 0:4, y_vals = {0.0,1.0,1.6,1.9,2.0}) annotation(Placement(visible = true, transformation(origin = {-176.0,35.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular BRPM(x_vals = 0:10:60, y_vals = {1.02,0.9,0.7,0.4,0.25,0.15,0.1}) annotation(Placement(visible = true, transformation(origin = {-135.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular BRCM(x_vals = 0:5, y_vals = {1.05,1.0,0.9,0.7,0.6,0.55}) annotation(Placement(visible = true, transformation(origin = {-95.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRCM(x_vals = 0:5, y_vals = {0.9,1.0,1.2,1.5,1.9,3.0}) annotation(Placement(visible = true, transformation(origin = {-25.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRPM(x_vals = 0:10:60, y_vals = {0.92,1.3,2.0,3.2,4.8,6.8,9.2}) annotation(Placement(visible = true, transformation(origin = {15.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRFM(x_vals = 0:0.25:2, y_vals = {30.0,3.0,2.0,1.4,1.0,0.7,0.6,0.5,0.5}) annotation(Placement(visible = true, transformation(origin = {55.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Auxiliary.Gain Crowd_Rat(k = 1.0 / (Land_Area * Pop_dens_norm)) annotation(Placement(visible = true, transformation(origin = {-60.0,35.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = -90)));
        SystemDynamics.Auxiliary.Gain NR_norm(k = NRUN) annotation(Placement(visible = true, transformation(origin = {163.0,103.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_1 annotation(Placement(visible = true, transformation(origin = {140.0,80.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular NRMM(x_vals = 0:10, y_vals = {0.0,1.0,1.8,2.4,2.9,3.3,3.6,3.8,3.9,3.95,4.0}) annotation(Placement(visible = true, transformation(origin = {201.0,80.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Sources.Source Source3 annotation(Placement(visible = true, transformation(origin = {50.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CI_Generation annotation(Placement(visible = true, transformation(origin = {80.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Capital_Investment(x0 = Cap_Invest_0) annotation(Placement(visible = true, transformation(origin = {130.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CI_Discard annotation(Placement(visible = true, transformation(origin = {180.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink3 annotation(Placement(visible = true, transformation(origin = {210.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIG_norm(k = CIGN) annotation(Placement(visible = true, transformation(origin = {65.0,-77.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_2 annotation(Placement(visible = true, transformation(origin = {40.0,-100.0}, extent = {{-14.0,-16.0},{14.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CID_norm(k = CIDN) annotation(Placement(visible = true, transformation(origin = {165.0,-77.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular CIM(x_vals = 0:5, y_vals = {0.1,1.0,1.8,2.4,2.8,3.0}) annotation(Placement(visible = true, transformation(origin = {113.0,-100.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain MSL(k = 1.0 / ECIRN) annotation(Placement(visible = true, transformation(origin = {277.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain NRFR(k = 1.0 / NRI) annotation(Placement(visible = true, transformation(origin = {137.0,51.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain ECIR(k = 1.0 / (1.0 - CIAFN)) annotation(Placement(visible = true, transformation(origin = {243.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_3 Prod_3_1 annotation(Placement(visible = true, transformation(origin = {215.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular NREM(x_vals = 0:0.25:1, y_vals = {0.0,0.15,0.5,0.85,1.0}) annotation(Placement(visible = true, transformation(origin = {175.0,51.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -180)));
        Modelica.Blocks.Math.Division CIR annotation(Placement(visible = true, transformation(origin = {107.0,6.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source4 annotation(Placement(visible = true, transformation(origin = {-150.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 P_Generation annotation(Placement(visible = true, transformation(origin = {-120.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Pollution(x0 = Pollution_0) annotation(Placement(visible = true, transformation(origin = {-70.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 P_Absorption annotation(Placement(visible = true, transformation(origin = {-20.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink4 annotation(Placement(visible = true, transformation(origin = {10.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Linear NotCIAF(m = -1.0, b = 1.0) annotation(Placement(visible = true, transformation(origin = {189.0,10.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain Pol_Ratio(k = 1.0 / POLS) annotation(Placement(visible = true, transformation(origin = {-29.0,-95.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular FCM(x_vals = 0:5, y_vals = {2.4,1.0,0.6,0.4,0.3,0.2}) annotation(Placement(visible = true, transformation(origin = {-85.0,-47.0}, extent = {{-15.0,15.0},{15.0,-15.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_3 Prod_3_2 annotation(Placement(visible = true, transformation(origin = {-117.0,-66.0}, extent = {{15.0,-14.0},{-15.0,14.0}}, rotation = 90)));
        SystemDynamics.Auxiliary.Gain Food_Ratio(k = FC / FN) annotation(Placement(visible = true, transformation(origin = {-149.0,-66.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source5 annotation(Placement(visible = true, transformation(origin = {50.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CIAFG annotation(Placement(visible = true, transformation(origin = {80.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level CIAF(x0 = CIAF_0) annotation(Placement(visible = true, transformation(origin = {130.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CIAFD annotation(Placement(visible = true, transformation(origin = {180.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink5 annotation(Placement(visible = true, transformation(origin = {210.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIAF_D(k = 1.0 / CIAFT) annotation(Placement(visible = true, transformation(origin = {163.0,-180.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIAF_G(k = 1.0 / CIAFT) annotation(Placement(visible = true, transformation(origin = {95.0,-180.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_3 annotation(Placement(visible = true, transformation(origin = {120.0,-230.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division P_Abs annotation(Placement(visible = true, transformation(origin = {-43.0,-180.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain Pol_norm(k = POLN) annotation(Placement(visible = true, transformation(origin = {-135.0,-180.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_4 annotation(Placement(visible = true, transformation(origin = {-160.0,-220.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular POLCM(x_vals = 0:5, y_vals = {0.05,1.0,3.0,5.4,7.4,8.0}) annotation(Placement(visible = true, transformation(origin = {-117.0,-220.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular POLAT(x_vals = 0:10:60, y_vals = {0.6,2.5,5.0,8.0,11.5,15.5,20.0}) annotation(Placement(visible = true, transformation(origin = {-39.0,-220.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular CFIFR(x_vals = 0:0.5:2, y_vals = {1.0,0.6,0.3,0.15,0.1}) annotation(Placement(visible = true, transformation(origin = {69.0,-230.0}, extent = {{15.0,16.0},{-15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular FPM(x_vals = 0:10:60, y_vals = {1.02,0.9,0.65,0.35,0.2,0.1,0.05}) annotation(Placement(visible = true, transformation(origin = {-33.0,-66.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_5 annotation(Placement(visible = true, transformation(origin = {229.0,-200.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIRA(k = 1.0 / CIAFN) annotation(Placement(visible = true, transformation(origin = {255.0,-225.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular FPCI(x_vals = 0:6, y_vals = {0.5,1.0,1.4,1.7,1.9,2.05,2.2}) annotation(Placement(visible = true, transformation(origin = {-153.0,-107.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular CIQR(x_vals = 0:0.5:2, y_vals = {0.7,0.8,1.0,1.5,2.0}) annotation(Placement(visible = true, transformation(origin = {167.0,-230.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        Modelica.Blocks.Math.Division QLMF annotation(Placement(visible = true, transformation(origin = {315.0,-223.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular QLM(x_vals = 0:5, y_vals = {0.2,1.0,1.7,2.3,2.7,2.9}) annotation(Placement(visible = true, transformation(origin = {319.0,-169.0}, extent = {{15.0,17.0},{-15.0,-17.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular QLF(x_vals = 0:4, y_vals = {0.0,1.0,1.8,2.4,2.7}) annotation(Placement(visible = true, transformation(origin = {381.0,-232.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_4 Prod_4_1 annotation(Placement(visible = true, transformation(origin = {375.0,-163.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular QLC(x_vals = 0:0.5:5, y_vals = {2.0,1.3,1.0,0.75,0.55,0.45,0.38,0.3,0.25,0.22,0.2}) annotation(Placement(visible = true, transformation(origin = {335.0,-113.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular QLP(x_vals = 0:10:60, y_vals = {1.04,0.85,0.6,0.3,0.15,0.05,0.02}) annotation(Placement(visible = true, transformation(origin = {375.0,-113.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -90)));
        SystemDynamics.Auxiliary.Gain Quality_of_Life(k = QLS) annotation(Placement(visible = true, transformation(origin = {412.0,-163.0}, extent = {{-20.0,-21.0},{20.0,21.0}}, rotation = 0)));
      equation
        connect(QLP.y,Prod_4_1.u1) annotation(Line(visible = true, points = {{375.0,-96.5},{375.0,-139.75},{375.0,-139.75},{375.0,-152.5}}, color = {0,0,191}));
        connect(QLC.y,Prod_4_1.u2) annotation(Line(visible = true, points = {{335.0,-96.5},{335.0,-157.0},{366.6,-157.0}}, color = {0,0,191}));
        connect(Prod_4_1.y,Quality_of_Life.u) annotation(Line(visible = true, points = {{385.5,-163.0},{398.0,-163.0}}, color = {0,0,191}));
        connect(QLF.y,Prod_4_1.u4) annotation(Line(visible = true, points = {{364.5,-232.0},{356.0,-232.0},{356.0,-190.0},{375.0,-190.0},{375.0,-173.5}}, color = {0,0,191}));
        connect(QLM.y,Prod_4_1.u3) annotation(Line(visible = true, points = {{335.5,-169.0},{366.6,-169.0}}, color = {0,0,191}));
        connect(Food_Ratio.y,BRFM.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-176.0,-66.0},{-176.0,22.2}}, color = {0,0,191}));
        connect(BRFM.y,Prod_5_1.u2) annotation(Line(visible = true, points = {{-176.0,52.6},{-176.0,60.0},{-167.36,60.0},{-167.36,72.0}}, color = {0,0,191}));
        connect(BRPM.u,QLP.u) annotation(Line(visible = true, points = {{-135.0,23.0},{-135.0,-20.0},{-214.0,-20.0},{-214.0,214.0},{375.0,214.0},{375.0,-125.0}}, color = {0,0,191}));
        connect(Pol_Ratio.y,BRPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,-20.0},{-135.0,-20.0},{-135.0,23.0}}, color = {0,0,191}));
        connect(BRPM.y,Prod_5_1.u3) annotation(Line(visible = true, points = {{-135.0,51.5},{-135.0,60.0},{-160.0,60.0},{-160.0,68.8}}, color = {0,0,191}));
        connect(Crowd_Rat.y,BRCM.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-95.0,8.0},{-95.0,23.0}}, color = {0,0,191}));
        connect(BRCM.y,Prod_5_1.u4) annotation(Line(visible = true, points = {{-95.0,51.5},{-95.0,66.0},{-152.64,66.0},{-152.64,72.0}}, color = {0,0,191}));
        connect(Crowd_Rat.y,DRCM.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-25.0,8.0},{-25.0,23.0}}, color = {0,0,191}));
        connect(DRCM.y,Prod_5_2.u2) annotation(Line(visible = true, points = {{-25.0,51.5},{-25.0,60.0},{12.64,60.0},{12.64,71.5}}, color = {0,0,191}));
        connect(Pol_Ratio.y,DRPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,8.0},{15.0,8.0},{15.0,23.0}}, color = {0,0,191}));
        connect(DRPM.y,Prod_5_2.u3) annotation(Line(visible = true, points = {{15.0,51.5},{15.0,60.0},{20.0,60.0},{20.0,68.5}}, color = {0,0,191}));
        connect(Food_Ratio.y,DRFM.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-176.0,-66.0},{-176.0,0.0},{55.0,0.0},{55.0,23.0}}, color = {0,0,191}));
        connect(DRFM.y,Prod_5_2.u4) annotation(Line(visible = true, points = {{55.0,51.5},{55.0,60.0},{27.36,60.0},{27.36,71.5}}, color = {0,0,191}));
        connect(Crowd_Rat.y,QLC.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-206.0,8.0},{-206.0,208.0},{335.0,208.0},{335.0,-125.0}}, color = {0,0,191}));
        connect(FCM.u,Crowd_Rat.y) annotation(Line(visible = true, points = {{-73.0,-47.0},{-60.0,-47.0},{-60.0,23.8}}, color = {0,0,191}));
        connect(Crowd_Rat.u,Population.y3) annotation(Line(visible = true, points = {{-60.0,46.2},{-60.0,66.0},{-70.0,66.0},{-70.0,127.0}}, color = {0,0,191}));
        connect(Prod_2_5.u2,Prod_3_1.u2) annotation(Line(visible = true, points = {{239.5,-200.0},{260.0,-200.0},{260.0,-20.0},{160.0,-20.0},{160.0,30.0},{203.8,30.0}}, color = {0,0,191}));
        connect(Prod_3_1.y,ECIR.u) annotation(Line(visible = true, points = {{226.2,30.0},{232.5,30.0}}, color = {0,0,191}));
        connect(CIR.y,Prod_3_1.u2) annotation(Line(visible = true, points = {{90.5,6.0},{80.0,6.0},{80.0,30.0},{203.8,30.0}}, color = {0,0,191}));
        connect(NotCIAF.y,Prod_3_1.u3) annotation(Line(visible = true, points = {{202.5,10.0},{215.0,10.0},{215.0,19.5}}, color = {0,0,191}));
        connect(Prod_3_1.u1,NREM.y) annotation(Line(visible = true, points = {{215.0,40.5},{215.0,51.0},{191.5,51.0}}, color = {0,0,191}));
        connect(QLF.y,QLMF.u2) annotation(Line(visible = true, points = {{364.5,-232.0},{333.0,-232.0}}, color = {0,0,191}));
        connect(QLMF.y,CIQR.u) annotation(Line(visible = true, points = {{298.5,-223.0},{290.0,-223.0},{290.0,-256.0},{190.0,-256.0},{190.0,-230.0},{179.0,-230.0}}, color = {0,0,191}));
        connect(P_Abs.u1,Pollution.y3) annotation(Line(visible = true, points = {{-61.0,-170.4},{-70.0,-170.4},{-70.0,-153.0}}, color = {0,0,191}));
        connect(CIG_norm.y,CI_Generation.u) annotation(Line(visible = true, points = {{75.5,-77.0},{80.0,-77.0},{80.0,-61.0}}, color = {0,0,191}));
        connect(CI_Discard.y1,Sink3.MassInPort1) annotation(Line(visible = true, points = {{190.0,-40.0},{199.0,-40.0}}, color = {191,0,191}));
        connect(Capital_Investment.u2,CI_Discard.y) annotation(Line(visible = true, points = {{152.0,-40.0},{170.0,-40.0}}, color = {191,0,191}));
        connect(Source3.MassInPort1,CI_Generation.y) annotation(Line(visible = true, points = {{61.0,-40.0},{70.0,-40.0}}, color = {191,0,191}));
        connect(CI_Generation.y1,Capital_Investment.u1) annotation(Line(visible = true, points = {{90.0,-40.0},{108.0,-40.0}}, color = {191,0,191}));
        connect(FPM.u,POLAT.u) annotation(Line(visible = true, points = {{-21.0,-66.0},{0.0,-66.0},{0.0,-220.0},{-27.0,-220.0}}, color = {0,0,191}));
        connect(CIRA.y,FPCI.u) annotation(Line(visible = true, points = {{265.5,-225.0},{280.0,-225.0},{280.0,-252.0},{-210.0,-252.0},{-210.0,-107.0},{-165.0,-107.0}}, color = {0,0,191}));
        connect(Prod_2_1.u1,Population.y1) annotation(Line(visible = true, points = {{128.8,80.0},{100.0,80.0},{100.0,194.0},{-46.0,194.0},{-46.0,151.0},{-53.0,151.0}}, color = {0,0,191}));
        connect(NRMM.u,DRMM.u) annotation(Line(visible = true, points = {{213.0,80.0},{300.0,80.0},{300.0,171.0},{63.0,171.0}}, color = {0,0,191}));
        connect(NRMM.u,BRMM.u) annotation(Line(visible = true, points = {{213.0,80.0},{300.0,80.0},{300.0,200.0},{-70.0,200.0},{-70.0,185.0},{-93.0,185.0}}, color = {0,0,191}));
        connect(CFIFR.u,QLF.u) annotation(Line(visible = true, points = {{57.0,-230.0},{40.0,-230.0},{40.0,-260.0},{400.0,-260.0},{400.0,-232.0},{393.0,-232.0}}, color = {0,0,191}));
        connect(Food_Ratio.y,QLF.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-220.0,-66.0},{-220.0,-260.0},{400.0,-260.0},{400.0,-232.0},{393.0,-232.0}}, color = {0,0,191}));
        connect(Pol_Ratio.y,FPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,-66.0},{-21.0,-66.0}}, color = {0,0,191}));
        connect(CIR.y,POLCM.u) annotation(Line(visible = true, points = {{90.5,6.0},{80.0,6.0},{80.0,-6.0},{10.0,-6.0},{10.0,-116.0},{-100.0,-116.0},{-100.0,-220.0},{-105.0,-220.0}}, color = {0,0,191}));
        connect(QLM.u,CIM.u) annotation(Line(visible = true, points = {{307.0,-169.0},{300.0,-169.0},{300.0,-100.0},{125.0,-100.0}}, color = {0,0,191}));
        connect(QLM.y,QLMF.u1) annotation(Line(visible = true, points = {{335.5,-169.0},{344.0,-169.0},{344.0,-214.0},{333.0,-214.0}}, color = {0,0,191}));
        connect(MSL.y,QLM.u) annotation(Line(visible = true, points = {{287.5,30.0},{300.0,30.0},{300.0,-169.0},{307.0,-169.0}}, color = {0,0,191}));
        connect(NRMM.y,Prod_2_1.u2) annotation(Line(visible = true, points = {{184.5,80.0},{151.2,80.0}}, color = {0,0,191}));
        connect(MSL.y,NRMM.u) annotation(Line(visible = true, points = {{287.5,30.0},{300.0,30.0},{300.0,80.0},{213.0,80.0}}, color = {0,0,191}));
        connect(ECIR.y,MSL.u) annotation(Line(visible = true, points = {{253.5,30.0},{266.5,30.0}}, color = {0,0,191}));
        connect(NotCIAF.u,CIAF.y1) annotation(Line(visible = true, points = {{175.5,10.0},{168.0,10.0},{168.0,-8.0},{240.0,-8.0},{240.0,-110.0},{154.0,-110.0},{154.0,-129.0},{147.0,-129.0}}, color = {0,0,191}));
        connect(Prod_2_2.u1,CIR.u2) annotation(Line(visible = true, points = {{30.2,-100.0},{20.0,-100.0},{20.0,-14.0},{140.0,-14.0},{140.0,-3.6},{125.0,-3.6}}, color = {0,0,191}));
        connect(Prod_2_2.u1,Population.y1) annotation(Line(visible = true, points = {{30.2,-100.0},{20.0,-100.0},{20.0,-14.0},{320.0,-14.0},{320.0,194.0},{-46.0,194.0},{-46.0,151.0},{-53.0,151.0}}, color = {0,0,191}));
        connect(Prod_2_4.u1,Population.y) annotation(Line(visible = true, points = {{-171.2,-220.0},{-200.0,-220.0},{-200.0,164.0},{-94.0,164.0},{-94.0,151.0},{-87.0,151.0}}, color = {0,0,191}));
        connect(CID_norm.u,Capital_Investment.y2) annotation(Line(visible = true, points = {{154.5,-77.0},{145.0,-77.0},{145.0,-53.0}}, color = {0,0,191}));
        connect(CID_norm.y,CI_Discard.u) annotation(Line(visible = true, points = {{175.5,-77.0},{180.0,-77.0},{180.0,-61.0}}, color = {0,0,191}));
        connect(CIM.y,Prod_2_2.u2) annotation(Line(visible = true, points = {{96.5,-100.0},{49.8,-100.0}}, color = {0,0,191}));
        connect(Prod_2_2.y,CIG_norm.u) annotation(Line(visible = true, points = {{40.0,-88.8},{40.0,-77.0},{54.5,-77.0}}, color = {0,0,191}));
        connect(Pol_Ratio.u,Pollution.y1) annotation(Line(visible = true, points = {{-39.5,-95.0},{-46.0,-95.0},{-46.0,-129.0},{-53.0,-129.0}}, color = {0,0,191}));
        connect(POLCM.y,Prod_2_4.u2) annotation(Line(visible = true, points = {{-133.5,-220.0},{-148.8,-220.0}}, color = {0,0,191}));
        connect(Prod_2_4.y,Pol_norm.u) annotation(Line(visible = true, points = {{-160.0,-208.8},{-160.0,-180.0},{-145.5,-180.0}}, color = {0,0,191}));
        connect(Pol_norm.y,P_Generation.u) annotation(Line(visible = true, points = {{-124.5,-180.0},{-120.0,-180.0},{-120.0,-161.0}}, color = {0,0,191}));
        connect(POLAT.y,P_Abs.u2) annotation(Line(visible = true, points = {{-55.5,-220.0},{-70.0,-220.0},{-70.0,-189.6},{-61.0,-189.6}}, color = {0,0,191}));
        connect(P_Abs.y,P_Absorption.u) annotation(Line(visible = true, points = {{-26.5,-180.0},{-20.0,-180.0},{-20.0,-161.0}}, color = {0,0,191}));
        connect(CFIFR.y,Prod_2_3.u1) annotation(Line(visible = true, points = {{85.5,-230.0},{108.8,-230.0}}, color = {0,0,191}));
        connect(CIQR.y,Prod_2_3.u2) annotation(Line(visible = true, points = {{150.5,-230.0},{131.2,-230.0}}, color = {0,0,191}));
        connect(CIAF_G.y,CIAFG.u) annotation(Line(visible = true, points = {{84.5,-180.0},{80.0,-180.0},{80.0,-161.0}}, color = {0,0,191}));
        connect(CIAF_G.u,Prod_2_3.y) annotation(Line(visible = true, points = {{105.5,-180.0},{120.0,-180.0},{120.0,-218.8}}, color = {0,0,191}));
        connect(CIRA.u,Prod_2_5.y) annotation(Line(visible = true, points = {{244.5,-225.0},{229.0,-225.0},{229.0,-211.2}}, color = {0,0,191}));
        connect(Prod_2_5.u1,CIAF.y3) annotation(Line(visible = true, points = {{218.5,-200.0},{130.0,-200.0},{130.0,-153.0}}, color = {0,0,191}));
        connect(CIAF_D.u,CIAF.y2) annotation(Line(visible = true, points = {{152.5,-180.0},{145.0,-180.0},{145.0,-153.0}}, color = {0,0,191}));
        connect(CIAF_D.y,CIAFD.u) annotation(Line(visible = true, points = {{173.5,-180.0},{180.0,-180.0},{180.0,-161.0}}, color = {0,0,191}));
        connect(CIAFD.y1,Sink5.MassInPort1) annotation(Line(visible = true, points = {{190.0,-140.0},{199.0,-140.0}}, color = {191,0,191}));
        connect(CIAF.u2,CIAFD.y) annotation(Line(visible = true, points = {{152.0,-140.0},{170.0,-140.0}}, color = {191,0,191}));
        connect(CIAFG.y1,CIAF.u1) annotation(Line(visible = true, points = {{90.0,-140.0},{108.0,-140.0}}, color = {191,0,191}));
        connect(Source5.MassInPort1,CIAFG.y) annotation(Line(visible = true, points = {{61.0,-140.0},{70.0,-140.0}}, color = {191,0,191}));
        connect(P_Absorption.y1,Sink4.MassInPort1) annotation(Line(visible = true, points = {{-10.0,-140.0},{-1.0,-140.0}}, color = {191,0,191}));
        connect(Pollution.u2,P_Absorption.y) annotation(Line(visible = true, points = {{-48.0,-140.0},{-30.0,-140.0}}, color = {191,0,191}));
        connect(P_Generation.y1,Pollution.u1) annotation(Line(visible = true, points = {{-110.0,-140.0},{-92.0,-140.0}}, color = {191,0,191}));
        connect(Source4.MassInPort1,P_Generation.y) annotation(Line(visible = true, points = {{-139.0,-140.0},{-130.0,-140.0}}, color = {191,0,191}));
        connect(FPCI.y,Prod_3_2.u3) annotation(Line(visible = true, points = {{-136.5,-107.0},{-117.0,-107.0},{-117.0,-76.5}}, color = {0,0,191}));
        connect(FPM.y,Prod_3_2.u2) annotation(Line(visible = true, points = {{-49.5,-66.0},{-107.2,-66.0}}, color = {0,0,191}));
        connect(Prod_3_2.u1,FCM.y) annotation(Line(visible = true, points = {{-117.0,-55.5},{-117.0,-47.0},{-101.5,-47.0}}, color = {0,0,191}));
        connect(Prod_3_2.y,Food_Ratio.u) annotation(Line(visible = true, points = {{-126.8,-66.0},{-138.5,-66.0}}, color = {0,0,191}));
        connect(CIR.u1,Capital_Investment.y1) annotation(Line(visible = true, points = {{125.0,15.6},{154.0,15.6},{154.0,-29.0},{147.0,-29.0}}, color = {0,0,191}));
        connect(NRFR.y,NREM.u) annotation(Line(visible = true, points = {{147.5,51.0},{153.75,51.0},{153.75,51.0},{163.0,51.0}}, color = {0,0,191}));
        connect(DRMM.y,Prod_5_2.u5) annotation(Line(visible = true, points = {{34.5,171.0},{26.0,171.0},{26.0,104.0},{40.0,104.0},{40.0,79.0},{31.2,79.0}}, color = {0,0,191}));
        connect(DR_norm.y,Death_Rate.u) annotation(Line(visible = true, points = {{-13.5,103.0},{-20.0,103.0},{-20.0,119.0}}, color = {0,0,191}));
        connect(Prod_5_2.y,DR_norm.u) annotation(Line(visible = true, points = {{20.0,89.5},{20.0,103.0},{7.5,103.0}}, color = {0,0,191}));
        connect(Prod_5_2.u1,Population.y2) annotation(Line(visible = true, points = {{8.8,79.0},{-55.0,79.0},{-55.0,127.0}}, color = {0,0,191}));
        connect(BRMM.y,Prod_5_1.u1) annotation(Line(visible = true, points = {{-121.5,185.0},{-180.0,185.0},{-180.0,80.0},{-171.2,80.0}}, color = {0,0,191}));
        connect(Prod_5_1.u5,Population.y4) annotation(Line(visible = true, points = {{-148.8,80.0},{-85.0,80.0},{-85.0,127.0}}, color = {0,0,191}));
        connect(Prod_5_1.y,BR_norm.u) annotation(Line(visible = true, points = {{-160.0,91.2},{-160.0,103.0},{-145.5,103.0}}, color = {0,0,191}));
        connect(BR_norm.y,Birth_Rate.u) annotation(Line(visible = true, points = {{-124.5,103.0},{-120.0,103.0},{-120.0,119.0}}, color = {0,0,191}));
        connect(Source1.MassInPort1,Birth_Rate.y) annotation(Line(visible = true, points = {{-139.0,140.0},{-130.0,140.0}}, color = {191,0,191}));
        connect(Birth_Rate.y1,Population.u1) annotation(Line(visible = true, points = {{-110.0,140.0},{-92.0,140.0}}, color = {191,0,191}));
        connect(Population.u2,Death_Rate.y) annotation(Line(visible = true, points = {{-48.0,140.0},{-30.0,140.0}}, color = {191,0,191}));
        connect(Death_Rate.y1,Sink1.MassInPort1) annotation(Line(visible = true, points = {{-10.0,140.0},{-1.0,140.0}}, color = {191,0,191}));
        connect(Source2.MassInPort1,Generation.y) annotation(Line(visible = true, points = {{61.0,140.0},{70.0,140.0}}, color = {191,0,191}));
        connect(Gen_Const.y,Generation.u) annotation(Line(visible = true, points = {{80.0,111.5},{80.0,119.0}}, color = {0,0,191}));
        connect(NRFR.u,Natural_Resources.y4) annotation(Line(visible = true, points = {{126.5,51.0},{115.0,51.0},{115.0,127.0}}, color = {0,0,191}));
        connect(Generation.y1,Natural_Resources.u1) annotation(Line(visible = true, points = {{90.0,140.0},{108.0,140.0}}, color = {191,0,191}));
        connect(Prod_2_1.y,NR_norm.u) annotation(Line(visible = true, points = {{140.0,91.2},{140.0,103.0},{152.5,103.0}}, color = {0,0,191}));
        connect(Natural_Resources.u2,Depletion.y) annotation(Line(visible = true, points = {{152.0,140.0},{170.0,140.0}}, color = {191,0,191}));
        connect(Depletion.y1,Sink2.MassInPort1) annotation(Line(visible = true, points = {{190.0,140.0},{199.0,140.0}}, color = {191,0,191}));
        connect(NR_norm.y,Depletion.u) annotation(Line(visible = true, points = {{173.5,103.0},{180.0,103.0},{180.0,119.0}}, color = {0,0,191}));
        Pop = Population.y;
        Pol = Pollution.y;
        Pol_rat = Pol_Ratio.y;
        Cap_inv = Capital_Investment.y;
        Qual_life = Quality_of_Life.y;
        Nat_res = Natural_Resources.y;
        annotation(Icon(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-220.0,-262.0},{442.0,220.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-70.0,102.0},{296.0,170.0}}, textString = "Forrester's", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-70.0,-22.0},{296.0,46.0}}, textString = "Original", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-70.0,-100.0},{296.0,-32.0}}, textString = "World Model", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-62.0,6.0},{-58.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{98.0,192.0},{102.0,196.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{138.0,-16.0},{142.0,-12.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-22.0},{2.0,-18.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-178.0,-2.0},{-174.0,2.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{78.0,4.0},{82.0,8.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-97.0},{2.0,-93.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-178.0,-68.0},{-174.0,-64.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-68.0},{2.0,-64.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{158.0,28.0},{162.0,32.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{38.0,-262.0},{42.0,-258.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,-102.0},{302.0,-98.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,28.0},{302.0,32.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,78.0},{302.0,82.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,169.0},{302.0,173.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{342.0,-171.0},{346.0,-167.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{354.0,-234.0},{358.0,-230.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-97.0,6.0},{-93.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-137.0,-22.0},{-133.0,-18.0}})}), Documentation(info = "<html>
This is the original scenario of <a href=\"http://en.wikipedia.org/wiki/Jay_Wright_Forrester\">Jay Forrester's</a> <font color=red><b>WORLD2</b></font> model as described in his <a href=\"http://www.amazon.com/World-Dynamics-Jay-W-Forrester/dp/1883823382/ref=ed_oe_h/103-2487145-1208659\">World Dynamics</a> book. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Cellier, F.E. (1991), <a href=\"http://www.amazon.com/Continuous-System-Modeling-Fran%C3%A7ois-Cellier/dp/0387975020\">Continuous System Modeling</a>, Springer-Verlag, New York, ISBN: 0-387-97502-0, 755p.
<li> Forrester, J.W. (1971), <a href=\"http://www.amazon.com/World-Dynamics-Jay-W-Forrester/dp/1883823382/ref=ed_oe_h/103-2487145-1208659\">World Dynamics</a>, Pegasus Communications, 160p.
</ol> <p>
 
<hr> <p>
 
Simulate the model across 200 years, and plot the state variables as well as the quality of life as functions of the calendar years: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/SD_Fig1.png\"> <p>
 
<hr> <p>
</html>", revisions = ""), experiment(StartTime = 1900, StopTime = 2100), experimentSetupOutput);
      end Scenario_1;
      model Scenario_2 "2nd Scenario"
        parameter Real Population_0 = 1650000000.0 "World population in 1900";
        parameter Real Pollution_0 = 200000000.0 "Pollution in 1900";
        parameter Real Nat_Resources_0(unit = "ton") = 900000000000.0 "Unrecoverable natural resources in 1900";
        parameter Real Cap_Invest_0(unit = "dollar") = 400000000.0 "Capital investment in 1900";
        parameter Real CIAF_0 = 0.2 "Proportion of capital investment in agriculture in 1900";
        parameter Real BRN(unit = "1/yr") = 0.04 "Normal birth rate";
        parameter Real CIAFN(unit = "dollar") = 0.3 "CIAF normalization";
        parameter Real CIAFT(unit = "yr") = 15.0 "CIAF time constant";
        parameter Real CIDN(unit = "dollar/yr") = 0.025 "Normal capital discard";
        parameter Real CIGN(unit = "dollar/yr") = 0.05 "Normal capital generation";
        parameter Real DRN(unit = "1/yr") = 0.028 "Normal death rate";
        parameter Real ECIRN(unit = "dollar") = 1.0 "Capital normalization";
        parameter Real FC(unit = "kg/yr") = 1.0 "Food coefficient";
        parameter Real FN(unit = "kg/yr") = 1.0 "Food normalization";
        parameter Real Land_Area(unit = "hectare") = 135000000.0 "Area of arable land";
        parameter Real NRI(unit = "ton") = 900000000000.0 "Initial natural resources";
        parameter Real POLN(unit = "1/yr") = 1.0 "Normal pollution";
        parameter Real POLS = 3599900000.0 "Standard pollution";
        parameter Real Pop_dens_norm(unit = "1/hectare") = 26.5 "Normal population density";
        parameter Real QLS = 1.0 "Standard quality of life";
        output Real Pop "World population";
        output Real Pol "Pollution";
        output Real Pol_rat "Pollution ratio";
        output Real Cap_inv(unit = "dollar") "Capital investment";
        output Real Qual_life "Quality of life";
        output Real Nat_res(unit = "ton") "Natural unrecoverable resources";
        Real NRUN(unit = "1/yr") "Normal resource utilization";
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {-150.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Birth_Rate annotation(Placement(visible = true, transformation(origin = {-120.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Population(x0 = Population_0) annotation(Placement(visible = true, transformation(origin = {-70.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Death_Rate annotation(Placement(visible = true, transformation(origin = {-20.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink1 annotation(Placement(visible = true, transformation(origin = {10.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source2 annotation(Placement(visible = true, transformation(origin = {50.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Generation annotation(Placement(visible = true, transformation(origin = {80.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Natural_Resources(x0 = Nat_Resources_0) annotation(Placement(visible = true, transformation(origin = {130.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Depletion annotation(Placement(visible = true, transformation(origin = {180.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink2 annotation(Placement(visible = true, transformation(origin = {210.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Const Gen_Const(k = 0) annotation(Placement(visible = true, transformation(origin = {80.0,101.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular BRMM(x_vals = {0,1,2,3,4,5,20}, y_vals = {1.2,1.0,0.85,0.75,0.7,0.7,0.7}) annotation(Placement(visible = true, transformation(origin = {-105.0,185.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular DRMM(x_vals = {0,0.5,1,1.5,2,2.5,3,3.5,4,4.5,5,20}, y_vals = {3.0,1.8,1.0,0.8,0.7,0.6,0.53,0.5,0.5,0.5,0.5,0.5}) annotation(Placement(visible = true, transformation(origin = {51.0,171.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain BR_norm(k = BRN) annotation(Placement(visible = true, transformation(origin = {-135.0,103.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_5 Prod_5_1 annotation(Placement(visible = true, transformation(origin = {-160.0,80.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain DR_norm(k = DRN) annotation(Placement(visible = true, transformation(origin = {-3.0,103.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_5 Prod_5_2 annotation(Placement(visible = true, transformation(origin = {20.0,79.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular BRFM(x_vals = {0,1,2,3,4,20}, y_vals = {0.0,1.0,1.6,1.9,2.0,2.0}) annotation(Placement(visible = true, transformation(origin = {-176.0,35.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular BRPM(x_vals = 0:10:60, y_vals = {1.02,0.9,0.7,0.4,0.25,0.15,0.1}) annotation(Placement(visible = true, transformation(origin = {-135.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular BRCM(x_vals = 0:5, y_vals = {1.05,1.0,0.9,0.7,0.6,0.55}) annotation(Placement(visible = true, transformation(origin = {-95.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRCM(x_vals = 0:5, y_vals = {0.9,1.0,1.2,1.5,1.9,3.0}) annotation(Placement(visible = true, transformation(origin = {-25.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRPM(x_vals = 0:10:60, y_vals = {0.92,1.3,2.0,3.2,4.8,6.8,9.2}) annotation(Placement(visible = true, transformation(origin = {15.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRFM(x_vals = {0,0.25,0.5,0.75,1,1.25,1.5,1.75,2,20}, y_vals = {30.0,3.0,2.0,1.4,1.0,0.7,0.6,0.5,0.5,0.5}) annotation(Placement(visible = true, transformation(origin = {55.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Auxiliary.Gain Crowd_Rat(k = 1.0 / (Land_Area * Pop_dens_norm)) annotation(Placement(visible = true, transformation(origin = {-60.0,35.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = -90)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_1 annotation(Placement(visible = true, transformation(origin = {140.0,80.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular NRMM(x_vals = 0:10, y_vals = {0.0,1.0,1.8,2.4,2.9,3.3,3.6,3.8,3.9,3.95,4.0}) annotation(Placement(visible = true, transformation(origin = {201.0,80.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Sources.Source Source3 annotation(Placement(visible = true, transformation(origin = {50.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CI_Generation annotation(Placement(visible = true, transformation(origin = {80.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Capital_Investment(x0 = Cap_Invest_0) annotation(Placement(visible = true, transformation(origin = {130.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CI_Discard annotation(Placement(visible = true, transformation(origin = {180.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink3 annotation(Placement(visible = true, transformation(origin = {210.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIG_norm(k = CIGN) annotation(Placement(visible = true, transformation(origin = {65.0,-77.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_2 annotation(Placement(visible = true, transformation(origin = {40.0,-100.0}, extent = {{-14.0,-16.0},{14.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CID_norm(k = CIDN) annotation(Placement(visible = true, transformation(origin = {165.0,-77.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular CIM(x_vals = {0,1,2,3,4,5,20}, y_vals = {0.1,1.0,1.8,2.4,2.8,3.0,3.0}) annotation(Placement(visible = true, transformation(origin = {113.0,-100.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain MSL(k = 1.0 / ECIRN) annotation(Placement(visible = true, transformation(origin = {277.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain NRFR(k = 1.0 / NRI) annotation(Placement(visible = true, transformation(origin = {137.0,51.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain ECIR(k = 1.0 / (1.0 - CIAFN)) annotation(Placement(visible = true, transformation(origin = {243.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_3 Prod_3_1 annotation(Placement(visible = true, transformation(origin = {215.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular NREM(x_vals = 0:0.25:1, y_vals = {0.0,0.15,0.5,0.85,1.0}) annotation(Placement(visible = true, transformation(origin = {175.0,51.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -180)));
        Modelica.Blocks.Math.Division CIR annotation(Placement(visible = true, transformation(origin = {107.0,6.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source4 annotation(Placement(visible = true, transformation(origin = {-150.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 P_Generation annotation(Placement(visible = true, transformation(origin = {-120.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Pollution(x0 = Pollution_0) annotation(Placement(visible = true, transformation(origin = {-70.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 P_Absorption annotation(Placement(visible = true, transformation(origin = {-20.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink4 annotation(Placement(visible = true, transformation(origin = {10.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Linear NotCIAF(m = -1.0, b = 1.0) annotation(Placement(visible = true, transformation(origin = {189.0,10.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain Pol_Ratio(k = 1.0 / POLS) annotation(Placement(visible = true, transformation(origin = {-29.0,-95.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular FCM(x_vals = 0:5, y_vals = {2.4,1.0,0.6,0.4,0.3,0.2}) annotation(Placement(visible = true, transformation(origin = {-85.0,-47.0}, extent = {{-15.0,15.0},{15.0,-15.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_3 Prod_3_2 annotation(Placement(visible = true, transformation(origin = {-117.0,-66.0}, extent = {{15.0,-14.0},{-15.0,14.0}}, rotation = 90)));
        SystemDynamics.Auxiliary.Gain Food_Ratio(k = FC / FN) annotation(Placement(visible = true, transformation(origin = {-149.0,-66.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source5 annotation(Placement(visible = true, transformation(origin = {50.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CIAFG annotation(Placement(visible = true, transformation(origin = {80.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level CIAF(x0 = CIAF_0) annotation(Placement(visible = true, transformation(origin = {130.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CIAFD annotation(Placement(visible = true, transformation(origin = {180.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink5 annotation(Placement(visible = true, transformation(origin = {210.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIAF_D(k = 1.0 / CIAFT) annotation(Placement(visible = true, transformation(origin = {163.0,-180.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIAF_G(k = 1.0 / CIAFT) annotation(Placement(visible = true, transformation(origin = {95.0,-180.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_3 annotation(Placement(visible = true, transformation(origin = {120.0,-230.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division P_Abs annotation(Placement(visible = true, transformation(origin = {-43.0,-180.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain Pol_norm(k = POLN) annotation(Placement(visible = true, transformation(origin = {-135.0,-180.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_4 annotation(Placement(visible = true, transformation(origin = {-160.0,-220.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular POLCM(x_vals = {0,1,2,3,4,5,100}, y_vals = {0.05,1.0,3.0,5.4,7.4,8.0,8.0}) annotation(Placement(visible = true, transformation(origin = {-117.0,-220.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular POLAT(x_vals = 0:10:60, y_vals = {0.6,2.5,5.0,8.0,11.5,15.5,20.0}) annotation(Placement(visible = true, transformation(origin = {-39.0,-220.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular CFIFR(x_vals = {0,0.5,1,1.5,2,20}, y_vals = {1.0,0.6,0.3,0.15,0.1,0.1}) annotation(Placement(visible = true, transformation(origin = {69.0,-230.0}, extent = {{15.0,16.0},{-15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular FPM(x_vals = 0:10:60, y_vals = {1.02,0.9,0.65,0.35,0.2,0.1,0.05}) annotation(Placement(visible = true, transformation(origin = {-33.0,-66.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_5 annotation(Placement(visible = true, transformation(origin = {229.0,-200.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIRA(k = 1.0 / CIAFN) annotation(Placement(visible = true, transformation(origin = {255.0,-225.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular FPCI(x_vals = {0,1,2,3,4,5,6,100}, y_vals = {0.5,1.0,1.4,1.7,1.9,2.05,2.2,2.2}) annotation(Placement(visible = true, transformation(origin = {-153.0,-107.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular CIQR(y_vals = {0.7,0.8,1.0,1.5,2.0,2.0}, x_vals = {0,0.5,1,1.5,2,10}) annotation(Placement(visible = true, transformation(origin = {167.0,-230.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        Modelica.Blocks.Math.Division QLMF annotation(Placement(visible = true, transformation(origin = {315.0,-223.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular QLM(x_vals = {0,1,2,3,4,5,20}, y_vals = {0.2,1.0,1.7,2.3,2.7,2.9,2.9}) annotation(Placement(visible = true, transformation(origin = {319.0,-169.0}, extent = {{15.0,17.0},{-15.0,-17.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular QLF(x_vals = {0,1,2,3,4,20}, y_vals = {0.0,1.0,1.8,2.4,2.7,2.7}) annotation(Placement(visible = true, transformation(origin = {381.0,-232.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_4 Prod_4_1 annotation(Placement(visible = true, transformation(origin = {375.0,-163.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular QLC(x_vals = 0:0.5:5, y_vals = {2.0,1.3,1.0,0.75,0.55,0.45,0.38,0.3,0.25,0.22,0.2}) annotation(Placement(visible = true, transformation(origin = {335.0,-113.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular QLP(x_vals = 0:10:60, y_vals = {1.04,0.85,0.6,0.3,0.15,0.05,0.02}) annotation(Placement(visible = true, transformation(origin = {375.0,-113.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -90)));
        SystemDynamics.Auxiliary.Gain Quality_of_Life(k = QLS) annotation(Placement(visible = true, transformation(origin = {412.0,-163.0}, extent = {{-20.0,-21.0},{20.0,21.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World2.Utilities.Parameter_Change NR_norm annotation(Placement(visible = true, transformation(origin = {161.6845,105.0774}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      equation
        connect(NR_norm.u1,Prod_2_1.y) annotation(Line(visible = true, points = {{152.6845,109.0774},{140.0,109.0774},{140.0,91.2}}, color = {0,0,191}));
        connect(Depletion.u,NR_norm.y) annotation(Line(visible = true, points = {{180.0,119.0},{180.0,105.0},{170.6845,105.0774}}, color = {0,0,191}));
        connect(Food_Ratio.y,BRFM.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-176.0,-66.0},{-176.0,22.2}}, color = {0,0,191}));
        connect(BRFM.y,Prod_5_1.u2) annotation(Line(visible = true, points = {{-176.0,52.6},{-176.0,60.0},{-167.36,60.0},{-167.36,72.0}}, color = {0,0,191}));
        connect(BRPM.u,QLP.u) annotation(Line(visible = true, points = {{-135.0,23.0},{-135.0,-20.0},{-214.0,-20.0},{-214.0,214.0},{375.0,214.0},{375.0,-125.0}}, color = {0,0,191}));
        connect(Pol_Ratio.y,BRPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,-20.0},{-135.0,-20.0},{-135.0,23.0}}, color = {0,0,191}));
        connect(BRPM.y,Prod_5_1.u3) annotation(Line(visible = true, points = {{-135.0,51.5},{-135.0,60.0},{-160.0,60.0},{-160.0,68.8}}, color = {0,0,191}));
        connect(Crowd_Rat.y,BRCM.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-95.0,8.0},{-95.0,23.0}}, color = {0,0,191}));
        connect(BRCM.y,Prod_5_1.u4) annotation(Line(visible = true, points = {{-95.0,51.5},{-95.0,66.0},{-152.64,66.0},{-152.64,72.0}}, color = {0,0,191}));
        connect(Crowd_Rat.y,DRCM.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-25.0,8.0},{-25.0,23.0}}, color = {0,0,191}));
        connect(DRCM.y,Prod_5_2.u2) annotation(Line(visible = true, points = {{-25.0,51.5},{-25.0,60.0},{12.64,60.0},{12.64,71.5}}, color = {0,0,191}));
        connect(Pol_Ratio.y,DRPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,8.0},{15.0,8.0},{15.0,23.0}}, color = {0,0,191}));
        connect(DRPM.y,Prod_5_2.u3) annotation(Line(visible = true, points = {{15.0,51.5},{15.0,60.0},{20.0,60.0},{20.0,68.5}}, color = {0,0,191}));
        connect(Food_Ratio.y,DRFM.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-176.0,-66.0},{-176.0,0.0},{55.0,0.0},{55.0,23.0}}, color = {0,0,191}));
        connect(DRFM.y,Prod_5_2.u4) annotation(Line(visible = true, points = {{55.0,51.5},{55.0,60.0},{27.36,60.0},{27.36,71.5}}, color = {0,0,191}));
        connect(Crowd_Rat.y,QLC.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-206.0,8.0},{-206.0,208.0},{335.0,208.0},{335.0,-125.0}}, color = {0,0,191}));
        connect(FCM.u,Crowd_Rat.y) annotation(Line(visible = true, points = {{-73.0,-47.0},{-60.0,-47.0},{-60.0,23.8}}, color = {0,0,191}));
        connect(Crowd_Rat.u,Population.y3) annotation(Line(visible = true, points = {{-60.0,46.2},{-60.0,66.0},{-70.0,66.0},{-70.0,127.0}}, color = {0,0,191}));
        connect(Prod_2_5.u2,Prod_3_1.u2) annotation(Line(visible = true, points = {{239.5,-200.0},{260.0,-200.0},{260.0,-20.0},{160.0,-20.0},{160.0,30.0},{203.8,30.0}}, color = {0,0,191}));
        connect(Prod_3_1.y,ECIR.u) annotation(Line(visible = true, points = {{226.2,30.0},{232.5,30.0}}, color = {0,0,191}));
        connect(CIR.y,Prod_3_1.u2) annotation(Line(visible = true, points = {{90.5,6.0},{80.0,6.0},{80.0,30.0},{203.8,30.0}}, color = {0,0,191}));
        connect(NotCIAF.y,Prod_3_1.u3) annotation(Line(visible = true, points = {{202.5,10.0},{215.0,10.0},{215.0,19.5}}, color = {0,0,191}));
        connect(Prod_3_1.u1,NREM.y) annotation(Line(visible = true, points = {{215.0,40.5},{215.0,51.0},{191.5,51.0}}, color = {0,0,191}));
        connect(QLP.y,Prod_4_1.u1) annotation(Line(visible = true, points = {{375.0,-96.5},{375.0,-152.5}}, color = {0,0,191}));
        connect(QLC.y,Prod_4_1.u2) annotation(Line(visible = true, points = {{335.0,-96.5},{335.0,-157.0},{366.6,-157.0}}, color = {0,0,191}));
        connect(Prod_4_1.y,Quality_of_Life.u) annotation(Line(visible = true, points = {{385.5,-163.0},{398.0,-163.0}}, color = {0,0,191}));
        connect(QLF.y,Prod_4_1.u4) annotation(Line(visible = true, points = {{364.5,-232.0},{356.0,-232.0},{356.0,-190.0},{375.0,-190.0},{375.0,-173.5}}, color = {0,0,191}));
        connect(QLM.y,Prod_4_1.u3) annotation(Line(visible = true, points = {{335.5,-169.0},{366.6,-169.0}}, color = {0,0,191}));
        connect(QLF.y,QLMF.u2) annotation(Line(visible = true, points = {{364.5,-232.0},{333.0,-232.0}}, color = {0,0,191}));
        connect(QLMF.y,CIQR.u) annotation(Line(visible = true, points = {{298.5,-223.0},{290.0,-223.0},{290.0,-256.0},{190.0,-256.0},{190.0,-230.0},{179.0,-230.0}}, color = {0,0,191}));
        connect(P_Abs.u1,Pollution.y3) annotation(Line(visible = true, points = {{-61.0,-170.4},{-70.0,-170.4},{-70.0,-153.0}}, color = {0,0,191}));
        connect(CIG_norm.y,CI_Generation.u) annotation(Line(visible = true, points = {{75.5,-77.0},{80.0,-77.0},{80.0,-61.0}}, color = {0,0,191}));
        connect(CI_Discard.y1,Sink3.MassInPort1) annotation(Line(visible = true, points = {{190.0,-40.0},{199.0,-40.0}}, color = {191,0,191}));
        connect(Capital_Investment.u2,CI_Discard.y) annotation(Line(visible = true, points = {{152.0,-40.0},{170.0,-40.0}}, color = {191,0,191}));
        connect(Source3.MassInPort1,CI_Generation.y) annotation(Line(visible = true, points = {{61.0,-40.0},{70.0,-40.0}}, color = {191,0,191}));
        connect(CI_Generation.y1,Capital_Investment.u1) annotation(Line(visible = true, points = {{90.0,-40.0},{108.0,-40.0}}, color = {191,0,191}));
        connect(FPM.u,POLAT.u) annotation(Line(visible = true, points = {{-21.0,-66.0},{0.0,-66.0},{0.0,-220.0},{-27.0,-220.0}}, color = {0,0,191}));
        connect(CIRA.y,FPCI.u) annotation(Line(visible = true, points = {{265.5,-225.0},{280.0,-225.0},{280.0,-252.0},{-210.0,-252.0},{-210.0,-107.0},{-165.0,-107.0}}, color = {0,0,191}));
        connect(Prod_2_1.u1,Population.y1) annotation(Line(visible = true, points = {{128.8,80.0},{100.0,80.0},{100.0,194.0},{-46.0,194.0},{-46.0,151.0},{-53.0,151.0}}, color = {0,0,191}));
        connect(NRMM.u,DRMM.u) annotation(Line(visible = true, points = {{213.0,80.0},{300.0,80.0},{300.0,171.0},{63.0,171.0}}, color = {0,0,191}));
        connect(NRMM.u,BRMM.u) annotation(Line(visible = true, points = {{213.0,80.0},{300.0,80.0},{300.0,200.0},{-70.0,200.0},{-70.0,185.0},{-93.0,185.0}}, color = {0,0,191}));
        connect(CFIFR.u,QLF.u) annotation(Line(visible = true, points = {{57.0,-230.0},{40.0,-230.0},{40.0,-260.0},{400.0,-260.0},{400.0,-232.0},{393.0,-232.0}}, color = {0,0,191}));
        connect(Food_Ratio.y,QLF.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-220.0,-66.0},{-220.0,-260.0},{400.0,-260.0},{400.0,-232.0},{393.0,-232.0}}, color = {0,0,191}));
        connect(Pol_Ratio.y,FPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,-66.0},{-21.0,-66.0}}, color = {0,0,191}));
        connect(CIR.y,POLCM.u) annotation(Line(visible = true, points = {{90.5,6.0},{80.0,6.0},{80.0,-6.0},{10.0,-6.0},{10.0,-116.0},{-100.0,-116.0},{-100.0,-220.0},{-105.0,-220.0}}, color = {0,0,191}));
        connect(QLM.u,CIM.u) annotation(Line(visible = true, points = {{307.0,-169.0},{300.0,-169.0},{300.0,-100.0},{125.0,-100.0}}, color = {0,0,191}));
        connect(QLM.y,QLMF.u1) annotation(Line(visible = true, points = {{335.5,-169.0},{344.0,-169.0},{344.0,-214.0},{333.0,-214.0}}, color = {0,0,191}));
        connect(MSL.y,QLM.u) annotation(Line(visible = true, points = {{287.5,30.0},{300.0,30.0},{300.0,-169.0},{307.0,-169.0}}, color = {0,0,191}));
        connect(NRMM.y,Prod_2_1.u2) annotation(Line(visible = true, points = {{184.5,80.0},{151.2,80.0}}, color = {0,0,191}));
        connect(MSL.y,NRMM.u) annotation(Line(visible = true, points = {{287.5,30.0},{300.0,30.0},{300.0,80.0},{213.0,80.0}}, color = {0,0,191}));
        connect(ECIR.y,MSL.u) annotation(Line(visible = true, points = {{253.5,30.0},{266.5,30.0}}, color = {0,0,191}));
        connect(NotCIAF.u,CIAF.y1) annotation(Line(visible = true, points = {{175.5,10.0},{168.0,10.0},{168.0,-8.0},{240.0,-8.0},{240.0,-110.0},{154.0,-110.0},{154.0,-129.0},{147.0,-129.0}}, color = {0,0,191}));
        connect(Prod_2_2.u1,CIR.u2) annotation(Line(visible = true, points = {{30.2,-100.0},{20.0,-100.0},{20.0,-14.0},{140.0,-14.0},{140.0,-3.6},{125.0,-3.6}}, color = {0,0,191}));
        connect(Prod_2_2.u1,Population.y1) annotation(Line(visible = true, points = {{30.2,-100.0},{20.0,-100.0},{20.0,-14.0},{320.0,-14.0},{320.0,194.0},{-46.0,194.0},{-46.0,151.0},{-53.0,151.0}}, color = {0,0,191}));
        connect(Prod_2_4.u1,Population.y) annotation(Line(visible = true, points = {{-171.2,-220.0},{-200.0,-220.0},{-200.0,164.0},{-94.0,164.0},{-94.0,151.0},{-87.0,151.0}}, color = {0,0,191}));
        connect(CID_norm.u,Capital_Investment.y2) annotation(Line(visible = true, points = {{154.5,-77.0},{145.0,-77.0},{145.0,-53.0}}, color = {0,0,191}));
        connect(CID_norm.y,CI_Discard.u) annotation(Line(visible = true, points = {{175.5,-77.0},{180.0,-77.0},{180.0,-61.0}}, color = {0,0,191}));
        connect(CIM.y,Prod_2_2.u2) annotation(Line(visible = true, points = {{96.5,-100.0},{49.8,-100.0}}, color = {0,0,191}));
        connect(Prod_2_2.y,CIG_norm.u) annotation(Line(visible = true, points = {{40.0,-88.8},{40.0,-77.0},{54.5,-77.0}}, color = {0,0,191}));
        connect(Pol_Ratio.u,Pollution.y1) annotation(Line(visible = true, points = {{-39.5,-95.0},{-46.0,-95.0},{-46.0,-129.0},{-53.0,-129.0}}, color = {0,0,191}));
        connect(POLCM.y,Prod_2_4.u2) annotation(Line(visible = true, points = {{-133.5,-220.0},{-148.8,-220.0}}, color = {0,0,191}));
        connect(Prod_2_4.y,Pol_norm.u) annotation(Line(visible = true, points = {{-160.0,-208.8},{-160.0,-180.0},{-145.5,-180.0}}, color = {0,0,191}));
        connect(Pol_norm.y,P_Generation.u) annotation(Line(visible = true, points = {{-124.5,-180.0},{-120.0,-180.0},{-120.0,-161.0}}, color = {0,0,191}));
        connect(POLAT.y,P_Abs.u2) annotation(Line(visible = true, points = {{-55.5,-220.0},{-70.0,-220.0},{-70.0,-189.6},{-61.0,-189.6}}, color = {0,0,191}));
        connect(P_Abs.y,P_Absorption.u) annotation(Line(visible = true, points = {{-26.5,-180.0},{-20.0,-180.0},{-20.0,-161.0}}, color = {0,0,191}));
        connect(CFIFR.y,Prod_2_3.u1) annotation(Line(visible = true, points = {{85.5,-230.0},{108.8,-230.0}}, color = {0,0,191}));
        connect(CIQR.y,Prod_2_3.u2) annotation(Line(visible = true, points = {{150.5,-230.0},{131.2,-230.0}}, color = {0,0,191}));
        connect(CIAF_G.y,CIAFG.u) annotation(Line(visible = true, points = {{84.5,-180.0},{80.0,-180.0},{80.0,-161.0}}, color = {0,0,191}));
        connect(CIAF_G.u,Prod_2_3.y) annotation(Line(visible = true, points = {{105.5,-180.0},{120.0,-180.0},{120.0,-218.8}}, color = {0,0,191}));
        connect(CIRA.u,Prod_2_5.y) annotation(Line(visible = true, points = {{244.5,-225.0},{229.0,-225.0},{229.0,-211.2}}, color = {0,0,191}));
        connect(Prod_2_5.u1,CIAF.y3) annotation(Line(visible = true, points = {{218.5,-200.0},{130.0,-200.0},{130.0,-153.0}}, color = {0,0,191}));
        connect(CIAF_D.u,CIAF.y2) annotation(Line(visible = true, points = {{152.5,-180.0},{145.0,-180.0},{145.0,-153.0}}, color = {0,0,191}));
        connect(CIAF_D.y,CIAFD.u) annotation(Line(visible = true, points = {{173.5,-180.0},{180.0,-180.0},{180.0,-161.0}}, color = {0,0,191}));
        connect(CIAFD.y1,Sink5.MassInPort1) annotation(Line(visible = true, points = {{190.0,-140.0},{199.0,-140.0}}, color = {191,0,191}));
        connect(CIAF.u2,CIAFD.y) annotation(Line(visible = true, points = {{152.0,-140.0},{170.0,-140.0}}, color = {191,0,191}));
        connect(CIAFG.y1,CIAF.u1) annotation(Line(visible = true, points = {{90.0,-140.0},{108.0,-140.0}}, color = {191,0,191}));
        connect(Source5.MassInPort1,CIAFG.y) annotation(Line(visible = true, points = {{61.0,-140.0},{70.0,-140.0}}, color = {191,0,191}));
        connect(P_Absorption.y1,Sink4.MassInPort1) annotation(Line(visible = true, points = {{-10.0,-140.0},{-1.0,-140.0}}, color = {191,0,191}));
        connect(Pollution.u2,P_Absorption.y) annotation(Line(visible = true, points = {{-48.0,-140.0},{-30.0,-140.0}}, color = {191,0,191}));
        connect(P_Generation.y1,Pollution.u1) annotation(Line(visible = true, points = {{-110.0,-140.0},{-92.0,-140.0}}, color = {191,0,191}));
        connect(Source4.MassInPort1,P_Generation.y) annotation(Line(visible = true, points = {{-139.0,-140.0},{-130.0,-140.0}}, color = {191,0,191}));
        connect(FPCI.y,Prod_3_2.u3) annotation(Line(visible = true, points = {{-136.5,-107.0},{-117.0,-107.0},{-117.0,-76.5}}, color = {0,0,191}));
        connect(FPM.y,Prod_3_2.u2) annotation(Line(visible = true, points = {{-49.5,-66.0},{-107.2,-66.0}}, color = {0,0,191}));
        connect(Prod_3_2.u1,FCM.y) annotation(Line(visible = true, points = {{-117.0,-55.5},{-117.0,-47.0},{-101.5,-47.0}}, color = {0,0,191}));
        connect(Prod_3_2.y,Food_Ratio.u) annotation(Line(visible = true, points = {{-126.8,-66.0},{-138.5,-66.0}}, color = {0,0,191}));
        connect(CIR.u1,Capital_Investment.y1) annotation(Line(visible = true, points = {{125.0,15.6},{154.0,15.6},{154.0,-29.0},{147.0,-29.0}}, color = {0,0,191}));
        connect(NRFR.y,NREM.u) annotation(Line(visible = true, points = {{147.5,51.0},{163.0,51.0}}, color = {0,0,191}));
        connect(DRMM.y,Prod_5_2.u5) annotation(Line(visible = true, points = {{34.5,171.0},{26.0,171.0},{26.0,104.0},{40.0,104.0},{40.0,79.0},{31.2,79.0}}, color = {0,0,191}));
        connect(DR_norm.y,Death_Rate.u) annotation(Line(visible = true, points = {{-13.5,103.0},{-20.0,103.0},{-20.0,119.0}}, color = {0,0,191}));
        connect(Prod_5_2.y,DR_norm.u) annotation(Line(visible = true, points = {{20.0,89.5},{20.0,103.0},{7.5,103.0}}, color = {0,0,191}));
        connect(Prod_5_2.u1,Population.y2) annotation(Line(visible = true, points = {{8.8,79.0},{-55.0,79.0},{-55.0,127.0}}, color = {0,0,191}));
        connect(BRMM.y,Prod_5_1.u1) annotation(Line(visible = true, points = {{-121.5,185.0},{-180.0,185.0},{-180.0,80.0},{-171.2,80.0}}, color = {0,0,191}));
        connect(Prod_5_1.u5,Population.y4) annotation(Line(visible = true, points = {{-148.8,80.0},{-85.0,80.0},{-85.0,127.0}}, color = {0,0,191}));
        connect(Prod_5_1.y,BR_norm.u) annotation(Line(visible = true, points = {{-160.0,91.2},{-160.0,103.0},{-145.5,103.0}}, color = {0,0,191}));
        connect(BR_norm.y,Birth_Rate.u) annotation(Line(visible = true, points = {{-124.5,103.0},{-120.0,103.0},{-120.0,119.0}}, color = {0,0,191}));
        connect(Source1.MassInPort1,Birth_Rate.y) annotation(Line(visible = true, points = {{-139.0,140.0},{-130.0,140.0}}, color = {191,0,191}));
        connect(Birth_Rate.y1,Population.u1) annotation(Line(visible = true, points = {{-110.0,140.0},{-92.0,140.0}}, color = {191,0,191}));
        connect(Population.u2,Death_Rate.y) annotation(Line(visible = true, points = {{-48.0,140.0},{-30.0,140.0}}, color = {191,0,191}));
        connect(Death_Rate.y1,Sink1.MassInPort1) annotation(Line(visible = true, points = {{-10.0,140.0},{-1.0,140.0}}, color = {191,0,191}));
        connect(Source2.MassInPort1,Generation.y) annotation(Line(visible = true, points = {{61.0,140.0},{70.0,140.0}}, color = {191,0,191}));
        connect(Gen_Const.y,Generation.u) annotation(Line(visible = true, points = {{80.0,111.5},{80.0,119.0}}, color = {0,0,191}));
        connect(NRFR.u,Natural_Resources.y4) annotation(Line(visible = true, points = {{126.5,51.0},{115.0,51.0},{115.0,127.0}}, color = {0,0,191}));
        connect(Generation.y1,Natural_Resources.u1) annotation(Line(visible = true, points = {{90.0,140.0},{108.0,140.0}}, color = {191,0,191}));
        connect(Natural_Resources.u2,Depletion.y) annotation(Line(visible = true, points = {{152.0,140.0},{170.0,140.0}}, color = {191,0,191}));
        connect(Depletion.y1,Sink2.MassInPort1) annotation(Line(visible = true, points = {{190.0,140.0},{199.0,140.0}}, color = {191,0,191}));
        Pop = Population.y;
        Pol = Pollution.y;
        Pol_rat = Pol_Ratio.y;
        Cap_inv = Capital_Investment.y;
        Qual_life = Quality_of_Life.y;
        Nat_res = Natural_Resources.y;
        NRUN = if time > 1970 then 0.25 else 1.0;
        NR_norm.u2 = NRUN;
        annotation(Diagram(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-62.0,6.0},{-58.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{98.0,192.0},{102.0,196.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{138.0,-16.0},{142.0,-12.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-22.0},{2.0,-18.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-178.0,-2.0},{-174.0,2.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{78.0,4.0},{82.0,8.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-97.0},{2.0,-93.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-178.0,-68.0},{-174.0,-64.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-68.0},{2.0,-64.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{158.0,28.0},{162.0,32.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{38.0,-262.0},{42.0,-258.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,-102.0},{302.0,-98.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,28.0},{302.0,32.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,78.0},{302.0,82.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,169.0},{302.0,173.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{342.0,-171.0},{346.0,-167.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{354.0,-234.0},{358.0,-230.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-97.0,6.0},{-93.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-137.0,-22.0},{-133.0,-18.0}})}), Icon(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-220.0,-260.0},{442.0,224.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-152.0,74.0},{368.0,144.0}}, textString = "1st Modification", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {127,0,255}, extent = {{-148.0,-22.0},{372.0,48.0}}, textString = "NRUN", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, extent = {{-150.0,-102.0},{370.0,-32.0}}, textString = "reduced to 0.25", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, extent = {{-150.0,-188.0},{370.0,-118.0}}, textString = "in 1970", fontName = "Arial")}), experiment(StartTime = 1900, StopTime = 2100), experimentSetupOutput, Documentation(info = "<html>
This is the first modification proposed by <a href=\"http://en.wikipedia.org/wiki/Jay_Wright_Forrester\">Jay Forrester</a> in his <font color=red><b>WORLD2</b></font> model. The utilization of natural resources is reduced in 1970 to 25% of their former value in order to extend their availability. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Cellier, F.E. (1991), <a href=\"http://www.amazon.com/Continuous-System-Modeling-Fran%C3%A7ois-Cellier/dp/0387975020\">Continuous System Modeling</a>, Springer-Verlag, New York, ISBN: 0-387-97502-0, 755p.
<li> Forrester, J.W. (1971), <a href=\"http://www.amazon.com/World-Dynamics-Jay-W-Forrester/dp/1883823382/ref=ed_oe_h/103-2487145-1208659\">World Dynamics</a>, Pegasus Communications, 160p.
</ol> <p>
 
<hr> <p>
 
Simulate the model across 200 years, and plot the state variables as well as the quality of life as functions of the calendar years: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/SD_Fig2.png\"> <p>
 
I had to artificially extend some of the tabular functions, because our software complains when an interpolation function attempts an extrapolation. <p>
 
<hr> <p>
</html>", revisions = ""));
      end Scenario_2;
      model Scenario_3 "3rd Scenario"
        parameter Real Population_0 = 1650000000.0 "World population in 1900";
        parameter Real Pollution_0 = 200000000.0 "Pollution in 1900";
        parameter Real Nat_Resources_0(unit = "ton") = 900000000000.0 "Unrecoverable natural resources in 1900";
        parameter Real Cap_Invest_0(unit = "dollar") = 400000000.0 "Capital investment in 1900";
        parameter Real CIAF_0 = 0.2 "Proportion of capital investment in agriculture in 1900";
        parameter Real BRN(unit = "1/yr") = 0.04 "Normal birth rate";
        parameter Real CIAFN(unit = "dollar") = 0.3 "CIAF normalization";
        parameter Real CIAFT(unit = "yr") = 15.0 "CIAF time constant";
        parameter Real CIDN(unit = "dollar/yr") = 0.025 "Normal capital discard";
        parameter Real CIGN(unit = "dollar/yr") = 0.05 "Normal capital generation";
        parameter Real DRN(unit = "1/yr") = 0.028 "Normal death rate";
        parameter Real ECIRN(unit = "dollar") = 1.0 "Capital normalization";
        parameter Real FC(unit = "kg/yr") = 1.0 "Food coefficient";
        parameter Real FN(unit = "kg/yr") = 1.0 "Food normalization";
        parameter Real Land_Area(unit = "hectare") = 135000000.0 "Area of arable land";
        parameter Real NRI(unit = "ton") = 900000000000.0 "Initial natural resources";
        parameter Real POLS = 3599900000.0 "Standard pollution";
        parameter Real Pop_dens_norm(unit = "1/hectare") = 26.5 "Normal population density";
        parameter Real QLS = 1.0 "Standard quality of life";
        output Real Pop "World population";
        output Real Pol "Pollution";
        output Real Pol_rat "Pollution ratio";
        output Real Cap_inv(unit = "dollar") "Capital investment";
        output Real Qual_life "Quality of life";
        output Real Nat_res(unit = "ton") "Natural unrecoverable resources";
        Real NRUN(unit = "1/yr") "Normal resource utilization";
        Real POLN(unit = "1/yr") "Normal pollution";
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {-150.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Birth_Rate annotation(Placement(visible = true, transformation(origin = {-120.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Population(x0 = Population_0) annotation(Placement(visible = true, transformation(origin = {-70.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Death_Rate annotation(Placement(visible = true, transformation(origin = {-20.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink1 annotation(Placement(visible = true, transformation(origin = {10.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source2 annotation(Placement(visible = true, transformation(origin = {50.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Generation annotation(Placement(visible = true, transformation(origin = {80.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Natural_Resources(x0 = Nat_Resources_0) annotation(Placement(visible = true, transformation(origin = {130.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Depletion annotation(Placement(visible = true, transformation(origin = {180.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink2 annotation(Placement(visible = true, transformation(origin = {210.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Const Gen_Const(k = 0) annotation(Placement(visible = true, transformation(origin = {80.0,101.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular BRMM(x_vals = {0,1,2,3,4,5,20}, y_vals = {1.2,1.0,0.85,0.75,0.7,0.7,0.7}) annotation(Placement(visible = true, transformation(origin = {-105.0,185.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular DRMM(x_vals = {0,0.5,1,1.5,2,2.5,3,3.5,4,4.5,5,20}, y_vals = {3.0,1.8,1.0,0.8,0.7,0.6,0.53,0.5,0.5,0.5,0.5,0.5}) annotation(Placement(visible = true, transformation(origin = {51.0,171.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain BR_norm(k = BRN) annotation(Placement(visible = true, transformation(origin = {-135.0,103.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_5 Prod_5_1 annotation(Placement(visible = true, transformation(origin = {-160.0,80.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain DR_norm(k = DRN) annotation(Placement(visible = true, transformation(origin = {-3.0,103.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_5 Prod_5_2 annotation(Placement(visible = true, transformation(origin = {20.0,79.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular BRFM(x_vals = {0,1,2,3,4,20}, y_vals = {0.0,1.0,1.6,1.9,2.0,2.0}) annotation(Placement(visible = true, transformation(origin = {-176.0,35.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular BRPM(x_vals = 0:10:60, y_vals = {1.02,0.9,0.7,0.4,0.25,0.15,0.1}) annotation(Placement(visible = true, transformation(origin = {-135.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular BRCM(x_vals = 0:5, y_vals = {1.05,1.0,0.9,0.7,0.6,0.55}) annotation(Placement(visible = true, transformation(origin = {-95.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRCM(x_vals = 0:5, y_vals = {0.9,1.0,1.2,1.5,1.9,3.0}) annotation(Placement(visible = true, transformation(origin = {-25.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRPM(x_vals = 0:10:60, y_vals = {0.92,1.3,2.0,3.2,4.8,6.8,9.2}) annotation(Placement(visible = true, transformation(origin = {15.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRFM(x_vals = {0,0.25,0.5,0.75,1,1.25,1.5,1.75,2,20}, y_vals = {30.0,3.0,2.0,1.4,1.0,0.7,0.6,0.5,0.5,0.5}) annotation(Placement(visible = true, transformation(origin = {55.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -90)));
        SystemDynamics.Auxiliary.Gain Crowd_Rat(k = 1.0 / (Land_Area * Pop_dens_norm)) annotation(Placement(visible = true, transformation(origin = {-60.0,35.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = -90)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_1 annotation(Placement(visible = true, transformation(origin = {140.0,80.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular NRMM(x_vals = 0:10, y_vals = {0.0,1.0,1.8,2.4,2.9,3.3,3.6,3.8,3.9,3.95,4.0}) annotation(Placement(visible = true, transformation(origin = {201.0,80.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Sources.Source Source3 annotation(Placement(visible = true, transformation(origin = {50.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CI_Generation annotation(Placement(visible = true, transformation(origin = {80.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Capital_Investment(x0 = Cap_Invest_0) annotation(Placement(visible = true, transformation(origin = {130.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CI_Discard annotation(Placement(visible = true, transformation(origin = {180.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink3 annotation(Placement(visible = true, transformation(origin = {210.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIG_norm(k = CIGN) annotation(Placement(visible = true, transformation(origin = {65.0,-77.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_2 annotation(Placement(visible = true, transformation(origin = {40.0,-100.0}, extent = {{-14.0,-16.0},{14.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CID_norm(k = CIDN) annotation(Placement(visible = true, transformation(origin = {165.0,-77.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular CIM(x_vals = {0,1,2,3,4,5,20}, y_vals = {0.1,1.0,1.8,2.4,2.8,3.0,3.0}) annotation(Placement(visible = true, transformation(origin = {113.0,-100.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain MSL(k = 1.0 / ECIRN) annotation(Placement(visible = true, transformation(origin = {277.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain NRFR(k = 1.0 / NRI) annotation(Placement(visible = true, transformation(origin = {137.0,51.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain ECIR(k = 1.0 / (1.0 - CIAFN)) annotation(Placement(visible = true, transformation(origin = {243.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_3 Prod_3_1 annotation(Placement(visible = true, transformation(origin = {215.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular NREM(x_vals = 0:0.25:1, y_vals = {0.0,0.15,0.5,0.85,1.0}) annotation(Placement(visible = true, transformation(origin = {175.0,51.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -180)));
        Modelica.Blocks.Math.Division CIR annotation(Placement(visible = true, transformation(origin = {107.0,6.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source4 annotation(Placement(visible = true, transformation(origin = {-150.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 P_Generation annotation(Placement(visible = true, transformation(origin = {-120.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Pollution(x0 = Pollution_0) annotation(Placement(visible = true, transformation(origin = {-70.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 P_Absorption annotation(Placement(visible = true, transformation(origin = {-20.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink4 annotation(Placement(visible = true, transformation(origin = {10.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Linear NotCIAF(m = -1.0, b = 1.0) annotation(Placement(visible = true, transformation(origin = {189.0,10.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain Pol_Ratio(k = 1.0 / POLS) annotation(Placement(visible = true, transformation(origin = {-29.0,-95.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular FCM(x_vals = 0:5, y_vals = {2.4,1.0,0.6,0.4,0.3,0.2}) annotation(Placement(visible = true, transformation(origin = {-85.0,-47.0}, extent = {{-15.0,15.0},{15.0,-15.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_3 Prod_3_2 annotation(Placement(visible = true, transformation(origin = {-117.0,-66.0}, extent = {{15.0,-14.0},{-15.0,14.0}}, rotation = 90)));
        SystemDynamics.Auxiliary.Gain Food_Ratio(k = FC / FN) annotation(Placement(visible = true, transformation(origin = {-149.0,-66.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source5 annotation(Placement(visible = true, transformation(origin = {50.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CIAFG annotation(Placement(visible = true, transformation(origin = {80.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level CIAF(x0 = CIAF_0) annotation(Placement(visible = true, transformation(origin = {130.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CIAFD annotation(Placement(visible = true, transformation(origin = {180.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink5 annotation(Placement(visible = true, transformation(origin = {210.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIAF_D(k = 1.0 / CIAFT) annotation(Placement(visible = true, transformation(origin = {163.0,-180.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIAF_G(k = 1.0 / CIAFT) annotation(Placement(visible = true, transformation(origin = {95.0,-180.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_3 annotation(Placement(visible = true, transformation(origin = {120.0,-230.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division P_Abs annotation(Placement(visible = true, transformation(origin = {-43.0,-180.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_4 annotation(Placement(visible = true, transformation(origin = {-160.0,-220.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular POLCM(x_vals = {0,1,2,3,4,5,100}, y_vals = {0.05,1.0,3.0,5.4,7.4,8.0,8.0}) annotation(Placement(visible = true, transformation(origin = {-117.0,-220.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular POLAT(x_vals = 0:10:60, y_vals = {0.6,2.5,5.0,8.0,11.5,15.5,20.0}) annotation(Placement(visible = true, transformation(origin = {-39.0,-220.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular CFIFR(x_vals = {0,0.5,1,1.5,2,20}, y_vals = {1.0,0.6,0.3,0.15,0.1,0.1}) annotation(Placement(visible = true, transformation(origin = {69.0,-230.0}, extent = {{15.0,16.0},{-15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular FPM(x_vals = 0:10:60, y_vals = {1.02,0.9,0.65,0.35,0.2,0.1,0.05}) annotation(Placement(visible = true, transformation(origin = {-33.0,-66.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_5 annotation(Placement(visible = true, transformation(origin = {229.0,-200.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIRA(k = 1.0 / CIAFN) annotation(Placement(visible = true, transformation(origin = {255.0,-225.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular FPCI(x_vals = {0,1,2,3,4,5,6,100}, y_vals = {0.5,1.0,1.4,1.7,1.9,2.05,2.2,2.2}) annotation(Placement(visible = true, transformation(origin = {-153.0,-107.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular CIQR(y_vals = {0.7,0.8,1.0,1.5,2.0,2.0}, x_vals = {0,0.5,1,1.5,2,10}) annotation(Placement(visible = true, transformation(origin = {167.0,-230.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        Modelica.Blocks.Math.Division QLMF annotation(Placement(visible = true, transformation(origin = {315.0,-223.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular QLM(x_vals = {0,1,2,3,4,5,20}, y_vals = {0.2,1.0,1.7,2.3,2.7,2.9,2.9}) annotation(Placement(visible = true, transformation(origin = {319.0,-169.0}, extent = {{15.0,17.0},{-15.0,-17.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular QLF(x_vals = {0,1,2,3,4,20}, y_vals = {0.0,1.0,1.8,2.4,2.7,2.7}) annotation(Placement(visible = true, transformation(origin = {381.0,-232.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_4 Prod_4_1 annotation(Placement(visible = true, transformation(origin = {375.0,-163.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular QLC(x_vals = 0:0.5:5, y_vals = {2.0,1.3,1.0,0.75,0.55,0.45,0.38,0.3,0.25,0.22,0.2}) annotation(Placement(visible = true, transformation(origin = {335.0,-113.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular QLP(x_vals = 0:10:60, y_vals = {1.04,0.85,0.6,0.3,0.15,0.05,0.02}) annotation(Placement(visible = true, transformation(origin = {375.0,-113.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -90)));
        SystemDynamics.Auxiliary.Gain Quality_of_Life(k = QLS) annotation(Placement(visible = true, transformation(origin = {412.0,-163.0}, extent = {{-20.0,-21.0},{20.0,21.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World2.Utilities.Parameter_Change NR_norm annotation(Placement(visible = true, transformation(origin = {162.5298,105.0774}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World2.Utilities.Parameter_Change Pol_norm annotation(Placement(visible = true, transformation(origin = {-140.0,-180.815}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
      equation
        connect(NR_norm.u1,Prod_2_1.y) annotation(Line(visible = true, points = {{153.5298,109.0774},{140.0,109.0774},{140.0,91.2}}, color = {0,0,191}));
        connect(P_Generation.u,Pol_norm.y) annotation(Line(visible = true, points = {{-120.0,-161.0},{-120.0,-171.0},{-120.0,-181.0},{-126.5,-180.815}}, color = {0,0,191}));
        connect(Pol_norm.u1,Prod_2_4.y) annotation(Line(visible = true, points = {{-153.5,-174.815},{-160.0,-175.0},{-160.0,-208.8}}, color = {0,0,191}));
        connect(Depletion.u,NR_norm.y) annotation(Line(visible = true, points = {{180.0,119.0},{180.0,105.0},{171.5298,105.0774}}, color = {0,0,191}));
        connect(Food_Ratio.y,BRFM.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-176.0,-66.0},{-176.0,22.2}}, color = {0,0,191}));
        connect(BRFM.y,Prod_5_1.u2) annotation(Line(visible = true, points = {{-176.0,52.6},{-176.0,60.0},{-167.36,60.0},{-167.36,72.0}}, color = {0,0,191}));
        connect(BRPM.u,QLP.u) annotation(Line(visible = true, points = {{-135.0,23.0},{-135.0,-20.0},{-214.0,-20.0},{-214.0,214.0},{375.0,214.0},{375.0,-125.0}}, color = {0,0,191}));
        connect(Pol_Ratio.y,BRPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,-20.0},{-135.0,-20.0},{-135.0,23.0}}, color = {0,0,191}));
        connect(BRPM.y,Prod_5_1.u3) annotation(Line(visible = true, points = {{-135.0,51.5},{-135.0,60.0},{-160.0,60.0},{-160.0,68.8}}, color = {0,0,191}));
        connect(Crowd_Rat.y,BRCM.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-95.0,8.0},{-95.0,23.0}}, color = {0,0,191}));
        connect(BRCM.y,Prod_5_1.u4) annotation(Line(visible = true, points = {{-95.0,51.5},{-95.0,66.0},{-152.64,66.0},{-152.64,72.0}}, color = {0,0,191}));
        connect(Crowd_Rat.y,DRCM.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-25.0,8.0},{-25.0,23.0}}, color = {0,0,191}));
        connect(DRCM.y,Prod_5_2.u2) annotation(Line(visible = true, points = {{-25.0,51.5},{-25.0,60.0},{12.64,60.0},{12.64,71.5}}, color = {0,0,191}));
        connect(Pol_Ratio.y,DRPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,8.0},{15.0,8.0},{15.0,23.0}}, color = {0,0,191}));
        connect(DRPM.y,Prod_5_2.u3) annotation(Line(visible = true, points = {{15.0,51.5},{15.0,60.0},{20.0,60.0},{20.0,68.5}}, color = {0,0,191}));
        connect(Crowd_Rat.y,QLC.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-206.0,8.0},{-206.0,208.0},{335.0,208.0},{335.0,-125.0}}, color = {0,0,191}));
        connect(FCM.u,Crowd_Rat.y) annotation(Line(visible = true, points = {{-73.0,-47.0},{-60.0,-47.0},{-60.0,23.8}}, color = {0,0,191}));
        connect(Crowd_Rat.u,Population.y3) annotation(Line(visible = true, points = {{-60.0,46.2},{-60.0,66.0},{-70.0,66.0},{-70.0,127.0}}, color = {0,0,191}));
        connect(Prod_2_5.u2,Prod_3_1.u2) annotation(Line(visible = true, points = {{239.5,-200.0},{260.0,-200.0},{260.0,-20.0},{160.0,-20.0},{160.0,30.0},{203.8,30.0}}, color = {0,0,191}));
        connect(Prod_3_1.y,ECIR.u) annotation(Line(visible = true, points = {{226.2,30.0},{232.5,30.0}}, color = {0,0,191}));
        connect(CIR.y,Prod_3_1.u2) annotation(Line(visible = true, points = {{90.5,6.0},{80.0,6.0},{80.0,30.0},{203.8,30.0}}, color = {0,0,191}));
        connect(NotCIAF.y,Prod_3_1.u3) annotation(Line(visible = true, points = {{202.5,10.0},{215.0,10.0},{215.0,19.5}}, color = {0,0,191}));
        connect(Prod_3_1.u1,NREM.y) annotation(Line(visible = true, points = {{215.0,40.5},{215.0,51.0},{191.5,51.0}}, color = {0,0,191}));
        connect(QLP.y,Prod_4_1.u1) annotation(Line(visible = true, points = {{375.0,-96.5},{375.0,-152.5}}, color = {0,0,191}));
        connect(QLC.y,Prod_4_1.u2) annotation(Line(visible = true, points = {{335.0,-96.5},{335.0,-157.0},{366.6,-157.0}}, color = {0,0,191}));
        connect(Prod_4_1.y,Quality_of_Life.u) annotation(Line(visible = true, points = {{385.5,-163.0},{398.0,-163.0}}, color = {0,0,191}));
        connect(QLF.y,Prod_4_1.u4) annotation(Line(visible = true, points = {{364.5,-232.0},{356.0,-232.0},{356.0,-190.0},{375.0,-190.0},{375.0,-173.5}}, color = {0,0,191}));
        connect(QLM.y,Prod_4_1.u3) annotation(Line(visible = true, points = {{335.5,-169.0},{366.6,-169.0}}, color = {0,0,191}));
        connect(QLF.y,QLMF.u2) annotation(Line(visible = true, points = {{364.5,-232.0},{333.0,-232.0}}, color = {0,0,191}));
        connect(QLMF.y,CIQR.u) annotation(Line(visible = true, points = {{298.5,-223.0},{290.0,-223.0},{290.0,-256.0},{190.0,-256.0},{190.0,-230.0},{179.0,-230.0}}, color = {0,0,191}));
        connect(P_Abs.u1,Pollution.y3) annotation(Line(visible = true, points = {{-61.0,-170.4},{-70.0,-170.4},{-70.0,-153.0}}, color = {0,0,191}));
        connect(CIG_norm.y,CI_Generation.u) annotation(Line(visible = true, points = {{75.5,-77.0},{80.0,-77.0},{80.0,-61.0}}, color = {0,0,191}));
        connect(CI_Discard.y1,Sink3.MassInPort1) annotation(Line(visible = true, points = {{190.0,-40.0},{199.0,-40.0}}, color = {191,0,191}));
        connect(Capital_Investment.u2,CI_Discard.y) annotation(Line(visible = true, points = {{152.0,-40.0},{170.0,-40.0}}, color = {191,0,191}));
        connect(Source3.MassInPort1,CI_Generation.y) annotation(Line(visible = true, points = {{61.0,-40.0},{70.0,-40.0}}, color = {191,0,191}));
        connect(CI_Generation.y1,Capital_Investment.u1) annotation(Line(visible = true, points = {{90.0,-40.0},{108.0,-40.0}}, color = {191,0,191}));
        connect(FPM.u,POLAT.u) annotation(Line(visible = true, points = {{-21.0,-66.0},{0.0,-66.0},{0.0,-220.0},{-27.0,-220.0}}, color = {0,0,191}));
        connect(CIRA.y,FPCI.u) annotation(Line(visible = true, points = {{265.5,-225.0},{280.0,-225.0},{280.0,-252.0},{-210.0,-252.0},{-210.0,-107.0},{-165.0,-107.0}}, color = {0,0,191}));
        connect(Prod_2_1.u1,Population.y1) annotation(Line(visible = true, points = {{128.8,80.0},{100.0,80.0},{100.0,194.0},{-46.0,194.0},{-46.0,151.0},{-53.0,151.0}}, color = {0,0,191}));
        connect(NRMM.u,DRMM.u) annotation(Line(visible = true, points = {{213.0,80.0},{300.0,80.0},{300.0,171.0},{63.0,171.0}}, color = {0,0,191}));
        connect(NRMM.u,BRMM.u) annotation(Line(visible = true, points = {{213.0,80.0},{300.0,80.0},{300.0,200.0},{-70.0,200.0},{-70.0,185.0},{-93.0,185.0}}, color = {0,0,191}));
        connect(CFIFR.u,QLF.u) annotation(Line(visible = true, points = {{57.0,-230.0},{40.0,-230.0},{40.0,-260.0},{400.0,-260.0},{400.0,-232.0},{393.0,-232.0}}, color = {0,0,191}));
        connect(Food_Ratio.y,QLF.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-220.0,-66.0},{-220.0,-260.0},{400.0,-260.0},{400.0,-232.0},{393.0,-232.0}}, color = {0,0,191}));
        connect(Food_Ratio.y,DRFM.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-176.0,-66.0},{-176.0,0.0},{55.0,0.0},{55.0,47.0}}, color = {0,0,191}));
        connect(Pol_Ratio.y,FPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,-66.0},{-21.0,-66.0}}, color = {0,0,191}));
        connect(CIR.y,POLCM.u) annotation(Line(visible = true, points = {{90.5,6.0},{80.0,6.0},{80.0,-6.0},{10.0,-6.0},{10.0,-116.0},{-100.0,-116.0},{-100.0,-220.0},{-105.0,-220.0}}, color = {0,0,191}));
        connect(QLM.u,CIM.u) annotation(Line(visible = true, points = {{307.0,-169.0},{300.0,-169.0},{300.0,-100.0},{125.0,-100.0}}, color = {0,0,191}));
        connect(QLM.y,QLMF.u1) annotation(Line(visible = true, points = {{335.5,-169.0},{344.0,-169.0},{344.0,-214.0},{333.0,-214.0}}, color = {0,0,191}));
        connect(MSL.y,QLM.u) annotation(Line(visible = true, points = {{287.5,30.0},{300.0,30.0},{300.0,-169.0},{307.0,-169.0}}, color = {0,0,191}));
        connect(NRMM.y,Prod_2_1.u2) annotation(Line(visible = true, points = {{184.5,80.0},{151.2,80.0}}, color = {0,0,191}));
        connect(MSL.y,NRMM.u) annotation(Line(visible = true, points = {{287.5,30.0},{300.0,30.0},{300.0,80.0},{213.0,80.0}}, color = {0,0,191}));
        connect(ECIR.y,MSL.u) annotation(Line(visible = true, points = {{253.5,30.0},{266.5,30.0}}, color = {0,0,191}));
        connect(NotCIAF.u,CIAF.y1) annotation(Line(visible = true, points = {{175.5,10.0},{168.0,10.0},{168.0,-8.0},{240.0,-8.0},{240.0,-110.0},{154.0,-110.0},{154.0,-129.0},{147.0,-129.0}}, color = {0,0,191}));
        connect(Prod_2_2.u1,CIR.u2) annotation(Line(visible = true, points = {{30.2,-100.0},{20.0,-100.0},{20.0,-14.0},{140.0,-14.0},{140.0,-3.6},{125.0,-3.6}}, color = {0,0,191}));
        connect(Prod_2_2.u1,Population.y1) annotation(Line(visible = true, points = {{30.2,-100.0},{20.0,-100.0},{20.0,-14.0},{320.0,-14.0},{320.0,194.0},{-46.0,194.0},{-46.0,151.0},{-53.0,151.0}}, color = {0,0,191}));
        connect(Prod_2_4.u1,Population.y) annotation(Line(visible = true, points = {{-171.2,-220.0},{-200.0,-220.0},{-200.0,164.0},{-94.0,164.0},{-94.0,151.0},{-87.0,151.0}}, color = {0,0,191}));
        connect(CID_norm.u,Capital_Investment.y2) annotation(Line(visible = true, points = {{154.5,-77.0},{145.0,-77.0},{145.0,-53.0}}, color = {0,0,191}));
        connect(CID_norm.y,CI_Discard.u) annotation(Line(visible = true, points = {{175.5,-77.0},{180.0,-77.0},{180.0,-61.0}}, color = {0,0,191}));
        connect(CIM.y,Prod_2_2.u2) annotation(Line(visible = true, points = {{96.5,-100.0},{49.8,-100.0}}, color = {0,0,191}));
        connect(Prod_2_2.y,CIG_norm.u) annotation(Line(visible = true, points = {{40.0,-88.8},{40.0,-77.0},{54.5,-77.0}}, color = {0,0,191}));
        connect(Pol_Ratio.u,Pollution.y1) annotation(Line(visible = true, points = {{-39.5,-95.0},{-46.0,-95.0},{-46.0,-129.0},{-53.0,-129.0}}, color = {0,0,191}));
        connect(POLCM.y,Prod_2_4.u2) annotation(Line(visible = true, points = {{-133.5,-220.0},{-148.8,-220.0}}, color = {0,0,191}));
        connect(POLAT.y,P_Abs.u2) annotation(Line(visible = true, points = {{-55.5,-220.0},{-70.0,-220.0},{-70.0,-189.6},{-61.0,-189.6}}, color = {0,0,191}));
        connect(P_Abs.y,P_Absorption.u) annotation(Line(visible = true, points = {{-26.5,-180.0},{-20.0,-180.0},{-20.0,-161.0}}, color = {0,0,191}));
        connect(CFIFR.y,Prod_2_3.u1) annotation(Line(visible = true, points = {{85.5,-230.0},{108.8,-230.0}}, color = {0,0,191}));
        connect(CIQR.y,Prod_2_3.u2) annotation(Line(visible = true, points = {{150.5,-230.0},{131.2,-230.0}}, color = {0,0,191}));
        connect(CIAF_G.y,CIAFG.u) annotation(Line(visible = true, points = {{84.5,-180.0},{80.0,-180.0},{80.0,-161.0}}, color = {0,0,191}));
        connect(CIAF_G.u,Prod_2_3.y) annotation(Line(visible = true, points = {{105.5,-180.0},{120.0,-180.0},{120.0,-218.8}}, color = {0,0,191}));
        connect(CIRA.u,Prod_2_5.y) annotation(Line(visible = true, points = {{244.5,-225.0},{229.0,-225.0},{229.0,-211.2}}, color = {0,0,191}));
        connect(Prod_2_5.u1,CIAF.y3) annotation(Line(visible = true, points = {{218.5,-200.0},{130.0,-200.0},{130.0,-153.0}}, color = {0,0,191}));
        connect(CIAF_D.u,CIAF.y2) annotation(Line(visible = true, points = {{152.5,-180.0},{145.0,-180.0},{145.0,-153.0}}, color = {0,0,191}));
        connect(CIAF_D.y,CIAFD.u) annotation(Line(visible = true, points = {{173.5,-180.0},{180.0,-180.0},{180.0,-161.0}}, color = {0,0,191}));
        connect(CIAFD.y1,Sink5.MassInPort1) annotation(Line(visible = true, points = {{190.0,-140.0},{199.0,-140.0}}, color = {191,0,191}));
        connect(CIAF.u2,CIAFD.y) annotation(Line(visible = true, points = {{152.0,-140.0},{170.0,-140.0}}, color = {191,0,191}));
        connect(CIAFG.y1,CIAF.u1) annotation(Line(visible = true, points = {{90.0,-140.0},{108.0,-140.0}}, color = {191,0,191}));
        connect(Source5.MassInPort1,CIAFG.y) annotation(Line(visible = true, points = {{61.0,-140.0},{70.0,-140.0}}, color = {191,0,191}));
        connect(P_Absorption.y1,Sink4.MassInPort1) annotation(Line(visible = true, points = {{-10.0,-140.0},{-1.0,-140.0}}, color = {191,0,191}));
        connect(Pollution.u2,P_Absorption.y) annotation(Line(visible = true, points = {{-48.0,-140.0},{-30.0,-140.0}}, color = {191,0,191}));
        connect(P_Generation.y1,Pollution.u1) annotation(Line(visible = true, points = {{-110.0,-140.0},{-92.0,-140.0}}, color = {191,0,191}));
        connect(Source4.MassInPort1,P_Generation.y) annotation(Line(visible = true, points = {{-139.0,-140.0},{-130.0,-140.0}}, color = {191,0,191}));
        connect(FPCI.y,Prod_3_2.u3) annotation(Line(visible = true, points = {{-136.5,-107.0},{-117.0,-107.0},{-117.0,-76.5}}, color = {0,0,191}));
        connect(FPM.y,Prod_3_2.u2) annotation(Line(visible = true, points = {{-49.5,-66.0},{-107.2,-66.0}}, color = {0,0,191}));
        connect(Prod_3_2.u1,FCM.y) annotation(Line(visible = true, points = {{-117.0,-55.5},{-117.0,-47.0},{-101.5,-47.0}}, color = {0,0,191}));
        connect(Prod_3_2.y,Food_Ratio.u) annotation(Line(visible = true, points = {{-126.8,-66.0},{-138.5,-66.0}}, color = {0,0,191}));
        connect(CIR.u1,Capital_Investment.y1) annotation(Line(visible = true, points = {{125.0,15.6},{154.0,15.6},{154.0,-29.0},{147.0,-29.0}}, color = {0,0,191}));
        connect(NRFR.y,NREM.u) annotation(Line(visible = true, points = {{147.5,51.0},{163.0,51.0}}, color = {0,0,191}));
        connect(DRMM.y,Prod_5_2.u5) annotation(Line(visible = true, points = {{34.5,171.0},{26.0,171.0},{26.0,104.0},{40.0,104.0},{40.0,79.0},{31.2,79.0}}, color = {0,0,191}));
        connect(DR_norm.y,Death_Rate.u) annotation(Line(visible = true, points = {{-13.5,103.0},{-20.0,103.0},{-20.0,119.0}}, color = {0,0,191}));
        connect(Prod_5_2.y,DR_norm.u) annotation(Line(visible = true, points = {{20.0,89.5},{20.0,103.0},{7.5,103.0}}, color = {0,0,191}));
        connect(Prod_5_2.u1,Population.y2) annotation(Line(visible = true, points = {{8.8,79.0},{-55.0,79.0},{-55.0,127.0}}, color = {0,0,191}));
        connect(BRMM.y,Prod_5_1.u1) annotation(Line(visible = true, points = {{-121.5,185.0},{-180.0,185.0},{-180.0,80.0},{-171.2,80.0}}, color = {0,0,191}));
        connect(Prod_5_1.u5,Population.y4) annotation(Line(visible = true, points = {{-148.8,80.0},{-85.0,80.0},{-85.0,127.0}}, color = {0,0,191}));
        connect(Prod_5_1.y,BR_norm.u) annotation(Line(visible = true, points = {{-160.0,91.2},{-160.0,103.0},{-145.5,103.0}}, color = {0,0,191}));
        connect(BR_norm.y,Birth_Rate.u) annotation(Line(visible = true, points = {{-124.5,103.0},{-120.0,103.0},{-120.0,119.0}}, color = {0,0,191}));
        connect(Source1.MassInPort1,Birth_Rate.y) annotation(Line(visible = true, points = {{-139.0,140.0},{-130.0,140.0}}, color = {191,0,191}));
        connect(Birth_Rate.y1,Population.u1) annotation(Line(visible = true, points = {{-110.0,140.0},{-92.0,140.0}}, color = {191,0,191}));
        connect(Population.u2,Death_Rate.y) annotation(Line(visible = true, points = {{-48.0,140.0},{-30.0,140.0}}, color = {191,0,191}));
        connect(Death_Rate.y1,Sink1.MassInPort1) annotation(Line(visible = true, points = {{-10.0,140.0},{-1.0,140.0}}, color = {191,0,191}));
        connect(Source2.MassInPort1,Generation.y) annotation(Line(visible = true, points = {{61.0,140.0},{70.0,140.0}}, color = {191,0,191}));
        connect(Gen_Const.y,Generation.u) annotation(Line(visible = true, points = {{80.0,111.5},{80.0,119.0}}, color = {0,0,191}));
        connect(NRFR.u,Natural_Resources.y4) annotation(Line(visible = true, points = {{126.5,51.0},{115.0,51.0},{115.0,127.0}}, color = {0,0,191}));
        connect(Generation.y1,Natural_Resources.u1) annotation(Line(visible = true, points = {{90.0,140.0},{108.0,140.0}}, color = {191,0,191}));
        connect(Natural_Resources.u2,Depletion.y) annotation(Line(visible = true, points = {{152.0,140.0},{170.0,140.0}}, color = {191,0,191}));
        connect(Depletion.y1,Sink2.MassInPort1) annotation(Line(visible = true, points = {{190.0,140.0},{199.0,140.0}}, color = {191,0,191}));
        connect(DRFM.y,Prod_5_2.u4) annotation(Line(visible = true, points = {{55.0,18.5},{55.0,60.0},{27.36,60.0},{27.36,71.5}}, color = {0,0,191}));
        Pop = Population.y;
        Pol = Pollution.y;
        Pol_rat = Pol_Ratio.y;
        Cap_inv = Capital_Investment.y;
        Qual_life = Quality_of_Life.y;
        Nat_res = Natural_Resources.y;
        NRUN = if time > 1970 then 0.25 else 1.0;
        NR_norm.u2 = NRUN;
        POLN = if time > 1970 then 0.25 else 1.0;
        Pol_norm.u2 = POLN;
        annotation(Icon(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-220.0,-264.0},{442.0,220.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-152.0,70.0},{368.0,140.0}}, textString = "2nd Modification", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {127,0,255}, extent = {{-148.0,-26.0},{372.0,44.0}}, textString = "POLN", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, extent = {{-150.0,-106.0},{370.0,-36.0}}, textString = "reduced to 0.25", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, extent = {{-150.0,-192.0},{370.0,-122.0}}, textString = "in 1970", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-62.0,6.0},{-58.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{98.0,192.0},{102.0,196.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{138.0,-16.0},{142.0,-12.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-22.0},{2.0,-18.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-178.0,-2.0},{-174.0,2.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{78.0,4.0},{82.0,8.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-97.0},{2.0,-93.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-178.0,-68.0},{-174.0,-64.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-68.0},{2.0,-64.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{158.0,28.0},{162.0,32.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{38.0,-262.0},{42.0,-258.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,-102.0},{302.0,-98.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,28.0},{302.0,32.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,78.0},{302.0,82.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,169.0},{302.0,173.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{342.0,-171.0},{346.0,-167.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{354.0,-234.0},{358.0,-230.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-97.0,6.0},{-93.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-137.0,-22.0},{-133.0,-18.0}})}), experiment(StartTime = 1900, StopTime = 2100), experimentSetupOutput, Documentation(info = "<html>
This is the second modification proposed by <a href=\"http://en.wikipedia.org/wiki/Jay_Wright_Forrester\">Jay Forrester</a> in his <font color=red><b>WORLD2</b></font> model. The generation of pollution is reduced in 1970 to 25% of their former value in order to reduce the effect of pollution on climate and food production. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Cellier, F.E. (1991), <a href=\"http://www.amazon.com/Continuous-System-Modeling-Fran%C3%A7ois-Cellier/dp/0387975020\">Continuous System Modeling</a>, Springer-Verlag, New York, ISBN: 0-387-97502-0, 755p.
<li> Forrester, J.W. (1971), <a href=\"http://www.amazon.com/World-Dynamics-Jay-W-Forrester/dp/1883823382/ref=ed_oe_h/103-2487145-1208659\">World Dynamics</a>, Pegasus Communications, 160p.
</ol> <p>
 
<hr> <p>
 
Simulate the model across 200 years, and plot the state variables as well as the quality of life as functions of the calendar years: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/SD_Fig3.png\"> <p>
 
<hr> <p>
</html>", revisions = ""));
      end Scenario_3;
      model Scenario_4 "4th Scenario"
        parameter Real Population_0 = 1650000000.0 "World population in 1900";
        parameter Real Pollution_0 = 200000000.0 "Pollution in 1900";
        parameter Real Nat_Resources_0(unit = "ton") = 900000000000.0 "Unrecoverable natural resources in 1900";
        parameter Real Cap_Invest_0(unit = "dollar") = 400000000.0 "Capital investment in 1900";
        parameter Real CIAF_0 = 0.2 "Proportion of capital investment in agriculture in 1900";
        parameter Real BRN(unit = "1/yr") = 0.04 "Normal birth rate";
        parameter Real CIAFN(unit = "dollar") = 0.3 "CIAF normalization";
        parameter Real CIAFT(unit = "yr") = 15.0 "CIAF time constant";
        parameter Real CIDN(unit = "dollar/yr") = 0.025 "Normal capital discard";
        parameter Real CIGN(unit = "dollar/yr") = 0.05 "Normal capital generation";
        parameter Real ECIRN(unit = "dollar") = 1.0 "Capital normalization";
        parameter Real FC(unit = "kg/yr") = 1.0 "Food coefficient";
        parameter Real FN(unit = "kg/yr") = 1.0 "Food normalization";
        parameter Real Land_Area(unit = "hectare") = 135000000.0 "Area of arable land";
        parameter Real NRI(unit = "ton") = 900000000000.0 "Initial natural resources";
        parameter Real POLS = 3599900000.0 "Standard pollution";
        parameter Real Pop_dens_norm(unit = "1/hectare") = 26.5 "Normal population density";
        parameter Real QLS = 1.0 "Standard quality of life";
        output Real Pop "World population";
        output Real Pol "Pollution";
        output Real Pol_rat "Pollution ratio";
        output Real Cap_inv(unit = "dollar") "Capital investment";
        output Real Qual_life "Quality of life";
        output Real Nat_res(unit = "ton") "Natural unrecoverable resources";
        Real NRUN(unit = "1/yr") "Normal resource utilization";
        Real POLN(unit = "1/yr") "Normal pollution";
        Real DRN(unit = "1/yr") "Normal death rate";
        Real Pop_meas;
      protected
        Real time_vals[12](each unit = "yr") = {1900,1910,1920,1930,1940,1950,1960,1970,1980,1990,2000,2500};
        Real pop_vals[12] = {1600000000.0,1750000000.0,1860000000.0,2070000000.0,2300000000.0,2556000000.0,3039500000.0,3706560000.0,4378100000.0,5190300000.0,5996200000.0,6000000000.0};
      public
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {-150.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Birth_Rate annotation(Placement(visible = true, transformation(origin = {-120.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Population(x0 = Population_0) annotation(Placement(visible = true, transformation(origin = {-70.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Death_Rate annotation(Placement(visible = true, transformation(origin = {-20.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink1 annotation(Placement(visible = true, transformation(origin = {10.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source2 annotation(Placement(visible = true, transformation(origin = {50.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Generation annotation(Placement(visible = true, transformation(origin = {80.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Natural_Resources(x0 = Nat_Resources_0) annotation(Placement(visible = true, transformation(origin = {130.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Depletion annotation(Placement(visible = true, transformation(origin = {180.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink2 annotation(Placement(visible = true, transformation(origin = {210.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Const Gen_Const(k = 0) annotation(Placement(visible = true, transformation(origin = {80.0,101.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular BRMM(x_vals = {0,1,2,3,4,5,20}, y_vals = {1.2,1.0,0.85,0.75,0.7,0.7,0.7}) annotation(Placement(visible = true, transformation(origin = {-105.0,185.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular DRMM(x_vals = {0,0.5,1,1.5,2,2.5,3,3.5,4,4.5,5,20}, y_vals = {3.0,1.8,1.0,0.8,0.7,0.6,0.53,0.5,0.5,0.5,0.5,0.5}) annotation(Placement(visible = true, transformation(origin = {51.0,171.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain BR_norm(k = BRN) annotation(Placement(visible = true, transformation(origin = {-135.0,103.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_5 Prod_5_1 annotation(Placement(visible = true, transformation(origin = {-160.0,80.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_5 Prod_5_2 annotation(Placement(visible = true, transformation(origin = {20.0,79.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular BRFM(x_vals = {0,1,2,3,4,20}, y_vals = {0.0,1.0,1.6,1.9,2.0,2.0}) annotation(Placement(visible = true, transformation(origin = {-176.0,35.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular BRPM(x_vals = 0:10:60, y_vals = {1.02,0.9,0.7,0.4,0.25,0.15,0.1}) annotation(Placement(visible = true, transformation(origin = {-135.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular BRCM(x_vals = 0:5, y_vals = {1.05,1.0,0.9,0.7,0.6,0.55}) annotation(Placement(visible = true, transformation(origin = {-95.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRCM(x_vals = 0:5, y_vals = {0.9,1.0,1.2,1.5,1.9,3.0}) annotation(Placement(visible = true, transformation(origin = {-25.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRPM(x_vals = 0:10:60, y_vals = {0.92,1.3,2.0,3.2,4.8,6.8,9.2}) annotation(Placement(visible = true, transformation(origin = {15.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRFM(x_vals = {0,0.25,0.5,0.75,1,1.25,1.5,1.75,2,20}, y_vals = {30.0,3.0,2.0,1.4,1.0,0.7,0.6,0.5,0.5,0.5}) annotation(Placement(visible = true, transformation(origin = {55.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Auxiliary.Gain Crowd_Rat(k = 1.0 / (Land_Area * Pop_dens_norm)) annotation(Placement(visible = true, transformation(origin = {-60.0,35.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = -90)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_1 annotation(Placement(visible = true, transformation(origin = {140.0,80.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular NRMM(x_vals = 0:10, y_vals = {0.0,1.0,1.8,2.4,2.9,3.3,3.6,3.8,3.9,3.95,4.0}) annotation(Placement(visible = true, transformation(origin = {201.0,80.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Sources.Source Source3 annotation(Placement(visible = true, transformation(origin = {50.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CI_Generation annotation(Placement(visible = true, transformation(origin = {80.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Capital_Investment(x0 = Cap_Invest_0) annotation(Placement(visible = true, transformation(origin = {130.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CI_Discard annotation(Placement(visible = true, transformation(origin = {180.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink3 annotation(Placement(visible = true, transformation(origin = {210.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIG_norm(k = CIGN) annotation(Placement(visible = true, transformation(origin = {65.0,-77.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_2 annotation(Placement(visible = true, transformation(origin = {40.0,-100.0}, extent = {{-14.0,-16.0},{14.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CID_norm(k = CIDN) annotation(Placement(visible = true, transformation(origin = {165.0,-77.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular CIM(x_vals = {0,1,2,3,4,5,20}, y_vals = {0.1,1.0,1.8,2.4,2.8,3.0,3.0}) annotation(Placement(visible = true, transformation(origin = {113.0,-100.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain MSL(k = 1.0 / ECIRN) annotation(Placement(visible = true, transformation(origin = {277.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain NRFR(k = 1.0 / NRI) annotation(Placement(visible = true, transformation(origin = {137.0,51.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain ECIR(k = 1.0 / (1.0 - CIAFN)) annotation(Placement(visible = true, transformation(origin = {243.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_3 Prod_3_1 annotation(Placement(visible = true, transformation(origin = {215.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular NREM(x_vals = 0:0.25:1, y_vals = {0.0,0.15,0.5,0.85,1.0}) annotation(Placement(visible = true, transformation(origin = {175.0,51.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -180)));
        Modelica.Blocks.Math.Division CIR annotation(Placement(visible = true, transformation(origin = {107.0,6.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source4 annotation(Placement(visible = true, transformation(origin = {-150.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 P_Generation annotation(Placement(visible = true, transformation(origin = {-120.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Pollution(x0 = Pollution_0) annotation(Placement(visible = true, transformation(origin = {-70.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 P_Absorption annotation(Placement(visible = true, transformation(origin = {-20.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink4 annotation(Placement(visible = true, transformation(origin = {10.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Linear NotCIAF(m = -1.0, b = 1.0) annotation(Placement(visible = true, transformation(origin = {189.0,10.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain Pol_Ratio(k = 1.0 / POLS) annotation(Placement(visible = true, transformation(origin = {-29.0,-95.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular FCM(x_vals = 0:5, y_vals = {2.4,1.0,0.6,0.4,0.3,0.2}) annotation(Placement(visible = true, transformation(origin = {-85.0,-47.0}, extent = {{-15.0,15.0},{15.0,-15.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_3 Prod_3_2 annotation(Placement(visible = true, transformation(origin = {-117.0,-66.0}, extent = {{15.0,-14.0},{-15.0,14.0}}, rotation = 90)));
        SystemDynamics.Auxiliary.Gain Food_Ratio(k = FC / FN) annotation(Placement(visible = true, transformation(origin = {-149.0,-66.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source5 annotation(Placement(visible = true, transformation(origin = {50.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CIAFG annotation(Placement(visible = true, transformation(origin = {80.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level CIAF(x0 = CIAF_0) annotation(Placement(visible = true, transformation(origin = {130.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CIAFD annotation(Placement(visible = true, transformation(origin = {180.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink5 annotation(Placement(visible = true, transformation(origin = {210.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIAF_D(k = 1.0 / CIAFT) annotation(Placement(visible = true, transformation(origin = {163.0,-180.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIAF_G(k = 1.0 / CIAFT) annotation(Placement(visible = true, transformation(origin = {95.0,-180.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_3 annotation(Placement(visible = true, transformation(origin = {120.0,-230.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division P_Abs annotation(Placement(visible = true, transformation(origin = {-43.0,-180.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_4 annotation(Placement(visible = true, transformation(origin = {-160.0,-220.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular POLCM(x_vals = {0,1,2,3,4,5,100}, y_vals = {0.05,1.0,3.0,5.4,7.4,8.0,8.0}) annotation(Placement(visible = true, transformation(origin = {-117.0,-220.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular POLAT(x_vals = 0:10:60, y_vals = {0.6,2.5,5.0,8.0,11.5,15.5,20.0}) annotation(Placement(visible = true, transformation(origin = {-39.0,-220.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular CFIFR(x_vals = {0,0.5,1,1.5,2,20}, y_vals = {1.0,0.6,0.3,0.15,0.1,0.1}) annotation(Placement(visible = true, transformation(origin = {69.0,-230.0}, extent = {{15.0,16.0},{-15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular FPM(x_vals = 0:10:60, y_vals = {1.02,0.9,0.65,0.35,0.2,0.1,0.05}) annotation(Placement(visible = true, transformation(origin = {-33.0,-66.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_5 annotation(Placement(visible = true, transformation(origin = {229.0,-200.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIRA(k = 1.0 / CIAFN) annotation(Placement(visible = true, transformation(origin = {255.0,-225.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular FPCI(x_vals = {0,1,2,3,4,5,6,100}, y_vals = {0.5,1.0,1.4,1.7,1.9,2.05,2.2,2.2}) annotation(Placement(visible = true, transformation(origin = {-153.0,-107.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular CIQR(y_vals = {0.7,0.8,1.0,1.5,2.0,2.0}, x_vals = {0,0.5,1,1.5,2,10}) annotation(Placement(visible = true, transformation(origin = {167.0,-230.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        Modelica.Blocks.Math.Division QLMF annotation(Placement(visible = true, transformation(origin = {315.0,-223.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular QLM(x_vals = {0,1,2,3,4,5,20}, y_vals = {0.2,1.0,1.7,2.3,2.7,2.9,2.9}) annotation(Placement(visible = true, transformation(origin = {319.0,-169.0}, extent = {{15.0,17.0},{-15.0,-17.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular QLF(x_vals = {0,1,2,3,4,20}, y_vals = {0.0,1.0,1.8,2.4,2.7,2.7}) annotation(Placement(visible = true, transformation(origin = {381.0,-232.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_4 Prod_4_1 annotation(Placement(visible = true, transformation(origin = {375.0,-163.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular QLC(x_vals = 0:0.5:5, y_vals = {2.0,1.3,1.0,0.75,0.55,0.45,0.38,0.3,0.25,0.22,0.2}) annotation(Placement(visible = true, transformation(origin = {335.0,-113.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular QLP(x_vals = 0:10:60, y_vals = {1.04,0.85,0.6,0.3,0.15,0.05,0.02}) annotation(Placement(visible = true, transformation(origin = {375.0,-113.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -90)));
        SystemDynamics.Auxiliary.Gain Quality_of_Life(k = QLS) annotation(Placement(visible = true, transformation(origin = {412.0,-163.0}, extent = {{-20.0,-21.0},{20.0,21.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World2.Utilities.Parameter_Change NR_norm annotation(Placement(visible = true, transformation(origin = {160.0,105.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -360)));
        SystemDynamics.WorldDynamics.World2.Utilities.Parameter_Change Pol_norm annotation(Placement(visible = true, transformation(origin = {-138.175,-180.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)), extent = {{-154,-196},{-124,-166}});
        SystemDynamics.WorldDynamics.World2.Utilities.Parameter_Change DR_norm annotation(Placement(visible = true, transformation(origin = {-0.0,100.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
      equation
        connect(Pol_norm.u1,Prod_2_4.y) annotation(Line(visible = true, points = {{-151.675,-174.0},{-160.0,-174.0},{-160.0,-208.8}}, color = {0,0,191}));
        connect(P_Generation.u,Pol_norm.y) annotation(Line(visible = true, points = {{-120.0,-161.0},{-120.0,-180.0},{-124.675,-180.0}}, color = {0,0,191}));
        connect(Depletion.u,NR_norm.y) annotation(Line(visible = true, origin = {177.8333,109.6667}, points = {{2.1667,9.3333},{2.1667,-4.6667},{-4.3333,-4.6667}}, color = {0,0,191}));
        connect(NR_norm.u1,Prod_2_1.y) annotation(Line(visible = true, points = {{146.5,111.0},{140.0,111.0},{140.0,91.2}}, color = {0,0,191}));
        connect(DR_norm.u1,Prod_5_2.y) annotation(Line(visible = true, points = {{13.5,106.0},{20.0,106.0},{20.0,89.5}}, color = {0,0,191}));
        connect(DR_norm.y,Death_Rate.u) annotation(Line(visible = true, origin = {-17.8333,106.3333}, points = {{4.3333,-6.3333},{-2.1667,-6.3333},{-2.1667,12.6667}}, color = {0,0,191}));
        connect(Food_Ratio.y,BRFM.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-176.0,-66.0},{-176.0,22.2}}, color = {0,0,191}));
        connect(BRFM.y,Prod_5_1.u2) annotation(Line(visible = true, points = {{-176.0,52.6},{-176.0,60.0},{-167.36,60.0},{-167.36,72.0}}, color = {0,0,191}));
        connect(BRPM.u,QLP.u) annotation(Line(visible = true, points = {{-135.0,23.0},{-135.0,-20.0},{-214.0,-20.0},{-214.0,214.0},{375.0,214.0},{375.0,-125.0}}, color = {0,0,191}));
        connect(Pol_Ratio.y,BRPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,-20.0},{-135.0,-20.0},{-135.0,23.0}}, color = {0,0,191}));
        connect(BRPM.y,Prod_5_1.u3) annotation(Line(visible = true, points = {{-135.0,51.5},{-135.0,60.0},{-160.0,60.0},{-160.0,68.8}}, color = {0,0,191}));
        connect(Crowd_Rat.y,BRCM.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-95.0,8.0},{-95.0,23.0}}, color = {0,0,191}));
        connect(BRCM.y,Prod_5_1.u4) annotation(Line(visible = true, points = {{-95.0,51.5},{-95.0,66.0},{-152.64,66.0},{-152.64,72.0}}, color = {0,0,191}));
        connect(Crowd_Rat.y,DRCM.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-25.0,8.0},{-25.0,23.0}}, color = {0,0,191}));
        connect(DRCM.y,Prod_5_2.u2) annotation(Line(visible = true, points = {{-25.0,51.5},{-25.0,60.0},{12.64,60.0},{12.64,71.5}}, color = {0,0,191}));
        connect(Pol_Ratio.y,DRPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,8.0},{15.0,8.0},{15.0,23.0}}, color = {0,0,191}));
        connect(DRPM.y,Prod_5_2.u3) annotation(Line(visible = true, points = {{15.0,51.5},{15.0,60.0},{20.0,60.0},{20.0,68.5}}, color = {0,0,191}));
        connect(Food_Ratio.y,DRFM.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-176.0,-66.0},{-176.0,0.0},{55.0,0.0},{55.0,23.0}}, color = {0,0,191}));
        connect(DRFM.y,Prod_5_2.u4) annotation(Line(visible = true, points = {{55.0,51.5},{55.0,60.0},{27.36,60.0},{27.36,71.5}}, color = {0,0,191}));
        connect(Crowd_Rat.y,QLC.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-206.0,8.0},{-206.0,208.0},{335.0,208.0},{335.0,-125.0}}, color = {0,0,191}));
        connect(FCM.u,Crowd_Rat.y) annotation(Line(visible = true, points = {{-73.0,-47.0},{-60.0,-47.0},{-60.0,23.8}}, color = {0,0,191}));
        connect(Crowd_Rat.u,Population.y3) annotation(Line(visible = true, points = {{-60.0,46.2},{-60.0,66.0},{-70.0,66.0},{-70.0,127.0}}, color = {0,0,191}));
        connect(Prod_2_5.u2,Prod_3_1.u2) annotation(Line(visible = true, points = {{239.5,-200.0},{260.0,-200.0},{260.0,-20.0},{160.0,-20.0},{160.0,30.0},{203.8,30.0}}, color = {0,0,191}));
        connect(Prod_3_1.y,ECIR.u) annotation(Line(visible = true, points = {{226.2,30.0},{232.5,30.0}}, color = {0,0,191}));
        connect(CIR.y,Prod_3_1.u2) annotation(Line(visible = true, points = {{90.5,6.0},{80.0,6.0},{80.0,30.0},{203.8,30.0}}, color = {0,0,191}));
        connect(NotCIAF.y,Prod_3_1.u3) annotation(Line(visible = true, points = {{202.5,10.0},{215.0,10.0},{215.0,19.5}}, color = {0,0,191}));
        connect(Prod_3_1.u1,NREM.y) annotation(Line(visible = true, points = {{215.0,40.5},{215.0,51.0},{191.5,51.0}}, color = {0,0,191}));
        connect(QLP.y,Prod_4_1.u1) annotation(Line(visible = true, points = {{375.0,-96.5},{375.0,-152.5}}, color = {0,0,191}));
        connect(QLC.y,Prod_4_1.u2) annotation(Line(visible = true, points = {{335.0,-96.5},{335.0,-157.0},{366.6,-157.0}}, color = {0,0,191}));
        connect(Prod_4_1.y,Quality_of_Life.u) annotation(Line(visible = true, points = {{385.5,-163.0},{398.0,-163.0}}, color = {0,0,191}));
        connect(QLF.y,Prod_4_1.u4) annotation(Line(visible = true, points = {{364.5,-232.0},{356.0,-232.0},{356.0,-190.0},{375.0,-190.0},{375.0,-173.5}}, color = {0,0,191}));
        connect(QLM.y,Prod_4_1.u3) annotation(Line(visible = true, points = {{335.5,-169.0},{366.6,-169.0}}, color = {0,0,191}));
        connect(QLF.y,QLMF.u2) annotation(Line(visible = true, points = {{364.5,-232.0},{333.0,-232.0}}, color = {0,0,191}));
        connect(QLMF.y,CIQR.u) annotation(Line(visible = true, points = {{298.5,-223.0},{290.0,-223.0},{290.0,-256.0},{190.0,-256.0},{190.0,-230.0},{179.0,-230.0}}, color = {0,0,191}));
        connect(P_Abs.u1,Pollution.y3) annotation(Line(visible = true, points = {{-61.0,-170.4},{-70.0,-170.4},{-70.0,-153.0}}, color = {0,0,191}));
        connect(CIG_norm.y,CI_Generation.u) annotation(Line(visible = true, points = {{75.5,-77.0},{80.0,-77.0},{80.0,-61.0}}, color = {0,0,191}));
        connect(CI_Discard.y1,Sink3.MassInPort1) annotation(Line(visible = true, points = {{190.0,-40.0},{199.0,-40.0}}, color = {191,0,191}));
        connect(Capital_Investment.u2,CI_Discard.y) annotation(Line(visible = true, points = {{152.0,-40.0},{170.0,-40.0}}, color = {191,0,191}));
        connect(Source3.MassInPort1,CI_Generation.y) annotation(Line(visible = true, points = {{61.0,-40.0},{70.0,-40.0}}, color = {191,0,191}));
        connect(CI_Generation.y1,Capital_Investment.u1) annotation(Line(visible = true, points = {{90.0,-40.0},{108.0,-40.0}}, color = {191,0,191}));
        connect(FPM.u,POLAT.u) annotation(Line(visible = true, points = {{-21.0,-66.0},{0.0,-66.0},{0.0,-220.0},{-27.0,-220.0}}, color = {0,0,191}));
        connect(CIRA.y,FPCI.u) annotation(Line(visible = true, points = {{265.5,-225.0},{280.0,-225.0},{280.0,-252.0},{-210.0,-252.0},{-210.0,-107.0},{-165.0,-107.0}}, color = {0,0,191}));
        connect(Prod_2_1.u1,Population.y1) annotation(Line(visible = true, points = {{128.8,80.0},{100.0,80.0},{100.0,194.0},{-46.0,194.0},{-46.0,151.0},{-53.0,151.0}}, color = {0,0,191}));
        connect(NRMM.u,DRMM.u) annotation(Line(visible = true, points = {{213.0,80.0},{300.0,80.0},{300.0,171.0},{63.0,171.0}}, color = {0,0,191}));
        connect(NRMM.u,BRMM.u) annotation(Line(visible = true, points = {{213.0,80.0},{300.0,80.0},{300.0,200.0},{-70.0,200.0},{-70.0,185.0},{-93.0,185.0}}, color = {0,0,191}));
        connect(CFIFR.u,QLF.u) annotation(Line(visible = true, points = {{57.0,-230.0},{40.0,-230.0},{40.0,-260.0},{400.0,-260.0},{400.0,-232.0},{393.0,-232.0}}, color = {0,0,191}));
        connect(Food_Ratio.y,QLF.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-220.0,-66.0},{-220.0,-260.0},{400.0,-260.0},{400.0,-232.0},{393.0,-232.0}}, color = {0,0,191}));
        connect(Pol_Ratio.y,FPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,-66.0},{-21.0,-66.0}}, color = {0,0,191}));
        connect(CIR.y,POLCM.u) annotation(Line(visible = true, points = {{90.5,6.0},{80.0,6.0},{80.0,-6.0},{10.0,-6.0},{10.0,-116.0},{-100.0,-116.0},{-100.0,-220.0},{-105.0,-220.0}}, color = {0,0,191}));
        connect(QLM.u,CIM.u) annotation(Line(visible = true, points = {{307.0,-169.0},{300.0,-169.0},{300.0,-100.0},{125.0,-100.0}}, color = {0,0,191}));
        connect(QLM.y,QLMF.u1) annotation(Line(visible = true, points = {{335.5,-169.0},{344.0,-169.0},{344.0,-214.0},{333.0,-214.0}}, color = {0,0,191}));
        connect(MSL.y,QLM.u) annotation(Line(visible = true, points = {{287.5,30.0},{300.0,30.0},{300.0,-169.0},{307.0,-169.0}}, color = {0,0,191}));
        connect(NRMM.y,Prod_2_1.u2) annotation(Line(visible = true, points = {{184.5,80.0},{151.2,80.0}}, color = {0,0,191}));
        connect(MSL.y,NRMM.u) annotation(Line(visible = true, points = {{287.5,30.0},{300.0,30.0},{300.0,80.0},{213.0,80.0}}, color = {0,0,191}));
        connect(ECIR.y,MSL.u) annotation(Line(visible = true, points = {{253.5,30.0},{266.5,30.0}}, color = {0,0,191}));
        connect(NotCIAF.u,CIAF.y1) annotation(Line(visible = true, points = {{175.5,10.0},{168.0,10.0},{168.0,-8.0},{240.0,-8.0},{240.0,-110.0},{154.0,-110.0},{154.0,-129.0},{147.0,-129.0}}, color = {0,0,191}));
        connect(Prod_2_2.u1,CIR.u2) annotation(Line(visible = true, points = {{30.2,-100.0},{20.0,-100.0},{20.0,-14.0},{140.0,-14.0},{140.0,-3.6},{125.0,-3.6}}, color = {0,0,191}));
        connect(Prod_2_2.u1,Population.y1) annotation(Line(visible = true, points = {{30.2,-100.0},{20.0,-100.0},{20.0,-14.0},{320.0,-14.0},{320.0,194.0},{-46.0,194.0},{-46.0,151.0},{-53.0,151.0}}, color = {0,0,191}));
        connect(Prod_2_4.u1,Population.y) annotation(Line(visible = true, points = {{-171.2,-220.0},{-200.0,-220.0},{-200.0,164.0},{-94.0,164.0},{-94.0,151.0},{-87.0,151.0}}, color = {0,0,191}));
        connect(CID_norm.u,Capital_Investment.y2) annotation(Line(visible = true, points = {{154.5,-77.0},{145.0,-77.0},{145.0,-53.0}}, color = {0,0,191}));
        connect(CID_norm.y,CI_Discard.u) annotation(Line(visible = true, points = {{175.5,-77.0},{180.0,-77.0},{180.0,-61.0}}, color = {0,0,191}));
        connect(CIM.y,Prod_2_2.u2) annotation(Line(visible = true, points = {{96.5,-100.0},{49.8,-100.0}}, color = {0,0,191}));
        connect(Prod_2_2.y,CIG_norm.u) annotation(Line(visible = true, points = {{40.0,-88.8},{40.0,-77.0},{54.5,-77.0}}, color = {0,0,191}));
        connect(Pol_Ratio.u,Pollution.y1) annotation(Line(visible = true, points = {{-39.5,-95.0},{-46.0,-95.0},{-46.0,-129.0},{-53.0,-129.0}}, color = {0,0,191}));
        connect(POLCM.y,Prod_2_4.u2) annotation(Line(visible = true, points = {{-133.5,-220.0},{-148.8,-220.0}}, color = {0,0,191}));
        connect(POLAT.y,P_Abs.u2) annotation(Line(visible = true, points = {{-55.5,-220.0},{-70.0,-220.0},{-70.0,-189.6},{-61.0,-189.6}}, color = {0,0,191}));
        connect(P_Abs.y,P_Absorption.u) annotation(Line(visible = true, points = {{-26.5,-180.0},{-20.0,-180.0},{-20.0,-161.0}}, color = {0,0,191}));
        connect(CFIFR.y,Prod_2_3.u1) annotation(Line(visible = true, points = {{85.5,-230.0},{108.8,-230.0}}, color = {0,0,191}));
        connect(CIQR.y,Prod_2_3.u2) annotation(Line(visible = true, points = {{150.5,-230.0},{131.2,-230.0}}, color = {0,0,191}));
        connect(CIAF_G.y,CIAFG.u) annotation(Line(visible = true, points = {{84.5,-180.0},{80.0,-180.0},{80.0,-161.0}}, color = {0,0,191}));
        connect(CIAF_G.u,Prod_2_3.y) annotation(Line(visible = true, points = {{105.5,-180.0},{120.0,-180.0},{120.0,-218.8}}, color = {0,0,191}));
        connect(CIRA.u,Prod_2_5.y) annotation(Line(visible = true, points = {{244.5,-225.0},{229.0,-225.0},{229.0,-211.2}}, color = {0,0,191}));
        connect(Prod_2_5.u1,CIAF.y3) annotation(Line(visible = true, points = {{218.5,-200.0},{130.0,-200.0},{130.0,-153.0}}, color = {0,0,191}));
        connect(CIAF_D.u,CIAF.y2) annotation(Line(visible = true, points = {{152.5,-180.0},{145.0,-180.0},{145.0,-153.0}}, color = {0,0,191}));
        connect(CIAF_D.y,CIAFD.u) annotation(Line(visible = true, points = {{173.5,-180.0},{180.0,-180.0},{180.0,-161.0}}, color = {0,0,191}));
        connect(CIAFD.y1,Sink5.MassInPort1) annotation(Line(visible = true, points = {{190.0,-140.0},{199.0,-140.0}}, color = {191,0,191}));
        connect(CIAF.u2,CIAFD.y) annotation(Line(visible = true, points = {{152.0,-140.0},{170.0,-140.0}}, color = {191,0,191}));
        connect(CIAFG.y1,CIAF.u1) annotation(Line(visible = true, points = {{90.0,-140.0},{108.0,-140.0}}, color = {191,0,191}));
        connect(Source5.MassInPort1,CIAFG.y) annotation(Line(visible = true, points = {{61.0,-140.0},{70.0,-140.0}}, color = {191,0,191}));
        connect(P_Absorption.y1,Sink4.MassInPort1) annotation(Line(visible = true, points = {{-10.0,-140.0},{-1.0,-140.0}}, color = {191,0,191}));
        connect(Pollution.u2,P_Absorption.y) annotation(Line(visible = true, points = {{-48.0,-140.0},{-30.0,-140.0}}, color = {191,0,191}));
        connect(P_Generation.y1,Pollution.u1) annotation(Line(visible = true, points = {{-110.0,-140.0},{-92.0,-140.0}}, color = {191,0,191}));
        connect(Source4.MassInPort1,P_Generation.y) annotation(Line(visible = true, points = {{-139.0,-140.0},{-130.0,-140.0}}, color = {191,0,191}));
        connect(FPCI.y,Prod_3_2.u3) annotation(Line(visible = true, points = {{-136.5,-107.0},{-117.0,-107.0},{-117.0,-76.5}}, color = {0,0,191}));
        connect(FPM.y,Prod_3_2.u2) annotation(Line(visible = true, points = {{-49.5,-66.0},{-107.2,-66.0}}, color = {0,0,191}));
        connect(Prod_3_2.u1,FCM.y) annotation(Line(visible = true, points = {{-117.0,-55.5},{-117.0,-47.0},{-101.5,-47.0}}, color = {0,0,191}));
        connect(Prod_3_2.y,Food_Ratio.u) annotation(Line(visible = true, points = {{-126.8,-66.0},{-138.5,-66.0}}, color = {0,0,191}));
        connect(CIR.u1,Capital_Investment.y1) annotation(Line(visible = true, points = {{125.0,15.6},{154.0,15.6},{154.0,-29.0},{147.0,-29.0}}, color = {0,0,191}));
        connect(NRFR.y,NREM.u) annotation(Line(visible = true, points = {{147.5,51.0},{163.0,51.0}}, color = {0,0,191}));
        connect(DRMM.y,Prod_5_2.u5) annotation(Line(visible = true, points = {{34.5,171.0},{26.0,171.0},{26.0,104.0},{40.0,104.0},{40.0,79.0},{31.2,79.0}}, color = {0,0,191}));
        connect(Prod_5_2.u1,Population.y2) annotation(Line(visible = true, points = {{8.8,79.0},{-55.0,79.0},{-55.0,127.0}}, color = {0,0,191}));
        connect(BRMM.y,Prod_5_1.u1) annotation(Line(visible = true, points = {{-121.5,185.0},{-180.0,185.0},{-180.0,80.0},{-171.2,80.0}}, color = {0,0,191}));
        connect(Prod_5_1.u5,Population.y4) annotation(Line(visible = true, points = {{-148.8,80.0},{-85.0,80.0},{-85.0,127.0}}, color = {0,0,191}));
        connect(Prod_5_1.y,BR_norm.u) annotation(Line(visible = true, points = {{-160.0,91.2},{-160.0,103.0},{-145.5,103.0}}, color = {0,0,191}));
        connect(BR_norm.y,Birth_Rate.u) annotation(Line(visible = true, points = {{-124.5,103.0},{-120.0,103.0},{-120.0,119.0}}, color = {0,0,191}));
        connect(Source1.MassInPort1,Birth_Rate.y) annotation(Line(visible = true, points = {{-139.0,140.0},{-130.0,140.0}}, color = {191,0,191}));
        connect(Birth_Rate.y1,Population.u1) annotation(Line(visible = true, points = {{-110.0,140.0},{-92.0,140.0}}, color = {191,0,191}));
        connect(Population.u2,Death_Rate.y) annotation(Line(visible = true, points = {{-48.0,140.0},{-30.0,140.0}}, color = {191,0,191}));
        connect(Death_Rate.y1,Sink1.MassInPort1) annotation(Line(visible = true, points = {{-10.0,140.0},{-1.0,140.0}}, color = {191,0,191}));
        connect(Source2.MassInPort1,Generation.y) annotation(Line(visible = true, points = {{61.0,140.0},{70.0,140.0}}, color = {191,0,191}));
        connect(Gen_Const.y,Generation.u) annotation(Line(visible = true, points = {{80.0,111.5},{80.0,119.0}}, color = {0,0,191}));
        connect(NRFR.u,Natural_Resources.y4) annotation(Line(visible = true, points = {{126.5,51.0},{115.0,51.0},{115.0,127.0}}, color = {0,0,191}));
        connect(Generation.y1,Natural_Resources.u1) annotation(Line(visible = true, points = {{90.0,140.0},{108.0,140.0}}, color = {191,0,191}));
        connect(Natural_Resources.u2,Depletion.y) annotation(Line(visible = true, points = {{152.0,140.0},{170.0,140.0}}, color = {191,0,191}));
        connect(Depletion.y1,Sink2.MassInPort1) annotation(Line(visible = true, points = {{190.0,140.0},{199.0,140.0}}, color = {191,0,191}));
        Pop = Population.y;
        Pol = Pollution.y;
        Pol_rat = Pol_Ratio.y;
        Cap_inv = Capital_Investment.y;
        Qual_life = Quality_of_Life.y;
        Nat_res = Natural_Resources.y;
        NRUN = if time > 1970 then 0.25 else 1.0;
        NR_norm.u2 = NRUN;
        POLN = if time > 1970 then 0.25 else 1.0;
        Pol_norm.u2 = POLN;
        DRN = if time > 1970 then 0.02 else 0.028;
        DR_norm.u2 = DRN;
        Pop_meas = Functions.Utilities.Piecewise(x = time, x_grid = time_vals, y_grid = pop_vals);
        annotation(Icon(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-220.0,-264.0},{442.0,220.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-152.0,70.0},{368.0,140.0}}, textString = "3rd Modification", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {127,0,255}, extent = {{-148.0,-26.0},{372.0,44.0}}, textString = "DRN", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, extent = {{-150.0,-106.0},{370.0,-36.0}}, textString = "reduced to 0.2", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, extent = {{-150.0,-192.0},{370.0,-122.0}}, textString = "in 1970", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-62.0,6.0},{-58.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{98.0,192.0},{102.0,196.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{138.0,-16.0},{142.0,-12.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-22.0},{2.0,-18.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-178.0,-2.0},{-174.0,2.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{78.0,4.0},{82.0,8.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-97.0},{2.0,-93.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-178.0,-68.0},{-174.0,-64.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-68.0},{2.0,-64.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{158.0,28.0},{162.0,32.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{38.0,-262.0},{42.0,-258.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,-102.0},{302.0,-98.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,28.0},{302.0,32.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,78.0},{302.0,82.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,169.0},{302.0,173.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{342.0,-171.0},{346.0,-167.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{354.0,-234.0},{358.0,-230.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-97.0,6.0},{-93.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-137.0,-22.0},{-133.0,-18.0}})}), experiment(StartTime = 1900, StopTime = 2000), experimentSetupOutput, Documentation(info = "<html>
This is a further modification of <a href=\"http://en.wikipedia.org/wiki/Jay_Wright_Forrester\">Jay Forrester's</a> <font color=red><b>WORLD2</b></font> model. The death rate is reduced in 1970 to better account for the recent advances in medicine. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Cellier, F.E. (1991), <a href=\"http://www.amazon.com/Continuous-System-Modeling-Fran%C3%A7ois-Cellier/dp/0387975020\">Continuous System Modeling</a>, Springer-Verlag, New York, ISBN: 0-387-97502-0, 755p.
<li> Forrester, J.W. (1971), <a href=\"http://www.amazon.com/World-Dynamics-Jay-W-Forrester/dp/1883823382/ref=ed_oe_h/103-2487145-1208659\">World Dynamics</a>, Pegasus Communications, 160p.
</ol> <p>
 
<hr> <p>
 
Simulate the model across 100 years, and plot the simulated and the measured world populations together on a single graph against the calendar years: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/SD_Fig4.png\"> <p>
 
<hr> <p>
</html>", revisions = ""));
      end Scenario_4;
      model Scenario_5 "5th Scenario"
        parameter Real Population_0 = 1650000000.0 "World population in 1900";
        parameter Real Pollution_0 = 200000000.0 "Pollution in 1900";
        parameter Real Nat_Resources_0(unit = "ton") = 900000000000.0 "Unrecoverable natural resources in 1900";
        parameter Real Cap_Invest_0(unit = "dollar") = 400000000.0 "Capital investment in 1900";
        parameter Real CIAF_0 = 0.2 "Proportion of capital investment in agriculture in 1900";
        parameter Real BRN(unit = "1/yr") = 0.04 "Normal birth rate";
        parameter Real CIAFN(unit = "dollar") = 0.3 "CIAF normalization";
        parameter Real CIAFT(unit = "yr") = 15.0 "CIAF time constant";
        parameter Real CIDN(unit = "dollar/yr") = 0.025 "Normal capital discard";
        parameter Real CIGN(unit = "dollar/yr") = 0.05 "Normal capital generation";
        parameter Real ECIRN(unit = "dollar") = 1.0 "Capital normalization";
        parameter Real FC(unit = "kg/yr") = 1.0 "Food coefficient";
        parameter Real FN(unit = "kg/yr") = 1.0 "Food normalization";
        parameter Real Land_Area(unit = "hectare") = 135000000.0 "Area of arable land";
        parameter Real NRI(unit = "ton") = 900000000000.0 "Initial natural resources";
        parameter Real POLS = 3599900000.0 "Standard pollution";
        parameter Real Pop_dens_norm(unit = "1/hectare") = 26.5 "Normal population density";
        parameter Real QLS = 1.0 "Standard quality of life";
        Real years(start = 1900) "Time in calendar years";
        output Real Pop "World population";
        output Real Pol "Pollution";
        output Real Pol_rat "Pollution ratio";
        output Real Cap_inv(unit = "dollar") "Capital investment";
        output Real Qual_life "Quality of life";
        output Real Nat_res(unit = "ton") "Natural unrecoverable resources";
        Real NRUN(unit = "1/yr") "Normal resource utilization";
        Real POLN(unit = "1/yr") "Normal pollution";
        Real DRN(unit = "1/yr") "Normal death rate";
        Real Pop_meas;
      protected
        Real time_vals[12] = {1900,1910,1920,1930,1940,1950,1960,1970,1980,1990,2000,2500};
        Real pop_vals[12] = {1600000000.0,1750000000.0,1860000000.0,2070000000.0,2300000000.0,2556000000.0,3039500000.0,3706560000.0,4378100000.0,5190300000.0,5996200000.0,6000000000.0};
        Real dir;
        parameter Real time_reverse(unit = "yr") = 200 "Time of time reversal";
      public
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {-150.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Birth_Rate annotation(Placement(visible = true, transformation(origin = {-120.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Reverse_Level Population(x0 = Population_0) annotation(Placement(visible = true, transformation(origin = {-70.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Death_Rate annotation(Placement(visible = true, transformation(origin = {-20.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink1 annotation(Placement(visible = true, transformation(origin = {10.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source2 annotation(Placement(visible = true, transformation(origin = {50.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Generation annotation(Placement(visible = true, transformation(origin = {80.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Reverse_Level Natural_Resources(x0 = Nat_Resources_0) annotation(Placement(visible = true, transformation(origin = {130.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Depletion annotation(Placement(visible = true, transformation(origin = {180.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink2 annotation(Placement(visible = true, transformation(origin = {210.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Const Gen_Const(k = 0) annotation(Placement(visible = true, transformation(origin = {80.0,101.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular BRMM(x_vals = {0,1,2,3,4,5,20}, y_vals = {1.2,1.0,0.85,0.75,0.7,0.7,0.7}) annotation(Placement(visible = true, transformation(origin = {-105.0,185.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular DRMM(x_vals = {0,0.5,1,1.5,2,2.5,3,3.5,4,4.5,5,20}, y_vals = {3.0,1.8,1.0,0.8,0.7,0.6,0.53,0.5,0.5,0.5,0.5,0.5}) annotation(Placement(visible = true, transformation(origin = {51.0,171.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain BR_norm(k = BRN) annotation(Placement(visible = true, transformation(origin = {-135.0,103.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_5 Prod_5_1 annotation(Placement(visible = true, transformation(origin = {-160.0,80.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_5 Prod_5_2 annotation(Placement(visible = true, transformation(origin = {20.0,79.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular BRFM(x_vals = {0,1,2,3,4,20}, y_vals = {0.0,1.0,1.6,1.9,2.0,2.0}) annotation(Placement(visible = true, transformation(origin = {-176.0,35.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular BRPM(x_vals = 0:10:60, y_vals = {1.02,0.9,0.7,0.4,0.25,0.15,0.1}) annotation(Placement(visible = true, transformation(origin = {-135.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular BRCM(x_vals = 0:5, y_vals = {1.05,1.0,0.9,0.7,0.6,0.55}) annotation(Placement(visible = true, transformation(origin = {-95.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRCM(x_vals = 0:5, y_vals = {0.9,1.0,1.2,1.5,1.9,3.0}) annotation(Placement(visible = true, transformation(origin = {-25.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRPM(x_vals = 0:10:60, y_vals = {0.92,1.3,2.0,3.2,4.8,6.8,9.2}) annotation(Placement(visible = true, transformation(origin = {15.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRFM(x_vals = {0,0.25,0.5,0.75,1,1.25,1.5,1.75,2,20}, y_vals = {30.0,3.0,2.0,1.4,1.0,0.7,0.6,0.5,0.5,0.5}) annotation(Placement(visible = true, transformation(origin = {55.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Auxiliary.Gain Crowd_Rat(k = 1.0 / (Land_Area * Pop_dens_norm)) annotation(Placement(visible = true, transformation(origin = {-60.0,35.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = -90)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_1 annotation(Placement(visible = true, transformation(origin = {140.0,80.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular NRMM(x_vals = 0:10, y_vals = {0.0,1.0,1.8,2.4,2.9,3.3,3.6,3.8,3.9,3.95,4.0}) annotation(Placement(visible = true, transformation(origin = {201.0,80.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Sources.Source Source3 annotation(Placement(visible = true, transformation(origin = {50.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CI_Generation annotation(Placement(visible = true, transformation(origin = {80.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Reverse_Level Capital_Investment(x0 = Cap_Invest_0) annotation(Placement(visible = true, transformation(origin = {130.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CI_Discard annotation(Placement(visible = true, transformation(origin = {180.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink3 annotation(Placement(visible = true, transformation(origin = {210.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIG_norm(k = CIGN) annotation(Placement(visible = true, transformation(origin = {65.0,-77.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_2 annotation(Placement(visible = true, transformation(origin = {40.0,-100.0}, extent = {{-14.0,-16.0},{14.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CID_norm(k = CIDN) annotation(Placement(visible = true, transformation(origin = {165.0,-77.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular CIM(x_vals = {0,1,2,3,4,5,20}, y_vals = {0.1,1.0,1.8,2.4,2.8,3.0,3.0}) annotation(Placement(visible = true, transformation(origin = {113.0,-100.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain MSL(k = 1.0 / ECIRN) annotation(Placement(visible = true, transformation(origin = {277.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain NRFR(k = 1.0 / NRI) annotation(Placement(visible = true, transformation(origin = {137.0,51.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain ECIR(k = 1.0 / (1.0 - CIAFN)) annotation(Placement(visible = true, transformation(origin = {243.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_3 Prod_3_1 annotation(Placement(visible = true, transformation(origin = {215.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular NREM(x_vals = 0:0.25:1, y_vals = {0.0,0.15,0.5,0.85,1.0}) annotation(Placement(visible = true, transformation(origin = {175.0,51.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -180)));
        Modelica.Blocks.Math.Division CIR annotation(Placement(visible = true, transformation(origin = {107.0,6.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source4 annotation(Placement(visible = true, transformation(origin = {-150.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 P_Generation annotation(Placement(visible = true, transformation(origin = {-120.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Reverse_Level Pollution(x0 = Pollution_0) annotation(Placement(visible = true, transformation(origin = {-70.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 P_Absorption annotation(Placement(visible = true, transformation(origin = {-20.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink4 annotation(Placement(visible = true, transformation(origin = {10.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Linear NotCIAF(m = -1.0, b = 1.0) annotation(Placement(visible = true, transformation(origin = {189.0,10.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain Pol_Ratio(k = 1.0 / POLS) annotation(Placement(visible = true, transformation(origin = {-29.0,-95.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular FCM(x_vals = 0:5, y_vals = {2.4,1.0,0.6,0.4,0.3,0.2}) annotation(Placement(visible = true, transformation(origin = {-85.0,-47.0}, extent = {{-15.0,15.0},{15.0,-15.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_3 Prod_3_2 annotation(Placement(visible = true, transformation(origin = {-117.0,-66.0}, extent = {{15.0,-14.0},{-15.0,14.0}}, rotation = 90)));
        SystemDynamics.Auxiliary.Gain Food_Ratio(k = FC / FN) annotation(Placement(visible = true, transformation(origin = {-149.0,-66.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source5 annotation(Placement(visible = true, transformation(origin = {50.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CIAFG annotation(Placement(visible = true, transformation(origin = {80.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Reverse_Level CIAF(x0 = CIAF_0) annotation(Placement(visible = true, transformation(origin = {130.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CIAFD annotation(Placement(visible = true, transformation(origin = {180.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink5 annotation(Placement(visible = true, transformation(origin = {210.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIAF_D(k = 1.0 / CIAFT) annotation(Placement(visible = true, transformation(origin = {163.0,-180.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIAF_G(k = 1.0 / CIAFT) annotation(Placement(visible = true, transformation(origin = {95.0,-180.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_3 annotation(Placement(visible = true, transformation(origin = {120.0,-230.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division P_Abs annotation(Placement(visible = true, transformation(origin = {-43.0,-180.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_4 annotation(Placement(visible = true, transformation(origin = {-160.0,-220.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular POLCM(x_vals = {0,1,2,3,4,5,100}, y_vals = {0.05,1.0,3.0,5.4,7.4,8.0,8.0}) annotation(Placement(visible = true, transformation(origin = {-117.0,-220.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular POLAT(x_vals = 0:10:60, y_vals = {0.6,2.5,5.0,8.0,11.5,15.5,20.0}) annotation(Placement(visible = true, transformation(origin = {-39.0,-220.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular CFIFR(x_vals = {0,0.5,1,1.5,2,20}, y_vals = {1.0,0.6,0.3,0.15,0.1,0.1}) annotation(Placement(visible = true, transformation(origin = {69.0,-230.0}, extent = {{15.0,16.0},{-15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular FPM(x_vals = 0:10:60, y_vals = {1.02,0.9,0.65,0.35,0.2,0.1,0.05}) annotation(Placement(visible = true, transformation(origin = {-33.0,-66.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_5 annotation(Placement(visible = true, transformation(origin = {229.0,-200.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIRA(k = 1.0 / CIAFN) annotation(Placement(visible = true, transformation(origin = {255.0,-225.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular FPCI(x_vals = {0,1,2,3,4,5,6,100}, y_vals = {0.5,1.0,1.4,1.7,1.9,2.05,2.2,2.2}) annotation(Placement(visible = true, transformation(origin = {-153.0,-107.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular CIQR(y_vals = {0.7,0.8,1.0,1.5,2.0,2.0}, x_vals = {0,0.5,1,1.5,2,10}) annotation(Placement(visible = true, transformation(origin = {167.0,-230.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        Modelica.Blocks.Math.Division QLMF annotation(Placement(visible = true, transformation(origin = {315.0,-223.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular QLM(x_vals = {0,1,2,3,4,5,20}, y_vals = {0.2,1.0,1.7,2.3,2.7,2.9,2.9}) annotation(Placement(visible = true, transformation(origin = {319.0,-169.0}, extent = {{15.0,17.0},{-15.0,-17.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular QLF(x_vals = {0,1,2,3,4,20}, y_vals = {0.0,1.0,1.8,2.4,2.7,2.7}) annotation(Placement(visible = true, transformation(origin = {381.0,-232.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_4 Prod_4_1 annotation(Placement(visible = true, transformation(origin = {375.0,-163.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular QLC(x_vals = 0:0.5:5, y_vals = {2.0,1.3,1.0,0.75,0.55,0.45,0.38,0.3,0.25,0.22,0.2}) annotation(Placement(visible = true, transformation(origin = {335.0,-113.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular QLP(x_vals = 0:10:60, y_vals = {1.04,0.85,0.6,0.3,0.15,0.05,0.02}) annotation(Placement(visible = true, transformation(origin = {375.0,-113.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -90)));
        SystemDynamics.Auxiliary.Gain Quality_of_Life(k = QLS) annotation(Placement(visible = true, transformation(origin = {412.0,-163.0}, extent = {{-20.0,-21.0},{20.0,21.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World2.Utilities.Parameter_Change NR_norm annotation(Placement(visible = true, transformation(origin = {160.0,105.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World2.Utilities.Parameter_Change Pol_norm annotation(Placement(visible = true, transformation(origin = {-140.0,-180.0}, extent = {{-14.1444,-14.1444},{14.1444,14.1444}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World2.Utilities.Parameter_Change DR_norm annotation(Placement(visible = true, transformation(origin = {-2.5,102.5}, extent = {{-12.5,12.5},{12.5,-12.5}}, rotation = 180)));
      equation
        connect(Depletion.u,NR_norm.y) annotation(Line(visible = true, origin = {177.8333,109.6667}, points = {{2.1667,9.3333},{2.1667,-4.6667},{-4.3333,-4.6667}}, color = {0,0,191}));
        connect(NR_norm.u1,Prod_2_1.y) annotation(Line(visible = true, points = {{146.5,111.0},{140.0,111.0},{140.0,91.2}}, color = {0,0,191}));
        connect(P_Generation.u,Pol_norm.y) annotation(Line(visible = true, points = {{-120.0,-161.0},{-120.0,-180.0},{-127.27,-180.0}}, color = {0,0,191}));
        connect(Pol_norm.u1,Prod_2_4.y) annotation(Line(visible = true, points = {{-152.73,-174.3422},{-160.0,-174.3422},{-160.0,-208.8}}, color = {0,0,191}));
        connect(DR_norm.u1,Prod_5_2.y) annotation(Line(visible = true, points = {{8.75,107.5},{20.0,107.5},{20.0,89.5}}, color = {0,0,191}));
        connect(DR_norm.y,Death_Rate.u) annotation(Line(visible = true, origin = {-17.9167,108.0}, points = {{4.1667,-5.5},{-2.0833,-5.5},{-2.0833,11.0}}, color = {0,0,191}));
        connect(Food_Ratio.y,BRFM.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-176.0,-66.0},{-176.0,22.2}}, color = {0,0,191}));
        connect(BRFM.y,Prod_5_1.u2) annotation(Line(visible = true, points = {{-176.0,52.6},{-176.0,60.0},{-167.36,60.0},{-167.36,72.0}}, color = {0,0,191}));
        connect(BRPM.u,QLP.u) annotation(Line(visible = true, points = {{-135.0,23.0},{-135.0,-20.0},{-214.0,-20.0},{-214.0,214.0},{375.0,214.0},{375.0,-125.0}}, color = {0,0,191}));
        connect(Pol_Ratio.y,BRPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,-20.0},{-135.0,-20.0},{-135.0,23.0}}, color = {0,0,191}));
        connect(BRPM.y,Prod_5_1.u3) annotation(Line(visible = true, points = {{-135.0,51.5},{-135.0,60.0},{-160.0,60.0},{-160.0,68.8}}, color = {0,0,191}));
        connect(Crowd_Rat.y,BRCM.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-95.0,8.0},{-95.0,23.0}}, color = {0,0,191}));
        connect(BRCM.y,Prod_5_1.u4) annotation(Line(visible = true, points = {{-95.0,51.5},{-95.0,66.0},{-152.64,66.0},{-152.64,72.0}}, color = {0,0,191}));
        connect(Crowd_Rat.y,DRCM.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-25.0,8.0},{-25.0,23.0}}, color = {0,0,191}));
        connect(DRCM.y,Prod_5_2.u2) annotation(Line(visible = true, points = {{-25.0,51.5},{-25.0,60.0},{12.64,60.0},{12.64,71.5}}, color = {0,0,191}));
        connect(Pol_Ratio.y,DRPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,8.0},{15.0,8.0},{15.0,23.0}}, color = {0,0,191}));
        connect(DRPM.y,Prod_5_2.u3) annotation(Line(visible = true, points = {{15.0,51.5},{15.0,60.0},{20.0,60.0},{20.0,68.5}}, color = {0,0,191}));
        connect(Food_Ratio.y,DRFM.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-176.0,-66.0},{-176.0,0.0},{55.0,0.0},{55.0,23.0}}, color = {0,0,191}));
        connect(DRFM.y,Prod_5_2.u4) annotation(Line(visible = true, points = {{55.0,51.5},{55.0,60.0},{27.36,60.0},{27.36,71.5}}, color = {0,0,191}));
        connect(Crowd_Rat.y,QLC.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-206.0,8.0},{-206.0,208.0},{335.0,208.0},{335.0,-125.0}}, color = {0,0,191}));
        connect(FCM.u,Crowd_Rat.y) annotation(Line(visible = true, points = {{-73.0,-47.0},{-60.0,-47.0},{-60.0,23.8}}, color = {0,0,191}));
        connect(Crowd_Rat.u,Population.y3) annotation(Line(visible = true, points = {{-60.0,46.2},{-60.0,66.0},{-70.0,66.0},{-70.0,127.0}}, color = {0,0,191}));
        connect(Prod_2_5.u2,Prod_3_1.u2) annotation(Line(visible = true, points = {{239.5,-200.0},{260.0,-200.0},{260.0,-20.0},{160.0,-20.0},{160.0,30.0},{203.8,30.0}}, color = {0,0,191}));
        connect(Prod_3_1.y,ECIR.u) annotation(Line(visible = true, points = {{226.2,30.0},{232.5,30.0}}, color = {0,0,191}));
        connect(CIR.y,Prod_3_1.u2) annotation(Line(visible = true, points = {{90.5,6.0},{80.0,6.0},{80.0,30.0},{203.8,30.0}}, color = {0,0,191}));
        connect(NotCIAF.y,Prod_3_1.u3) annotation(Line(visible = true, points = {{202.5,10.0},{215.0,10.0},{215.0,19.5}}, color = {0,0,191}));
        connect(Prod_3_1.u1,NREM.y) annotation(Line(visible = true, points = {{215.0,40.5},{215.0,51.0},{191.5,51.0}}, color = {0,0,191}));
        connect(QLP.y,Prod_4_1.u1) annotation(Line(visible = true, points = {{375.0,-96.5},{375.0,-152.5}}, color = {0,0,191}));
        connect(QLC.y,Prod_4_1.u2) annotation(Line(visible = true, points = {{335.0,-96.5},{335.0,-157.0},{366.6,-157.0}}, color = {0,0,191}));
        connect(Prod_4_1.y,Quality_of_Life.u) annotation(Line(visible = true, points = {{385.5,-163.0},{398.0,-163.0}}, color = {0,0,191}));
        connect(QLF.y,Prod_4_1.u4) annotation(Line(visible = true, points = {{364.5,-232.0},{356.0,-232.0},{356.0,-190.0},{375.0,-190.0},{375.0,-173.5}}, color = {0,0,191}));
        connect(QLM.y,Prod_4_1.u3) annotation(Line(visible = true, points = {{335.5,-169.0},{366.6,-169.0}}, color = {0,0,191}));
        connect(QLF.y,QLMF.u2) annotation(Line(visible = true, points = {{364.5,-232.0},{333.0,-232.0}}, color = {0,0,191}));
        connect(QLMF.y,CIQR.u) annotation(Line(visible = true, points = {{298.5,-223.0},{290.0,-223.0},{290.0,-256.0},{190.0,-256.0},{190.0,-230.0},{179.0,-230.0}}, color = {0,0,191}));
        connect(P_Abs.u1,Pollution.y3) annotation(Line(visible = true, points = {{-61.0,-170.4},{-70.0,-170.4},{-70.0,-153.0}}, color = {0,0,191}));
        connect(CIG_norm.y,CI_Generation.u) annotation(Line(visible = true, points = {{75.5,-77.0},{80.0,-77.0},{80.0,-61.0}}, color = {0,0,191}));
        connect(CI_Discard.y1,Sink3.MassInPort1) annotation(Line(visible = true, points = {{190.0,-40.0},{199.0,-40.0}}, color = {191,0,191}));
        connect(Capital_Investment.u2,CI_Discard.y) annotation(Line(visible = true, points = {{152.0,-40.0},{170.0,-40.0}}, color = {191,0,191}));
        connect(Source3.MassInPort1,CI_Generation.y) annotation(Line(visible = true, points = {{61.0,-40.0},{70.0,-40.0}}, color = {191,0,191}));
        connect(CI_Generation.y1,Capital_Investment.u1) annotation(Line(visible = true, points = {{90.0,-40.0},{108.0,-40.0}}, color = {191,0,191}));
        connect(FPM.u,POLAT.u) annotation(Line(visible = true, points = {{-21.0,-66.0},{0.0,-66.0},{0.0,-220.0},{-27.0,-220.0}}, color = {0,0,191}));
        connect(CIRA.y,FPCI.u) annotation(Line(visible = true, points = {{265.5,-225.0},{280.0,-225.0},{280.0,-252.0},{-210.0,-252.0},{-210.0,-107.0},{-165.0,-107.0}}, color = {0,0,191}));
        connect(Prod_2_1.u1,Population.y1) annotation(Line(visible = true, points = {{128.8,80.0},{100.0,80.0},{100.0,194.0},{-46.0,194.0},{-46.0,151.0},{-53.0,151.0}}, color = {0,0,191}));
        connect(NRMM.u,DRMM.u) annotation(Line(visible = true, points = {{213.0,80.0},{300.0,80.0},{300.0,171.0},{63.0,171.0}}, color = {0,0,191}));
        connect(NRMM.u,BRMM.u) annotation(Line(visible = true, points = {{213.0,80.0},{300.0,80.0},{300.0,200.0},{-70.0,200.0},{-70.0,185.0},{-93.0,185.0}}, color = {0,0,191}));
        connect(CFIFR.u,QLF.u) annotation(Line(visible = true, points = {{57.0,-230.0},{40.0,-230.0},{40.0,-260.0},{400.0,-260.0},{400.0,-232.0},{393.0,-232.0}}, color = {0,0,191}));
        connect(Food_Ratio.y,QLF.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-220.0,-66.0},{-220.0,-260.0},{400.0,-260.0},{400.0,-232.0},{393.0,-232.0}}, color = {0,0,191}));
        connect(Pol_Ratio.y,FPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,-66.0},{-21.0,-66.0}}, color = {0,0,191}));
        connect(CIR.y,POLCM.u) annotation(Line(visible = true, points = {{90.5,6.0},{80.0,6.0},{80.0,-6.0},{10.0,-6.0},{10.0,-116.0},{-100.0,-116.0},{-100.0,-220.0},{-105.0,-220.0}}, color = {0,0,191}));
        connect(QLM.u,CIM.u) annotation(Line(visible = true, points = {{307.0,-169.0},{300.0,-169.0},{300.0,-100.0},{125.0,-100.0}}, color = {0,0,191}));
        connect(QLM.y,QLMF.u1) annotation(Line(visible = true, points = {{335.5,-169.0},{344.0,-169.0},{344.0,-214.0},{333.0,-214.0}}, color = {0,0,191}));
        connect(MSL.y,QLM.u) annotation(Line(visible = true, points = {{287.5,30.0},{300.0,30.0},{300.0,-169.0},{307.0,-169.0}}, color = {0,0,191}));
        connect(NRMM.y,Prod_2_1.u2) annotation(Line(visible = true, points = {{184.5,80.0},{151.2,80.0}}, color = {0,0,191}));
        connect(MSL.y,NRMM.u) annotation(Line(visible = true, points = {{287.5,30.0},{300.0,30.0},{300.0,80.0},{213.0,80.0}}, color = {0,0,191}));
        connect(ECIR.y,MSL.u) annotation(Line(visible = true, points = {{253.5,30.0},{266.5,30.0}}, color = {0,0,191}));
        connect(NotCIAF.u,CIAF.y1) annotation(Line(visible = true, points = {{175.5,10.0},{168.0,10.0},{168.0,-8.0},{240.0,-8.0},{240.0,-110.0},{154.0,-110.0},{154.0,-129.0},{147.0,-129.0}}, color = {0,0,191}));
        connect(Prod_2_2.u1,CIR.u2) annotation(Line(visible = true, points = {{30.2,-100.0},{20.0,-100.0},{20.0,-14.0},{140.0,-14.0},{140.0,-3.6},{125.0,-3.6}}, color = {0,0,191}));
        connect(Prod_2_2.u1,Population.y1) annotation(Line(visible = true, points = {{30.2,-100.0},{20.0,-100.0},{20.0,-14.0},{320.0,-14.0},{320.0,194.0},{-46.0,194.0},{-46.0,151.0},{-53.0,151.0}}, color = {0,0,191}));
        connect(Prod_2_4.u1,Population.y) annotation(Line(visible = true, points = {{-171.2,-220.0},{-200.0,-220.0},{-200.0,164.0},{-94.0,164.0},{-94.0,151.0},{-87.0,151.0}}, color = {0,0,191}));
        connect(CID_norm.u,Capital_Investment.y2) annotation(Line(visible = true, points = {{154.5,-77.0},{145.0,-77.0},{145.0,-53.0}}, color = {0,0,191}));
        connect(CID_norm.y,CI_Discard.u) annotation(Line(visible = true, points = {{175.5,-77.0},{180.0,-77.0},{180.0,-61.0}}, color = {0,0,191}));
        connect(CIM.y,Prod_2_2.u2) annotation(Line(visible = true, points = {{96.5,-100.0},{49.8,-100.0}}, color = {0,0,191}));
        connect(Prod_2_2.y,CIG_norm.u) annotation(Line(visible = true, points = {{40.0,-88.8},{40.0,-77.0},{54.5,-77.0}}, color = {0,0,191}));
        connect(Pol_Ratio.u,Pollution.y1) annotation(Line(visible = true, points = {{-39.5,-95.0},{-46.0,-95.0},{-46.0,-129.0},{-53.0,-129.0}}, color = {0,0,191}));
        connect(POLCM.y,Prod_2_4.u2) annotation(Line(visible = true, points = {{-133.5,-220.0},{-148.8,-220.0}}, color = {0,0,191}));
        connect(POLAT.y,P_Abs.u2) annotation(Line(visible = true, points = {{-55.5,-220.0},{-70.0,-220.0},{-70.0,-189.6},{-61.0,-189.6}}, color = {0,0,191}));
        connect(P_Abs.y,P_Absorption.u) annotation(Line(visible = true, points = {{-26.5,-180.0},{-20.0,-180.0},{-20.0,-161.0}}, color = {0,0,191}));
        connect(CFIFR.y,Prod_2_3.u1) annotation(Line(visible = true, points = {{85.5,-230.0},{108.8,-230.0}}, color = {0,0,191}));
        connect(CIQR.y,Prod_2_3.u2) annotation(Line(visible = true, points = {{150.5,-230.0},{131.2,-230.0}}, color = {0,0,191}));
        connect(CIAF_G.y,CIAFG.u) annotation(Line(visible = true, points = {{84.5,-180.0},{80.0,-180.0},{80.0,-161.0}}, color = {0,0,191}));
        connect(CIAF_G.u,Prod_2_3.y) annotation(Line(visible = true, points = {{105.5,-180.0},{120.0,-180.0},{120.0,-218.8}}, color = {0,0,191}));
        connect(CIRA.u,Prod_2_5.y) annotation(Line(visible = true, points = {{244.5,-225.0},{229.0,-225.0},{229.0,-211.2}}, color = {0,0,191}));
        connect(Prod_2_5.u1,CIAF.y3) annotation(Line(visible = true, points = {{218.5,-200.0},{130.0,-200.0},{130.0,-153.0}}, color = {0,0,191}));
        connect(CIAF_D.u,CIAF.y2) annotation(Line(visible = true, points = {{152.5,-180.0},{145.0,-180.0},{145.0,-153.0}}, color = {0,0,191}));
        connect(CIAF_D.y,CIAFD.u) annotation(Line(visible = true, points = {{173.5,-180.0},{180.0,-180.0},{180.0,-161.0}}, color = {0,0,191}));
        connect(CIAFD.y1,Sink5.MassInPort1) annotation(Line(visible = true, points = {{190.0,-140.0},{199.0,-140.0}}, color = {191,0,191}));
        connect(CIAF.u2,CIAFD.y) annotation(Line(visible = true, points = {{152.0,-140.0},{170.0,-140.0}}, color = {191,0,191}));
        connect(CIAFG.y1,CIAF.u1) annotation(Line(visible = true, points = {{90.0,-140.0},{108.0,-140.0}}, color = {191,0,191}));
        connect(Source5.MassInPort1,CIAFG.y) annotation(Line(visible = true, points = {{61.0,-140.0},{70.0,-140.0}}, color = {191,0,191}));
        connect(P_Absorption.y1,Sink4.MassInPort1) annotation(Line(visible = true, points = {{-10.0,-140.0},{-1.0,-140.0}}, color = {191,0,191}));
        connect(Pollution.u2,P_Absorption.y) annotation(Line(visible = true, points = {{-48.0,-140.0},{-30.0,-140.0}}, color = {191,0,191}));
        connect(P_Generation.y1,Pollution.u1) annotation(Line(visible = true, points = {{-110.0,-140.0},{-92.0,-140.0}}, color = {191,0,191}));
        connect(Source4.MassInPort1,P_Generation.y) annotation(Line(visible = true, points = {{-139.0,-140.0},{-130.0,-140.0}}, color = {191,0,191}));
        connect(FPCI.y,Prod_3_2.u3) annotation(Line(visible = true, points = {{-136.5,-107.0},{-117.0,-107.0},{-117.0,-76.5}}, color = {0,0,191}));
        connect(FPM.y,Prod_3_2.u2) annotation(Line(visible = true, points = {{-49.5,-66.0},{-107.2,-66.0}}, color = {0,0,191}));
        connect(Prod_3_2.u1,FCM.y) annotation(Line(visible = true, points = {{-117.0,-55.5},{-117.0,-47.0},{-101.5,-47.0}}, color = {0,0,191}));
        connect(Prod_3_2.y,Food_Ratio.u) annotation(Line(visible = true, points = {{-126.8,-66.0},{-138.5,-66.0}}, color = {0,0,191}));
        connect(CIR.u1,Capital_Investment.y1) annotation(Line(visible = true, points = {{125.0,15.6},{154.0,15.6},{154.0,-29.0},{147.0,-29.0}}, color = {0,0,191}));
        connect(NRFR.y,NREM.u) annotation(Line(visible = true, points = {{147.5,51.0},{163.0,51.0}}, color = {0,0,191}));
        connect(DRMM.y,Prod_5_2.u5) annotation(Line(visible = true, points = {{34.5,171.0},{26.0,171.0},{26.0,104.0},{40.0,104.0},{40.0,79.0},{31.2,79.0}}, color = {0,0,191}));
        connect(Prod_5_2.u1,Population.y2) annotation(Line(visible = true, points = {{8.8,79.0},{-55.0,79.0},{-55.0,127.0}}, color = {0,0,191}));
        connect(BRMM.y,Prod_5_1.u1) annotation(Line(visible = true, points = {{-121.5,185.0},{-180.0,185.0},{-180.0,80.0},{-171.2,80.0}}, color = {0,0,191}));
        connect(Prod_5_1.u5,Population.y4) annotation(Line(visible = true, points = {{-148.8,80.0},{-85.0,80.0},{-85.0,127.0}}, color = {0,0,191}));
        connect(Prod_5_1.y,BR_norm.u) annotation(Line(visible = true, points = {{-160.0,91.2},{-160.0,103.0},{-145.5,103.0}}, color = {0,0,191}));
        connect(BR_norm.y,Birth_Rate.u) annotation(Line(visible = true, points = {{-124.5,103.0},{-120.0,103.0},{-120.0,119.0}}, color = {0,0,191}));
        connect(Source1.MassInPort1,Birth_Rate.y) annotation(Line(visible = true, points = {{-139.0,140.0},{-130.0,140.0}}, color = {191,0,191}));
        connect(Birth_Rate.y1,Population.u1) annotation(Line(visible = true, points = {{-110.0,140.0},{-92.0,140.0}}, color = {191,0,191}));
        connect(Population.u2,Death_Rate.y) annotation(Line(visible = true, points = {{-48.0,140.0},{-30.0,140.0}}, color = {191,0,191}));
        connect(Death_Rate.y1,Sink1.MassInPort1) annotation(Line(visible = true, points = {{-10.0,140.0},{-1.0,140.0}}, color = {191,0,191}));
        connect(Source2.MassInPort1,Generation.y) annotation(Line(visible = true, points = {{61.0,140.0},{70.0,140.0}}, color = {191,0,191}));
        connect(Gen_Const.y,Generation.u) annotation(Line(visible = true, points = {{80.0,111.5},{80.0,119.0}}, color = {0,0,191}));
        connect(NRFR.u,Natural_Resources.y4) annotation(Line(visible = true, points = {{126.5,51.0},{115.0,51.0},{115.0,127.0}}, color = {0,0,191}));
        connect(Generation.y1,Natural_Resources.u1) annotation(Line(visible = true, points = {{90.0,140.0},{108.0,140.0}}, color = {191,0,191}));
        connect(Natural_Resources.u2,Depletion.y) annotation(Line(visible = true, points = {{152.0,140.0},{170.0,140.0}}, color = {191,0,191}));
        connect(Depletion.y1,Sink2.MassInPort1) annotation(Line(visible = true, points = {{190.0,140.0},{199.0,140.0}}, color = {191,0,191}));
        Pop = Population.y;
        Pol = Pollution.y;
        Pol_rat = Pol_Ratio.y;
        Cap_inv = Capital_Investment.y;
        Qual_life = Quality_of_Life.y;
        Nat_res = Natural_Resources.y;
        NRUN = if years > 1970 then 0.25 else 1.0;
        NR_norm.u2 = NRUN;
        POLN = if years > 1970 then 0.25 else 1.0;
        Pol_norm.u2 = POLN;
        DRN = if years > 1970 then 0.02 else 0.028;
        DR_norm.u2 = DRN;
        Pop_meas = Functions.Utilities.Piecewise(x = years, x_grid = time_vals, y_grid = pop_vals);
        dir = if time > time_reverse then -1 else 1;
        Population.dir = dir;
        Pollution.dir = dir;
        Natural_Resources.dir = dir;
        Capital_Investment.dir = dir;
        CIAF.dir = dir;
        der(years) = dir;
        annotation(Icon(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-220.0,-264.0},{442.0,220.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-152.0,70.0},{368.0,140.0}}, textString = "4th Modification", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, extent = {{-148.0,-26.0},{372.0,44.0}}, textString = "Simulation", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, extent = {{-150.0,-106.0},{370.0,-36.0}}, textString = "made", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, extent = {{-150.0,-192.0},{370.0,-122.0}}, textString = "reversible in time", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-62.0,6.0},{-58.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{98.0,192.0},{102.0,196.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{138.0,-16.0},{142.0,-12.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-22.0},{2.0,-18.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-178.0,-2.0},{-174.0,2.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{78.0,4.0},{82.0,8.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-97.0},{2.0,-93.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-178.0,-68.0},{-174.0,-64.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-68.0},{2.0,-64.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{158.0,28.0},{162.0,32.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{38.0,-262.0},{42.0,-258.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,-102.0},{302.0,-98.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,28.0},{302.0,32.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,78.0},{302.0,82.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,169.0},{302.0,173.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{342.0,-171.0},{346.0,-167.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{354.0,-234.0},{358.0,-230.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-97.0,6.0},{-93.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-137.0,-22.0},{-133.0,-18.0}})}), experiment(StopTime = 233, Tolerance = 0.0000000001), experimentSetupOutput, Documentation(info = "<html>
This is yet another modification of <a href=\"http://en.wikipedia.org/wiki/Jay_Wright_Forrester\">Jay Forrester's</a> <font color=red><b>WORLD2</b></font> model. We wish to introduce time reversal to the simulation.  To this end, we replace the regular continuous levels of the System Dynamics methodology by reversible levels. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Cellier, F.E. (1991), <a href=\"http://www.amazon.com/Continuous-System-Modeling-Fran%C3%A7ois-Cellier/dp/0387975020\">Continuous System Modeling</a>, Springer-Verlag, New York, ISBN: 0-387-97502-0, 755p.
<li> Forrester, J.W. (1971), <a href=\"http://www.amazon.com/World-Dynamics-Jay-W-Forrester/dp/1883823382/ref=ed_oe_h/103-2487145-1208659\">World Dynamics</a>, Pegasus Communications, 160p.
</ol> <p>
 
<hr> <p>
 
Simulate the model across 233 years with time reversal starting at year 200, i.e., we simulate the world model forward through time from the year 1900 until the year 2100, and then backward through time from the year 2100 until the year 2067.  For this experiment, you need to set the simulation accuracy to 1e-10. <p>
 
Plot the simulated world population against the calendar years. <p>
<img src=\"modelica://SystemDynamics/Resources/Images/SD_Fig5.png\"> <p>
 
You shall notice that time reversal in the world model is numerically problematic.  The simulation is numerically unstable in backward direction.  The time-reversed model simulates correctly for a few years, but as numerical errors get the trajectory away from its designed path, the backward trajectory starts diverging quickly from the forward trajectory. <p>
 
<hr> <p>
</html>", revisions = ""));
      end Scenario_5;
      model Scenario_6 "6th Scenario"
        parameter Real Population_0 = 1650000000.0 "World population in 1900";
        parameter Real Pollution_0 = 200000000.0 "Pollution in 1900";
        parameter Real Nat_Resources_0(unit = "ton") = 900000000000.0 "Unrecoverable natural resources in 1900";
        parameter Real Cap_Invest_0(unit = "dollar") = 400000000.0 "Capital investment in 1900";
        parameter Real CIAF_0 = 0.2 "Proportion of capital investment in agriculture in 1900";
        parameter Real BRN(unit = "1/yr") = 0.04 "Normal birth rate";
        parameter Real CIAFN(unit = "dollar") = 0.3 "CIAF normalization";
        parameter Real CIAFT(unit = "yr") = 15.0 "CIAF time constant";
        parameter Real CIDN(unit = "dollar/yr") = 0.025 "Normal capital discard";
        parameter Real CIGN(unit = "dollar/yr") = 0.05 "Normal capital generation";
        parameter Real DRN(unit = "1/yr") = 0.028 "Normal death rate";
        parameter Real ECIRN(unit = "dollar") = 1.0 "Capital normalization";
        parameter Real FC(unit = "kg/yr") = 1.0 "Food coefficient";
        parameter Real FN(unit = "kg/yr") = 1.0 "Food normalization";
        parameter Real Land_Area(unit = "hectare") = 135000000.0 "Area of arable land";
        parameter Real NRI(unit = "ton") = 900000000000.0 "Initial natural resources";
        parameter Real POLN(unit = "1/yr") = 1.0 "Normal pollution";
        parameter Real POLS = 3599900000.0 "Standard pollution";
        parameter Real Pop_dens_norm(unit = "1/hectare") = 26.5 "Normal population density";
        parameter Real QLS = 1.0 "Standard quality of life";
        output Real Pop "World population";
        output Real Pol "Pollution";
        output Real Pol_rat "Pollution ratio";
        output Real Cap_inv(unit = "dollar") "Capital investment";
        output Real Qual_life "Quality of life";
        output Real Nat_res(unit = "ton") "Natural unrecoverable resources";
        parameter Real NRUN2(unit = "1/yr") = 1.0 "Resource utilization after 1970";
        Real NRUN(unit = "1/yr") "Normal resource utilization";
        Real rel_dPop(unit = "1/yr") "Relative derivative of population";
        Real min_dPop(unit = "1/yr") "Minimal derivative of population";
        Real min_QL(start = 1) "Minimum quality of life";
        Real Perf_Index "Preformance index";
        constant Real dPop_fact(unit = "1/yr") = 1 "Dimensionality factor";
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {-150.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Birth_Rate annotation(Placement(visible = true, transformation(origin = {-120.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Population(x0 = Population_0) annotation(Placement(visible = true, transformation(origin = {-70.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Death_Rate annotation(Placement(visible = true, transformation(origin = {-20.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink1 annotation(Placement(visible = true, transformation(origin = {10.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source2 annotation(Placement(visible = true, transformation(origin = {50.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Generation annotation(Placement(visible = true, transformation(origin = {80.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Natural_Resources(x0 = Nat_Resources_0) annotation(Placement(visible = true, transformation(origin = {130.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Depletion annotation(Placement(visible = true, transformation(origin = {180.0,140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink2 annotation(Placement(visible = true, transformation(origin = {210.0,140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Const Gen_Const(k = 0) annotation(Placement(visible = true, transformation(origin = {80.0,101.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular BRMM(x_vals = {0,1,2,3,4,5,20}, y_vals = {1.2,1.0,0.85,0.75,0.7,0.7,0.7}) annotation(Placement(visible = true, transformation(origin = {-105.0,185.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular DRMM(x_vals = {0,0.5,1,1.5,2,2.5,3,3.5,4,4.5,5,20}, y_vals = {3.0,1.8,1.0,0.8,0.7,0.6,0.53,0.5,0.5,0.5,0.5,0.5}) annotation(Placement(visible = true, transformation(origin = {51.0,171.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain BR_norm(k = BRN) annotation(Placement(visible = true, transformation(origin = {-135.0,103.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_5 Prod_5_1 annotation(Placement(visible = true, transformation(origin = {-160.0,80.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain DR_norm(k = DRN) annotation(Placement(visible = true, transformation(origin = {-3.0,103.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_5 Prod_5_2 annotation(Placement(visible = true, transformation(origin = {20.0,79.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular BRFM(x_vals = {0,1,2,3,4,20}, y_vals = {0.0,1.0,1.6,1.9,2.0,2.0}) annotation(Placement(visible = true, transformation(origin = {-176.0,35.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular BRPM(x_vals = 0:10:60, y_vals = {1.02,0.9,0.7,0.4,0.25,0.15,0.1}) annotation(Placement(visible = true, transformation(origin = {-135.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular BRCM(x_vals = 0:5, y_vals = {1.05,1.0,0.9,0.7,0.6,0.55}) annotation(Placement(visible = true, transformation(origin = {-95.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRCM(x_vals = 0:5, y_vals = {0.9,1.0,1.2,1.5,1.9,3.0}) annotation(Placement(visible = true, transformation(origin = {-25.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRPM(x_vals = 0:10:60, y_vals = {0.92,1.3,2.0,3.2,4.8,6.8,9.2}) annotation(Placement(visible = true, transformation(origin = {15.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular DRFM(x_vals = {0,0.25,0.5,0.75,1,1.25,1.5,1.75,2,20}, y_vals = {30.0,3.0,2.0,1.4,1.0,0.7,0.6,0.5,0.5,0.5}) annotation(Placement(visible = true, transformation(origin = {55.0,35.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -270)));
        SystemDynamics.Auxiliary.Gain Crowd_Rat(k = 1.0 / (Land_Area * Pop_dens_norm)) annotation(Placement(visible = true, transformation(origin = {-60.0,35.0}, extent = {{-16.0,-15.0},{16.0,15.0}}, rotation = -90)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_1 annotation(Placement(visible = true, transformation(origin = {140.0,80.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular NRMM(x_vals = 0:10, y_vals = {0.0,1.0,1.8,2.4,2.9,3.3,3.6,3.8,3.9,3.95,4.0}) annotation(Placement(visible = true, transformation(origin = {201.0,80.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Sources.Source Source3 annotation(Placement(visible = true, transformation(origin = {50.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CI_Generation annotation(Placement(visible = true, transformation(origin = {80.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Capital_Investment(x0 = Cap_Invest_0) annotation(Placement(visible = true, transformation(origin = {130.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CI_Discard annotation(Placement(visible = true, transformation(origin = {180.0,-40.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink3 annotation(Placement(visible = true, transformation(origin = {210.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIG_norm(k = CIGN) annotation(Placement(visible = true, transformation(origin = {65.0,-77.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_2 annotation(Placement(visible = true, transformation(origin = {40.0,-100.0}, extent = {{-14.0,-16.0},{14.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CID_norm(k = CIDN) annotation(Placement(visible = true, transformation(origin = {165.0,-77.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular CIM(x_vals = {0,1,2,3,4,5,20}, y_vals = {0.1,1.0,1.8,2.4,2.8,3.0,3.0}) annotation(Placement(visible = true, transformation(origin = {113.0,-100.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain MSL(k = 1.0 / ECIRN) annotation(Placement(visible = true, transformation(origin = {277.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain NRFR(k = 1.0 / NRI) annotation(Placement(visible = true, transformation(origin = {137.0,51.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain ECIR(k = 1.0 / (1.0 - CIAFN)) annotation(Placement(visible = true, transformation(origin = {243.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_3 Prod_3_1 annotation(Placement(visible = true, transformation(origin = {215.0,30.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular NREM(x_vals = 0:0.25:1, y_vals = {0.0,0.15,0.5,0.85,1.0}) annotation(Placement(visible = true, transformation(origin = {175.0,51.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -180)));
        Modelica.Blocks.Math.Division CIR annotation(Placement(visible = true, transformation(origin = {107.0,6.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source4 annotation(Placement(visible = true, transformation(origin = {-150.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 P_Generation annotation(Placement(visible = true, transformation(origin = {-120.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Pollution(x0 = Pollution_0) annotation(Placement(visible = true, transformation(origin = {-70.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 P_Absorption annotation(Placement(visible = true, transformation(origin = {-20.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink4 annotation(Placement(visible = true, transformation(origin = {10.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Linear NotCIAF(m = -1.0, b = 1.0) annotation(Placement(visible = true, transformation(origin = {189.0,10.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain Pol_Ratio(k = 1.0 / POLS) annotation(Placement(visible = true, transformation(origin = {-29.0,-95.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular FCM(x_vals = 0:5, y_vals = {2.4,1.0,0.6,0.4,0.3,0.2}) annotation(Placement(visible = true, transformation(origin = {-85.0,-47.0}, extent = {{-15.0,15.0},{15.0,-15.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_3 Prod_3_2 annotation(Placement(visible = true, transformation(origin = {-117.0,-66.0}, extent = {{15.0,-14.0},{-15.0,14.0}}, rotation = 90)));
        SystemDynamics.Auxiliary.Gain Food_Ratio(k = FC / FN) annotation(Placement(visible = true, transformation(origin = {-149.0,-66.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source5 annotation(Placement(visible = true, transformation(origin = {50.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CIAFG annotation(Placement(visible = true, transformation(origin = {80.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Levels.Level CIAF(x0 = CIAF_0) annotation(Placement(visible = true, transformation(origin = {130.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 CIAFD annotation(Placement(visible = true, transformation(origin = {180.0,-140.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink5 annotation(Placement(visible = true, transformation(origin = {210.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIAF_D(k = 1.0 / CIAFT) annotation(Placement(visible = true, transformation(origin = {163.0,-180.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIAF_G(k = 1.0 / CIAFT) annotation(Placement(visible = true, transformation(origin = {95.0,-180.0}, extent = {{15.0,-16.0},{-15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_3 annotation(Placement(visible = true, transformation(origin = {120.0,-230.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division P_Abs annotation(Placement(visible = true, transformation(origin = {-43.0,-180.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain Pol_norm(k = POLN) annotation(Placement(visible = true, transformation(origin = {-135.0,-180.0}, extent = {{-15.0,-16.0},{15.0,16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_4 annotation(Placement(visible = true, transformation(origin = {-160.0,-220.0}, extent = {{-16.0,-16.0},{16.0,16.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular POLCM(x_vals = {0,1,2,3,4,5,100}, y_vals = {0.05,1.0,3.0,5.4,7.4,8.0,8.0}) annotation(Placement(visible = true, transformation(origin = {-117.0,-220.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular POLAT(x_vals = 0:10:60, y_vals = {0.6,2.5,5.0,8.0,11.5,15.5,20.0}) annotation(Placement(visible = true, transformation(origin = {-39.0,-220.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular CFIFR(x_vals = {0,0.5,1,1.5,2,20}, y_vals = {1.0,0.6,0.3,0.15,0.1,0.1}) annotation(Placement(visible = true, transformation(origin = {69.0,-230.0}, extent = {{15.0,16.0},{-15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular FPM(x_vals = 0:10:60, y_vals = {1.02,0.9,0.65,0.35,0.2,0.1,0.05}) annotation(Placement(visible = true, transformation(origin = {-33.0,-66.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_2 Prod_2_5 annotation(Placement(visible = true, transformation(origin = {229.0,-200.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Gain CIRA(k = 1.0 / CIAFN) annotation(Placement(visible = true, transformation(origin = {255.0,-225.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular FPCI(x_vals = {0,1,2,3,4,5,6,100}, y_vals = {0.5,1.0,1.4,1.7,1.9,2.05,2.2,2.2}) annotation(Placement(visible = true, transformation(origin = {-153.0,-107.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular CIQR(y_vals = {0.7,0.8,1.0,1.5,2.0,2.0}, x_vals = {0,0.5,1,1.5,2,10}) annotation(Placement(visible = true, transformation(origin = {167.0,-230.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        Modelica.Blocks.Math.Division QLMF annotation(Placement(visible = true, transformation(origin = {315.0,-223.0}, extent = {{15.0,-15.0},{-15.0,15.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular QLM(x_vals = {0,1,2,3,4,5,20}, y_vals = {0.2,1.0,1.7,2.3,2.7,2.9,2.9}) annotation(Placement(visible = true, transformation(origin = {319.0,-169.0}, extent = {{15.0,17.0},{-15.0,-17.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular QLF(x_vals = {0,1,2,3,4,20}, y_vals = {0.0,1.0,1.8,2.4,2.7,2.7}) annotation(Placement(visible = true, transformation(origin = {381.0,-232.0}, extent = {{-15.0,16.0},{15.0,-16.0}}, rotation = -180)));
        SystemDynamics.Auxiliary.Prod_4 Prod_4_1 annotation(Placement(visible = true, transformation(origin = {375.0,-163.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular QLC(x_vals = 0:0.5:5, y_vals = {2.0,1.3,1.0,0.75,0.55,0.45,0.38,0.3,0.25,0.22,0.2}) annotation(Placement(visible = true, transformation(origin = {335.0,-113.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular QLP(x_vals = 0:10:60, y_vals = {1.04,0.85,0.6,0.3,0.15,0.05,0.02}) annotation(Placement(visible = true, transformation(origin = {375.0,-113.0}, extent = {{15.0,15.0},{-15.0,-15.0}}, rotation = -90)));
        SystemDynamics.Auxiliary.Gain Quality_of_Life(k = QLS) annotation(Placement(visible = true, transformation(origin = {412.0,-163.0}, extent = {{-20.0,-21.0},{20.0,21.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World2.Utilities.Parameter_Change NR_norm annotation(Placement(visible = true, transformation(origin = {158.8111,105.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
      equation
        connect(Depletion.u,NR_norm.y) annotation(Line(visible = true, origin = {177.437,109.6667}, points = {{2.563,9.3333},{2.563,-4.6667},{-5.1259,-4.6667}}, color = {0,0,191}));
        connect(NR_norm.u1,Prod_2_1.y) annotation(Line(visible = true, points = {{145.3111,111.0},{140.0,111.0},{140.0,91.2}}, color = {0,0,191}));
        connect(Food_Ratio.y,BRFM.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-176.0,-66.0},{-176.0,22.2}}, color = {0,0,191}));
        connect(BRFM.y,Prod_5_1.u2) annotation(Line(visible = true, points = {{-176.0,52.6},{-176.0,60.0},{-167.36,60.0},{-167.36,72.0}}, color = {0,0,191}));
        connect(BRPM.u,QLP.u) annotation(Line(visible = true, points = {{-135.0,23.0},{-135.0,-20.0},{-214.0,-20.0},{-214.0,214.0},{375.0,214.0},{375.0,-125.0}}, color = {0,0,191}));
        connect(Pol_Ratio.y,BRPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,-20.0},{-135.0,-20.0},{-135.0,23.0}}, color = {0,0,191}));
        connect(BRPM.y,Prod_5_1.u3) annotation(Line(visible = true, points = {{-135.0,51.5},{-135.0,60.0},{-160.0,60.0},{-160.0,68.8}}, color = {0,0,191}));
        connect(Crowd_Rat.y,BRCM.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-95.0,8.0},{-95.0,23.0}}, color = {0,0,191}));
        connect(BRCM.y,Prod_5_1.u4) annotation(Line(visible = true, points = {{-95.0,51.5},{-95.0,66.0},{-152.64,66.0},{-152.64,72.0}}, color = {0,0,191}));
        connect(Crowd_Rat.y,DRCM.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-25.0,8.0},{-25.0,23.0}}, color = {0,0,191}));
        connect(DRCM.y,Prod_5_2.u2) annotation(Line(visible = true, points = {{-25.0,51.5},{-25.0,60.0},{12.64,60.0},{12.64,71.5}}, color = {0,0,191}));
        connect(Pol_Ratio.y,DRPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,8.0},{15.0,8.0},{15.0,23.0}}, color = {0,0,191}));
        connect(DRPM.y,Prod_5_2.u3) annotation(Line(visible = true, points = {{15.0,51.5},{15.0,60.0},{20.0,60.0},{20.0,68.5}}, color = {0,0,191}));
        connect(Food_Ratio.y,DRFM.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-176.0,-66.0},{-176.0,0.0},{55.0,0.0},{55.0,23.0}}, color = {0,0,191}));
        connect(DRFM.y,Prod_5_2.u4) annotation(Line(visible = true, points = {{55.0,51.5},{55.0,60.0},{27.36,60.0},{27.36,71.5}}, color = {0,0,191}));
        connect(Crowd_Rat.y,QLC.u) annotation(Line(visible = true, points = {{-60.0,23.8},{-60.0,8.0},{-206.0,8.0},{-206.0,208.0},{335.0,208.0},{335.0,-125.0}}, color = {0,0,191}));
        connect(FCM.u,Crowd_Rat.y) annotation(Line(visible = true, points = {{-73.0,-47.0},{-60.0,-47.0},{-60.0,23.8}}, color = {0,0,191}));
        connect(Crowd_Rat.u,Population.y3) annotation(Line(visible = true, points = {{-60.0,46.2},{-60.0,66.0},{-70.0,66.0},{-70.0,127.0}}, color = {0,0,191}));
        connect(Prod_2_5.u2,Prod_3_1.u2) annotation(Line(visible = true, points = {{239.5,-200.0},{260.0,-200.0},{260.0,-20.0},{160.0,-20.0},{160.0,30.0},{203.8,30.0}}, color = {0,0,191}));
        connect(Prod_3_1.y,ECIR.u) annotation(Line(visible = true, points = {{226.2,30.0},{232.5,30.0}}, color = {0,0,191}));
        connect(CIR.y,Prod_3_1.u2) annotation(Line(visible = true, points = {{90.5,6.0},{80.0,6.0},{80.0,30.0},{203.8,30.0}}, color = {0,0,191}));
        connect(NotCIAF.y,Prod_3_1.u3) annotation(Line(visible = true, points = {{202.5,10.0},{215.0,10.0},{215.0,19.5}}, color = {0,0,191}));
        connect(Prod_3_1.u1,NREM.y) annotation(Line(visible = true, points = {{215.0,40.5},{215.0,51.0},{191.5,51.0}}, color = {0,0,191}));
        connect(QLP.y,Prod_4_1.u1) annotation(Line(visible = true, points = {{375.0,-96.5},{375.0,-152.5}}, color = {0,0,191}));
        connect(QLC.y,Prod_4_1.u2) annotation(Line(visible = true, points = {{335.0,-96.5},{335.0,-157.0},{366.6,-157.0}}, color = {0,0,191}));
        connect(Prod_4_1.y,Quality_of_Life.u) annotation(Line(visible = true, points = {{385.5,-163.0},{398.0,-163.0}}, color = {0,0,191}));
        connect(QLF.y,Prod_4_1.u4) annotation(Line(visible = true, points = {{364.5,-232.0},{356.0,-232.0},{356.0,-190.0},{375.0,-190.0},{375.0,-173.5}}, color = {0,0,191}));
        connect(QLM.y,Prod_4_1.u3) annotation(Line(visible = true, points = {{335.5,-169.0},{366.6,-169.0}}, color = {0,0,191}));
        connect(QLF.y,QLMF.u2) annotation(Line(visible = true, points = {{364.5,-232.0},{333.0,-232.0}}, color = {0,0,191}));
        connect(QLMF.y,CIQR.u) annotation(Line(visible = true, points = {{298.5,-223.0},{290.0,-223.0},{290.0,-256.0},{190.0,-256.0},{190.0,-230.0},{179.0,-230.0}}, color = {0,0,191}));
        connect(P_Abs.u1,Pollution.y3) annotation(Line(visible = true, points = {{-61.0,-170.4},{-70.0,-170.4},{-70.0,-153.0}}, color = {0,0,191}));
        connect(CIG_norm.y,CI_Generation.u) annotation(Line(visible = true, points = {{75.5,-77.0},{80.0,-77.0},{80.0,-61.0}}, color = {0,0,191}));
        connect(CI_Discard.y1,Sink3.MassInPort1) annotation(Line(visible = true, points = {{190.0,-40.0},{199.0,-40.0}}, color = {191,0,191}));
        connect(Capital_Investment.u2,CI_Discard.y) annotation(Line(visible = true, points = {{152.0,-40.0},{170.0,-40.0}}, color = {191,0,191}));
        connect(Source3.MassInPort1,CI_Generation.y) annotation(Line(visible = true, points = {{61.0,-40.0},{70.0,-40.0}}, color = {191,0,191}));
        connect(CI_Generation.y1,Capital_Investment.u1) annotation(Line(visible = true, points = {{90.0,-40.0},{108.0,-40.0}}, color = {191,0,191}));
        connect(FPM.u,POLAT.u) annotation(Line(visible = true, points = {{-21.0,-66.0},{0.0,-66.0},{0.0,-220.0},{-27.0,-220.0}}, color = {0,0,191}));
        connect(CIRA.y,FPCI.u) annotation(Line(visible = true, points = {{265.5,-225.0},{280.0,-225.0},{280.0,-252.0},{-210.0,-252.0},{-210.0,-107.0},{-165.0,-107.0}}, color = {0,0,191}));
        connect(Prod_2_1.u1,Population.y1) annotation(Line(visible = true, points = {{128.8,80.0},{100.0,80.0},{100.0,194.0},{-46.0,194.0},{-46.0,151.0},{-53.0,151.0}}, color = {0,0,191}));
        connect(NRMM.u,DRMM.u) annotation(Line(visible = true, points = {{213.0,80.0},{300.0,80.0},{300.0,171.0},{63.0,171.0}}, color = {0,0,191}));
        connect(NRMM.u,BRMM.u) annotation(Line(visible = true, points = {{213.0,80.0},{300.0,80.0},{300.0,200.0},{-70.0,200.0},{-70.0,185.0},{-93.0,185.0}}, color = {0,0,191}));
        connect(CFIFR.u,QLF.u) annotation(Line(visible = true, points = {{57.0,-230.0},{40.0,-230.0},{40.0,-260.0},{400.0,-260.0},{400.0,-232.0},{393.0,-232.0}}, color = {0,0,191}));
        connect(Food_Ratio.y,QLF.u) annotation(Line(visible = true, points = {{-159.5,-66.0},{-220.0,-66.0},{-220.0,-260.0},{400.0,-260.0},{400.0,-232.0},{393.0,-232.0}}, color = {0,0,191}));
        connect(Pol_Ratio.y,FPM.u) annotation(Line(visible = true, points = {{-18.5,-95.0},{0.0,-95.0},{0.0,-66.0},{-21.0,-66.0}}, color = {0,0,191}));
        connect(CIR.y,POLCM.u) annotation(Line(visible = true, points = {{90.5,6.0},{80.0,6.0},{80.0,-6.0},{10.0,-6.0},{10.0,-116.0},{-100.0,-116.0},{-100.0,-220.0},{-105.0,-220.0}}, color = {0,0,191}));
        connect(QLM.u,CIM.u) annotation(Line(visible = true, points = {{307.0,-169.0},{300.0,-169.0},{300.0,-100.0},{125.0,-100.0}}, color = {0,0,191}));
        connect(QLM.y,QLMF.u1) annotation(Line(visible = true, points = {{335.5,-169.0},{344.0,-169.0},{344.0,-214.0},{333.0,-214.0}}, color = {0,0,191}));
        connect(MSL.y,QLM.u) annotation(Line(visible = true, points = {{287.5,30.0},{300.0,30.0},{300.0,-169.0},{307.0,-169.0}}, color = {0,0,191}));
        connect(NRMM.y,Prod_2_1.u2) annotation(Line(visible = true, points = {{184.5,80.0},{151.2,80.0}}, color = {0,0,191}));
        connect(MSL.y,NRMM.u) annotation(Line(visible = true, points = {{287.5,30.0},{300.0,30.0},{300.0,80.0},{213.0,80.0}}, color = {0,0,191}));
        connect(ECIR.y,MSL.u) annotation(Line(visible = true, points = {{253.5,30.0},{266.5,30.0}}, color = {0,0,191}));
        connect(NotCIAF.u,CIAF.y1) annotation(Line(visible = true, points = {{175.5,10.0},{168.0,10.0},{168.0,-8.0},{240.0,-8.0},{240.0,-110.0},{154.0,-110.0},{154.0,-129.0},{147.0,-129.0}}, color = {0,0,191}));
        connect(Prod_2_2.u1,CIR.u2) annotation(Line(visible = true, points = {{30.2,-100.0},{20.0,-100.0},{20.0,-14.0},{140.0,-14.0},{140.0,-3.6},{125.0,-3.6}}, color = {0,0,191}));
        connect(Prod_2_2.u1,Population.y1) annotation(Line(visible = true, points = {{30.2,-100.0},{20.0,-100.0},{20.0,-14.0},{320.0,-14.0},{320.0,194.0},{-46.0,194.0},{-46.0,151.0},{-53.0,151.0}}, color = {0,0,191}));
        connect(Prod_2_4.u1,Population.y) annotation(Line(visible = true, points = {{-171.2,-220.0},{-200.0,-220.0},{-200.0,164.0},{-94.0,164.0},{-94.0,151.0},{-87.0,151.0}}, color = {0,0,191}));
        connect(CID_norm.u,Capital_Investment.y2) annotation(Line(visible = true, points = {{154.5,-77.0},{145.0,-77.0},{145.0,-53.0}}, color = {0,0,191}));
        connect(CID_norm.y,CI_Discard.u) annotation(Line(visible = true, points = {{175.5,-77.0},{180.0,-77.0},{180.0,-61.0}}, color = {0,0,191}));
        connect(CIM.y,Prod_2_2.u2) annotation(Line(visible = true, points = {{96.5,-100.0},{49.8,-100.0}}, color = {0,0,191}));
        connect(Prod_2_2.y,CIG_norm.u) annotation(Line(visible = true, points = {{40.0,-88.8},{40.0,-77.0},{54.5,-77.0}}, color = {0,0,191}));
        connect(Pol_Ratio.u,Pollution.y1) annotation(Line(visible = true, points = {{-39.5,-95.0},{-46.0,-95.0},{-46.0,-129.0},{-53.0,-129.0}}, color = {0,0,191}));
        connect(POLCM.y,Prod_2_4.u2) annotation(Line(visible = true, points = {{-133.5,-220.0},{-148.8,-220.0}}, color = {0,0,191}));
        connect(Prod_2_4.y,Pol_norm.u) annotation(Line(visible = true, points = {{-160.0,-208.8},{-160.0,-180.0},{-145.5,-180.0}}, color = {0,0,191}));
        connect(Pol_norm.y,P_Generation.u) annotation(Line(visible = true, points = {{-124.5,-180.0},{-120.0,-180.0},{-120.0,-161.0}}, color = {0,0,191}));
        connect(POLAT.y,P_Abs.u2) annotation(Line(visible = true, points = {{-55.5,-220.0},{-70.0,-220.0},{-70.0,-189.6},{-61.0,-189.6}}, color = {0,0,191}));
        connect(P_Abs.y,P_Absorption.u) annotation(Line(visible = true, points = {{-26.5,-180.0},{-20.0,-180.0},{-20.0,-161.0}}, color = {0,0,191}));
        connect(CFIFR.y,Prod_2_3.u1) annotation(Line(visible = true, points = {{85.5,-230.0},{108.8,-230.0}}, color = {0,0,191}));
        connect(CIQR.y,Prod_2_3.u2) annotation(Line(visible = true, points = {{150.5,-230.0},{131.2,-230.0}}, color = {0,0,191}));
        connect(CIAF_G.y,CIAFG.u) annotation(Line(visible = true, points = {{84.5,-180.0},{80.0,-180.0},{80.0,-161.0}}, color = {0,0,191}));
        connect(CIAF_G.u,Prod_2_3.y) annotation(Line(visible = true, points = {{105.5,-180.0},{120.0,-180.0},{120.0,-218.8}}, color = {0,0,191}));
        connect(CIRA.u,Prod_2_5.y) annotation(Line(visible = true, points = {{244.5,-225.0},{229.0,-225.0},{229.0,-211.2}}, color = {0,0,191}));
        connect(Prod_2_5.u1,CIAF.y3) annotation(Line(visible = true, points = {{218.5,-200.0},{130.0,-200.0},{130.0,-153.0}}, color = {0,0,191}));
        connect(CIAF_D.u,CIAF.y2) annotation(Line(visible = true, points = {{152.5,-180.0},{145.0,-180.0},{145.0,-153.0}}, color = {0,0,191}));
        connect(CIAF_D.y,CIAFD.u) annotation(Line(visible = true, points = {{173.5,-180.0},{180.0,-180.0},{180.0,-161.0}}, color = {0,0,191}));
        connect(CIAFD.y1,Sink5.MassInPort1) annotation(Line(visible = true, points = {{190.0,-140.0},{199.0,-140.0}}, color = {191,0,191}));
        connect(CIAF.u2,CIAFD.y) annotation(Line(visible = true, points = {{152.0,-140.0},{170.0,-140.0}}, color = {191,0,191}));
        connect(CIAFG.y1,CIAF.u1) annotation(Line(visible = true, points = {{90.0,-140.0},{108.0,-140.0}}, color = {191,0,191}));
        connect(Source5.MassInPort1,CIAFG.y) annotation(Line(visible = true, points = {{61.0,-140.0},{70.0,-140.0}}, color = {191,0,191}));
        connect(P_Absorption.y1,Sink4.MassInPort1) annotation(Line(visible = true, points = {{-10.0,-140.0},{-1.0,-140.0}}, color = {191,0,191}));
        connect(Pollution.u2,P_Absorption.y) annotation(Line(visible = true, points = {{-48.0,-140.0},{-30.0,-140.0}}, color = {191,0,191}));
        connect(P_Generation.y1,Pollution.u1) annotation(Line(visible = true, points = {{-110.0,-140.0},{-92.0,-140.0}}, color = {191,0,191}));
        connect(Source4.MassInPort1,P_Generation.y) annotation(Line(visible = true, points = {{-139.0,-140.0},{-130.0,-140.0}}, color = {191,0,191}));
        connect(FPCI.y,Prod_3_2.u3) annotation(Line(visible = true, points = {{-136.5,-107.0},{-117.0,-107.0},{-117.0,-76.5}}, color = {0,0,191}));
        connect(FPM.y,Prod_3_2.u2) annotation(Line(visible = true, points = {{-49.5,-66.0},{-107.2,-66.0}}, color = {0,0,191}));
        connect(Prod_3_2.u1,FCM.y) annotation(Line(visible = true, points = {{-117.0,-55.5},{-117.0,-47.0},{-101.5,-47.0}}, color = {0,0,191}));
        connect(Prod_3_2.y,Food_Ratio.u) annotation(Line(visible = true, points = {{-126.8,-66.0},{-138.5,-66.0}}, color = {0,0,191}));
        connect(CIR.u1,Capital_Investment.y1) annotation(Line(visible = true, points = {{125.0,15.6},{154.0,15.6},{154.0,-29.0},{147.0,-29.0}}, color = {0,0,191}));
        connect(NRFR.y,NREM.u) annotation(Line(visible = true, points = {{147.5,51.0},{163.0,51.0}}, color = {0,0,191}));
        connect(DRMM.y,Prod_5_2.u5) annotation(Line(visible = true, points = {{34.5,171.0},{26.0,171.0},{26.0,104.0},{40.0,104.0},{40.0,79.0},{31.2,79.0}}, color = {0,0,191}));
        connect(DR_norm.y,Death_Rate.u) annotation(Line(visible = true, points = {{-13.5,103.0},{-20.0,103.0},{-20.0,119.0}}, color = {0,0,191}));
        connect(Prod_5_2.y,DR_norm.u) annotation(Line(visible = true, points = {{20.0,89.5},{20.0,103.0},{7.5,103.0}}, color = {0,0,191}));
        connect(Prod_5_2.u1,Population.y2) annotation(Line(visible = true, points = {{8.8,79.0},{-55.0,79.0},{-55.0,127.0}}, color = {0,0,191}));
        connect(BRMM.y,Prod_5_1.u1) annotation(Line(visible = true, points = {{-121.5,185.0},{-180.0,185.0},{-180.0,80.0},{-171.2,80.0}}, color = {0,0,191}));
        connect(Prod_5_1.u5,Population.y4) annotation(Line(visible = true, points = {{-148.8,80.0},{-85.0,80.0},{-85.0,127.0}}, color = {0,0,191}));
        connect(Prod_5_1.y,BR_norm.u) annotation(Line(visible = true, points = {{-160.0,91.2},{-160.0,103.0},{-145.5,103.0}}, color = {0,0,191}));
        connect(BR_norm.y,Birth_Rate.u) annotation(Line(visible = true, points = {{-124.5,103.0},{-120.0,103.0},{-120.0,119.0}}, color = {0,0,191}));
        connect(Source1.MassInPort1,Birth_Rate.y) annotation(Line(visible = true, points = {{-139.0,140.0},{-130.0,140.0}}, color = {191,0,191}));
        connect(Birth_Rate.y1,Population.u1) annotation(Line(visible = true, points = {{-110.0,140.0},{-92.0,140.0}}, color = {191,0,191}));
        connect(Population.u2,Death_Rate.y) annotation(Line(visible = true, points = {{-48.0,140.0},{-30.0,140.0}}, color = {191,0,191}));
        connect(Death_Rate.y1,Sink1.MassInPort1) annotation(Line(visible = true, points = {{-10.0,140.0},{-1.0,140.0}}, color = {191,0,191}));
        connect(Source2.MassInPort1,Generation.y) annotation(Line(visible = true, points = {{61.0,140.0},{70.0,140.0}}, color = {191,0,191}));
        connect(Gen_Const.y,Generation.u) annotation(Line(visible = true, points = {{80.0,111.5},{80.0,119.0}}, color = {0,0,191}));
        connect(NRFR.u,Natural_Resources.y4) annotation(Line(visible = true, points = {{126.5,51.0},{115.0,51.0},{115.0,127.0}}, color = {0,0,191}));
        connect(Generation.y1,Natural_Resources.u1) annotation(Line(visible = true, points = {{90.0,140.0},{108.0,140.0}}, color = {191,0,191}));
        connect(Natural_Resources.u2,Depletion.y) annotation(Line(visible = true, points = {{152.0,140.0},{170.0,140.0}}, color = {191,0,191}));
        connect(Depletion.y1,Sink2.MassInPort1) annotation(Line(visible = true, points = {{190.0,140.0},{199.0,140.0}}, color = {191,0,191}));
        Pop = Population.y;
        Pol = Pollution.y;
        Pol_rat = Pol_Ratio.y;
        Cap_inv = Capital_Investment.y;
        Qual_life = Quality_of_Life.y;
        Nat_res = Natural_Resources.y;
        NRUN = if time > 1970 then NRUN2 else 1.0;
        NR_norm.u2 = NRUN;
        rel_dPop = (Birth_Rate.rate - Death_Rate.rate) / Population.level;
        when sample(100, 1) then
                  min_dPop = min([pre(min_dPop),rel_dPop]);
          min_QL = min([pre(min_QL),Quality_of_Life.y]);
        
        end when;
        Perf_Index = min_QL + 5 * min_dPop / dPop_fact;
        annotation(Icon(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-220.0,-264.0},{442.0,220.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-152.0,70.0},{368.0,140.0}}, textString = "5th Modification", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, extent = {{-148.0,-26.0},{372.0,44.0}}, textString = "Optimization", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, extent = {{-150.0,-106.0},{370.0,-36.0}}, textString = "of use of", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, extent = {{-150.0,-192.0},{370.0,-122.0}}, textString = "natural resources", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-62.0,6.0},{-58.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{98.0,192.0},{102.0,196.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{138.0,-16.0},{142.0,-12.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-22.0},{2.0,-18.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-178.0,-2.0},{-174.0,2.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{78.0,4.0},{82.0,8.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-97.0},{2.0,-93.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-178.0,-68.0},{-174.0,-64.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-2.0,-68.0},{2.0,-64.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{158.0,28.0},{162.0,32.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{38.0,-262.0},{42.0,-258.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,-102.0},{302.0,-98.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,28.0},{302.0,32.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,78.0},{302.0,82.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{298.0,169.0},{302.0,173.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{342.0,-171.0},{346.0,-167.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{354.0,-234.0},{358.0,-230.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-97.0,6.0},{-93.0,10.0}}),Ellipse(visible = true, lineColor = {0,0,255}, fillColor = {0,0,255}, fillPattern = FillPattern.Solid, extent = {{-137.0,-22.0},{-133.0,-18.0}})}), experiment(StartTime = 1900, StopTime = 2100), experimentSetupOutput, Documentation(info = "<html>
The 6<sup>th</sup> scenario starts out from the 2<sup>nd</sup> scenario.  We now wish to vary the resource utilization rate after the year 1970. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Cellier, F.E. (1991), <a href=\"http://www.amazon.com/Continuous-System-Modeling-Fran%C3%A7ois-Cellier/dp/0387975020\">Continuous System Modeling</a>, Springer-Verlag, New York, ISBN: 0-387-97502-0, 755p.
<li> Forrester, J.W. (1971), <a href=\"http://www.amazon.com/World-Dynamics-Jay-W-Forrester/dp/1883823382/ref=ed_oe_h/103-2487145-1208659\">World Dynamics</a>, Pegasus Communications, 160p.
</ol> <p>
 
<hr> <p>
 
Simulate the model six times across 200 years while keeping all six trajectories (plot setup menu).  For the six simulation runs, choose different levels of resource utilization after the year 1970: <br>
<font color=red><b>NRUN2 = {0.25, 0.5, 0.75, 1.0, 1.25, 1.5}</b></font>. <p>
 
Compute a performance index: <br>
<font color=red><b>Perf_Index = min_QL + 5*min_dPop;</b></font> <br>
where <b>min_QL</b> is the minimal quality of life observed between 2000 and 2100, and <b>min_dPop</b> is the largest negative population gradient observed in the same time period. <p>
 
Plot the performance index across the calendar years between 2000 and 2100 for all six scenarios on a single graph: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/SD_Fig6.png\"> <p>
 
<b>NRUN2 = 0.25</b> and <b>NRUN2 = 0.5</b> lead to massive die-off, whereas the other scenarios avoid this problem.  However in the short run, those scenarios that offer the worst long-term performance are characterized by the best short-term performance.  This is the predicament that humanity is currently facing. <p>
 
<hr> <p>
</html>", revisions = ""));
      end Scenario_6;
      package Utilities "Utility models of Forrester's WORLD2 model"
        extends Modelica.Icons.Library;
        block Parameter_Change "Parameter variation of WORLD2 model"
          extends Interfaces.Nonlin_2;
        equation
          y = u1 * u2;
        end Parameter_Change;
        annotation(preferedView = "info", Documentation(info = "<html>
Utility models of Forrester's <font color=red><b>WORLD2</b></font> model.
</html>"));
      end Utilities;
      annotation(preferedView = "info", Documentation(info = "<html>
This model implements <a href=\"http://en.wikipedia.org/wiki/Jay_Wright_Forrester\">Jay Forrester's</a> <font color=red><b>WORLD2</b></font> model as described in his 1971 book on <a href=\"http://www.amazon.com/World-Dynamics-Jay-W-Forrester/dp/1883823382/ref=ed_oe_h/103-2487145-1208659\">World Dynamics</a>.  <p>
 
It is a very simply model that contains only five state variables: <br>
 
<ol>
<li>total human population,
<li>total persistent pollution,
<li>remaining non-recoverable natural resources,
<li>total capital investment, and
<li>fraction of capital investment allocated to the agricultural sector.
</ol> <p>
 
The aim of the model is to demonstrate, in very simple terms, that physical systems remain always constrained.  The production of goods (especially food) on this globe is limited by the available resources, and energy constraints will prevent production to grow indefinitely. <p>
 
While these are very simple facts, it is useful to investigate, when our globe will reach its limits.  Forrester showed that this will inevitably happen during the first half of the 21st century.  After that, humanity will invariably have to learn to transform itself from a society of (seemingly perpetual) exponential growth to one of (truly perpetual) stagnation, at least as long as humanity limits itself to the resources available on this one planet. <p>
 
Forrester listed his entire model in his book, which made it easy for other researcher to reproduce his results.  Many people have done so using a variety of different tools.  Whereas the original model had been coded in <font color=red><b>Dynamo</b></font>, a rather clumsy and old-fashioned alphanumerical M&S environment, the most popular tool for coding System Dynamics models today is <font color=red><b>STELLA</b></font>. <p>
 
In this library, we offer a <font color=red><b>Modelica</b></font> implementation of Forrester's <font color=red><b>WORLD2</b></font> model. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Cellier, F.E. (1991), <a href=\"http://www.amazon.com/Continuous-System-Modeling-Fran%C3%A7ois-Cellier/dp/0387975020\">Continuous System Modeling</a>, Springer-Verlag, New York, ISBN: 0-387-97502-0, 755p.
<li> Forrester, J.W. (1971), <a href=\"http://www.amazon.com/World-Dynamics-Jay-W-Forrester/dp/1883823382/ref=ed_oe_h/103-2487145-1208659\">World Dynamics</a>, Pegasus Communications, 160p.
</ol> <p>
 
<hr> <p>
</html>", revisions = ""), Diagram(coordinateSystem(extent = {{-100,-100},{100,100}}), graphics = {Rectangle(extent = {{-100,100},{100,-100}}, fillPattern = FillPattern.Solid, lineColor = {0,0,0}, lineThickness = 0.5, fillColor = {160,160,160}),Text(lineColor = {0,0,255}, extent = {{-80,90},{76,36}}, fillColor = {0,0,0}, lineThickness = 0.5, textString = "WORLD2"),Text(lineColor = {0,0,255}, extent = {{-73.9,25.05},{72,-18}}, fillColor = {0,0,0}, lineThickness = 0.5, textString = "world model as proposed"),Text(lineColor = {0,0,255}, extent = {{-65.8,38.1},{64,12}}, fillColor = {0,0,0}, lineThickness = 0.5, textString = "describes the original"),Text(lineColor = {0,0,255}, extent = {{-77.7,-14.85},{82,-64}}, fillColor = {0,0,0}, lineThickness = 0.5, textString = "his book World Dynamics."),Text(lineColor = {0,0,255}, extent = {{-61.6,-4.8},{56,-31}}, fillColor = {0,0,0}, lineThickness = 0.5, textString = "by Jay Forrester in")}));
    end World2;
    package World3 "Meadows' World Model (2004 Edition)"
      extends Modelica.Icons.Example;
      block Arable_Land_Dynamics "Arable land dynamics"
        parameter Real arable_land_init(unit = "hectare") = 900000000.0 "Initial arable land";
        parameter Real avg_life_land_norm(unit = "yr") = 1000 "Normal life span of land";
        parameter Real inherent_land_fert(unit = "kg/(hectare.yr)") = 600 "Inherent land fertility";
        parameter Real pot_arable_land_init(unit = "hectare") = 2300000000.0 "Initial potential arable land";
        parameter Real pot_arable_land_tot(unit = "hectare") = 3200000000.0 "Total potential arable land";
        parameter Real social_discount(unit = "1/yr") = 0.07 "Social discount";
        parameter Real t_land_life_time(unit = "yr") = 4000 "Land life time";
        parameter Real urban_ind_land_init(unit = "hectare") = 8200000.0 "Initial urban and industrial land";
        parameter Real urb_ind_land_dev_time(unit = "yr") = 10 "Urban and industrial land development time";
        SystemDynamics.Rates.RRate Land_Devel_Rt "p.289 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-27.0,0.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Levels.Level2b Arable_Land(x0 = arable_land_init) "p.279 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {35.0,0.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Rates.RRate Land_Rem_Urb_Ind_Use "p.322 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {106.0,6.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Rates.RRate Land_Erosion_Rt "p.318 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {65.0,-40.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 90)));
        SystemDynamics.Sources.Sink sink annotation(Placement(visible = true, transformation(origin = {65.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
        SystemDynamics.Levels.Level1b Pot_Arable_Land(x0 = pot_arable_land_init) "p.279 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-82.5489,0.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Levels.Level1a Urban_Ind_Land(x0 = urban_ind_land_init) "p.322 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {164.0,6.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Land_Devel_Rt Land_Dev_Rt "p.289 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-27.0,-38.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.WorldDynamics.World3.Utilities.Land_Rem_Urb_Ind_Use Land_Rem_UI_Use(urb_ind_land_dev_time = urb_ind_land_dev_time) "p.322 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {104.0827,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        Modelica.Blocks.Math.Division Land_Er_Rt "p.318 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {30.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Gain Avg_Life_Land(k = avg_life_land_norm) "p.316 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {10.0,-73.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular Dev_Cost_Per_Hect(x_vals = 0.0:0.1:1.0, y_vals = {100000,7400,5200,3500,2400,1500,750,300,150,75,50}) "p.291 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-78.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Gain Gain1(k = 1 / pot_arable_land_tot) "p.291 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-130.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Marg_Prod_Land_Dev Marg_Prod_Land_Dev(social_discount = social_discount) "p.312 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-18.0827,83.8496}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Fr_Inp_Al_Land_Dev(x_vals = 0.0:0.25:2.0, y_vals = {0.0,0.05,0.15,0.3,0.5,0.7,0.85,0.95,1.0}) "p.311 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {67.0,78.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division Div1 "p.311 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {30.0,78.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Gain Land_Fr_Cult(k = 1 / pot_arable_land_tot) "p.278 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {35.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Urb_Ind_Land_Req "p.321 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {100.0,-110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Urb_Ind_Land_PC(x_vals = 0:200:1600, y_vals = {0.005,0.008,0.015,0.025,0.04,0.055,0.07,0.08,0.09}) "p.321 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {60.0,-110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_Land_Life_Mlt_Yield S_Land_Life_Mlt_Yield(t_land_life_time = t_land_life_time) "p.317 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-34.0,-98.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_Land_Life_Mlt_Yield_1(y_vals = {1.2,1.0,0.63,0.36,0.16,0.055,0.04,0.025,0.015,0.01,0.01}, x_vals = {0,1,2,3,4,5,6,7,8,9,50}) "p.317 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-100.0,-76.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_Land_Life_Mlt_Yield_2(y_vals = {1.2,1.0,0.63,0.36,0.29,0.26,0.24,0.22,0.21,0.2,0.2}, x_vals = {0,1,2,3,4,5,6,7,8,9,50}) "p.318 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-100.0,-104.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Gain Gain2(k = 1 / inherent_land_fert) "p.317 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-157.0,-90.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort land_yield(unit = "kg/(hectare.yr)") "Land yield" annotation(Placement(visible = true, transformation(origin = {-210.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort marg_prod_agr_inp(unit = "kg/dollar") "Development of marginally producing land" annotation(Placement(visible = true, transformation(origin = {-210.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort population "Population" annotation(Placement(visible = true, transformation(origin = {-210.0,-130.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort ind_out_pc "Per capita annual industrial output" annotation(Placement(visible = true, transformation(origin = {-210.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort tot_agric_invest(unit = "dollar/yr") "Total investment in the agricultural sector" annotation(Placement(visible = true, transformation(origin = {-210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort arable_land(unit = "hectare") "Arable land" annotation(Placement(visible = true, transformation(origin = {210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort urban_ind_land(unit = "hectare") "Land occupied by human dwellings and industry" annotation(Placement(visible = true, transformation(origin = {210.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort fr_inp_al_land_dev "Fraction of capital invested in agriculture allocated to the devlopment of arable land" annotation(Placement(visible = true, transformation(origin = {210.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      equation
        connect(Fr_Inp_Al_Land_Dev.y,fr_inp_al_land_dev) annotation(Line(visible = true, points = {{78.0,78.0},{180.0,78.0},{180.0,50.0},{210.0,50.0}}, color = {0,0,191}));
        connect(Fr_Inp_Al_Land_Dev.u,Div1.y) annotation(Line(visible = true, origin = {50.0,78.0}, points = {{9.0,0.0},{-9.0,0.0}}, color = {0,0,191}));
        connect(Fr_Inp_Al_Land_Dev.y,Land_Dev_Rt.u2) annotation(Line(visible = true, points = {{78.0,78.0},{100.0,78.0},{100.0,60.0},{-8.0,60.0},{-8.0,-60.0},{-27.0,-60.0},{-27.0,-47.0}}, color = {0,0,191}));
        connect(Div1.u2,marg_prod_agr_inp) annotation(Line(visible = true, points = {{18.0,72.0},{-20.0,72.0},{-20.0,50.0},{-210.0,50.0}}, color = {0,0,191}));
        connect(Marg_Prod_Land_Dev.y,Div1.u1) annotation(Line(visible = true, origin = {9.7293,83.9248}, points = {{-18.812,-0.0752},{5.2707,-0.0752},{5.2707,0.0752},{8.2707,0.0752}}, color = {0,0,191}));
        connect(Marg_Prod_Land_Dev.u1,land_yield) annotation(Line(visible = true, points = {{-27.0827,87.8496},{-46.0,87.8496},{-46.0,110.0},{-210.0,110.0}}, color = {0,0,191}));
        connect(Dev_Cost_Per_Hect.y,Marg_Prod_Land_Dev.u2) annotation(Line(visible = true, origin = {-39.062,79.9248}, points = {{-27.938,0.0752},{7.9793,0.0752},{7.9793,-0.0752},{11.9793,-0.0752}}, color = {0,0,191}));
        connect(Gain1.y,Dev_Cost_Per_Hect.u) annotation(Line(visible = true, origin = {-102.5,80.0}, points = {{-16.5,0.0},{16.5,0.0}}, color = {0,0,191}));
        connect(Dev_Cost_Per_Hect.y,Land_Dev_Rt.u1) annotation(Line(visible = true, points = {{-67.0,80.0},{-46.0,80.0},{-46.0,-60.0},{-33.0,-60.0},{-33.0,-47.0}}, color = {0,0,191}));
        connect(Gain1.u,Pot_Arable_Land.y) annotation(Line(visible = true, points = {{-142.0,80.0},{-170.0,80.0},{-170.0,8.25},{-103.5326,8.25},{-95.2989,8.25}}, color = {0,0,191}));
        connect(Land_Fr_Cult.u,Arable_Land.y) annotation(Line(visible = true, points = {{23.0,40.0},{0.0,40.0},{0.0,8.25},{22.25,8.25}}, color = {0,0,191}));
        connect(arable_land,Arable_Land.y1) annotation(Line(visible = true, points = {{210.0,-10.0},{190.0,-10.0},{190.0,30.0},{70.0,30.0},{70.0,-17.6389},{46.25,-17.6389},{46.25,-9.75}}, color = {0,0,191}));
        connect(Land_Er_Rt.u1,Arable_Land.y2) annotation(Line(visible = true, points = {{18.0,-34.0},{10.0,-34.0},{10.0,-18.0223},{35.0,-18.0223},{35.0,-9.75}}, color = {0,0,191}));
        connect(Land_Rem_Urb_Ind_Use.y,Arable_Land.u2) annotation(Line(visible = true, points = {{98.5,6.0},{53.8,6.0},{51.5,6.0}}, color = {95,0,191}));
        connect(Land_Erosion_Rt.y,Arable_Land.u3) annotation(Line(visible = true, points = {{65.0,-47.5},{65.0,-6.4},{51.5,-6.0}}, color = {95,0,191}));
        connect(Land_Devel_Rt.y1,Arable_Land.u1) annotation(Line(visible = true, points = {{-19.5,0.0},{-10.0,0.0},{-10.0,0.0},{18.5,0.0}}, color = {95,0,191}));
        connect(Land_Dev_Rt.y,Land_Devel_Rt.u) annotation(Line(visible = true, origin = {-27.0,-22.375}, points = {{0.0,-6.625},{0.0,6.625}}, color = {0,0,191}));
        connect(Pot_Arable_Land.u1,Land_Devel_Rt.y) annotation(Line(visible = true, points = {{-66.0489,0.0},{-50.0,0.0},{-50.0,0.0},{-34.5,0.0}}, color = {95,0,191}));
        connect(P_Land_Life_Mlt_Yield_1.y,S_Land_Life_Mlt_Yield.u1) annotation(Line(visible = true, points = {{-89.0,-76.0},{-64.0,-76.0},{-64.0,-92.0},{-64.0,-94.0},{-60.0,-94.0},{-43.0,-94.0}}, color = {0,0,191}));
        connect(P_Land_Life_Mlt_Yield_2.y,S_Land_Life_Mlt_Yield.u2) annotation(Line(visible = true, points = {{-89.0,-104.0},{-64.0,-104.0},{-64.0,-102.0},{-43.0,-102.0}}, color = {0,0,191}));
        connect(Gain2.u,land_yield) annotation(Line(visible = true, points = {{-169.0,-90.0},{-186.0,-90.0},{-186.0,110.0},{-210.0,110.0}}, color = {0,0,191}));
        connect(Gain2.y,P_Land_Life_Mlt_Yield_2.u) annotation(Line(visible = true, points = {{-146.0,-90.0},{-130.0,-90.0},{-130.0,-104.0},{-108.0,-104.0}}, color = {0,0,191}));
        connect(Gain2.y,P_Land_Life_Mlt_Yield_1.u) annotation(Line(visible = true, points = {{-146.0,-90.0},{-130.0,-90.0},{-130.0,-76.0},{-108.0,-76.0}}, color = {0,0,191}));
        connect(S_Land_Life_Mlt_Yield.y,Avg_Life_Land.u) annotation(Line(visible = true, points = {{-25.0,-98.0},{10.0,-98.0},{10.0,-61.0}}, color = {0,0,191}));
        connect(Urb_Ind_Land_PC.u,ind_out_pc) annotation(Line(visible = true, points = {{52.0,-110.0},{20.0,-110.0},{20.0,-120.0},{-190.0,-120.0},{-190.0,-70.0},{-210.0,-70.0}}, color = {0,0,191}));
        connect(Urb_Ind_Land_PC.y,Urb_Ind_Land_Req.u1) annotation(Line(visible = true, origin = {82.0,-110.0}, points = {{-11.0,0.0},{11.0,0.0}}, color = {0,0,191}));
        connect(Urb_Ind_Land_Req.u2,population) annotation(Line(visible = true, points = {{107.0,-110.0},{120.0,-110.0},{120.0,-130.0},{-210.0,-130.0}}, color = {0,0,191}));
        connect(Urb_Ind_Land_Req.y,Land_Rem_UI_Use.u1) annotation(Line(visible = true, points = {{100.0,-103.0},{100.0,-54.4},{100.0827,-49.0}}, color = {0,0,191}));
        connect(sink.MassInPort1,Land_Erosion_Rt.y1) annotation(Line(visible = true, points = {{65.0,-59.0},{65.0,-32.5}}, color = {95,0,191}));
        connect(Land_Er_Rt.u2,Avg_Life_Land.y) annotation(Line(visible = true, points = {{18.0,-46.0},{10.0,-46.0},{10.0,-84.0}}, color = {0,0,191}));
        connect(urban_ind_land,Urban_Ind_Land.y2) annotation(Line(visible = true, points = {{210.0,-70.0},{175.25,-70.0},{175.25,-3.75}}, color = {0,0,191}));
        connect(Land_Rem_UI_Use.u2,Urban_Ind_Land.y4) annotation(Line(visible = true, points = {{108.0827,-49.0},{108.0827,-70.0},{152.75,-70.0},{152.75,-3.75}}, color = {0,0,191}));
        connect(Urban_Ind_Land.u1,Land_Rem_Urb_Ind_Use.y1) annotation(Line(visible = true, origin = {130.5,6.0}, points = {{17.0,0.0},{-17.0,0.0}}, color = {95,0,191}));
        connect(Land_Rem_UI_Use.y,Land_Rem_Urb_Ind_Use.u) annotation(Line(visible = true, origin = {105.0414,-16.5625}, points = {{-0.9586,-14.4375},{-0.9586,3.8125},{0.9586,3.8125},{0.9586,6.8125}}, color = {0,0,191}));
        connect(Land_Er_Rt.y,Land_Erosion_Rt.u) annotation(Line(visible = true, points = {{41.0,-40.0},{80.75,-40.0}}, color = {0,0,191}));
        connect(tot_agric_invest,Land_Dev_Rt.u3) annotation(Line(visible = true, points = {{-210.0,-10.0},{-170.0,-10.0},{-170.0,-50.0},{-60.0,-50.0},{-60.0,-70.0},{-21.0,-70.0},{-21.0,-47.0}}, color = {0,0,191}));
        annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, fillColor = {170,255,170}, fillPattern = FillPattern.Solid, lineThickness = 1.0, extent = {{-100.0,-100.0},{100.0,100.0}}, radius = 0.5),Text(visible = true, origin = {0.0,50.0}, fillPattern = FillPattern.Solid, extent = {{-73.6862,-36.6918},{73.6862,36.6918}}, textString = "Arable", fontSize = 102.0, fontName = "Arial"),Text(visible = true, origin = {0.0,-50.0}, fillPattern = FillPattern.Solid, extent = {{-87.9194,-20.8986},{87.9194,20.8986}}, textString = "Dynamics", fontSize = 102.0, fontName = "Arial"),Text(visible = true, fillPattern = FillPattern.Solid, extent = {{-72.1931,-36.6918},{72.1931,36.6918}}, textString = "Land", fontSize = 102.0, fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-200.0,-160.0},{200.0,140.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-47.0,79.0},{-45.0,81.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-131.0,-91.0},{-129.0,-89.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-187.0,109.0},{-185.0,111.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{99.0,77.0},{101.0,79.0}})}), Documentation(info = "<html>
This model quantifies the land that is being used to produce food, i.e., the arable land.  The model contains three state variables, representing the potential arable land, i.e., land that could be developed to grow crop, but so far hasn't been developed; the arable land that is currently being used for agriculture; and finally, the land that is being used for urban development and industry (including mining).  Some of the potential arable land is getting developed over time, whereas some of the arable land is being lost either to urban development or to erosion. <p>
 
The arable land is measured in units of hectares.
</html>"));
      end Arable_Land_Dynamics;
      block Food_Production "Food production"
        parameter Real agr_inp_init(unit = "dollar/yr") = 5000000000.0 "Initial agricultural input";
        parameter Real food_short_perc_del(unit = "yr") = 2 "Food shortage perception delay";
        parameter Real land_fr_harvested = 0.7 "Land fraction harvested";
        parameter Real p_avg_life_agr_inp_1(unit = "yr") = 2 "Default average life of agricultural input";
        parameter Real p_avg_life_agr_inp_2(unit = "yr") = 2 "Controlled average life of agricultural input";
        parameter Real p_land_yield_fact_1 = 1 "Default land yield factor";
        parameter Real perc_food_ratio_init = 1 "Initial perceived food ratio";
        parameter Real processing_loss = 0.1 "Processing loss";
        parameter Real subsist_food_pc(unit = "kg/yr") = 230 "Available per capita food";
        parameter Real t_policy_year(unit = "yr") = 4000 "Year of policy change";
        parameter Real tech_dev_del_TDD(unit = "yr") = 20 "Technology development time";
        SystemDynamics.Interfaces.MassInPort ind_out_pc(unit = "dollar/yr") "Per capita annual industrial output" annotation(Placement(visible = true, transformation(origin = {-210.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_Indic_Food_PC_1(x_vals = 0:200:1600, y_vals = {230,480,690,850,970,1070,1150,1210,1250}) "p.286 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-142.0,-90.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_Indic_Food_PC_2(x_vals = 0:200:1600, y_vals = {230,480,690,850,970,1070,1150,1210,1250}) "p.286 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-142.0,-130.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_Indic_Food_PC S_Indic_Food_PC(t_policy_year = t_policy_year) "p.286 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-79.0,-110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_Fr_IO_Al_Agr_1(x_vals = 0:0.5:2.5, y_vals = {0.4,0.2,0.1,0.025,0,0}) "p.287 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {18.0,-90.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division Div1 annotation(Placement(visible = true, transformation(origin = {-30.0,-104.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_Fr_IO_Al_Agr_2(x_vals = 0:0.5:2.5, y_vals = {0.4,0.2,0.1,0.025,0,0}) "p.287 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {18.0,-130.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_FIOA_Agr S_FIOA_Agr(t_policy_year = t_policy_year) "p.287 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {76.0,-110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Tot_Agric_Invest "p.287 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {120.0,-110.0}, extent = {{-12.0,-12.0},{12.0,12.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort industrial_output(unit = "dollar/yr") "Annual industrial output" annotation(Placement(visible = true, transformation(origin = {-210.0,-130.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {-150.0,10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Chg_Agr_Inp annotation(Placement(visible = true, transformation(origin = {-100.0,10.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Levels.Level1a Agr_Inp(x0 = agr_inp_init) annotation(Placement(visible = true, transformation(origin = {-40.0,10.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Ch_Agr_Inp Ch_Agr_Inp "p.292 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-100.0,-26.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.WorldDynamics.World3.Utilities.Current_Agr_Inp Current_Agr_Inp "p.292 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-141.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort fr_inp_al_land_dev "Fraction of capital invested in agriculture allocated to the devlopment of arable land" annotation(Placement(visible = true, transformation(origin = {-210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_Avg_Life_Agr_Inp S_Avg_Life_Agr_Inp(t_policy_year = t_policy_year, p_avg_life_agr_inp_1 = p_avg_life_agr_inp_1, p_avg_life_agr_inp_2 = p_avg_life_agr_inp_2) "p.293 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {0.0,-50.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_Fr_Inp_For_Land_Maint(x_vals = 0:4, y_vals = {0,0.04,0.07,0.09,0.1}) "p.332 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-126.5,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source2 annotation(Placement(visible = true, transformation(origin = {-40.0,90.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Chg_Perc_Food_Ratio annotation(Placement(visible = true, transformation(origin = {10.0,90.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Levels.Level1a Perc_Food_Ratio(x0 = perc_food_ratio_init) annotation(Placement(visible = true, transformation(origin = {60.0,90.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Agr_Inp_Per_Hect Agr_Inp_Per_Hect "p.294 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-53.3528,44.0583}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort arable_land(unit = "hectare") "Arable land" annotation(Placement(visible = true, transformation(origin = {-210.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Marg_Land_Yield_Mlt_Cap(y_vals = {0.075,0.03,0.015,0.011,0.009,0.008,0.007,0.006,0.005,0.005,0.005,0.005,0.005,0.005,0.005,0.005,0.005}, x_vals = {0,40,80,120,160,200,240,280,320,360,400,440,480,520,560,600,10000}) "p.313 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {23.5,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Land_Yield_Mlt_Cap(y_vals = {1,3,4.5,5,5.3,5.6,5.9,6.1,6.35,6.6,6.9,7.2,7.4,7.6,7.8,8,8.2,8.4,8.6,8.8,9,9.2,9.4,9.6,9.8,10,10}, x_vals = {0,40,80,120,160,200,240,280,320,360,400,440,480,520,560,600,640,680,720,760,800,840,880,920,960,1000,10000}) "p.306 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {21.5,20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_4 Land_Yield "p.307 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {90.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
        SystemDynamics.WorldDynamics.World3.Utilities.Marg_Prod_Agr_Inp Marg_Prod_Agr_Inp "p.313 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {160.0,-5.9417}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort marg_prod_agr_inp(unit = "kg/dollar") "Development of marginally producing land" annotation(Placement(visible = true, transformation(origin = {210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort land_yield(unit = "kg/(hectare.yr)") "Land yield" annotation(Placement(visible = true, transformation(origin = {210.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Food Food(land_fr_harvested = land_fr_harvested, processing_loss = processing_loss) "p.280 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-158.0,100.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division Food_PC "p.281 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-126.0,94.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort population "Population" annotation(Placement(visible = true, transformation(origin = {-210.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Gain Food_Ratio(k = 1 / subsist_food_pc) "p.332 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-83.0,94.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Ch_Perc_Food_Ratio Ch_Perc_Food_Ratio(food_short_perc_del = food_short_perc_del) "p.333 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {121.825,106.9111}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_Land_Yield_Fact S_Land_Yield_Fact(t_policy_year = t_policy_year, p_land_yield_fact_1 = p_land_yield_fact_1) "p.307 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {115.0,50.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.SMTH3 P_Land_Yield_Fact_2(averaging_time(unit = "yr") = tech_dev_del_TDD) annotation(Placement(visible = true, transformation(origin = {157.0,50.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort yield_tech_LYTD "Technology induced absolute yield enhancement" annotation(Placement(visible = true, transformation(origin = {-170.0,150.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 90), iconTransformation(origin = {-60.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
        SystemDynamics.Interfaces.MassInPort land_fertility(unit = "kg/(hectare.yr)") "Land fertility" annotation(Placement(visible = true, transformation(origin = {-170.0,-170.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90), iconTransformation(origin = {-60.0,-110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Interfaces.MassInPort s_yield_mlt_air_poll "Land yield multiplier from air pollution" annotation(Placement(visible = true, transformation(origin = {-110.0,-170.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90), iconTransformation(origin = {-20.0,-110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Interfaces.MassOutPort food_ratio "Food ratio" annotation(Placement(visible = true, transformation(origin = {210.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort tot_agric_invest(unit = "dollar/yr") "Total investment in the agricultural sector" annotation(Placement(visible = true, transformation(origin = {170.0,150.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90), iconTransformation(origin = {60.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Interfaces.MassOutPort agr_inp_per_hect(unit = "dollar/(hectare.yr)") "Agricultural investments per hectare" annotation(Placement(visible = true, transformation(origin = {210.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort s_fioa_agr "Fraction of total investments allocated to the agricultural sector" annotation(Placement(visible = true, transformation(origin = {210.0,-130.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort p_fr_inp_for_land_maint "Fraction of investments in the agricultural sector allocated to land maintenance" annotation(Placement(visible = true, transformation(origin = {170.0,-170.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 90), iconTransformation(origin = {60.0,-110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular Yield_Tech_Mult_Icor_COYM(y_vals = {1,1,1.05,1.12,1.25,1.35,1.5,1.5}, x_vals = {0,1,1.2,1.4,1.6,1.8,2,50}) annotation(Placement(visible = true, transformation(origin = {50.0,50.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort yield_tech_mult_icor_COYM "Technology induced relative enhancement of land yield" annotation(Placement(visible = true, transformation(origin = {110.0,-170.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 90), iconTransformation(origin = {20.0,-110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
        SystemDynamics.Interfaces.MassOutPort food_pc(unit = "kg/yr") "Per capita annually consumed food" annotation(Placement(visible = true, transformation(origin = {110.0,150.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90), iconTransformation(origin = {20.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
      equation
        connect(Marg_Prod_Agr_Inp.y,marg_prod_agr_inp) annotation(Line(visible = true, points = {{169.0,-5.9417},{190.0,-5.9417},{190.0,-10.0},{210.0,-10.0}}, color = {0,0,191}));
        connect(S_FIOA_Agr.u1,P_Fr_IO_Al_Agr_1.y) annotation(Line(visible = true, points = {{67.0,-106.0},{54.0,-106.0},{54.0,-90.0},{29.0,-90.0}}, color = {0,0,191}));
        connect(S_FIOA_Agr.u2,P_Fr_IO_Al_Agr_2.y) annotation(Line(visible = true, points = {{67.0,-114.0},{54.0,-114.0},{54.0,-130.0},{29.0,-130.0}}, color = {0,0,191}));
        connect(S_Indic_Food_PC.u2,P_Indic_Food_PC_2.y) annotation(Line(visible = true, points = {{-88.0,-114.0},{-106.0,-114.0},{-106.0,-130.0},{-131.0,-130.0}}, color = {0,0,191}));
        connect(S_Indic_Food_PC.u1,P_Indic_Food_PC_1.y) annotation(Line(visible = true, points = {{-88.0,-106.0},{-106.0,-106.0},{-106.0,-90.0},{-131.0,-90.0}}, color = {0,0,191}));
        connect(Tot_Agric_Invest.y,Current_Agr_Inp.u2) annotation(Line(visible = true, points = {{120.0,-101.6},{120.0,-64.0},{-168.0,-64.0},{-168.0,-54.0},{-150.0,-54.0}}, color = {0,0,191}));
        connect(Current_Agr_Inp.u1,fr_inp_al_land_dev) annotation(Line(visible = true, points = {{-150.0,-46.0},{-180.0,-46.0},{-180.0,-10.0},{-210.0,-10.0}}, color = {0,0,191}));
        connect(Ch_Agr_Inp.u3,Agr_Inp.y4) annotation(Line(visible = true, points = {{-94.0,-35.0},{-94.0,-50.0},{-51.25,-50.0},{-51.25,-5.9972},{-51.25,0.25}}, color = {0,0,191}));
        connect(Agr_Inp_Per_Hect.u3,Agr_Inp.y) annotation(Line(visible = true, points = {{-62.3528,38.0583},{-76.0,38.0583},{-76.0,20.0},{-76.0,18.25},{-52.75,18.25}}, color = {0,0,191}));
        connect(Marg_Land_Yield_Mlt_Cap.u,Agr_Inp_Per_Hect.y) annotation(Line(visible = true, points = {{15.5,-10.0},{-10.0,-10.0},{-10.0,44.0583},{-44.3528,44.0583}}, color = {0,0,191}));
        connect(Land_Yield_Mlt_Cap.u,Agr_Inp_Per_Hect.y) annotation(Line(visible = true, points = {{13.5,20.0},{-10.0,20.0},{-10.0,44.0583},{-44.3528,44.0583}}, color = {0,0,191}));
        connect(arable_land,Agr_Inp_Per_Hect.u2) annotation(Line(visible = true, points = {{-210.0,50.0},{-180.0,50.0},{-180.0,30.0},{-90.0,30.0},{-90.0,44.0583},{-62.3528,44.0583}}, color = {0,0,191}));
        connect(Agr_Inp_Per_Hect.u1,P_Fr_Inp_For_Land_Maint.y) annotation(Line(visible = true, points = {{-62.3528,50.0583},{-78.95,50.2},{-78.95,50.0},{-115.5,50.0}}, color = {0,0,191}));
        connect(S_FIOA_Agr.y,s_fioa_agr) annotation(Line(visible = true, points = {{85.0,-110.0},{100.0,-110.0},{100.0,-130.0},{210.0,-130.0}}, color = {0,0,191}));
        connect(Tot_Agric_Invest.u1,S_FIOA_Agr.y) annotation(Line(visible = true, origin = {98.3,-110.0}, points = {{13.3,0.0},{-13.3,0.0}}, color = {0,0,191}));
        connect(P_Fr_IO_Al_Agr_2.u,Div1.y) annotation(Line(visible = true, points = {{10.0,-130.0},{-10.0,-130.0},{-10.0,-104.0},{-19.0,-104.0}}, color = {0,0,191}));
        connect(P_Fr_IO_Al_Agr_1.u,Div1.y) annotation(Line(visible = true, points = {{10.0,-90.0},{-10.0,-90.0},{-10.0,-104.0},{-19.0,-104.0}}, color = {0,0,191}));
        connect(Div1.u2,S_Indic_Food_PC.y) annotation(Line(visible = true, origin = {-56.0,-110.0}, points = {{14.0,0.0},{-14.0,0.0}}, color = {0,0,191}));
        connect(P_Indic_Food_PC_2.u,ind_out_pc) annotation(Line(visible = true, points = {{-150.0,-130.0},{-180.0,-130.0},{-180.0,-70.0},{-210.0,-70.0}}, color = {0,0,191}));
        connect(P_Indic_Food_PC_1.u,ind_out_pc) annotation(Line(visible = true, points = {{-150.0,-90.0},{-180.0,-90.0},{-180.0,-70.0},{-210.0,-70.0}}, color = {0,0,191}));
        connect(S_Avg_Life_Agr_Inp.y,Marg_Prod_Agr_Inp.u4) annotation(Line(visible = true, points = {{-9.0,-50.0},{-40.0,-50.0},{-40.0,-30.0},{110.0,-30.0},{110.0,-11.9417},{151.0,-11.9417}}, color = {0,0,191}));
        connect(S_Avg_Life_Agr_Inp.y,Ch_Agr_Inp.u2) annotation(Line(visible = true, points = {{-9.0,-50.0},{-40.0,-50.0},{-40.0,-60.0},{-100.0,-60.0},{-100.0,-35.0}}, color = {0,0,191}));
        connect(Current_Agr_Inp.y,Ch_Agr_Inp.u1) annotation(Line(visible = true, origin = {-114.6667,-45.0}, points = {{-17.3333,-5.0},{8.6667,-5.0},{8.6667,10.0}}, color = {0,0,191}));
        connect(Ch_Agr_Inp.y,Chg_Agr_Inp.u) annotation(Line(visible = true, origin = {-100.0,-11.375}, points = {{0.0,-5.625},{0.0,5.625}}, color = {0,0,191}));
        connect(Source1.MassInPort1,Chg_Agr_Inp.y) annotation(Line(visible = true, origin = {-123.25,10.0}, points = {{-15.75,0.0},{15.75,0.0}}, color = {95,0,191}));
        connect(Chg_Agr_Inp.y1,Agr_Inp.u1) annotation(Line(visible = true, origin = {-74.5,10.0}, points = {{-18.0,0.0},{18.0,0.0}}, color = {95,0,191}));
        connect(Marg_Land_Yield_Mlt_Cap.u,agr_inp_per_hect) annotation(Line(visible = true, points = {{15.5,-10.0},{-10.0,-10.0},{-10.0,-26.0},{166.0,-26.0},{166.0,-70.0},{210.0,-70.0}}, color = {0,0,191}));
        connect(Marg_Land_Yield_Mlt_Cap.y,Marg_Prod_Agr_Inp.u3) annotation(Line(visible = true, points = {{34.5,-10.0},{98.675,-10.0},{98.675,-7.9417},{151.0,-7.9417}}, color = {0,0,191}));
        connect(Land_Yield.y,Marg_Prod_Agr_Inp.u1) annotation(Line(visible = true, points = {{97.0,30.0},{130.0,30.0},{130.0,0.0},{151.0,0.0583}}, color = {0,0,191}));
        connect(Land_Yield_Mlt_Cap.y,Marg_Prod_Agr_Inp.u2) annotation(Line(visible = true, points = {{32.5,20.0},{64.0,20.0},{64.0,-4.0},{151.0,-3.9417}}, color = {0,0,191}));
        connect(Land_Yield.u3,s_yield_mlt_air_poll) annotation(Line(visible = true, points = {{84.4,26.0},{76.0,26.0},{76.0,-40.0},{160.0,-40.0},{160.0,-152.0},{-110.0,-152.0},{-110.0,-170.0}}, color = {0,0,191}));
        connect(Land_Yield.u2,land_fertility) annotation(Line(visible = true, points = {{84.4,34.0},{70.0,34.0},{70.0,-46.0},{154.0,-46.0},{154.0,-148.0},{-170.0,-148.0},{-170.0,-170.0}}, color = {0,0,191}));
        connect(Ch_Perc_Food_Ratio.y,Chg_Perc_Food_Ratio.u) annotation(Line(visible = true, points = {{130.825,106.9111},{160.0,106.9111},{160.0,68.0},{10.0,68.0},{10.0,74.25}}, color = {0,0,191}));
        connect(Perc_Food_Ratio.y1,Ch_Perc_Food_Ratio.u2) annotation(Line(visible = true, points = {{72.75,98.25},{90.0,98.25},{90.0,102.9111},{112.825,102.9111}}, color = {95,0,191}));
        connect(Food_Ratio.y,Ch_Perc_Food_Ratio.u1) annotation(Line(visible = true, points = {{-72.0,94.0},{-60.0,94.0},{-60.0,111.0},{90.0,111.0},{95.9556,111.0},{112.825,110.9111}}, color = {0,0,191}));
        connect(P_Fr_Inp_For_Land_Maint.u,Perc_Food_Ratio.y) annotation(Line(visible = true, points = {{-134.5,50.0},{-162.0,50.0},{-162.0,72.0},{-52.0,72.0},{-52.0,106.0},{40.0,106.0},{40.0,98.25},{47.25,98.25}}, color = {0,0,191}));
        connect(Chg_Perc_Food_Ratio.y1,Perc_Food_Ratio.u1) annotation(Line(visible = true, origin = {30.5,90.0}, points = {{-13.0,0.0},{13.0,0.0}}, color = {95,0,191}));
        connect(Land_Yield.y,Food.u1) annotation(Line(visible = true, points = {{97.0,30.0},{180.0,30.0},{180.0,124.0},{-180.0,124.0},{-180.0,104.0},{-167.0,104.0}}, color = {0,0,191}));
        connect(Food.u2,arable_land) annotation(Line(visible = true, points = {{-167.0,96.0},{-180.0,96.0},{-180.0,50.0},{-210.0,50.0}}, color = {0,0,191}));
        connect(Land_Yield.u4,Land_Yield_Mlt_Cap.y) annotation(Line(visible = true, points = {{90.0,23.0},{90.0,-4.0},{64.0,-4.0},{64.0,20.0},{32.5,20.0}}, color = {0,0,191}));
        connect(S_Land_Yield_Fact.y,Land_Yield.u1) annotation(Line(visible = true, points = {{106.0,50.0},{90.0,50.0},{90.0,37.0}}, color = {0,0,191}));
        connect(Land_Yield.y,land_yield) annotation(Line(visible = true, points = {{97.0,30.0},{180.0,30.0},{180.0,50.0},{210.0,50.0}}, color = {0,0,191}));
        connect(P_Land_Yield_Fact_2.u,yield_tech_LYTD) annotation(Line(visible = true, points = {{168.0,50.0},{176.0,50.0},{176.0,128.0},{-170.0,128.0},{-170.0,150.0}}, color = {0,0,191}));
        connect(P_Land_Yield_Fact_2.y,S_Land_Yield_Fact.u) annotation(Line(visible = true, origin = {135.0,50.0}, points = {{11.0,0.0},{-11.0,0.0}}, color = {0,0,191}));
        connect(S_Land_Yield_Fact.y,Yield_Tech_Mult_Icor_COYM.u) annotation(Line(visible = true, origin = {82.0,50.0}, points = {{24.0,-0.0},{-24.0,0.0}}, color = {0,0,191}));
        connect(Yield_Tech_Mult_Icor_COYM.y,yield_tech_mult_icor_COYM) annotation(Line(visible = true, points = {{39.0,50.0},{-14.0,50.0},{-14.0,-24.0},{106.0,-24.0},{106.0,-140.0},{110.0,-140.0},{110.0,-170.0}}, color = {0,0,191}));
        connect(P_Fr_Inp_For_Land_Maint.y,p_fr_inp_for_land_maint) annotation(Line(visible = true, points = {{-115.5,50.0},{-80.0,50.0},{-80.0,-56.0},{-50.0,-56.0},{-50.0,-156.0},{170.0,-156.0},{170.0,-170.0}}, color = {0,0,191}));
        connect(Source2.MassInPort1,Chg_Perc_Food_Ratio.y) annotation(Line(visible = true, origin = {-13.25,90.0}, points = {{-15.75,0.0},{15.75,0.0}}, color = {95,0,191}));
        connect(Food_Ratio.y,food_ratio) annotation(Line(visible = true, points = {{-72.0,94.0},{-60.0,94.0},{-60.0,132.0},{184.0,132.0},{184.0,110.0},{210.0,110.0}}, color = {0,0,191}));
        connect(Food_Ratio.u,Food_PC.y) annotation(Line(visible = true, origin = {-105.0,94.0}, points = {{10.0,0.0},{-10.0,0.0}}, color = {0,0,191}));
        connect(Food_PC.u1,Food.y) annotation(Line(visible = true, origin = {-143.5,100.0}, points = {{5.5,0.0},{-5.5,0.0}}, color = {0,0,191}));
        connect(Tot_Agric_Invest.u2,industrial_output) annotation(Line(points = {{128.4,-110.0},{140.0,-110.0},{140.0,-144.0},{-190.0,-144.0},{-190.0,-130.0},{-210.0,-130.0}}, color = {0,0,191}, visible = true));
        connect(population,Food_PC.u2) annotation(Line(points = {{-210.0,110.0},{-190.0,110.0},{-190.0,80.0},{-146.0,80.0},{-146.0,88.0},{-138.0,88.0}}, color = {0,0,191}, visible = true));
        connect(Food_PC.y,Div1.u1) annotation(Line(points = {{-115.0,94.0},{-110.0,94.0},{-110.0,76.0},{-174.0,76.0},{-174.0,-68.0},{-46.0,-68.0},{-46.0,-98.0},{-42.0,-98.0}}, color = {0,0,191}, visible = true));
        connect(Food_PC.y,food_pc) annotation(Line(points = {{-115.0,94.0},{-110.0,94.0},{-110.0,136.0},{110.0,136.0},{110.0,150.0}}, color = {0,0,191}, visible = true));
        connect(Tot_Agric_Invest.y,tot_agric_invest) annotation(Line(points = {{120.0,-101.6},{120.0,-64.0},{194.0,-64.0},{194.0,136.0},{170.0,136.0},{170.0,150.0}}, color = {0,0,191}, visible = true));
        annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, fillColor = {170,255,170}, fillPattern = FillPattern.Solid, lineThickness = 1.0, extent = {{-100.0,-100.0},{100.0,100.0}}, radius = 0.5),Text(visible = true, origin = {0.0,30.0}, fillPattern = FillPattern.Solid, extent = {{-47.3724,-36.6918},{47.3724,36.6918}}, textString = "Food", fontSize = 102.0, fontName = "Arial"),Text(visible = true, origin = {0.0,-30.0}, fillPattern = FillPattern.Solid, extent = {{-95.13,-24.9532},{95.13,24.9532}}, textString = "Production", fontSize = 102.0, fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-200.0,-160.0},{200.0,140.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-181.0,-91.0},{-179.0,-89.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-11.0,-105.0},{-9.0,-103.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-11.0,19.0},{-9.0,21.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-41.0,-51.0},{-39.0,-49.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{89.0,-5.0},{91.0,-3.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-181.0,49.0},{-179.0,51.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-111.0,93.0},{-109.0,95.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{179.0,49.0},{181.0,51.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{129.0,29.0},{131.0,31.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-61.0,110.0},{-59.0,112.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{99.0,-111.0},{101.0,-109.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-81.0,49.0},{-79.0,51.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{89.0,49.0},{91.0,51.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{119.0,-65.0},{121.0,-63.0}})}), Documentation(info = "<html>
This model describes the production of food.  Food is a controlled variable, as farmers decide what to grow and how much to grow on the basis of their perception of what they can sell.  Consequently, the model uses the perceived food ratio as a state variable.  If the farmers decide that too much food is being produced already, they'll produce less next year, and vice versa. <p>
 
A second state variable describes the capital that is being invested in growing food.  Since that capital (for example, the capital expended in buying food harvesting machines) cannot be allocated or freed instantaneously, it makes sense to maintain also that variable as a state variable. <p>
 
The agricultural input is measured in dollars/year, whereas the food ratio is dimensionless.
</html>"));
      end Food_Production;
      block Human_Ecological_Footprint "Ecological footprint"
        SystemDynamics.Interfaces.MassInPort urban_ind_land(unit = "hectare") "Land occupied by human dwellings and industry" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort arable_land(unit = "hectare") "Arable land" annotation(Placement(visible = true, transformation(origin = {-110.0,64.2333}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort ppoll_gen_rt(unit = "1/yr") "Persistent pollution generation rate" annotation(Placement(visible = true, transformation(origin = {-110.0,-58.4806}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Gain Arable_Land_In_Gha(k = 0.000000001) annotation(Placement(visible = true, transformation(origin = {-31.0,64.2333}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Gain Urban_Land_In_Gha(k = 0.000000001) annotation(Placement(visible = true, transformation(origin = {-27.1694,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Absorption_Land_In_Gha annotation(Placement(visible = true, transformation(origin = {-20.0,-57.775}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Const Gha_Per_Unit_Of_Pollution(k = 0.000000004) annotation(Placement(visible = true, transformation(origin = {68.0,-68.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.HEF_Human_Ecological_Footprint HEF_Human_Ecological_Footprint annotation(Placement(visible = true, transformation(origin = {20.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort hef_human_ecological_footprint(unit = "hectare") "Ecological footprint" annotation(Placement(visible = true, transformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      equation
        connect(Gha_Per_Unit_Of_Pollution.y,Absorption_Land_In_Gha.u2) annotation(Line(visible = true, origin = {34.7556,-45.43}, points = {{33.2444,-15.57},{33.2444,12.43},{-10.7667,12.43},{-10.7667,-12.345},{-47.7556,-12.345}}, color = {0,0,191}));
        connect(Absorption_Land_In_Gha.y,HEF_Human_Ecological_Footprint.u3) annotation(Line(visible = true, points = {{-20.0,-50.775},{-20.0,-40.0},{0.0,-40.0},{0.0,-6.0},{11.0,-6.0}}, color = {0,0,191}));
        connect(Absorption_Land_In_Gha.u1,ppoll_gen_rt) annotation(Line(visible = true, origin = {-82.6375,-58.1278}, points = {{55.6375,0.3528},{-14.1375,0.3528},{-14.1375,-0.3528},{-27.3625,-0.3528}}, color = {0,0,191}));
        connect(Urban_Land_In_Gha.y,HEF_Human_Ecological_Footprint.u2) annotation(Line(visible = true, origin = {-2.5847,0.0}, points = {{-13.5847,0.0},{13.5847,0.0}}, color = {0,0,191}));
        connect(Urban_Land_In_Gha.u,urban_ind_land) annotation(Line(visible = true, origin = {-74.5847,0.0}, points = {{35.4153,0.0},{-35.4153,0.0}}, color = {0,0,191}));
        connect(Arable_Land_In_Gha.y,HEF_Human_Ecological_Footprint.u1) annotation(Line(visible = true, origin = {80.0,3.8806}, points = {{-100.0,60.3527},{-80.0,60.0},{-80.0,2.0},{-69.0,2.1194}}, color = {0,0,191}));
        connect(arable_land,Arable_Land_In_Gha.u) annotation(Line(visible = true, origin = {-76.5,64.2333}, points = {{-33.5,0.0},{33.5,0.0}}, color = {0,0,191}));
        connect(HEF_Human_Ecological_Footprint.y,hef_human_ecological_footprint) annotation(Line(points = {{-40.5,0.0},{40.5,0.0}}, color = {0,0,191}, visible = true, origin = {69.5,0.0}));
        annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, fillColor = {170,255,170}, fillPattern = FillPattern.Solid, lineThickness = 1.0, extent = {{-100.0,-100.0},{100.0,100.0}}, radius = 0.5),Text(visible = true, origin = {0.0,50.0}, fillPattern = FillPattern.Solid, extent = {{-78.6862,-24.4985},{78.6862,24.4985}}, textString = "Human", fontSize = 102.0, fontName = "Arial"),Text(visible = true, origin = {1.7338,-49.1014}, fillPattern = FillPattern.Solid, extent = {{-96.4561,-20.8986},{96.4561,20.8986}}, textString = "Footprint", fontSize = 102.0, fontName = "Arial"),Text(visible = true, fillPattern = FillPattern.Solid, extent = {{-88.0175,-25.4584},{88.0175,25.4584}}, textString = "Ecological", fontSize = 102.0, fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Documentation(info = "<html>
The <a href=\"http://www.footprintnetwork.org/gfn_sub.php?content=footprint_overview\">ecological footprint</a> is a concept developed by <a href=\"http://www.footprintnetwork.org/gfn_sub.php?content=whoweare\">Mathis Wackernagel</a> and his co-workers.  It attempts to quantify the percentage of the planet's carrying capacity being used up by humanity at any given point in time. Wackernagel showed that, since the 1980s, humanity is spending more resources per time unit than the planet can regenerate in a sustainable fashion. <p>
 
The human ecological footprint is measured in hectares per person.
</html>"));
      end Human_Ecological_Footprint;
      block Human_Fertility "Human fertility"
        parameter Real des_compl_fam_size_norm = 3.8 "Desired normal complete family size";
        parameter Real hlth_serv_impact_del(unit = "yr") = 20 "Health service impact delay";
        parameter Real income_expect_avg_time(unit = "yr") = 3 "Income expected average time";
        parameter Real lifet_perc_del(unit = "yr") = 20 "Perceived life-time delay";
        parameter Real max_tot_fert_norm = 12 "Normal maximal total fertility";
        parameter Real social_adj_del(unit = "yr") = 20 "Social adjustment delay";
        parameter Real t_fert_cont_eff_time(unit = "yr") = 4000 "Year of continued fertility change";
        parameter Real t_zero_pop_grow_time(unit = "yr") = 4000 "Time to zero population growth";
        SystemDynamics.Interfaces.MassInPort ind_out_pc(unit = "dollar/yr") "Per capita annual industrial output" annotation(Placement(visible = true, transformation(origin = {-210.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.SMTH1 Avg_Ind_Out_PC(averaging_time(unit = "yr") = income_expect_avg_time, smooth_init(unit = "dollar/yr") = 43.3) "p.122 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-125.0,-100.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Fam_Income_Expect Fam_Income_Expect "p.122 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-65.0,-96.1504}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Fam_Resp_To_Soc_Norm(x_vals = (-0.2):0.1:0.2, y_vals = {0.5,0.6,0.7,0.85,1}) "p.122 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-4.3835,-96.1504}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.SMTH3 Del_Ind_Out_PC(averaging_time(unit = "yr") = social_adj_del, smooth_init(unit = "dollar/yr") = 0) "p.119 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-125.0,-60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Soc_Fam_Size_Norm(x_vals = 0:200:800, y_vals = {1.25,0.94,0.715,0.59,0.5}) "p.119 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-67.0,-60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Des_Compl_Fam_Size Des_Compl_Fam_Size(t_zero_pop_grow_time = t_zero_pop_grow_time, des_compl_fam_size_norm = des_compl_fam_size_norm) "p.113 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {72.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort life_expectancy(unit = "yr") "Average life expectancy of human population" annotation(Placement(visible = true, transformation(origin = {-210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.SMTH3 Perc_Life_Expectancy(averaging_time(unit = "yr") = lifet_perc_del, smooth_init(unit = "yr") = 0) "p.112 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-125.0,-20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Compl_Mlt_Perc_Lifet(x_vals = 0:10:90, y_vals = {3,2.1,1.6,1.4,1.3,1.2,1.1,1.05,1,1}) "p.112 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-67.0,-20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Des_Tot_Fert "p.107 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-10.0,-30.0}, extent = {{-12.0,-12.0},{12.0,12.0}}, rotation = -90)));
        SystemDynamics.Functions.Tabular Fecundity_Mult(x_vals = 0:10:90, y_vals = {0,0.2,0.4,0.6,0.7,0.75,0.79,0.84,0.87,0.87}) "p.104 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-124.8043,20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Gain Max_Tot_Fert(k = max_tot_fert_norm) "p.104 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-50.0,20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Need_For_Fert_Cont Need_For_Fert_Cont "p.131 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {41.5,-24.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Fr_Serv_Al_Fert_Cont(x_vals = 0:2:10, y_vals = {0,0.005,0.015,0.025,0.03,0.035}) "p.132 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {99.0,-24.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort serv_out_pc(unit = "dollar/yr") "Per capita annual expenditure in services" annotation(Placement(visible = true, transformation(origin = {-210.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Fert_Cont_Al_PC "p.132 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-140.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.SMTH3 Fert_Cont_Facil_PC(averaging_time(unit = "yr") = hlth_serv_impact_del, smooth_init(unit = "dollar/yr") = 0) "p.133 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-113.0,90.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Fert_Cont_Eff_Table(y_vals = {0.75,0.85,0.9,0.95,0.98,0.99,1,1}, x_vals = {0,0.5,1,1.5,2,2.5,3,15}) "p.133 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-63.0,90.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Fert_Cont_Eff Fert_Cont_Eff(t_fert_cont_eff_time = t_fert_cont_eff_time) "p.133 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-6.0,90.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Total_Fertility Total_Fertility "p.97 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {120.0,100.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort total_fertility "Average human fertility" annotation(Placement(visible = true, transformation(origin = {210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      equation
        connect(Fam_Income_Expect.u1,ind_out_pc) annotation(Line(visible = true, points = {{-74.0,-92.1504},{-100.0,-92.1504},{-100.0,-80.0},{-210.0,-80.0}}, color = {0,0,191}));
        connect(Des_Compl_Fam_Size.u2,Fam_Resp_To_Soc_Norm.y) annotation(Line(visible = true, points = {{63.0,-84.0},{40.0,-84.0},{40.0,-96.1504},{6.6165,-96.1504}}, color = {0,0,191}));
        connect(Fam_Resp_To_Soc_Norm.u,Fam_Income_Expect.y) annotation(Line(visible = true, origin = {-34.1917,-96.1504}, points = {{21.8083,0.0},{-21.8083,0.0}}, color = {0,0,191}));
        connect(Fam_Income_Expect.u2,Avg_Ind_Out_PC.y) annotation(Line(visible = true, points = {{-74.0,-100.1504},{-100.0,-100.0},{-100.0,-100.0},{-114.0,-100.0}}, color = {0,0,191}));
        connect(Max_Tot_Fert.u,Fecundity_Mult.y) annotation(Line(visible = true, origin = {-87.9022,20.0}, points = {{25.9022,0.0},{-25.9022,0.0}}, color = {0,0,191}));
        connect(Fecundity_Mult.u,life_expectancy) annotation(Line(visible = true, points = {{-132.8043,20.0},{-150.0,20.0},{-150.0,-10.0},{-210.0,-10.0}}, color = {0,0,191}));
        connect(Fr_Serv_Al_Fert_Cont.y,Fert_Cont_Al_PC.u2) annotation(Line(visible = true, points = {{110.0,-24.0},{140.0,-24.0},{140.0,60.0},{-133.0,60.0}}, color = {0,0,191}));
        connect(Fr_Serv_Al_Fert_Cont.u,Need_For_Fert_Cont.y) annotation(Line(visible = true, origin = {70.75,-24.0}, points = {{20.25,0.0},{-20.25,0.0}}, color = {0,0,191}));
        connect(Need_For_Fert_Cont.u1,Max_Tot_Fert.y) annotation(Line(visible = true, points = {{32.5,-20.0},{10.0,-20.0},{10.0,20.0},{-39.0,20.0}}, color = {0,0,191}));
        connect(Des_Tot_Fert.y,Need_For_Fert_Cont.u2) annotation(Line(visible = true, origin = {21.975,-29.0}, points = {{-23.575,-1.0},{6.525,-1.0},{6.525,1.0},{10.525,1.0}}, color = {0,0,191}));
        connect(Soc_Fam_Size_Norm.y,Des_Compl_Fam_Size.u1) annotation(Line(visible = true, points = {{-56.0,-60.0},{40.0,-60.0},{40.0,-76.0},{63.0,-76.0}}, color = {0,0,191}));
        connect(Des_Compl_Fam_Size.y,Des_Tot_Fert.u2) annotation(Line(visible = true, points = {{81.0,-80.0},{110.0,-80.0},{110.0,-50.0},{-10.0,-50.0},{-10.0,-38.4}}, color = {0,0,191}));
        connect(Soc_Fam_Size_Norm.u,Del_Ind_Out_PC.y) annotation(Line(visible = true, origin = {-94.5,-60.0}, points = {{19.5,0.0},{-19.5,0.0}}, color = {0,0,191}));
        connect(Compl_Mlt_Perc_Lifet.y,Des_Tot_Fert.u1) annotation(Line(visible = true, points = {{-56.0,-20.0},{-30.0,-20.0},{-30.0,-10.0},{-10.0,-10.0},{-10.0,-21.6}}, color = {0,0,191}));
        connect(Compl_Mlt_Perc_Lifet.u,Perc_Life_Expectancy.y) annotation(Line(visible = true, origin = {-94.5,-20.0}, points = {{19.5,0.0},{-19.5,0.0}}, color = {0,0,191}));
        connect(Avg_Ind_Out_PC.u,ind_out_pc) annotation(Line(visible = true, points = {{-136.0,-100.0},{-150.0,-100.0},{-150.0,-80.0},{-210.0,-80.0}}, color = {0,0,191}));
        connect(Del_Ind_Out_PC.u,ind_out_pc) annotation(Line(visible = true, points = {{-136.0,-60.0},{-150.0,-60.0},{-150.0,-80.0},{-210.0,-80.0}}, color = {0,0,191}));
        connect(Perc_Life_Expectancy.u,life_expectancy) annotation(Line(visible = true, points = {{-136.0,-20.0},{-150.0,-20.0},{-150.0,-10.0},{-210.0,-10.0}}, color = {0,0,191}));
        connect(Max_Tot_Fert.y,Total_Fertility.u2) annotation(Line(visible = true, points = {{-39.0,20.0},{80.0,20.0},{80.0,100.0},{111.0,100.0}}, color = {0,0,191}));
        connect(Des_Tot_Fert.y,Total_Fertility.u3) annotation(Line(visible = true, points = {{-1.6,-30.0},{10.0,-30.0},{10.0,-40.0},{150.0,-40.0},{150.0,70.0},{90.0,70.0},{90.0,94.0},{111.0,94.0}}, color = {0,0,191}));
        connect(Fert_Cont_Eff.y,Total_Fertility.u1) annotation(Line(visible = true, points = {{3.0,90.0},{70.0,90.0},{70.0,106.0},{111.0,106.0}}, color = {0,0,191}));
        connect(Total_Fertility.y,total_fertility) annotation(Line(visible = true, points = {{129.0,100.0},{180.0,100.0},{180.0,-10.0},{210.0,-10.0}}, color = {0,0,191}));
        connect(Fert_Cont_Eff.u,Fert_Cont_Eff_Table.y) annotation(Line(visible = true, origin = {-33.5,90.0}, points = {{18.5,0.0},{-18.5,0.0}}, color = {0,0,191}));
        connect(Fert_Cont_Eff_Table.u,Fert_Cont_Facil_PC.y) annotation(Line(visible = true, origin = {-86.5,90.0}, points = {{15.5,0.0},{-15.5,0.0}}, color = {0,0,191}));
        connect(Fert_Cont_Facil_PC.u,Fert_Cont_Al_PC.y) annotation(Line(visible = true, origin = {-134.6667,82.3333}, points = {{10.6667,7.6667},{-5.3333,7.6667},{-5.3333,-15.3333}}, color = {0,0,191}));
        connect(Fert_Cont_Al_PC.u1,serv_out_pc) annotation(Line(visible = true, origin = {-178.5,60.0}, points = {{31.5,0.0},{-31.5,0.0}}, color = {0,0,191}));
        annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, fillColor = {170,255,170}, fillPattern = FillPattern.Solid, lineThickness = 1.0, extent = {{-100.0,-100.0},{100.0,100.0}}, radius = 0.5),Text(visible = true, origin = {-0.0,30.0}, fillPattern = FillPattern.Solid, extent = {{-65.7919,-36.6918},{65.7919,36.6918}}, textString = "Human", fontSize = 102.0, fontName = "Arial"),Text(visible = true, origin = {0.0,-30.0}, fillPattern = FillPattern.Solid, extent = {{-65.0807,-36.6918},{65.0807,36.6918}}, textString = "Fertility", fontSize = 102.0, fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-200.0,-160.0},{200.0,140.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-151.0,-81.0},{-149.0,-79.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-151.0,-11.0},{-149.0,-9.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{9.0,19.0},{11.0,21.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{9.0,-31.0},{11.0,-29.0}})}), Documentation(info = "<html>
This model estimates the human fertility factor used by the population dynamics model. <p>
 
The human fertility factor is dimensionless.
</html>"), experiment(StopTime = 200), experimentSetupOutput);
      end Human_Fertility;
      block Human_Welfare_Index "Human welfare index"
        SystemDynamics.Interfaces.MassInPort life_expectancy(unit = "yr") "Average life expectancy of human population" annotation(Placement(visible = true, transformation(origin = {-210.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort ind_out_pc(unit = "dollar/yr") "Per capita annual industrial output" annotation(Placement(visible = true, transformation(origin = {-210.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Life_Expectancy_Index(x_vals = {0,25,35,45,55,65,75,85,95}, y_vals = {0,0,0.16,0.33,0.5,0.67,0.84,1,1}) annotation(Placement(visible = true, transformation(origin = {-140.5,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular GDP_Per_Capita(x_vals = 0:200:1000, y_vals = {120,600,1200,1800,2500,3200}) annotation(Placement(visible = true, transformation(origin = {-140.5,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Education_Index(x_vals = 0:1000:7000, y_vals = {0,0.81,0.88,0.92,0.95,0.98,0.99,1}) annotation(Placement(visible = true, transformation(origin = {-40.5,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.GDP_Index GDP_Index annotation(Placement(visible = true, transformation(origin = {-48.1039,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.HWI_Human_Welfare_Index HWI_Human_Welfare_Index annotation(Placement(visible = true, transformation(origin = {43.7138,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort hwi_human_welfare_index "Human welfare index" annotation(Placement(visible = true, transformation(origin = {210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      equation
        connect(Education_Index.y,HWI_Human_Welfare_Index.u2) annotation(Line(visible = true, points = {{-29.5,-10.0},{3.625,-10.0},{3.625,-10.0},{34.7138,-10.0}}, color = {0,0,191}));
        connect(GDP_Per_Capita.y,Education_Index.u) annotation(Line(visible = true, points = {{-129.5,-80.0},{-90.0,-80.0},{-90.0,-10.0},{-48.5,-10.0}}, color = {0,0,191}));
        connect(GDP_Index.u,GDP_Per_Capita.y) annotation(Line(visible = true, origin = {-93.302,-80.0}, points = {{36.1981,0.0},{-36.198,0.0}}, color = {0,0,191}));
        connect(GDP_Per_Capita.u,ind_out_pc) annotation(Line(visible = true, origin = {-179.25,-80.0}, points = {{30.75,0.0},{-30.75,0.0}}, color = {0,0,191}));
        connect(Life_Expectancy_Index.y,HWI_Human_Welfare_Index.u1) annotation(Line(visible = true, points = {{-129.5,60.0},{0.0,60.0},{0.0,-4.0},{34.7138,-4.0}}, color = {0,0,191}));
        connect(Life_Expectancy_Index.u,life_expectancy) annotation(Line(visible = true, origin = {-179.25,60.0}, points = {{30.75,0.0},{-30.75,0.0}}, color = {0,0,191}));
        connect(GDP_Index.y,HWI_Human_Welfare_Index.u3) annotation(Line(points = {{-39.1039,-80.0},{0.0,-80.0},{0.0,-16.0},{34.7138,-16.0}}, color = {0,0,191}, visible = true));
        connect(HWI_Human_Welfare_Index.y,hwi_human_welfare_index) annotation(Line(points = {{80.5,-10.0},{210.0,-10.0}}, color = {0,0,191}, visible = true));
        annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, fillColor = {170,255,170}, fillPattern = FillPattern.Solid, lineThickness = 1.0, extent = {{-100.0,-100.0},{100.0,100.0}}, radius = 0.5),Text(visible = true, origin = {0.0,50.0}, fillPattern = FillPattern.Solid, extent = {{-63.6862,-26.9167},{63.6862,26.9167}}, textString = "Human", fontSize = 102.0, fontName = "Arial"),Text(visible = true, origin = {0.0,-50.0}, fillPattern = FillPattern.Solid, extent = {{-62.9121,-20.8986},{62.9121,20.8986}}, textString = "Index", fontSize = 102.0, fontName = "Arial"),Text(visible = true, fillPattern = FillPattern.Solid, extent = {{-62.5202,-26.0486},{62.5202,26.0486}}, textString = "Welfare", fontSize = 102.0, fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-200.0,-160.0},{200.0,140.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-91.0,-81.0},{-89.0,-79.0}})}), Documentation(info = "<html>
The <a href=\"http://links.jstor.org/sici?sici=0004-5608(198812)78%3A4%3C580%3AWVIHWA%3E2.0.CO%3B2-A\">human welfare index</a> is a measure of how well humans are living on average.  It is similar in concept to the <b>quality of life</b> variable introduced by Forrester in his <font color=red><b>WORLD2</b></font> model.  Yet, the human welfare index has been better standardized, and therefore, has replaced the former <b>quality of life</b> variable in the enhanced <font color=red><b>WORLD3</b></font> model. <p>
 
The human welfare index is dimensionless.
</html>"));
      end Human_Welfare_Index;
      block Industrial_Investment "Investments in the military/industrial sector"
        parameter Real industrial_capital_init(unit = "dollar") = 210000000000.0 "Initial industrial investment";
        parameter Real ind_out_pc_des(unit = "dollar/yr") = 400 "Desired annual industrial per capita output";
        parameter Real p_avg_life_ind_cap_1(unit = "yr") = 14 "Default average life of industrial capital";
        parameter Real p_avg_life_ind_cap_2(unit = "yr") = 14 "Controlled average life of industrial capital";
        parameter Real p_fioa_cons_const_1 = 0.43 "Default fraction of industrial output allocated to consumption";
        parameter Real p_fioa_cons_const_2 = 0.43 "Controlled fraction of industrial output allocated to consumption";
        parameter Real p_ind_cap_out_ratio_1(unit = "yr") = 3 "Default industrial capital output ratio";
        parameter Real t_ind_equil_time(unit = "yr") = 4000 "Year of industrial equilibrium";
        parameter Real t_policy_year(unit = "yr") = 4000 "Year of policy change";
        SystemDynamics.Levels.Level Industrial_Capital(x0 = industrial_capital_init) "p.218 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-60.0,-10.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Rates.RRate Ind_Cap_Invest "p.222 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-120.0,-10.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Rates.RRate Ind_Cap_Deprec "p.221 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {0.0,-10.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Ind_Cap_Inv "p.222 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-120.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division Ind_Cap_Dep "p.221 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {0.0,-49.0}, extent = {{-10.0,-11.0},{10.0,11.0}}, rotation = -270)));
        SystemDynamics.WorldDynamics.World3.Utilities.Industrial_Output Industrial_Output "p.216 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-70.0,40.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_Ind_Cap_Out_Ratio S_Ind_Cap_Out_Ratio(t_policy_year = t_policy_year, p_ind_cap_out_ratio_1 = p_ind_cap_out_ratio_1) "p.216 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-5.0,40.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_3 P_Ind_Cap_Out_Ratio_2 annotation(Placement(visible = true, transformation(origin = {50.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_Avg_Life_Ind_Cap S_Avg_Life_Ind_Cap(t_policy_year = t_policy_year, p_avg_life_ind_cap_1 = p_avg_life_ind_cap_1, p_avg_life_ind_cap_2 = p_avg_life_ind_cap_2) "p.221 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {41.0,-70.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {-170.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink1 annotation(Placement(visible = true, transformation(origin = {50.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.FIOA_Ind FIOA_Ind "p.223 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-140.0,-75.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_FIOA_Cons S_FIOA_Cons(t_ind_equil_time = t_ind_equil_time) "p.223 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-97.0,-110.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_FIOA_Cons_Const S_FIOA_Cons_Const(p_fioa_cons_const_1 = p_fioa_cons_const_1, p_fioa_cons_const_2 = p_fioa_cons_const_2, t_policy_year = t_policy_year) "p.223 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-40.0,-100.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_FIOA_Cons_Var(x_vals = 0:0.2:2, y_vals = {0.3,0.32,0.34,0.36,0.38,0.43,0.73,0.77,0.81,0.82,0.83}) "p.225 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-36.0,-130.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Gain Gain1(k = 1 / ind_out_pc_des) annotation(Placement(visible = true, transformation(origin = {21.0,-130.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division Ind_Out_PC "p.214 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {70.0,-130.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort population "Population" annotation(Placement(visible = true, transformation(origin = {-110.0,-170.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90), iconTransformation(origin = {-20.0,-110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Interfaces.MassInPort s_fioa_serv "Fraction of total investments allocated to the service sector" annotation(Placement(visible = true, transformation(origin = {-170.0,-170.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90), iconTransformation(origin = {-60.0,-110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Interfaces.MassInPort s_fioa_agr "Fraction of total investments allocated to the agricultural sector" annotation(Placement(visible = true, transformation(origin = {-210.0,-130.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort s_fr_cap_al_obt_res "Fraction of capital allocated to resource use efficiency" annotation(Placement(visible = true, transformation(origin = {-210.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort capital_util_fr "Capital utilization fraction" annotation(Placement(visible = true, transformation(origin = {-210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort ppoll_tech_mult_icor_COPM "Technology induced reduction in persistent pollution release" annotation(Placement(visible = true, transformation(origin = {-210.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort yield_tech_mult_icor_COYM "Technology induced relative enhancement of land yield" annotation(Placement(visible = true, transformation(origin = {-210.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort ind_cap_out_ratio_2_ICOR2T "Industrial capital output ratio" annotation(Placement(visible = true, transformation(origin = {-170.0,150.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 90), iconTransformation(origin = {-60.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
        SystemDynamics.Interfaces.MassOutPort industrial_output(unit = "dollar/yr") "Annual industrial output" annotation(Placement(visible = true, transformation(origin = {210.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,21.707}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort industrial_capital(unit = "dollar") "Total capital invested in the military/industrial sector" annotation(Placement(visible = true, transformation(origin = {210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort ind_out_pc(unit = "dollar/yr") "Per capita annual industrial output" annotation(Placement(visible = true, transformation(origin = {210.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Cons_Ind_Out annotation(Placement(visible = true, transformation(origin = {-100.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
        Modelica.Blocks.Math.Division Cons_Ind_Out_PC annotation(Placement(visible = true, transformation(origin = {146.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort cons_ind_out_pc1(unit = "dollar/yr") "Per capita general industrial output" annotation(Placement(visible = true, transformation(origin = {210.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      equation
        connect(P_FIOA_Cons_Var.y,S_FIOA_Cons.u2) annotation(Line(visible = true, points = {{-47.0,-130.0},{-74.0,-130.0},{-74.0,-114.0},{-88.0,-114.0}}, color = {0,0,191}));
        connect(S_FIOA_Cons_Const.y,S_FIOA_Cons.u1) annotation(Line(visible = true, points = {{-49.0,-100.0},{-74.0,-100.0},{-74.0,-106.0},{-88.0,-106.0}}, color = {0,0,191}));
        connect(FIOA_Ind.u1,s_fioa_agr) annotation(Line(visible = true, points = {{-146.0,-84.0},{-146.0,-130.0},{-210.0,-130.0}}, color = {0,0,191}));
        connect(FIOA_Ind.u3,S_FIOA_Cons.y) annotation(Line(visible = true, points = {{-134.0,-84.0},{-134.0,-110.0},{-106.0,-110.0}}, color = {0,0,191}));
        connect(industrial_capital,Industrial_Capital.y2) annotation(Line(visible = true, points = {{210.0,-10.0},{160.0,-10.0},{160.0,-32.0},{-48.75,-32.0},{-48.75,-19.75}}, color = {0,0,191}));
        connect(Industrial_Output.u4,Industrial_Capital.y1) annotation(Line(visible = true, points = {{-61.0,34.0},{-36.0,34.0},{-36.0,-1.75},{-47.25,-1.75}}, color = {0,0,191}));
        connect(S_Ind_Cap_Out_Ratio.y,Industrial_Output.u3) annotation(Line(visible = true, points = {{-14.0,40.0},{-36.0,40.0},{-36.0,38.0},{-61.0,38.0}}, color = {0,0,191}));
        connect(Industrial_Output.u2,capital_util_fr) annotation(Line(visible = true, points = {{-61.0,42.0},{-40.0,42.0},{-40.0,90.0},{-190.0,90.0},{-190.0,-10.0},{-210.0,-10.0}}, color = {0,0,191}));
        connect(Industrial_Output.u1,s_fr_cap_al_obt_res) annotation(Line(visible = true, points = {{-61.0,46.0},{-44.0,46.0},{-44.0,80.0},{-186.0,80.0},{-186.0,-70.0},{-210.0,-70.0}}, color = {0,0,191}));
        connect(Cons_Ind_Out.y,Cons_Ind_Out_PC.u1) annotation(Line(visible = true, points = {{-93.0,-70.0},{-80.0,-70.0},{-80.0,-78.0},{0.0,-78.0},{0.0,-90.0},{110.0,-90.0},{110.0,116.0},{134.0,116.0}}, color = {0,0,191}));
        connect(Ind_Out_PC.u2,Cons_Ind_Out_PC.u2) annotation(Line(visible = true, points = {{82.0,-136.0},{120.0,-136.0},{120.0,104.0},{134.0,104.0}}, color = {0,0,191}));
        connect(Ind_Cap_Dep.u2,S_Avg_Life_Ind_Cap.y) annotation(Line(visible = true, origin = {15.0667,-67.0}, points = {{-8.4667,6.0},{-8.4667,-3.0},{16.9333,-3.0}}, color = {0,0,191}));
        connect(Ind_Out_PC.y,Gain1.u) annotation(Line(visible = true, origin = {46.0,-130.0}, points = {{13.0,0.0},{-13.0,0.0}}, color = {0,0,191}));
        connect(Gain1.y,P_FIOA_Cons_Var.u) annotation(Line(visible = true, origin = {-9.0,-130.0}, points = {{19.0,0.0},{-19.0,0.0}}, color = {0,0,191}));
        connect(S_FIOA_Cons.y,Cons_Ind_Out.u2) annotation(Line(visible = true, points = {{-106.0,-110.0},{-120.0,-110.0},{-120.0,-86.0},{-100.0,-86.0},{-100.0,-77.0}}, color = {0,0,191}));
        connect(Industrial_Output.y,Cons_Ind_Out.u1) annotation(Line(visible = true, origin = {-93.0,5.6667}, points = {{14.0,34.3333},{-7.0,34.3333},{-7.0,-68.6667}}, color = {0,0,191}));
        connect(FIOA_Ind.u2,s_fioa_serv) annotation(Line(visible = true, points = {{-140.0,-84.0},{-140.0,-150.0},{-170.0,-150.0},{-170.0,-170.0}}, color = {0,0,191}));
        connect(Ind_Cap_Inv.u1,FIOA_Ind.y) annotation(Line(visible = true, points = {{-127.0,-50.0},{-140.0,-50.0},{-140.0,-66.0}}, color = {0,0,191}));
        connect(Industrial_Output.y,Ind_Cap_Inv.u2) annotation(Line(visible = true, points = {{-79.0,40.0},{-100.0,40.0},{-100.0,-50.0},{-113.0,-50.0}}, color = {0,0,191}));
        connect(Ind_Cap_Inv.y,Ind_Cap_Invest.u) annotation(Line(visible = true, origin = {-120.0,-34.375}, points = {{0.0,-8.625},{0.0,8.625}}, color = {0,0,191}));
        connect(Sink1.MassInPort1,Ind_Cap_Deprec.y1) annotation(Line(visible = true, origin = {23.25,-10.0}, points = {{15.75,0.0},{-15.75,0.0}}, color = {95,0,191}));
        connect(Ind_Cap_Dep.y,Ind_Cap_Deprec.u) annotation(Line(visible = true, points = {{0.0,-38.0},{0.0,-33.5},{0.0,-33.5},{0.0,-25.75}}, color = {0,0,191}));
        connect(Industrial_Capital.u2,Ind_Cap_Deprec.y) annotation(Line(visible = true, origin = {-25.5,-10.0}, points = {{-18.0,0.0},{18.0,0.0}}, color = {95,0,191}));
        connect(Ind_Cap_Dep.u1,Industrial_Capital.y3) annotation(Line(visible = true, points = {{-6.6,-61.0},{-6.6,-70.0},{-60.0,-70.0},{-60.0,-19.75}}, color = {0,0,191}));
        connect(Ind_Cap_Invest.y1,Industrial_Capital.u1) annotation(Line(visible = true, origin = {-94.5,-10.0}, points = {{-18.0,0.0},{18.0,0.0}}, color = {95,0,191}));
        connect(Source1.MassInPort1,Ind_Cap_Invest.y) annotation(Line(visible = true, origin = {-143.25,-10.0}, points = {{-15.75,0.0},{15.75,0.0}}, color = {95,0,191}));
        connect(Industrial_Output.y,industrial_output) annotation(Line(visible = true, points = {{-79.0,40.0},{-100.0,40.0},{-100.0,20.0},{160.0,20.0},{160.0,50.0},{210.0,50.0}}, color = {0,0,191}));
        connect(Ind_Out_PC.u1,Industrial_Output.y) annotation(Line(visible = true, points = {{82.0,-124.0},{90.0,-124.0},{90.0,20.0},{-100.0,20.0},{-100.0,40.0},{-79.0,40.0}}, color = {0,0,191}));
        connect(P_Ind_Cap_Out_Ratio_2.y,S_Ind_Cap_Out_Ratio.u) annotation(Line(visible = true, origin = {23.5,40.0}, points = {{19.5,0.0},{-19.5,0.0}}, color = {0,0,191}));
        connect(P_Ind_Cap_Out_Ratio_2.u3,ppoll_tech_mult_icor_COPM) annotation(Line(visible = true, points = {{50.0,47.0},{50.0,100.0},{-194.0,100.0},{-194.0,50.0},{-210.0,50.0}}, color = {0,0,191}));
        connect(P_Ind_Cap_Out_Ratio_2.u2,yield_tech_mult_icor_COYM) annotation(Line(visible = true, points = {{57.0,40.0},{70.0,40.0},{70.0,110.0},{-210.0,110.0}}, color = {0,0,191}));
        connect(P_Ind_Cap_Out_Ratio_2.u1,ind_cap_out_ratio_2_ICOR2T) annotation(Line(visible = true, points = {{50.0,33.0},{50.0,26.0},{90.0,26.0},{90.0,120.0},{-170.0,120.0},{-170.0,150.0}}, color = {0,0,191}));
        connect(Cons_Ind_Out_PC.y,cons_ind_out_pc1) annotation(Line(visible = true, origin = {183.5,110.0}, points = {{-26.5,0.0},{26.5,0.0}}, color = {0,0,191}));
        connect(Ind_Out_PC.u2,population) annotation(Line(points = {{82.0,-136.0},{90.0,-136.0},{90.0,-150.0},{-110.0,-150.0},{-110.0,-170.0}}, color = {0,0,191}, visible = true));
        connect(Ind_Out_PC.y,ind_out_pc) annotation(Line(points = {{59.0,-130.0},{52.0,-130.0},{52.0,-160.0},{160.0,-160.0},{160.0,-70.0},{210.0,-70.0}}, color = {0,0,191}, visible = true));
        annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, fillColor = {170,255,170}, fillPattern = FillPattern.Solid, lineThickness = 1.0, extent = {{-100.0,-100.0},{100.0,100.0}}, radius = 0.5),Text(visible = true, origin = {0.0,30.0}, fillPattern = FillPattern.Solid, extent = {{-86.8431,-36.6918},{86.8431,36.6918}}, textString = "Industrial", fontSize = 102.0, fontName = "Arial"),Text(visible = true, origin = {-0.0,-30.0}, fillPattern = FillPattern.Solid, extent = {{-92.5404,-36.6918},{92.5404,36.6918}}, textString = "Investment", fontSize = 102.0, fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-200.0,-160.0},{200.0,140.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-101.0,19.0},{-99.0,21.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{89.0,19.0},{91.0,21.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{51.0,-131.0},{53.0,-129.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-121.0,-111.0},{-119.0,-109.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-101.0,-51.0},{-99.0,-49.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{89.0,-137.0},{91.0,-135.0}})}), Documentation(info = "<html>
This model calculates the capital invested in industry. <p>
 
Whereas the earlier <font color=red><b>WORLD2</b></font> model lumped all capital investment into a single state variable from which it then split off the percentage of capital invested in the agricultural sector as a second state variable (a funny choice for a state variable, since percentages don't accumulate), the more modern <font color=red><b>WORLD3</b></font> model differentiates between capital invested in industry, capital invested in the service sector, capital invested in agriculture, and other capital investment. <p>
 
The industrial capital is measured in U.S. dollars.
</html>"));
      end Industrial_Investment;
      block Labor_Utilization "Utilization of the labor force"
        parameter Real labor_util_fr_del_init = 1 "Initial delayed labor utilization fraction";
        parameter Real labor_util_fr_del_time(unit = "yr") = 2 "Labor utilization fraction delay time";
        SystemDynamics.Levels.Level1a Labor_Util_Fr_Del(x0 = labor_util_fr_del_init) annotation(Placement(visible = true, transformation(origin = {0.0,-5.1833}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.Rates.Rate_1 Chg_Lab_Util_Fr_Del annotation(Placement(visible = true, transformation(origin = {-40.0,-4.6542}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {-70.0,-13.65}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Ch_Lab_Util_Fr_Del Ch_Lab_Util_Fr_Del(labor_util_fr_del_time = labor_util_fr_del_time) "p.241 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-40.0,-37.4563}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)), Dialog(group = "Variables"));
        SystemDynamics.Functions.Tabular Capital_Util_Fr(x_vals = (-1):2:11, y_vals = {1,1,0.9,0.7,0.3,0.1,0.1}) "p.241 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {57.225,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)), Dialog(group = "Variables"));
        Modelica.Blocks.Math.Division Labor_Util_Fr "p.241 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-44.7625,-72.0625}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)), Dialog(group = "Variables"));
        Modelica.Blocks.Math.Add3 Jobs "p.233 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-65.4417,30.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Pot_Jobs_Agr_Sector "p.237 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-75.075,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular Jobs_Per_Hect(x_vals = {0,2,6,10,14,18,22,26,30,10000}, y_vals = {2,2,0.5,0.4,0.3,0.27,0.24,0.2,0.2,0.2}) "p.239 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-50.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.Auxiliary.Prod_2 Pot_Jobs_Ind_Sector "p.233 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-22.0,44.6542}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular Jobs_Per_Ind_Cap_Unit(x_vals = (-100):150:800, y_vals = {0.00037,0.00037,0.00018,0.00012,0.00009,0.00007,0.00006}) "p.236 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {10.0,44.6542}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.Auxiliary.Prod_2 Pot_Jobs_In_Serv_Sector "p.236 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {56.1333,35.2917}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular Jobs_Per_Serv_Cap_Unit(x_vals = {0,50,200,350,500,650,800,1200}, y_vals = {0.0011,0.0011,0.0006,0.00035,0.0002,0.00015,0.00015,0.00015}) "p.237 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {80.0,50.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.Interfaces.MassInPort service_capital(unit = "dollar") "Total capital invested in service sector" annotation(Placement(visible = true, transformation(origin = {-110.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort ind_out_pc(unit = "dollar/yr") "Per capita annual industrial output" annotation(Placement(visible = true, transformation(origin = {-110.0,41.5333}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort industrial_capital(unit = "dollar") "Total capital invested in the military/industrial sector" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort agr_inp_per_hect(unit = "dollar/(hectare.yr)") "Agricultural investments per hectare" annotation(Placement(visible = true, transformation(origin = {-110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort arable_land(unit = "hectare") "Arable land" annotation(Placement(visible = true, transformation(origin = {-110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort labor_force "Total human labor force" annotation(Placement(visible = true, transformation(origin = {-76.1597,-102.9722}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90), iconTransformation(origin = {-60.0,-110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Interfaces.MassInPort serv_out_pc(unit = "dollar/yr") "Per capita annual expenditure in services" annotation(Placement(visible = true, transformation(origin = {-77.925,103.0806}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 90), iconTransformation(origin = {-60.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
        SystemDynamics.Interfaces.MassOutPort capital_util_fr "Capital utilization fraction" annotation(Placement(visible = true, transformation(origin = {111.575,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      equation
        connect(Jobs_Per_Hect.u,agr_inp_per_hect) annotation(Line(visible = true, points = {{-42.0,80.0},{-17.4625,80.0},{-17.4625,93.1333},{-90.0,93.1333},{-90.0,-40.0},{-110.0,-40.0}}, color = {0,0,191}));
        connect(Pot_Jobs_In_Serv_Sector.u2,service_capital) annotation(Line(visible = true, points = {{49.1333,35.2917},{45.5083,35.2917},{45.5083,100.0},{-100.0,100.0},{-100.0,80.0},{-110.0,80.0}}, color = {0,0,191}));
        connect(Capital_Util_Fr.u,Labor_Util_Fr_Del.y1) annotation(Line(visible = true, points = {{49.225,0.0},{30.0,0.0},{30.0,3.0667},{12.75,3.0667}}, color = {0,0,191}));
        connect(Ch_Lab_Util_Fr_Del.u2,Labor_Util_Fr_Del.y4) annotation(Line(visible = true, points = {{-36.0,-46.4563},{-36.0,-54.5042},{-11.25,-54.5042},{-11.25,-14.9333}}, color = {0,0,191}));
        connect(Labor_Util_Fr.y,Ch_Lab_Util_Fr_Del.u1) annotation(Line(visible = true, points = {{-44.7625,-61.0625},{-44.0,-61.0625},{-44.0,-46.4563}}, color = {0,0,191}));
        connect(Ch_Lab_Util_Fr_Del.y,Chg_Lab_Util_Fr_Del.u) annotation(Line(visible = true, origin = {-40.0,-24.4302}, points = {{0.0,-4.026},{0.0,4.026}}, color = {0,0,191}));
        connect(Chg_Lab_Util_Fr_Del.y1,Labor_Util_Fr_Del.u1) annotation(Line(visible = true, origin = {-22.0,-4.9188}, points = {{-10.5,0.2645},{2.5,0.2645},{2.5,-0.2645},{5.5,-0.2645}}, color = {95,0,191}));
        connect(Source1.MassInPort1,Chg_Lab_Util_Fr_Del.y) annotation(Line(visible = true, origin = {-54.625,-9.1521}, points = {{-4.375,-4.4979},{-1.375,-4.4979},{-1.375,4.4979},{7.125,4.4979}}, color = {95,0,191}));
        connect(Pot_Jobs_In_Serv_Sector.y,Jobs.u3) annotation(Line(visible = true, origin = {19.6083,24.0972}, points = {{36.525,4.1945},{36.525,-2.0972},{-73.05,-2.0972}}, color = {0,0,191}));
        connect(Jobs_Per_Serv_Cap_Unit.y,Pot_Jobs_In_Serv_Sector.u1) annotation(Line(visible = true, origin = {66.6,42.6458}, points = {{2.4,7.3542},{0.5333,7.3542},{0.5333,-7.3541},{-3.4667,-7.3541}}, color = {0,0,191}));
        connect(Pot_Jobs_Ind_Sector.u2,industrial_capital) annotation(Line(visible = true, points = {{-29.0,44.6542},{-40.0,44.6542},{-40.0,94.1917},{-94.1917,94.1917},{-94.1917,0.0},{-110.0,0.0}}, color = {0,0,191}));
        connect(Jobs_Per_Ind_Cap_Unit.y,Pot_Jobs_Ind_Sector.u1) annotation(Line(visible = true, origin = {-8.0,44.6542}, points = {{7.0,0.0},{-7.0,0.0}}, color = {0,0,191}));
        connect(Pot_Jobs_Ind_Sector.y,Jobs.u2) annotation(Line(visible = true, origin = {-32.4806,32.5514}, points = {{10.4806,5.1028},{10.4806,-2.5514},{-20.9611,-2.5514}}, color = {0,0,191}));
        connect(Pot_Jobs_Agr_Sector.u2,arable_land) annotation(Line(visible = true, points = {{-82.075,80.0},{-83.6083,80.0},{-83.6083,-80.0},{-110.0,-80.0}}, color = {0,0,191}));
        connect(Jobs_Per_Hect.y,Pot_Jobs_Agr_Sector.u1) annotation(Line(visible = true, origin = {-64.5375,80.0}, points = {{3.5375,0.0},{-3.5375,0.0}}, color = {0,0,191}));
        connect(Pot_Jobs_Agr_Sector.y,Jobs.u1) annotation(Line(visible = true, origin = {-60.895,56.6}, points = {{-14.18,16.4},{-14.18,10.4},{10.4533,10.4},{10.4533,-18.6},{7.4533,-18.6}}, color = {0,0,191}));
        connect(Labor_Util_Fr.u1,Jobs.y) annotation(Line(points = {{-50.7625,-84.0625},{-50.7625,-90.0},{-80.0,-90.0},{-80.0,30.0},{-76.4417,30.0}}, color = {0,0,191}, visible = true));
        connect(Labor_Util_Fr.u2,labor_force) annotation(Line(points = {{-38.7625,-84.0625},{-40.0,-94.7208},{-76.2,-94.7208},{-76.1597,-102.9722}}, color = {0,0,191}, visible = true));
        connect(Jobs_Per_Ind_Cap_Unit.u,ind_out_pc) annotation(Line(points = {{18.0,44.6542},{30.0,44.6542},{30.0,95.7792},{-97.3667,95.7792},{-97.3667,41.5333},{-110.0,41.5333}}, color = {0,0,191}, visible = true));
        connect(Jobs_Per_Serv_Cap_Unit.u,serv_out_pc) annotation(Line(points = {{88.0,50.0},{97.3667,50.0},{97.3667,96.8375},{-77.925,98.425},{-77.925,103.0806}}, color = {0,0,191}, visible = true));
        connect(capital_util_fr,Capital_Util_Fr.y) annotation(Line(points = {{111.575,0.0},{90.0,0.0},{87.3125,0.0},{68.225,0.0}}, color = {0,0,191}, visible = true));
        annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, fillColor = {170,255,170}, fillPattern = FillPattern.Solid, lineThickness = 1.0, extent = {{-100.0,-100.0},{100.0,100.0}}, radius = 0.5),Text(visible = true, origin = {0.0,30.0}, fillPattern = FillPattern.Solid, extent = {{-47.3724,-36.6918},{47.3724,36.6918}}, textString = "Labor", fontSize = 102.0, fontName = "Arial"),Text(visible = true, origin = {0.0,-30.0}, fillPattern = FillPattern.Solid, extent = {{-95.4145,-36.6918},{95.4145,36.6918}}, textString = "Utilization", fontSize = 102.0, fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Documentation(info = "<html>
This model computes the available labor work force that is available to be employed by agriculture, industry, and in the service sector. <p>
 
The labor utilization fraction is dimensionless.
</html>"));
      end Labor_Utilization;
      block Land_Fertility "Land fertility"
        parameter Real des_food_ratio_dfr = 2 "Desired food ratio";
        parameter Real inherent_land_fert(unit = "kg/(hectare.yr)") = 600 "Inherent land fertility";
        parameter Real land_fertility_init(unit = "kg/(hectare.yr)") = 600 "Initial industrial investment";
        parameter Real t_policy_year(unit = "yr") = 4000 "Year of policy change";
        parameter Real yield_tech_init = 1 "Initial yield technology factor";
        parameter Real p_yield_tech_chg_mlt[:] = {0,0,0,0} "Yield technology change multiplier";
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {-66.2556,-17.775}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.RRate Land_Fert_Regen "p.328 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-35.3056,-17.775}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.Levels.Level Land_Fertility(x0 = land_fertility_init) "p.324 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {1.852,-17.6868}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.Rates.RRate Land_Fert_Degr "p.326 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {40.0,-27.3}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.Sources.Sink Sink1 annotation(Placement(visible = true, transformation(origin = {63.0806,-27.3}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Land_Fert_Reg Land_Fert_Reg(inherent_land_fert = inherent_land_fert) "p.328 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-35.3056,-47.3694}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)), Dialog(group = "Variables"));
        SystemDynamics.Auxiliary.Prod_2 Land_Fert_Deg "p.326 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {40.0,-60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Land_Fert_Regen_Time(x_vals = 0:0.02:0.1, y_vals = {20,13,8,4,2,2}) "p.330 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-60.0,-74.7625}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.Functions.Tabular Land_Fert_Degr_Rt(x_vals = {0,10,20,30,100}, y_vals = {0,0.1,0.3,0.5,0.5}) "p.326 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {51.9556,-90.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.Interfaces.MassInPort p_fr_inp_for_land_maint "Fraction of investments in the agricultural sector allocated to land maintenance" annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort ppoll_index "Persistent pollution index" annotation(Placement(visible = true, transformation(origin = {-110.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort land_fertility(unit = "kg/(hectare.yr)") "Land fertility" annotation(Placement(visible = true, transformation(origin = {110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Yield_Tech_Chg_Rt_LYTDR annotation(Placement(visible = true, transformation(origin = {15.9833,77.775}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Utilities.Yield_Tech_Chg_Rt Yield_Tech_Chg_Rt(t_policy_year = t_policy_year) annotation(Placement(visible = true, transformation(origin = {16.1257,46.8312}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular P_Yield_Tech_Chg_Mlt_LYCM(y_vals = p_yield_tech_chg_mlt, x_vals = {-2,0,1,2}) annotation(Placement(visible = true, transformation(origin = {-20.0,20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.Interfaces.MassInPort food_ratio "Food ratio" annotation(Placement(visible = true, transformation(origin = {-110.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Add Add1(k2 = -1) annotation(Placement(visible = true, transformation(origin = {-56.1556,20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Const Des_Food_Ratio_DFR(k = des_food_ratio_dfr) annotation(Placement(visible = true, transformation(origin = {-73.9222,73.5389}, extent = {{-10.0,10.0},{10.0,-10.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.Interfaces.MassOutPort yield_tech_LYTD "Technology induced absolute yield enhancement" annotation(Placement(visible = true, transformation(origin = {110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source2 annotation(Placement(visible = true, transformation(origin = {-27.9111,77.775}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Levels.Level1a Yield_Tech_LYTD(x0 = yield_tech_init) annotation(Placement(visible = true, transformation(origin = {55.0,70.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)), Dialog(group = "Variables"));
      equation
        connect(Land_Fert_Regen_Time.u,p_fr_inp_for_land_maint) annotation(Line(visible = true, points = {{-68.0,-74.7625},{-80.0,-74.4979},{-80.0,0.0},{-110.0,0.0}}, color = {0,0,191}));
        connect(Land_Fert_Regen_Time.y,Land_Fert_Reg.u1) annotation(Line(visible = true, origin = {-42.5371,-68.6315}, points = {{-6.4629,-6.131},{3.2315,-6.131},{3.2315,12.2621}}, color = {0,0,191}));
        connect(land_fertility,Land_Fertility.y1) annotation(Line(visible = true, points = {{110.0,-40.0},{90.0,-40.0},{90.0,-9.4368},{14.6021,-9.4368}}, color = {0,0,191}));
        connect(Land_Fert_Deg.u1,Land_Fertility.y2) annotation(Line(visible = true, points = {{33.0,-60.0},{13.1021,-60.0},{13.1021,-27.4368}}, color = {0,0,191}));
        connect(Land_Fert_Reg.u2,Land_Fertility.y4) annotation(Line(visible = true, origin = {10.2306,18.3444}, points = {{-41.5362,-74.7138},{-41.5362,-86.3423},{-19.6286,-86.3423},{-19.6286,-45.7812}}, color = {0,0,191}));
        connect(Land_Fertility.u2,Land_Fert_Degr.y) annotation(Line(visible = true, origin = {23.713,-22.4934}, points = {{-5.361,4.8066},{-1.713,4.8066},{-1.713,-4.8066},{8.787,-4.8066}}, color = {95,0,191}));
        connect(Land_Fert_Regen.y1,Land_Fertility.u1) annotation(Line(visible = true, origin = {-18.4374,-17.7309}, points = {{-9.3682,-0.0441},{2.7894,-0.0441},{2.7894,0.0441},{3.7894,0.0441}}, color = {95,0,191}));
        connect(Land_Fert_Degr_Rt.y,Land_Fert_Deg.u2) annotation(Line(visible = true, points = {{62.9556,-90.0},{80.0,-90.0},{80.0,-60.0},{47.0,-60.0}}, color = {0,0,191}));
        connect(Land_Fert_Deg.y,Land_Fert_Degr.u) annotation(Line(visible = true, origin = {40.0,-48.025}, points = {{0.0,-4.975},{0.0,4.975}}, color = {0,0,191}));
        connect(Land_Fert_Degr.y1,Sink1.MassInPort1) annotation(Line(visible = true, origin = {49.0269,-27.3}, points = {{-1.5269,0.0},{-1.5269,0.0},{3.0537,0.0}}, color = {95,0,191}));
        connect(Land_Fert_Reg.y,Land_Fert_Regen.u) annotation(Line(visible = true, origin = {-35.3056,-35.9472}, points = {{0.0,-2.4222},{0.0,2.4222}}, color = {0,0,191}));
        connect(Source1.MassInPort1,Land_Fert_Regen.y) annotation(Line(visible = true, origin = {-49.0306,-17.775}, points = {{-6.225,0.0},{6.225,0.0}}, color = {95,0,191}));
        connect(yield_tech_LYTD,Yield_Tech_LYTD.y2) annotation(Line(visible = true, points = {{110.0,40.0},{66.25,40.0},{66.25,60.25}}, color = {0,0,191}));
        connect(Yield_Tech_Chg_Rt.u2,Yield_Tech_LYTD.y4) annotation(Line(visible = true, points = {{20.1257,37.8312},{20.0,20.0},{44.25,20.0},{43.75,60.25}}, color = {0,0,191}));
        connect(Yield_Tech_Chg_Rt_LYTDR.y1,Yield_Tech_LYTD.u1) annotation(Line(visible = true, origin = {33.2458,73.8875}, points = {{-9.7625,3.8875},{2.2542,3.8875},{2.2542,-3.8875},{5.2542,-3.8875}}, color = {95,0,191}));
        connect(Source2.MassInPort1,Yield_Tech_Chg_Rt_LYTDR.y) annotation(Line(visible = true, origin = {-4.2139,77.775}, points = {{-12.6972,0.0},{12.6972,0.0}}, color = {95,0,191}));
        connect(Yield_Tech_Chg_Rt.y,Yield_Tech_Chg_Rt_LYTDR.u) annotation(Line(visible = true, points = {{16.1257,55.8312},{15.9833,62.025}}, color = {0,0,191}));
        connect(food_ratio,Add1.u2) annotation(Line(points = {{-110.0,60.0},{-90.0,60.0},{-90.0,14.0},{-68.1556,14.0}}, color = {0,0,191}, visible = true));
        connect(Add1.u1,Des_Food_Ratio_DFR.y) annotation(Line(points = {{3.8444,-13.513},{-1.9222,-13.513},{-1.9222,27.0259}}, color = {0,0,191}, visible = true, origin = {-72.0,39.513}));
        connect(Add1.y,P_Yield_Tech_Chg_Mlt_LYCM.u) annotation(Line(points = {{-8.5778,0.0},{8.5778,0.0}}, color = {0,0,191}, visible = true, origin = {-36.5778,20.0}));
        connect(Yield_Tech_Chg_Rt.u1,P_Yield_Tech_Chg_Mlt_LYCM.y) annotation(Line(points = {{7.0419,11.8875},{7.0419,-5.9437},{-14.0838,-5.9437}}, color = {0,0,191}, visible = true, origin = {5.0838,25.9437}));
        connect(Land_Fert_Degr_Rt.u,ppoll_index) annotation(Line(points = {{43.9556,-90.0},{-90.0,-90.0},{-90.0,-50.0},{-110.0,-50.0}}, color = {0,0,191}, visible = true));
        annotation(Diagram(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, fillColor = {170,255,170}, fillPattern = FillPattern.Solid, lineThickness = 1.0, extent = {{-100.0,-100.0},{100.0,100.0}}, radius = 0.5),Text(visible = true, origin = {0.0,30.0}, fillPattern = FillPattern.Solid, extent = {{-47.3724,-36.6918},{47.3724,36.6918}}, textString = "Land", fontSize = 102.0, fontName = "Arial"),Text(visible = true, origin = {0.0,-30.0}, fillPattern = FillPattern.Solid, extent = {{-65.0807,-36.6918},{65.0807,36.6918}}, textString = "Fertility", fontSize = 102.0, fontName = "Arial")}), Documentation(info = "<html>
This model computes the land fertility as a function of its utilization.  Land can be overused.  If that happens, the fertility of the land shrinks.  This is likely to happen as the overall world population continues to climb. <p>
 
The land fertility is measured in kilograms of equivalent vegetable food per hectare and year.  Arable land can be used to either grow vegetable food directly, or alternatively, to raise animals that then enter the human food chain.  Yet it takes more land per human to raise animals than to grow vegetable food.  For this reason, it is meaningful to convert proteins obtained from eating meat to equivalent units of vegetable food. <p>
 
Notice that this model uses two different types of rate models.  Whereas the yield technology rate is a so-called bi-flow, i.e., it can assume both positive and negative values leading to a change of flow direction, the land fertility integrator uses uni-flows, i.e., flows that are allowed to flow in the indicated direction only.
</html>", revisions = ""));
      end Land_Fertility;
      block Life_Expectancy "Life expectancy"
        parameter Real hlth_serv_impact_del(unit = "yr") = 20 "Health services impact delay";
        parameter Real life_expect_norm(unit = "yr") = 28 "Normal life expectancy";
        parameter Real subsist_food_pc(unit = "kg/yr") = 230 "Available per capita food";
        SystemDynamics.Auxiliary.Prod_5 Life_Expectancy annotation(Placement(visible = true, transformation(origin = {140.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
        SystemDynamics.Interfaces.MassOutPort life_expectancy(unit = "yr") "Average life expectancy of human population" annotation(Placement(visible = true, transformation(origin = {210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Const Life_Expect_Norm(k = life_expect_norm) annotation(Placement(visible = true, transformation(origin = {140.0,103.0}, extent = {{-10.0,10.0},{10.0,-10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Lifet_Mult_Hlth_Serv Lifet_Mult_Hlth_Serv "p.76 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {85.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Lifet_Mlt_Hlth_Serv_1(x_vals = {0,20,40,60,80,100,200}, y_vals = {1,1.1,1.4,1.6,1.7,1.8,1.8}) "p.76 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {10.0,100.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Lifet_Mlt_Hlth_Serv_2(x_vals = {0,20,40,60,80,100,200}, y_vals = {1,1.5,1.9,2,2,2,2}) "p.76 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {10.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.SMTH1 Eff_Hlth_Serv_PC(averaging_time(unit = "yr") = hlth_serv_impact_del, smooth_init(unit = "dollar/yr") = 0) "p.71 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-58.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Hlth_Serv_Al_PC(x_vals = 0:250:2000, y_vals = {0,20,50,95,140,175,200,220,230}) "p.70 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-114.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort serv_out_pc(unit = "dollar/yr") "Per capita annual expenditure in services" annotation(Placement(visible = true, transformation(origin = {-210.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Lifet_Mlt_Crowd Lifet_Mlt_Crowd "p.90 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {81.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Crowd_Mult_Ind(x_vals = 0:200:1600, y_vals = {0.5,0.05,-0.1,-0.08,-0.02,0.05,0.1,0.15,0.2}) "p.90 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {10.0,20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Fr_Pop_Urban(x_vals = 0:2000000000.0:16000000000.0, y_vals = {0,0.2,0.4,0.5,0.58,0.65,0.72,0.78,0.8}) "p.88 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {10.0,-20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort ind_out_pc(unit = "dollar/yr") "Per capita annual industrial output" annotation(Placement(visible = true, transformation(origin = {-210.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort population "Population" annotation(Placement(visible = true, transformation(origin = {-210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Lifet_Mlt_Food(x_vals = 0:5, y_vals = {0,1,1.43,1.5,1.5,1.5}) "p.66 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {10.0,-60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Gain Gain1(k = 1 / subsist_food_pc) annotation(Placement(visible = true, transformation(origin = {-51.0,-60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort food_pc(unit = "kg/yr") "Per capita annually consumed food" annotation(Placement(visible = true, transformation(origin = {-210.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Lifet_Mlt_PPoll(x_vals = 0:10:100, y_vals = {1,0.99,0.97,0.95,0.9,0.85,0.75,0.65,0.55,0.4,0.2}) "p.94 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {10.0,-100.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort ppoll_index "Persistent pollution index" annotation(Placement(visible = true, transformation(origin = {-210.0,-130.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      equation
        connect(Life_Expect_Norm.y,Life_Expectancy.u1) annotation(Line(visible = true, origin = {140.0,46.5}, points = {{0.0,49.5},{0.0,-49.5}}, color = {0,0,191}));
        connect(Lifet_Mlt_Hlth_Serv_1.y,Lifet_Mult_Hlth_Serv.u1) annotation(Line(visible = true, points = {{21.0,100.0},{50.0,100.0},{50.0,84.0},{76.0,84.0}}, color = {0,0,191}));
        connect(Lifet_Mlt_Hlth_Serv_2.y,Lifet_Mult_Hlth_Serv.u2) annotation(Line(visible = true, points = {{21.0,60.0},{50.0,60.0},{50.0,76.0},{76.0,76.0}}, color = {0,0,191}));
        connect(Lifet_Mult_Hlth_Serv.y,Life_Expectancy.u2) annotation(Line(visible = true, points = {{94.0,80.0},{120.0,80.0},{120.0,-5.4},{135.0,-5.4}}, color = {0,0,191}));
        connect(Lifet_Mlt_Food.y,Life_Expectancy.u4) annotation(Line(visible = true, points = {{21.0,-60.0},{120.0,-60.0},{120.0,-14.6},{135.0,-14.6}}, color = {0,0,191}));
        connect(Lifet_Mlt_PPoll.y,Life_Expectancy.u5) annotation(Line(visible = true, origin = {100.3333,-72.3333}, points = {{-79.3333,-27.6667},{39.6667,-27.6667},{39.6667,55.3333}}, color = {0,0,191}));
        connect(Lifet_Mlt_Crowd.y,Life_Expectancy.u3) annotation(Line(visible = true, points = {{90.0,0.0},{110.0,0.0},{110.0,-10.0},{133.0,-10.0}}, color = {0,0,191}));
        connect(Life_Expectancy.y,life_expectancy) annotation(Line(visible = true, origin = {178.5,-10.0}, points = {{-31.5,0.0},{31.5,0.0}}, color = {0,0,191}));
        connect(Crowd_Mult_Ind.y,Lifet_Mlt_Crowd.u1) annotation(Line(visible = true, points = {{21.0,20.0},{50.0,20.0},{50.0,4.0},{72.0,4.0}}, color = {0,0,191}));
        connect(Fr_Pop_Urban.y,Lifet_Mlt_Crowd.u2) annotation(Line(visible = true, points = {{21.0,-20.0},{50.0,-20.0},{50.0,-4.0},{72.0,-4.0}}, color = {0,0,191}));
        connect(Gain1.u,food_pc) annotation(Line(visible = true, points = {{-63.0,-60.0},{-180.0,-60.0},{-180.0,-70.0},{-210.0,-70.0}}, color = {0,0,191}));
        connect(Gain1.y,Lifet_Mlt_Food.u) annotation(Line(visible = true, origin = {-19.0,-60.0}, points = {{-21.0,0.0},{21.0,0.0}}, color = {0,0,191}));
        connect(Lifet_Mlt_PPoll.u,ppoll_index) annotation(Line(visible = true, points = {{2.0,-100.0},{-180.0,-100.0},{-180.0,-130.0},{-210.0,-130.0}}, color = {0,0,191}));
        connect(Fr_Pop_Urban.u,population) annotation(Line(visible = true, points = {{2.0,-20.0},{-180.0,-20.0},{-180.0,-10.0},{-210.0,-10.0}}, color = {0,0,191}));
        connect(Crowd_Mult_Ind.u,ind_out_pc) annotation(Line(visible = true, points = {{2.0,20.0},{-180.0,20.0},{-180.0,50.0},{-210.0,50.0}}, color = {0,0,191}));
        connect(Eff_Hlth_Serv_PC.y,Lifet_Mlt_Hlth_Serv_2.u) annotation(Line(visible = true, points = {{-47.0,80.0},{-30.0,80.0},{-30.0,60.0},{2.0,60.0}}, color = {0,0,191}));
        connect(Eff_Hlth_Serv_PC.y,Lifet_Mlt_Hlth_Serv_1.u) annotation(Line(visible = true, points = {{-47.0,80.0},{-30.0,80.0},{-30.0,100.0},{2.0,100.0}}, color = {0,0,191}));
        connect(Hlth_Serv_Al_PC.y,Eff_Hlth_Serv_PC.u) annotation(Line(visible = true, origin = {-86.0,80.0}, points = {{-17.0,0.0},{17.0,0.0}}, color = {0,0,191}));
        connect(Hlth_Serv_Al_PC.u,serv_out_pc) annotation(Line(visible = true, points = {{-122.0,80.0},{-180.0,80.0},{-180.0,110.0},{-210.0,110.0}}, color = {0,0,191}));
        annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, fillColor = {170,255,170}, fillPattern = FillPattern.Solid, lineThickness = 1.0, extent = {{-100.0,-100.0},{100.0,100.0}}, radius = 0.5),Text(visible = true, origin = {0.0,30.0}, fillPattern = FillPattern.Solid, extent = {{-47.3724,-36.6918},{47.3724,36.6918}}, textString = "Life", fontSize = 102.0, fontName = "Arial"),Text(visible = true, origin = {-0.0,-30.0}, fillPattern = FillPattern.Solid, extent = {{-95.2722,-36.6918},{95.2722,36.6918}}, textString = "Expectancy", fontSize = 102.0, fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-200.0,-160.0},{200.0,140.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-31.0,79.0},{-29.0,81.0}})}), Documentation(info = "<html>
This model computes the average life expectancy of the human inhabitants of this planet as a function of their living conditions. <p>
 
The life expectancy is measured in numbers of years. <p>
 
This model is described on p.61 of <a href=\"http://www.pegasuscom.com/BookDetail.asp?BookQuery_Action=Find('ISBN','XDYGO')\">Dynamics of Growth in a Finite World</a>.
</html>"));
      end Life_Expectancy;
      block NR_Resource_Utilization "Utilization of non-recoverable natural resources"
        parameter Real des_res_use_rt_DNRUR(unit = "ton/yr") = 4800000000.0 "Desired resource utilization rate";
        parameter Real nr_resources_init(unit = "ton") = 1000000000000.0 "Initial available non-recoverable resources";
        parameter Real p_nr_res_use_fact_1 = 1 "Default non-recoverable resource utilization factor";
        parameter Real res_tech_init = 1 "Initial non-recoverable resource technology factor";
        parameter Real t_policy_year(unit = "yr") = 4000 "Year of policy change";
        parameter Real t_fcaor_time(unit = "yr") = 4000 "Year of capital allocation to resource use efficiency";
        parameter Real tech_dev_del_TDD(unit = "yr") = 20 "Technology development time";
        parameter Real p_fr_cap_al_obt_res_2[:] = {1,0.2,0.1,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05} "Non-renewable resource fraction remaining";
        parameter Real p_res_tech_chg_mlt[:] = {0,0,0,0} "Resource technology change multiplier";
        SystemDynamics.Levels.Level1b NR_Resources(x0 = nr_resources_init) "p.387 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {40.0,100.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Rates.RRate NR_Res_Use_Rate "p.389 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {100.0,100.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink1 annotation(Placement(visible = true, transformation(origin = {150.0,100.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_3 NR_Res_Use_Rt "p.389 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {100.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_NR_Res_Use_Fact S_NR_Res_Use_Fact(t_policy_year = t_policy_year, p_nr_res_use_fact_1 = p_nr_res_use_fact_1) "p.390 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {69.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.SMTH3 P_Nr_Res_Use_Fact_2(averaging_time(unit = "yr") = tech_dev_del_TDD) annotation(Placement(visible = true, transformation(origin = {14.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Levels.Level1a Res_Tech_NRTD(x0 = res_tech_init) annotation(Placement(visible = true, transformation(origin = {-40.0,-18.0827}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 Res_Tech_Ch_Rt_NRATE annotation(Placement(visible = true, transformation(origin = {-100.0,-18.0827}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {-145.782,-18.0827}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Res_Tech_Ch_Rt_NRATE Res_Tech_Chg_Rt(t_policy_year = t_policy_year) annotation(Placement(visible = true, transformation(origin = {-100.0,-66.5489}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular P_Res_Tech_Chg_Mlt_NRCM(x_vals = {-10,-1,0,1}, y_vals = p_res_tech_chg_mlt) annotation(Placement(visible = true, transformation(origin = {-46.0,-130.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.P_Res_Tech_Chg P_Res_Tech_Chg(des_res_use_rt_DNRUR = des_res_use_rt_DNRUR) annotation(Placement(visible = true, transformation(origin = {77.3158,-130.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -180)));
        SystemDynamics.Functions.Tabular PC_Res_Use_Mlt(x_vals = 0:200:1600, y_vals = {0,0.85,2.6,3.4,3.8,4.1,4.4,4.7,5}) "p.390 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {46.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort ind_out_pc(unit = "dollar/yr") "Per capita annual industrial output" annotation(Placement(visible = true, transformation(origin = {-210.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort population "Population" annotation(Placement(visible = true, transformation(origin = {-210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort pc_res_use_mlt(unit = "ton/yr") "Per capita resource utilization" annotation(Placement(visible = true, transformation(origin = {210.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Gain NR_Res_Fr_Remain(k = 1 / nr_resources_init) "p.393 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-20.0,100.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_Fr_Cap_Al_Obt_Res_1(x_vals = 0:0.1:1, y_vals = {1,0.9,0.7,0.5,0.2,0.1,0.05,0.05,0.05,0.05,0.05}) "p.394 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-86.0,120.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_Fr_Cap_Al_Obt_Res_2(x_vals = 0:0.1:1, y_vals = p_fr_cap_al_obt_res_2) "p.394 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-86.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_Fr_Cap_Al_Obt_Res S_Fr_Cap_Al_Obt_Res(t_fcaor_time = t_fcaor_time) "p.393 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-159.0,100.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort s_fr_cap_al_obt_res "Fraction of capital allocated to resource use efficiency" annotation(Placement(visible = true, transformation(origin = {210.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular Ind_Cap_Out_Ratio_2(x_vals = {0,0.1,0.2,0.3,0.4,0.5,0.6,0.7,0.8,0.9,1,2}, y_vals = {3.75,3.6,3.47,3.36,3.25,3.16,3.1,3.06,3.02,3.01,3,3}) annotation(Placement(visible = true, transformation(origin = {131.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort ind_cap_out_ratio_2_ICOR2T "Industrial capital output ratio" annotation(Placement(visible = true, transformation(origin = {210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort industrial_output(unit = "dollar/yr") "Annual industrial output" annotation(Placement(visible = true, transformation(origin = {-210.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division Res_Intens annotation(Placement(visible = true, transformation(origin = {112.0,-68.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Interfaces.MassOutPort res_intens(unit = "ton/dollar") "Resource utilization intensity" annotation(Placement(visible = true, transformation(origin = {210.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      equation
        connect(P_Fr_Cap_Al_Obt_Res_1.y,S_Fr_Cap_Al_Obt_Res.u1) annotation(Line(visible = true, points = {{-97.0,120.0},{-128.0,120.0},{-128.0,104.0},{-150.0,104.0}}, color = {0,0,191}));
        connect(P_Fr_Cap_Al_Obt_Res_2.y,S_Fr_Cap_Al_Obt_Res.u2) annotation(Line(visible = true, points = {{-97.0,80.0},{-128.0,80.0},{-128.0,96.0},{-150.0,96.0}}, color = {0,0,191}));
        connect(P_Res_Tech_Chg.y,P_Res_Tech_Chg_Mlt_NRCM.u) annotation(Line(visible = true, origin = {15.1579,-130.0}, points = {{53.1579,0.0},{-53.1579,0.0}}, color = {0,0,191}));
        connect(P_Res_Tech_Chg_Mlt_NRCM.y,Res_Tech_Chg_Rt.u1) annotation(Line(visible = true, origin = {-88.3333,-111.8496}, points = {{31.3333,-18.1504},{-15.6667,-18.1504},{-15.6667,36.3007}}, color = {0,0,191}));
        connect(industrial_output,Res_Intens.u2) annotation(Line(visible = true, points = {{-210.0,-80.0},{-180.0,-80.0},{-180.0,-106.0},{118.0,-106.0},{118.0,-80.0}}, color = {0,0,191}));
        connect(Res_Intens.y,res_intens) annotation(Line(visible = true, points = {{112.0,-57.0},{112.0,-40.0},{190.0,-40.0},{190.0,50.0},{210.0,50.0}}, color = {0,0,191}));
        connect(P_Res_Tech_Chg.u,Res_Intens.u1) annotation(Line(visible = true, origin = {99.4386,-113.3333}, points = {{-13.1228,-16.6667},{6.5614,-16.6667},{6.5614,33.3333}}, color = {0,0,191}));
        connect(Res_Tech_Ch_Rt_NRATE.y,Source1.MassInPort1) annotation(Line(visible = true, origin = {-121.141,-18.0827}, points = {{13.641,0.0},{-13.641,0.0}}, color = {95,0,191}));
        connect(Res_Tech_NRTD.u1,Res_Tech_Ch_Rt_NRATE.y1) annotation(Line(visible = true, origin = {-74.5,-18.0827}, points = {{18.0,0.0},{-18.0,0.0}}, color = {95,0,191}));
        connect(Res_Tech_Chg_Rt.y,Res_Tech_Ch_Rt_NRATE.u) annotation(Line(visible = true, origin = {-100.0,-45.6908}, points = {{0.0,-11.8581},{0.0,11.8581}}, color = {0,0,191}));
        connect(P_Nr_Res_Use_Fact_2.u,Res_Tech_NRTD.y1) annotation(Line(visible = true, points = {{3.0,-10.0},{-9.2,-10.0},{-9.2,-10.1},{-27.25,-9.8327}}, color = {0,0,191}));
        connect(Res_Tech_Chg_Rt.u2,Res_Tech_NRTD.y4) annotation(Line(visible = true, points = {{-96.0,-75.5489},{-96.0,-100.0},{-51.25,-100.0},{-51.25,-27.8327}}, color = {0,0,191}));
        connect(Ind_Cap_Out_Ratio_2.y,ind_cap_out_ratio_2_ICOR2T) annotation(Line(visible = true, points = {{142.0,-10.0},{184.25,-10.0},{184.25,-10.0},{210.0,-10.0}}, color = {0,0,191}));
        connect(Ind_Cap_Out_Ratio_2.u,S_NR_Res_Use_Fact.y) annotation(Line(visible = true, points = {{123.0,-10.0},{106.125,-10.0},{106.125,-10.0},{101.25,-10.0},{101.25,-10.0},{78.0,-10.0}}, color = {0,0,191}));
        connect(P_Nr_Res_Use_Fact_2.y,S_NR_Res_Use_Fact.u) annotation(Line(visible = true, points = {{25.0,-10.0},{37.95,-10.0},{37.95,-10.0},{60.0,-10.0}}, color = {0,0,191}));
        connect(S_NR_Res_Use_Fact.y,NR_Res_Use_Rt.u2) annotation(Line(visible = true, origin = {92.6667,11.0}, points = {{-14.6667,-21.0},{7.3333,-21.0},{7.3333,42.0}}, color = {0,0,191}));
        connect(NR_Res_Use_Rt.u3,population) annotation(Line(visible = true, points = {{107.0,60.0},{120.0,60.0},{120.0,30.0},{-180.0,30.0},{-180.0,-10.0},{-210.0,-10.0}}, color = {0,0,191}));
        connect(PC_Res_Use_Mlt.y,NR_Res_Use_Rt.u1) annotation(Line(visible = true, origin = {75.0,60.0}, points = {{-18.0,0.0},{18.0,0.0}}, color = {0,0,191}));
        connect(NR_Res_Use_Rt.y,P_Res_Tech_Chg.u) annotation(Line(visible = true, points = {{100.0,67.0},{100.0,80.0},{170.0,80.0},{170.0,-130.0},{86.3158,-130.0}}, color = {0,0,191}));
        connect(NR_Res_Use_Rt.y,NR_Res_Use_Rate.u) annotation(Line(visible = true, origin = {100.0,75.625}, points = {{0.0,-8.625},{0.0,8.625}}, color = {0,0,191}));
        connect(PC_Res_Use_Mlt.y,pc_res_use_mlt) annotation(Line(visible = true, points = {{57.0,60.0},{80.0,60.0},{80.0,20.0},{180.0,20.0},{180.0,110.0},{210.0,110.0}}, color = {0,0,191}));
        connect(PC_Res_Use_Mlt.u,ind_out_pc) annotation(Line(visible = true, origin = {-86.0,60.0}, points = {{124.0,0.0},{-124.0,0.0}}, color = {0,0,191}));
        connect(NR_Res_Use_Rate.y1,Sink1.MassInPort1) annotation(Line(visible = true, origin = {123.25,100.0}, points = {{-15.75,0.0},{15.75,0.0}}, color = {95,0,191}));
        connect(NR_Res_Use_Rate.y,NR_Resources.u1) annotation(Line(visible = true, origin = {74.5,100.0}, points = {{18.0,0.0},{-18.0,0.0}}, color = {95,0,191}));
        connect(NR_Res_Fr_Remain.u,NR_Resources.y) annotation(Line(visible = true, points = {{-8.0,100.0},{16.0,100.0},{16.0,108.25},{27.25,108.25}}, color = {0,0,191}));
        connect(NR_Res_Fr_Remain.y,P_Fr_Cap_Al_Obt_Res_2.u) annotation(Line(visible = true, points = {{-31.0,100.0},{-50.0,100.0},{-50.0,80.0},{-78.0,80.0}}, color = {0,0,191}));
        connect(NR_Res_Fr_Remain.y,P_Fr_Cap_Al_Obt_Res_1.u) annotation(Line(visible = true, points = {{-31.0,100.0},{-50.0,100.0},{-50.0,120.0},{-78.0,120.0}}, color = {0,0,191}));
        connect(S_Fr_Cap_Al_Obt_Res.y,s_fr_cap_al_obt_res) annotation(Line(visible = true, points = {{-168.0,100.0},{-190.0,100.0},{-190.0,-150.0},{180.0,-150.0},{180.0,-70.0},{210.0,-70.0}}, color = {0,0,191}));
        annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, fillColor = {170,255,170}, fillPattern = FillPattern.Solid, lineThickness = 1.0, extent = {{-100.0,-100.0},{100.0,100.0}}, radius = 0.5),Text(visible = true, origin = {0.0,50.0}, fillPattern = FillPattern.Solid, extent = {{-47.3724,-36.6918},{47.3724,36.6918}}, textString = "NR", fontSize = 102.0, fontName = "Arial"),Text(visible = true, origin = {-0.0,-49.1014}, fillPattern = FillPattern.Solid, extent = {{-88.3557,-20.8986},{88.3557,20.8986}}, textString = "Utilization", fontSize = 102.0, fontName = "Arial"),Text(visible = true, fillPattern = FillPattern.Solid, extent = {{-78.7713,-36.6918},{78.7713,36.6918}}, textString = "Resource", fontSize = 102.0, fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-200.0,-160.0},{200.0,140.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{99.0,79.0},{101.0,81.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{79.0,59.0},{81.0,61.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-51.0,99.0},{-49.0,101.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{99.0,-11.0},{101.0,-9.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{104.0,-131.0},{106.0,-129.0}})}), Documentation(info = "<html>
This model describes the utilization of non-recoverable natural resources.  In the model (created in the early 1970s), the non-recoverable resources considered are mostly metals.  They are thus measured in metric tons. <p>
 
From today's perspective, it might make more sense to concentrate on the remaining fossil fuels, which is not the approach that Meadows and his coworkers took.  Yet, the effects of the dwindling resources on the overall economy are comparable, whether we can no longer produce goods, because we lack the raw materials or because we lack the energy to do so, results ultimately in the same predicament.  Due to the laws of exponential growth, we are running out of all kind of natural resources (fossil fuels, minerals, fresh water) almost simultaneously. <p>
 
In the case of minerals, recovery is partly possible, as discarded materials can be recycled.  However doing so requires energy for the re-concentration of these scrap materials.  The materials themselves don't get used up.  They only get dissipated further and further, until their density is so low that they cannot be collected any longer within reasonable cost limits. <p>
 
In the case of fossil fuels, these truly get used up.  These resources are non-recoverable within human time constants.  They were developed over many millions of years and essentially represent \"fossil sunshine.\"  Humanity is using all of these resources up within the very short time span (in geological terms) of a few hundred years.  <font color=red><b>Peak Oil</b></font>, i.e., the time when we shall have used up 50% of the available oil, and when supply can no longer keep up with demand, is now just around the corner. <p>
 
The fantastic recent developments of technology and the medical sciences, accompanied by an unprecedented growth of human population, would not have been possible without these resources, and won't be maintainable, once they shall have been used up. <p>
 
Whereas the earlier <font color=red><b>WORLD2</b></font> model only accounted for the dwindling resources themselves, the newer <font color=red><b>WORLD3</b></font> model offers a second state variable representing technological change.  Through more advanced technology, it is possible to use the available resources more efficiently, and that effect is accounted for by the second state variable in the model. <p>
 
In the <font color=red><b>WORLD3</b></font> model, the natural resources themselves are measured in metric tons, whereas the technology change has no units.
</html>", revisions = ""));
      end NR_Resource_Utilization;
      block Pollution_Dynamics "Pollution dynamics"
        parameter Real agr_mtl_toxic_index(unit = "1/dollar") = 1 "Agricultural materials toxicity index";
        parameter Real assim_half_life_1970(unit = "yr") = 1.5 "Pollution assimilation half life in 1970";
        parameter Real des_ppoll_index_DPOLX = 1.2 "Desired persistent pollution index";
        parameter Real fr_agr_inp_pers_mtl = 0.001 "Effective fraction of agricultural pollution input";
        parameter Real frac_res_pers_mtl = 0.02 "Effective fraction of resource utilization on pollution generation";
        parameter Real ind_mtl_emiss_fact(unit = "1/ton") = 0.1 "Industrial materials emission factor";
        parameter Real ind_mtl_toxic_index = 10.0 "Industrial materials toxicity index";
        parameter Real ind_out_in_1970(unit = "dollar/yr") = 790000000000.0 "Industrial output in 1970";
        parameter Real p_ppoll_gen_fact_1 = 1 "Default persistent pollution generation factor";
        parameter Real pers_pollution_init = 25000000.0 "Initial persistent pollution";
        parameter Real ppoll_in_1970 = 136000000.0 "Persistent pollution in 1970";
        parameter Real ppoll_tech_init = 1 "Initial persistent pollution technology change factor";
        parameter Real ppoll_trans_del(unit = "yr") = 20 "Persistent pollution transmission delay";
        parameter Real t_air_poll_time(unit = "yr") = 4000 "Air pollution change time";
        parameter Real t_policy_year(unit = "yr") = 4000 "Year of policy change";
        parameter Real tech_dev_del_TDD(unit = "yr") = 20 "Technology development time";
        parameter Real p_ppoll_tech_chg_mlt[:] = {0,0,0,0} "Persistent pollution technology change multiplier";
        SystemDynamics.Levels.Level Pers_Pollution(x0 = pers_pollution_init) "p.440 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-10.0,-10.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Rates.RRate PPoll_Appear_Rt "p.435 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-70.0,-10.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Rates.RRate PPoll_Assim_Rt "p.442 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {50.0,-10.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {-110.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink1 annotation(Placement(visible = true, transformation(origin = {90.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.SMTH3 PPoll_Appear_Rate(smooth_init(unit = "1/yr") = 0, averaging_time(unit = "yr") = ppoll_trans_del) "p.435 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-70.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
        SystemDynamics.WorldDynamics.World3.Utilities.PPoll_Assim_Rt PPoll_Ass_Rt "p.442 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {50.0,-44.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular Assim_Half_Life_Mlt(x_vals = (-249):250:1001, y_vals = {1,1,11,21,31,41}) "p.453 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {17.0,-112.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Gain PPoll_Index(k = 1 / ppoll_in_1970) "p.441 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-32.0,-112.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.PPoll_Gen_Rt PPoll_Gen_Rt "p.428 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-98.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.PPoll_Gen_Agr PPoll_Gen_Agr(agr_mtl_toxic_index = agr_mtl_toxic_index, fr_agr_inp_pers_mtl = fr_agr_inp_pers_mtl) "p.433 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-147.0,-120.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.PPoll_Gen_Ind PPoll_Gen_Ind(frac_res_pers_mtl = frac_res_pers_mtl, ind_mtl_emiss_fact = ind_mtl_emiss_fact, ind_mtl_toxic_index = ind_mtl_toxic_index) "p.429 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-155.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Levels.Level1a PPoll_Tech_PTD(x0 = ppoll_tech_init) annotation(Placement(visible = true, transformation(origin = {140.0,100.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Rates.Rate_1 PPoll_Tech_Chg_Rt annotation(Placement(visible = true, transformation(origin = {80.0,100.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Sources.Source Source2 annotation(Placement(visible = true, transformation(origin = {30.0,100.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.PPoll_Tech_Chg_Rt PPoll_Tech_Ch_Rt(t_policy_year = t_policy_year) annotation(Placement(visible = true, transformation(origin = {80.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular P_PPoll_Tech_Chg_mlt_POLGFM(y_vals = p_ppoll_tech_chg_mlt, x_vals = {-100,-1,0,1}) annotation(Placement(visible = true, transformation(origin = {23.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.P_PPoll_Tech_Chg P_PPoll_Tech_Chg(des_ppoll_index_DPOLX = des_ppoll_index_DPOLX) annotation(Placement(visible = true, transformation(origin = {-40.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.SMTH3 P_PPoll_Gen_Fact_2(averaging_time(unit = "yr") = tech_dev_del_TDD) annotation(Placement(visible = true, transformation(origin = {-30.0,90.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_PPoll_Gen_Fact S_PPoll_Gen_Fact(t_policy_year = t_policy_year, p_ppoll_gen_fact_1 = p_ppoll_gen_fact_1) "p.428 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-82.0,90.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort population "Population" annotation(Placement(visible = true, transformation(origin = {-210.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort pc_res_use_mlt(unit = "ton/yr") "Per capita resource utilization" annotation(Placement(visible = true, transformation(origin = {-210.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort arable_land(unit = "hectare") "Arable land" annotation(Placement(visible = true, transformation(origin = {-210.0,-130.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort agr_inp_per_hect(unit = "dollar/(hectare.yr)") "Agricultural investments per hectare" annotation(Placement(visible = true, transformation(origin = {-210.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular PPoll_Tech_Mult_Icor_COPM(x_vals = {0,0.1,0.2,0.3,0.4,0.5,0.6,0.7,0.8,0.9,1,2}, y_vals = {1.25,1.2,1.15,1.11,1.08,1.05,1.03,1.02,1.01,1,1,1}) annotation(Placement(visible = true, transformation(origin = {-81.0,120.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort ppoll_tech_mult_icor_COPM "Technology induced reduction in persistent pollution release" annotation(Placement(visible = true, transformation(origin = {210.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort ppoll_index "Persistent pollution index" annotation(Placement(visible = true, transformation(origin = {210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort ppoll_gen_rt(unit = "1/yr") "Persistent pollution generation rate" annotation(Placement(visible = true, transformation(origin = {210.0,-130.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort industrial_output(unit = "dollar/yr") "Annual industrial output" annotation(Placement(visible = true, transformation(origin = {-210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Poll_Intens_Ind Poll_Intens_Ind annotation(Placement(visible = true, transformation(origin = {-150.0,67.6993}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Interfaces.MassOutPort poll_intens_ind(unit = "1/dollar") "Persistent pollution intensity index" annotation(Placement(visible = true, transformation(origin = {210.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Gain Gain1(k = 1 / ind_out_in_1970) annotation(Placement(visible = true, transformation(origin = {-103.0,-140.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_Yield_Mlt_Air_Poll_1(x_vals = 0:10:30, y_vals = {1,1,0.7,0.4}) "p.310 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {104.0,-48.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_Yield_Mlt_Air_Poll_2(x_vals = 0:10:30, y_vals = {1,1,0.98,0.95}) "p.310 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {104.0,-90.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_Yield_Mlt_Air_Poll S_Yield_Mlt_Air_Poll(t_air_poll_time = t_air_poll_time) "p.310 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {170.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort s_yield_mlt_air_poll "Land yield multiplier to air pollution release" annotation(Placement(visible = true, transformation(origin = {210.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-40.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Gain Assim_Half_Life(k = assim_half_life_1970) "p.453 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {56.0,-84.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
      equation
        connect(S_Yield_Mlt_Air_Poll.u1,P_Yield_Mlt_Air_Poll_1.y) annotation(Line(visible = true, points = {{161.0,-66.0},{140.0,-66.0},{140.0,-48.0},{115.0,-48.0}}, color = {0,0,191}));
        connect(P_Yield_Mlt_Air_Poll_2.y,S_Yield_Mlt_Air_Poll.u2) annotation(Line(visible = true, points = {{115.0,-90.0},{140.0,-90.0},{140.0,-74.0},{161.0,-74.0}}, color = {0,0,191}));
        connect(arable_land,PPoll_Gen_Agr.u2) annotation(Line(visible = true, points = {{-210.0,-130.0},{-180.0,-130.0},{-180.0,-124.0},{-156.0,-124.0}}, color = {0,0,191}));
        connect(agr_inp_per_hect,PPoll_Gen_Agr.u1) annotation(Line(visible = true, points = {{-210.0,-70.0},{-190.0,-70.0},{-190.0,-116.0},{-156.0,-116.0}}, color = {0,0,191}));
        connect(PPoll_Gen_Ind.u2,population) annotation(Line(visible = true, points = {{-164.0,-84.0},{-180.0,-84.0},{-180.0,50.0},{-210.0,50.0}}, color = {0,0,191}));
        connect(PPoll_Gen_Ind.u1,pc_res_use_mlt) annotation(Line(visible = true, points = {{-164.0,-76.0},{-176.0,-76.0},{-176.0,-74.0},{-176.0,110.0},{-210.0,110.0}}, color = {0,0,191}));
        connect(S_PPoll_Gen_Fact.y,PPoll_Gen_Rt.u1) annotation(Line(visible = true, points = {{-91.0,90.0},{-126.0,90.0},{-126.0,-74.0},{-107.0,-74.0}}, color = {0,0,191}));
        connect(PPoll_Gen_Agr.y,PPoll_Gen_Rt.u3) annotation(Line(visible = true, points = {{-138.0,-120.0},{-126.0,-120.0},{-126.0,-86.0},{-107.0,-86.0}}, color = {0,0,191}));
        connect(Assim_Half_Life.y,PPoll_Ass_Rt.u2) annotation(Line(visible = true, points = {{56.0,-95.0},{56.0,-65.1},{54.0,-65.1},{54.0,-53.0}}, color = {0,0,191}));
        connect(PPoll_Ass_Rt.u1,Pers_Pollution.y2) annotation(Line(visible = true, points = {{46.0,-53.0},{46.0,-64.0},{1.25,-64.0},{1.25,-19.75}}, color = {0,0,191}));
        connect(S_Yield_Mlt_Air_Poll.y,s_yield_mlt_air_poll) annotation(Line(visible = true, origin = {194.5,-70.0}, points = {{-15.5,0.0},{15.5,0.0}}, color = {0,0,191}));
        connect(Gain1.y,P_Yield_Mlt_Air_Poll_2.u) annotation(Line(visible = true, points = {{-92.0,-140.0},{76.0,-140.0},{76.0,-90.0},{96.0,-90.0}}, color = {0,0,191}));
        connect(Gain1.y,P_Yield_Mlt_Air_Poll_1.u) annotation(Line(visible = true, points = {{-92.0,-140.0},{76.0,-140.0},{76.0,-48.0},{96.0,-48.0}}, color = {0,0,191}));
        connect(PPoll_Ass_Rt.y,PPoll_Assim_Rt.u) annotation(Line(visible = true, origin = {50.0,-30.375}, points = {{0.0,-4.625},{0.0,4.625}}, color = {0,0,191}));
        connect(Assim_Half_Life_Mlt.y,Assim_Half_Life.u) annotation(Line(visible = true, points = {{28.0,-112.0},{56.0,-112.0},{56.0,-72.0}}, color = {0,0,191}));
        connect(PPoll_Index.y,Assim_Half_Life_Mlt.u) annotation(Line(visible = true, origin = {-6.0,-112.0}, points = {{-15.0,0.0},{15.0,0.0}}, color = {0,0,191}));
        connect(PPoll_Index.y,ppoll_index) annotation(Line(visible = true, points = {{-21.0,-112.0},{-10.0,-112.0},{-10.0,-126.0},{72.0,-126.0},{72.0,12.0},{110.0,12.0},{110.0,-10.0},{210.0,-10.0}}, color = {0,0,191}));
        connect(PPoll_Index.y,P_PPoll_Tech_Chg.u) annotation(Line(visible = true, points = {{-21.0,-112.0},{-10.0,-112.0},{-10.0,-126.0},{72.0,-126.0},{72.0,12.0},{-70.0,12.0},{-70.0,30.0},{-49.0,30.0}}, color = {0,0,191}));
        connect(PPoll_Index.u,Pers_Pollution.y3) annotation(Line(visible = true, points = {{-44.0,-112.0},{-56.0,-112.0},{-56.0,-72.0},{-10.0,-72.0},{-10.0,-19.75}}, color = {0,0,191}));
        connect(PPoll_Gen_Rt.y,ppoll_gen_rt) annotation(Line(visible = true, points = {{-89.0,-80.0},{-70.0,-80.0},{-70.0,-130.0},{210.0,-130.0}}, color = {0,0,191}));
        connect(PPoll_Gen_Ind.y,PPoll_Gen_Rt.u2) annotation(Line(visible = true, origin = {-126.5,-80.0}, points = {{-19.5,0.0},{19.5,0.0}}, color = {0,0,191}));
        connect(PPoll_Gen_Rt.y,PPoll_Appear_Rate.u) annotation(Line(visible = true, points = {{-89.0,-80.0},{-70.0,-80.0},{-70.0,-39.0}}, color = {0,0,191}));
        connect(Gain1.u,industrial_output) annotation(Line(visible = true, points = {{-115.0,-140.0},{-184.0,-140.0},{-184.0,-10.0},{-210.0,-10.0}}, color = {0,0,191}));
        connect(PPoll_Gen_Ind.y,Poll_Intens_Ind.u2) annotation(Line(visible = true, points = {{-146.0,-80.0},{-134.0,-80.0},{-134.0,-40.0},{-150.0,-40.0},{-150.0,58.6993}}, color = {0,0,191}));
        connect(PPoll_Appear_Rate.y,PPoll_Appear_Rt.u) annotation(Line(visible = true, points = {{-70.0,-61.0},{-70.0,-25.75}}, color = {0,0,191}));
        connect(PPoll_Assim_Rt.y,Pers_Pollution.u2) annotation(Line(visible = true, points = {{42.5,-10.0},{24.9,-10.0},{24.9,-10.0},{6.5,-10.0}}, color = {95,0,191}));
        connect(Sink1.MassInPort1,PPoll_Assim_Rt.y1) annotation(Line(visible = true, points = {{79.0,-10.0},{69.5,-10.0},{69.5,-10.0},{57.5,-10.0}}, color = {95,0,191}));
        connect(Source1.MassInPort1,PPoll_Appear_Rt.y) annotation(Line(visible = true, points = {{-99.0,-10.0},{-89.5,-10.0},{-89.5,-10.0},{-77.5,-10.0}}, color = {95,0,191}));
        connect(PPoll_Appear_Rt.y1,Pers_Pollution.u1) annotation(Line(visible = true, points = {{-62.5,-10.0},{-44.9,-10.0},{-44.9,-10.0},{-26.5,-10.0}}, color = {95,0,191}));
        connect(P_PPoll_Tech_Chg.y,P_PPoll_Tech_Chg_mlt_POLGFM.u) annotation(Line(visible = true, origin = {-8.0,30.0}, points = {{-23.0,0.0},{23.0,0.0}}, color = {0,0,191}));
        connect(PPoll_Tech_Ch_Rt.u1,P_PPoll_Tech_Chg_mlt_POLGFM.y) annotation(Line(visible = true, origin = {62.0,37.0}, points = {{14.0,14.0},{14.0,-7.0},{-28.0,-7.0}}, color = {0,0,191}));
        connect(PPoll_Tech_Ch_Rt.u2,PPoll_Tech_PTD.y4) annotation(Line(visible = true, points = {{84.0,51.0},{83.9764,35.6612},{83.9764,30.0},{128.75,30.0},{128.75,90.25}}, color = {0,0,191}));
        connect(P_PPoll_Gen_Fact_2.u,PPoll_Tech_PTD.y) annotation(Line(visible = true, points = {{-19.0,90.0},{0.0,90.0},{0.0,120.0},{110.0,120.0},{110.0,108.25},{127.25,108.25}}, color = {0,0,191}));
        connect(PPoll_Tech_Chg_Rt.y1,PPoll_Tech_PTD.u1) annotation(Line(visible = true, origin = {105.5,100.0}, points = {{-18.0,0.0},{18.0,0.0}}, color = {95,0,191}));
        connect(PPoll_Tech_Ch_Rt.y,PPoll_Tech_Chg_Rt.u) annotation(Line(visible = true, origin = {80.0,76.625}, points = {{0.0,-7.625},{0.0,7.625}}, color = {0,0,191}));
        connect(Source2.MassInPort1,PPoll_Tech_Chg_Rt.y) annotation(Line(visible = true, origin = {56.75,100.0}, points = {{-15.75,0.0},{15.75,0.0}}, color = {95,0,191}));
        connect(S_PPoll_Gen_Fact.u,P_PPoll_Gen_Fact_2.y) annotation(Line(visible = true, origin = {-57.0,90.0}, points = {{-16.0,0.0},{16.0,0.0}}, color = {0,0,191}));
        connect(S_PPoll_Gen_Fact.y,Poll_Intens_Ind.u3) annotation(Line(visible = true, points = {{-91.0,90.0},{-126.0,90.0},{-126.0,-10.0},{-144.0,-10.0},{-144.0,58.6993}}, color = {0,0,191}));
        connect(S_PPoll_Gen_Fact.y,PPoll_Tech_Mult_Icor_COPM.u) annotation(Line(visible = true, points = {{-91.0,90.0},{-126.0,90.0},{-126.0,120.0},{-89.0,120.0}}, color = {0,0,191}));
        connect(PPoll_Tech_Mult_Icor_COPM.y,ppoll_tech_mult_icor_COPM) annotation(Line(visible = true, points = {{-70.0,120.0},{-20.0,120.0},{-20.0,130.0},{180.0,130.0},{180.0,50.0},{210.0,50.0}}, color = {0,0,191}));
        connect(Poll_Intens_Ind.u1,industrial_output) annotation(Line(points = {{-156.0,58.6993},{-156.0,-10.0},{-210.0,-10.0}}, color = {0,0,191}, visible = true));
        connect(Poll_Intens_Ind.y,poll_intens_ind) annotation(Line(points = {{-150.0,76.6993},{-150.0,140.0},{190.0,140.0},{190.0,110.0},{210.0,110.0}}, color = {0,0,191}, visible = true));
        annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, fillColor = {170,255,170}, fillPattern = FillPattern.Solid, lineThickness = 1.0, extent = {{-100.0,-100.0},{100.0,100.0}}, radius = 0.5),Text(visible = true, origin = {0.2251,30.0}, fillPattern = FillPattern.Solid, extent = {{-69.7749,-36.6918},{69.7749,36.6918}}, textString = "Pollution", fontSize = 102.0, fontName = "Arial"),Text(visible = true, origin = {-0.3752,-30.0}, fillPattern = FillPattern.Solid, extent = {{-79.6248,-36.6918},{79.6248,36.6918}}, textString = "Dynamics", fontSize = 102.0, fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-200.0,-160.0},{200.0,140.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-11.0,-113.0},{-9.0,-111.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-127.0,89.0},{-125.0,91.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-71.0,-81.0},{-69.0,-79.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{71.0,11.0},{73.0,13.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-135.0,-81.0},{-133.0,-79.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-127.0,-11.0},{-125.0,-9.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-185.0,-11.0},{-183.0,-9.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{75.0,-91.0},{77.0,-89.0}})}), Documentation(info = "<html>
This model describes the appearance of persistent pollutants.  These pollutants are an unwelcome yet inevitable byproduct of our industrial production.  Especially air pollution is problematic, because these pollutants are difficult to re-capture, and they remain in the air for a long time, somewhere in the order of 100 years.  These then lead to <font color=red><b>Global Warming</b></font>. <p>
 
Whereas the earlier <font color=red><b>WORLD2</b></font> model listed only the appearance and assimilation of pollution itself, the newer <font color=red><b>WORLD3</b></font> model accounts for the possibility of reducing the release of pollutants into the air by improved technology using a second state variable.  For example, it may be possible to burn coal in a power plant that captures the released CO<sub>2</sub> rather than releasing it into the atmosphere.  In this way, the greenhouse gas emissions could be significantly reduced, and thereby, the effects of global warming could be mitigated. <p>
 
Pollutants can take many forms.  For this reason, the <font color=red><b>WORLD3</b></font> model normalizes the pollutants and expresses them in \"units of pollution.\"  In accordance with engineering practice, normalized quantities are assumed to be dimensionless (the dimensions are filtered out in the process of normalization), and consequently, pollution in this model is assumed dimensionless.  The technology change factor is also dimensionless.
</html>"));
      end Pollution_Dynamics;
      block Population_Dynamics "Population dynamics"
        parameter Real pop1_init = 650000000.0 "Initial population 14 years and younger";
        parameter Real pop2_init = 700000000.0 "Initial population 15 to 44 years old";
        parameter Real pop3_init = 190000000.0 "Initial population 45 to 64 years old";
        parameter Real pop4_init = 60000000.0 "Initial population 65 years and older";
        parameter Real labor_force_partic = 0.75 "Percentage of participating labor force";
        parameter Real reproductive_lifetime(unit = "yr") = 30 "Reproductive life time";
        parameter Real t_pop_equil_time(unit = "yr") = 4000 "Population equilibrium time";
        SystemDynamics.Rates.RRate Births "p.96 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-140.0,0.0}, extent = {{-12.0,-12.0},{12.0,12.0}}, rotation = 0)));
        SystemDynamics.Sources.Source source annotation(Placement(visible = true, transformation(origin = {-170.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Levels.Level2b Pop_0_14(x0 = pop1_init) annotation(Placement(visible = true, transformation(origin = {-110.0,0.0}, extent = {{-12.0,-12.0},{12.0,12.0}}, rotation = 0)));
        SystemDynamics.Rates.RRate Matur_14 "p.141 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-70.0,5.0}, extent = {{-12.0,-12.0},{12.0,12.0}}, rotation = 0)));
        SystemDynamics.Levels.Level2b Pop_15_44(x0 = pop2_init) "p.57 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-39.0,5.0}, extent = {{-12.0,-12.0},{12.0,12.0}}, rotation = 0)));
        SystemDynamics.Rates.RRate Matur_44 "p.141 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {0.0,10.0}, extent = {{-12.0,-12.0},{12.0,12.0}}, rotation = 0)));
        SystemDynamics.Levels.Level2b Pop_45_64(x0 = pop3_init) "p.57 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {30.0,10.0}, extent = {{-12.0,-12.0},{12.0,12.0}}, rotation = 0)));
        SystemDynamics.Rates.RRate Matur_64 "p.141 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {70.0,15.0}, extent = {{-12.0,-12.0},{12.0,12.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Pop_65plus(x0 = pop4_init) "p.57 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {102.0,15.0}, extent = {{-12.0,-12.0},{12.0,12.0}}, rotation = 0)));
        SystemDynamics.Rates.RRate Deaths_65p "p.57 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {132.0,15.0}, extent = {{-12.0,-12.0},{12.0,12.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink sink annotation(Placement(visible = true, transformation(origin = {160.0,15.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Birth_Factors births(Repro_Life = reproductive_lifetime, t_pop_equil_time = t_pop_equil_time) "p.96 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-140.0,-29.0}, extent = {{-10.0,-11.0},{10.0,11.0}}, rotation = -270)));
        SystemDynamics.Rates.RRate Deaths_0_14 "p.57 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-90.0,-60.0}, extent = {{-12.0,-12.0},{12.0,12.0}}, rotation = -270)));
        SystemDynamics.Sources.Sink sink1 annotation(Placement(visible = true, transformation(origin = {-80.0,-90.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
        SystemDynamics.Rates.RRate Deaths_15_44 "p.57 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-20.0,-60.0}, extent = {{-12.0,-12.0},{12.0,12.0}}, rotation = -270)));
        SystemDynamics.Sources.Sink sink2 annotation(Placement(visible = true, transformation(origin = {-10.0,-90.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
        SystemDynamics.Rates.RRate Deaths_45_64 "p.57 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {50.0,-60.0}, extent = {{-12.0,-12.0},{12.0,12.0}}, rotation = -270)));
        SystemDynamics.Sources.Sink sink3 annotation(Placement(visible = true, transformation(origin = {60.0,-90.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -90)));
        SystemDynamics.WorldDynamics.World3.Utilities.Matur_Factors matur_14(bracket = 15) "p.141 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-70.0,-25.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.WorldDynamics.World3.Utilities.Matur_Factors matur_44(bracket = 30) "p.141 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {0.0,-19.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.WorldDynamics.World3.Utilities.Matur_Factors matur_64(bracket = 20) "p.141 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {70.0,-15.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.WorldDynamics.World3.Utilities.Death_Factors deaths_0_14 "p.57 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-110.0,-82.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.WorldDynamics.World3.Utilities.Death_Factors deaths_15_44 "p.57 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-40.0,-82.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.WorldDynamics.World3.Utilities.Death_Factors deaths_45_64 "p.57 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {30.0,-82.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.WorldDynamics.World3.Utilities.Death_Factors deaths_65p "p.57 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {132.0,-23.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular Mort_0_14(x_vals = 20:10:90, y_vals = {0.0567,0.0366,0.0243,0.0155,0.0082,0.0023,0.001,0.001}) "p.57 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-70.0,-124.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular Mort_15_44(x_vals = 20:10:90, y_vals = {0.0266,0.0171,0.011,0.0065,0.004,0.0016,0.0008,0.0008}) "p.57 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {0.0,-124.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular Mort_45_64(x_vals = 20:10:90, y_vals = {0.0562,0.0373,0.0252,0.0171,0.0118,0.0083,0.006,0.006}) "p.57 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {70.0,-124.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Functions.Tabular Mort_65p(x_vals = 20:10:90, y_vals = {0.13,0.11,0.09,0.07,0.06,0.05,0.04,0.04}) "p.57 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {138.0,-77.2554}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.Interfaces.MassInPort life_expectancy(unit = "yr") "Average life expectancy of human population" annotation(Placement(visible = true, transformation(origin = {-210.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Add Pop_Young annotation(Placement(visible = true, transformation(origin = {-150.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Add Pop_Old annotation(Placement(visible = true, transformation(origin = {-150.0,80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Add Population annotation(Placement(visible = true, transformation(origin = {-112.0,94.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Add Death_Young annotation(Placement(visible = true, transformation(origin = {149.0,110.0}, extent = {{11.0,-10.0},{-11.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Add Death_Old annotation(Placement(visible = true, transformation(origin = {149.0,80.0}, extent = {{11.0,-10.0},{-11.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Add Deaths annotation(Placement(visible = true, transformation(origin = {110.0,94.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Add Work_Pop "p.241 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {50.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Gain Labor_Force(k = labor_force_partic) "p.241 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {96.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort labor_force "Total human labor force" annotation(Placement(visible = true, transformation(origin = {210.0,-80.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.BD_Rates Birth_Rate "p.97 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-73.0,99.0}, extent = {{-13.0,-13.0},{13.0,13.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.BD_Rates Death_Rate "p.140 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {68.0,99.0}, extent = {{14.0,-13.0},{-14.0,13.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort total_fertility "Average human fertility" annotation(Placement(visible = true, transformation(origin = {-210.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.Interfaces.MassOutPort population "Population" annotation(Placement(visible = true, transformation(origin = {210.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      equation
        connect(Labor_Force.y,labor_force) annotation(Line(visible = true, points = {{107.0,50.0},{188.0,50.0},{188.0,-80.0},{210.0,-80.0}}, color = {0,0,191}));
        connect(Labor_Force.u,Work_Pop.y) annotation(Line(visible = true, origin = {72.5,50.0}, points = {{11.5,0.0},{-11.5,0.0}}, color = {0,0,191}));
        connect(matur_64.u1,Pop_45_64.y1) annotation(Line(visible = true, points = {{66.0,-24.0},{66.0,-30.0},{39.0,-30.0},{39.0,2.2}}, color = {0,0,191}));
        connect(matur_44.u1,Pop_15_44.y1) annotation(Line(visible = true, points = {{-4.0,-28.0},{-4.0,-34.0},{-29.25,-34.0},{-30.0,-2.8}}, color = {0,0,191}));
        connect(matur_14.u1,Pop_0_14.y1) annotation(Line(visible = true, points = {{-74.0,-34.0},{-74.0,-40.0},{-101.0,-40.0},{-101.0,-7.8}}, color = {0,0,191}));
        connect(deaths_0_14.u1,Pop_0_14.y2) annotation(Line(visible = true, points = {{-114.0,-91.0},{-114.0,-104.0},{-126.0,-104.0},{-126.0,-48.0},{-110.0,-48.0},{-110.0,-7.8}}, color = {0,0,191}));
        connect(Mort_0_14.y,deaths_0_14.u2) annotation(Line(visible = true, points = {{-70.0,-113.0},{-70.0,-104.0},{-106.0,-104.0},{-106.0,-91.0}}, color = {0,0,191}));
        connect(deaths_15_44.u1,Pop_15_44.y2) annotation(Line(visible = true, points = {{-44.0,-91.0},{-44.0,-104.0},{-56.0,-104.0},{-56.0,-48.0},{-39.0,-48.0},{-39.0,-2.8}}, color = {0,0,191}));
        connect(Mort_15_44.y,deaths_15_44.u2) annotation(Line(visible = true, points = {{0.0,-113.0},{0.0,-104.0},{-36.0,-104.0},{-36.0,-91.0}}, color = {0,0,191}));
        connect(deaths_45_64.u1,Pop_45_64.y2) annotation(Line(visible = true, points = {{26.0,-91.0},{26.0,-104.0},{14.0,-104.0},{14.0,-30.0},{30.0,-30.0},{30.0,2.2}}, color = {0,0,191}));
        connect(Mort_45_64.y,deaths_45_64.u2) annotation(Line(visible = true, points = {{70.0,-113.0},{70.0,-104.0},{34.0,-104.0},{34.0,-91.0}}, color = {0,0,191}));
        connect(Work_Pop.u2,Pop_45_64.y) annotation(Line(visible = true, points = {{38.0,44.0},{14.0,44.0},{14.0,16.6},{19.8,16.6}}, color = {0,0,191}));
        connect(Pop_Old.u1,Pop_45_64.y) annotation(Line(visible = true, points = {{-162.0,86.0},{-172.0,86.0},{-172.0,28.0},{14.0,28.0},{14.0,16.6},{19.8,16.6}}, color = {0,0,191}));
        connect(Pop_45_64.u3,Deaths_45_64.y) annotation(Line(visible = true, origin = {47.7333,-18.5333}, points = {{-4.5333,23.7333},{2.2667,23.7333},{2.2667,-47.4667}}, color = {191,0,191}));
        connect(Pop_45_64.u2,Matur_64.y) annotation(Line(visible = true, points = {{43.2,14.8},{52.6,14.8},{52.6,15.0},{64.0,15.0}}, color = {191,0,191}));
        connect(Matur_44.y1,Pop_45_64.u1) annotation(Line(visible = true, origin = {11.4,10.0}, points = {{-5.4,0.0},{5.4,0.0}}, color = {191,0,191}));
        connect(deaths_65p.u1,Pop_65plus.y2) annotation(Line(visible = true, points = {{128.0,-32.0},{128.0,-40.0},{111.0,-40.0},{111.0,7.2}}, color = {0,0,191}));
        connect(Pop_Old.u2,Pop_65plus.y) annotation(Line(visible = true, points = {{-162.0,74.0},{-168.0,74.0},{-168.0,32.0},{84.0,32.0},{84.0,22.15},{91.8,21.6}}, color = {0,0,191}));
        connect(Pop_65plus.u2,Deaths_65p.y) annotation(Line(visible = true, origin = {120.6,15.0}, points = {{-5.4,0.0},{5.4,0.0}}, color = {191,0,191}));
        connect(Matur_64.y1,Pop_65plus.u1) annotation(Line(visible = true, origin = {82.4,15.0}, points = {{-6.4,0.0},{6.4,0.0}}, color = {191,0,191}));
        connect(matur_64.y,Matur_64.u) annotation(Line(visible = true, origin = {70.0,-1.8}, points = {{0.0,-4.2},{0.0,4.2}}, color = {0,0,191}));
        connect(Matur_14.y1,Pop_15_44.u1) annotation(Line(visible = true, origin = {-58.1,5.0}, points = {{-5.9,0.0},{5.9,0.0}}, color = {191,0,191}));
        connect(Work_Pop.u1,Pop_15_44.y) annotation(Line(visible = true, points = {{38.0,56.0},{-56.0,56.0},{-56.0,12.15},{-49.2,11.6}}, color = {0,0,191}));
        connect(Pop_Young.u2,Pop_15_44.y) annotation(Line(visible = true, points = {{-162.0,104.0},{-176.0,104.0},{-176.0,24.0},{-56.0,24.0},{-56.0,12.15},{-49.2,11.6}}, color = {0,0,191}));
        connect(births.u3,Pop_15_44.y3) annotation(Line(visible = true, points = {{-133.4,-38.0},{-133.4,-44.0},{-48.75,-44.0},{-48.0,-2.8}}, color = {0,0,191}));
        connect(Pop_15_44.u3,Deaths_15_44.y) annotation(Line(visible = true, origin = {-21.9333,-21.8667}, points = {{-3.8667,22.0667},{1.9333,22.0667},{1.9333,-44.1333}}, color = {191,0,191}));
        connect(Pop_15_44.u2,Matur_44.y) annotation(Line(visible = true, points = {{-25.8,9.8},{-17.4,10.2},{-17.4,10.0},{-6.0,10.0}}, color = {191,0,191}));
        connect(matur_14.y,Matur_14.u) annotation(Line(visible = true, origin = {-70.0,-11.8}, points = {{0.0,-4.2},{0.0,4.2}}, color = {0,0,191}));
        connect(Pop_0_14.u2,Matur_14.y) annotation(Line(visible = true, points = {{-96.8,4.8},{-88.4,4.8},{-88.4,5.0},{-76.0,5.0}}, color = {191,0,191}));
        connect(Deaths_65p.y1,sink.MassInPort1) annotation(Line(visible = true, origin = {143.5,15.0}, points = {{-5.5,0.0},{5.5,0.0}}, color = {191,0,191}));
        connect(deaths_65p.y,Deaths_65p.u) annotation(Line(visible = true, origin = {132.0,-5.8}, points = {{0.0,-8.2},{0.0,8.2}}, color = {0,0,191}));
        connect(deaths_45_64.y,Deaths_45_64.u) annotation(Line(visible = true, origin = {40.8667,-64.3333}, points = {{-10.8667,-8.6667},{-10.8667,4.3333},{21.7333,4.3333}}, color = {0,0,191}));
        connect(Deaths_45_64.y1,sink3.MassInPort1) annotation(Line(visible = true, origin = {55.0,-71.25}, points = {{-5.0,17.25},{-5.0,-4.75},{5.0,-4.75},{5.0,-7.75}}, color = {191,0,191}));
        connect(deaths_45_64.y,Death_Old.u1) annotation(Line(visible = true, points = {{30.0,-73.0},{30.0,-42.0},{176.0,-42.0},{176.0,86.0},{162.2,86.0}}, color = {0,0,191}));
        connect(deaths_15_44.y,Death_Young.u2) annotation(Line(visible = true, points = {{-40.0,-73.0},{-40.0,-60.0},{-60.0,-60.0},{-60.0,-144.0},{180.0,-144.0},{180.0,104.0},{162.2,104.0}}, color = {0,0,191}));
        connect(deaths_15_44.y,Deaths_15_44.u) annotation(Line(visible = true, origin = {-29.1333,-64.3333}, points = {{-10.8667,-8.6667},{-10.8667,4.3333},{21.7333,4.3333}}, color = {0,0,191}));
        connect(Deaths_15_44.y1,sink2.MassInPort1) annotation(Line(visible = true, origin = {-15.0,-71.25}, points = {{-5.0,17.25},{-5.0,-4.75},{5.0,-4.75},{5.0,-7.75}}, color = {191,0,191}));
        connect(deaths_0_14.y,Deaths_0_14.u) annotation(Line(visible = true, origin = {-99.1333,-64.3333}, points = {{-10.8667,-8.6667},{-10.8667,4.3333},{21.7333,4.3333}}, color = {0,0,191}));
        connect(Deaths_0_14.y1,sink1.MassInPort1) annotation(Line(visible = true, origin = {-85.0,-71.25}, points = {{-5.0,17.25},{-5.0,-4.75},{5.0,-4.75},{5.0,-7.75}}, color = {191,0,191}));
        connect(Pop_0_14.u3,Deaths_0_14.y) annotation(Line(visible = true, origin = {-92.2667,-25.2}, points = {{-4.5333,20.4},{2.2667,20.4},{2.2667,-40.8}}, color = {191,0,191}));
        connect(Mort_0_14.y,matur_14.u2) annotation(Line(visible = true, points = {{-70.0,-113.0},{-70.0,-104.0},{-66.0,-104.0},{-66.0,-34.0}}, color = {0,0,191}));
        connect(Mort_15_44.y,matur_44.u2) annotation(Line(visible = true, points = {{0.0,-113.0},{0.0,-104.0},{5.2,-104.0},{4.0,-28.0}}, color = {0,0,191}));
        connect(matur_44.y,Matur_44.u) annotation(Line(visible = true, points = {{0.0,-10.0},{0.0,-6.55},{0.0,-6.55},{0.0,-2.6}}, color = {0,0,191}));
        connect(Mort_45_64.y,matur_64.u2) annotation(Line(visible = true, points = {{70.0,-113.0},{70.0,-104.0},{75.2,-104.0},{74.0,-24.0}}, color = {0,0,191}));
        connect(deaths_65p.y,Death_Old.u2) annotation(Line(visible = true, points = {{132.0,-14.0},{132.0,-4.0},{172.0,-4.0},{172.0,74.0},{162.2,74.0}}, color = {0,0,191}));
        connect(Mort_65p.y,deaths_65p.u2) annotation(Line(visible = true, origin = {137.0,-42.5639}, points = {{1.0,-23.6915},{1.0,6.5639},{-1.0,6.5639},{-1.0,10.5639}}, color = {0,0,191}));
        connect(Mort_65p.u,Mort_45_64.u) annotation(Line(visible = true, points = {{138.0,-85.2554},{138.0,-140.0},{70.0,-140.0},{70.0,-132.0}}, color = {0,0,191}));
        connect(Mort_45_64.u,Mort_15_44.u) annotation(Line(visible = true, points = {{70.0,-132.0},{70.0,-140.0},{0.0,-140.0},{0.0,-132.0}}, color = {0,0,191}));
        connect(Mort_15_44.u,Mort_0_14.u) annotation(Line(visible = true, points = {{0.0,-132.0},{0.0,-140.0},{-70.0,-140.0},{-70.0,-132.0}}, color = {0,0,191}));
        connect(Mort_0_14.u,life_expectancy) annotation(Line(visible = true, points = {{-70.0,-132.0},{-70.0,-140.0},{-180.0,-140.0},{-180.0,-80.0},{-210.0,-80.0}}, color = {0,0,191}));
        connect(deaths_0_14.y,Death_Young.u1) annotation(Line(visible = true, points = {{-110.0,-73.0},{-110.0,-60.0},{-130.0,-60.0},{-130.0,-148.0},{184.0,-148.0},{184.0,116.0},{162.2,116.0}}, color = {0,0,191}));
        connect(source.MassInPort1,Births.y) annotation(Line(points = {{-159.0,0.0},{-146.0,0.0}}, color = {191,0,191}, visible = true));
        connect(Births.y1,Pop_0_14.u1) annotation(Line(points = {{-134.0,0.0},{-123.2,0.0}}, color = {191,0,191}, visible = true));
        connect(births.y,Births.u) annotation(Line(points = {{-140.0,-20.0},{-140.0,-12.6}}, color = {0,0,191}, visible = true));
        connect(Pop_Young.u1,Pop_0_14.y) annotation(Line(points = {{-162.0,116.0},{-180.0,116.0},{-180.0,20.0},{-126.0,20.0},{-126.0,6.6},{-120.2,6.6}}, color = {0,0,191}, visible = true));
        connect(Pop_Young.y,Population.u1) annotation(Line(points = {{-139.0,110.0},{-132.0,110.0},{-132.0,100.0},{-124.0,100.0}}, color = {0,0,191}, visible = true));
        connect(Pop_Old.y,Population.u2) annotation(Line(points = {{-139.0,80.0},{-132.0,80.0},{-132.0,88.0},{-124.0,88.0}}, color = {0,0,191}, visible = true));
        connect(Death_Young.y,Deaths.u1) annotation(Line(points = {{136.9,110.0},{130.0,110.0},{130.0,100.0},{122.0,100.0}}, color = {0,0,191}, visible = true));
        connect(Death_Old.y,Deaths.u2) annotation(Line(points = {{136.9,80.0},{130.0,80.0},{130.0,88.0},{122.0,88.0}}, color = {0,0,191}, visible = true));
        connect(Birth_Rate.u2,Population.y) annotation(Line(points = {{-84.7,93.8},{-92.35,93.8},{-92.35,94.0},{-101.0,94.0}}, color = {0,0,191}, visible = true));
        connect(births.y,Birth_Rate.u1) annotation(Line(points = {{-140.0,-20.0},{-140.0,-16.0},{-184.0,-16.0},{-184.0,128.0},{-94.0,128.0},{-94.0,104.2},{-84.7,104.2}}, color = {0,0,191}, visible = true));
        connect(Population.y,Death_Rate.u2) annotation(Line(points = {{-101.0,94.0},{-94.0,94.0},{-94.0,74.0},{86.0,74.0},{86.0,93.8},{80.6,93.8}}, color = {0,0,191}, visible = true));
        connect(Deaths.y,Death_Rate.u1) annotation(Line(points = {{99.0,94.0},{92.0,94.0},{92.0,104.2},{80.6,104.2}}, color = {0,0,191}, visible = true));
        connect(Deaths.y,births.u2) annotation(Line(points = {{99.0,94.0},{92.0,94.0},{92.0,70.0},{-60.0,70.0},{-60.0,36.0},{-156.0,36.0},{-156.0,-60.0},{-140.0,-60.0},{-140.0,-38.0}}, color = {0,0,191}, visible = true));
        connect(births.u1,total_fertility) annotation(Line(points = {{-146.6,-38.0},{-146.6,-48.0},{-188.0,-48.0},{-188.0,60.0},{-210.0,60.0}}, color = {0,0,191}, visible = true));
        connect(Population.y,population) annotation(Line(points = {{-101.0,94.0},{-94.0,94.0},{-94.0,74.0},{120.0,74.0},{120.0,60.0},{210.0,60.0}}, color = {0,0,191}, visible = true));
        annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, fillColor = {170,255,170}, fillPattern = FillPattern.Solid, lineThickness = 1.0, extent = {{-100.0,-100.0},{100.0,100.0}}, radius = 0.5),Text(visible = true, origin = {-0.5365,30.0}, fillPattern = FillPattern.Solid, extent = {{-89.4635,-36.6918},{89.4635,36.6918}}, textString = "Population", fontSize = 102.0, fontName = "Arial"),Text(visible = true, origin = {0.0,-30.0}, fillPattern = FillPattern.Solid, extent = {{-81.6353,-36.6918},{81.6353,36.6918}}, textString = "Dynamics", fontSize = 102.0, fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-200.0,-160.0},{200.0,140.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-71.0,-105.0},{-69.0,-103.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-1.0,-105.0},{1.0,-103.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{69.0,-105.0},{71.0,-103.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-71.0,-141.0},{-69.0,-139.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-1.0,-141.0},{1.0,-139.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{69.0,-141.0},{71.0,-139.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{29.0,-61.0},{31.0,-59.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-41.0,-61.0},{-39.0,-59.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-111.0,-61.0},{-109.0,-59.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{131.0,-5.0},{133.0,-3.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{13.0,27.0},{15.0,29.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-57.0,23.0},{-55.0,25.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-141.0,-17.0},{-139.0,-15.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-95.0,93.0},{-93.0,95.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{91.0,93.0},{93.0,95.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{85.0,73.0},{87.0,75.0}})}), Documentation(info = "<html>
This model describes the population dynamics of the <font color=red><b>WORLD3</b></font> model.  Whereas the earlier <font color=red><b>WORLD2</b></font> model had lumped all of the population together into a single state variable, <font color=red><b>WORLD3</b></font> offers a demographic model that distinguishes between four age groups, those of the children, those of the younger adults representing the child-bearing population, those of the older adults who are still in the work force, and finally, those of the retired population. <p>
 
Population is measured in numbers of living humans.  In accordance with engineering practice, which (arbitrarily) assigns measurement units only to real-valued quantities and not to integer-valued quantities, this model lists the population as dimensionless. <p>
 
This model is described on p.142 of <a href=\"http://www.pegasuscom.com/BookDetail.asp?BookQuery_Action=Find('ISBN','XDYGO')\">Dynamics of Growth in a Finite World</a>.
</html>"));
      end Population_Dynamics;
      block Service_Sector_Investment "Investments in the service sector"
        parameter Real t_policy_year(unit = "yr") = 4000 "Year of policy change";
        parameter Real p_avg_life_serv_cap_1(unit = "yr") = 20 "Default average life of service sector capital";
        parameter Real p_avg_life_serv_cap_2(unit = "yr") = 20 "Controlled average life of service sector capital";
        parameter Real p_serv_cap_out_ratio_1(unit = "yr") = 1.0 "Default fraction of service sector output ratio";
        parameter Real p_serv_cap_out_ratio_2(unit = "yr") = 1.0 "Controlled fraction of service sector output ratio";
        parameter Real service_capital_init(unit = "dollar") = 144000000000.0 "Initial service sector investment";
        SystemDynamics.Sources.Source Source1 annotation(Placement(visible = true, transformation(origin = {-170.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Rates.RRate Serv_Cap_Invest "p.230 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-120.0,30.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Levels.Level Service_Capital(x0 = service_capital_init) "p.230 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-60.0,30.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Rates.RRate Serv_Cap_Deprec "p.231 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {0.0,30.0}, extent = {{-15.0,-15.0},{15.0,15.0}}, rotation = 0)));
        SystemDynamics.Sources.Sink Sink1 annotation(Placement(visible = true, transformation(origin = {50.0,30.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Auxiliary.Prod_2 Serv_Cap_Inv "p.230 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-120.0,-10.0}, extent = {{-12.0,-12.0},{12.0,12.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division Serv_Cap_Dep "p.231 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {0.0,-12.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = -270)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_Avg_Life_Serv_Cap S_Avg_Life_Serv_Cap(t_policy_year = t_policy_year, p_avg_life_serv_cap_1 = p_avg_life_serv_cap_1, p_avg_life_serv_cap_2 = p_avg_life_serv_cap_2) "p.231 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {43.0,-36.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.Service_Output Service_Output "p.231 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-70.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_Serv_Cap_Out_Ratio S_Serv_Cap_Out_Ratio(t_policy_year = t_policy_year, p_serv_cap_out_ratio_1 = p_serv_cap_out_ratio_1, p_serv_cap_out_ratio_2 = p_serv_cap_out_ratio_2) "p.232 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-2.0,80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_FIOA_Serv S_FIOA_Serv(t_policy_year = t_policy_year) "p.229 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-105.0,-80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_Fr_IO_Al_Serv_1(x_vals = {0,0.5,1,1.5,2,5}, y_vals = {0.3,0.2,0.1,0.05,0,0}) "p.229 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-42.0,-60.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_Fr_IO_Al_Serv_2(x_vals = {0,0.5,1,1.5,2,5}, y_vals = {0.3,0.2,0.1,0.05,0,0}) "p.229 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-42.0,-100.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division Div1 "p.229 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {10.0,-80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Utilities.S_Indic_Serv_PC S_Indic_Serv_PC(t_policy_year = t_policy_year) "p.227 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {63.0,-100.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_Indic_Serv_PC_1(x_vals = 0:200:1600, y_vals = {40,300,640,1000,1220,1450,1650,1800,2000}) "p.227 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {128.0,-80.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        SystemDynamics.Functions.Tabular P_Indic_Serv_PC_2(x_vals = 0:200:1600, y_vals = {40,300,640,1000,1220,1450,1650,1800,2000}) "p.227 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {128.0,-120.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = 0)));
        Modelica.Blocks.Math.Division Serv_Out_PC "p.232 of Dynamics of Growth in a Finite World" annotation(Placement(visible = true, transformation(origin = {-110.0,102.0}, extent = {{10.0,-10.0},{-10.0,10.0}}, rotation = -90)));
        SystemDynamics.Interfaces.MassInPort population "Population" annotation(Placement(visible = true, transformation(origin = {-210.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort capital_util_fr "Capital utilization fraction" annotation(Placement(visible = true, transformation(origin = {-210.0,110.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort industrial_output(unit = "dollar/yr") "Annual industrial output" annotation(Placement(visible = true, transformation(origin = {-210.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-20.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassInPort ind_out_pc(unit = "dollar/yr") "Per capita annual industrial output" annotation(Placement(visible = true, transformation(origin = {-210.0,-130.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {-110.0,-60.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort serv_out_pc(unit = "dollar/yr") "Per capita annual expenditure in services" annotation(Placement(visible = true, transformation(origin = {210.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort service_capital(unit = "dollar") "Total capital invested in service sector" annotation(Placement(visible = true, transformation(origin = {210.0,-10.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,0.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
        SystemDynamics.Interfaces.MassOutPort s_fioa_serv "Fraction of total investments allocated to the service sector" annotation(Placement(visible = true, transformation(origin = {210.0,-70.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0), iconTransformation(origin = {110.0,-50.0}, extent = {{-10.0,-10.0},{10.0,10.0}}, rotation = 0)));
      equation
        connect(Service_Output.u1,capital_util_fr) annotation(Line(visible = true, points = {{-61.0,86.0},{-36.0,86.0},{-36.0,130.0},{-180.0,130.0},{-180.0,110.0},{-210.0,110.0}}, color = {0,0,191}));
        connect(Service_Output.u3,Service_Capital.y1) annotation(Line(visible = true, points = {{-61.0,74.0},{-36.0,74.0},{-36.0,38.25},{-47.25,38.25}}, color = {0,0,191}));
        connect(service_capital,Service_Capital.y2) annotation(Line(visible = true, points = {{210.0,-10.0},{180.0,-10.0},{180.0,8.0},{-48.75,8.0},{-48.75,20.25}}, color = {0,0,191}));
        connect(Serv_Cap_Dep.u1,Service_Capital.y3) annotation(Line(visible = true, points = {{-6.0,-24.0},{-6.0,-36.0},{-60.0,-36.0},{-60.0,20.25}}, color = {0,0,191}));
        connect(P_Fr_IO_Al_Serv_1.y,S_FIOA_Serv.u1) annotation(Line(visible = true, points = {{-53.0,-60.0},{-80.0,-60.0},{-80.0,-76.0},{-96.0,-76.0}}, color = {0,0,191}));
        connect(P_Fr_IO_Al_Serv_2.y,S_FIOA_Serv.u2) annotation(Line(visible = true, points = {{-53.0,-100.0},{-80.0,-100.0},{-80.0,-84.0},{-96.0,-84.0}}, color = {0,0,191}));
        connect(P_Indic_Serv_PC_1.y,S_Indic_Serv_PC.u1) annotation(Line(visible = true, points = {{117.0,-80.0},{90.0,-80.0},{90.0,-96.0},{72.0,-96.0}}, color = {0,0,191}));
        connect(P_Indic_Serv_PC_2.y,S_Indic_Serv_PC.u2) annotation(Line(visible = true, points = {{117.0,-120.0},{90.0,-120.0},{90.0,-104.0},{72.0,-104.0}}, color = {0,0,191}));
        connect(P_Indic_Serv_PC_1.u,P_Indic_Serv_PC_2.u) annotation(Line(visible = true, points = {{136.0,-80.0},{160.0,-80.0},{160.0,-120.0},{136.0,-120.0}}, color = {0,0,191}));
        connect(P_Indic_Serv_PC_2.u,ind_out_pc) annotation(Line(visible = true, points = {{136.0,-120.0},{160.0,-120.0},{160.0,-140.0},{-180.0,-140.0},{-180.0,-130.0},{-210.0,-130.0}}, color = {0,0,191}));
        connect(S_Indic_Serv_PC.y,Div1.u2) annotation(Line(visible = true, points = {{54.0,-100.0},{30.0,-100.0},{30.0,-86.0},{22.0,-86.0}}, color = {0,0,191}));
        connect(S_Avg_Life_Serv_Cap.y,Serv_Cap_Dep.u2) annotation(Line(visible = true, origin = {15.3333,-32.0}, points = {{18.6667,-4.0},{-9.3333,-4.0},{-9.3333,8.0}}, color = {0,0,191}));
        connect(Div1.y,P_Fr_IO_Al_Serv_2.u) annotation(Line(visible = true, points = {{-1.0,-80.0},{-10.0,-80.0},{-10.0,-100.0},{-34.0,-100.0}}, color = {0,0,191}));
        connect(Div1.y,P_Fr_IO_Al_Serv_1.u) annotation(Line(visible = true, points = {{-1.0,-80.0},{-10.0,-80.0},{-10.0,-60.0},{-34.0,-60.0}}, color = {0,0,191}));
        connect(S_FIOA_Serv.y,s_fioa_serv) annotation(Line(visible = true, points = {{-114.0,-80.0},{-130.0,-80.0},{-130.0,-150.0},{180.0,-150.0},{180.0,-70.0},{210.0,-70.0}}, color = {0,0,191}));
        connect(S_FIOA_Serv.y,Serv_Cap_Inv.u2) annotation(Line(visible = true, points = {{-114.0,-80.0},{-130.0,-80.0},{-130.0,-40.0},{-100.0,-40.0},{-100.0,-10.0},{-111.6,-10.0}}, color = {0,0,191}));
        connect(Serv_Cap_Dep.y,Serv_Cap_Deprec.u) annotation(Line(visible = true, points = {{0.0,-1.0},{0.0,7.55},{0.0,7.55},{0.0,14.25}}, color = {0,0,191}));
        connect(Serv_Cap_Deprec.y1,Sink1.MassInPort1) annotation(Line(visible = true, origin = {23.25,30.0}, points = {{-15.75,0.0},{15.75,0.0}}, color = {95,0,191}));
        connect(Service_Capital.u2,Serv_Cap_Deprec.y) annotation(Line(visible = true, origin = {-25.5,30.0}, points = {{-18.0,0.0},{18.0,0.0}}, color = {95,0,191}));
        connect(Serv_Cap_Invest.y1,Service_Capital.u1) annotation(Line(visible = true, origin = {-94.5,30.0}, points = {{-18.0,0.0},{18.0,0.0}}, color = {95,0,191}));
        connect(Serv_Cap_Inv.y,Serv_Cap_Invest.u) annotation(Line(visible = true, origin = {-120.0,6.325}, points = {{0.0,-7.925},{0.0,7.925}}, color = {0,0,191}));
        connect(Source1.MassInPort1,Serv_Cap_Invest.y) annotation(Line(visible = true, origin = {-143.25,30.0}, points = {{-15.75,0.0},{15.75,0.0}}, color = {95,0,191}));
        connect(S_Serv_Cap_Out_Ratio.y,Service_Output.u2) annotation(Line(visible = true, origin = {-36.0,80.0}, points = {{25.0,0.0},{-25.0,0.0}}, color = {0,0,191}));
        connect(Serv_Out_PC.u1,Service_Output.y) annotation(Line(visible = true, origin = {-95.6667,83.3333}, points = {{-8.3333,6.6667},{-8.3333,-3.3333},{16.6667,-3.3333}}, color = {0,0,191}));
        connect(Serv_Out_PC.y,serv_out_pc) annotation(Line(visible = true, points = {{-110.0,113.0},{-110.0,120.0},{180.0,120.0},{180.0,50.0},{210.0,50.0}}, color = {0,0,191}));
        connect(Serv_Out_PC.u2,population) annotation(Line(visible = true, origin = {-147.3333,63.3333}, points = {{31.3333,26.6667},{31.3333,-13.3333},{-62.6667,-13.3333}}, color = {0,0,191}));
        connect(Serv_Out_PC.y,Div1.u1) annotation(Line(visible = true, points = {{-110.0,113.0},{-110.0,120.0},{80.0,120.0},{80.0,-74.0},{22.0,-74.0}}, color = {0,0,191}));
        connect(Serv_Cap_Inv.u1,industrial_output) annotation(Line(points = {{-128.4,-10.0},{-180.0,-10.0},{-180.0,-70.0},{-210.0,-70.0}}, color = {0,0,191}, visible = true));
        annotation(Icon(coordinateSystem(extent = {{-100.0,-100.0},{100.0,100.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, fillColor = {170,255,170}, fillPattern = FillPattern.Solid, lineThickness = 1.0, extent = {{-100.0,-100.0},{100.0,100.0}}, radius = 0.5),Text(visible = true, origin = {0.1825,50.0}, fillPattern = FillPattern.Solid, extent = {{-59.8175,-36.6918},{59.8175,36.6918}}, textString = "Service", fontSize = 102.0, fontName = "Arial"),Text(visible = true, origin = {-0.0,-50.0}, fillPattern = FillPattern.Solid, extent = {{-95.1836,-20.8986},{95.1836,20.8986}}, textString = "Investment", fontSize = 102.0, fontName = "Arial"),Text(visible = true, fillPattern = FillPattern.Solid, extent = {{-62.5202,-36.6918},{62.5202,36.6918}}, textString = "Sector", fontSize = 102.0, fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-200.0,-160.0},{200.0,140.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-11.0,-81.0},{-9.0,-79.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{159.0,-121.0},{161.0,-119.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{79.0,119.0},{81.0,121.0}}),Ellipse(visible = true, lineColor = {0,0,191}, fillColor = {0,0,191}, fillPattern = FillPattern.Solid, extent = {{-131.0,-81.0},{-129.0,-79.0}})}), Documentation(info = "<html>
This model calculates the capital invested in the service sector. <p>
 
Whereas the earlier <font color=red><b>WORLD2</b></font> model lumped all capital investment into a single state variable from which it then split off the percentage of capital invested in the agricultural sector as a second state variable (a funny choice for a state variable, since percentages don't accumulate), the more modern <font color=red><b>WORLD3</b></font> model differentiates between capital invested in industry, capital invested in the service sector, capital invested in agriculture, and other capital investment. <p>
 
The service capital is measured in U.S. dollars.
</html>"));
      end Service_Sector_Investment;
      model Scenario_1 "Original WORLD3 model"
        parameter Real agr_mtl_toxic_index(unit = "1/dollar") = 1 "Agricultural materials toxicity index";
        parameter Real assim_half_life_1970(unit = "yr") = 1.5 "Pollution assimilation half life in 1970";
        parameter Real avg_life_land_norm(unit = "yr") = 1000 "Normal life span of land";
        parameter Real des_compl_fam_size_norm = 3.8 "Desired normal complete family size";
        parameter Real des_food_ratio_dfr = 2 "Desired food ratio";
        parameter Real des_ppoll_index_DPOLX = 1.2 "Desired persistent pollution index";
        parameter Real des_res_use_rt_DNRUR(unit = "ton/yr") = 4800000000.0 "Desired resource utilization rate";
        parameter Real food_short_perc_del(unit = "yr") = 2 "Food shortage perception delay";
        parameter Real fr_agr_inp_pers_mtl = 0.001 "Effective fraction of agricultural pollution input";
        parameter Real frac_res_pers_mtl = 0.02 "Effective fraction of resource utilization on pollution generation";
        parameter Real hlth_serv_impact_del(unit = "yr") = 20 "Health service impact delay";
        parameter Real income_expect_avg_time(unit = "yr") = 3 "Income expected average time";
        parameter Real ind_mtl_emiss_fact(unit = "1/ton") = 0.1 "Industrial materials emission factor";
        parameter Real ind_mtl_toxic_index = 10.0 "Industrial materials toxicity index";
        parameter Real ind_out_pc_des(unit = "dollar/yr") = 400 "Desired annual industrial per capita output";
        parameter Real ind_out_in_1970(unit = "dollar/yr") = 790000000000.0 "Industrial output in 1970";
        parameter Real inherent_land_fert(unit = "kg/(hectare.yr)") = 600 "Inherent land fertility";
        parameter Real labor_force_partic = 0.75 "Percentage of participating labor force";
        parameter Real labor_util_fr_del_time(unit = "yr") = 2 "Labor utilization fraction delay time";
        parameter Real land_fr_harvested = 0.7 "Land fraction harvested";
        parameter Real life_expect_norm(unit = "yr") = 28 "Normal life expectancy";
        parameter Real lifet_perc_del(unit = "yr") = 20 "Perceived life-time delay";
        parameter Real max_tot_fert_norm = 12 "Normal maximal total fertility";
        parameter Real p_avg_life_agr_inp_1(unit = "yr") = 2 "Default average life of agricultural input";
        parameter Real p_avg_life_agr_inp_2(unit = "yr") = 2 "Controlled average life of agricultural input";
        parameter Real p_avg_life_ind_cap_1(unit = "yr") = 14 "Default average life of industrial capital";
        parameter Real p_avg_life_ind_cap_2(unit = "yr") = 14 "Controlled average life of industrial capital";
        parameter Real p_avg_life_serv_cap_1(unit = "yr") = 20 "Default average life of service sector capital";
        parameter Real p_avg_life_serv_cap_2(unit = "yr") = 20 "Controlled average life of service sector capital";
        parameter Real p_fioa_cons_const_1 = 0.43 "Default fraction of industrial output allocated to consumption";
        parameter Real p_fioa_cons_const_2 = 0.43 "Controlled fraction of industrial output allocated to consumption";
        parameter Real p_ind_cap_out_ratio_1(unit = "yr") = 3 "Default industrial capital output ratio";
        parameter Real p_land_yield_fact_1 = 1 "Default land yield factor";
        parameter Real p_nr_res_use_fact_1 = 1 "Default non-recoverable resource utilization factor";
        parameter Real p_ppoll_gen_fact_1 = 1 "Default persistent pollution generation factor";
        parameter Real p_serv_cap_out_ratio_1(unit = "yr") = 1.0 "Default fraction of service sector output ratio";
        parameter Real p_serv_cap_out_ratio_2(unit = "yr") = 1.0 "Controlled fraction of service sector output ratio";
        parameter Real pot_arable_land_tot(unit = "hectare") = 3200000000.0 "Total potential arable land";
        parameter Real ppoll_in_1970 = 136000000.0 "Persistent pollution in 1970";
        parameter Real ppoll_trans_del(unit = "yr") = 20 "Persistent pollution transmission delay";
        parameter Real processing_loss = 0.1 "Processing loss";
        parameter Real reproductive_lifetime(unit = "yr") = 30.0 "Reproductive life time";
        parameter Real social_adj_del(unit = "yr") = 20 "Social adjustment delay";
        parameter Real social_discount(unit = "1/yr") = 0.07 "Social discount";
        parameter Real subsist_food_pc(unit = "kg/yr") = 230 "Available per capita food";
        parameter Real tech_dev_del_TDD(unit = "yr") = 20 "Technology development time";
        parameter Real urb_ind_land_dev_time(unit = "yr") = 10 "Urban and industrial land development time";
        parameter Real t_air_poll_time(unit = "yr") = 4000 "Air pollution change time";
        parameter Real t_fcaor_time(unit = "yr") = 4000 "Year of capital allocation to resource use efficiency";
        parameter Real t_fert_cont_eff_time(unit = "yr") = 4000 "Year of continued fertility change";
        parameter Real t_ind_equil_time(unit = "yr") = 4000 "Year of industrial equilibrium";
        parameter Real t_land_life_time(unit = "yr") = 4000 "Land life time";
        parameter Real t_policy_year(unit = "yr") = 4000 "Year of policy change";
        parameter Real t_pop_equil_time(unit = "yr") = 4000 "Population equilibrium time";
        parameter Real t_zero_pop_grow_time(unit = "yr") = 4000 "Time to zero population growth";
        parameter Real p_fr_cap_al_obt_res_2[:] = {1,0.2,0.1,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05} "Non-renewable resource fraction remaining";
        parameter Real p_ppoll_tech_chg_mlt[:] = {0,0,0,0} "Persistent pollution technology change multiplier";
        parameter Real p_res_tech_chg_mlt[:] = {0,0,0,0} "Resource technology change multiplier";
        parameter Real p_yield_tech_chg_mlt[:] = {0,0,0,0} "Yield technology change multiplier";
        parameter Real agr_inp_init(unit = "dollar/yr") = 5000000000.0 "Initial agricultural input";
        parameter Real arable_land_init(unit = "hectare") = 900000000.0 "Initial arable land";
        parameter Real industrial_capital_init(unit = "dollar") = 210000000000.0 "Initial industrial investment";
        parameter Real labor_util_fr_del_init = 1 "Initial delayed labor utilization fraction";
        parameter Real land_fertility_init(unit = "kg/(hectare.yr)") = 600 "Initial industrial investment";
        parameter Real nr_resources_init(unit = "ton") = 1000000000000.0 "Initial available non-recoverable resources";
        parameter Real perc_food_ratio_init = 1 "Initial perceived food ratio";
        parameter Real pers_pollution_init = 25000000.0 "Initial persistent pollution";
        parameter Real pop1_init = 650000000.0 "Initial population 14 years and younger";
        parameter Real pop2_init = 700000000.0 "Initial population 15 to 44 years old";
        parameter Real pop3_init = 190000000.0 "Initial population 45 to 64 years old";
        parameter Real pop4_init = 60000000.0 "Initial population 65 years and older";
        parameter Real pot_arable_land_init(unit = "hectare") = 2300000000.0 "Initial potential arable land";
        parameter Real ppoll_tech_init = 1 "Initial persistent pollution technology change factor";
        parameter Real res_tech_init = 1 "Initial non-recoverable resource technology factor";
        parameter Real service_capital_init(unit = "dollar") = 144000000000.0 "Initial service sector investment";
        parameter Real urban_ind_land_init(unit = "hectare") = 8200000.0 "Initial urban and industrial land";
        parameter Real yield_tech_init = 1 "Initial yield technology factor";
        output Real population "Total human world population";
        output Real food(unit = "dollar/yr") "Total annually produced food";
        output Real industrial_output(unit = "dollar/yr") "Total annual world industrial output";
        output Real ppoll_index "Persistent pollution index";
        output Real nr_resources "Remaining non-recoverable natural resources";
        output Real fioa_ind "Fraction of industrial output allocated to industrial/military complex";
        output Real s_fioa_agr "Fraction of industrial output allocated to food production";
        output Real s_fioa_cons "Fraction of industrial output allocated to consumption";
        output Real s_fioa_serv "Fraction of industrial output allocated to service sector";
        output Real s_fr_cap_al_obt_res "Fraction of capital allocated to resource use efficiency";
        output Real life_expectancy(unit = "yr") "Life expectancy";
        output Real food_pc(unit = "dollar/yr") "Total annual food per person";
        output Real serv_out_pc(unit = "dollar/yr") "Total annual services per person";
        output Real ind_out_pc(unit = "dollar/yr") "Total annual consumer goods per person";
        output Real human_ecological_footprint(unit = "Gha") "Human ecological footprint";
        output Real human_welfare_index "Human welfare index";
        SystemDynamics.WorldDynamics.World3.Population_Dynamics Population_Dynamics1(pop1_init = pop1_init, pop2_init = pop2_init, pop3_init = pop3_init, pop4_init = pop4_init, labor_force_partic = labor_force_partic, reproductive_lifetime = reproductive_lifetime, t_pop_equil_time = t_pop_equil_time) "Population dynamics" annotation(Placement(visible = true, transformation(origin = {-100.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Pollution_Dynamics Pollution_Dynamics1(agr_mtl_toxic_index = agr_mtl_toxic_index, assim_half_life_1970 = assim_half_life_1970, des_ppoll_index_DPOLX = des_ppoll_index_DPOLX, fr_agr_inp_pers_mtl = fr_agr_inp_pers_mtl, frac_res_pers_mtl = frac_res_pers_mtl, ind_mtl_emiss_fact = ind_mtl_emiss_fact, ind_mtl_toxic_index = ind_mtl_toxic_index, ind_out_in_1970 = ind_out_in_1970, p_ppoll_gen_fact_1 = p_ppoll_gen_fact_1, pers_pollution_init = pers_pollution_init, ppoll_in_1970 = ppoll_in_1970, ppoll_tech_init = ppoll_tech_init, ppoll_trans_del = ppoll_trans_del, t_air_poll_time = t_air_poll_time, t_policy_year = t_policy_year, tech_dev_del_TDD = tech_dev_del_TDD, p_ppoll_tech_chg_mlt = p_ppoll_tech_chg_mlt) "Persistent pollution generation" annotation(Placement(visible = true, transformation(origin = {-100.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Arable_Land_Dynamics Arable_Land_Dynamics1(arable_land_init = arable_land_init, avg_life_land_norm = avg_life_land_norm, inherent_land_fert = inherent_land_fert, pot_arable_land_init = pot_arable_land_init, pot_arable_land_tot = pot_arable_land_tot, social_discount = social_discount, t_land_life_time = t_land_life_time, urban_ind_land_init = urban_ind_land_init, urb_ind_land_dev_time = urb_ind_land_dev_time) "Arable land dynamics" annotation(Placement(visible = true, transformation(origin = {-100.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Food_Production Food_Production1(agr_inp_init = agr_inp_init, food_short_perc_del = food_short_perc_del, p_avg_life_agr_inp_1 = p_avg_life_agr_inp_1, p_avg_life_agr_inp_2 = p_avg_life_agr_inp_2, p_land_yield_fact_1 = p_land_yield_fact_1, perc_food_ratio_init = perc_food_ratio_init, processing_loss = processing_loss, subsist_food_pc = subsist_food_pc, t_policy_year = t_policy_year, tech_dev_del_TDD = tech_dev_del_TDD, land_fr_harvested = land_fr_harvested) "Food production" annotation(Placement(visible = true, transformation(origin = {-100.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Human_Ecological_Footprint Human_Ecological_Footprint1 "Human ecological footprint" annotation(Placement(visible = true, transformation(origin = {340.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Human_Fertility Human_Fertility1(des_compl_fam_size_norm = des_compl_fam_size_norm, hlth_serv_impact_del = hlth_serv_impact_del, income_expect_avg_time = income_expect_avg_time, lifet_perc_del = lifet_perc_del, max_tot_fert_norm = max_tot_fert_norm, social_adj_del = social_adj_del, t_fert_cont_eff_time = t_fert_cont_eff_time, t_zero_pop_grow_time = t_zero_pop_grow_time) "Human fertility" annotation(Placement(visible = true, transformation(origin = {120.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Human_Welfare_Index Human_Welfare_Index1 "Human welfare index" annotation(Placement(visible = true, transformation(origin = {340.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Industrial_Investment Industrial_Investment1(industrial_capital_init = industrial_capital_init, ind_out_pc_des = ind_out_pc_des, p_avg_life_ind_cap_1 = p_avg_life_ind_cap_1, p_avg_life_ind_cap_2 = p_avg_life_ind_cap_2, p_fioa_cons_const_1 = p_fioa_cons_const_1, p_fioa_cons_const_2 = p_fioa_cons_const_2, p_ind_cap_out_ratio_1 = p_ind_cap_out_ratio_1, t_ind_equil_time = t_ind_equil_time, t_policy_year = t_policy_year) "Industrial investment" annotation(Placement(visible = true, transformation(origin = {120.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Labor_Utilization Labor_Utilization1(labor_util_fr_del_init = labor_util_fr_del_init, labor_util_fr_del_time = labor_util_fr_del_time) "Labor utilization" annotation(Placement(visible = true, transformation(origin = {120.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Land_Fertility Land_Fertility1(des_food_ratio_dfr = des_food_ratio_dfr, inherent_land_fert = inherent_land_fert, land_fertility_init = land_fertility_init, t_policy_year = t_policy_year, yield_tech_init = yield_tech_init, p_yield_tech_chg_mlt = p_yield_tech_chg_mlt) "Land fertility" annotation(Placement(visible = true, transformation(origin = {120.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Life_Expectancy Life_Expectancy1(hlth_serv_impact_del = hlth_serv_impact_del, life_expect_norm = life_expect_norm, subsist_food_pc = subsist_food_pc) "Life expectancy" annotation(Placement(visible = true, transformation(origin = {340.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.NR_Resource_Utilization NR_Resource_Utilization1(des_res_use_rt_DNRUR = des_res_use_rt_DNRUR, nr_resources_init = nr_resources_init, p_nr_res_use_fact_1 = p_nr_res_use_fact_1, res_tech_init = res_tech_init, t_policy_year = t_policy_year, t_fcaor_time = t_fcaor_time, tech_dev_del_TDD = tech_dev_del_TDD, p_fr_cap_al_obt_res_2 = p_fr_cap_al_obt_res_2, p_res_tech_chg_mlt = p_res_tech_chg_mlt) "Non-recoverable natural resource utilization" annotation(Placement(visible = true, transformation(origin = {340.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Service_Sector_Investment Service_Sector_Investment1(t_policy_year = t_policy_year, p_avg_life_serv_cap_1 = p_avg_life_serv_cap_1, p_avg_life_serv_cap_2 = p_avg_life_serv_cap_2, p_serv_cap_out_ratio_1 = p_serv_cap_out_ratio_1, p_serv_cap_out_ratio_2 = p_serv_cap_out_ratio_2, service_capital_init = service_capital_init) "Service sector investment" annotation(Placement(visible = true, transformation(origin = {-100.0,-220.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
      equation
        connect(Food_Production1.tot_agric_invest,Arable_Land_Dynamics1.tot_agric_invest) annotation(Line(visible = true, points = {{-88.0,-98.0},{-88.0,-52.0},{-200.0,-52.0},{-200.0,-20.0},{-122.0,-20.0}}, color = {0,0,191}));
        connect(Food_Production1.yield_tech_mult_icor_COYM,Industrial_Investment1.yield_tech_mult_icor_COYM) annotation(Line(visible = true, points = {{-96.0,-142.0},{-96.0,-174.0},{28.0,-174.0},{28.0,96.0},{98.0,96.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.urban_ind_land,Human_Ecological_Footprint1.urban_ind_land) annotation(Line(visible = true, points = {{-78.0,-30.0},{-24.0,-30.0},{-24.0,136.0},{252.0,136.0},{252.0,180.0},{318.0,180.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.ppoll_gen_rt,Human_Ecological_Footprint1.ppoll_gen_rt) annotation(Line(visible = true, points = {{-78.0,64.0},{-20.0,64.0},{-20.0,148.0},{256.0,148.0},{256.0,170.0},{318.0,170.0}}, color = {0,0,191}));
        connect(NR_Resource_Utilization1.s_fr_cap_al_obt_res,Industrial_Investment1.s_fr_cap_al_obt_res) annotation(Line(visible = true, points = {{362.0,-132.0},{432.0,-132.0},{432.0,132.0},{44.0,132.0},{44.0,72.0},{98.0,72.0}}, color = {0,0,191}));
        connect(Labor_Utilization1.capital_util_fr,Industrial_Investment1.capital_util_fr) annotation(Line(visible = true, points = {{142.0,-28.0},{224.0,-28.0},{224.0,-180.0},{24.0,-180.0},{24.0,80.0},{98.0,80.0}}, color = {0,0,191}));
        connect(NR_Resource_Utilization1.ind_cap_out_ratio_2_ICOR2T,Industrial_Investment1.ind_cap_out_ratio_2_ICOR2T) annotation(Line(visible = true, points = {{362.0,-124.0},{428.0,-124.0},{428.0,120.0},{108.0,120.0},{108.0,102.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.ppoll_tech_mult_icor_COPM,Industrial_Investment1.ppoll_tech_mult_icor_COPM) annotation(Line(visible = true, points = {{-78.0,88.0},{98.0,88.0}}, color = {0,0,191}));
        connect(Labor_Utilization1.capital_util_fr,Service_Sector_Investment1.capital_util_fr) annotation(Line(visible = true, points = {{142.0,-28.0},{224.0,-28.0},{224.0,-180.0},{-172.0,-180.0},{-172.0,-208.0},{-122.0,-208.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.s_yield_mlt_air_poll,Food_Production1.s_yield_mlt_air_poll) annotation(Line(visible = true, points = {{-78.0,72.0},{32.0,72.0},{32.0,-168.0},{-104.0,-168.0},{-104.0,-142.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.fr_inp_al_land_dev,Food_Production1.fr_inp_al_land_dev) annotation(Line(visible = true, points = {{-78.0,-10.0},{-28.0,-10.0},{-28.0,-68.0},{-172.0,-68.0},{-172.0,-120.0},{-122.0,-120.0}}, color = {0,0,191}));
        connect(Land_Fertility1.yield_tech_LYTD,Food_Production1.yield_tech_LYTD) annotation(Line(visible = true, points = {{142.0,-110.0},{192.0,-110.0},{192.0,-82.0},{-112.0,-82.0},{-112.0,-87.9893},{-112.0,-98.0}}, color = {0,0,191}));
        connect(NR_Resource_Utilization1.pc_res_use_mlt,Pollution_Dynamics1.pc_res_use_mlt) annotation(Line(visible = true, points = {{362.0,-108.0},{424.0,-108.0},{424.0,124.0},{-176.0,124.0},{-176.0,96.0},{-122.0,96.0}}, color = {0,0,191}));
        connect(Human_Fertility1.life_expectancy,Population_Dynamics1.life_expectancy) annotation(Line(visible = true, points = {{98.0,180.0},{32.0,180.0},{32.0,128.0},{-190.0,128.0},{-190.0,170.0},{-122.0,170.0}}, color = {0,0,191}));
        connect(Human_Welfare_Index1.life_expectancy,Human_Fertility1.life_expectancy) annotation(Line(visible = true, points = {{318.0,90.0},{250.0,90.0},{250.0,128.0},{32.0,128.0},{32.0,180.0},{98.0,180.0}}, color = {0,0,191}));
        connect(Life_Expectancy1.life_expectancy,Human_Welfare_Index1.life_expectancy) annotation(Line(visible = true, points = {{362.0,-20.0},{420.0,-20.0},{420.0,40.0},{250.0,40.0},{250.0,90.0},{318.0,90.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.industrial_capital,Labor_Utilization1.industrial_capital) annotation(Line(visible = true, points = {{142.0,76.0},{196.0,76.0},{196.0,-78.0},{44.0,-78.0},{44.0,-20.0},{98.0,-20.0}}, color = {0,0,191}));
        connect(Human_Fertility1.total_fertility,Population_Dynamics1.total_fertility) annotation(Line(visible = true, points = {{142.0,180.0},{196.0,180.0},{196.0,144.0},{-200.0,144.0},{-200.0,190.0},{-122.0,190.0}}, color = {0,0,191}));
        connect(Population_Dynamics1.labor_force,Labor_Utilization1.labor_force) annotation(Line(visible = true, points = {{-78.0,170.0},{12.0,170.0},{12.0,-60.0},{108.0,-60.0},{108.0,-42.0}}, color = {0,0,191}));
        connect(Service_Sector_Investment1.s_fioa_serv,Industrial_Investment1.s_fioa_serv) annotation(Line(visible = true, points = {{-78.0,-230.0},{16.0,-230.0},{16.0,38.0},{108.0,38.0},{108.0,58.0}}, color = {0,0,191}));
        connect(Food_Production1.p_fr_inp_for_land_maint,Land_Fertility1.p_fr_inp_for_land_maint) annotation(Line(visible = true, points = {{-88.0,-142.0},{-88.0,-160.0},{44.0,-160.0},{44.0,-120.0},{98.0,-120.0}}, color = {0,0,191}));
        connect(Food_Production1.s_fioa_agr,Industrial_Investment1.s_fioa_agr) annotation(Line(visible = true, points = {{-78.0,-136.0},{-4.0,-136.0},{-4.0,64.0},{98.0,64.0}}, color = {0,0,191}));
        connect(Food_Production1.agr_inp_per_hect,Labor_Utilization1.agr_inp_per_hect) annotation(Line(visible = true, points = {{-78.0,-128.0},{-16.0,-128.0},{-16.0,-28.0},{98.0,-28.0}}, color = {0,0,191}));
        connect(Food_Production1.agr_inp_per_hect,Pollution_Dynamics1.agr_inp_per_hect) annotation(Line(visible = true, points = {{-78.0,-128.0},{-16.0,-128.0},{-16.0,-56.0},{-196.0,-56.0},{-196.0,72.0},{-122.0,72.0}}, color = {0,0,191}));
        connect(Food_Production1.marg_prod_agr_inp,Arable_Land_Dynamics1.marg_prod_agr_inp) annotation(Line(visible = true, points = {{-78.0,-120.0},{-24.0,-120.0},{-24.0,-64.0},{-172.0,-64.0},{-172.0,-12.0},{-122.0,-12.0}}, color = {0,0,191}));
        connect(Food_Production1.food_pc,Life_Expectancy1.food_pc) annotation(Line(visible = true, points = {{-96.0,-98.0},{-96.0,-74.0},{256.0,-74.0},{256.0,-28.0},{318.0,-28.0}}, color = {0,0,191}));
        connect(Food_Production1.land_yield,Arable_Land_Dynamics1.land_yield) annotation(Line(visible = true, points = {{-78.0,-112.0},{-20.0,-112.0},{-20.0,-60.0},{-192.0,-60.0},{-192.0,-4.0},{-122.0,-4.0}}, color = {0,0,191}));
        connect(Land_Fertility1.ppoll_index,Life_Expectancy1.ppoll_index) annotation(Line(visible = true, points = {{98.0,-130.0},{20.0,-130.0},{20.0,-70.0},{268.0,-70.0},{268.0,-36.0},{318.0,-36.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.ppoll_index,Land_Fertility1.ppoll_index) annotation(Line(visible = true, points = {{-78.0,80.0},{20.0,80.0},{20.0,-130.0},{98.0,-130.0}}, color = {0,0,191}));
        connect(Service_Sector_Investment1.serv_out_pc,Labor_Utilization1.serv_out_pc) annotation(Line(visible = true, points = {{-78.0,-210.0},{0.0,-210.0},{0.0,18.0},{70.0,18.0},{108.0,18.0},{108.0,2.0}}, color = {0,0,191}));
        connect(Service_Sector_Investment1.serv_out_pc,Human_Fertility1.serv_out_pc) annotation(Line(visible = true, points = {{-78.0,-210.0},{0.0,-210.0},{0.0,190.0},{98.0,190.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,NR_Resource_Utilization1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,-110.0},{318.0,-110.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Life_Expectancy1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,-12.0},{318.0,-12.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Human_Welfare_Index1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,70.0},{318.0,70.0}}, color = {0,0,191}));
        connect(Human_Fertility1.ind_out_pc,Labor_Utilization1.ind_out_pc) annotation(Line(visible = true, points = {{98.0,170.0},{36.0,170.0},{36.0,-12.0},{98.0,-12.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Human_Fertility1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,30.0},{36.0,30.0},{36.0,170.0},{98.0,170.0}}, color = {0,0,191}));
        connect(Food_Production1.ind_out_pc,Service_Sector_Investment1.ind_out_pc) annotation(Line(visible = true, points = {{-122.0,-128.0},{-188.0,-128.0},{-188.0,-232.0},{-122.0,-232.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.ind_out_pc,Food_Production1.ind_out_pc) annotation(Line(visible = true, points = {{-122.0,-28.0},{-188.0,-28.0},{-188.0,-128.0},{-122.0,-128.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Arable_Land_Dynamics1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,30.0},{-188.0,30.0},{-188.0,-28.0},{-122.0,-28.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.industrial_output,NR_Resource_Utilization1.industrial_output) annotation(Line(visible = true, points = {{142.0,84.3414},{210.0,84.3414},{210.0,-130.0},{318.0,-130.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.industrial_output,Food_Production1.industrial_output) annotation(Line(visible = true, points = {{-122.0,80.0},{-184.0,80.0},{-184.0,-136.0},{-122.0,-136.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.industrial_output,Pollution_Dynamics1.industrial_output) annotation(Line(visible = true, points = {{142.0,84.3414},{210.0,84.3414},{210.0,22.0},{-184.0,22.0},{-184.0,80.0},{-122.0,80.0}}, color = {0,0,191}));
        connect(Labor_Utilization1.arable_land,Human_Ecological_Footprint1.arable_land) annotation(Line(visible = true, points = {{98.0,-36.0},{40.0,-36.0},{40.0,-64.0},{264.0,-64.0},{264.0,190.0},{318.0,190.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.arable_land,Labor_Utilization1.arable_land) annotation(Line(visible = true, points = {{-78.0,-20.0},{40.0,-20.0},{40.0,-36.0},{98.0,-36.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.arable_land,Food_Production1.arable_land) annotation(Line(visible = true, points = {{-122.0,64.0},{-176.0,64.0},{-176.0,-112.0},{-122.0,-112.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.arable_land,Pollution_Dynamics1.arable_land) annotation(Line(visible = true, points = {{-78.0,-20.0},{-20.0,-20.0},{-20.0,26.0},{-176.0,26.0},{-176.0,64.0},{-122.0,64.0}}, color = {0,0,191}));
        connect(Life_Expectancy1.population,NR_Resource_Utilization1.population) annotation(Line(visible = true, points = {{318.0,-20.0},{260.0,-20.0},{260.0,-120.0},{318.0,-120.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.population,Life_Expectancy1.population) annotation(Line(visible = true, points = {{116.0,58.0},{116.0,34.0},{260.0,34.0},{260.0,-20.0},{318.0,-20.0}}, color = {0,0,191}));
        connect(Population_Dynamics1.population,Industrial_Investment1.population) annotation(Line(visible = true, points = {{-78.0,190.0},{-10.0,190.0},{-10.0,34.0},{116.0,34.0},{116.0,58.0}}, color = {0,0,191}));
        connect(Food_Production1.population,Service_Sector_Investment1.population) annotation(Line(visible = true, points = {{-122.0,-104.0},{-180.0,-104.0},{-180.0,-216.0},{-122.0,-216.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.population,Food_Production1.population) annotation(Line(visible = true, points = {{-122.0,-36.0},{-180.0,-36.0},{-180.0,-104.0},{-122.0,-104.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.population,Arable_Land_Dynamics1.population) annotation(Line(visible = true, points = {{-122.0,88.0},{-180.0,88.0},{-180.0,-36.0},{-122.0,-36.0}}, color = {0,0,191}));
        connect(Population_Dynamics1.population,Pollution_Dynamics1.population) annotation(Line(visible = true, points = {{-78.0,190.0},{-10.0,190.0},{-10.0,140.0},{-180.0,140.0},{-180.0,88.0},{-122.0,88.0}}, color = {0,0,191}));
        connect(Food_Production1.food_ratio,Land_Fertility1.food_ratio) annotation(Line(visible = true, points = {{-78.0,-104.0},{10.0,-104.0},{10.0,-110.0},{98.0,-110.0}}, color = {0,0,191}));
        connect(Life_Expectancy1.serv_out_pc,Labor_Utilization1.serv_out_pc) annotation(Line(visible = true, points = {{318.0,-4.0},{218.0,-4.0},{218.0,18.0},{108.0,18.0},{108.0,18.0},{108.0,2.0}}, color = {0,0,191}));
        connect(Service_Sector_Investment1.service_capital,Labor_Utilization1.service_capital) annotation(Line(visible = true, points = {{-78.0,-220.0},{4.0,-220.0},{4.0,-4.0},{98.0,-4.0}}, color = {0,0,191}));
        connect(Land_Fertility1.land_fertility,Food_Production1.land_fertility) annotation(Line(visible = true, points = {{142.0,-120.0},{192.0,-120.0},{192.0,-164.0},{-112.0,-164.0},{-112.0,-142.0}}, color = {0,0,191}));
        connect(Food_Production1.industrial_output,Service_Sector_Investment1.industrial_output) annotation(Line(visible = true, origin = {0.6153,6.0}, points = {{-122.6153,-142.0},{-184.0,-142.0},{-184.0,-230.0},{-122.6153,-230.0}}, color = {0,0,191}));
        population = Population_Dynamics1.Population.y;
        food = Food_Production1.Food.food;
        industrial_output = Industrial_Investment1.Industrial_Output.industrial_output;
        ppoll_index = Pollution_Dynamics1.PPoll_Index.y;
        nr_resources = NR_Resource_Utilization1.NR_Resources.y;
        fioa_ind = Industrial_Investment1.FIOA_Ind.fioa_ind;
        s_fioa_agr = Food_Production1.S_FIOA_Agr.s_fioa_agr;
        s_fioa_cons = Industrial_Investment1.S_FIOA_Cons.s_fioa_cons;
        s_fioa_serv = Service_Sector_Investment1.S_FIOA_Serv.s_fioa_serv;
        s_fr_cap_al_obt_res = NR_Resource_Utilization1.S_Fr_Cap_Al_Obt_Res.s_fr_cap_al_obt_res;
        life_expectancy = Life_Expectancy1.Life_Expectancy.y;
        food_pc = Food_Production1.Food_PC.y;
        serv_out_pc = Service_Sector_Investment1.Serv_Out_PC.y;
        ind_out_pc = Industrial_Investment1.Ind_Out_PC.y;
        human_ecological_footprint = Human_Ecological_Footprint1.HEF_Human_Ecological_Footprint.hef_human_ecological_footprint;
        human_welfare_index = Human_Welfare_Index1.HWI_Human_Welfare_Index.hwi_human_welfare_index;
        annotation(Icon(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-220.0,-260.0},{442.0,222.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-70.0,104.0},{296.0,172.0}}, textString = "Meadows'", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-78.0,-20.0},{296.0,46.0}}, textString = "Original", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-70.0,-98.0},{296.0,-30.0}}, textString = "World Model", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), experiment(StartTime = 1900, StopTime = 2100), experimentSetupOutput, Documentation(info = "<html>
This is the original <font color=red><b>WORLD3</b></font> model without any interventions.  In the book <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a> it is referred to as <font color=red><b>Scenario #1</b></font>. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Meadows, D.H., D.L. Meadows, J. Randers, and W.W. Behrens III (1972), <i>Limits to Growth: A Report for the Club of Rome's Project on the Predicament of Mankind</i>, Universe Books, New York, 205p.
<li> Meadows, D.L., W.W. Behrens III, D.M., Meadows, R.F. Naill, J. Randers, and E.K.O. Zahn (1974), <a href=\"http://www.pegasuscom.com/BookDetail.asp?BookQuery_Action=Find('ISBN','XDYGO')\">Dynamics of Growth in a Finite World</a>, Wright-Allen Press, 637p.
<li> Meadows, D.H., D.L. Meadows, and J. Randers (1992), <i>Beyond the Limits</i>, Chelsea Green, 300p.
<li> Meadows, D.H., J. Randers, and D.L. Meadows (2004), <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a>, Chelsea Green, 368p.
</ol> <p>
 
<hr> <p>
 
Simulate the model from 1900 until 2100, and display the same variables as in the book <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a> at page 169: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_1.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_2.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_3.png\"> <p>
 
<hr> <p>
 
The results obtained are not exactly the same as those shown in the book <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a> due to the integration algorithm in use.  Most <font color=red><b>Modelica</b></font> simulation environments uses by default a variable-step / variable-order DASSL algorithm, whereas <font color=red><b>STELLA</b></font>, just like the older <font color=red><b>Dynamo</b></font> software, uses by default a fixed-step Euler algorithm with a step size of 1 time unit. <p>
 
A second even more important difference is that my <font color=red><b>Modelica</b></font> code treats all variables as real-valued floating-point numbers, whereas the <font color=red><b>STELLA</b></font> model treated some variables (e.g., population) as integers, whereas it treated other variables as fixed-point variables with only two significant digits after the comma.  This led to a quite noticeable quantization error. <p>
 
As I didn't see any good reason for unnecessarily mutilating the capabilities of the <font color=red><b>Modelica</b></font> simulation environment, I decided not to replicate these features of the <font color=red><b>STELLA</b></font> code here. <p>
 
<hr> <p>
</html>", revisions = ""));
      end Scenario_1;
      model Scenario_2 "More abundant non-recoverable natural resources"
        parameter Real agr_mtl_toxic_index(unit = "1/dollar") = 1 "Agricultural materials toxicity index";
        parameter Real assim_half_life_1970(unit = "yr") = 1.5 "Pollution assimilation half life in 1970";
        parameter Real avg_life_land_norm(unit = "yr") = 1000 "Normal life span of land";
        parameter Real des_compl_fam_size_norm = 3.8 "Desired normal complete family size";
        parameter Real des_food_ratio_dfr = 2 "Desired food ratio";
        parameter Real des_ppoll_index_DPOLX = 1.2 "Desired persistent pollution index";
        parameter Real des_res_use_rt_DNRUR(unit = "ton/yr") = 4800000000.0 "Desired resource utilization rate";
        parameter Real food_short_perc_del(unit = "yr") = 2 "Food shortage perception delay";
        parameter Real fr_agr_inp_pers_mtl = 0.001 "Effective fraction of agricultural pollution input";
        parameter Real frac_res_pers_mtl = 0.02 "Effective fraction of resource utilization on pollution generation";
        parameter Real hlth_serv_impact_del(unit = "yr") = 20 "Health service impact delay";
        parameter Real income_expect_avg_time(unit = "yr") = 3 "Income expected average time";
        parameter Real ind_mtl_emiss_fact(unit = "1/ton") = 0.1 "Industrial materials emission factor";
        parameter Real ind_mtl_toxic_index = 10.0 "Industrial materials toxicity index";
        parameter Real ind_out_pc_des(unit = "dollar/yr") = 400 "Desired annual industrial per capita output";
        parameter Real ind_out_in_1970(unit = "dollar/yr") = 790000000000.0 "Industrial output in 1970";
        parameter Real inherent_land_fert(unit = "kg/(hectare.yr)") = 600 "Inherent land fertility";
        parameter Real labor_force_partic = 0.75 "Percentage of participating labor force";
        parameter Real labor_util_fr_del_time(unit = "yr") = 2 "Labor utilization fraction delay time";
        parameter Real land_fr_harvested = 0.7 "Land fraction harvested";
        parameter Real life_expect_norm(unit = "yr") = 28 "Normal life expectancy";
        parameter Real lifet_perc_del(unit = "yr") = 20 "Perceived life-time delay";
        parameter Real max_tot_fert_norm = 12 "Normal maximal total fertility";
        parameter Real p_avg_life_agr_inp_1(unit = "yr") = 2 "Default average life of agricultural input";
        parameter Real p_avg_life_agr_inp_2(unit = "yr") = 2 "Controlled average life of agricultural input";
        parameter Real p_avg_life_ind_cap_1(unit = "yr") = 14 "Default average life of industrial capital";
        parameter Real p_avg_life_ind_cap_2(unit = "yr") = 14 "Controlled average life of industrial capital";
        parameter Real p_avg_life_serv_cap_1(unit = "yr") = 20 "Default average life of service sector capital";
        parameter Real p_avg_life_serv_cap_2(unit = "yr") = 20 "Controlled average life of service sector capital";
        parameter Real p_fioa_cons_const_1 = 0.43 "Default fraction of industrial output allocated to consumption";
        parameter Real p_fioa_cons_const_2 = 0.43 "Controlled fraction of industrial output allocated to consumption";
        parameter Real p_ind_cap_out_ratio_1(unit = "yr") = 3 "Default industrial capital output ratio";
        parameter Real p_land_yield_fact_1 = 1 "Default land yield factor";
        parameter Real p_nr_res_use_fact_1 = 1 "Default non-recoverable resource utilization factor";
        parameter Real p_ppoll_gen_fact_1 = 1 "Default persistent pollution generation factor";
        parameter Real p_serv_cap_out_ratio_1(unit = "yr") = 1.0 "Default fraction of service sector output ratio";
        parameter Real p_serv_cap_out_ratio_2(unit = "yr") = 1.0 "Controlled fraction of service sector output ratio";
        parameter Real pot_arable_land_tot(unit = "hectare") = 3200000000.0 "Total potential arable land";
        parameter Real ppoll_in_1970 = 136000000.0 "Persistent pollution in 1970";
        parameter Real ppoll_trans_del(unit = "yr") = 20 "Persistent pollution transmission delay";
        parameter Real processing_loss = 0.1 "Processing loss";
        parameter Real reproductive_lifetime(unit = "yr") = 30.0 "Reproductive life time";
        parameter Real social_adj_del(unit = "yr") = 20 "Social adjustment delay";
        parameter Real social_discount(unit = "1/yr") = 0.07 "Social discount";
        parameter Real subsist_food_pc(unit = "kg/yr") = 230 "Available per capita food";
        parameter Real tech_dev_del_TDD(unit = "yr") = 20 "Technology development time";
        parameter Real urb_ind_land_dev_time(unit = "yr") = 10 "Urban and industrial land development time";
        parameter Real t_air_poll_time(unit = "yr") = 4000 "Air pollution change time";
        parameter Real t_fcaor_time(unit = "yr") = 2002 "Year of capital allocation to resource use efficiency";
        parameter Real t_fert_cont_eff_time(unit = "yr") = 4000 "Year of continued fertility change";
        parameter Real t_ind_equil_time(unit = "yr") = 4000 "Year of industrial equilibrium";
        parameter Real t_land_life_time(unit = "yr") = 4000 "Land life time";
        parameter Real t_policy_year(unit = "yr") = 4000 "Year of policy change";
        parameter Real t_pop_equil_time(unit = "yr") = 4000 "Population equilibrium time";
        parameter Real t_zero_pop_grow_time(unit = "yr") = 4000 "Time to zero population growth";
        parameter Real p_fr_cap_al_obt_res_2[:] = {1,0.1,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05} "Non-renewable resource fraction remaining";
        parameter Real p_ppoll_tech_chg_mlt[:] = {0,0,0,0} "Persistent pollution technology change multiplier";
        parameter Real p_res_tech_chg_mlt[:] = {0,0,0,0} "Resource technology change multiplier";
        parameter Real p_yield_tech_chg_mlt[:] = {0,0,0,0} "Yield technology change multiplier";
        parameter Real agr_inp_init(unit = "dollar/yr") = 5000000000.0 "Initial agricultural input";
        parameter Real arable_land_init(unit = "hectare") = 900000000.0 "Initial arable land";
        parameter Real industrial_capital_init(unit = "dollar") = 210000000000.0 "Initial industrial investment";
        parameter Real labor_util_fr_del_init = 1 "Initial delayed labor utilization fraction";
        parameter Real land_fertility_init(unit = "kg/(hectare.yr)") = 600 "Initial industrial investment";
        parameter Real nr_resources_init(unit = "ton") = 2000000000000.0 "Initial available non-recoverable resources";
        parameter Real perc_food_ratio_init = 1 "Initial perceived food ratio";
        parameter Real pers_pollution_init = 25000000.0 "Initial persistent pollution";
        parameter Real pop1_init = 650000000.0 "Initial population 14 years and younger";
        parameter Real pop2_init = 700000000.0 "Initial population 15 to 44 years old";
        parameter Real pop3_init = 190000000.0 "Initial population 45 to 64 years old";
        parameter Real pop4_init = 60000000.0 "Initial population 65 years and older";
        parameter Real pot_arable_land_init(unit = "hectare") = 2300000000.0 "Initial potential arable land";
        parameter Real ppoll_tech_init = 1 "Initial persistent pollution technology change factor";
        parameter Real res_tech_init = 1 "Initial non-recoverable resource technology factor";
        parameter Real service_capital_init(unit = "dollar") = 144000000000.0 "Initial service sector investment";
        parameter Real urban_ind_land_init(unit = "hectare") = 8200000.0 "Initial urban and industrial land";
        parameter Real yield_tech_init = 1 "Initial yield technology factor";
        output Real population "Total human world population";
        output Real food(unit = "dollar/yr") "Total annually produced food";
        output Real industrial_output(unit = "dollar/yr") "Total annual world industrial output";
        output Real ppoll_index "Persistent pollution index";
        output Real nr_resources "Remaining non-recoverable natural resources";
        output Real fioa_ind "Fraction of industrial output allocated to industrial/military complex";
        output Real s_fioa_agr "Fraction of industrial output allocated to food production";
        output Real s_fioa_cons "Fraction of industrial output allocated to consumption";
        output Real s_fioa_serv "Fraction of industrial output allocated to service sector";
        output Real s_fr_cap_al_obt_res "Fraction of capital allocated to resource use efficiency";
        output Real life_expectancy(unit = "yr") "Life expectancy";
        output Real food_pc(unit = "dollar/yr") "Total annual food per person";
        output Real serv_out_pc(unit = "dollar/yr") "Total annual services per person";
        output Real ind_out_pc(unit = "dollar/yr") "Total annual consumer goods per person";
        output Real human_ecological_footprint(unit = "Gha") "Human ecological footprint";
        output Real human_welfare_index "Human welfare index";
        SystemDynamics.WorldDynamics.World3.Population_Dynamics Population_Dynamics1(pop1_init = pop1_init, pop2_init = pop2_init, pop3_init = pop3_init, pop4_init = pop4_init, labor_force_partic = labor_force_partic, reproductive_lifetime = reproductive_lifetime, t_pop_equil_time = t_pop_equil_time) "Population dynamics" annotation(Placement(visible = true, transformation(origin = {-100.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Pollution_Dynamics Pollution_Dynamics1(agr_mtl_toxic_index = agr_mtl_toxic_index, assim_half_life_1970 = assim_half_life_1970, des_ppoll_index_DPOLX = des_ppoll_index_DPOLX, fr_agr_inp_pers_mtl = fr_agr_inp_pers_mtl, frac_res_pers_mtl = frac_res_pers_mtl, ind_mtl_emiss_fact = ind_mtl_emiss_fact, ind_mtl_toxic_index = ind_mtl_toxic_index, ind_out_in_1970 = ind_out_in_1970, p_ppoll_gen_fact_1 = p_ppoll_gen_fact_1, pers_pollution_init = pers_pollution_init, ppoll_in_1970 = ppoll_in_1970, ppoll_tech_init = ppoll_tech_init, ppoll_trans_del = ppoll_trans_del, t_air_poll_time = t_air_poll_time, t_policy_year = t_policy_year, tech_dev_del_TDD = tech_dev_del_TDD, p_ppoll_tech_chg_mlt = p_ppoll_tech_chg_mlt) "Persistent pollution generation" annotation(Placement(visible = true, transformation(origin = {-100.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Arable_Land_Dynamics Arable_Land_Dynamics1(arable_land_init = arable_land_init, avg_life_land_norm = avg_life_land_norm, inherent_land_fert = inherent_land_fert, pot_arable_land_init = pot_arable_land_init, pot_arable_land_tot = pot_arable_land_tot, social_discount = social_discount, t_land_life_time = t_land_life_time, urban_ind_land_init = urban_ind_land_init, urb_ind_land_dev_time = urb_ind_land_dev_time) "Arable land dynamics" annotation(Placement(visible = true, transformation(origin = {-100.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Food_Production Food_Production1(agr_inp_init = agr_inp_init, food_short_perc_del = food_short_perc_del, p_avg_life_agr_inp_1 = p_avg_life_agr_inp_1, p_avg_life_agr_inp_2 = p_avg_life_agr_inp_2, p_land_yield_fact_1 = p_land_yield_fact_1, perc_food_ratio_init = perc_food_ratio_init, processing_loss = processing_loss, subsist_food_pc = subsist_food_pc, t_policy_year = t_policy_year, tech_dev_del_TDD = tech_dev_del_TDD, land_fr_harvested = land_fr_harvested) "Food production" annotation(Placement(visible = true, transformation(origin = {-100.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Human_Ecological_Footprint Human_Ecological_Footprint1 "Human ecological footprint" annotation(Placement(visible = true, transformation(origin = {340.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Human_Fertility Human_Fertility1(des_compl_fam_size_norm = des_compl_fam_size_norm, hlth_serv_impact_del = hlth_serv_impact_del, income_expect_avg_time = income_expect_avg_time, lifet_perc_del = lifet_perc_del, max_tot_fert_norm = max_tot_fert_norm, social_adj_del = social_adj_del, t_fert_cont_eff_time = t_fert_cont_eff_time, t_zero_pop_grow_time = t_zero_pop_grow_time) "Human fertility" annotation(Placement(visible = true, transformation(origin = {120.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Human_Welfare_Index Human_Welfare_Index1 "Human welfare index" annotation(Placement(visible = true, transformation(origin = {340.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Industrial_Investment Industrial_Investment1(industrial_capital_init = industrial_capital_init, ind_out_pc_des = ind_out_pc_des, p_avg_life_ind_cap_1 = p_avg_life_ind_cap_1, p_avg_life_ind_cap_2 = p_avg_life_ind_cap_2, p_fioa_cons_const_1 = p_fioa_cons_const_1, p_fioa_cons_const_2 = p_fioa_cons_const_2, p_ind_cap_out_ratio_1 = p_ind_cap_out_ratio_1, t_ind_equil_time = t_ind_equil_time, t_policy_year = t_policy_year) "Industrial investment" annotation(Placement(visible = true, transformation(origin = {120.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Labor_Utilization Labor_Utilization1(labor_util_fr_del_init = labor_util_fr_del_init, labor_util_fr_del_time = labor_util_fr_del_time) "Labor utilization" annotation(Placement(visible = true, transformation(origin = {120.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Land_Fertility Land_Fertility1(des_food_ratio_dfr = des_food_ratio_dfr, inherent_land_fert = inherent_land_fert, land_fertility_init = land_fertility_init, t_policy_year = t_policy_year, yield_tech_init = yield_tech_init, p_yield_tech_chg_mlt = p_yield_tech_chg_mlt) "Land fertility" annotation(Placement(visible = true, transformation(origin = {120.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Life_Expectancy Life_Expectancy1(hlth_serv_impact_del = hlth_serv_impact_del, life_expect_norm = life_expect_norm, subsist_food_pc = subsist_food_pc) "Life expectancy" annotation(Placement(visible = true, transformation(origin = {340.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.NR_Resource_Utilization NR_Resource_Utilization1(des_res_use_rt_DNRUR = des_res_use_rt_DNRUR, nr_resources_init = nr_resources_init, p_nr_res_use_fact_1 = p_nr_res_use_fact_1, res_tech_init = res_tech_init, t_policy_year = t_policy_year, t_fcaor_time = t_fcaor_time, tech_dev_del_TDD = tech_dev_del_TDD, p_res_tech_chg_mlt = p_res_tech_chg_mlt, p_fr_cap_al_obt_res_2 = p_fr_cap_al_obt_res_2) "Non-recoverable natural resource utilization" annotation(Placement(visible = true, transformation(origin = {340.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Service_Sector_Investment Service_Sector_Investment1(t_policy_year = t_policy_year, p_avg_life_serv_cap_1 = p_avg_life_serv_cap_1, p_avg_life_serv_cap_2 = p_avg_life_serv_cap_2, p_serv_cap_out_ratio_1 = p_serv_cap_out_ratio_1, p_serv_cap_out_ratio_2 = p_serv_cap_out_ratio_2, service_capital_init = service_capital_init) "Service sector investment" annotation(Placement(visible = true, transformation(origin = {-100.0,-220.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
      equation
        population = Population_Dynamics1.Population.y;
        food = Food_Production1.Food.food;
        industrial_output = Industrial_Investment1.Industrial_Output.industrial_output;
        ppoll_index = Pollution_Dynamics1.PPoll_Index.y;
        nr_resources = NR_Resource_Utilization1.NR_Resources.y;
        fioa_ind = Industrial_Investment1.FIOA_Ind.fioa_ind;
        s_fioa_agr = Food_Production1.S_FIOA_Agr.s_fioa_agr;
        s_fioa_cons = Industrial_Investment1.S_FIOA_Cons.s_fioa_cons;
        s_fioa_serv = Service_Sector_Investment1.S_FIOA_Serv.s_fioa_serv;
        s_fr_cap_al_obt_res = NR_Resource_Utilization1.S_Fr_Cap_Al_Obt_Res.s_fr_cap_al_obt_res;
        life_expectancy = Life_Expectancy1.Life_Expectancy.y;
        food_pc = Food_Production1.Food_PC.y;
        serv_out_pc = Service_Sector_Investment1.Serv_Out_PC.y;
        ind_out_pc = Industrial_Investment1.Ind_Out_PC.y;
        human_ecological_footprint = Human_Ecological_Footprint1.HEF_Human_Ecological_Footprint.hef_human_ecological_footprint;
        human_welfare_index = Human_Welfare_Index1.HWI_Human_Welfare_Index.hwi_human_welfare_index;
        connect(Population_Dynamics1.population,Pollution_Dynamics1.population) annotation(Line(points = {{-78.0,190.0},{-10.0,190.0},{-10.0,140.0},{-180.0,140.0},{-180.0,88.0},{-122.0,88.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.population,Arable_Land_Dynamics1.population) annotation(Line(points = {{-122.0,88.0},{-180.0,88.0},{-180.0,-36.0},{-122.0,-36.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.population,Food_Production1.population) annotation(Line(points = {{-122.0,-36.0},{-180.0,-36.0},{-180.0,-104.0},{-122.0,-104.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.population,Service_Sector_Investment1.population) annotation(Line(points = {{-122.0,-104.0},{-180.0,-104.0},{-180.0,-216.0},{-122.0,-216.0}}, color = {0,0,191}, visible = true));
        connect(Population_Dynamics1.population,Industrial_Investment1.population) annotation(Line(points = {{-78.0,190.0},{-10.0,190.0},{-10.0,34.0},{116.0,34.0},{116.0,58.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.population,Life_Expectancy1.population) annotation(Line(points = {{116.0,58.0},{116.0,34.0},{260.0,34.0},{260.0,-20.0},{318.0,-20.0}}, color = {0,0,191}, visible = true));
        connect(Life_Expectancy1.population,NR_Resource_Utilization1.population) annotation(Line(points = {{318.0,-20.0},{260.0,-20.0},{260.0,-120.0},{318.0,-120.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.arable_land,Pollution_Dynamics1.arable_land) annotation(Line(points = {{-78.0,-20.0},{-20.0,-20.0},{-20.0,26.0},{-176.0,26.0},{-176.0,64.0},{-122.0,64.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.arable_land,Food_Production1.arable_land) annotation(Line(points = {{-122.0,64.0},{-176.0,64.0},{-176.0,-112.0},{-122.0,-112.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.arable_land,Labor_Utilization1.arable_land) annotation(Line(points = {{-78.0,-20.0},{40.0,-20.0},{40.0,-36.0},{98.0,-36.0}}, color = {0,0,191}, visible = true));
        connect(Labor_Utilization1.arable_land,Human_Ecological_Footprint1.arable_land) annotation(Line(points = {{98.0,-36.0},{40.0,-36.0},{40.0,-64.0},{264.0,-64.0},{264.0,190.0},{318.0,190.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.industrial_output,Pollution_Dynamics1.industrial_output) annotation(Line(points = {{142.0,84.3414},{210.0,84.3414},{210.0,22.0},{-184.0,22.0},{-184.0,80.0},{-122.0,80.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.industrial_output,Food_Production1.industrial_output) annotation(Line(points = {{-122.0,80.0},{-184.0,80.0},{-184.0,-136.0},{-122.0,-136.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.industrial_output,Service_Sector_Investment1.industrial_output) annotation(Line(points = {{-122.0,-136.0},{-184.0,-136.0},{-184.0,-224.0},{-122.0,-224.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.industrial_output,NR_Resource_Utilization1.industrial_output) annotation(Line(points = {{142.0,84.3414},{210.0,84.3414},{210.0,-130.0},{318.0,-130.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.ind_out_pc,Arable_Land_Dynamics1.ind_out_pc) annotation(Line(points = {{142.0,68.0},{200.0,68.0},{200.0,30.0},{-188.0,30.0},{-188.0,-28.0},{-122.0,-28.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.ind_out_pc,Food_Production1.ind_out_pc) annotation(Line(points = {{-122.0,-28.0},{-188.0,-28.0},{-188.0,-128.0},{-122.0,-128.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.ind_out_pc,Service_Sector_Investment1.ind_out_pc) annotation(Line(points = {{-122.0,-128.0},{-188.0,-128.0},{-188.0,-232.0},{-122.0,-232.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.ind_out_pc,Human_Fertility1.ind_out_pc) annotation(Line(points = {{142.0,68.0},{200.0,68.0},{200.0,30.0},{36.0,30.0},{36.0,170.0},{98.0,170.0}}, color = {0,0,191}, visible = true));
        connect(Human_Fertility1.ind_out_pc,Labor_Utilization1.ind_out_pc) annotation(Line(points = {{98.0,170.0},{36.0,170.0},{36.0,-12.0},{98.0,-12.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.ind_out_pc,Human_Welfare_Index1.ind_out_pc) annotation(Line(points = {{142.0,68.0},{200.0,68.0},{200.0,70.0},{318.0,70.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.ind_out_pc,Life_Expectancy1.ind_out_pc) annotation(Line(points = {{142.0,68.0},{200.0,68.0},{200.0,-12.0},{318.0,-12.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.ind_out_pc,NR_Resource_Utilization1.ind_out_pc) annotation(Line(points = {{142.0,68.0},{200.0,68.0},{200.0,-110.0},{318.0,-110.0}}, color = {0,0,191}, visible = true));
        connect(Service_Sector_Investment1.serv_out_pc,Human_Fertility1.serv_out_pc) annotation(Line(points = {{-78.0,-210.0},{0.0,-210.0},{0.0,190.0},{98.0,190.0}}, color = {0,0,191}, visible = true));
        connect(Service_Sector_Investment1.serv_out_pc,Labor_Utilization1.serv_out_pc) annotation(Line(points = {{-78.0,-210.0},{0.0,-210.0},{0.0,18.0},{70.0,18.0},{108.0,18.0},{108.0,2.0}}, color = {0,0,191}, visible = true));
        connect(Life_Expectancy1.serv_out_pc,Labor_Utilization1.serv_out_pc) annotation(Line(points = {{318.0,-4.0},{218.0,-4.0},{218.0,18.0},{108.0,18.0},{108.0,11.6909},{108.0,2.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.ppoll_index,Land_Fertility1.ppoll_index) annotation(Line(points = {{-78.0,80.0},{20.0,80.0},{20.0,-130.0},{98.0,-130.0}}, color = {0,0,191}, visible = true));
        connect(Land_Fertility1.ppoll_index,Life_Expectancy1.ppoll_index) annotation(Line(points = {{98.0,-130.0},{20.0,-130.0},{20.0,-70.0},{268.0,-70.0},{268.0,-36.0},{318.0,-36.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.food_ratio,Land_Fertility1.food_ratio) annotation(Line(points = {{-78.0,-104.0},{10.0,-104.0},{10.0,-110.0},{98.0,-110.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.land_yield,Arable_Land_Dynamics1.land_yield) annotation(Line(points = {{-78.0,-112.0},{-20.0,-112.0},{-20.0,-60.0},{-192.0,-60.0},{-192.0,-4.0},{-122.0,-4.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.food_pc,Life_Expectancy1.food_pc) annotation(Line(points = {{-96.0,-98.0},{-96.0,-74.0},{256.0,-74.0},{256.0,-28.0},{318.0,-28.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.marg_prod_agr_inp,Arable_Land_Dynamics1.marg_prod_agr_inp) annotation(Line(points = {{-78.0,-120.0},{-24.0,-120.0},{-24.0,-64.0},{-172.0,-64.0},{-172.0,-12.0},{-122.0,-12.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.agr_inp_per_hect,Pollution_Dynamics1.agr_inp_per_hect) annotation(Line(points = {{-78.0,-128.0},{-16.0,-128.0},{-16.0,-56.0},{-196.0,-56.0},{-196.0,72.0},{-122.0,72.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.agr_inp_per_hect,Labor_Utilization1.agr_inp_per_hect) annotation(Line(points = {{-78.0,-128.0},{-16.0,-128.0},{-16.0,-28.0},{98.0,-28.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.s_fioa_agr,Industrial_Investment1.s_fioa_agr) annotation(Line(points = {{-78.0,-136.0},{-4.0,-136.0},{-4.0,64.0},{98.0,64.0}}, color = {0,0,191}, visible = true));
        connect(Service_Sector_Investment1.service_capital,Labor_Utilization1.service_capital) annotation(Line(points = {{-78.0,-220.0},{4.0,-220.0},{4.0,-4.0},{98.0,-4.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.p_fr_inp_for_land_maint,Land_Fertility1.p_fr_inp_for_land_maint) annotation(Line(points = {{-88.0,-142.0},{-88.0,-160.0},{44.0,-160.0},{44.0,-120.0},{98.0,-120.0}}, color = {0,0,191}, visible = true));
        connect(Service_Sector_Investment1.s_fioa_serv,Industrial_Investment1.s_fioa_serv) annotation(Line(points = {{-78.0,-230.0},{16.0,-230.0},{16.0,38.0},{108.0,38.0},{108.0,58.0}}, color = {0,0,191}, visible = true));
        connect(Population_Dynamics1.labor_force,Labor_Utilization1.labor_force) annotation(Line(points = {{-78.0,170.0},{12.0,170.0},{12.0,-60.0},{108.0,-60.0},{108.0,-42.0}}, color = {0,0,191}, visible = true));
        connect(Human_Fertility1.total_fertility,Population_Dynamics1.total_fertility) annotation(Line(points = {{142.0,180.0},{196.0,180.0},{196.0,144.0},{-200.0,144.0},{-200.0,190.0},{-122.0,190.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.industrial_capital,Labor_Utilization1.industrial_capital) annotation(Line(points = {{142.0,76.0},{196.0,76.0},{196.0,-78.0},{44.0,-78.0},{44.0,-20.0},{98.0,-20.0}}, color = {0,0,191}, visible = true));
        connect(Land_Fertility1.land_fertility,Food_Production1.land_fertility) annotation(Line(points = {{142.0,-120.0},{192.0,-120.0},{192.0,-164.0},{-112.0,-164.0},{-112.0,-142.0}}, color = {0,0,191}, visible = true));
        connect(Life_Expectancy1.life_expectancy,Human_Welfare_Index1.life_expectancy) annotation(Line(points = {{362.0,-20.0},{420.0,-20.0},{420.0,40.0},{250.0,40.0},{250.0,90.0},{318.0,90.0}}, color = {0,0,191}, visible = true));
        connect(Human_Welfare_Index1.life_expectancy,Human_Fertility1.life_expectancy) annotation(Line(points = {{318.0,90.0},{250.0,90.0},{250.0,128.0},{32.0,128.0},{32.0,180.0},{98.0,180.0}}, color = {0,0,191}, visible = true));
        connect(Human_Fertility1.life_expectancy,Population_Dynamics1.life_expectancy) annotation(Line(points = {{98.0,180.0},{32.0,180.0},{32.0,128.0},{-190.0,128.0},{-190.0,170.0},{-122.0,170.0}}, color = {0,0,191}, visible = true));
        connect(NR_Resource_Utilization1.pc_res_use_mlt,Pollution_Dynamics1.pc_res_use_mlt) annotation(Line(points = {{362.0,-108.0},{424.0,-108.0},{424.0,124.0},{-176.0,124.0},{-176.0,96.0},{-122.0,96.0}}, color = {0,0,191}, visible = true));
        connect(Land_Fertility1.yield_tech_LYTD,Food_Production1.yield_tech_LYTD) annotation(Line(points = {{142.0,-110.0},{192.0,-110.0},{192.0,-82.0},{-112.0,-82.0},{-112.0,-84.9128},{-112.0,-98.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.fr_inp_al_land_dev,Food_Production1.fr_inp_al_land_dev) annotation(Line(points = {{-78.0,-10.0},{-28.0,-10.0},{-28.0,-68.0},{-172.0,-68.0},{-172.0,-120.0},{-122.0,-120.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.s_yield_mlt_air_poll,Food_Production1.s_yield_mlt_air_poll) annotation(Line(points = {{-78.0,72.0},{32.0,72.0},{32.0,-168.0},{-104.0,-168.0},{-104.0,-142.0}}, color = {0,0,191}, visible = true));
        connect(Labor_Utilization1.capital_util_fr,Service_Sector_Investment1.capital_util_fr) annotation(Line(points = {{142.0,-28.0},{224.0,-28.0},{224.0,-180.0},{-172.0,-180.0},{-172.0,-208.0},{-122.0,-208.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.ppoll_tech_mult_icor_COPM,Industrial_Investment1.ppoll_tech_mult_icor_COPM) annotation(Line(points = {{-88.0,0.0},{88.0,0.0}}, color = {0,0,191}, visible = true, origin = {10.0,88.0}));
        connect(NR_Resource_Utilization1.ind_cap_out_ratio_2_ICOR2T,Industrial_Investment1.ind_cap_out_ratio_2_ICOR2T) annotation(Line(points = {{362.0,-124.0},{428.0,-124.0},{428.0,120.0},{108.0,120.0},{108.0,102.0}}, color = {0,0,191}, visible = true));
        connect(Labor_Utilization1.capital_util_fr,Industrial_Investment1.capital_util_fr) annotation(Line(points = {{142.0,-28.0},{224.0,-28.0},{224.0,-180.0},{24.0,-180.0},{24.0,80.0},{98.0,80.0}}, color = {0,0,191}, visible = true));
        connect(NR_Resource_Utilization1.s_fr_cap_al_obt_res,Industrial_Investment1.s_fr_cap_al_obt_res) annotation(Line(points = {{362.0,-132.0},{432.0,-132.0},{432.0,132.0},{44.0,132.0},{44.0,72.0},{98.0,72.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.ppoll_gen_rt,Human_Ecological_Footprint1.ppoll_gen_rt) annotation(Line(points = {{-78.0,64.0},{-20.0,64.0},{-20.0,148.0},{256.0,148.0},{256.0,170.0},{318.0,170.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.urban_ind_land,Human_Ecological_Footprint1.urban_ind_land) annotation(Line(points = {{-78.0,-30.0},{-24.0,-30.0},{-24.0,136.0},{252.0,136.0},{252.0,180.0},{318.0,180.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.yield_tech_mult_icor_COYM,Industrial_Investment1.yield_tech_mult_icor_COYM) annotation(Line(points = {{-96.0,-142.0},{-96.0,-174.0},{28.0,-174.0},{28.0,96.0},{98.0,96.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.tot_agric_invest,Arable_Land_Dynamics1.tot_agric_invest) annotation(Line(points = {{-88.0,-98.0},{-88.0,-52.0},{-200.0,-52.0},{-200.0,-20.0},{-122.0,-20.0}}, color = {0,0,191}, visible = true));
        annotation(Icon(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-222.0,-260.0},{440.0,222.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-70.0,58.0},{286.0,132.0}}, textString = "More Abundant", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-84.0,-14.0},{302.0,52.0}}, textString = "Non-recoverable", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-114.0,-106.0},{324.0,-2.0}}, textString = "Natural Resources", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10})), experiment(StartTime = 1900, StopTime = 2100), experimentSetupOutput, Documentation(info = "<html>
This is <font color=red><b>Scenario #2</b></font> of the <font color=red><b>WORLD3</b></font> model. In the original scenario, industry stopped growing around 2040 due to an exhaustion of the remaining non-recoverable natural resources. <p>
 
In this modified scenario, the initially available non-recoverable natural resources are doubled to make them last longer.  This is a reasonable assumption, as indeed, the projections of available resources have repetitively been raised during recent decades.  Consequently, industry is able to grow unabatedly for 20 more years, before the increasing extraction cost finally puts a damper on its further growth.  Also postulated in this scenario are advances in resource extraction technology that make it cheaper and more efficient to produce the available resources. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Meadows, D.H., D.L. Meadows, J. Randers, and W.W. Behrens III (1972), <i>Limits to Growth: A Report for the Club of Rome's Project on the Predicament of Mankind</i>, Universe Books, New York, 205p.
<li> Meadows, D.L., W.W. Behrens III, D.M., Meadows, R.F. Naill, J. Randers, and E.K.O. Zahn (1974), <a href=\"http://www.pegasuscom.com/BookDetail.asp?BookQuery_Action=Find('ISBN','XDYGO')\">Dynamics of Growth in a Finite World</a>, Wright-Allen Press, 637p.
<li> Meadows, D.H., D.L. Meadows, and J. Randers (1992), <i>Beyond the Limits</i>, Chelsea Green, 300p.
<li> Meadows, D.H., J. Randers, and D.L. Meadows (2004), <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a>, Chelsea Green, 368p.
</ol> <p>
 
<hr> <p>
 
In order to accomplish this change, you need to modify the initial value of the natural resources as follows: <p>
<font color=red>parameter Real nr_resources_init(unit=\"ton\") = <b>2e12</b> \"Initial available non-recoverable resources\";</font>, <p>
you need to modify the table P_Fr_Cap_Al_Obt_Res_2 that describes the fraction of capital allocated to obtaining resources: <p>
<font color=red>parameter Real p_fr_cap_al_obt_res_2[:] = {1,<b>0.1</b>,<b>0.05</b>,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05} \"Non-renewable resource fraction remaining\";</font>, <p>
and finally, you need to reset one of the switching times in the model: <p>
<font color=red>parameter Real t_fcaor_time(unit=\"yr\") = <b>2002</b> \"Year of capital allocation to resource use efficiency\";</font>. <p>
 
<hr> <p>
 
Simulate the model from 1900 until 2100, and display the same variables as in the book <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a> at page 173: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_4.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_5.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_6.png\"> <p>
 
<hr> <p>
</html>", revisions = ""));
      end Scenario_2;
      model Scenario_3 "More accessible non-recoverable natural resources and pollution control"
        parameter Real agr_mtl_toxic_index(unit = "1/dollar") = 1 "Agricultural materials toxicity index";
        parameter Real assim_half_life_1970(unit = "yr") = 1.5 "Pollution assimilation half life in 1970";
        parameter Real avg_life_land_norm(unit = "yr") = 1000 "Normal life span of land";
        parameter Real des_compl_fam_size_norm = 3.8 "Desired normal complete family size";
        parameter Real des_food_ratio_dfr = 2 "Desired food ratio";
        parameter Real des_ppoll_index_DPOLX = 1.2 "Desired persistent pollution index";
        parameter Real des_res_use_rt_DNRUR(unit = "ton/yr") = 4800000000.0 "Desired resource utilization rate";
        parameter Real food_short_perc_del(unit = "yr") = 2 "Food shortage perception delay";
        parameter Real fr_agr_inp_pers_mtl = 0.001 "Effective fraction of agricultural pollution input";
        parameter Real frac_res_pers_mtl = 0.02 "Effective fraction of resource utilization on pollution generation";
        parameter Real hlth_serv_impact_del(unit = "yr") = 20 "Health service impact delay";
        parameter Real income_expect_avg_time(unit = "yr") = 3 "Income expected average time";
        parameter Real ind_mtl_emiss_fact(unit = "1/ton") = 0.1 "Industrial materials emission factor";
        parameter Real ind_mtl_toxic_index = 10.0 "Industrial materials toxicity index";
        parameter Real ind_out_pc_des(unit = "dollar/yr") = 400 "Desired annual industrial per capita output";
        parameter Real ind_out_in_1970(unit = "dollar/yr") = 790000000000.0 "Industrial output in 1970";
        parameter Real inherent_land_fert(unit = "kg/(hectare.yr)") = 600 "Inherent land fertility";
        parameter Real labor_force_partic = 0.75 "Percentage of participating labor force";
        parameter Real labor_util_fr_del_time(unit = "yr") = 2 "Labor utilization fraction delay time";
        parameter Real land_fr_harvested = 0.7 "Land fraction harvested";
        parameter Real life_expect_norm(unit = "yr") = 28 "Normal life expectancy";
        parameter Real lifet_perc_del(unit = "yr") = 20 "Perceived life-time delay";
        parameter Real max_tot_fert_norm = 12 "Normal maximal total fertility";
        parameter Real p_avg_life_agr_inp_1(unit = "yr") = 2 "Default average life of agricultural input";
        parameter Real p_avg_life_agr_inp_2(unit = "yr") = 2 "Controlled average life of agricultural input";
        parameter Real p_avg_life_ind_cap_1(unit = "yr") = 14 "Default average life of industrial capital";
        parameter Real p_avg_life_ind_cap_2(unit = "yr") = 14 "Controlled average life of industrial capital";
        parameter Real p_avg_life_serv_cap_1(unit = "yr") = 20 "Default average life of service sector capital";
        parameter Real p_avg_life_serv_cap_2(unit = "yr") = 20 "Controlled average life of service sector capital";
        parameter Real p_fioa_cons_const_1 = 0.43 "Default fraction of industrial output allocated to consumption";
        parameter Real p_fioa_cons_const_2 = 0.43 "Controlled fraction of industrial output allocated to consumption";
        parameter Real p_ind_cap_out_ratio_1(unit = "yr") = 3 "Default industrial capital output ratio";
        parameter Real p_land_yield_fact_1 = 1 "Default land yield factor";
        parameter Real p_nr_res_use_fact_1 = 1 "Default non-recoverable resource utilization factor";
        parameter Real p_ppoll_gen_fact_1 = 1 "Default persistent pollution generation factor";
        parameter Real p_serv_cap_out_ratio_1(unit = "yr") = 1.0 "Default fraction of service sector output ratio";
        parameter Real p_serv_cap_out_ratio_2(unit = "yr") = 1.0 "Controlled fraction of service sector output ratio";
        parameter Real pot_arable_land_tot(unit = "hectare") = 3200000000.0 "Total potential arable land";
        parameter Real ppoll_in_1970 = 136000000.0 "Persistent pollution in 1970";
        parameter Real ppoll_trans_del(unit = "yr") = 20 "Persistent pollution transmission delay";
        parameter Real processing_loss = 0.1 "Processing loss";
        parameter Real reproductive_lifetime(unit = "yr") = 30.0 "Reproductive life time";
        parameter Real social_adj_del(unit = "yr") = 20 "Social adjustment delay";
        parameter Real social_discount(unit = "1/yr") = 0.07 "Social discount";
        parameter Real subsist_food_pc(unit = "kg/yr") = 230 "Available per capita food";
        parameter Real tech_dev_del_TDD(unit = "yr") = 20 "Technology development time";
        parameter Real urb_ind_land_dev_time(unit = "yr") = 10 "Urban and industrial land development time";
        parameter Real t_air_poll_time(unit = "yr") = 4000 "Air pollution change time";
        parameter Real t_fcaor_time(unit = "yr") = 2002 "Year of capital allocation to resource use efficiency";
        parameter Real t_fert_cont_eff_time(unit = "yr") = 4000 "Year of continued fertility change";
        parameter Real t_ind_equil_time(unit = "yr") = 4000 "Year of industrial equilibrium";
        parameter Real t_land_life_time(unit = "yr") = 4000 "Land life time";
        parameter Real t_policy_year(unit = "yr") = 2002 "Year of policy change";
        parameter Real t_pop_equil_time(unit = "yr") = 4000 "Population equilibrium time";
        parameter Real t_zero_pop_grow_time(unit = "yr") = 4000 "Time to zero population growth";
        parameter Real p_fr_cap_al_obt_res_2[:] = {1,0.1,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05} "Non-renewable resource fraction remaining";
        parameter Real p_ppoll_tech_chg_mlt[:] = {-0.04,-0.04,0,0} "Persistent pollution technology change multiplier";
        parameter Real p_res_tech_chg_mlt[:] = {0,0,0,0} "Resource technology change multiplier";
        parameter Real p_yield_tech_chg_mlt[:] = {0,0,0,0} "Yield technology change multiplier";
        parameter Real agr_inp_init(unit = "dollar/yr") = 5000000000.0 "Initial agricultural input";
        parameter Real arable_land_init(unit = "hectare") = 900000000.0 "Initial arable land";
        parameter Real industrial_capital_init(unit = "dollar") = 210000000000.0 "Initial industrial investment";
        parameter Real labor_util_fr_del_init = 1 "Initial delayed labor utilization fraction";
        parameter Real land_fertility_init(unit = "kg/(hectare.yr)") = 600 "Initial industrial investment";
        parameter Real nr_resources_init(unit = "ton") = 2000000000000.0 "Initial available non-recoverable resources";
        parameter Real perc_food_ratio_init = 1 "Initial perceived food ratio";
        parameter Real pers_pollution_init = 25000000.0 "Initial persistent pollution";
        parameter Real pop1_init = 650000000.0 "Initial population 14 years and younger";
        parameter Real pop2_init = 700000000.0 "Initial population 15 to 44 years old";
        parameter Real pop3_init = 190000000.0 "Initial population 45 to 64 years old";
        parameter Real pop4_init = 60000000.0 "Initial population 65 years and older";
        parameter Real pot_arable_land_init(unit = "hectare") = 2300000000.0 "Initial potential arable land";
        parameter Real ppoll_tech_init = 1 "Initial persistent pollution technology change factor";
        parameter Real res_tech_init = 1 "Initial non-recoverable resource technology factor";
        parameter Real service_capital_init(unit = "dollar") = 144000000000.0 "Initial service sector investment";
        parameter Real urban_ind_land_init(unit = "hectare") = 8200000.0 "Initial urban and industrial land";
        parameter Real yield_tech_init = 1 "Initial yield technology factor";
        output Real population "Total human world population";
        output Real food(unit = "dollar/yr") "Total annually produced food";
        output Real industrial_output(unit = "dollar/yr") "Total annual world industrial output";
        output Real ppoll_index "Persistent pollution index";
        output Real nr_resources "Remaining non-recoverable natural resources";
        output Real fioa_ind "Fraction of industrial output allocated to industrial/military complex";
        output Real s_fioa_agr "Fraction of industrial output allocated to food production";
        output Real s_fioa_cons "Fraction of industrial output allocated to consumption";
        output Real s_fioa_serv "Fraction of industrial output allocated to service sector";
        output Real s_fr_cap_al_obt_res "Fraction of capital allocated to resource use efficiency";
        output Real life_expectancy(unit = "yr") "Life expectancy";
        output Real food_pc(unit = "dollar/yr") "Total annual food per person";
        output Real serv_out_pc(unit = "dollar/yr") "Total annual services per person";
        output Real ind_out_pc(unit = "dollar/yr") "Total annual consumer goods per person";
        output Real human_ecological_footprint(unit = "Gha") "Human ecological footprint";
        output Real human_welfare_index "Human welfare index";
        SystemDynamics.WorldDynamics.World3.Population_Dynamics Population_Dynamics1(pop1_init = pop1_init, pop2_init = pop2_init, pop3_init = pop3_init, pop4_init = pop4_init, labor_force_partic = labor_force_partic, reproductive_lifetime = reproductive_lifetime, t_pop_equil_time = t_pop_equil_time) "Population dynamics" annotation(Placement(visible = true, transformation(origin = {-100.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Pollution_Dynamics Pollution_Dynamics1(agr_mtl_toxic_index = agr_mtl_toxic_index, assim_half_life_1970 = assim_half_life_1970, des_ppoll_index_DPOLX = des_ppoll_index_DPOLX, fr_agr_inp_pers_mtl = fr_agr_inp_pers_mtl, frac_res_pers_mtl = frac_res_pers_mtl, ind_mtl_emiss_fact = ind_mtl_emiss_fact, ind_mtl_toxic_index = ind_mtl_toxic_index, ind_out_in_1970 = ind_out_in_1970, p_ppoll_gen_fact_1 = p_ppoll_gen_fact_1, pers_pollution_init = pers_pollution_init, ppoll_in_1970 = ppoll_in_1970, ppoll_tech_init = ppoll_tech_init, ppoll_trans_del = ppoll_trans_del, t_air_poll_time = t_air_poll_time, t_policy_year = t_policy_year, tech_dev_del_TDD = tech_dev_del_TDD, p_ppoll_tech_chg_mlt = p_ppoll_tech_chg_mlt) "Persistent pollution generation" annotation(Placement(visible = true, transformation(origin = {-100.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Arable_Land_Dynamics Arable_Land_Dynamics1(arable_land_init = arable_land_init, avg_life_land_norm = avg_life_land_norm, inherent_land_fert = inherent_land_fert, pot_arable_land_init = pot_arable_land_init, pot_arable_land_tot = pot_arable_land_tot, social_discount = social_discount, t_land_life_time = t_land_life_time, urban_ind_land_init = urban_ind_land_init, urb_ind_land_dev_time = urb_ind_land_dev_time) "Arable land dynamics" annotation(Placement(visible = true, transformation(origin = {-100.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Food_Production Food_Production1(agr_inp_init = agr_inp_init, food_short_perc_del = food_short_perc_del, p_avg_life_agr_inp_1 = p_avg_life_agr_inp_1, p_avg_life_agr_inp_2 = p_avg_life_agr_inp_2, p_land_yield_fact_1 = p_land_yield_fact_1, perc_food_ratio_init = perc_food_ratio_init, processing_loss = processing_loss, subsist_food_pc = subsist_food_pc, t_policy_year = t_policy_year, tech_dev_del_TDD = tech_dev_del_TDD, land_fr_harvested = land_fr_harvested) "Food production" annotation(Placement(visible = true, transformation(origin = {-100.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Human_Ecological_Footprint Human_Ecological_Footprint1 "Human ecological footprint" annotation(Placement(visible = true, transformation(origin = {340.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Human_Fertility Human_Fertility1(des_compl_fam_size_norm = des_compl_fam_size_norm, hlth_serv_impact_del = hlth_serv_impact_del, income_expect_avg_time = income_expect_avg_time, lifet_perc_del = lifet_perc_del, max_tot_fert_norm = max_tot_fert_norm, social_adj_del = social_adj_del, t_fert_cont_eff_time = t_fert_cont_eff_time, t_zero_pop_grow_time = t_zero_pop_grow_time) "Human fertility" annotation(Placement(visible = true, transformation(origin = {120.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Human_Welfare_Index Human_Welfare_Index1 "Human welfare index" annotation(Placement(visible = true, transformation(origin = {340.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Industrial_Investment Industrial_Investment1(industrial_capital_init = industrial_capital_init, ind_out_pc_des = ind_out_pc_des, p_avg_life_ind_cap_1 = p_avg_life_ind_cap_1, p_avg_life_ind_cap_2 = p_avg_life_ind_cap_2, p_fioa_cons_const_1 = p_fioa_cons_const_1, p_fioa_cons_const_2 = p_fioa_cons_const_2, p_ind_cap_out_ratio_1 = p_ind_cap_out_ratio_1, t_ind_equil_time = t_ind_equil_time, t_policy_year = t_policy_year) "Industrial investment" annotation(Placement(visible = true, transformation(origin = {120.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Labor_Utilization Labor_Utilization1(labor_util_fr_del_init = labor_util_fr_del_init, labor_util_fr_del_time = labor_util_fr_del_time) "Labor utilization" annotation(Placement(visible = true, transformation(origin = {120.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Land_Fertility Land_Fertility1(des_food_ratio_dfr = des_food_ratio_dfr, inherent_land_fert = inherent_land_fert, land_fertility_init = land_fertility_init, t_policy_year = t_policy_year, yield_tech_init = yield_tech_init, p_yield_tech_chg_mlt = p_yield_tech_chg_mlt) "Land fertility" annotation(Placement(visible = true, transformation(origin = {120.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Life_Expectancy Life_Expectancy1(hlth_serv_impact_del = hlth_serv_impact_del, life_expect_norm = life_expect_norm, subsist_food_pc = subsist_food_pc) "Life expectancy" annotation(Placement(visible = true, transformation(origin = {340.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.NR_Resource_Utilization NR_Resource_Utilization1(des_res_use_rt_DNRUR = des_res_use_rt_DNRUR, nr_resources_init = nr_resources_init, p_nr_res_use_fact_1 = p_nr_res_use_fact_1, res_tech_init = res_tech_init, t_policy_year = t_policy_year, t_fcaor_time = t_fcaor_time, tech_dev_del_TDD = tech_dev_del_TDD, p_fr_cap_al_obt_res_2 = p_fr_cap_al_obt_res_2, p_res_tech_chg_mlt = p_res_tech_chg_mlt) "Non-recoverable natural resource utilization" annotation(Placement(visible = true, transformation(origin = {340.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Service_Sector_Investment Service_Sector_Investment1(t_policy_year = t_policy_year, p_avg_life_serv_cap_1 = p_avg_life_serv_cap_1, p_avg_life_serv_cap_2 = p_avg_life_serv_cap_2, p_serv_cap_out_ratio_1 = p_serv_cap_out_ratio_1, p_serv_cap_out_ratio_2 = p_serv_cap_out_ratio_2, service_capital_init = service_capital_init) "Service sector investment" annotation(Placement(visible = true, transformation(origin = {-100.0,-220.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
      equation
        connect(Food_Production1.s_fioa_agr,Industrial_Investment1.s_fioa_agr) annotation(Line(visible = true, points = {{-78.0,-136.0},{-4.0,-136.0},{-4.0,64.0},{98.0,64.0}}, color = {0,0,191}));
        population = Population_Dynamics1.Population.y;
        food = Food_Production1.Food.food;
        industrial_output = Industrial_Investment1.Industrial_Output.industrial_output;
        ppoll_index = Pollution_Dynamics1.PPoll_Index.y;
        nr_resources = NR_Resource_Utilization1.NR_Resources.y;
        fioa_ind = Industrial_Investment1.FIOA_Ind.fioa_ind;
        s_fioa_agr = Food_Production1.S_FIOA_Agr.s_fioa_agr;
        s_fioa_cons = Industrial_Investment1.S_FIOA_Cons.s_fioa_cons;
        s_fioa_serv = Service_Sector_Investment1.S_FIOA_Serv.s_fioa_serv;
        s_fr_cap_al_obt_res = NR_Resource_Utilization1.S_Fr_Cap_Al_Obt_Res.s_fr_cap_al_obt_res;
        life_expectancy = Life_Expectancy1.Life_Expectancy.y;
        food_pc = Food_Production1.Food_PC.y;
        serv_out_pc = Service_Sector_Investment1.Serv_Out_PC.y;
        ind_out_pc = Industrial_Investment1.Ind_Out_PC.y;
        human_ecological_footprint = Human_Ecological_Footprint1.HEF_Human_Ecological_Footprint.hef_human_ecological_footprint;
        human_welfare_index = Human_Welfare_Index1.HWI_Human_Welfare_Index.hwi_human_welfare_index;
        connect(Population_Dynamics1.population,Pollution_Dynamics1.population) annotation(Line(points = {{-78.0,190.0},{-10.0,190.0},{-10.0,140.0},{-180.0,140.0},{-180.0,88.0},{-122.0,88.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.population,Arable_Land_Dynamics1.population) annotation(Line(points = {{-122.0,88.0},{-180.0,88.0},{-180.0,-36.0},{-122.0,-36.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.population,Food_Production1.population) annotation(Line(points = {{-122.0,-36.0},{-180.0,-36.0},{-180.0,-104.0},{-122.0,-104.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.population,Service_Sector_Investment1.population) annotation(Line(points = {{-122.0,-104.0},{-180.0,-104.0},{-180.0,-216.0},{-122.0,-216.0}}, color = {0,0,191}, visible = true));
        connect(Population_Dynamics1.population,Industrial_Investment1.population) annotation(Line(points = {{-78.0,190.0},{-10.0,190.0},{-10.0,34.0},{116.0,34.0},{116.0,58.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.population,Life_Expectancy1.population) annotation(Line(points = {{116.0,58.0},{116.0,34.0},{260.0,34.0},{260.0,-20.0},{318.0,-20.0}}, color = {0,0,191}, visible = true));
        connect(Life_Expectancy1.population,NR_Resource_Utilization1.population) annotation(Line(points = {{318.0,-20.0},{260.0,-20.0},{260.0,-120.0},{318.0,-120.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.arable_land,Pollution_Dynamics1.arable_land) annotation(Line(points = {{-78.0,-20.0},{-20.0,-20.0},{-20.0,26.0},{-176.0,26.0},{-176.0,64.0},{-122.0,64.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.arable_land,Food_Production1.arable_land) annotation(Line(points = {{-122.0,64.0},{-176.0,64.0},{-176.0,-112.0},{-122.0,-112.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.arable_land,Labor_Utilization1.arable_land) annotation(Line(points = {{-78.0,-20.0},{40.0,-20.0},{40.0,-36.0},{98.0,-36.0}}, color = {0,0,191}, visible = true));
        connect(Labor_Utilization1.arable_land,Human_Ecological_Footprint1.arable_land) annotation(Line(points = {{98.0,-36.0},{40.0,-36.0},{40.0,-64.0},{264.0,-64.0},{264.0,190.0},{318.0,190.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.industrial_output,Pollution_Dynamics1.industrial_output) annotation(Line(points = {{142.0,84.3414},{210.0,84.3414},{210.0,22.0},{-184.0,22.0},{-184.0,80.0},{-122.0,80.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.industrial_output,Food_Production1.industrial_output) annotation(Line(points = {{-122.0,80.0},{-184.0,80.0},{-184.0,-136.0},{-122.0,-136.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.industrial_output,Service_Sector_Investment1.industrial_output) annotation(Line(points = {{-122.0,-136.0},{-184.0,-136.0},{-184.0,-224.0},{-122.0,-224.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.industrial_output,NR_Resource_Utilization1.industrial_output) annotation(Line(points = {{142.0,84.3414},{210.0,84.3414},{210.0,-130.0},{318.0,-130.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.ind_out_pc,Arable_Land_Dynamics1.ind_out_pc) annotation(Line(points = {{142.0,68.0},{200.0,68.0},{200.0,30.0},{-188.0,30.0},{-188.0,-28.0},{-122.0,-28.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.ind_out_pc,Food_Production1.ind_out_pc) annotation(Line(points = {{-122.0,-28.0},{-188.0,-28.0},{-188.0,-128.0},{-122.0,-128.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.ind_out_pc,Service_Sector_Investment1.ind_out_pc) annotation(Line(points = {{-122.0,-128.0},{-188.0,-128.0},{-188.0,-232.0},{-122.0,-232.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.ind_out_pc,Human_Fertility1.ind_out_pc) annotation(Line(points = {{142.0,68.0},{200.0,68.0},{200.0,30.0},{36.0,30.0},{36.0,170.0},{98.0,170.0}}, color = {0,0,191}, visible = true));
        connect(Human_Fertility1.ind_out_pc,Labor_Utilization1.ind_out_pc) annotation(Line(points = {{98.0,170.0},{36.0,170.0},{36.0,-12.0},{98.0,-12.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.ind_out_pc,Human_Welfare_Index1.ind_out_pc) annotation(Line(points = {{142.0,68.0},{200.0,68.0},{200.0,70.0},{318.0,70.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.ind_out_pc,Life_Expectancy1.ind_out_pc) annotation(Line(points = {{142.0,68.0},{200.0,68.0},{200.0,-12.0},{318.0,-12.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.ind_out_pc,NR_Resource_Utilization1.ind_out_pc) annotation(Line(points = {{142.0,68.0},{200.0,68.0},{200.0,-110.0},{318.0,-110.0}}, color = {0,0,191}, visible = true));
        connect(Service_Sector_Investment1.serv_out_pc,Human_Fertility1.serv_out_pc) annotation(Line(points = {{-78.0,-210.0},{0.0,-210.0},{0.0,190.0},{98.0,190.0}}, color = {0,0,191}, visible = true));
        connect(Service_Sector_Investment1.serv_out_pc,Labor_Utilization1.serv_out_pc) annotation(Line(points = {{-78.0,-210.0},{0.0,-210.0},{0.0,18.0},{70.0,18.0},{108.0,18.0},{108.0,2.0}}, color = {0,0,191}, visible = true));
        connect(Life_Expectancy1.serv_out_pc,Labor_Utilization1.serv_out_pc) annotation(Line(points = {{318.0,-4.0},{218.0,-4.0},{218.0,18.0},{108.0,18.0},{108.0,12.9215},{108.0,2.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.ppoll_index,Land_Fertility1.ppoll_index) annotation(Line(points = {{-78.0,80.0},{20.0,80.0},{20.0,-130.0},{98.0,-130.0}}, color = {0,0,191}, visible = true));
        connect(Land_Fertility1.ppoll_index,Life_Expectancy1.ppoll_index) annotation(Line(points = {{98.0,-130.0},{20.0,-130.0},{20.0,-70.0},{268.0,-70.0},{268.0,-36.0},{318.0,-36.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.food_ratio,Land_Fertility1.food_ratio) annotation(Line(points = {{-78.0,-104.0},{10.0,-104.0},{10.0,-110.0},{98.0,-110.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.land_yield,Arable_Land_Dynamics1.land_yield) annotation(Line(points = {{-78.0,-112.0},{-20.0,-112.0},{-20.0,-60.0},{-192.0,-60.0},{-192.0,-4.0},{-122.0,-4.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.food_pc,Life_Expectancy1.food_pc) annotation(Line(points = {{-96.0,-98.0},{-96.0,-74.0},{256.0,-74.0},{256.0,-28.0},{318.0,-28.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.marg_prod_agr_inp,Arable_Land_Dynamics1.marg_prod_agr_inp) annotation(Line(points = {{-78.0,-120.0},{-24.0,-120.0},{-24.0,-64.0},{-172.0,-64.0},{-172.0,-12.0},{-122.0,-12.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.agr_inp_per_hect,Pollution_Dynamics1.agr_inp_per_hect) annotation(Line(points = {{-78.0,-128.0},{-16.0,-128.0},{-16.0,-56.0},{-196.0,-56.0},{-196.0,72.0},{-122.0,72.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.agr_inp_per_hect,Labor_Utilization1.agr_inp_per_hect) annotation(Line(points = {{-78.0,-128.0},{-16.0,-128.0},{-16.0,-28.0},{98.0,-28.0}}, color = {0,0,191}, visible = true));
        connect(Service_Sector_Investment1.service_capital,Labor_Utilization1.service_capital) annotation(Line(points = {{-78.0,-220.0},{4.0,-220.0},{4.0,-4.0},{98.0,-4.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.p_fr_inp_for_land_maint,Land_Fertility1.p_fr_inp_for_land_maint) annotation(Line(points = {{-88.0,-142.0},{-88.0,-160.0},{44.0,-160.0},{44.0,-120.0},{98.0,-120.0}}, color = {0,0,191}, visible = true));
        connect(Service_Sector_Investment1.s_fioa_serv,Industrial_Investment1.s_fioa_serv) annotation(Line(points = {{-78.0,-230.0},{16.0,-230.0},{16.0,38.0},{108.0,38.0},{108.0,58.0}}, color = {0,0,191}, visible = true));
        connect(Population_Dynamics1.labor_force,Labor_Utilization1.labor_force) annotation(Line(points = {{-78.0,170.0},{12.0,170.0},{12.0,-60.0},{108.0,-60.0},{108.0,-42.0}}, color = {0,0,191}, visible = true));
        connect(Human_Fertility1.total_fertility,Population_Dynamics1.total_fertility) annotation(Line(points = {{142.0,180.0},{196.0,180.0},{196.0,144.0},{-200.0,144.0},{-200.0,190.0},{-122.0,190.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.industrial_capital,Labor_Utilization1.industrial_capital) annotation(Line(points = {{142.0,76.0},{196.0,76.0},{196.0,-78.0},{44.0,-78.0},{44.0,-20.0},{98.0,-20.0}}, color = {0,0,191}, visible = true));
        connect(Land_Fertility1.land_fertility,Food_Production1.land_fertility) annotation(Line(points = {{142.0,-120.0},{192.0,-120.0},{192.0,-164.0},{-112.0,-164.0},{-112.0,-142.0}}, color = {0,0,191}, visible = true));
        connect(Life_Expectancy1.life_expectancy,Human_Welfare_Index1.life_expectancy) annotation(Line(points = {{362.0,-20.0},{420.0,-20.0},{420.0,40.0},{250.0,40.0},{250.0,90.0},{318.0,90.0}}, color = {0,0,191}, visible = true));
        connect(Human_Welfare_Index1.life_expectancy,Human_Fertility1.life_expectancy) annotation(Line(points = {{318.0,90.0},{250.0,90.0},{250.0,128.0},{32.0,128.0},{32.0,180.0},{98.0,180.0}}, color = {0,0,191}, visible = true));
        connect(Human_Fertility1.life_expectancy,Population_Dynamics1.life_expectancy) annotation(Line(points = {{98.0,180.0},{32.0,180.0},{32.0,128.0},{-190.0,128.0},{-190.0,170.0},{-122.0,170.0}}, color = {0,0,191}, visible = true));
        connect(NR_Resource_Utilization1.pc_res_use_mlt,Pollution_Dynamics1.pc_res_use_mlt) annotation(Line(points = {{362.0,-108.0},{424.0,-108.0},{424.0,124.0},{-176.0,124.0},{-176.0,96.0},{-122.0,96.0}}, color = {0,0,191}, visible = true));
        connect(Land_Fertility1.yield_tech_LYTD,Food_Production1.yield_tech_LYTD) annotation(Line(points = {{142.0,-110.0},{192.0,-110.0},{192.0,-82.0},{-112.0,-82.0},{-112.0,-87.374},{-112.0,-98.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.fr_inp_al_land_dev,Food_Production1.fr_inp_al_land_dev) annotation(Line(points = {{-78.0,-10.0},{-28.0,-10.0},{-28.0,-68.0},{-172.0,-68.0},{-172.0,-120.0},{-122.0,-120.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.s_yield_mlt_air_poll,Food_Production1.s_yield_mlt_air_poll) annotation(Line(points = {{-78.0,72.0},{32.0,72.0},{32.0,-168.0},{-104.0,-168.0},{-104.0,-142.0}}, color = {0,0,191}, visible = true));
        connect(Labor_Utilization1.capital_util_fr,Service_Sector_Investment1.capital_util_fr) annotation(Line(points = {{142.0,-28.0},{224.0,-28.0},{224.0,-180.0},{-172.0,-180.0},{-172.0,-208.0},{-122.0,-208.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.ppoll_tech_mult_icor_COPM,Industrial_Investment1.ppoll_tech_mult_icor_COPM) annotation(Line(points = {{-88.0,0.0},{88.0,0.0}}, color = {0,0,191}, visible = true, origin = {10.0,88.0}));
        connect(NR_Resource_Utilization1.ind_cap_out_ratio_2_ICOR2T,Industrial_Investment1.ind_cap_out_ratio_2_ICOR2T) annotation(Line(points = {{362.0,-124.0},{428.0,-124.0},{428.0,120.0},{108.0,120.0},{108.0,102.0}}, color = {0,0,191}, visible = true));
        connect(Labor_Utilization1.capital_util_fr,Industrial_Investment1.capital_util_fr) annotation(Line(points = {{142.0,-28.0},{224.0,-28.0},{224.0,-180.0},{24.0,-180.0},{24.0,80.0},{98.0,80.0}}, color = {0,0,191}, visible = true));
        connect(NR_Resource_Utilization1.s_fr_cap_al_obt_res,Industrial_Investment1.s_fr_cap_al_obt_res) annotation(Line(points = {{362.0,-132.0},{432.0,-132.0},{432.0,132.0},{44.0,132.0},{44.0,72.0},{98.0,72.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.ppoll_gen_rt,Human_Ecological_Footprint1.ppoll_gen_rt) annotation(Line(points = {{-78.0,64.0},{-20.0,64.0},{-20.0,148.0},{256.0,148.0},{256.0,170.0},{318.0,170.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.urban_ind_land,Human_Ecological_Footprint1.urban_ind_land) annotation(Line(points = {{-78.0,-30.0},{-24.0,-30.0},{-24.0,136.0},{252.0,136.0},{252.0,180.0},{318.0,180.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.yield_tech_mult_icor_COYM,Industrial_Investment1.yield_tech_mult_icor_COYM) annotation(Line(points = {{-96.0,-142.0},{-96.0,-174.0},{28.0,-174.0},{28.0,96.0},{98.0,96.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.tot_agric_invest,Arable_Land_Dynamics1.tot_agric_invest) annotation(Line(points = {{-88.0,-98.0},{-88.0,-52.0},{-200.0,-52.0},{-200.0,-20.0},{-122.0,-20.0}}, color = {0,0,191}, visible = true));
        annotation(Icon(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-220.0,-260.0},{442.0,222.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,60.0},{300.0,132.0}}, textString = "More Accessible", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-82.0,-14.0},{304.0,52.0}}, textString = "Non-recoverable", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-112.0,-106.0},{326.0,-2.0}}, textString = "Natural Resources", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-136.0,-184.0},{348.0,-80.0}}, textString = "and Pollution Control", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10})), experiment(StartTime = 1900, StopTime = 2100), experimentSetupOutput, Documentation(info = "<html>
This is <font color=red><b>Scenario #3</b></font> of the <font color=red><b>WORLD3</b></font> model. This scenario starts out with the same assumptions as <font color=red><b>Scenario #2</b></font>.  In that scenario, the continued production of industrial goods caused the persistent pollution to accumulate.  That pollution led to a deterioration of the fertility of our agricultural lands, which caused a collapse over time. <p>
 
For this reason, it is postulated here that improved pollution control technology can curbe the amount of pollutants released into the atmosphere by 4% starting in the year 2002.  The hope is that such a reduction can be reasonably well accomplished without suffocating the industry in the process.  In fact, our government already talk about measures that are considerably more stringent than just a 4% reduction in air pollution, although the already implemented measures have proven ineffective until now.  The greenhouse gas emissions are still growing essentially proportional with the industrial production. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Meadows, D.H., D.L. Meadows, J. Randers, and W.W. Behrens III (1972), <i>Limits to Growth: A Report for the Club of Rome's Project on the Predicament of Mankind</i>, Universe Books, New York, 205p.
<li> Meadows, D.L., W.W. Behrens III, D.M., Meadows, R.F. Naill, J. Randers, and E.K.O. Zahn (1974), <a href=\"http://www.pegasuscom.com/BookDetail.asp?BookQuery_Action=Find('ISBN','XDYGO')\">Dynamics of Growth in a Finite World</a>, Wright-Allen Press, 637p.
<li> Meadows, D.H., D.L. Meadows, and J. Randers (1992), <i>Beyond the Limits</i>, Chelsea Green, 300p.
<li> Meadows, D.H., J. Randers, and D.L. Meadows (2004), <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a>, Chelsea Green, 368p.
</ol> <p>
 
<hr> <p>
 
In order to accomplish this change, you need to modify the table P_PPoll_Tech_Chg_Mlt_POLGFM that models a multiplier to the generation of pollution as a function of available pollution technology: <p>
<font color=red>parameter Real p_ppoll_tech_chg_mlt[:] = {<b>-0.04</b>,<b>-0.04</b>,0,0} \"Persistent pollution technology change multiplier\";</font>, <p>
and you also you need to reset yet another one of the switching times in the model: <p>
<font color=red>parameter Real t_policy_year(unit=\"yr\") = <b>2002</b> \"Year of policy change\";</font>. <p>
The latter change has lots of consequences, as this policy control parameter appears in 16 different places in the model. <p>
 
<hr> <p>
 
Simulate the model from 1900 until 2100, and display the same variables as in the book <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a> at page 211: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_7.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_8.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_9.png\"> <p>
 
<hr> <p>
</html>", revisions = ""));
      end Scenario_3;
      model Scenario_4 "More accessible non-recoverable natural resources, pollution control, and land yield enhancement"
        parameter Real agr_mtl_toxic_index(unit = "1/dollar") = 1 "Agricultural materials toxicity index";
        parameter Real assim_half_life_1970(unit = "yr") = 1.5 "Pollution assimilation half life in 1970";
        parameter Real avg_life_land_norm(unit = "yr") = 1000 "Normal life span of land";
        parameter Real des_compl_fam_size_norm = 3.8 "Desired normal complete family size";
        parameter Real des_food_ratio_dfr = 2 "Desired food ratio";
        parameter Real des_ppoll_index_DPOLX = 1.2 "Desired persistent pollution index";
        parameter Real des_res_use_rt_DNRUR(unit = "ton/yr") = 4800000000.0 "Desired resource utilization rate";
        parameter Real food_short_perc_del(unit = "yr") = 2 "Food shortage perception delay";
        parameter Real fr_agr_inp_pers_mtl = 0.001 "Effective fraction of agricultural pollution input";
        parameter Real frac_res_pers_mtl = 0.02 "Effective fraction of resource utilization on pollution generation";
        parameter Real hlth_serv_impact_del(unit = "yr") = 20 "Health service impact delay";
        parameter Real income_expect_avg_time(unit = "yr") = 3 "Income expected average time";
        parameter Real ind_mtl_emiss_fact(unit = "1/ton") = 0.1 "Industrial materials emission factor";
        parameter Real ind_mtl_toxic_index = 10.0 "Industrial materials toxicity index";
        parameter Real ind_out_pc_des(unit = "dollar/yr") = 400 "Desired annual industrial per capita output";
        parameter Real ind_out_in_1970(unit = "dollar/yr") = 790000000000.0 "Industrial output in 1970";
        parameter Real inherent_land_fert(unit = "kg/(hectare.yr)") = 600 "Inherent land fertility";
        parameter Real labor_force_partic = 0.75 "Percentage of participating labor force";
        parameter Real labor_util_fr_del_time(unit = "yr") = 2 "Labor utilization fraction delay time";
        parameter Real land_fr_harvested = 0.7 "Land fraction harvested";
        parameter Real life_expect_norm(unit = "yr") = 28 "Normal life expectancy";
        parameter Real lifet_perc_del(unit = "yr") = 20 "Perceived life-time delay";
        parameter Real max_tot_fert_norm = 12 "Normal maximal total fertility";
        parameter Real p_avg_life_agr_inp_1(unit = "yr") = 2 "Default average life of agricultural input";
        parameter Real p_avg_life_agr_inp_2(unit = "yr") = 2 "Controlled average life of agricultural input";
        parameter Real p_avg_life_ind_cap_1(unit = "yr") = 14 "Default average life of industrial capital";
        parameter Real p_avg_life_ind_cap_2(unit = "yr") = 14 "Controlled average life of industrial capital";
        parameter Real p_avg_life_serv_cap_1(unit = "yr") = 20 "Default average life of service sector capital";
        parameter Real p_avg_life_serv_cap_2(unit = "yr") = 20 "Controlled average life of service sector capital";
        parameter Real p_fioa_cons_const_1 = 0.43 "Default fraction of industrial output allocated to consumption";
        parameter Real p_fioa_cons_const_2 = 0.43 "Controlled fraction of industrial output allocated to consumption";
        parameter Real p_ind_cap_out_ratio_1(unit = "yr") = 3 "Default industrial capital output ratio";
        parameter Real p_land_yield_fact_1 = 1 "Default land yield factor";
        parameter Real p_nr_res_use_fact_1 = 1 "Default non-recoverable resource utilization factor";
        parameter Real p_ppoll_gen_fact_1 = 1 "Default persistent pollution generation factor";
        parameter Real p_serv_cap_out_ratio_1(unit = "yr") = 1.0 "Default fraction of service sector output ratio";
        parameter Real p_serv_cap_out_ratio_2(unit = "yr") = 1.0 "Controlled fraction of service sector output ratio";
        parameter Real pot_arable_land_tot(unit = "hectare") = 3200000000.0 "Total potential arable land";
        parameter Real ppoll_in_1970 = 136000000.0 "Persistent pollution in 1970";
        parameter Real ppoll_trans_del(unit = "yr") = 20 "Persistent pollution transmission delay";
        parameter Real processing_loss = 0.1 "Processing loss";
        parameter Real reproductive_lifetime(unit = "yr") = 30.0 "Reproductive life time";
        parameter Real social_adj_del(unit = "yr") = 20 "Social adjustment delay";
        parameter Real social_discount(unit = "1/yr") = 0.07 "Social discount";
        parameter Real subsist_food_pc(unit = "kg/yr") = 230 "Available per capita food";
        parameter Real tech_dev_del_TDD(unit = "yr") = 20 "Technology development time";
        parameter Real urb_ind_land_dev_time(unit = "yr") = 10 "Urban and industrial land development time";
        parameter Real t_air_poll_time(unit = "yr") = 4000 "Air pollution change time";
        parameter Real t_fcaor_time(unit = "yr") = 2002 "Year of capital allocation to resource use efficiency";
        parameter Real t_fert_cont_eff_time(unit = "yr") = 4000 "Year of continued fertility change";
        parameter Real t_ind_equil_time(unit = "yr") = 4000 "Year of industrial equilibrium";
        parameter Real t_land_life_time(unit = "yr") = 4000 "Land life time";
        parameter Real t_policy_year(unit = "yr") = 2002 "Year of policy change";
        parameter Real t_pop_equil_time(unit = "yr") = 4000 "Population equilibrium time";
        parameter Real t_zero_pop_grow_time(unit = "yr") = 4000 "Time to zero population growth";
        parameter Real p_fr_cap_al_obt_res_2[:] = {1,0.1,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05} "Non-renewable resource fraction remaining";
        parameter Real p_ppoll_tech_chg_mlt[:] = {-0.04,-0.04,0,0} "Persistent pollution technology change multiplier";
        parameter Real p_res_tech_chg_mlt[:] = {0,0,0,0} "Resource technology change multiplier";
        parameter Real p_yield_tech_chg_mlt[:] = {0,0,0.04,0.04} "Yield technology change multiplier";
        parameter Real agr_inp_init(unit = "dollar/yr") = 5000000000.0 "Initial agricultural input";
        parameter Real arable_land_init(unit = "hectare") = 900000000.0 "Initial arable land";
        parameter Real industrial_capital_init(unit = "dollar") = 210000000000.0 "Initial industrial investment";
        parameter Real labor_util_fr_del_init = 1 "Initial delayed labor utilization fraction";
        parameter Real land_fertility_init(unit = "kg/(hectare.yr)") = 600 "Initial industrial investment";
        parameter Real nr_resources_init(unit = "ton") = 2000000000000.0 "Initial available non-recoverable resources";
        parameter Real perc_food_ratio_init = 1 "Initial perceived food ratio";
        parameter Real pers_pollution_init = 25000000.0 "Initial persistent pollution";
        parameter Real pop1_init = 650000000.0 "Initial population 14 years and younger";
        parameter Real pop2_init = 700000000.0 "Initial population 15 to 44 years old";
        parameter Real pop3_init = 190000000.0 "Initial population 45 to 64 years old";
        parameter Real pop4_init = 60000000.0 "Initial population 65 years and older";
        parameter Real pot_arable_land_init(unit = "hectare") = 2300000000.0 "Initial potential arable land";
        parameter Real ppoll_tech_init = 1 "Initial persistent pollution technology change factor";
        parameter Real res_tech_init = 1 "Initial non-recoverable resource technology factor";
        parameter Real service_capital_init(unit = "dollar") = 144000000000.0 "Initial service sector investment";
        parameter Real urban_ind_land_init(unit = "hectare") = 8200000.0 "Initial urban and industrial land";
        parameter Real yield_tech_init = 1 "Initial yield technology factor";
        output Real population "Total human world population";
        output Real food(unit = "dollar/yr") "Total annually produced food";
        output Real industrial_output(unit = "dollar/yr") "Total annual world industrial output";
        output Real ppoll_index "Persistent pollution index";
        output Real nr_resources "Remaining non-recoverable natural resources";
        output Real fioa_ind "Fraction of industrial output allocated to industrial/military complex";
        output Real s_fioa_agr "Fraction of industrial output allocated to food production";
        output Real s_fioa_cons "Fraction of industrial output allocated to consumption";
        output Real s_fioa_serv "Fraction of industrial output allocated to service sector";
        output Real s_fr_cap_al_obt_res "Fraction of capital allocated to resource use efficiency";
        output Real life_expectancy(unit = "yr") "Life expectancy";
        output Real food_pc(unit = "dollar/yr") "Total annual food per person";
        output Real serv_out_pc(unit = "dollar/yr") "Total annual services per person";
        output Real ind_out_pc(unit = "dollar/yr") "Total annual consumer goods per person";
        output Real human_ecological_footprint(unit = "Gha") "Human ecological footprint";
        output Real human_welfare_index "Human welfare index";
        SystemDynamics.WorldDynamics.World3.Population_Dynamics Population_Dynamics1(pop1_init = pop1_init, pop2_init = pop2_init, pop3_init = pop3_init, pop4_init = pop4_init, labor_force_partic = labor_force_partic, reproductive_lifetime = reproductive_lifetime, t_pop_equil_time = t_pop_equil_time) "Population dynamics" annotation(Placement(visible = true, transformation(origin = {-100.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Pollution_Dynamics Pollution_Dynamics1(agr_mtl_toxic_index = agr_mtl_toxic_index, assim_half_life_1970 = assim_half_life_1970, des_ppoll_index_DPOLX = des_ppoll_index_DPOLX, fr_agr_inp_pers_mtl = fr_agr_inp_pers_mtl, frac_res_pers_mtl = frac_res_pers_mtl, ind_mtl_emiss_fact = ind_mtl_emiss_fact, ind_mtl_toxic_index = ind_mtl_toxic_index, ind_out_in_1970 = ind_out_in_1970, p_ppoll_gen_fact_1 = p_ppoll_gen_fact_1, pers_pollution_init = pers_pollution_init, ppoll_in_1970 = ppoll_in_1970, ppoll_tech_init = ppoll_tech_init, ppoll_trans_del = ppoll_trans_del, t_air_poll_time = t_air_poll_time, t_policy_year = t_policy_year, tech_dev_del_TDD = tech_dev_del_TDD, p_ppoll_tech_chg_mlt = p_ppoll_tech_chg_mlt) "Persistent pollution generation" annotation(Placement(visible = true, transformation(origin = {-100.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Arable_Land_Dynamics Arable_Land_Dynamics1(arable_land_init = arable_land_init, avg_life_land_norm = avg_life_land_norm, inherent_land_fert = inherent_land_fert, pot_arable_land_init = pot_arable_land_init, pot_arable_land_tot = pot_arable_land_tot, social_discount = social_discount, t_land_life_time = t_land_life_time, urban_ind_land_init = urban_ind_land_init, urb_ind_land_dev_time = urb_ind_land_dev_time) "Arable land dynamics" annotation(Placement(visible = true, transformation(origin = {-100.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Food_Production Food_Production1(agr_inp_init = agr_inp_init, food_short_perc_del = food_short_perc_del, p_avg_life_agr_inp_1 = p_avg_life_agr_inp_1, p_avg_life_agr_inp_2 = p_avg_life_agr_inp_2, p_land_yield_fact_1 = p_land_yield_fact_1, perc_food_ratio_init = perc_food_ratio_init, processing_loss = processing_loss, subsist_food_pc = subsist_food_pc, t_policy_year = t_policy_year, tech_dev_del_TDD = tech_dev_del_TDD, land_fr_harvested = land_fr_harvested) "Food production" annotation(Placement(visible = true, transformation(origin = {-100.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Human_Ecological_Footprint Human_Ecological_Footprint1 "Human ecological footprint" annotation(Placement(visible = true, transformation(origin = {340.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Human_Fertility Human_Fertility1(des_compl_fam_size_norm = des_compl_fam_size_norm, hlth_serv_impact_del = hlth_serv_impact_del, income_expect_avg_time = income_expect_avg_time, lifet_perc_del = lifet_perc_del, max_tot_fert_norm = max_tot_fert_norm, social_adj_del = social_adj_del, t_fert_cont_eff_time = t_fert_cont_eff_time, t_zero_pop_grow_time = t_zero_pop_grow_time) "Human fertility" annotation(Placement(visible = true, transformation(origin = {120.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Human_Welfare_Index Human_Welfare_Index1 "Human welfare index" annotation(Placement(visible = true, transformation(origin = {340.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Industrial_Investment Industrial_Investment1(industrial_capital_init = industrial_capital_init, ind_out_pc_des = ind_out_pc_des, p_avg_life_ind_cap_1 = p_avg_life_ind_cap_1, p_avg_life_ind_cap_2 = p_avg_life_ind_cap_2, p_fioa_cons_const_1 = p_fioa_cons_const_1, p_fioa_cons_const_2 = p_fioa_cons_const_2, p_ind_cap_out_ratio_1 = p_ind_cap_out_ratio_1, t_ind_equil_time = t_ind_equil_time, t_policy_year = t_policy_year) "Industrial investment" annotation(Placement(visible = true, transformation(origin = {120.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Labor_Utilization Labor_Utilization1(labor_util_fr_del_init = labor_util_fr_del_init, labor_util_fr_del_time = labor_util_fr_del_time) "Labor utilization" annotation(Placement(visible = true, transformation(origin = {120.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Land_Fertility Land_Fertility1(des_food_ratio_dfr = des_food_ratio_dfr, inherent_land_fert = inherent_land_fert, land_fertility_init = land_fertility_init, t_policy_year = t_policy_year, yield_tech_init = yield_tech_init, p_yield_tech_chg_mlt = p_yield_tech_chg_mlt) "Land fertility" annotation(Placement(visible = true, transformation(origin = {120.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Life_Expectancy Life_Expectancy1(hlth_serv_impact_del = hlth_serv_impact_del, life_expect_norm = life_expect_norm, subsist_food_pc = subsist_food_pc) "Life expectancy" annotation(Placement(visible = true, transformation(origin = {340.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.NR_Resource_Utilization NR_Resource_Utilization1(des_res_use_rt_DNRUR = des_res_use_rt_DNRUR, nr_resources_init = nr_resources_init, p_nr_res_use_fact_1 = p_nr_res_use_fact_1, res_tech_init = res_tech_init, t_policy_year = t_policy_year, t_fcaor_time = t_fcaor_time, tech_dev_del_TDD = tech_dev_del_TDD, p_fr_cap_al_obt_res_2 = p_fr_cap_al_obt_res_2, p_res_tech_chg_mlt = p_res_tech_chg_mlt) "Non-recoverable natural resource utilization" annotation(Placement(visible = true, transformation(origin = {340.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
        SystemDynamics.WorldDynamics.World3.Service_Sector_Investment Service_Sector_Investment1(t_policy_year = t_policy_year, p_avg_life_serv_cap_1 = p_avg_life_serv_cap_1, p_avg_life_serv_cap_2 = p_avg_life_serv_cap_2, p_serv_cap_out_ratio_1 = p_serv_cap_out_ratio_1, p_serv_cap_out_ratio_2 = p_serv_cap_out_ratio_2, service_capital_init = service_capital_init) "Service sector investment" annotation(Placement(visible = true, transformation(origin = {-100.0,-220.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)), Dialog(group = "Variables"));
      equation
        population = Population_Dynamics1.Population.y;
        food = Food_Production1.Food.food;
        industrial_output = Industrial_Investment1.Industrial_Output.industrial_output;
        ppoll_index = Pollution_Dynamics1.PPoll_Index.y;
        nr_resources = NR_Resource_Utilization1.NR_Resources.y;
        fioa_ind = Industrial_Investment1.FIOA_Ind.fioa_ind;
        s_fioa_agr = Food_Production1.S_FIOA_Agr.s_fioa_agr;
        s_fioa_cons = Industrial_Investment1.S_FIOA_Cons.s_fioa_cons;
        s_fioa_serv = Service_Sector_Investment1.S_FIOA_Serv.s_fioa_serv;
        s_fr_cap_al_obt_res = NR_Resource_Utilization1.S_Fr_Cap_Al_Obt_Res.s_fr_cap_al_obt_res;
        life_expectancy = Life_Expectancy1.Life_Expectancy.y;
        food_pc = Food_Production1.Food_PC.y;
        serv_out_pc = Service_Sector_Investment1.Serv_Out_PC.y;
        ind_out_pc = Industrial_Investment1.Ind_Out_PC.y;
        human_ecological_footprint = Human_Ecological_Footprint1.HEF_Human_Ecological_Footprint.hef_human_ecological_footprint;
        human_welfare_index = Human_Welfare_Index1.HWI_Human_Welfare_Index.hwi_human_welfare_index;
        connect(Population_Dynamics1.population,Pollution_Dynamics1.population) annotation(Line(points = {{-78.0,190.0},{-10.0,190.0},{-10.0,140.0},{-180.0,140.0},{-180.0,88.0},{-122.0,88.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.population,Arable_Land_Dynamics1.population) annotation(Line(points = {{-122.0,88.0},{-180.0,88.0},{-180.0,-36.0},{-122.0,-36.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.population,Food_Production1.population) annotation(Line(points = {{-122.0,-36.0},{-180.0,-36.0},{-180.0,-104.0},{-122.0,-104.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.population,Service_Sector_Investment1.population) annotation(Line(points = {{-122.0,-104.0},{-180.0,-104.0},{-180.0,-216.0},{-122.0,-216.0}}, color = {0,0,191}, visible = true));
        connect(Population_Dynamics1.population,Industrial_Investment1.population) annotation(Line(points = {{-78.0,190.0},{-10.0,190.0},{-10.0,34.0},{116.0,34.0},{116.0,58.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.population,Life_Expectancy1.population) annotation(Line(points = {{116.0,58.0},{116.0,34.0},{260.0,34.0},{260.0,-20.0},{318.0,-20.0}}, color = {0,0,191}, visible = true));
        connect(Life_Expectancy1.population,NR_Resource_Utilization1.population) annotation(Line(points = {{318.0,-20.0},{260.0,-20.0},{260.0,-120.0},{318.0,-120.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.arable_land,Pollution_Dynamics1.arable_land) annotation(Line(points = {{-78.0,-20.0},{-20.0,-20.0},{-20.0,26.0},{-176.0,26.0},{-176.0,64.0},{-122.0,64.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.arable_land,Food_Production1.arable_land) annotation(Line(points = {{-122.0,64.0},{-176.0,64.0},{-176.0,-112.0},{-122.0,-112.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.arable_land,Labor_Utilization1.arable_land) annotation(Line(points = {{-78.0,-20.0},{40.0,-20.0},{40.0,-36.0},{98.0,-36.0}}, color = {0,0,191}, visible = true));
        connect(Labor_Utilization1.arable_land,Human_Ecological_Footprint1.arable_land) annotation(Line(points = {{98.0,-36.0},{40.0,-36.0},{40.0,-64.0},{264.0,-64.0},{264.0,190.0},{318.0,190.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.industrial_output,Pollution_Dynamics1.industrial_output) annotation(Line(points = {{142.0,84.3414},{210.0,84.3414},{210.0,22.0},{-184.0,22.0},{-184.0,80.0},{-122.0,80.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.industrial_output,Food_Production1.industrial_output) annotation(Line(points = {{-122.0,80.0},{-184.0,80.0},{-184.0,-136.0},{-122.0,-136.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.industrial_output,Service_Sector_Investment1.industrial_output) annotation(Line(points = {{-122.0,-136.0},{-184.0,-136.0},{-184.0,-224.0},{-122.0,-224.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.industrial_output,NR_Resource_Utilization1.industrial_output) annotation(Line(points = {{142.0,84.3414},{210.0,84.3414},{210.0,-130.0},{318.0,-130.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.ind_out_pc,Arable_Land_Dynamics1.ind_out_pc) annotation(Line(points = {{142.0,68.0},{200.0,68.0},{200.0,30.0},{-188.0,30.0},{-188.0,-28.0},{-122.0,-28.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.ind_out_pc,Food_Production1.ind_out_pc) annotation(Line(points = {{-122.0,-28.0},{-188.0,-28.0},{-188.0,-128.0},{-122.0,-128.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.ind_out_pc,Service_Sector_Investment1.ind_out_pc) annotation(Line(points = {{-122.0,-128.0},{-188.0,-128.0},{-188.0,-232.0},{-122.0,-232.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.ind_out_pc,Human_Fertility1.ind_out_pc) annotation(Line(points = {{142.0,68.0},{200.0,68.0},{200.0,30.0},{36.0,30.0},{36.0,170.0},{98.0,170.0}}, color = {0,0,191}, visible = true));
        connect(Human_Fertility1.ind_out_pc,Labor_Utilization1.ind_out_pc) annotation(Line(points = {{98.0,170.0},{36.0,170.0},{36.0,-12.0},{98.0,-12.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.ind_out_pc,Human_Welfare_Index1.ind_out_pc) annotation(Line(points = {{142.0,68.0},{200.0,68.0},{200.0,70.0},{318.0,70.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.ind_out_pc,Life_Expectancy1.ind_out_pc) annotation(Line(points = {{142.0,68.0},{200.0,68.0},{200.0,-12.0},{318.0,-12.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.ind_out_pc,NR_Resource_Utilization1.ind_out_pc) annotation(Line(points = {{142.0,68.0},{200.0,68.0},{200.0,-110.0},{318.0,-110.0}}, color = {0,0,191}, visible = true));
        connect(Service_Sector_Investment1.serv_out_pc,Human_Fertility1.serv_out_pc) annotation(Line(points = {{-78.0,-210.0},{0.0,-210.0},{0.0,190.0},{98.0,190.0}}, color = {0,0,191}, visible = true));
        connect(Service_Sector_Investment1.serv_out_pc,Labor_Utilization1.serv_out_pc) annotation(Line(points = {{-78.0,-210.0},{0.0,-210.0},{0.0,18.0},{70.0,18.0},{108.0,18.0},{108.0,2.0}}, color = {0,0,191}, visible = true));
        connect(Life_Expectancy1.serv_out_pc,Labor_Utilization1.serv_out_pc) annotation(Line(points = {{318.0,-4.0},{218.0,-4.0},{218.0,18.0},{108.0,18.0},{108.0,10.0},{108.0,2.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.ppoll_index,Land_Fertility1.ppoll_index) annotation(Line(points = {{-78.0,80.0},{20.0,80.0},{20.0,-130.0},{98.0,-130.0}}, color = {0,0,191}, visible = true));
        connect(Land_Fertility1.ppoll_index,Life_Expectancy1.ppoll_index) annotation(Line(points = {{98.0,-130.0},{20.0,-130.0},{20.0,-70.0},{268.0,-70.0},{268.0,-36.0},{318.0,-36.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.food_ratio,Land_Fertility1.food_ratio) annotation(Line(points = {{-78.0,-104.0},{10.0,-104.0},{10.0,-110.0},{98.0,-110.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.land_yield,Arable_Land_Dynamics1.land_yield) annotation(Line(points = {{-78.0,-112.0},{-20.0,-112.0},{-20.0,-60.0},{-192.0,-60.0},{-192.0,-4.0},{-122.0,-4.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.food_pc,Life_Expectancy1.food_pc) annotation(Line(points = {{-96.0,-98.0},{-96.0,-74.0},{256.0,-74.0},{256.0,-28.0},{318.0,-28.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.marg_prod_agr_inp,Arable_Land_Dynamics1.marg_prod_agr_inp) annotation(Line(points = {{-78.0,-120.0},{-24.0,-120.0},{-24.0,-64.0},{-172.0,-64.0},{-172.0,-12.0},{-122.0,-12.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.agr_inp_per_hect,Pollution_Dynamics1.agr_inp_per_hect) annotation(Line(points = {{-78.0,-128.0},{-16.0,-128.0},{-16.0,-56.0},{-196.0,-56.0},{-196.0,72.0},{-122.0,72.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.agr_inp_per_hect,Labor_Utilization1.agr_inp_per_hect) annotation(Line(points = {{-78.0,-128.0},{-16.0,-128.0},{-16.0,-28.0},{98.0,-28.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.s_fioa_agr,Industrial_Investment1.s_fioa_agr) annotation(Line(points = {{-78.0,-136.0},{-4.0,-136.0},{-4.0,64.0},{98.0,64.0}}, color = {0,0,191}, visible = true));
        connect(Service_Sector_Investment1.service_capital,Labor_Utilization1.service_capital) annotation(Line(points = {{-78.0,-220.0},{4.0,-220.0},{4.0,-4.0},{98.0,-4.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.p_fr_inp_for_land_maint,Land_Fertility1.p_fr_inp_for_land_maint) annotation(Line(points = {{-88.0,-142.0},{-88.0,-160.0},{44.0,-160.0},{44.0,-120.0},{98.0,-120.0}}, color = {0,0,191}, visible = true));
        connect(Service_Sector_Investment1.s_fioa_serv,Industrial_Investment1.s_fioa_serv) annotation(Line(points = {{-78.0,-230.0},{16.0,-230.0},{16.0,38.0},{108.0,38.0},{108.0,58.0}}, color = {0,0,191}, visible = true));
        connect(Population_Dynamics1.labor_force,Labor_Utilization1.labor_force) annotation(Line(points = {{-78.0,170.0},{12.0,170.0},{12.0,-60.0},{108.0,-60.0},{108.0,-42.0}}, color = {0,0,191}, visible = true));
        connect(Human_Fertility1.total_fertility,Population_Dynamics1.total_fertility) annotation(Line(points = {{142.0,180.0},{196.0,180.0},{196.0,144.0},{-200.0,144.0},{-200.0,190.0},{-122.0,190.0}}, color = {0,0,191}, visible = true));
        connect(Industrial_Investment1.industrial_capital,Labor_Utilization1.industrial_capital) annotation(Line(points = {{142.0,76.0},{196.0,76.0},{196.0,-78.0},{44.0,-78.0},{44.0,-20.0},{98.0,-20.0}}, color = {0,0,191}, visible = true));
        connect(Land_Fertility1.land_fertility,Food_Production1.land_fertility) annotation(Line(points = {{142.0,-120.0},{192.0,-120.0},{192.0,-164.0},{-112.0,-164.0},{-112.0,-142.0}}, color = {0,0,191}, visible = true));
        connect(Life_Expectancy1.life_expectancy,Human_Welfare_Index1.life_expectancy) annotation(Line(points = {{362.0,-20.0},{420.0,-20.0},{420.0,40.0},{250.0,40.0},{250.0,90.0},{318.0,90.0}}, color = {0,0,191}, visible = true));
        connect(Human_Welfare_Index1.life_expectancy,Human_Fertility1.life_expectancy) annotation(Line(points = {{318.0,90.0},{250.0,90.0},{250.0,128.0},{32.0,128.0},{32.0,180.0},{98.0,180.0}}, color = {0,0,191}, visible = true));
        connect(Human_Fertility1.life_expectancy,Population_Dynamics1.life_expectancy) annotation(Line(points = {{98.0,180.0},{32.0,180.0},{32.0,128.0},{-190.0,128.0},{-190.0,170.0},{-122.0,170.0}}, color = {0,0,191}, visible = true));
        connect(NR_Resource_Utilization1.pc_res_use_mlt,Pollution_Dynamics1.pc_res_use_mlt) annotation(Line(points = {{362.0,-108.0},{424.0,-108.0},{424.0,124.0},{-176.0,124.0},{-176.0,96.0},{-122.0,96.0}}, color = {0,0,191}, visible = true));
        connect(Land_Fertility1.yield_tech_LYTD,Food_Production1.yield_tech_LYTD) annotation(Line(points = {{142.0,-110.0},{192.0,-110.0},{192.0,-82.0},{-112.0,-82.0},{-112.0,-87.9893},{-112.0,-98.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.fr_inp_al_land_dev,Food_Production1.fr_inp_al_land_dev) annotation(Line(points = {{-78.0,-10.0},{-28.0,-10.0},{-28.0,-68.0},{-172.0,-68.0},{-172.0,-120.0},{-122.0,-120.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.s_yield_mlt_air_poll,Food_Production1.s_yield_mlt_air_poll) annotation(Line(points = {{-78.0,72.0},{32.0,72.0},{32.0,-168.0},{-104.0,-168.0},{-104.0,-142.0}}, color = {0,0,191}, visible = true));
        connect(Labor_Utilization1.capital_util_fr,Service_Sector_Investment1.capital_util_fr) annotation(Line(points = {{142.0,-28.0},{224.0,-28.0},{224.0,-180.0},{-172.0,-180.0},{-172.0,-208.0},{-122.0,-208.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.ppoll_tech_mult_icor_COPM,Industrial_Investment1.ppoll_tech_mult_icor_COPM) annotation(Line(points = {{-88.0,0.0},{88.0,0.0}}, color = {0,0,191}, visible = true, origin = {10.0,88.0}));
        connect(NR_Resource_Utilization1.ind_cap_out_ratio_2_ICOR2T,Industrial_Investment1.ind_cap_out_ratio_2_ICOR2T) annotation(Line(points = {{362.0,-124.0},{428.0,-124.0},{428.0,120.0},{108.0,120.0},{108.0,102.0}}, color = {0,0,191}, visible = true));
        connect(Labor_Utilization1.capital_util_fr,Industrial_Investment1.capital_util_fr) annotation(Line(points = {{142.0,-28.0},{224.0,-28.0},{224.0,-180.0},{24.0,-180.0},{24.0,80.0},{98.0,80.0}}, color = {0,0,191}, visible = true));
        connect(NR_Resource_Utilization1.s_fr_cap_al_obt_res,Industrial_Investment1.s_fr_cap_al_obt_res) annotation(Line(points = {{362.0,-132.0},{432.0,-132.0},{432.0,132.0},{44.0,132.0},{44.0,72.0},{98.0,72.0}}, color = {0,0,191}, visible = true));
        connect(Pollution_Dynamics1.ppoll_gen_rt,Human_Ecological_Footprint1.ppoll_gen_rt) annotation(Line(points = {{-78.0,64.0},{-20.0,64.0},{-20.0,148.0},{256.0,148.0},{256.0,170.0},{318.0,170.0}}, color = {0,0,191}, visible = true));
        connect(Arable_Land_Dynamics1.urban_ind_land,Human_Ecological_Footprint1.urban_ind_land) annotation(Line(points = {{-78.0,-30.0},{-24.0,-30.0},{-24.0,136.0},{252.0,136.0},{252.0,180.0},{318.0,180.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.yield_tech_mult_icor_COYM,Industrial_Investment1.yield_tech_mult_icor_COYM) annotation(Line(points = {{-96.0,-142.0},{-96.0,-174.0},{28.0,-174.0},{28.0,96.0},{98.0,96.0}}, color = {0,0,191}, visible = true));
        connect(Food_Production1.tot_agric_invest,Arable_Land_Dynamics1.tot_agric_invest) annotation(Line(points = {{-88.0,-98.0},{-88.0,-52.0},{-200.0,-52.0},{-200.0,-20.0},{-122.0,-20.0}}, color = {0,0,191}, visible = true));
        annotation(Icon(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-220.0,-260.0},{442.0,222.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,98.0},{300.0,170.0}}, textString = "More Accessible", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-82.0,24.0},{304.0,90.0}}, textString = "Non-recoverable", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-112.0,-68.0},{326.0,36.0}}, textString = "Natural Resources,", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-138.0,-146.0},{360.0,-42.0}}, textString = "Pollution Control, and", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-200.0,-324.0},{418.0,-16.0}}, textString = "Land Yield Enhancement", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10})), experiment(StartTime = 1900, StopTime = 2100), experimentSetupOutput, Documentation(info = "<html>
This is <font color=red><b>Scenario #4</b></font> of the <font color=red><b>WORLD3</b></font> model. This scenario starts out with the same assumptions as <font color=red><b>Scenario #3</b></font>.  Improvements in pollution control technology have led to an increase in the population. There are additional mouths to be fed, but unfortunately, not enough food is being grown to feed the population in the long run.  The population stagnates, and there is massive hunger in the world. <p>
 
In this new scenario, it is thus postulated that additional efforts are being undertaken to increase the yield of food per hectare of arable land. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Meadows, D.H., D.L. Meadows, J. Randers, and W.W. Behrens III (1972), <i>Limits to Growth: A Report for the Club of Rome's Project on the Predicament of Mankind</i>, Universe Books, New York, 205p.
<li> Meadows, D.L., W.W. Behrens III, D.M., Meadows, R.F. Naill, J. Randers, and E.K.O. Zahn (1974), <a href=\"http://www.pegasuscom.com/BookDetail.asp?BookQuery_Action=Find('ISBN','XDYGO')\">Dynamics of Growth in a Finite World</a>, Wright-Allen Press, 637p.
<li> Meadows, D.H., D.L. Meadows, and J. Randers (1992), <i>Beyond the Limits</i>, Chelsea Green, 300p.
<li> Meadows, D.H., J. Randers, and D.L. Meadows (2004), <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a>, Chelsea Green, 368p.
</ol> <p>
 
<hr> <p>
 
In order to accomplish this change, you need to modify the table P_Yield_Tech_Chg_Mlt_LYCM  that models a multiplier to the investment in technology to improve land yield: <p>
<font color=red>parameter Real p_yield_tech_chg_mlt[:] = {0,0,<b>0.04</b>,<b>0.04</b>} \"Yield technology change multiplier\";</font>. <p>
 
<hr> <p>
 
Simulate the model from 1900 until 2100, and display the same variables as in the book <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a> at page 215: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_10.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_11.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_12.png\"> <p>
 
<hr> <p>
 
The results obtained are visibly different from those shown in the book <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a> primarily due to the  quantization error. <p>
 
In this model, some of the gains are huge.  For example, the variable <b>agr_inp_per_hect</b>, i.e., the investment in agricultural production per hectare of arable land, that in earlier scenarios assumed values in the range <b>[2,30] dollars per hectare and year</b> suddenly assumes values of several <b>thousands of dollars per hectare and year</b>.  We observe hyperinflation at work. <p>
 
Due to the amplified quantization errors, the collapse happens more than 20 years earlier than shown in the book.  If we were to simulate beyond the year 2100, the results obtained using the <font color=red><b>Modelica</b></font> implementation could be considered similar to those obtained using <font color=red><b>STELLA</b></font>.  The trajectories simply get shifted in time.  However, since we stop the simulation in 2100, our results look qualitatively different. <p>
 
<hr> <p>
</html>", revisions = ""));
      end Scenario_4;
      model Scenario_5 "More accessible non-recoverable natural resources, pollution control, land yield enhancement, and erosion protection"
        parameter Real agr_mtl_toxic_index(unit = "1/dollar") = 1 "Agricultural materials toxicity index";
        parameter Real assim_half_life_1970(unit = "yr") = 1.5 "Pollution assimilation half life in 1970";
        parameter Real avg_life_land_norm(unit = "yr") = 1000 "Normal life span of land";
        parameter Real des_compl_fam_size_norm = 3.8 "Desired normal complete family size";
        parameter Real des_food_ratio_dfr = 2 "Desired food ratio";
        parameter Real des_ppoll_index_DPOLX = 1.2 "Desired persistent pollution index";
        parameter Real des_res_use_rt_DNRUR(unit = "ton/yr") = 4800000000.0 "Desired resource utilization rate";
        parameter Real food_short_perc_del(unit = "yr") = 2 "Food shortage perception delay";
        parameter Real fr_agr_inp_pers_mtl = 0.001 "Effective fraction of agricultural pollution input";
        parameter Real frac_res_pers_mtl = 0.02 "Effective fraction of resource utilization on pollution generation";
        parameter Real hlth_serv_impact_del(unit = "yr") = 20 "Health service impact delay";
        parameter Real income_expect_avg_time(unit = "yr") = 3 "Income expected average time";
        parameter Real ind_mtl_emiss_fact(unit = "1/ton") = 0.1 "Industrial materials emission factor";
        parameter Real ind_mtl_toxic_index = 10.0 "Industrial materials toxicity index";
        parameter Real ind_out_pc_des(unit = "dollar/yr") = 400 "Desired annual industrial per capita output";
        parameter Real ind_out_in_1970(unit = "dollar/yr") = 790000000000.0 "Industrial output in 1970";
        parameter Real inherent_land_fert(unit = "kg/(hectare.yr)") = 600 "Inherent land fertility";
        parameter Real labor_force_partic = 0.75 "Percentage of participating labor force";
        parameter Real labor_util_fr_del_time(unit = "yr") = 2 "Labor utilization fraction delay time";
        parameter Real land_fr_harvested = 0.7 "Land fraction harvested";
        parameter Real life_expect_norm(unit = "yr") = 28 "Normal life expectancy";
        parameter Real lifet_perc_del(unit = "yr") = 20 "Perceived life-time delay";
        parameter Real max_tot_fert_norm = 12 "Normal maximal total fertility";
        parameter Real p_avg_life_agr_inp_1(unit = "yr") = 2 "Default average life of agricultural input";
        parameter Real p_avg_life_agr_inp_2(unit = "yr") = 2 "Controlled average life of agricultural input";
        parameter Real p_avg_life_ind_cap_1(unit = "yr") = 14 "Default average life of industrial capital";
        parameter Real p_avg_life_ind_cap_2(unit = "yr") = 14 "Controlled average life of industrial capital";
        parameter Real p_avg_life_serv_cap_1(unit = "yr") = 20 "Default average life of service sector capital";
        parameter Real p_avg_life_serv_cap_2(unit = "yr") = 20 "Controlled average life of service sector capital";
        parameter Real p_fioa_cons_const_1 = 0.43 "Default fraction of industrial output allocated to consumption";
        parameter Real p_fioa_cons_const_2 = 0.43 "Controlled fraction of industrial output allocated to consumption";
        parameter Real p_ind_cap_out_ratio_1(unit = "yr") = 3 "Default industrial capital output ratio";
        parameter Real p_land_yield_fact_1 = 1 "Default land yield factor";
        parameter Real p_nr_res_use_fact_1 = 1 "Default non-recoverable resource utilization factor";
        parameter Real p_ppoll_gen_fact_1 = 1 "Default persistent pollution generation factor";
        parameter Real p_serv_cap_out_ratio_1(unit = "yr") = 1.0 "Default fraction of service sector output ratio";
        parameter Real p_serv_cap_out_ratio_2(unit = "yr") = 1.0 "Controlled fraction of service sector output ratio";
        parameter Real pot_arable_land_tot(unit = "hectare") = 3200000000.0 "Total potential arable land";
        parameter Real ppoll_in_1970 = 136000000.0 "Persistent pollution in 1970";
        parameter Real ppoll_trans_del(unit = "yr") = 20 "Persistent pollution transmission delay";
        parameter Real processing_loss = 0.1 "Processing loss";
        parameter Real reproductive_lifetime(unit = "yr") = 30.0 "Reproductive life time";
        parameter Real social_adj_del(unit = "yr") = 20 "Social adjustment delay";
        parameter Real social_discount(unit = "1/yr") = 0.07 "Social discount";
        parameter Real subsist_food_pc(unit = "kg/yr") = 230 "Available per capita food";
        parameter Real tech_dev_del_TDD(unit = "yr") = 20 "Technology development time";
        parameter Real urb_ind_land_dev_time(unit = "yr") = 10 "Urban and industrial land development time";
        parameter Real t_air_poll_time(unit = "yr") = 4000 "Air pollution change time";
        parameter Real t_fcaor_time(unit = "yr") = 2002 "Year of capital allocation to resource use efficiency";
        parameter Real t_fert_cont_eff_time(unit = "yr") = 4000 "Year of continued fertility change";
        parameter Real t_ind_equil_time(unit = "yr") = 4000 "Year of industrial equilibrium";
        parameter Real t_land_life_time(unit = "yr") = 2002 "Land life time";
        parameter Real t_policy_year(unit = "yr") = 2002 "Year of policy change";
        parameter Real t_pop_equil_time(unit = "yr") = 4000 "Population equilibrium time";
        parameter Real t_zero_pop_grow_time(unit = "yr") = 4000 "Time to zero population growth";
        parameter Real p_fr_cap_al_obt_res_2[:] = {1,0.1,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05} "Non-renewable resource fraction remaining";
        parameter Real p_ppoll_tech_chg_mlt[:] = {-0.04,-0.04,0,0} "Persistent pollution technology change multiplier";
        parameter Real p_res_tech_chg_mlt[:] = {0,0,0,0} "Resource technology change multiplier";
        parameter Real p_yield_tech_chg_mlt[:] = {0,0,0.04,0.04} "Yield technology change multiplier";
        parameter Real agr_inp_init(unit = "dollar/yr") = 5000000000.0 "Initial agricultural input";
        parameter Real arable_land_init(unit = "hectare") = 900000000.0 "Initial arable land";
        parameter Real industrial_capital_init(unit = "dollar") = 210000000000.0 "Initial industrial investment";
        parameter Real labor_util_fr_del_init = 1 "Initial delayed labor utilization fraction";
        parameter Real land_fertility_init(unit = "kg/(hectare.yr)") = 600 "Initial industrial investment";
        parameter Real nr_resources_init(unit = "ton") = 2000000000000.0 "Initial available non-recoverable resources";
        parameter Real perc_food_ratio_init = 1 "Initial perceived food ratio";
        parameter Real pers_pollution_init = 25000000.0 "Initial persistent pollution";
        parameter Real pop1_init = 650000000.0 "Initial population 14 years and younger";
        parameter Real pop2_init = 700000000.0 "Initial population 15 to 44 years old";
        parameter Real pop3_init = 190000000.0 "Initial population 45 to 64 years old";
        parameter Real pop4_init = 60000000.0 "Initial population 65 years and older";
        parameter Real pot_arable_land_init(unit = "hectare") = 2300000000.0 "Initial potential arable land";
        parameter Real ppoll_tech_init = 1 "Initial persistent pollution technology change factor";
        parameter Real res_tech_init = 1 "Initial non-recoverable resource technology factor";
        parameter Real service_capital_init(unit = "dollar") = 144000000000.0 "Initial service sector investment";
        parameter Real urban_ind_land_init(unit = "hectare") = 8200000.0 "Initial urban and industrial land";
        parameter Real yield_tech_init = 1 "Initial yield technology factor";
        output Real population "Total human world population";
        output Real food(unit = "dollar/yr") "Total annually produced food";
        output Real industrial_output(unit = "dollar/yr") "Total annual world industrial output";
        output Real ppoll_index "Persistent pollution index";
        output Real nr_resources "Remaining non-recoverable natural resources";
        output Real fioa_ind "Fraction of industrial output allocated to industrial/military complex";
        output Real s_fioa_agr "Fraction of industrial output allocated to food production";
        output Real s_fioa_cons "Fraction of industrial output allocated to consumption";
        output Real s_fioa_serv "Fraction of industrial output allocated to service sector";
        output Real s_fr_cap_al_obt_res "Fraction of capital allocated to resource use efficiency";
        output Real life_expectancy(unit = "yr") "Life expectancy";
        output Real food_pc(unit = "dollar/yr") "Total annual food per person";
        output Real serv_out_pc(unit = "dollar/yr") "Total annual services per person";
        output Real ind_out_pc(unit = "dollar/yr") "Total annual consumer goods per person";
        output Real human_ecological_footprint(unit = "Gha") "Human ecological footprint";
        output Real human_welfare_index "Human welfare index";
        SystemDynamics.WorldDynamics.World3.Population_Dynamics Population_Dynamics1(pop1_init = pop1_init, pop2_init = pop2_init, pop3_init = pop3_init, pop4_init = pop4_init, labor_force_partic = labor_force_partic, reproductive_lifetime = reproductive_lifetime, t_pop_equil_time = t_pop_equil_time) "Population dynamics" annotation(Placement(visible = true, transformation(origin = {-100.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Pollution_Dynamics Pollution_Dynamics1(agr_mtl_toxic_index = agr_mtl_toxic_index, assim_half_life_1970 = assim_half_life_1970, des_ppoll_index_DPOLX = des_ppoll_index_DPOLX, fr_agr_inp_pers_mtl = fr_agr_inp_pers_mtl, frac_res_pers_mtl = frac_res_pers_mtl, ind_mtl_emiss_fact = ind_mtl_emiss_fact, ind_mtl_toxic_index = ind_mtl_toxic_index, ind_out_in_1970 = ind_out_in_1970, p_ppoll_gen_fact_1 = p_ppoll_gen_fact_1, pers_pollution_init = pers_pollution_init, ppoll_in_1970 = ppoll_in_1970, ppoll_tech_init = ppoll_tech_init, ppoll_trans_del = ppoll_trans_del, t_air_poll_time = t_air_poll_time, t_policy_year = t_policy_year, tech_dev_del_TDD = tech_dev_del_TDD, p_ppoll_tech_chg_mlt = p_ppoll_tech_chg_mlt) "Persistent pollution generation" annotation(Placement(visible = true, transformation(origin = {-100.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Arable_Land_Dynamics Arable_Land_Dynamics1(arable_land_init = arable_land_init, avg_life_land_norm = avg_life_land_norm, inherent_land_fert = inherent_land_fert, pot_arable_land_init = pot_arable_land_init, pot_arable_land_tot = pot_arable_land_tot, social_discount = social_discount, t_land_life_time = t_land_life_time, urban_ind_land_init = urban_ind_land_init, urb_ind_land_dev_time = urb_ind_land_dev_time) "Arable land dynamics" annotation(Placement(visible = true, transformation(origin = {-100.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Food_Production Food_Production1(agr_inp_init = agr_inp_init, food_short_perc_del = food_short_perc_del, p_avg_life_agr_inp_1 = p_avg_life_agr_inp_1, p_avg_life_agr_inp_2 = p_avg_life_agr_inp_2, p_land_yield_fact_1 = p_land_yield_fact_1, perc_food_ratio_init = perc_food_ratio_init, processing_loss = processing_loss, subsist_food_pc = subsist_food_pc, t_policy_year = t_policy_year, tech_dev_del_TDD = tech_dev_del_TDD, land_fr_harvested = land_fr_harvested) "Food production" annotation(Placement(visible = true, transformation(origin = {-100.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Human_Ecological_Footprint Human_Ecological_Footprint1 "Human ecological footprint" annotation(Placement(visible = true, transformation(origin = {340.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Human_Fertility Human_Fertility1(des_compl_fam_size_norm = des_compl_fam_size_norm, hlth_serv_impact_del = hlth_serv_impact_del, income_expect_avg_time = income_expect_avg_time, lifet_perc_del = lifet_perc_del, max_tot_fert_norm = max_tot_fert_norm, social_adj_del = social_adj_del, t_fert_cont_eff_time = t_fert_cont_eff_time, t_zero_pop_grow_time = t_zero_pop_grow_time) "Human fertility" annotation(Placement(visible = true, transformation(origin = {120.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Human_Welfare_Index Human_Welfare_Index1 "Human welfare index" annotation(Placement(visible = true, transformation(origin = {340.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Industrial_Investment Industrial_Investment1(industrial_capital_init = industrial_capital_init, ind_out_pc_des = ind_out_pc_des, p_avg_life_ind_cap_1 = p_avg_life_ind_cap_1, p_avg_life_ind_cap_2 = p_avg_life_ind_cap_2, p_fioa_cons_const_1 = p_fioa_cons_const_1, p_fioa_cons_const_2 = p_fioa_cons_const_2, p_ind_cap_out_ratio_1 = p_ind_cap_out_ratio_1, t_ind_equil_time = t_ind_equil_time, t_policy_year = t_policy_year) "Industrial investment" annotation(Placement(visible = true, transformation(origin = {120.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Labor_Utilization Labor_Utilization1(labor_util_fr_del_init = labor_util_fr_del_init, labor_util_fr_del_time = labor_util_fr_del_time) "Labor utilization" annotation(Placement(visible = true, transformation(origin = {120.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Land_Fertility Land_Fertility1(des_food_ratio_dfr = des_food_ratio_dfr, inherent_land_fert = inherent_land_fert, land_fertility_init = land_fertility_init, t_policy_year = t_policy_year, yield_tech_init = yield_tech_init, p_yield_tech_chg_mlt = p_yield_tech_chg_mlt) "Land fertility" annotation(Placement(visible = true, transformation(origin = {120.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Life_Expectancy Life_Expectancy1(hlth_serv_impact_del = hlth_serv_impact_del, life_expect_norm = life_expect_norm, subsist_food_pc = subsist_food_pc) "Life expectancy" annotation(Placement(visible = true, transformation(origin = {340.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.NR_Resource_Utilization NR_Resource_Utilization1(des_res_use_rt_DNRUR = des_res_use_rt_DNRUR, nr_resources_init = nr_resources_init, p_nr_res_use_fact_1 = p_nr_res_use_fact_1, res_tech_init = res_tech_init, t_policy_year = t_policy_year, t_fcaor_time = t_fcaor_time, tech_dev_del_TDD = tech_dev_del_TDD, p_fr_cap_al_obt_res_2 = p_fr_cap_al_obt_res_2, p_res_tech_chg_mlt = p_res_tech_chg_mlt) "Non-recoverable natural resource utilization" annotation(Placement(visible = true, transformation(origin = {340.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Service_Sector_Investment Service_Sector_Investment1(t_policy_year = t_policy_year, p_avg_life_serv_cap_1 = p_avg_life_serv_cap_1, p_avg_life_serv_cap_2 = p_avg_life_serv_cap_2, p_serv_cap_out_ratio_1 = p_serv_cap_out_ratio_1, p_serv_cap_out_ratio_2 = p_serv_cap_out_ratio_2, service_capital_init = service_capital_init) "Service sector investment" annotation(Placement(visible = true, transformation(origin = {-100.0,-220.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
      equation
        connect(Service_Sector_Investment1.serv_out_pc,Human_Fertility1.serv_out_pc) annotation(Line(visible = true, points = {{-78.0,-210.0},{0.0,-210.0},{0.0,190.0},{98.0,190.0}}, color = {0,0,191}));
        connect(Food_Production1.ind_out_pc,Service_Sector_Investment1.ind_out_pc) annotation(Line(visible = true, points = {{-122.0,-128.0},{-188.0,-128.0},{-188.0,-232.0},{-122.0,-232.0}}, color = {0,0,191}));
        connect(Food_Production1.industrial_output,Service_Sector_Investment1.industrial_output) annotation(Line(visible = true, points = {{-122.0,-136.0},{-184.0,-136.0},{-184.0,-224.0},{-122.0,-224.0}}, color = {0,0,191}));
        connect(Food_Production1.population,Service_Sector_Investment1.population) annotation(Line(visible = true, points = {{-122.0,-104.0},{-180.0,-104.0},{-180.0,-216.0},{-122.0,-216.0}}, color = {0,0,191}));
        connect(Labor_Utilization1.capital_util_fr,Service_Sector_Investment1.capital_util_fr) annotation(Line(visible = true, points = {{142.0,-28.0},{224.0,-28.0},{224.0,-180.0},{-172.0,-180.0},{-172.0,-208.0},{-122.0,-208.0}}, color = {0,0,191}));
        connect(Service_Sector_Investment1.serv_out_pc,Labor_Utilization1.serv_out_pc) annotation(Line(visible = true, points = {{-78.0,-210.0},{0.0,-210.0},{0.0,18.0},{70.0,18.0},{108.0,18.0},{108.0,2.0}}, color = {0,0,191}));
        connect(Service_Sector_Investment1.service_capital,Labor_Utilization1.service_capital) annotation(Line(visible = true, points = {{-78.0,-220.0},{4.0,-220.0},{4.0,-4.0},{98.0,-4.0}}, color = {0,0,191}));
        connect(Service_Sector_Investment1.s_fioa_serv,Industrial_Investment1.s_fioa_serv) annotation(Line(visible = true, points = {{-78.0,-230.0},{16.0,-230.0},{16.0,38.0},{108.0,38.0},{108.0,58.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.ppoll_tech_mult_icor_COPM,Industrial_Investment1.ppoll_tech_mult_icor_COPM) annotation(Line(visible = true, origin = {10.0,88.0}, points = {{-88.0,0.0},{88.0,0.0}}, color = {0,0,191}));
        connect(Life_Expectancy1.population,NR_Resource_Utilization1.population) annotation(Line(visible = true, points = {{318.0,-20.0},{260.0,-20.0},{260.0,-120.0},{318.0,-120.0}}, color = {0,0,191}));
        connect(Human_Fertility1.life_expectancy,Population_Dynamics1.life_expectancy) annotation(Line(visible = true, points = {{98.0,180.0},{32.0,180.0},{32.0,128.0},{-190.0,128.0},{-190.0,170.0},{-122.0,170.0}}, color = {0,0,191}));
        connect(Human_Fertility1.total_fertility,Population_Dynamics1.total_fertility) annotation(Line(visible = true, points = {{142.0,180.0},{196.0,180.0},{196.0,144.0},{-200.0,144.0},{-200.0,190.0},{-122.0,190.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Human_Fertility1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,30.0},{36.0,30.0},{36.0,170.0},{98.0,170.0}}, color = {0,0,191}));
        connect(Population_Dynamics1.population,Pollution_Dynamics1.population) annotation(Line(visible = true, points = {{-78.0,190.0},{-10.0,190.0},{-10.0,140.0},{-180.0,140.0},{-180.0,88.0},{-122.0,88.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.arable_land,Pollution_Dynamics1.arable_land) annotation(Line(visible = true, points = {{-78.0,-20.0},{-20.0,-20.0},{-20.0,26.0},{-176.0,26.0},{-176.0,64.0},{-122.0,64.0}}, color = {0,0,191}));
        connect(Food_Production1.marg_prod_agr_inp,Arable_Land_Dynamics1.marg_prod_agr_inp) annotation(Line(visible = true, points = {{-78.0,-120.0},{-24.0,-120.0},{-24.0,-64.0},{-172.0,-64.0},{-172.0,-12.0},{-122.0,-12.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Arable_Land_Dynamics1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,30.0},{-188.0,30.0},{-188.0,-28.0},{-122.0,-28.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.population,Arable_Land_Dynamics1.population) annotation(Line(visible = true, points = {{-122.0,88.0},{-180.0,88.0},{-180.0,-36.0},{-122.0,-36.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.urban_ind_land,Human_Ecological_Footprint1.urban_ind_land) annotation(Line(visible = true, points = {{-78.0,-30.0},{-24.0,-30.0},{-24.0,136.0},{252.0,136.0},{252.0,180.0},{318.0,180.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.fr_inp_al_land_dev,Food_Production1.fr_inp_al_land_dev) annotation(Line(visible = true, points = {{-78.0,-10.0},{-28.0,-10.0},{-28.0,-68.0},{-172.0,-68.0},{-172.0,-120.0},{-122.0,-120.0}}, color = {0,0,191}));
        connect(Food_Production1.tot_agric_invest,Arable_Land_Dynamics1.tot_agric_invest) annotation(Line(visible = true, points = {{-88.0,-98.0},{-88.0,-52.0},{-200.0,-52.0},{-200.0,-20.0},{-122.0,-20.0}}, color = {0,0,191}));
        connect(Food_Production1.food_ratio,Land_Fertility1.food_ratio) annotation(Line(visible = true, points = {{-78.0,-104.0},{10.0,-104.0},{10.0,-110.0},{98.0,-110.0}}, color = {0,0,191}));
        connect(Food_Production1.land_yield,Arable_Land_Dynamics1.land_yield) annotation(Line(visible = true, points = {{-78.0,-112.0},{-20.0,-112.0},{-20.0,-60.0},{-192.0,-60.0},{-192.0,-4.0},{-122.0,-4.0}}, color = {0,0,191}));
        connect(Food_Production1.agr_inp_per_hect,Pollution_Dynamics1.agr_inp_per_hect) annotation(Line(visible = true, points = {{-78.0,-128.0},{-16.0,-128.0},{-16.0,-56.0},{-196.0,-56.0},{-196.0,72.0},{-122.0,72.0}}, color = {0,0,191}));
        connect(Food_Production1.p_fr_inp_for_land_maint,Land_Fertility1.p_fr_inp_for_land_maint) annotation(Line(visible = true, points = {{-88.0,-142.0},{-88.0,-160.0},{44.0,-160.0},{44.0,-120.0},{98.0,-120.0}}, color = {0,0,191}));
        connect(Land_Fertility1.land_fertility,Food_Production1.land_fertility) annotation(Line(visible = true, points = {{142.0,-120.0},{192.0,-120.0},{192.0,-164.0},{-112.0,-164.0},{-112.0,-142.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.ind_out_pc,Food_Production1.ind_out_pc) annotation(Line(visible = true, points = {{-122.0,-28.0},{-188.0,-28.0},{-188.0,-128.0},{-122.0,-128.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.industrial_output,Food_Production1.industrial_output) annotation(Line(visible = true, points = {{-122.0,80.0},{-184.0,80.0},{-184.0,-136.0},{-122.0,-136.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.arable_land,Food_Production1.arable_land) annotation(Line(visible = true, points = {{-122.0,64.0},{-176.0,64.0},{-176.0,-112.0},{-122.0,-112.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.population,Food_Production1.population) annotation(Line(visible = true, points = {{-122.0,-36.0},{-180.0,-36.0},{-180.0,-104.0},{-122.0,-104.0}}, color = {0,0,191}));
        connect(Land_Fertility1.yield_tech_LYTD,Food_Production1.yield_tech_LYTD) annotation(Line(visible = true, points = {{142.0,-110.0},{192.0,-110.0},{192.0,-82.0},{-112.0,-82.0},{-112.0,-86.7587},{-112.0,-98.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.ppoll_index,Land_Fertility1.ppoll_index) annotation(Line(visible = true, points = {{-78.0,80.0},{20.0,80.0},{20.0,-130.0},{98.0,-130.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,NR_Resource_Utilization1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,-110.0},{318.0,-110.0}}, color = {0,0,191}));
        connect(NR_Resource_Utilization1.ind_cap_out_ratio_2_ICOR2T,Industrial_Investment1.ind_cap_out_ratio_2_ICOR2T) annotation(Line(visible = true, points = {{362.0,-124.0},{428.0,-124.0},{428.0,120.0},{108.0,120.0},{108.0,102.0}}, color = {0,0,191}));
        connect(NR_Resource_Utilization1.pc_res_use_mlt,Pollution_Dynamics1.pc_res_use_mlt) annotation(Line(visible = true, points = {{362.0,-108.0},{424.0,-108.0},{424.0,124.0},{-176.0,124.0},{-176.0,96.0},{-122.0,96.0}}, color = {0,0,191}));
        connect(Food_Production1.food_pc,Life_Expectancy1.food_pc) annotation(Line(visible = true, points = {{-96.0,-98.0},{-96.0,-74.0},{256.0,-74.0},{256.0,-28.0},{318.0,-28.0}}, color = {0,0,191}));
        connect(Land_Fertility1.ppoll_index,Life_Expectancy1.ppoll_index) annotation(Line(visible = true, points = {{98.0,-130.0},{20.0,-130.0},{20.0,-70.0},{268.0,-70.0},{268.0,-36.0},{318.0,-36.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Life_Expectancy1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,-12.0},{318.0,-12.0}}, color = {0,0,191}));
        connect(Labor_Utilization1.capital_util_fr,Industrial_Investment1.capital_util_fr) annotation(Line(visible = true, points = {{142.0,-28.0},{224.0,-28.0},{224.0,-180.0},{24.0,-180.0},{24.0,80.0},{98.0,80.0}}, color = {0,0,191}));
        connect(Life_Expectancy1.serv_out_pc,Labor_Utilization1.serv_out_pc) annotation(Line(visible = true, points = {{318.0,-4.0},{218.0,-4.0},{218.0,18.0},{108.0,18.0},{108.0,11.6909},{108.0,2.0}}, color = {0,0,191}));
        connect(Human_Fertility1.ind_out_pc,Labor_Utilization1.ind_out_pc) annotation(Line(visible = true, points = {{98.0,170.0},{36.0,170.0},{36.0,-12.0},{98.0,-12.0}}, color = {0,0,191}));
        connect(Food_Production1.agr_inp_per_hect,Labor_Utilization1.agr_inp_per_hect) annotation(Line(visible = true, points = {{-78.0,-128.0},{-16.0,-128.0},{-16.0,-28.0},{98.0,-28.0}}, color = {0,0,191}));
        connect(Population_Dynamics1.labor_force,Labor_Utilization1.labor_force) annotation(Line(visible = true, points = {{-78.0,170.0},{12.0,170.0},{12.0,-60.0},{108.0,-60.0},{108.0,-42.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.arable_land,Labor_Utilization1.arable_land) annotation(Line(visible = true, points = {{-78.0,-20.0},{40.0,-20.0},{40.0,-36.0},{98.0,-36.0}}, color = {0,0,191}));
        connect(Labor_Utilization1.arable_land,Human_Ecological_Footprint1.arable_land) annotation(Line(visible = true, points = {{98.0,-36.0},{40.0,-36.0},{40.0,-64.0},{264.0,-64.0},{264.0,190.0},{318.0,190.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.ppoll_gen_rt,Human_Ecological_Footprint1.ppoll_gen_rt) annotation(Line(visible = true, points = {{-78.0,64.0},{-20.0,64.0},{-20.0,148.0},{256.0,148.0},{256.0,170.0},{318.0,170.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.s_yield_mlt_air_poll,Food_Production1.s_yield_mlt_air_poll) annotation(Line(visible = true, points = {{-78.0,72.0},{32.0,72.0},{32.0,-168.0},{-104.0,-168.0},{-104.0,-142.0}}, color = {0,0,191}));
        connect(Food_Production1.yield_tech_mult_icor_COYM,Industrial_Investment1.yield_tech_mult_icor_COYM) annotation(Line(visible = true, points = {{-96.0,-142.0},{-96.0,-174.0},{28.0,-174.0},{28.0,96.0},{98.0,96.0}}, color = {0,0,191}));
        connect(NR_Resource_Utilization1.s_fr_cap_al_obt_res,Industrial_Investment1.s_fr_cap_al_obt_res) annotation(Line(visible = true, points = {{362.0,-132.0},{432.0,-132.0},{432.0,132.0},{44.0,132.0},{44.0,72.0},{98.0,72.0}}, color = {0,0,191}));
        connect(Food_Production1.s_fioa_agr,Industrial_Investment1.s_fioa_agr) annotation(Line(visible = true, points = {{-78.0,-136.0},{-4.0,-136.0},{-4.0,64.0},{98.0,64.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.population,Life_Expectancy1.population) annotation(Line(visible = true, points = {{116.0,58.0},{116.0,34.0},{260.0,34.0},{260.0,-20.0},{318.0,-20.0}}, color = {0,0,191}));
        connect(Population_Dynamics1.population,Industrial_Investment1.population) annotation(Line(visible = true, points = {{-78.0,190.0},{-10.0,190.0},{-10.0,34.0},{116.0,34.0},{116.0,58.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.industrial_capital,Labor_Utilization1.industrial_capital) annotation(Line(visible = true, points = {{142.0,76.0},{196.0,76.0},{196.0,-78.0},{44.0,-78.0},{44.0,-20.0},{98.0,-20.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.industrial_output,NR_Resource_Utilization1.industrial_output) annotation(Line(visible = true, points = {{142.0,84.3414},{210.0,84.3414},{210.0,-130.0},{318.0,-130.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.industrial_output,Pollution_Dynamics1.industrial_output) annotation(Line(visible = true, points = {{142.0,84.3414},{210.0,84.3414},{210.0,22.0},{-184.0,22.0},{-184.0,80.0},{-122.0,80.0}}, color = {0,0,191}));
        connect(Human_Welfare_Index1.life_expectancy,Human_Fertility1.life_expectancy) annotation(Line(visible = true, points = {{318.0,90.0},{250.0,90.0},{250.0,128.0},{32.0,128.0},{32.0,180.0},{98.0,180.0}}, color = {0,0,191}));
        connect(Life_Expectancy1.life_expectancy,Human_Welfare_Index1.life_expectancy) annotation(Line(visible = true, points = {{362.0,-20.0},{420.0,-20.0},{420.0,40.0},{250.0,40.0},{250.0,90.0},{318.0,90.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Human_Welfare_Index1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,70.0},{318.0,70.0}}, color = {0,0,191}));
        population = Population_Dynamics1.Population.y;
        food = Food_Production1.Food.food;
        industrial_output = Industrial_Investment1.Industrial_Output.industrial_output;
        ppoll_index = Pollution_Dynamics1.PPoll_Index.y;
        nr_resources = NR_Resource_Utilization1.NR_Resources.y;
        fioa_ind = Industrial_Investment1.FIOA_Ind.fioa_ind;
        s_fioa_agr = Food_Production1.S_FIOA_Agr.s_fioa_agr;
        s_fioa_cons = Industrial_Investment1.S_FIOA_Cons.s_fioa_cons;
        s_fioa_serv = Service_Sector_Investment1.S_FIOA_Serv.s_fioa_serv;
        s_fr_cap_al_obt_res = NR_Resource_Utilization1.S_Fr_Cap_Al_Obt_Res.s_fr_cap_al_obt_res;
        life_expectancy = Life_Expectancy1.Life_Expectancy.y;
        food_pc = Food_Production1.Food_PC.y;
        serv_out_pc = Service_Sector_Investment1.Serv_Out_PC.y;
        ind_out_pc = Industrial_Investment1.Ind_Out_PC.y;
        human_ecological_footprint = Human_Ecological_Footprint1.HEF_Human_Ecological_Footprint.hef_human_ecological_footprint;
        human_welfare_index = Human_Welfare_Index1.HWI_Human_Welfare_Index.hwi_human_welfare_index;
        annotation(Icon(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-220.0,-260.0},{442.0,222.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,118.0},{300.0,190.0}}, textString = "More Accessible", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-82.0,64.0},{304.0,130.0}}, textString = "Non-recoverable", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-112.0,-14.0},{332.0,90.0}}, textString = "Natural Resources,", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-108.0,-72.0},{330.0,14.0}}, textString = "Pollution Control,", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-197.0,-248.0},{421.0,60.0}}, textString = "Land Yield Enhancement,", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-170.0,-276.0},{377.0,-34.0}}, textString = "and Erosion Protection", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), experiment(StartTime = 1900, StopTime = 2100), experimentSetupOutput, Documentation(info = "<html>
This is <font color=red><b>Scenario #5</b></font> of the <font color=red><b>WORLD3</b></font> model. This scenario starts out with the same assumptions as <font color=red><b>Scenario #4</b></font>.  Improvements in land yield enhancement have allowed more people to be fed a little while longer.  However, the effort is not sustainable.  The agricultural land loses its fertility rather quickly, and in the end, the collapse arrives even earlier than in <font color=red><b>Scenario #4</b></font>.  Humanity spends all of its money and more in trying to grow more food at all cost.  This leads to rampant hyperinflation. <p>
 
In this new scenario, it is thus postulated to spend more money earlier on for the protection of the arable land against erosion. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Meadows, D.H., D.L. Meadows, J. Randers, and W.W. Behrens III (1972), <i>Limits to Growth: A Report for the Club of Rome's Project on the Predicament of Mankind</i>, Universe Books, New York, 205p.
<li> Meadows, D.L., W.W. Behrens III, D.M., Meadows, R.F. Naill, J. Randers, and E.K.O. Zahn (1974), <a href=\"http://www.pegasuscom.com/BookDetail.asp?BookQuery_Action=Find('ISBN','XDYGO')\">Dynamics of Growth in a Finite World</a>, Wright-Allen Press, 637p.
<li> Meadows, D.H., D.L. Meadows, and J. Randers (1992), <i>Beyond the Limits</i>, Chelsea Green, 300p.
<li> Meadows, D.H., J. Randers, and D.L. Meadows (2004), <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a>, Chelsea Green, 368p.
</ol> <p>
 
<hr> <p>
 
In order to accomplish this change, you need to reset another of the switching times in the model: <p>
<font color=red>parameter Real t_land_life_time(unit=\"yr\") = <b>2002</b> \"Land life time\";</font>. <p>
 
<hr> <p>
 
Simulate the model from 1900 until 2100, and display the same variables as in the book <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a> at page 215: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_13.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_14.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_15.png\"> <p>
 
<hr> <p>
</html>", revisions = ""));
      end Scenario_5;
      model Scenario_6 "More accessible non-recoverable natural resources, pollution control, land yield enhancement, erosion protection, and resource efficiency"
        parameter Real agr_mtl_toxic_index(unit = "1/dollar") = 1 "Agricultural materials toxicity index";
        parameter Real assim_half_life_1970(unit = "yr") = 1.5 "Pollution assimilation half life in 1970";
        parameter Real avg_life_land_norm(unit = "yr") = 1000 "Normal life span of land";
        parameter Real des_compl_fam_size_norm = 3.8 "Desired normal complete family size";
        parameter Real des_food_ratio_dfr = 2 "Desired food ratio";
        parameter Real des_ppoll_index_DPOLX = 1.2 "Desired persistent pollution index";
        parameter Real des_res_use_rt_DNRUR(unit = "ton/yr") = 4800000000.0 "Desired resource utilization rate";
        parameter Real food_short_perc_del(unit = "yr") = 2 "Food shortage perception delay";
        parameter Real fr_agr_inp_pers_mtl = 0.001 "Effective fraction of agricultural pollution input";
        parameter Real frac_res_pers_mtl = 0.02 "Effective fraction of resource utilization on pollution generation";
        parameter Real hlth_serv_impact_del(unit = "yr") = 20 "Health service impact delay";
        parameter Real income_expect_avg_time(unit = "yr") = 3 "Income expected average time";
        parameter Real ind_mtl_emiss_fact(unit = "1/ton") = 0.1 "Industrial materials emission factor";
        parameter Real ind_mtl_toxic_index = 10.0 "Industrial materials toxicity index";
        parameter Real ind_out_pc_des(unit = "dollar/yr") = 400 "Desired annual industrial per capita output";
        parameter Real ind_out_in_1970(unit = "dollar/yr") = 790000000000.0 "Industrial output in 1970";
        parameter Real inherent_land_fert(unit = "kg/(hectare.yr)") = 600 "Inherent land fertility";
        parameter Real labor_force_partic = 0.75 "Percentage of participating labor force";
        parameter Real labor_util_fr_del_time(unit = "yr") = 2 "Labor utilization fraction delay time";
        parameter Real land_fr_harvested = 0.7 "Land fraction harvested";
        parameter Real life_expect_norm(unit = "yr") = 28 "Normal life expectancy";
        parameter Real lifet_perc_del(unit = "yr") = 20 "Perceived life-time delay";
        parameter Real max_tot_fert_norm = 12 "Normal maximal total fertility";
        parameter Real p_avg_life_agr_inp_1(unit = "yr") = 2 "Default average life of agricultural input";
        parameter Real p_avg_life_agr_inp_2(unit = "yr") = 2 "Controlled average life of agricultural input";
        parameter Real p_avg_life_ind_cap_1(unit = "yr") = 14 "Default average life of industrial capital";
        parameter Real p_avg_life_ind_cap_2(unit = "yr") = 14 "Controlled average life of industrial capital";
        parameter Real p_avg_life_serv_cap_1(unit = "yr") = 20 "Default average life of service sector capital";
        parameter Real p_avg_life_serv_cap_2(unit = "yr") = 20 "Controlled average life of service sector capital";
        parameter Real p_fioa_cons_const_1 = 0.43 "Default fraction of industrial output allocated to consumption";
        parameter Real p_fioa_cons_const_2 = 0.43 "Controlled fraction of industrial output allocated to consumption";
        parameter Real p_ind_cap_out_ratio_1(unit = "yr") = 3 "Default industrial capital output ratio";
        parameter Real p_land_yield_fact_1 = 1 "Default land yield factor";
        parameter Real p_nr_res_use_fact_1 = 1 "Default non-recoverable resource utilization factor";
        parameter Real p_ppoll_gen_fact_1 = 1 "Default persistent pollution generation factor";
        parameter Real p_serv_cap_out_ratio_1(unit = "yr") = 1.0 "Default fraction of service sector output ratio";
        parameter Real p_serv_cap_out_ratio_2(unit = "yr") = 1.0 "Controlled fraction of service sector output ratio";
        parameter Real pot_arable_land_tot(unit = "hectare") = 3200000000.0 "Total potential arable land";
        parameter Real ppoll_in_1970 = 136000000.0 "Persistent pollution in 1970";
        parameter Real ppoll_trans_del(unit = "yr") = 20 "Persistent pollution transmission delay";
        parameter Real processing_loss = 0.1 "Processing loss";
        parameter Real reproductive_lifetime(unit = "yr") = 30.0 "Reproductive life time";
        parameter Real social_adj_del(unit = "yr") = 20 "Social adjustment delay";
        parameter Real social_discount(unit = "1/yr") = 0.07 "Social discount";
        parameter Real subsist_food_pc(unit = "kg/yr") = 230 "Available per capita food";
        parameter Real tech_dev_del_TDD(unit = "yr") = 20 "Technology development time";
        parameter Real urb_ind_land_dev_time(unit = "yr") = 10 "Urban and industrial land development time";
        parameter Real t_air_poll_time(unit = "yr") = 4000 "Air pollution change time";
        parameter Real t_fcaor_time(unit = "yr") = 2002 "Year of capital allocation to resource use efficiency";
        parameter Real t_fert_cont_eff_time(unit = "yr") = 4000 "Year of continued fertility change";
        parameter Real t_ind_equil_time(unit = "yr") = 4000 "Year of industrial equilibrium";
        parameter Real t_land_life_time(unit = "yr") = 2002 "Land life time";
        parameter Real t_policy_year(unit = "yr") = 2002 "Year of policy change";
        parameter Real t_pop_equil_time(unit = "yr") = 4000 "Population equilibrium time";
        parameter Real t_zero_pop_grow_time(unit = "yr") = 4000 "Time to zero population growth";
        parameter Real p_fr_cap_al_obt_res_2[:] = {1,0.1,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05} "Non-renewable resource fraction remaining";
        parameter Real p_ppoll_tech_chg_mlt[:] = {-0.04,-0.04,0,0} "Persistent pollution technology change multiplier";
        parameter Real p_res_tech_chg_mlt[:] = {-0.04,-0.04,0,0} "Resource technology change multiplier";
        parameter Real p_yield_tech_chg_mlt[:] = {0,0,0.04,0.04} "Yield technology change multiplier";
        parameter Real agr_inp_init(unit = "dollar/yr") = 5000000000.0 "Initial agricultural input";
        parameter Real arable_land_init(unit = "hectare") = 900000000.0 "Initial arable land";
        parameter Real industrial_capital_init(unit = "dollar") = 210000000000.0 "Initial industrial investment";
        parameter Real labor_util_fr_del_init = 1 "Initial delayed labor utilization fraction";
        parameter Real land_fertility_init(unit = "kg/(hectare.yr)") = 600 "Initial industrial investment";
        parameter Real nr_resources_init(unit = "ton") = 2000000000000.0 "Initial available non-recoverable resources";
        parameter Real perc_food_ratio_init = 1 "Initial perceived food ratio";
        parameter Real pers_pollution_init = 25000000.0 "Initial persistent pollution";
        parameter Real pop1_init = 650000000.0 "Initial population 14 years and younger";
        parameter Real pop2_init = 700000000.0 "Initial population 15 to 44 years old";
        parameter Real pop3_init = 190000000.0 "Initial population 45 to 64 years old";
        parameter Real pop4_init = 60000000.0 "Initial population 65 years and older";
        parameter Real pot_arable_land_init(unit = "hectare") = 2300000000.0 "Initial potential arable land";
        parameter Real ppoll_tech_init = 1 "Initial persistent pollution technology change factor";
        parameter Real res_tech_init = 1 "Initial non-recoverable resource technology factor";
        parameter Real service_capital_init(unit = "dollar") = 144000000000.0 "Initial service sector investment";
        parameter Real urban_ind_land_init(unit = "hectare") = 8200000.0 "Initial urban and industrial land";
        parameter Real yield_tech_init = 1 "Initial yield technology factor";
        output Real population "Total human world population";
        output Real food(unit = "dollar/yr") "Total annually produced food";
        output Real industrial_output(unit = "dollar/yr") "Total annual world industrial output";
        output Real ppoll_index "Persistent pollution index";
        output Real nr_resources "Remaining non-recoverable natural resources";
        output Real fioa_ind "Fraction of industrial output allocated to industrial/military complex";
        output Real s_fioa_agr "Fraction of industrial output allocated to food production";
        output Real s_fioa_cons "Fraction of industrial output allocated to consumption";
        output Real s_fioa_serv "Fraction of industrial output allocated to service sector";
        output Real s_fr_cap_al_obt_res "Fraction of capital allocated to resource use efficiency";
        output Real life_expectancy(unit = "yr") "Life expectancy";
        output Real food_pc(unit = "dollar/yr") "Total annual food per person";
        output Real serv_out_pc(unit = "dollar/yr") "Total annual services per person";
        output Real ind_out_pc(unit = "dollar/yr") "Total annual consumer goods per person";
        output Real human_ecological_footprint(unit = "Gha") "Human ecological footprint";
        output Real human_welfare_index "Human welfare index";
        SystemDynamics.WorldDynamics.World3.Population_Dynamics Population_Dynamics1(pop1_init = pop1_init, pop2_init = pop2_init, pop3_init = pop3_init, pop4_init = pop4_init, labor_force_partic = labor_force_partic, reproductive_lifetime = reproductive_lifetime, t_pop_equil_time = t_pop_equil_time) "Population dynamics" annotation(Placement(visible = true, transformation(origin = {-100.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Pollution_Dynamics Pollution_Dynamics1(agr_mtl_toxic_index = agr_mtl_toxic_index, assim_half_life_1970 = assim_half_life_1970, des_ppoll_index_DPOLX = des_ppoll_index_DPOLX, fr_agr_inp_pers_mtl = fr_agr_inp_pers_mtl, frac_res_pers_mtl = frac_res_pers_mtl, ind_mtl_emiss_fact = ind_mtl_emiss_fact, ind_mtl_toxic_index = ind_mtl_toxic_index, ind_out_in_1970 = ind_out_in_1970, p_ppoll_gen_fact_1 = p_ppoll_gen_fact_1, pers_pollution_init = pers_pollution_init, ppoll_in_1970 = ppoll_in_1970, ppoll_tech_init = ppoll_tech_init, ppoll_trans_del = ppoll_trans_del, t_air_poll_time = t_air_poll_time, t_policy_year = t_policy_year, tech_dev_del_TDD = tech_dev_del_TDD, p_ppoll_tech_chg_mlt = p_ppoll_tech_chg_mlt) "Persistent pollution generation" annotation(Placement(visible = true, transformation(origin = {-100.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Arable_Land_Dynamics Arable_Land_Dynamics1(arable_land_init = arable_land_init, avg_life_land_norm = avg_life_land_norm, inherent_land_fert = inherent_land_fert, pot_arable_land_init = pot_arable_land_init, pot_arable_land_tot = pot_arable_land_tot, social_discount = social_discount, t_land_life_time = t_land_life_time, urban_ind_land_init = urban_ind_land_init, urb_ind_land_dev_time = urb_ind_land_dev_time) "Arable land dynamics" annotation(Placement(visible = true, transformation(origin = {-100.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Food_Production Food_Production1(agr_inp_init = agr_inp_init, food_short_perc_del = food_short_perc_del, p_avg_life_agr_inp_1 = p_avg_life_agr_inp_1, p_avg_life_agr_inp_2 = p_avg_life_agr_inp_2, p_land_yield_fact_1 = p_land_yield_fact_1, perc_food_ratio_init = perc_food_ratio_init, processing_loss = processing_loss, subsist_food_pc = subsist_food_pc, t_policy_year = t_policy_year, tech_dev_del_TDD = tech_dev_del_TDD, land_fr_harvested = land_fr_harvested) "Food production" annotation(Placement(visible = true, transformation(origin = {-100.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Human_Ecological_Footprint Human_Ecological_Footprint1 "Human ecological footprint" annotation(Placement(visible = true, transformation(origin = {340.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Human_Fertility Human_Fertility1(des_compl_fam_size_norm = des_compl_fam_size_norm, hlth_serv_impact_del = hlth_serv_impact_del, income_expect_avg_time = income_expect_avg_time, lifet_perc_del = lifet_perc_del, max_tot_fert_norm = max_tot_fert_norm, social_adj_del = social_adj_del, t_fert_cont_eff_time = t_fert_cont_eff_time, t_zero_pop_grow_time = t_zero_pop_grow_time) "Human fertility" annotation(Placement(visible = true, transformation(origin = {120.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Human_Welfare_Index Human_Welfare_Index1 "Human welfare index" annotation(Placement(visible = true, transformation(origin = {340.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Industrial_Investment Industrial_Investment1(industrial_capital_init = industrial_capital_init, ind_out_pc_des = ind_out_pc_des, p_avg_life_ind_cap_1 = p_avg_life_ind_cap_1, p_avg_life_ind_cap_2 = p_avg_life_ind_cap_2, p_fioa_cons_const_1 = p_fioa_cons_const_1, p_fioa_cons_const_2 = p_fioa_cons_const_2, p_ind_cap_out_ratio_1 = p_ind_cap_out_ratio_1, t_ind_equil_time = t_ind_equil_time, t_policy_year = t_policy_year) "Industrial investment" annotation(Placement(visible = true, transformation(origin = {120.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Labor_Utilization Labor_Utilization1(labor_util_fr_del_init = labor_util_fr_del_init, labor_util_fr_del_time = labor_util_fr_del_time) "Labor utilization" annotation(Placement(visible = true, transformation(origin = {120.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Land_Fertility Land_Fertility1(des_food_ratio_dfr = des_food_ratio_dfr, inherent_land_fert = inherent_land_fert, land_fertility_init = land_fertility_init, t_policy_year = t_policy_year, yield_tech_init = yield_tech_init, p_yield_tech_chg_mlt = p_yield_tech_chg_mlt) "Land fertility" annotation(Placement(visible = true, transformation(origin = {120.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Life_Expectancy Life_Expectancy1(hlth_serv_impact_del = hlth_serv_impact_del, life_expect_norm = life_expect_norm, subsist_food_pc = subsist_food_pc) "Life expectancy" annotation(Placement(visible = true, transformation(origin = {340.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.NR_Resource_Utilization NR_Resource_Utilization1(des_res_use_rt_DNRUR = des_res_use_rt_DNRUR, nr_resources_init = nr_resources_init, p_nr_res_use_fact_1 = p_nr_res_use_fact_1, res_tech_init = res_tech_init, t_policy_year = t_policy_year, t_fcaor_time = t_fcaor_time, tech_dev_del_TDD = tech_dev_del_TDD, p_fr_cap_al_obt_res_2 = p_fr_cap_al_obt_res_2, p_res_tech_chg_mlt = p_res_tech_chg_mlt) "Non-recoverable natural resource utilization" annotation(Placement(visible = true, transformation(origin = {340.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Service_Sector_Investment Service_Sector_Investment1(t_policy_year = t_policy_year, p_avg_life_serv_cap_1 = p_avg_life_serv_cap_1, p_avg_life_serv_cap_2 = p_avg_life_serv_cap_2, p_serv_cap_out_ratio_1 = p_serv_cap_out_ratio_1, p_serv_cap_out_ratio_2 = p_serv_cap_out_ratio_2, service_capital_init = service_capital_init) "Service sector investment" annotation(Placement(visible = true, transformation(origin = {-100.0,-220.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
      equation
        connect(Service_Sector_Investment1.service_capital,Labor_Utilization1.service_capital) annotation(Line(visible = true, points = {{-78.0,-220.0},{4.0,-220.0},{4.0,-4.0},{98.0,-4.0}}, color = {0,0,191}));
        connect(NR_Resource_Utilization1.ind_cap_out_ratio_2_ICOR2T,Industrial_Investment1.ind_cap_out_ratio_2_ICOR2T) annotation(Line(visible = true, points = {{362.0,-124.0},{428.0,-124.0},{428.0,120.0},{108.0,120.0},{108.0,102.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Human_Welfare_Index1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,70.0},{318.0,70.0}}, color = {0,0,191}));
        connect(Life_Expectancy1.life_expectancy,Human_Welfare_Index1.life_expectancy) annotation(Line(visible = true, points = {{362.0,-20.0},{420.0,-20.0},{420.0,40.0},{250.0,40.0},{250.0,90.0},{318.0,90.0}}, color = {0,0,191}));
        connect(Human_Welfare_Index1.life_expectancy,Human_Fertility1.life_expectancy) annotation(Line(visible = true, points = {{318.0,90.0},{250.0,90.0},{250.0,128.0},{32.0,128.0},{32.0,180.0},{98.0,180.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.industrial_capital,Labor_Utilization1.industrial_capital) annotation(Line(visible = true, points = {{142.0,76.0},{196.0,76.0},{196.0,-78.0},{44.0,-78.0},{44.0,-20.0},{98.0,-20.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.industrial_output,Pollution_Dynamics1.industrial_output) annotation(Line(visible = true, points = {{142.0,84.3414},{210.0,84.3414},{210.0,22.0},{-184.0,22.0},{-184.0,80.0},{-122.0,80.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.industrial_output,NR_Resource_Utilization1.industrial_output) annotation(Line(visible = true, points = {{142.0,84.3414},{210.0,84.3414},{210.0,-130.0},{318.0,-130.0}}, color = {0,0,191}));
        connect(Food_Production1.yield_tech_mult_icor_COYM,Industrial_Investment1.yield_tech_mult_icor_COYM) annotation(Line(visible = true, points = {{-96.0,-142.0},{-96.0,-174.0},{28.0,-174.0},{28.0,96.0},{98.0,96.0}}, color = {0,0,191}));
        connect(NR_Resource_Utilization1.s_fr_cap_al_obt_res,Industrial_Investment1.s_fr_cap_al_obt_res) annotation(Line(visible = true, points = {{362.0,-132.0},{432.0,-132.0},{432.0,132.0},{44.0,132.0},{44.0,72.0},{98.0,72.0}}, color = {0,0,191}));
        connect(Food_Production1.s_fioa_agr,Industrial_Investment1.s_fioa_agr) annotation(Line(visible = true, points = {{-78.0,-136.0},{-4.0,-136.0},{-4.0,64.0},{98.0,64.0}}, color = {0,0,191}));
        connect(Service_Sector_Investment1.s_fioa_serv,Industrial_Investment1.s_fioa_serv) annotation(Line(visible = true, points = {{-78.0,-230.0},{16.0,-230.0},{16.0,38.0},{108.0,38.0},{108.0,58.0}}, color = {0,0,191}));
        connect(Population_Dynamics1.population,Industrial_Investment1.population) annotation(Line(visible = true, points = {{-78.0,190.0},{-10.0,190.0},{-10.0,34.0},{116.0,34.0},{116.0,58.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.population,Life_Expectancy1.population) annotation(Line(visible = true, points = {{116.0,58.0},{116.0,34.0},{260.0,34.0},{260.0,-20.0},{318.0,-20.0}}, color = {0,0,191}));
        connect(Service_Sector_Investment1.serv_out_pc,Labor_Utilization1.serv_out_pc) annotation(Line(visible = true, points = {{-78.0,-210.0},{0.0,-210.0},{0.0,18.0},{70.0,18.0},{108.0,18.0},{108.0,2.0}}, color = {0,0,191}));
        connect(Life_Expectancy1.serv_out_pc,Labor_Utilization1.serv_out_pc) annotation(Line(visible = true, points = {{318.0,-4.0},{218.0,-4.0},{218.0,18.0},{108.0,18.0},{108.0,10.0},{108.0,2.0}}, color = {0,0,191}));
        connect(Human_Fertility1.ind_out_pc,Labor_Utilization1.ind_out_pc) annotation(Line(visible = true, points = {{98.0,170.0},{36.0,170.0},{36.0,-12.0},{98.0,-12.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.arable_land,Labor_Utilization1.arable_land) annotation(Line(visible = true, points = {{-78.0,-20.0},{40.0,-20.0},{40.0,-36.0},{98.0,-36.0}}, color = {0,0,191}));
        connect(Labor_Utilization1.arable_land,Human_Ecological_Footprint1.arable_land) annotation(Line(visible = true, points = {{98.0,-36.0},{40.0,-36.0},{40.0,-64.0},{264.0,-64.0},{264.0,190.0},{318.0,190.0}}, color = {0,0,191}));
        connect(Food_Production1.agr_inp_per_hect,Labor_Utilization1.agr_inp_per_hect) annotation(Line(visible = true, points = {{-78.0,-128.0},{-16.0,-128.0},{-16.0,-28.0},{98.0,-28.0}}, color = {0,0,191}));
        connect(Population_Dynamics1.labor_force,Labor_Utilization1.labor_force) annotation(Line(visible = true, points = {{-78.0,170.0},{12.0,170.0},{12.0,-60.0},{108.0,-60.0},{108.0,-42.0}}, color = {0,0,191}));
        connect(Labor_Utilization1.capital_util_fr,Service_Sector_Investment1.capital_util_fr) annotation(Line(visible = true, points = {{142.0,-28.0},{224.0,-28.0},{224.0,-180.0},{-172.0,-180.0},{-172.0,-208.0},{-122.0,-208.0}}, color = {0,0,191}));
        connect(Labor_Utilization1.capital_util_fr,Industrial_Investment1.capital_util_fr) annotation(Line(visible = true, points = {{142.0,-28.0},{224.0,-28.0},{224.0,-180.0},{24.0,-180.0},{24.0,80.0},{98.0,80.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Life_Expectancy1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,-12.0},{318.0,-12.0}}, color = {0,0,191}));
        connect(Land_Fertility1.ppoll_index,Life_Expectancy1.ppoll_index) annotation(Line(visible = true, points = {{98.0,-130.0},{20.0,-130.0},{20.0,-70.0},{268.0,-70.0},{268.0,-36.0},{318.0,-36.0}}, color = {0,0,191}));
        connect(Food_Production1.food_pc,Life_Expectancy1.food_pc) annotation(Line(visible = true, points = {{-96.0,-98.0},{-96.0,-74.0},{256.0,-74.0},{256.0,-28.0},{318.0,-28.0}}, color = {0,0,191}));
        connect(NR_Resource_Utilization1.pc_res_use_mlt,Pollution_Dynamics1.pc_res_use_mlt) annotation(Line(visible = true, points = {{362.0,-108.0},{424.0,-108.0},{424.0,124.0},{-176.0,124.0},{-176.0,96.0},{-122.0,96.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,NR_Resource_Utilization1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,-110.0},{318.0,-110.0}}, color = {0,0,191}));
        connect(Land_Fertility1.yield_tech_LYTD,Food_Production1.yield_tech_LYTD) annotation(Line(visible = true, points = {{142.0,-110.0},{192.0,-110.0},{192.0,-82.0},{-112.0,-82.0},{-112.0,-90.0},{-112.0,-98.0}}, color = {0,0,191}));
        connect(Food_Production1.p_fr_inp_for_land_maint,Land_Fertility1.p_fr_inp_for_land_maint) annotation(Line(visible = true, points = {{-88.0,-142.0},{-88.0,-160.0},{44.0,-160.0},{44.0,-120.0},{98.0,-120.0}}, color = {0,0,191}));
        connect(Land_Fertility1.land_fertility,Food_Production1.land_fertility) annotation(Line(visible = true, points = {{142.0,-120.0},{192.0,-120.0},{192.0,-164.0},{-112.0,-164.0},{-112.0,-142.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.s_yield_mlt_air_poll,Food_Production1.s_yield_mlt_air_poll) annotation(Line(visible = true, points = {{-78.0,72.0},{32.0,72.0},{32.0,-168.0},{-104.0,-168.0},{-104.0,-142.0}}, color = {0,0,191}));
        connect(Service_Sector_Investment1.serv_out_pc,Human_Fertility1.serv_out_pc) annotation(Line(visible = true, points = {{-78.0,-210.0},{0.0,-210.0},{0.0,190.0},{98.0,190.0}}, color = {0,0,191}));
        connect(Food_Production1.ind_out_pc,Service_Sector_Investment1.ind_out_pc) annotation(Line(visible = true, points = {{-122.0,-128.0},{-188.0,-128.0},{-188.0,-232.0},{-122.0,-232.0}}, color = {0,0,191}));
        connect(Food_Production1.industrial_output,Service_Sector_Investment1.industrial_output) annotation(Line(visible = true, points = {{-122.0,-136.0},{-184.0,-136.0},{-184.0,-224.0},{-122.0,-224.0}}, color = {0,0,191}));
        connect(Food_Production1.population,Service_Sector_Investment1.population) annotation(Line(visible = true, points = {{-122.0,-104.0},{-180.0,-104.0},{-180.0,-216.0},{-122.0,-216.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.industrial_output,Food_Production1.industrial_output) annotation(Line(visible = true, points = {{-122.0,80.0},{-184.0,80.0},{-184.0,-136.0},{-122.0,-136.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.ind_out_pc,Food_Production1.ind_out_pc) annotation(Line(visible = true, points = {{-122.0,-28.0},{-188.0,-28.0},{-188.0,-128.0},{-122.0,-128.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.population,Food_Production1.population) annotation(Line(visible = true, points = {{-122.0,-36.0},{-180.0,-36.0},{-180.0,-104.0},{-122.0,-104.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.arable_land,Food_Production1.arable_land) annotation(Line(visible = true, points = {{-122.0,64.0},{-176.0,64.0},{-176.0,-112.0},{-122.0,-112.0}}, color = {0,0,191}));
        connect(Food_Production1.land_yield,Arable_Land_Dynamics1.land_yield) annotation(Line(visible = true, points = {{-78.0,-112.0},{-20.0,-112.0},{-20.0,-60.0},{-192.0,-60.0},{-192.0,-4.0},{-122.0,-4.0}}, color = {0,0,191}));
        connect(Food_Production1.agr_inp_per_hect,Pollution_Dynamics1.agr_inp_per_hect) annotation(Line(visible = true, points = {{-78.0,-128.0},{-16.0,-128.0},{-16.0,-56.0},{-196.0,-56.0},{-196.0,72.0},{-122.0,72.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.ppoll_index,Land_Fertility1.ppoll_index) annotation(Line(visible = true, points = {{-78.0,80.0},{20.0,80.0},{20.0,-130.0},{98.0,-130.0}}, color = {0,0,191}));
        connect(Food_Production1.food_ratio,Land_Fertility1.food_ratio) annotation(Line(visible = true, points = {{-78.0,-104.0},{10.0,-104.0},{10.0,-110.0},{98.0,-110.0}}, color = {0,0,191}));
        connect(Food_Production1.tot_agric_invest,Arable_Land_Dynamics1.tot_agric_invest) annotation(Line(visible = true, points = {{-88.0,-98.0},{-88.0,-52.0},{-200.0,-52.0},{-200.0,-20.0},{-122.0,-20.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.urban_ind_land,Human_Ecological_Footprint1.urban_ind_land) annotation(Line(visible = true, points = {{-78.0,-30.0},{-24.0,-30.0},{-24.0,136.0},{252.0,136.0},{252.0,180.0},{318.0,180.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.fr_inp_al_land_dev,Food_Production1.fr_inp_al_land_dev) annotation(Line(visible = true, points = {{-78.0,-10.0},{-28.0,-10.0},{-28.0,-68.0},{-172.0,-68.0},{-172.0,-120.0},{-122.0,-120.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.population,Arable_Land_Dynamics1.population) annotation(Line(visible = true, points = {{-122.0,88.0},{-180.0,88.0},{-180.0,-36.0},{-122.0,-36.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Arable_Land_Dynamics1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,30.0},{-188.0,30.0},{-188.0,-28.0},{-122.0,-28.0}}, color = {0,0,191}));
        connect(Food_Production1.marg_prod_agr_inp,Arable_Land_Dynamics1.marg_prod_agr_inp) annotation(Line(visible = true, points = {{-78.0,-120.0},{-24.0,-120.0},{-24.0,-64.0},{-172.0,-64.0},{-172.0,-12.0},{-122.0,-12.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.ppoll_gen_rt,Human_Ecological_Footprint1.ppoll_gen_rt) annotation(Line(visible = true, points = {{-78.0,64.0},{-20.0,64.0},{-20.0,148.0},{256.0,148.0},{256.0,170.0},{318.0,170.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.arable_land,Pollution_Dynamics1.arable_land) annotation(Line(visible = true, points = {{-78.0,-20.0},{-20.0,-20.0},{-20.0,26.0},{-176.0,26.0},{-176.0,64.0},{-122.0,64.0}}, color = {0,0,191}));
        connect(Population_Dynamics1.population,Pollution_Dynamics1.population) annotation(Line(visible = true, points = {{-78.0,190.0},{-10.0,190.0},{-10.0,140.0},{-180.0,140.0},{-180.0,88.0},{-122.0,88.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Human_Fertility1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,30.0},{36.0,30.0},{36.0,170.0},{98.0,170.0}}, color = {0,0,191}));
        connect(Human_Fertility1.life_expectancy,Population_Dynamics1.life_expectancy) annotation(Line(visible = true, points = {{98.0,180.0},{32.0,180.0},{32.0,128.0},{-190.0,128.0},{-190.0,170.0},{-122.0,170.0}}, color = {0,0,191}));
        connect(Human_Fertility1.total_fertility,Population_Dynamics1.total_fertility) annotation(Line(visible = true, points = {{142.0,180.0},{196.0,180.0},{196.0,144.0},{-200.0,144.0},{-200.0,190.0},{-122.0,190.0}}, color = {0,0,191}));
        connect(Life_Expectancy1.population,NR_Resource_Utilization1.population) annotation(Line(visible = true, points = {{318.0,-20.0},{260.0,-20.0},{260.0,-120.0},{318.0,-120.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.ppoll_tech_mult_icor_COPM,Industrial_Investment1.ppoll_tech_mult_icor_COPM) annotation(Line(visible = true, origin = {10.0,88.0}, points = {{-88.0,0.0},{88.0,0.0}}, color = {0,0,191}));
        population = Population_Dynamics1.Population.y;
        food = Food_Production1.Food.food;
        industrial_output = Industrial_Investment1.Industrial_Output.industrial_output;
        ppoll_index = Pollution_Dynamics1.PPoll_Index.y;
        nr_resources = NR_Resource_Utilization1.NR_Resources.y;
        fioa_ind = Industrial_Investment1.FIOA_Ind.fioa_ind;
        s_fioa_agr = Food_Production1.S_FIOA_Agr.s_fioa_agr;
        s_fioa_cons = Industrial_Investment1.S_FIOA_Cons.s_fioa_cons;
        s_fioa_serv = Service_Sector_Investment1.S_FIOA_Serv.s_fioa_serv;
        s_fr_cap_al_obt_res = NR_Resource_Utilization1.S_Fr_Cap_Al_Obt_Res.s_fr_cap_al_obt_res;
        life_expectancy = Life_Expectancy1.Life_Expectancy.y;
        food_pc = Food_Production1.Food_PC.y;
        serv_out_pc = Service_Sector_Investment1.Serv_Out_PC.y;
        ind_out_pc = Industrial_Investment1.Ind_Out_PC.y;
        human_ecological_footprint = Human_Ecological_Footprint1.HEF_Human_Ecological_Footprint.hef_human_ecological_footprint;
        human_welfare_index = Human_Welfare_Index1.HWI_Human_Welfare_Index.hwi_human_welfare_index;
        annotation(Icon(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-220.0,-260.0},{442.0,222.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-80.0,118.0},{300.0,190.0}}, textString = "More Accessible", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-82.0,64.0},{304.0,130.0}}, textString = "Non-recoverable", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-112.0,-14.0},{332.0,90.0}}, textString = "Natural Resources,", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-108.0,-72.0},{330.0,14.0}}, textString = "Pollution Control,", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-197.0,-248.0},{421.0,60.0}}, textString = "Land Yield Enhancement,", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-162.0,-264.0},{381.0,-38.0}}, textString = "Erosion Protection, and", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-138.0,-324.0},{369.0,-98.0}}, textString = "Resource Efficiency", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10})), experiment(StartTime = 1900, StopTime = 2100), experimentSetupOutput, Documentation(info = "<html>
This is <font color=red><b>Scenario #6</b></font> of the <font color=red><b>WORLD3</b></font> model. This scenario starts out with the same assumptions as <font color=red><b>Scenario #5</b></font>.  Land erosion protection has indeed helped.  Yet more people can live on this planet for yet more years.  In the end, however, we are once again running out of natural resources. <p>
 
In this new scenario, it is thus postulated to spend more money on increasing the efficiency of use of the remaining natural resources. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Meadows, D.H., D.L. Meadows, J. Randers, and W.W. Behrens III (1972), <i>Limits to Growth: A Report for the Club of Rome's Project on the Predicament of Mankind</i>, Universe Books, New York, 205p.
<li> Meadows, D.L., W.W. Behrens III, D.M., Meadows, R.F. Naill, J. Randers, and E.K.O. Zahn (1974), <a href=\"http://www.pegasuscom.com/BookDetail.asp?BookQuery_Action=Find('ISBN','XDYGO')\">Dynamics of Growth in a Finite World</a>, Wright-Allen Press, 637p.
<li> Meadows, D.H., D.L. Meadows, and J. Randers (1992), <i>Beyond the Limits</i>, Chelsea Green, 300p.
<li> Meadows, D.H., J. Randers, and D.L. Meadows (2004), <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a>, Chelsea Green, 368p.
</ol> <p>
 
<hr> <p>
 
In order to accomplish this change, you need to modify the table P_Res_Tech_Chg_Mlt_NRCM that models a multiplier to the investment in technology to increase resource use efficiency: <p>
<font color=red>parameter Real p_res_tech_chg_mlt[:] = {<b>-0.04</b>,<b>-0.04</b>,0,0} \"Resource technology change multiplier\";</font>. <p>
 
<hr> <p>
 
Simulate the model from 1900 until 2100, and display the same variables as in the book <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a> at page 219: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_16.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_17.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_18.png\"> <p>
 
<hr> <p>
 
This seems to have done the trick.  The population no longer declines during the 21<sup>st</sup> century.  Is this effort sustainable? <p>
 
To answer this question, let us simulate the model once more, this time from 1900 until 2500: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_19.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_20.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_21.png\"> <p>
 
<hr> <p>
 
The effort is indeed sustainable.  The world population hovers around 10 billion people.  Yet, these people live very poorly.  In the end, no money is left for industrial production and/or services.  All of the human efforts are spent on feeding the world population in a sustainable fashion.  The life expectancy is back at its pre-industrial value of 27.6 years. <p>
 
No generation before ours lived as long and as prosperous as we do, and the likelihood is large that none coming after us will do so either. <p>
 
<hr> <p>
</html>", revisions = ""));
      end Scenario_6;
      model Scenario_7 "More abundant non-recoverable natural resources and birth control"
        parameter Real agr_mtl_toxic_index(unit = "1/dollar") = 1 "Agricultural materials toxicity index";
        parameter Real assim_half_life_1970(unit = "yr") = 1.5 "Pollution assimilation half life in 1970";
        parameter Real avg_life_land_norm(unit = "yr") = 1000 "Normal life span of land";
        parameter Real des_compl_fam_size_norm = 3.8 "Desired normal complete family size";
        parameter Real des_food_ratio_dfr = 2 "Desired food ratio";
        parameter Real des_ppoll_index_DPOLX = 1.2 "Desired persistent pollution index";
        parameter Real des_res_use_rt_DNRUR(unit = "ton/yr") = 4800000000.0 "Desired resource utilization rate";
        parameter Real food_short_perc_del(unit = "yr") = 2 "Food shortage perception delay";
        parameter Real fr_agr_inp_pers_mtl = 0.001 "Effective fraction of agricultural pollution input";
        parameter Real frac_res_pers_mtl = 0.02 "Effective fraction of resource utilization on pollution generation";
        parameter Real hlth_serv_impact_del(unit = "yr") = 20 "Health service impact delay";
        parameter Real income_expect_avg_time(unit = "yr") = 3 "Income expected average time";
        parameter Real ind_mtl_emiss_fact(unit = "1/ton") = 0.1 "Industrial materials emission factor";
        parameter Real ind_mtl_toxic_index = 10.0 "Industrial materials toxicity index";
        parameter Real ind_out_pc_des(unit = "dollar/yr") = 400 "Desired annual industrial per capita output";
        parameter Real ind_out_in_1970(unit = "dollar/yr") = 790000000000.0 "Industrial output in 1970";
        parameter Real inherent_land_fert(unit = "kg/(hectare.yr)") = 600 "Inherent land fertility";
        parameter Real labor_force_partic = 0.75 "Percentage of participating labor force";
        parameter Real labor_util_fr_del_time(unit = "yr") = 2 "Labor utilization fraction delay time";
        parameter Real land_fr_harvested = 0.7 "Land fraction harvested";
        parameter Real life_expect_norm(unit = "yr") = 28 "Normal life expectancy";
        parameter Real lifet_perc_del(unit = "yr") = 20 "Perceived life-time delay";
        parameter Real max_tot_fert_norm = 12 "Normal maximal total fertility";
        parameter Real p_avg_life_agr_inp_1(unit = "yr") = 2 "Default average life of agricultural input";
        parameter Real p_avg_life_agr_inp_2(unit = "yr") = 2 "Controlled average life of agricultural input";
        parameter Real p_avg_life_ind_cap_1(unit = "yr") = 14 "Default average life of industrial capital";
        parameter Real p_avg_life_ind_cap_2(unit = "yr") = 14 "Controlled average life of industrial capital";
        parameter Real p_avg_life_serv_cap_1(unit = "yr") = 20 "Default average life of service sector capital";
        parameter Real p_avg_life_serv_cap_2(unit = "yr") = 20 "Controlled average life of service sector capital";
        parameter Real p_fioa_cons_const_1 = 0.43 "Default fraction of industrial output allocated to consumption";
        parameter Real p_fioa_cons_const_2 = 0.43 "Controlled fraction of industrial output allocated to consumption";
        parameter Real p_ind_cap_out_ratio_1(unit = "yr") = 3 "Default industrial capital output ratio";
        parameter Real p_land_yield_fact_1 = 1 "Default land yield factor";
        parameter Real p_nr_res_use_fact_1 = 1 "Default non-recoverable resource utilization factor";
        parameter Real p_ppoll_gen_fact_1 = 1 "Default persistent pollution generation factor";
        parameter Real p_serv_cap_out_ratio_1(unit = "yr") = 1.0 "Default fraction of service sector output ratio";
        parameter Real p_serv_cap_out_ratio_2(unit = "yr") = 1.0 "Controlled fraction of service sector output ratio";
        parameter Real pot_arable_land_tot(unit = "hectare") = 3200000000.0 "Total potential arable land";
        parameter Real ppoll_in_1970 = 136000000.0 "Persistent pollution in 1970";
        parameter Real ppoll_trans_del(unit = "yr") = 20 "Persistent pollution transmission delay";
        parameter Real processing_loss = 0.1 "Processing loss";
        parameter Real reproductive_lifetime(unit = "yr") = 30.0 "Reproductive life time";
        parameter Real social_adj_del(unit = "yr") = 20 "Social adjustment delay";
        parameter Real social_discount(unit = "1/yr") = 0.07 "Social discount";
        parameter Real subsist_food_pc(unit = "kg/yr") = 230 "Available per capita food";
        parameter Real tech_dev_del_TDD(unit = "yr") = 20 "Technology development time";
        parameter Real urb_ind_land_dev_time(unit = "yr") = 10 "Urban and industrial land development time";
        parameter Real t_air_poll_time(unit = "yr") = 4000 "Air pollution change time";
        parameter Real t_fcaor_time(unit = "yr") = 2002 "Year of capital allocation to resource use efficiency";
        parameter Real t_fert_cont_eff_time(unit = "yr") = 2002 "Year of continued fertility change";
        parameter Real t_ind_equil_time(unit = "yr") = 4000 "Year of industrial equilibrium";
        parameter Real t_land_life_time(unit = "yr") = 4000 "Land life time";
        parameter Real t_policy_year(unit = "yr") = 4000 "Year of policy change";
        parameter Real t_pop_equil_time(unit = "yr") = 4000 "Population equilibrium time";
        parameter Real t_zero_pop_grow_time(unit = "yr") = 2002 "Time to zero population growth";
        parameter Real p_fr_cap_al_obt_res_2[:] = {1,0.1,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05} "Non-renewable resource fraction remaining";
        parameter Real p_ppoll_tech_chg_mlt[:] = {0,0,0,0} "Persistent pollution technology change multiplier";
        parameter Real p_res_tech_chg_mlt[:] = {0,0,0,0} "Resource technology change multiplier";
        parameter Real p_yield_tech_chg_mlt[:] = {0,0,0,0} "Yield technology change multiplier";
        parameter Real agr_inp_init(unit = "dollar/yr") = 5000000000.0 "Initial agricultural input";
        parameter Real arable_land_init(unit = "hectare") = 900000000.0 "Initial arable land";
        parameter Real industrial_capital_init(unit = "dollar") = 210000000000.0 "Initial industrial investment";
        parameter Real labor_util_fr_del_init = 1 "Initial delayed labor utilization fraction";
        parameter Real land_fertility_init(unit = "kg/(hectare.yr)") = 600 "Initial industrial investment";
        parameter Real nr_resources_init(unit = "ton") = 2000000000000.0 "Initial available non-recoverable resources";
        parameter Real perc_food_ratio_init = 1 "Initial perceived food ratio";
        parameter Real pers_pollution_init = 25000000.0 "Initial persistent pollution";
        parameter Real pop1_init = 650000000.0 "Initial population 14 years and younger";
        parameter Real pop2_init = 700000000.0 "Initial population 15 to 44 years old";
        parameter Real pop3_init = 190000000.0 "Initial population 45 to 64 years old";
        parameter Real pop4_init = 60000000.0 "Initial population 65 years and older";
        parameter Real pot_arable_land_init(unit = "hectare") = 2300000000.0 "Initial potential arable land";
        parameter Real ppoll_tech_init = 1 "Initial persistent pollution technology change factor";
        parameter Real res_tech_init = 1 "Initial non-recoverable resource technology factor";
        parameter Real service_capital_init(unit = "dollar") = 144000000000.0 "Initial service sector investment";
        parameter Real urban_ind_land_init(unit = "hectare") = 8200000.0 "Initial urban and industrial land";
        parameter Real yield_tech_init = 1 "Initial yield technology factor";
        output Real population "Total human world population";
        output Real food(unit = "dollar/yr") "Total annually produced food";
        output Real industrial_output(unit = "dollar/yr") "Total annual world industrial output";
        output Real ppoll_index "Persistent pollution index";
        output Real nr_resources "Remaining non-recoverable natural resources";
        output Real fioa_ind "Fraction of industrial output allocated to industrial/military complex";
        output Real s_fioa_agr "Fraction of industrial output allocated to food production";
        output Real s_fioa_cons "Fraction of industrial output allocated to consumption";
        output Real s_fioa_serv "Fraction of industrial output allocated to service sector";
        output Real s_fr_cap_al_obt_res "Fraction of capital allocated to resource use efficiency";
        output Real life_expectancy(unit = "yr") "Life expectancy";
        output Real food_pc(unit = "dollar/yr") "Total annual food per person";
        output Real serv_out_pc(unit = "dollar/yr") "Total annual services per person";
        output Real ind_out_pc(unit = "dollar/yr") "Total annual consumer goods per person";
        output Real human_ecological_footprint(unit = "Gha") "Human ecological footprint";
        output Real human_welfare_index "Human welfare index";
        SystemDynamics.WorldDynamics.World3.Population_Dynamics Population_Dynamics1(pop1_init = pop1_init, pop2_init = pop2_init, pop3_init = pop3_init, pop4_init = pop4_init, labor_force_partic = labor_force_partic, reproductive_lifetime = reproductive_lifetime, t_pop_equil_time = t_pop_equil_time) "Population dynamics" annotation(Placement(visible = true, transformation(origin = {-100.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Pollution_Dynamics Pollution_Dynamics1(agr_mtl_toxic_index = agr_mtl_toxic_index, assim_half_life_1970 = assim_half_life_1970, des_ppoll_index_DPOLX = des_ppoll_index_DPOLX, fr_agr_inp_pers_mtl = fr_agr_inp_pers_mtl, frac_res_pers_mtl = frac_res_pers_mtl, ind_mtl_emiss_fact = ind_mtl_emiss_fact, ind_mtl_toxic_index = ind_mtl_toxic_index, ind_out_in_1970 = ind_out_in_1970, p_ppoll_gen_fact_1 = p_ppoll_gen_fact_1, pers_pollution_init = pers_pollution_init, ppoll_in_1970 = ppoll_in_1970, ppoll_tech_init = ppoll_tech_init, ppoll_trans_del = ppoll_trans_del, t_air_poll_time = t_air_poll_time, t_policy_year = t_policy_year, tech_dev_del_TDD = tech_dev_del_TDD, p_ppoll_tech_chg_mlt = p_ppoll_tech_chg_mlt) "Persistent pollution generation" annotation(Placement(visible = true, transformation(origin = {-100.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Arable_Land_Dynamics Arable_Land_Dynamics1(arable_land_init = arable_land_init, avg_life_land_norm = avg_life_land_norm, inherent_land_fert = inherent_land_fert, pot_arable_land_init = pot_arable_land_init, pot_arable_land_tot = pot_arable_land_tot, social_discount = social_discount, t_land_life_time = t_land_life_time, urban_ind_land_init = urban_ind_land_init, urb_ind_land_dev_time = urb_ind_land_dev_time) "Arable land dynamics" annotation(Placement(visible = true, transformation(origin = {-100.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Food_Production Food_Production1(agr_inp_init = agr_inp_init, food_short_perc_del = food_short_perc_del, p_avg_life_agr_inp_1 = p_avg_life_agr_inp_1, p_avg_life_agr_inp_2 = p_avg_life_agr_inp_2, p_land_yield_fact_1 = p_land_yield_fact_1, perc_food_ratio_init = perc_food_ratio_init, processing_loss = processing_loss, subsist_food_pc = subsist_food_pc, t_policy_year = t_policy_year, tech_dev_del_TDD = tech_dev_del_TDD, land_fr_harvested = land_fr_harvested) "Food production" annotation(Placement(visible = true, transformation(origin = {-100.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Human_Ecological_Footprint Human_Ecological_Footprint1 "Human ecological footprint" annotation(Placement(visible = true, transformation(origin = {340.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Human_Fertility Human_Fertility1(des_compl_fam_size_norm = des_compl_fam_size_norm, hlth_serv_impact_del = hlth_serv_impact_del, income_expect_avg_time = income_expect_avg_time, lifet_perc_del = lifet_perc_del, max_tot_fert_norm = max_tot_fert_norm, social_adj_del = social_adj_del, t_fert_cont_eff_time = t_fert_cont_eff_time, t_zero_pop_grow_time = t_zero_pop_grow_time) "Human fertility" annotation(Placement(visible = true, transformation(origin = {120.0,180.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Human_Welfare_Index Human_Welfare_Index1 "Human welfare index" annotation(Placement(visible = true, transformation(origin = {340.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Industrial_Investment Industrial_Investment1(industrial_capital_init = industrial_capital_init, ind_out_pc_des = ind_out_pc_des, p_avg_life_ind_cap_1 = p_avg_life_ind_cap_1, p_avg_life_ind_cap_2 = p_avg_life_ind_cap_2, p_fioa_cons_const_1 = p_fioa_cons_const_1, p_fioa_cons_const_2 = p_fioa_cons_const_2, p_ind_cap_out_ratio_1 = p_ind_cap_out_ratio_1, t_ind_equil_time = t_ind_equil_time, t_policy_year = t_policy_year) "Industrial investment" annotation(Placement(visible = true, transformation(origin = {120.0,80.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Labor_Utilization Labor_Utilization1(labor_util_fr_del_init = labor_util_fr_del_init, labor_util_fr_del_time = labor_util_fr_del_time) "Labor utilization" annotation(Placement(visible = true, transformation(origin = {120.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Land_Fertility Land_Fertility1(des_food_ratio_dfr = des_food_ratio_dfr, inherent_land_fert = inherent_land_fert, land_fertility_init = land_fertility_init, t_policy_year = t_policy_year, yield_tech_init = yield_tech_init, p_yield_tech_chg_mlt = p_yield_tech_chg_mlt) "Land fertility" annotation(Placement(visible = true, transformation(origin = {120.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Life_Expectancy Life_Expectancy1(hlth_serv_impact_del = hlth_serv_impact_del, life_expect_norm = life_expect_norm, subsist_food_pc = subsist_food_pc) "Life expectancy" annotation(Placement(visible = true, transformation(origin = {340.0,-20.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.NR_Resource_Utilization NR_Resource_Utilization1(des_res_use_rt_DNRUR = des_res_use_rt_DNRUR, nr_resources_init = nr_resources_init, p_nr_res_use_fact_1 = p_nr_res_use_fact_1, res_tech_init = res_tech_init, t_policy_year = t_policy_year, t_fcaor_time = t_fcaor_time, tech_dev_del_TDD = tech_dev_del_TDD, p_res_tech_chg_mlt = p_res_tech_chg_mlt, p_fr_cap_al_obt_res_2 = p_fr_cap_al_obt_res_2) "Non-recoverable natural resource utilization" annotation(Placement(visible = true, transformation(origin = {340.0,-120.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
        SystemDynamics.WorldDynamics.World3.Service_Sector_Investment Service_Sector_Investment1(t_policy_year = t_policy_year, p_avg_life_serv_cap_1 = p_avg_life_serv_cap_1, p_avg_life_serv_cap_2 = p_avg_life_serv_cap_2, p_serv_cap_out_ratio_1 = p_serv_cap_out_ratio_1, p_serv_cap_out_ratio_2 = p_serv_cap_out_ratio_2, service_capital_init = service_capital_init) "Service sector investment" annotation(Placement(visible = true, transformation(origin = {-100.0,-220.0}, extent = {{-20.0,-20.0},{20.0,20.0}}, rotation = 0)));
      equation
        connect(Service_Sector_Investment1.serv_out_pc,Human_Fertility1.serv_out_pc) annotation(Line(visible = true, points = {{-78.0,-210.0},{0.0,-210.0},{0.0,190.0},{98.0,190.0}}, color = {0,0,191}));
        connect(Service_Sector_Investment1.s_fioa_serv,Industrial_Investment1.s_fioa_serv) annotation(Line(visible = true, points = {{-78.0,-230.0},{16.0,-230.0},{16.0,38.0},{108.0,38.0},{108.0,58.0}}, color = {0,0,191}));
        connect(Food_Production1.population,Service_Sector_Investment1.population) annotation(Line(visible = true, points = {{-122.0,-104.0},{-180.0,-104.0},{-180.0,-216.0},{-122.0,-216.0}}, color = {0,0,191}));
        connect(Food_Production1.ind_out_pc,Service_Sector_Investment1.ind_out_pc) annotation(Line(visible = true, points = {{-122.0,-128.0},{-188.0,-128.0},{-188.0,-232.0},{-122.0,-232.0}}, color = {0,0,191}));
        connect(Food_Production1.industrial_output,Service_Sector_Investment1.industrial_output) annotation(Line(visible = true, points = {{-122.0,-136.0},{-184.0,-136.0},{-184.0,-224.0},{-122.0,-224.0}}, color = {0,0,191}));
        connect(Labor_Utilization1.capital_util_fr,Service_Sector_Investment1.capital_util_fr) annotation(Line(visible = true, points = {{142.0,-28.0},{224.0,-28.0},{224.0,-180.0},{-172.0,-180.0},{-172.0,-208.0},{-122.0,-208.0}}, color = {0,0,191}));
        connect(Service_Sector_Investment1.serv_out_pc,Labor_Utilization1.serv_out_pc) annotation(Line(visible = true, points = {{-78.0,-210.0},{0.0,-210.0},{0.0,18.0},{70.0,18.0},{108.0,18.0},{108.0,2.0}}, color = {0,0,191}));
        connect(Service_Sector_Investment1.service_capital,Labor_Utilization1.service_capital) annotation(Line(visible = true, points = {{-78.0,-220.0},{4.0,-220.0},{4.0,-4.0},{98.0,-4.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.ppoll_tech_mult_icor_COPM,Industrial_Investment1.ppoll_tech_mult_icor_COPM) annotation(Line(visible = true, origin = {10.0,88.0}, points = {{-88.0,0.0},{88.0,0.0}}, color = {0,0,191}));
        connect(Life_Expectancy1.population,NR_Resource_Utilization1.population) annotation(Line(visible = true, points = {{318.0,-20.0},{260.0,-20.0},{260.0,-120.0},{318.0,-120.0}}, color = {0,0,191}));
        connect(Human_Fertility1.total_fertility,Population_Dynamics1.total_fertility) annotation(Line(visible = true, points = {{142.0,180.0},{196.0,180.0},{196.0,144.0},{-200.0,144.0},{-200.0,190.0},{-122.0,190.0}}, color = {0,0,191}));
        connect(Human_Fertility1.life_expectancy,Population_Dynamics1.life_expectancy) annotation(Line(visible = true, points = {{98.0,180.0},{32.0,180.0},{32.0,128.0},{-190.0,128.0},{-190.0,170.0},{-122.0,170.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Human_Fertility1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,30.0},{36.0,30.0},{36.0,170.0},{98.0,170.0}}, color = {0,0,191}));
        connect(Population_Dynamics1.population,Pollution_Dynamics1.population) annotation(Line(visible = true, points = {{-78.0,190.0},{-10.0,190.0},{-10.0,140.0},{-180.0,140.0},{-180.0,88.0},{-122.0,88.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.arable_land,Pollution_Dynamics1.arable_land) annotation(Line(visible = true, points = {{-78.0,-20.0},{-20.0,-20.0},{-20.0,26.0},{-176.0,26.0},{-176.0,64.0},{-122.0,64.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.ppoll_gen_rt,Human_Ecological_Footprint1.ppoll_gen_rt) annotation(Line(visible = true, points = {{-78.0,64.0},{-20.0,64.0},{-20.0,148.0},{256.0,148.0},{256.0,170.0},{318.0,170.0}}, color = {0,0,191}));
        connect(Food_Production1.marg_prod_agr_inp,Arable_Land_Dynamics1.marg_prod_agr_inp) annotation(Line(visible = true, points = {{-78.0,-120.0},{-24.0,-120.0},{-24.0,-64.0},{-172.0,-64.0},{-172.0,-12.0},{-122.0,-12.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Arable_Land_Dynamics1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,30.0},{-188.0,30.0},{-188.0,-28.0},{-122.0,-28.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.population,Arable_Land_Dynamics1.population) annotation(Line(visible = true, points = {{-122.0,88.0},{-180.0,88.0},{-180.0,-36.0},{-122.0,-36.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.fr_inp_al_land_dev,Food_Production1.fr_inp_al_land_dev) annotation(Line(visible = true, points = {{-78.0,-10.0},{-28.0,-10.0},{-28.0,-68.0},{-172.0,-68.0},{-172.0,-120.0},{-122.0,-120.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.urban_ind_land,Human_Ecological_Footprint1.urban_ind_land) annotation(Line(visible = true, points = {{-78.0,-30.0},{-24.0,-30.0},{-24.0,136.0},{252.0,136.0},{252.0,180.0},{318.0,180.0}}, color = {0,0,191}));
        connect(Land_Fertility1.yield_tech_LYTD,Food_Production1.yield_tech_LYTD) annotation(Line(visible = true, points = {{142.0,-110.0},{192.0,-110.0},{192.0,-82.0},{-112.0,-82.0},{-112.0,-87.9893},{-112.0,-98.0}}, color = {0,0,191}));
        connect(Food_Production1.food_ratio,Land_Fertility1.food_ratio) annotation(Line(visible = true, points = {{-78.0,-104.0},{10.0,-104.0},{10.0,-110.0},{98.0,-110.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.ppoll_index,Land_Fertility1.ppoll_index) annotation(Line(visible = true, points = {{-78.0,80.0},{20.0,80.0},{20.0,-130.0},{98.0,-130.0}}, color = {0,0,191}));
        connect(Food_Production1.land_yield,Arable_Land_Dynamics1.land_yield) annotation(Line(visible = true, points = {{-78.0,-112.0},{-20.0,-112.0},{-20.0,-60.0},{-192.0,-60.0},{-192.0,-4.0},{-122.0,-4.0}}, color = {0,0,191}));
        connect(Food_Production1.agr_inp_per_hect,Pollution_Dynamics1.agr_inp_per_hect) annotation(Line(visible = true, points = {{-78.0,-128.0},{-16.0,-128.0},{-16.0,-56.0},{-196.0,-56.0},{-196.0,72.0},{-122.0,72.0}}, color = {0,0,191}));
        connect(Food_Production1.p_fr_inp_for_land_maint,Land_Fertility1.p_fr_inp_for_land_maint) annotation(Line(visible = true, points = {{-88.0,-142.0},{-88.0,-160.0},{44.0,-160.0},{44.0,-120.0},{98.0,-120.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.s_yield_mlt_air_poll,Food_Production1.s_yield_mlt_air_poll) annotation(Line(visible = true, points = {{-78.0,72.0},{32.0,72.0},{32.0,-168.0},{-104.0,-168.0},{-104.0,-142.0}}, color = {0,0,191}));
        connect(Land_Fertility1.land_fertility,Food_Production1.land_fertility) annotation(Line(visible = true, points = {{142.0,-120.0},{192.0,-120.0},{192.0,-164.0},{-112.0,-164.0},{-112.0,-142.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,NR_Resource_Utilization1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,-110.0},{318.0,-110.0}}, color = {0,0,191}));
        connect(NR_Resource_Utilization1.pc_res_use_mlt,Pollution_Dynamics1.pc_res_use_mlt) annotation(Line(visible = true, points = {{362.0,-108.0},{424.0,-108.0},{424.0,124.0},{-176.0,124.0},{-176.0,96.0},{-122.0,96.0}}, color = {0,0,191}));
        connect(Human_Welfare_Index1.life_expectancy,Human_Fertility1.life_expectancy) annotation(Line(visible = true, points = {{318.0,90.0},{250.0,90.0},{250.0,128.0},{32.0,128.0},{32.0,180.0},{98.0,180.0}}, color = {0,0,191}));
        connect(Life_Expectancy1.life_expectancy,Human_Welfare_Index1.life_expectancy) annotation(Line(visible = true, points = {{362.0,-20.0},{420.0,-20.0},{420.0,40.0},{250.0,40.0},{250.0,90.0},{318.0,90.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Human_Welfare_Index1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,70.0},{318.0,70.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.industrial_capital,Labor_Utilization1.industrial_capital) annotation(Line(visible = true, points = {{142.0,76.0},{196.0,76.0},{196.0,-78.0},{44.0,-78.0},{44.0,-20.0},{98.0,-20.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.industrial_output,NR_Resource_Utilization1.industrial_output) annotation(Line(visible = true, points = {{142.0,84.3414},{210.0,84.3414},{210.0,-130.0},{318.0,-130.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.industrial_output,Pollution_Dynamics1.industrial_output) annotation(Line(visible = true, points = {{142.0,84.3414},{210.0,84.3414},{210.0,22.0},{-184.0,22.0},{-184.0,80.0},{-122.0,80.0}}, color = {0,0,191}));
        connect(NR_Resource_Utilization1.ind_cap_out_ratio_2_ICOR2T,Industrial_Investment1.ind_cap_out_ratio_2_ICOR2T) annotation(Line(visible = true, points = {{362.0,-124.0},{428.0,-124.0},{428.0,120.0},{108.0,120.0},{108.0,102.0}}, color = {0,0,191}));
        connect(Food_Production1.yield_tech_mult_icor_COYM,Industrial_Investment1.yield_tech_mult_icor_COYM) annotation(Line(visible = true, points = {{-96.0,-142.0},{-96.0,-174.0},{28.0,-174.0},{28.0,96.0},{98.0,96.0}}, color = {0,0,191}));
        connect(NR_Resource_Utilization1.s_fr_cap_al_obt_res,Industrial_Investment1.s_fr_cap_al_obt_res) annotation(Line(visible = true, points = {{362.0,-132.0},{432.0,-132.0},{432.0,132.0},{44.0,132.0},{44.0,72.0},{98.0,72.0}}, color = {0,0,191}));
        connect(Food_Production1.s_fioa_agr,Industrial_Investment1.s_fioa_agr) annotation(Line(visible = true, points = {{-78.0,-136.0},{-4.0,-136.0},{-4.0,64.0},{98.0,64.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.population,Life_Expectancy1.population) annotation(Line(visible = true, points = {{116.0,58.0},{116.0,34.0},{260.0,34.0},{260.0,-20.0},{318.0,-20.0}}, color = {0,0,191}));
        connect(Population_Dynamics1.population,Industrial_Investment1.population) annotation(Line(visible = true, points = {{-78.0,190.0},{-10.0,190.0},{-10.0,34.0},{116.0,34.0},{116.0,58.0}}, color = {0,0,191}));
        connect(Food_Production1.tot_agric_invest,Arable_Land_Dynamics1.tot_agric_invest) annotation(Line(visible = true, points = {{-88.0,-98.0},{-88.0,-52.0},{-200.0,-52.0},{-200.0,-20.0},{-122.0,-20.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.arable_land,Food_Production1.arable_land) annotation(Line(visible = true, points = {{-122.0,64.0},{-176.0,64.0},{-176.0,-112.0},{-122.0,-112.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.population,Food_Production1.population) annotation(Line(visible = true, points = {{-122.0,-36.0},{-180.0,-36.0},{-180.0,-104.0},{-122.0,-104.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.ind_out_pc,Food_Production1.ind_out_pc) annotation(Line(visible = true, points = {{-122.0,-28.0},{-188.0,-28.0},{-188.0,-128.0},{-122.0,-128.0}}, color = {0,0,191}));
        connect(Pollution_Dynamics1.industrial_output,Food_Production1.industrial_output) annotation(Line(visible = true, points = {{-122.0,80.0},{-184.0,80.0},{-184.0,-136.0},{-122.0,-136.0}}, color = {0,0,191}));
        connect(Labor_Utilization1.capital_util_fr,Industrial_Investment1.capital_util_fr) annotation(Line(visible = true, points = {{142.0,-28.0},{224.0,-28.0},{224.0,-180.0},{24.0,-180.0},{24.0,80.0},{98.0,80.0}}, color = {0,0,191}));
        connect(Industrial_Investment1.ind_out_pc,Life_Expectancy1.ind_out_pc) annotation(Line(visible = true, points = {{142.0,68.0},{200.0,68.0},{200.0,-12.0},{318.0,-12.0}}, color = {0,0,191}));
        connect(Life_Expectancy1.serv_out_pc,Labor_Utilization1.serv_out_pc) annotation(Line(visible = true, points = {{318.0,-4.0},{218.0,-4.0},{218.0,18.0},{108.0,18.0},{108.0,12.9215},{108.0,2.0}}, color = {0,0,191}));
        connect(Human_Fertility1.ind_out_pc,Labor_Utilization1.ind_out_pc) annotation(Line(visible = true, points = {{98.0,170.0},{36.0,170.0},{36.0,-12.0},{98.0,-12.0}}, color = {0,0,191}));
        connect(Food_Production1.agr_inp_per_hect,Labor_Utilization1.agr_inp_per_hect) annotation(Line(visible = true, points = {{-78.0,-128.0},{-16.0,-128.0},{-16.0,-28.0},{98.0,-28.0}}, color = {0,0,191}));
        connect(Labor_Utilization1.arable_land,Human_Ecological_Footprint1.arable_land) annotation(Line(visible = true, points = {{98.0,-36.0},{40.0,-36.0},{40.0,-64.0},{264.0,-64.0},{264.0,190.0},{318.0,190.0}}, color = {0,0,191}));
        connect(Arable_Land_Dynamics1.arable_land,Labor_Utilization1.arable_land) annotation(Line(visible = true, points = {{-78.0,-20.0},{40.0,-20.0},{40.0,-36.0},{98.0,-36.0}}, color = {0,0,191}));
        connect(Population_Dynamics1.labor_force,Labor_Utilization1.labor_force) annotation(Line(visible = true, points = {{-78.0,170.0},{12.0,170.0},{12.0,-60.0},{108.0,-60.0},{108.0,-42.0}}, color = {0,0,191}));
        connect(Food_Production1.food_pc,Life_Expectancy1.food_pc) annotation(Line(visible = true, points = {{-96.0,-98.0},{-96.0,-74.0},{256.0,-74.0},{256.0,-28.0},{318.0,-28.0}}, color = {0,0,191}));
        connect(Land_Fertility1.ppoll_index,Life_Expectancy1.ppoll_index) annotation(Line(visible = true, points = {{98.0,-130.0},{20.0,-130.0},{20.0,-70.0},{268.0,-70.0},{268.0,-36.0},{318.0,-36.0}}, color = {0,0,191}));
        population = Population_Dynamics1.Population.y;
        food = Food_Production1.Food.food;
        industrial_output = Industrial_Investment1.Industrial_Output.industrial_output;
        ppoll_index = Pollution_Dynamics1.PPoll_Index.y;
        nr_resources = NR_Resource_Utilization1.NR_Resources.y;
        fioa_ind = Industrial_Investment1.FIOA_Ind.fioa_ind;
        s_fioa_agr = Food_Production1.S_FIOA_Agr.s_fioa_agr;
        s_fioa_cons = Industrial_Investment1.S_FIOA_Cons.s_fioa_cons;
        s_fioa_serv = Service_Sector_Investment1.S_FIOA_Serv.s_fioa_serv;
        s_fr_cap_al_obt_res = NR_Resource_Utilization1.S_Fr_Cap_Al_Obt_Res.s_fr_cap_al_obt_res;
        life_expectancy = Life_Expectancy1.Life_Expectancy.y;
        food_pc = Food_Production1.Food_PC.y;
        serv_out_pc = Service_Sector_Investment1.Serv_Out_PC.y;
        ind_out_pc = Industrial_Investment1.Ind_Out_PC.y;
        human_ecological_footprint = Human_Ecological_Footprint1.HEF_Human_Ecological_Footprint.hef_human_ecological_footprint;
        human_welfare_index = Human_Welfare_Index1.HWI_Human_Welfare_Index.hwi_human_welfare_index;
        annotation(Icon(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics = {Rectangle(visible = true, origin = {2.0,-1.0}, lineColor = {0,0,255}, fillColor = {255,255,0}, fillPattern = FillPattern.Solid, extent = {{-222.0,-260.0},{440.0,222.0}}),Text(visible = true, lineColor = {0,0,255}, extent = {{-70.0,58.0},{286.0,132.0}}, textString = "More Abundant", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-84.0,-14.0},{302.0,52.0}}, textString = "Non-recoverable", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-114.0,-106.0},{324.0,-2.0}}, textString = "Natural Resources", fontName = "Arial"),Text(visible = true, lineColor = {0,0,255}, extent = {{-100.0,-178.0},{316.0,-78.0}}, textString = "and Birth Control", fontName = "Arial")}), Diagram(coordinateSystem(extent = {{-220.0,-260.0},{440.0,220.0}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10,10}), graphics), experiment(StartTime = 1900, StopTime = 2100), experimentSetupOutput, Documentation(info = "<html>
This is <font color=red><b>Scenario #7</b></font> of the <font color=red><b>WORLD3</b></font> model. This scenario starts out with the same assumptions as <font color=red><b>Scenario #2</b></font>. <p>
 
We now want to try out another approach.  Instead of improving the means of production, we shall attempt strict birth control.  Every family is limited to two children only.  Due to the age structure momentum, the population keeps growing for another generation.  Then it will finally start on its path of natural attrition. <p>
 
<hr> <p>
 
<b>References:</b> <p>
 
<ol>
<li> Meadows, D.H., D.L. Meadows, J. Randers, and W.W. Behrens III (1972), <i>Limits to Growth: A Report for the Club of Rome's Project on the Predicament of Mankind</i>, Universe Books, New York, 205p.
<li> Meadows, D.L., W.W. Behrens III, D.M., Meadows, R.F. Naill, J. Randers, and E.K.O. Zahn (1974), <a href=\"http://www.pegasuscom.com/BookDetail.asp?BookQuery_Action=Find('ISBN','XDYGO')\">Dynamics of Growth in a Finite World</a>, Wright-Allen Press, 637p.
<li> Meadows, D.H., D.L. Meadows, and J. Randers (1992), <i>Beyond the Limits</i>, Chelsea Green, 300p.
<li> Meadows, D.H., J. Randers, and D.L. Meadows (2004), <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a>, Chelsea Green, 368p.
</ol> <p>
 
<hr> <p>
 
In order to accomplish this change, you need to reset two of the switching times in the model: <p>
<font color=red>parameter Real t_fert_cont_eff_time(unit=\"yr\") = <b>2002</b> \"Year of continued fertility change\";</font>. <p>
<font color=red>parameter Real t_zero_pop_grow_time(unit=\"yr\") = <b>2002</b> \"Time to zero population growth\";</font>. <p>
 
<hr> <p>
 
Simulate the model from 1900 until 2100, and display the same variables as in the book <a href=\"http://www.amazon.com/Limits-Growth-Donella-H-Meadows/dp/193149858X\">Limits to Growth: The 30-Year Update</a> at page 241: <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_22.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_23.png\"> <p>
<img src=\"modelica://SystemDynamics/Resources/Images/World3_24.png\"> <p>
 
<hr> <p>
</html>", revisions = ""));
      end Scenario_7;
      model Scenario_8 "More abundant non-recoverable natural resources, birth control, and stable industrial output"
        parameter Real agr_mtl_toxic_index(unit = "1/dollar") = 1 "Agricultural materials toxicity index";
        parameter Real assim_half_life_1970(unit = "yr") = 1.5 "Pollution assimilation half life in 1970";
        parameter Real avg_life_land_norm(unit = "yr") = 1000 "Normal life span of land";
        parameter Real des_compl_fam_size_norm = 3.8 "Desired normal complete family size";
        parameter Real des_food_ratio_dfr = 2 "Desired food ratio";
        parameter Real des_ppoll_index_DPOLX = 1.2 "Desired persistent pollution index";
        parameter Real des_res_use_rt_DNRUR(unit = "ton/yr") = 4800000000.0 "Desired resource utilization rate";
        parameter Real food_short_perc_del(unit = "yr") = 2 "Food shortage perception delay";
        parameter Real fr_agr_inp_pers_mtl = 0.001 "Effective fraction of agricultural pollution input";
        parameter Real frac_res_pers_mtl = 0.02 "Effective fraction of resource utilization on pollution generation";
        parameter Real hlth_serv_impact_del(unit = "yr") = 20 "Health service impact delay";
        parameter Real income_expect_avg_time(unit = "yr") = 3 "Income expected average time";
        parameter Real ind_mtl_emiss_fact(unit = "1/ton") = 0.1 "Industrial materials emission factor";
        parameter Real ind_mtl_toxic_index = 10.0 "Industrial materials toxicity index";
        parameter Real ind_out_pc_des(unit = "dollar/yr") = 350 "Desired annual industrial per capita output";
        parameter Real ind_out_in_1970(unit = "dollar/yr") = 790000000000.0 "Industrial output in 1970";
        parameter Real inherent_land_fert(unit = "kg/(hectare.yr)") = 600 "Inherent land fertility";
        parameter Real labor_force_partic = 0.75 "Percentage of participating labor force";
        parameter Real labor_util_fr_del_time(unit = "yr") = 2 "Labor utilization fraction delay time";
        parameter Real land_fr_harvested = 0.7 "Land fraction harvested";
        parameter Real life_expect_norm(unit = "yr") = 28 "Normal life expectancy";
        parameter Real lifet_perc_del(unit = "yr") = 20 "Perceived life-time delay";
        parameter Real max_tot_fert_norm = 12 "Normal maximal total fertility";
        parameter Real p_avg_life_agr_inp_1(unit = "yr") = 2 "Default average life of agricultural input";
        parameter Real p_avg_life_agr_inp_2(unit = "yr") = 2.5 "Controlled average life of agricultural input";
        parameter Real p_avg_life_ind_cap_1(unit = "yr") = 14 "Default average life of industrial capital";
        parameter Real p_avg_life_ind_cap_2(unit = "yr") = 18 "Controlled average life of industrial capital";
        parameter Real p_avg_life_serv_cap_1(unit = "yr") = 20 "Default average life of service sector capital";
        parameter Real p_avg_life_serv_cap_2(unit = "yr") = 25 "Controlled average life of service sector capital";
        parameter Real p_fioa_cons_const_1 = 0.43 "Default fraction of industrial output allocated to consumption";
        parameter Real p_fioa_cons_const_2 = 0.43 "Controlled fraction of industrial output allocated to consumption";
        parameter Real p_ind_cap_out_ratio_1(unit = "yr") = 3 "Default industrial capital output ratio";
        parameter Real p_land_yield_fact_1 = 1 "Default land yield factor";
        parameter Real p_nr_res_use_fact_1 = 1 "Default non-recoverable resource utilization factor";
        parameter Real p_ppoll_gen_fact_1 = 1 "Default persistent pollution generation factor";
        parameter Real p_serv_cap_out_ratio_1(unit = "yr") = 1.0 "Default fraction of service sector output ratio";
        parameter Real p_serv_cap_out_ratio_2(unit = "yr") = 1.0 "Controlled fraction of service sector output ratio";
        parameter Real pot_arable_land_tot(unit = "hectare") = 3200000000.0 "Total potential arable land";
        parameter Real ppoll_in_1970 = 136000000.0 "Persistent pollution in 1970";
        parameter Real ppoll_trans_del(unit = "yr") = 20 "Persistent pollution transmission delay";
        parameter Real processing_loss = 0.1 "Processing loss";
        parameter Real reproductive_lifetime(unit = "yr") = 30.0 "Reproductive life time";
        parameter Real social_adj_del(unit = "yr") = 20 "Social adjustment delay";
        parameter Real social_discount(unit = "1/yr") = 0.07 "Social discount";
        parameter Real subsist_food_pc(unit = "kg/yr") = 230 "Available per capita food";
        parameter Real tech_dev_del_TDD(unit = "yr") = 20 "Technology development time";
        parameter Real urb_ind_land_dev_time(unit = "yr") = 10 "Urban and industrial land development time";
        parameter Real t_air_poll_time(unit = "yr") = 4000 "Air pollution change time";
        parameter Real t_fcaor_time(unit = "yr") = 2002 "Year of capital allocation to resource use efficiency";
        parameter Real t_fert_cont_eff_time(unit = "yr") = 2002 "Year of continued fertility change";
        parameter Real t_ind_equil_time(unit = "yr") = 2002 "Year of industrial equilibrium";
        parameter Real t_land_life_time(unit = "yr") = 4000 "Land life time";
        parameter Real t_policy_year(unit = "yr") = 2002 "Year of policy change";
        parameter Real t_pop_equil_time(unit = "yr") = 4000 "Population equilibrium time";
        parameter Real t_zero_pop_grow_time(unit = "yr") = 2002 "Time to zero population growth";
        parameter Real p_fr_cap_al_obt_res_2[:] = {1,0.1,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05,0.05} "Non-renewable resource fraction remaining";
        parameter Real p_ppoll_tech_chg_mlt[:] = {0,0,0,0} "Persistent pollution technology change multiplier";
        parameter Real p_res_tech_chg_mlt[:] = {0,0,0,0} "Resource technology change multiplier";
        parameter Real p_yield_tech_chg_mlt[:] = {0,0,0,0} "Yield technology change multiplier";
        parameter Real agr_inp_init(unit = "dollar/yr") = 5000000000.0 "Initial agricultural input";
        parameter Real arable_land_init(unit = "hectare") = 900000000.0 "Initial arable land";
        parameter Real industrial_capital_init(unit = "dollar") = 210000000000.0 "Initial industrial investment";
        parameter Real labor_util_fr_del_init = 1 "Initial delayed labor utilization fraction";
        parameter Real land_fertility_init(unit = "kg/(hectare