/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.common.cooperation2.datastore.internal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import java.io.IOException;
import java.time.Duration;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonatype.nexus.common.cooperation2.Config;
import org.sonatype.nexus.common.cooperation2.CooperationException;
import org.sonatype.nexus.common.cooperation2.CooperationKey;

public class CooperatingFuture<T>
extends CompletableFuture<T> {
    protected static final Logger log = LoggerFactory.getLogger(CooperatingFuture.class);
    private static final ThreadLocal<Boolean> callInProgress = new ThreadLocal();
    private final AtomicLong staggerTimeMillis = new AtomicLong(System.currentTimeMillis());
    private final AtomicInteger threadCount = new AtomicInteger(1);
    private final CooperationKey requestKey;
    private final Config config;

    public CooperatingFuture(CooperationKey requestKey, Config config) {
        this.requestKey = Preconditions.checkNotNull(requestKey);
        this.config = Preconditions.checkNotNull(config);
    }

    public T call(Function<Boolean, T> request) throws IOException {
        return this.performCall(request, false);
    }

    public T cooperate(Function<Boolean, T> request) throws IOException {
        this.increaseCooperation();
        try {
            if (CooperatingFuture.isNestedCall()) {
                T t2 = this.waitForCall(request, this.config.minorTimeout(), true);
                return t2;
            }
            T t3 = this.waitForCall(request, this.config.majorTimeout(), false);
            return t3;
        }
        catch (ExecutionException e) {
            log.debug("Cooperative wait failed on {}", (Object)this, (Object)e.getCause());
            Throwables.propagateIfPossible(e.getCause(), IOException.class);
            throw new IOException("Cooperative wait failed on " + this, e.getCause());
        }
        catch (InterruptedException | CancellationException e) {
            log.debug("Cooperative wait cancelled on {}", (Object)this, (Object)e);
            throw new CooperationException("Cooperative wait cancelled on " + this);
        }
        finally {
            this.decreaseCooperation();
        }
    }

    @VisibleForTesting
    public String getRequestKey() {
        return this.requestKey.getKey();
    }

    @VisibleForTesting
    public int getThreadCount() {
        return this.threadCount.get();
    }

    @Override
    public String toString() {
        return String.valueOf(this.requestKey.getKey()) + " (" + this.threadCount.get() + " threads cooperating)";
    }

    protected T performCall(Function<Boolean, T> request, boolean failover) throws IOException {
        boolean nested = CooperatingFuture.isNestedCall();
        try {
            if (!nested) {
                callInProgress.set(Boolean.TRUE);
            }
            log.debug("Requesting {}", (Object)this);
            T value = request.apply(failover);
            log.debug("Completing {}", (Object)this);
            this.complete(value);
            T t2 = value;
            return t2;
        }
        catch (Error | Exception e) {
            log.debug("Completing {} with exception", (Object)this, (Object)e);
            this.completeExceptionally(e);
            throw e;
        }
        finally {
            if (!nested) {
                callInProgress.remove();
            }
        }
    }

    protected T waitForCall(Function<Boolean, T> request, Duration initialTimeout, boolean failover) throws InterruptedException, ExecutionException, IOException {
        if (initialTimeout.isZero() || initialTimeout.isNegative()) {
            log.debug("Attempt cooperative wait on {}", (Object)this);
            return this.get();
        }
        Duration timeout = initialTimeout;
        if (failover) {
            timeout = this.staggerTimeout(timeout);
        }
        try {
            log.debug("Attempt cooperative wait on {} for {}", (Object)this, (Object)timeout);
            return this.get(timeout.toMillis(), TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            log.debug("Cooperative wait timed out on {}", (Object)this, (Object)e);
            if (failover) {
                return this.performCall(request, true);
            }
            throw new CooperationException("Cooperative wait timed out on " + this);
        }
    }

    private static boolean isNestedCall() {
        return Boolean.TRUE.equals(callInProgress.get());
    }

    private void increaseCooperation() {
        int limit = this.config.threadsPerKey();
        this.threadCount.getAndUpdate(count -> {
            if (limit > 0 && count >= limit) {
                log.debug("Thread cooperation maxed for {}", (Object)this);
                throw new CooperationException("Thread cooperation maxed for " + this);
            }
            return count + 1;
        });
    }

    private void decreaseCooperation() {
        this.threadCount.decrementAndGet();
    }

    @VisibleForTesting
    Duration staggerTimeout(Duration gap) {
        long nextTimeMillis;
        long prevTimeMillis;
        long currentTimeMillis = System.currentTimeMillis();
        while (!this.staggerTimeMillis.compareAndSet(prevTimeMillis = this.staggerTimeMillis.get(), nextTimeMillis = Math.max(prevTimeMillis + gap.toMillis(), currentTimeMillis))) {
        }
        return Duration.ofMillis(nextTimeMillis - currentTimeMillis);
    }
}

