-- Create tables
CREATE TABLE IF NOT EXISTS deployment_id
(
    id            INT          NOT NULL CHECK (id = 1),
    deployment_id VARCHAR(200) NOT NULL,

    CONSTRAINT pk_deployment_id PRIMARY KEY (id)
);

CREATE TABLE capability_storage_item
(
    id         UUID         NOT NULL,
    version    INT          NOT NULL,
    type       VARCHAR(100) NOT NULL,
    enabled    BOOLEAN      NOT NULL,
    notes      VARCHAR(400) NULL,
    properties JSON         NOT NULL,
    CONSTRAINT pk_capability_storage_item_id PRIMARY KEY (id)
);

CREATE TABLE ldap_configuration
(
    id         UUID         NOT NULL,
    name       VARCHAR(200) NOT NULL,
    "order"    INT          NOT NULL,
    connection JSON         NOT NULL,
    mapping    JSON         NOT NULL,

    CONSTRAINT pk_ldap_configuration_id PRIMARY KEY (id),
    CONSTRAINT uk_ldap_configuration_name UNIQUE (name)
);

CREATE TABLE saml_configuration (
    id                           INT            NOT NULL CHECK (id = 1),
    entity_id                    TEXT           NOT NULL,
    idp_metadata                 TEXT           NOT NULL,
    mapping                      JSON           NOT NULL,
    validate_response_signature  BOOLEAN        NULL,
    validate_assertion_signature BOOLEAN        NULL,
    key_store_password           VARCHAR(200)   NOT NULL,
    key_store_bytes              BYTEA          NOT NULL,

    CONSTRAINT pk_saml_configuration_id PRIMARY KEY (id)
);

CREATE TABLE saml_user
(
    id         VARCHAR(200) NOT NULL,
    first_name VARCHAR(200) NULL,
    last_name  VARCHAR(200) NULL,
    status     VARCHAR(100) NOT NULL,
    email      VARCHAR(200) NULL,
    roles      JSON         NULL,

    CONSTRAINT pk_saml_user_id PRIMARY KEY (id)
);

CREATE TABLE security_user
(
    id         VARCHAR(200) NOT NULL,
    first_name VARCHAR(200) NULL,
    last_name  VARCHAR(200) NULL,
    password   VARCHAR(200) NOT NULL,
    status     VARCHAR(100) NOT NULL,
    email      VARCHAR(200) NOT NULL,
    version    INT          NOT NULL DEFAULT 1,
    CONSTRAINT pk_security_user_id PRIMARY KEY (id)
);

CREATE TABLE repository_routing_rule
(
    id          UUID         NOT NULL,
    name        VARCHAR(200) NOT NULL,
    mode        VARCHAR(100) NOT NULL,
    description VARCHAR(400) NULL,
    matchers    JSON         NOT NULL,

    CONSTRAINT pk_repository_routing_rule_id PRIMARY KEY (id),
    CONSTRAINT pk_repository_routing_rule_name UNIQUE (name)
);

CREATE TABLE repository
(
    id              UUID         NOT NULL,
    name            VARCHAR(200) NOT NULL,
    recipe_name     VARCHAR(200) NOT NULL,
    online          BOOLEAN      NOT NULL,
    routing_rule_id UUID         NULL,
    attributes      JSON         NOT NULL,

    CONSTRAINT pk_repository_id PRIMARY KEY (id),
    CONSTRAINT uk_repository_name UNIQUE (name)
);

CREATE TABLE blob_store_configuration
(
    id         UUID         NOT NULL,
    name       VARCHAR(256) NOT NULL,
    type       VARCHAR(100) NOT NULL,
    attributes JSON         NOT NULL,

    CONSTRAINT pk_blob_store_configuration_id PRIMARY KEY (id),
    CONSTRAINT uk_blob_store_configuration_name UNIQUE (name)
);

CREATE TABLE qrtz_job_details
(
    sched_name        VARCHAR(120) NOT NULL,
    job_name          VARCHAR(200) NOT NULL,
    job_group         VARCHAR(200) NOT NULL,
    description       TEXT NULL,
    job_class_name    VARCHAR(250) NOT NULL,
    is_durable        BOOL         NOT NULL,
    is_nonconcurrent  BOOL         NOT NULL,
    is_update_data    BOOL         NOT NULL,
    requests_recovery BOOL         NOT NULL,
    job_data          BYTEA        NULL,
    CONSTRAINT pk_qrtz_job_details PRIMARY KEY (sched_name, job_name, job_group)
);

CREATE TABLE qrtz_triggers
(
    sched_name     VARCHAR(120) NOT NULL,
    trigger_name   VARCHAR(200) NOT NULL,
    trigger_group  VARCHAR(200) NOT NULL,
    job_name       VARCHAR(200) NOT NULL,
    job_group      VARCHAR(200) NOT NULL,
    description    TEXT NULL,
    next_fire_time BIGINT       NULL,
    prev_fire_time BIGINT       NULL,
    priority       INTEGER      NULL,
    trigger_state  VARCHAR(16)  NOT NULL,
    trigger_type   VARCHAR(8)   NOT NULL,
    start_time     BIGINT       NOT NULL,
    end_time       BIGINT       NULL,
    calendar_name  VARCHAR(200) NULL,
    misfire_instr  SMALLINT     NULL,
    job_data       BYTEA        NULL,
    CONSTRAINT pk_qrtz_triggers PRIMARY KEY (sched_name, trigger_name, trigger_group),
    CONSTRAINT fk_qrtz_triggers_qrtz_job_details FOREIGN KEY (sched_name, job_name, job_group)
        REFERENCES qrtz_job_details (sched_name, job_name, job_group)
);

CREATE TABLE qrtz_simple_triggers
(
    sched_name      VARCHAR(120) NOT NULL,
    trigger_name    VARCHAR(200) NOT NULL,
    trigger_group   VARCHAR(200) NOT NULL,
    repeat_count    BIGINT       NOT NULL,
    repeat_interval BIGINT       NOT NULL,
    times_triggered BIGINT       NOT NULL,
    CONSTRAINT pk_qrtz_simple_triggers PRIMARY KEY (sched_name, trigger_name, trigger_group),
    CONSTRAINT fk_qrtz_simple_triggers_qrtz_triggers FOREIGN KEY (sched_name, trigger_name, trigger_group)
        REFERENCES qrtz_triggers (sched_name, trigger_name, trigger_group)
        ON DELETE CASCADE
);

CREATE TABLE qrtz_cron_triggers
(
    sched_name      VARCHAR(120) NOT NULL,
    trigger_name    VARCHAR(200) NOT NULL,
    trigger_group   VARCHAR(200) NOT NULL,
    cron_expression VARCHAR(120) NOT NULL,
    time_zone_id    VARCHAR(80),
    CONSTRAINT pk_qrtz_cron_triggers PRIMARY KEY (sched_name, trigger_name, trigger_group),
    CONSTRAINT fk_qrtz_cron_triggers_qrtz_triggers FOREIGN KEY (sched_name, trigger_name, trigger_group)
        REFERENCES qrtz_triggers (sched_name, trigger_name, trigger_group)
        ON DELETE CASCADE
);

CREATE TABLE qrtz_simprop_triggers
(
    sched_name    VARCHAR(120)   NOT NULL,
    trigger_name  VARCHAR(200)   NOT NULL,
    trigger_group VARCHAR(200)   NOT NULL,
    str_prop_1    VARCHAR(512)   NULL,
    str_prop_2    VARCHAR(512)   NULL,
    str_prop_3    VARCHAR(512)   NULL,
    int_prop_1    INT            NULL,
    int_prop_2    INT            NULL,
    long_prop_1   BIGINT         NULL,
    long_prop_2   BIGINT         NULL,
    dec_prop_1    NUMERIC(13, 4) NULL,
    dec_prop_2    NUMERIC(13, 4) NULL,
    bool_prop_1   BOOL           NULL,
    bool_prop_2   BOOL           NULL,
    CONSTRAINT pk_qrtz_simprop_triggers PRIMARY KEY (sched_name, trigger_name, trigger_group),
    CONSTRAINT fk_qrtz_simprop_triggers_qrtz_triggers FOREIGN KEY (sched_name, trigger_name, trigger_group)
        REFERENCES qrtz_triggers (sched_name, trigger_name, trigger_group)
        ON DELETE CASCADE
);

CREATE TABLE qrtz_blob_triggers
(
    sched_name    VARCHAR(120) NOT NULL,
    trigger_name  VARCHAR(200) NOT NULL,
    trigger_group VARCHAR(200) NOT NULL,
    blob_data     BYTEA        NULL,
    CONSTRAINT pk_qrtz_blob_triggers PRIMARY KEY (sched_name, trigger_name, trigger_group),
    CONSTRAINT fk_qrtz_blob_triggers_qrtz_triggers FOREIGN KEY (sched_name, trigger_name, trigger_group)
        REFERENCES qrtz_triggers (sched_name, trigger_name, trigger_group)
        ON DELETE CASCADE
);

CREATE TABLE qrtz_paused_trigger_grps
(
    sched_name    VARCHAR(120) NOT NULL,
    trigger_group VARCHAR(200) NOT NULL,
    CONSTRAINT pk_qrtz_paused_trigger_grps PRIMARY KEY (sched_name, trigger_group)
);

CREATE TABLE qrtz_fired_triggers
(
    sched_name        VARCHAR(120) NOT NULL,
    entry_id          VARCHAR(95)  NOT NULL,
    trigger_name      VARCHAR(200) NOT NULL,
    trigger_group     VARCHAR(200) NOT NULL,
    instance_name     VARCHAR(200) NOT NULL,
    fired_time        BIGINT       NOT NULL,
    sched_time        BIGINT       NOT NULL,
    priority          INTEGER      NOT NULL,
    state             VARCHAR(16)  NOT NULL,
    job_name          VARCHAR(200) NULL,
    job_group         VARCHAR(200) NULL,
    is_nonconcurrent  BOOL         NULL,
    requests_recovery BOOL         NULL,
    CONSTRAINT pk_qrtz_fired_triggers PRIMARY KEY (sched_name, entry_id)
);

CREATE TABLE qrtz_calendars
(
    sched_name    VARCHAR(120) NOT NULL,
    calendar_name VARCHAR(200) NOT NULL,
    calendar      BYTEA        NOT NULL,
    CONSTRAINT pk_qrtz_calendars PRIMARY KEY (sched_name, calendar_name)
);

CREATE TABLE qrtz_scheduler_state
(
    sched_name        VARCHAR(120) NOT NULL,
    instance_name     VARCHAR(200) NOT NULL,
    last_checkin_time BIGINT       NOT NULL,
    checkin_interval  BIGINT       NOT NULL,
    CONSTRAINT pk_qrtz_scheduler_state PRIMARY KEY (sched_name, instance_name)
);

CREATE TABLE qrtz_locks
(
    sched_name VARCHAR(120) NOT NULL,
    lock_name  VARCHAR(40)  NOT NULL,
    CONSTRAINT pk_qrtz_locks PRIMARY KEY (sched_name, lock_name)
);

CREATE TABLE realm_configuration
(
    id          INT  NOT NULL CHECK (id = 1),
    realm_names JSON NOT NULL,

    CONSTRAINT pk_realm_configuration_id PRIMARY KEY (id)
);

CREATE TABLE role
(
    id          VARCHAR(200) NOT NULL,
    name        VARCHAR(200) NOT NULL,
    description VARCHAR(400) NULL,
    privileges  JSON         NOT NULL,
    roles       JSON         NOT NULL,
    read_only   BOOLEAN      NOT NULL,
    version     INT          NOT NULL DEFAULT 1,

    CONSTRAINT pk_role_id PRIMARY KEY (id)
);

CREATE TABLE user_role_mapping
(
    user_id VARCHAR(200) NOT NULL,
    user_lo VARCHAR(200) NULL,
    source  VARCHAR(100) NOT NULL,
    roles   JSON         NOT NULL,
    version INT          NOT NULL DEFAULT 1,

    CONSTRAINT pk_user_role_mapping_user_id_source PRIMARY KEY (user_id, source)
);

CREATE TABLE email_configuration
(
    id                            INT          NOT NULL CHECK (id = 1),
    enabled                       BOOLEAN      NOT NULL,
    host                          VARCHAR(200) NULL,
    port                          INT          NOT NULL,
    username                      VARCHAR(200) NULL,
    password                      VARCHAR(200) NULL,
    fromAddress                   VARCHAR(200) NULL,
    subjectPrefix                 VARCHAR(200) NULL,
    startTlsEnabled               BOOLEAN      NOT NULL,
    startTlsRequired              BOOLEAN      NOT NULL,
    sslOnConnectEnabled           BOOLEAN      NOT NULL,
    sslCheckServerIdentityEnabled BOOLEAN      NOT NULL,
    nexusTrustStoreEnabled        BOOLEAN      NOT NULL,

    CONSTRAINT pk_email_configuration_id PRIMARY KEY (id)
);

CREATE TABLE cleanup_policy
(
    name     varchar(200) NOT NULL,
    notes    varchar(400) NULL,
    format   varchar(100) NOT NULL,
    mode     varchar(100) NOT NULL,
    criteria JSON         NOT NULL,

    CONSTRAINT pk_cleanup_policy_name PRIMARY KEY (name)
);

CREATE TABLE firewall_ignore_patterns
(
    id                INT  NOT NULL CHECK (id = 1),
    regexps_by_format JSON NOT NULL,

    CONSTRAINT pk_firewall_ignore_patterns_id PRIMARY KEY (id)
);

CREATE TABLE privilege
(
    id          VARCHAR(200) NOT NULL,
    name        VARCHAR(200) NOT NULL,
    description VARCHAR(400) NULL,
    type        VARCHAR(100) NOT NULL,
    properties  JSON         NOT NULL,
    read_only   BOOLEAN      NOT NULL,
    version     INT          NOT NULL DEFAULT 1,

    CONSTRAINT pk_privilege_id PRIMARY KEY (id)
);

CREATE TABLE http_client_configuration
(
    id             int  NOT NULL CHECK (id = 1),
    connection     JSON NULL,
    proxy          JSON NULL,
    authentication JSON NULL,

    CONSTRAINT pk_http_client_configuration_id PRIMARY KEY (id)
);

CREATE TABLE selector_configuration
(
    name        VARCHAR(200) NOT NULL,
    type        VARCHAR(100) NOT NULL,
    description VARCHAR(400) NULL,
    attributes  JSON         NOT NULL,
    CONSTRAINT pk_selector_configuration PRIMARY KEY (name)
);

CREATE TABLE script
(
    name    varchar(200) NOT NULL,
    type    varchar(100) NOT NULL,
    content text         NOT NULL,
    CONSTRAINT pk_script_name PRIMARY KEY (name)
);

CREATE TABLE anonymous_configuration
(
    id         INT          NOT NULL CHECK (id = 1),
    enabled    BOOLEAN      NOT NULL,
    user_id    VARCHAR(200) NOT NULL,
    realm_name VARCHAR(200) NOT NULL,

    CONSTRAINT pk_anonymous_configuration_id PRIMARY KEY (id)
);

CREATE TABLE key_store_data
(
    name  varchar(200) NOT NULL,
    bytes BYTEA        NOT NULL,

    CONSTRAINT pk_key_store_data_name PRIMARY KEY (name)
);

CREATE TABLE api_key
(
    primary_principal VARCHAR(200) NOT NULL,
    domain            VARCHAR(200) NOT NULL,
    token             VARCHAR(200) NOT NULL,
    principals        BYTEA        NOT NULL,
    created           TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT CURRENT_TIMESTAMP,

    CONSTRAINT pk_api_key_primaryprincipal_domain_principals PRIMARY KEY (primary_principal, domain, principals),
    CONSTRAINT uk_api_key_domain_token UNIQUE (domain, token)
);

CREATE TABLE user_token
(
    user_name  VARCHAR(200) NOT NULL,
    name_code  VARCHAR(200) NOT NULL,
    pass_code  VARCHAR(200) NOT NULL,
    principals BYTEA        NOT NULL,
    created    TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT CURRENT_TIMESTAMP,

    CONSTRAINT pk_user_token_name_code PRIMARY KEY (name_code)
);

CREATE TABLE rhc_configuration
(
    id                 INT          NOT NULL CHECK (id = 1),
    insight_url        VARCHAR(200) NULL,
    use_trust_store    BOOLEAN      NULL,
    scan_rate          INT          NULL,
    http_timeout       INT          NULL,
    repository_enabled JSON         NOT NULL,

    CONSTRAINT pk_rhc_configuration_id PRIMARY KEY (id)
);

CREATE TABLE replication_connection
(
    id                            VARCHAR(200) NOT NULL,
    name                          VARCHAR(200) NOT NULL,
    source_repository_name        VARCHAR(200) NOT NULL,
    destination_instance_url      VARCHAR(200) NOT NULL,
    destination_instance_username VARCHAR(200) NULL,
    destination_instance_password VARCHAR(200) NULL,
    destination_repository_name   VARCHAR(200) NOT NULL,
    content_regexes               JSON NULL,
    include_existing_content      BOOLEAN NULL,

    CONSTRAINT pk_replication_connection_id PRIMARY KEY (id)
);

CREATE TABLE change_blobstore
(
    name                   VARCHAR(200) NOT NULL,
    source_blobstore_name  VARCHAR(200) NOT NULL,
    target_blobstore_name  VARCHAR(200) NOT NULL,
    started                TIMESTAMP WITH TIME ZONE NOT NULL,

    CONSTRAINT pk_change_blobstore PRIMARY KEY (name)
);

-- Create indexes
CREATE INDEX IF NOT EXISTS idx_qrtz_j_req_recovery ON qrtz_job_details (sched_name, requests_recovery);
CREATE INDEX IF NOT EXISTS idx_qrtz_j_grp ON qrtz_job_details (sched_name, job_group);

CREATE INDEX IF NOT EXISTS idx_qrtz_t_j ON qrtz_triggers (sched_name, job_name, job_group);
CREATE INDEX IF NOT EXISTS idx_qrtz_t_jg ON qrtz_triggers (sched_name, job_group);
CREATE INDEX IF NOT EXISTS idx_qrtz_t_c ON qrtz_triggers (sched_name, calendar_name);
CREATE INDEX IF NOT EXISTS idx_qrtz_t_g ON qrtz_triggers (sched_name, trigger_group);
CREATE INDEX IF NOT EXISTS idx_qrtz_t_state ON qrtz_triggers (sched_name, trigger_state);
CREATE INDEX IF NOT EXISTS idx_qrtz_t_n_state ON qrtz_triggers (sched_name, trigger_name, trigger_group, trigger_state);
CREATE INDEX IF NOT EXISTS idx_qrtz_t_n_g_state ON qrtz_triggers (sched_name, trigger_group, trigger_state);
CREATE INDEX IF NOT EXISTS idx_qrtz_t_next_fire_time ON qrtz_triggers (sched_name, next_fire_time);
CREATE INDEX IF NOT EXISTS idx_qrtz_t_nft_st ON qrtz_triggers (sched_name, trigger_state, next_fire_time);
CREATE INDEX IF NOT EXISTS idx_qrtz_t_nft_misfire ON qrtz_triggers (sched_name, misfire_instr, next_fire_time);
CREATE INDEX IF NOT EXISTS idx_qrtz_t_nft_st_misfire ON qrtz_triggers (sched_name, misfire_instr, next_fire_time, trigger_state);
CREATE INDEX IF NOT EXISTS idx_qrtz_t_nft_st_misfire_grp ON qrtz_triggers (sched_name, misfire_instr, next_fire_time,
                                                                           trigger_group, trigger_state);

CREATE INDEX IF NOT EXISTS idx_qrtz_ft_trig_inst_name ON qrtz_fired_triggers (sched_name, instance_name);
CREATE INDEX IF NOT EXISTS idx_qrtz_ft_inst_job_req_rcvry ON qrtz_fired_triggers (sched_name, instance_name, requests_recovery);
CREATE INDEX IF NOT EXISTS idx_qrtz_ft_j_g ON qrtz_fired_triggers (sched_name, job_name, job_group);
CREATE INDEX IF NOT EXISTS idx_qrtz_ft_jg ON qrtz_fired_triggers (sched_name, job_group);
CREATE INDEX IF NOT EXISTS idx_qrtz_ft_t_g ON qrtz_fired_triggers (sched_name, trigger_name, trigger_group);
CREATE INDEX IF NOT EXISTS idx_qrtz_ft_tg ON qrtz_fired_triggers (sched_name, trigger_group);

CREATE INDEX IF NOT EXISTS idx_user_token_user_name ON user_token (user_name);
