within ModelicaByExample.Subsystems.LotkaVolterra.Components;
model TwoSpecies "Lotka-Volterra two species configuration"
  // Import several component models from ModelicaByExample.Components.LotkaVolterra
  import ModelicaByExample.Components.LotkaVolterra.Components.RegionalPopulation;
  import ModelicaByExample.Components.LotkaVolterra.Components.Reproduction;
  import ModelicaByExample.Components.LotkaVolterra.Components.Starvation;
  import ModelicaByExample.Components.LotkaVolterra.Components.Predation;

  parameter Real alpha=0.1 "Birth rate";
  parameter Real gamma=0.4 "Starvation coefficient";
  parameter Real initial_rabbit_population=10 "Initial rabbit population";
  parameter Real initial_fox_population=10 "Initial fox population";
  parameter Real beta=0.02 "Prey (rabbits) consumed";
  parameter Real delta=0.02 "Predators (foxes) fed";

  ModelicaByExample.Components.LotkaVolterra.Interfaces.Species rabbits
    "Population of rabbits in this region"
    annotation (Placement(transformation(extent={{-110,-10},{-90,10}})));
  ModelicaByExample.Components.LotkaVolterra.Interfaces.Species foxes
    "Population of foxes in this region"
    annotation (Placement(transformation(extent={{90,-10},{110,10}})));
protected
  RegionalPopulation rabbit_population(
      initial_population=initial_rabbit_population,
      init=RegionalPopulation.InitializationOptions.FixedPopulation) "Rabbit population"
    annotation (Placement(transformation(extent={{-50,-60},{-30,-40}})));
  Reproduction reproduction(alpha=alpha) "Reproduction of rabbits"
    annotation (Placement(transformation(
        extent={{-10,-10},{10,10}},
        origin={-80,-50})));
  RegionalPopulation fox_population(
      init=RegionalPopulation.InitializationOptions.FixedPopulation,
      initial_population=initial_fox_population)
    annotation (Placement(transformation(extent={{30,-60},{50,-40}})));
  Starvation fox_starvation(gamma=gamma) "Starvation of foxes"
    annotation (Placement(transformation(extent={{70,-60},{90,-40}})));
  Predation fox_predation(beta=beta, delta=delta)
    "Foxes eating rabbits"
    annotation (Placement(transformation(extent={{-10,-30},{10,-10}})));
equation
  connect(reproduction.species, rabbit_population.species)
    annotation (Line(
      points={{-80,-40},{-80,-20},{-40,-20},{-40,-40}},
      color={0,127,0},
      smooth=Smooth.None));
  connect(fox_predation.a, rabbit_population.species)
    annotation (Line(
      points={{-10,-20},{-40,-20},{-40,-40}},
      color={0,127,0},
      smooth=Smooth.None));
  connect(fox_starvation.species, fox_population.species)
    annotation (Line(
      points={{80,-40},{80,-20},{40,-20},{40,-40}},
      color={0,127,0},
      smooth=Smooth.None));
  connect(fox_population.species, fox_predation.b)
    annotation (Line(
      points={{40,-40},{40,-20},{10,-20}},
      color={0,127,0},
      smooth=Smooth.None));
  connect(rabbit_population.species, rabbits)
    annotation (Line(
      points={{-40,-40},{-40,0},{-100,0}},
      color={0,127,0},
      smooth=Smooth.None));
  connect(fox_population.species, foxes)
    annotation (Line(
      points={{40,-40},{40,0},{100,0}},
      color={0,127,0},
      smooth=Smooth.None));
  annotation (Icon(graphics={Bitmap(
          extent={{-100,80},{100,-80}},
          imageSource="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",
          fileName="modelica://ModelicaByExample/../../../../../Desktop/fox-and-rabbit.jpg"),
          Text(
          extent={{-100,120},{100,80}},
          lineColor={0,127,0},
          textString="%name")}));
end TwoSpecies;
