-- Create tables
CREATE TABLE IF NOT EXISTS deployment_id
(
    id            INT          NOT NULL CHECK (id = 1),
    deployment_id VARCHAR(200) NOT NULL,

    CONSTRAINT pk_deployment_id PRIMARY KEY (id)
);

CREATE TABLE capability_storage_item
(
    id         UUID         NOT NULL,
    version    INT          NOT NULL,
    type       VARCHAR(100) NOT NULL,
    enabled    BOOLEAN      NOT NULL,
    notes      VARCHAR(400) NULL,
    properties JSONB        NOT NULL,

    CONSTRAINT pk_capability_storage_item_id PRIMARY KEY (id)
);

CREATE TABLE ldap_configuration
(
    id         UUID         NOT NULL,
    name       VARCHAR(200) NOT NULL,
    "order"    INT          NOT NULL,
    connection JSONB        NOT NULL,
    mapping    JSONB        NOT NULL,

    CONSTRAINT pk_ldap_configuration_id PRIMARY KEY (id),
    CONSTRAINT uk_ldap_configuration_name UNIQUE (name)
);

CREATE TABLE saml_configuration (
    id                           INT            NOT NULL CHECK (id = 1),
    entity_id                    TEXT           NOT NULL,
    idp_metadata                 TEXT           NOT NULL,
    mapping                      JSONB          NOT NULL,
    validate_response_signature  BOOLEAN        NULL,
    validate_assertion_signature BOOLEAN        NULL,
    key_store_password           VARCHAR(200)   NOT NULL,
    key_store_bytes              BYTEA          NOT NULL,

    CONSTRAINT pk_saml_configuration_id PRIMARY KEY (id)
);

CREATE TABLE saml_user
(
    id         VARCHAR(200) NOT NULL,
    first_name VARCHAR(200) NULL,
    last_name  VARCHAR(200) NULL,
    status     VARCHAR(100) NOT NULL,
    email      VARCHAR(200) NULL,
    roles      JSONB        NULL,

    CONSTRAINT pk_saml_user_id PRIMARY KEY (id)
);


CREATE TABLE security_user
(
    id         VARCHAR(200) NOT NULL,
    first_name VARCHAR(200) NULL,
    last_name  VARCHAR(200) NULL,
    password   VARCHAR(200) NOT NULL,
    status     VARCHAR(100) NOT NULL,
    email      VARCHAR(200) NOT NULL,
    version    INT          NOT NULL DEFAULT 1,

    CONSTRAINT pk_security_user_id PRIMARY KEY (id)
);


CREATE TABLE repository_routing_rule
(
    id          UUID         NOT NULL,
    name        VARCHAR(200) NOT NULL,
    mode        VARCHAR(100) NOT NULL,
    description VARCHAR(400) NULL,
    matchers    JSONB        NOT NULL,

    CONSTRAINT pk_repository_routing_rule_id PRIMARY KEY (id),
    CONSTRAINT pk_repository_routing_rule_name UNIQUE (name)
);

CREATE TABLE repository
(
    id              UUID         NOT NULL,
    name            VARCHAR(200) NOT NULL,
    recipe_name     VARCHAR(200) NOT NULL,
    online          BOOLEAN      NOT NULL,
    routing_rule_id UUID         NULL,
    attributes      JSONB        NOT NULL,

    CONSTRAINT pk_repository_id PRIMARY KEY (id),
    CONSTRAINT uk_repository_name UNIQUE (name)
);

CREATE TABLE blob_store_configuration
(
    id         UUID         NOT NULL,
    name       VARCHAR(256) NOT NULL,
    type       VARCHAR(100) NOT NULL,
    attributes JSONB        NOT NULL,

    CONSTRAINT pk_blob_store_configuration_id PRIMARY KEY (id),
    CONSTRAINT uk_blob_store_configuration_name UNIQUE (name)
);

CREATE TABLE qrtz_job_details
(
    SCHED_NAME        VARCHAR(120) NOT NULL,
    JOB_NAME          VARCHAR(200) NOT NULL,
    JOB_GROUP         VARCHAR(200) NOT NULL,
    DESCRIPTION       TEXT NULL,
    JOB_CLASS_NAME    VARCHAR(250) NOT NULL,
    IS_DURABLE        BOOL         NOT NULL,
    IS_NONCONCURRENT  BOOL         NOT NULL,
    IS_UPDATE_DATA    BOOL         NOT NULL,
    REQUESTS_RECOVERY BOOL         NOT NULL,
    JOB_DATA          BYTEA        NULL,
    PRIMARY KEY (SCHED_NAME, JOB_NAME, JOB_GROUP)
);

CREATE TABLE qrtz_triggers
(
    SCHED_NAME     VARCHAR(120) NOT NULL,
    TRIGGER_NAME   VARCHAR(200) NOT NULL,
    TRIGGER_GROUP  VARCHAR(200) NOT NULL,
    JOB_NAME       VARCHAR(200) NOT NULL,
    JOB_GROUP      VARCHAR(200) NOT NULL,
    DESCRIPTION    TEXT NULL,
    NEXT_FIRE_TIME BIGINT       NULL,
    PREV_FIRE_TIME BIGINT       NULL,
    PRIORITY       INTEGER      NULL,
    TRIGGER_STATE  VARCHAR(16)  NOT NULL,
    TRIGGER_TYPE   VARCHAR(8)   NOT NULL,
    START_TIME     BIGINT       NOT NULL,
    END_TIME       BIGINT       NULL,
    CALENDAR_NAME  VARCHAR(200) NULL,
    MISFIRE_INSTR  SMALLINT     NULL,
    JOB_DATA       BYTEA        NULL,
    PRIMARY KEY (SCHED_NAME, TRIGGER_NAME, TRIGGER_GROUP),
    FOREIGN KEY (SCHED_NAME, JOB_NAME, JOB_GROUP)
        REFERENCES QRTZ_JOB_DETAILS (SCHED_NAME, JOB_NAME, JOB_GROUP)
);

CREATE TABLE qrtz_simple_triggers
(
    SCHED_NAME      VARCHAR(120) NOT NULL,
    TRIGGER_NAME    VARCHAR(200) NOT NULL,
    TRIGGER_GROUP   VARCHAR(200) NOT NULL,
    REPEAT_COUNT    BIGINT       NOT NULL,
    REPEAT_INTERVAL BIGINT       NOT NULL,
    TIMES_TRIGGERED BIGINT       NOT NULL,
    PRIMARY KEY (SCHED_NAME, TRIGGER_NAME, TRIGGER_GROUP),
    FOREIGN KEY (SCHED_NAME, TRIGGER_NAME, TRIGGER_GROUP)
        REFERENCES QRTZ_TRIGGERS (SCHED_NAME, TRIGGER_NAME, TRIGGER_GROUP)
);

CREATE TABLE qrtz_cron_triggers
(
    SCHED_NAME      VARCHAR(120) NOT NULL,
    TRIGGER_NAME    VARCHAR(200) NOT NULL,
    TRIGGER_GROUP   VARCHAR(200) NOT NULL,
    CRON_EXPRESSION VARCHAR(120) NOT NULL,
    TIME_ZONE_ID    VARCHAR(80),
    PRIMARY KEY (SCHED_NAME, TRIGGER_NAME, TRIGGER_GROUP),
    FOREIGN KEY (SCHED_NAME, TRIGGER_NAME, TRIGGER_GROUP)
        REFERENCES QRTZ_TRIGGERS (SCHED_NAME, TRIGGER_NAME, TRIGGER_GROUP)
);

CREATE TABLE qrtz_simprop_triggers
(
    SCHED_NAME    VARCHAR(120)   NOT NULL,
    TRIGGER_NAME  VARCHAR(200)   NOT NULL,
    TRIGGER_GROUP VARCHAR(200)   NOT NULL,
    STR_PROP_1    VARCHAR(512)   NULL,
    STR_PROP_2    VARCHAR(512)   NULL,
    STR_PROP_3    VARCHAR(512)   NULL,
    INT_PROP_1    INT            NULL,
    INT_PROP_2    INT            NULL,
    LONG_PROP_1   BIGINT         NULL,
    LONG_PROP_2   BIGINT         NULL,
    DEC_PROP_1    NUMERIC(13, 4) NULL,
    DEC_PROP_2    NUMERIC(13, 4) NULL,
    BOOL_PROP_1   BOOL           NULL,
    BOOL_PROP_2   BOOL           NULL,
    PRIMARY KEY (SCHED_NAME, TRIGGER_NAME, TRIGGER_GROUP),
    FOREIGN KEY (SCHED_NAME, TRIGGER_NAME, TRIGGER_GROUP)
        REFERENCES QRTZ_TRIGGERS (SCHED_NAME, TRIGGER_NAME, TRIGGER_GROUP)
);

CREATE TABLE qrtz_blob_triggers
(
    SCHED_NAME    VARCHAR(120) NOT NULL,
    TRIGGER_NAME  VARCHAR(200) NOT NULL,
    TRIGGER_GROUP VARCHAR(200) NOT NULL,
    BLOB_DATA     BYTEA        NULL,
    PRIMARY KEY (SCHED_NAME, TRIGGER_NAME, TRIGGER_GROUP),
    FOREIGN KEY (SCHED_NAME, TRIGGER_NAME, TRIGGER_GROUP)
        REFERENCES QRTZ_TRIGGERS (SCHED_NAME, TRIGGER_NAME, TRIGGER_GROUP)
);

CREATE TABLE qrtz_calendars
(
    SCHED_NAME    VARCHAR(120) NOT NULL,
    CALENDAR_NAME VARCHAR(200) NOT NULL,
    CALENDAR      BYTEA        NOT NULL,
    PRIMARY KEY (SCHED_NAME, CALENDAR_NAME)
);


CREATE TABLE qrtz_paused_trigger_grps
(
    SCHED_NAME    VARCHAR(120) NOT NULL,
    TRIGGER_GROUP VARCHAR(200) NOT NULL,
    PRIMARY KEY (SCHED_NAME, TRIGGER_GROUP)
);

CREATE TABLE qrtz_fired_triggers
(
    SCHED_NAME        VARCHAR(120) NOT NULL,
    ENTRY_ID          VARCHAR(95)  NOT NULL,
    TRIGGER_NAME      VARCHAR(200) NOT NULL,
    TRIGGER_GROUP     VARCHAR(200) NOT NULL,
    INSTANCE_NAME     VARCHAR(200) NOT NULL,
    FIRED_TIME        BIGINT       NOT NULL,
    SCHED_TIME        BIGINT       NOT NULL,
    PRIORITY          INTEGER      NOT NULL,
    STATE             VARCHAR(16)  NOT NULL,
    JOB_NAME          VARCHAR(200) NULL,
    JOB_GROUP         VARCHAR(200) NULL,
    IS_NONCONCURRENT  BOOL         NULL,
    REQUESTS_RECOVERY BOOL         NULL,
    PRIMARY KEY (SCHED_NAME, ENTRY_ID)
);

CREATE TABLE qrtz_scheduler_state
(
    SCHED_NAME        VARCHAR(120) NOT NULL,
    INSTANCE_NAME     VARCHAR(200) NOT NULL,
    LAST_CHECKIN_TIME BIGINT       NOT NULL,
    CHECKIN_INTERVAL  BIGINT       NOT NULL,
    PRIMARY KEY (SCHED_NAME, INSTANCE_NAME)
);

CREATE TABLE qrtz_locks
(
    SCHED_NAME VARCHAR(120) NOT NULL,
    LOCK_NAME  VARCHAR(40)  NOT NULL,
    PRIMARY KEY (SCHED_NAME, LOCK_NAME)
);

CREATE TABLE realm_configuration
(
    id          INT   NOT NULL CHECK (id = 1),
    realm_names JSONB NOT NULL,

    CONSTRAINT pk_realm_configuration_id PRIMARY KEY (id)
);

CREATE TABLE role
(
    id          VARCHAR(200) NOT NULL,
    name        VARCHAR(200) NOT NULL,
    description VARCHAR(400) NULL,
    privileges  JSONB        NOT NULL,
    roles       JSONB        NOT NULL,
    read_only   BOOLEAN      NOT NULL,
    version     INT          NOT NULL DEFAULT 1,

    CONSTRAINT pk_role_id PRIMARY KEY (id)
);

CREATE TABLE user_role_mapping
(
    user_id VARCHAR(200) NOT NULL,
    user_lo VARCHAR(200) NULL,
    source  VARCHAR(100) NOT NULL,
    roles   JSONB        NOT NULL,
    version INT          NOT NULL DEFAULT 1,
    CONSTRAINT pk_user_role_mapping_user_id_source PRIMARY KEY (user_id, source)
);

CREATE TABLE email_configuration
(
    id                            INT          NOT NULL CHECK (id = 1),
    enabled                       BOOLEAN      NOT NULL,
    host                          VARCHAR(200) NULL,
    port                          INT          NOT NULL,
    username                      VARCHAR(200) NULL,
    password                      VARCHAR(200) NULL,
    fromAddress                   VARCHAR(200) NULL,
    subjectPrefix                 VARCHAR(200) NULL,
    startTlsEnabled               BOOLEAN      NOT NULL,
    startTlsRequired              BOOLEAN      NOT NULL,
    sslOnConnectEnabled           BOOLEAN      NOT NULL,
    sslCheckServerIdentityEnabled BOOLEAN      NOT NULL,
    nexusTrustStoreEnabled        BOOLEAN      NOT NULL,

    CONSTRAINT pk_email_configuration_id PRIMARY KEY (id)
);

CREATE TABLE cleanup_policy
(
    name     varchar(200) NOT NULL,
    notes    varchar(400) NULL,
    format   varchar(100) NOT NULL,
    mode     varchar(100) NOT NULL,
    criteria JSONB        NOT NULL,

    CONSTRAINT pk_cleanup_policy_name PRIMARY KEY (name)
);

CREATE TABLE firewall_ignore_patterns
(
    id                INT   NOT NULL CHECK (id = 1),
    regexps_by_format JSONB NOT NULL,

    CONSTRAINT pk_firewall_ignore_patterns_id PRIMARY KEY (id)
);

CREATE TABLE privilege
(
    id          VARCHAR(200) NOT NULL,
    name        VARCHAR(200) NOT NULL,
    description VARCHAR(400) NULL,
    type        VARCHAR(100) NOT NULL,
    properties  JSONB        NOT NULL,
    read_only   BOOLEAN      NOT NULL,
    version     INT          NOT NULL DEFAULT 1,
    CONSTRAINT pk_privilege_id PRIMARY KEY (id)
);

CREATE TABLE http_client_configuration
(
    id             int   NOT NULL CHECK (id = 1),
    connection     JSONB NULL,
    proxy          JSONB NULL,
    authentication JSONB NULL,

    CONSTRAINT pk_http_client_configuration_id PRIMARY KEY (id)
);

CREATE TABLE selector_configuration
(
    name        VARCHAR(200) NOT NULL,
    type        VARCHAR(100) NOT NULL,
    description VARCHAR(400) NULL,
    attributes  JSONB        NOT NULL,

    CONSTRAINT pk_selector_configuration_name PRIMARY KEY (name)
);

CREATE TABLE script
(
    name    varchar(200) NOT NULL,
    type    varchar(100) NOT NULL,
    content text         NOT NULL,
    CONSTRAINT pk_script_name PRIMARY KEY (name)
);

CREATE TABLE anonymous_configuration
(
    id         INT          NOT NULL CHECK (id = 1),
    enabled    BOOLEAN      NOT NULL,
    user_id    VARCHAR(200) NOT NULL,
    realm_name VARCHAR(200) NOT NULL,

    CONSTRAINT pk_anonymous_configuration_id PRIMARY KEY (id)
);

CREATE TABLE key_store_data
(
    name  varchar(200) NOT NULL,
    bytes BYTEA        NOT NULL,

    CONSTRAINT pk_key_store_data_name PRIMARY KEY (name)
);

CREATE TABLE api_key
(
    primary_principal VARCHAR(200) NOT NULL,
    domain            VARCHAR(200) NOT NULL,
    token             VARCHAR(200) NOT NULL,
    principals        BYTEA        NOT NULL,
    created           TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT CURRENT_TIMESTAMP,

    CONSTRAINT pk_api_key_primaryprincipal_domain_principals PRIMARY KEY (primary_principal, domain, principals),
    CONSTRAINT uk_api_key_domain_token UNIQUE (domain, token)
);

CREATE TABLE user_token
(
    user_name  VARCHAR(200) NOT NULL,
    name_code  VARCHAR(200) NOT NULL,
    pass_code  VARCHAR(200) NOT NULL,
    principals BYTEA        NOT NULL,
    created    TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT CURRENT_TIMESTAMP,

    CONSTRAINT pk_user_token_name_code PRIMARY KEY (name_code)
);

CREATE TABLE rhc_configuration
(
    id                 INT          NOT NULL CHECK (id = 1),
    insight_url        VARCHAR(200) NULL,
    use_trust_store    BOOLEAN      NULL,
    scan_rate          INT          NULL,
    http_timeout       INT          NULL,
    repository_enabled JSONB        NOT NULL,

    CONSTRAINT pk_rhc_configuration_id PRIMARY KEY (id)
);

CREATE TABLE replication_connection
(
    id                            VARCHAR(200) NOT NULL,
    name                          VARCHAR(200) NOT NULL,
    source_repository_name        VARCHAR(200) NOT NULL,
    destination_instance_url      VARCHAR(200) NOT NULL,
    destination_instance_username VARCHAR(200) NULL,
    destination_instance_password VARCHAR(200) NULL,
    destination_repository_name   VARCHAR(200) NOT NULL,
    content_regexes               JSONB NULL,
    include_existing_content      BOOLEAN NULL,

    CONSTRAINT pk_replication_connection_id PRIMARY KEY (id)
);

CREATE TABLE change_blobstore
(
    name                   VARCHAR(200) NOT NULL,
    source_blobstore_name  VARCHAR(200) NOT NULL,
    target_blobstore_name  VARCHAR(200) NOT NULL,
    started                TIMESTAMP WITH TIME ZONE NOT NULL,

    CONSTRAINT pk_change_blobstore PRIMARY KEY (name)
);

-- create index if not exists
create index if not exists idx_qrtz_j_req_recovery on qrtz_job_details (SCHED_NAME, REQUESTS_RECOVERY);
create index if not exists idx_qrtz_j_grp on qrtz_job_details (SCHED_NAME, JOB_GROUP);

create index if not exists idx_qrtz_t_j on qrtz_triggers (SCHED_NAME, JOB_NAME, JOB_GROUP);
create index if not exists idx_qrtz_t_jg on qrtz_triggers (SCHED_NAME, JOB_GROUP);
create index if not exists idx_qrtz_t_c on qrtz_triggers (SCHED_NAME, CALENDAR_NAME);
create index if not exists idx_qrtz_t_g on qrtz_triggers (SCHED_NAME, TRIGGER_GROUP);
create index if not exists idx_qrtz_t_state on qrtz_triggers (SCHED_NAME, TRIGGER_STATE);
create index if not exists idx_qrtz_t_n_state on qrtz_triggers (SCHED_NAME, TRIGGER_NAME, TRIGGER_GROUP, TRIGGER_STATE);
create index if not exists idx_qrtz_t_n_g_state on qrtz_triggers (SCHED_NAME, TRIGGER_GROUP, TRIGGER_STATE);
create index if not exists idx_qrtz_t_next_fire_time on qrtz_triggers (SCHED_NAME, NEXT_FIRE_TIME);
create index if not exists idx_qrtz_t_nft_st on qrtz_triggers (SCHED_NAME, TRIGGER_STATE, NEXT_FIRE_TIME);
create index if not exists idx_qrtz_t_nft_misfire on qrtz_triggers (SCHED_NAME, MISFIRE_INSTR, NEXT_FIRE_TIME);
create index if not exists idx_qrtz_t_nft_st_misfire on qrtz_triggers (SCHED_NAME, MISFIRE_INSTR, NEXT_FIRE_TIME, TRIGGER_STATE);
create index if not exists idx_qrtz_t_nft_st_misfire_grp on qrtz_triggers (SCHED_NAME, MISFIRE_INSTR, NEXT_FIRE_TIME, TRIGGER_GROUP,
                                                             TRIGGER_STATE);

create index if not exists idx_qrtz_ft_trig_inst_name on qrtz_fired_triggers (SCHED_NAME, INSTANCE_NAME);
create index if not exists idx_qrtz_ft_inst_job_req_rcvry on qrtz_fired_triggers (SCHED_NAME, INSTANCE_NAME, REQUESTS_RECOVERY);
create index if not exists idx_qrtz_ft_j_g on qrtz_fired_triggers (SCHED_NAME, JOB_NAME, JOB_GROUP);
create index if not exists idx_qrtz_ft_jg on qrtz_fired_triggers (SCHED_NAME, JOB_GROUP);
create index if not exists idx_qrtz_ft_t_g on qrtz_fired_triggers (SCHED_NAME, TRIGGER_NAME, TRIGGER_GROUP);
create index if not exists idx_qrtz_ft_tg on qrtz_fired_triggers (SCHED_NAME, TRIGGER_GROUP);

CREATE INDEX IF NOT EXISTS idx_user_token_user_name ON user_token (user_name);

commit;
