﻿within ;
package Nuclear "nuclear reactor simulations"

  model SingleFluidMoltenSaltReactor
    "molten salt reactor circuit with fuel and coolant as single fluid"
      // (c) 2016 Dr.Sdl
      // this model represents a simple molten salt reactor in which fuel salt and coolant salt
      // are identical. The "reactor" consists of a single pipe 3.9m in height and with 16 axial nodes.
      // The circuit is made up of a hotleg and a coldleg pipe and of a BasicHX component as heat exchanger
      // in between. The reactor power is generated by a point kinetic model which is connected as
      // a single heat source at the bottom of the "reactor pipe". In more advanced models you would
      // split the reactor power into 16 axial sections and connect it to the reactor pipe according
      // to the axial power profile. A heat capacity is also connected to the reactor pipe to
      // mimic the heat structure of the solid reactor structure, for example graphite.
      //
      // In order to reach the correct, initial equilibrium condition a number of artificial
      // ramps, valves and feedbacks have been introduced. First of all the is a valve in the
      // primary circuit which only fully opens at 20s. Before that point in time there is only
      // minimal flow and now reactor power and this ensures that the whole system can be
      // easily initalized by the Modelica compiler.
      //
      // The reactor is forced sub-critical at 0s by means of block "rho0". With the help of
      // block "ramp" the reactivity is very slowly increased to +0.2pm at 120s.
      //
      // The reactor temperature feedback is modeled as a single feedback loop and consists
      // of both Doppler feedback and density feedback. In more advanced models you would need to
      // separate the two contributions. There is an artificial ramp "Tfuel_reference" in order
      // to introduce an artifical reactivity contribution until the primary circuit salt
      // temperature has reached its predefined setpoint. In practice this "feedback" would
      // be achieved by removing some control rods of the reactor to successively increase
      // the power. This feedback starts at 100s and lasts for 300s. At the end of the ramp
      // the reactor power decreases fast towards its subsequent equilibrium value.
      //
      // When the reactor power increases also the heat transfer in the steam generator needs
      // to increase in order to remove the heat. The is acomplished by block "HEXT_ramp"
      // which decreases the steam generator inlet temperature from 610C to 410C at 20s until
      // 100s. This needs to be carefully timed in accordance with the fuel salt temperature setpoint
      // in order to not start a subcooled salt transient.

      replaceable package Medium =
         Nuclear.FuelSaltLiFBeF2 (final
            singleState=true, final enthalpyOfT=true);

     inner Modelica.Fluid.System system(
        m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
      T_ambient=923.15,
      m_flow_start=50,
      T_start=923.15)     annotation (Placement(transformation(extent={{-142,120},
              {-122,140}},rotation=0)));

    Modelica.Fluid.Vessels.OpenTank tank(
      redeclare package Medium = Medium,
      nPorts=2,
      massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
      use_HeatTransfer=true,
      redeclare model HeatTransfer =
          Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
      ports(each p(start=110000)),
      height=4,
      portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
          Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
      T_start=Modelica.SIunits.Conversions.from_degC(650),
      level_start=0.5,
      crossArea=3.00,
      m_flow_nominal=0.005)
                annotation (Placement(transformation(extent={{-120,26},{-100,46}},
            rotation=0)));
    Modelica.Fluid.Machines.ControlledPump pump(
      redeclare package Medium = Medium,
      N_nominal=1500,
      use_T_start=true,
      control_m_flow=false,
      allowFlowReversal=false,
      use_p_set=true,
      T_start=Modelica.SIunits.Conversions.from_degC(650),
      m_flow_start=system.m_flow_start,
      m_flow_nominal=system.m_flow_start,
      p_a_start=110000,
      p_b_start=130000,
      p_a_nominal=110000,
      p_b_nominal=130000)
      annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
    Modelica.Fluid.Valves.ValveIncompressible valve(
      redeclare package Medium = Medium,
      CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
      show_T=true,
      allowFlowReversal=false,
      m_flow_nominal=system.m_flow_start,
      dp_start=18000,
      dp_nominal=10000)
      annotation (Placement(transformation(extent={{102,-82},{82,-62}},
                                                                     rotation=0)));

    Modelica.Fluid.Pipes.DynamicPipe reactor(
      redeclare package Medium = Medium,
      use_T_start=true,
      redeclare model HeatTransfer =
          Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
      redeclare model FlowModel =
          Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
      use_HeatTransfer=true,
      modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
      nNodes=16,
      diameter=0.2,
      height_ab=0,
      length=3.9,
      T_start=Modelica.SIunits.Conversions.from_degC(650),
      p_a_start=130000) annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                                                                                       rotation=90,
          origin={50,82})));

    Modelica.Fluid.Pipes.DynamicPipe hotleg(
      redeclare package Medium = Medium,
      use_T_start=true,
      redeclare model HeatTransfer =
          Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
      redeclare model FlowModel =
          Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
      diameter=0.2,
      T_start=Modelica.SIunits.Conversions.from_degC(650),
      length=10.01,
      p_a_start=130000) annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=-90,
          origin={122,12})));

    Modelica.Fluid.Pipes.DynamicPipe coldleg(
      redeclare package Medium = Medium,
      use_T_start=true,
      redeclare model HeatTransfer =
          Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
      redeclare model FlowModel =
          Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
      diameter=0.2,
      nNodes=1,
      modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
      T_start=Modelica.SIunits.Conversions.from_degC(650),
      length=10.01,
      p_a_start=110000) annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=90,
          origin={-112,-20})));

    Modelica.Fluid.Examples.HeatExchanger.BaseClasses.BasicHX HEX(
      c_wall=500,
      s_wall=0.005,
      rho_wall=900,
      redeclare package Medium_1 =
          Medium,
      redeclare package Medium_2 =
          Medium,
      modelStructure_1=Modelica.Fluid.Types.ModelStructure.av_b,
      modelStructure_2=Modelica.Fluid.Types.ModelStructure.a_vb,
      crossArea_1=3.14e-2,
      perimeter_1=0.628,
      T_start_1=Modelica.SIunits.Conversions.from_degC(610),
      T_start_2=Modelica.SIunits.Conversions.from_degC(650),
      k_wall=100,
      crossArea_2=3.14e-2,
      perimeter_2=0.628,
      nNodes=2,
      use_T_start=false,
      Twall_start=Modelica.SIunits.Conversions.from_degC(625),
      m_flow_start_2=system.m_flow_start,
      redeclare model HeatTransfer_1 =
          Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
          (alpha0=2000),
           redeclare model HeatTransfer_2 =
          Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
          (alpha0=2000),
      m_flow_start_1=150,
      length=11.0003,
      area_h_1=0.628*11.0003,
      area_h_2=0.628*11.0003,
      dT=1,
      p_a_start1=100000,
      p_b_start1=100000,
      p_a_start2=117000,
      p_b_start2=117000)   annotation (Placement(transformation(extent={{-18,-86},
              {12,-56}}, rotation=0)));

    Modelica.Fluid.Sources.Boundary_pT HEXambient(
      nPorts=1,
      redeclare package Medium = Medium,
      T=Modelica.SIunits.Conversions.from_degC(610),
      p=100000) annotation (Placement(transformation(
          extent={{7,-7},{-7,7}},
          rotation=0,
          origin={46,-56})));

    Modelica.Fluid.Sources.MassFlowSource_T HEXmassFlowRate(
      nPorts=1,
      redeclare package Medium = Medium,
      use_X_in=false,
      use_m_flow_in=false,
      T=Modelica.SIunits.Conversions.from_degC(610),
      m_flow=150,
      use_T_in=true)
                annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=0,
          origin={-38,-86})));

  public
    Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow(redeclare package Medium
        =                                                                        Medium)
      annotation (Placement(transformation(extent={{-12,10},{8,30}}, rotation=0)));
    Modelica.Fluid.Sensors.Temperature sensor_T_hotleg(redeclare package Medium
        =                                                                         Medium)
      annotation (Placement(transformation(extent={{40,108},{60,128}},
                                                                     rotation=0)));
    Modelica.Fluid.Sensors.Temperature sensor_T_coldleg(redeclare package
        Medium =                                                                   Medium)
      annotation (Placement(transformation(extent={{-42,-56},{-62,-36}}, rotation=0)));
    Modelica.Blocks.Sources.Step handle(
      height=0.9,
      startTime=20,
      offset=0.1)   annotation (Placement(transformation(extent={{74,-29},{88,-15}},
                    rotation=0)));

  protected
    Modelica.Blocks.Interfaces.RealOutput T_hotleg
      annotation (Placement(transformation(extent={{80,112},{92,124}}, rotation=0)));
    Modelica.Blocks.Interfaces.RealOutput T_coldleg
      annotation (Placement(transformation(extent={{-66,-52},{-78,-40}}, rotation=0)));
    Modelica.Blocks.Interfaces.RealOutput tankLevel
                                   annotation (Placement(transformation(extent={{-94,26},{-82,38}},
                                  rotation=0)));
    Modelica.Blocks.Interfaces.RealOutput m_flow
      annotation (Placement(transformation(extent={{-4,40},{8,52}},   rotation=
              0)));
    Modelica.Blocks.Interfaces.RealOutput r_power
    annotation (Placement(transformation(extent={{20,60},{30,70}})));

  public
    Modelica.Blocks.Sources.Ramp PumpRamp(
      duration=100,
      offset=130000,
      startTime=9,
      height=145000)
                   annotation (Placement(transformation(extent={{-80,54},{-60,74}})));

    Modelica.Thermal.HeatTransfer.Components.HeatCapacitor ReactorHeatStructure(C=100000, T(start=923.15))
      "heatCapacitor"
      annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=90,
          origin={-6,124})));

    PointKineticMoltenSaltReactor_P1 pointKineticMoltenSaltReactor annotation (Placement(transformation(extent={{-16,84},{4,104}})));

    Modelica.Blocks.Sources.Ramp ramp(
      startTime=20,
      duration=100,
      height=2.2)  annotation (Placement(transformation(extent={{-104,94},{-84,114}})));
    Modelica.Blocks.Sources.Constant rho0(k=-2)   annotation (Placement(transformation(extent={{-104,124},{-84,144}})));
    Modelica.Blocks.Math.Add add1
      annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=180,
          origin={-12,178})));
    Modelica.Blocks.Math.Gain Tfuel_rho_feedback(k=-9.5)
      "reactivity feedback in pcm/K"
      annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=180,
          origin={-46,178})));
    Modelica.Blocks.Math.Add3 add3_1 annotation (Placement(transformation(extent={{-64,88},{-44,108}})));
    Modelica.Blocks.Sources.Ramp Tfuel_reference(
      offset=-890,
      duration=300,
      startTime=100,
      height=-11)    annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=180,
          origin={30,162})));
    Modelica.Blocks.Sources.Ramp HEXT_ramp(
      duration=100,
      startTime=20,
      offset=883,
      height=-200) annotation (Placement(transformation(extent={{-140,-142},{-120,-122}})));
    Modelica.Blocks.Math.Add add annotation (Placement(transformation(extent={{-80,-136},{-60,-116}})));
    Modelica.Blocks.Sources.Ramp subcool(
      duration=10,
      startTime=700,
      height=0) "subcool transient ramp" annotation (Placement(transformation(extent={{-140,-102},{-120,-82}})));
    Modelica.Blocks.Sources.Pulse pulse(
      width=95,
      period=5,
      nperiod=1,
      startTime=700,
      amplitude=40) "reactivity burst pulse" annotation (Placement(transformation(extent={{-140,58},{-120,78}})));
  equation
  tankLevel = tank.level;
    connect(sensor_m_flow.m_flow, m_flow)         annotation (Line(points={{-2,31},{-2,46},{2,46}},
                                                  color={0,0,127}));
    connect(sensor_m_flow.port_b, reactor.port_a)
      annotation (Line(points={{8,20},{8,20},{50,20},{50,72}},
                                                       color={0,127,255}));
    connect(sensor_T_hotleg.T, T_hotleg)
      annotation (Line(points={{57,118},{82,118},{86,118}}, color={0,0,127}));
    connect(handle.y, valve.opening)       annotation (Line(
        points={{88.7,-22},{92,-22},{92,-64}},
        color={0,0,127},
        smooth=Smooth.None));
    connect(pump.port_b, sensor_m_flow.port_a)
                                              annotation (Line(
        points={{-30,20},{-12,20}},
        color={0,127,255},
        smooth=Smooth.None));
    connect(sensor_T_coldleg.T, T_coldleg) annotation (Line(
        points={{-59,-46},{-72,-46}},
        color={0,0,127},
        smooth=Smooth.None));
    connect(sensor_T_hotleg.port, reactor.port_b) annotation (Line(
        points={{50,108},{50,92}},
        color={0,127,255},
        smooth=Smooth.None));
    connect(reactor.port_b, hotleg.port_a) annotation (Line(
        points={{50,92},{50,104},{122,104},{122,22}},
        color={0,127,255},
        smooth=Smooth.None));
    connect(hotleg.port_b, valve.port_a) annotation (Line(
        points={{122,2},{122,-72},{102,-72}},
        color={0,127,255},
        smooth=Smooth.None));
    connect(tank.ports[1], pump.port_a)
      annotation (Line(points={{-112,26},{-112,20},{-50,20}},        color={0,127,255}));

    connect(coldleg.port_b, tank.ports[2])
      annotation (Line(points={{-112,-10},{-112,26},{-108,26}},
                                                            color={0,127,255}));

    // connect(radiator.port_b, tank.ports[2]);
    connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
            {-35,28.2}}, color={0,0,127}));
    connect(valve.port_b, HEX.port_a2) annotation (Line(points={{82,-72},{58,-72},
            {58,-77.9},{13.5,-77.9}}, color={0,127,255}));
    connect(HEX.port_b2, coldleg.port_a) annotation (Line(points={{-19.5,-64.1},{-112,
            -64.1},{-112,-30}},      color={0,127,255}));
    connect(sensor_T_coldleg.port, coldleg.port_a) annotation (Line(points={{-52,-56},
            {-52,-64},{-112,-64},{-112,-30}},        color={0,127,255}));
    connect(HEXmassFlowRate.ports[1], HEX.port_a1) annotation (Line(points={{-28,-86},{-22,-86},{-22,-71.3},{-19.5,-71.3}},
                                                       color={0,127,255}));
    connect(HEX.port_b1, HEXambient.ports[1]) annotation (Line(points={{13.5,-71.3},
            {13.5,-63.65},{39,-63.65},{39,-56}},        color={0,127,255}));

    connect(ReactorHeatStructure.port, reactor.heatPorts[2])
      annotation (Line(points={{4,124},{38,124},{38,79.5813},{45.6,79.5813}}, color={191,0,0}));

    connect(pointKineticMoltenSaltReactor.port_b, reactor.heatPorts[1])
      annotation (Line(points={{3,101},{26,101},{26,79.1937},{45.6,79.1937}},
                                                                            color={191,0,0}));
    connect(pointKineticMoltenSaltReactor.ReactorPowerOutput, r_power)
      annotation (Line(points={{3.5,90.5},{10,90.5},{10,65},{25,65}}, color={0,0,127}));
    connect(sensor_T_hotleg.T, add1.u2) annotation (Line(points={{57,118},{58,118},{58,176},{58,184},{0,184}}, color={0,0,127}));
    connect(Tfuel_rho_feedback.u, add1.y) annotation (Line(points={{-34,178},{-32,178},{-23,178}}, color={0,0,127}));
    connect(ramp.y, add3_1.u3) annotation (Line(points={{-83,104},{-76,104},{-76,90},{-66,90}}, color={0,0,127}));
    connect(rho0.y, add3_1.u2) annotation (Line(points={{-83,134},{-74,134},{-74,98},{-66,98}}, color={0,0,127}));
    connect(Tfuel_rho_feedback.y, add3_1.u1) annotation (Line(points={{-57,178},{-70,178},{-70,106},{-66,106}}, color={0,0,127}));
    connect(pointKineticMoltenSaltReactor.rho, add3_1.y)
      annotation (Line(points={{-15,101},{-31.5,101},{-31.5,98},{-43,98}},
                                                                         color={0,0,127}));
    connect(Tfuel_reference.y, add1.u1) annotation (Line(points={{19,162},{19,162},{14,162},{14,172},{0,172}}, color={0,0,127}));
    connect(HEXT_ramp.y, add.u2) annotation (Line(points={{-119,-132},{-82,-132}}, color={0,0,127}));
    connect(HEXmassFlowRate.T_in, add.y) annotation (Line(points={{-50,-82},{-54,-82},{-54,-126},{-59,-126}}, color={0,0,127}));
    connect(subcool.y, add.u1) annotation (Line(points={{-119,-92},{-90,-92},{-90,-120},{-82,-120}}, color={0,0,127}));
    connect(pulse.y, pointKineticMoltenSaltReactor.rho_control)
      annotation (Line(points={{-119,68},{-102,68},{-102,82},{-30,82},{-30,89.5},{-15,89.5}}, color={0,0,127}));
    annotation (Icon(coordinateSystem(preserveAspectRatio=false,
          extent={{-150,-150},{150,200}},
          initialScale=0.1)),                                      Diagram(
          coordinateSystem(preserveAspectRatio=false,
          extent={{-150,-150},{150,200}},
          initialScale=0.1)));
  end SingleFluidMoltenSaltReactor;

  model FuelSaltLiFBeF2 "fuel salt 7LiF-BeF2-ZrF4-UF4 (65-29.1-5-0.9 mole %)"
    // (c) 2016 Dr.Sdl
    // data taken from "A multi-physics modelling approach to the dynamics of Molten Salt
    // Reactors", Antonio Cammi et al., Annals of Nuclear Energy 38 (2011) 1356–1372
    // and ORNL/TM-2006/12 "Assessment of candidate molten salt coolants for the AHTR"
    //
    // Attention: for practical reasons the temperature range runs from 10 to 1500 Celsius
    // and the formulas are of course not valid for this full range and there can be
    // phase transitions in between. Therefore you should check if your fluid
    // state is in the range of validity as given by the literature.
    // You can narrow the range of validity with T_min and T_max.
    //
    // density in kg/m3
    // heat capacity in J/kg/K
    // thermal conductivity in W/m/K
    // dynamic viscosity in  kg/m/s
    // vapor pressure in Pa
    //
    extends Modelica.Media.Incompressible.TableBased(
      mediumName="FuelSalt 7LiF-BeF2",
      T_min=Modelica.SIunits.Conversions.from_degC(10),
      T_max=Modelica.SIunits.Conversions.from_degC(1500),
      TinK=false,
      T0=273.15,
      tableDensity=[10,3746.4; 100,3686.1; 500,3418.1; 600,3351.1; 634.85,3327.7; 700,3284.7;
                    800,3217.1; 900,3150.1; 1000,3083.1; 1100,3016.1; 1200,2949.1; 1500,2748.1],
      tableHeatCapacity=[10,1357; 100,1357; 500,1357; 600,1357; 634.85,1357; 700,1357; 800,1357;
                         900,1357; 1000,1357; 1100,1357; 1200,1357; 1500,1357],
      tableConductivity=[10,0.01; 100,0.131; 500,1.151; 600,1.221; 634.85,1.228; 700,1.217; 800,1.139;
                         900,0.987; 1000,0.761; 1100,0.461; 1200,0.087; 1500,0.01],
      tableViscosity=[10,10.000; 100,6.275; 500,0.022; 600,0.012; 634.85,0.010; 700,0.007; 800,0.005;
                      900,0.004; 1000,0.003; 1100,0.002; 1200,0.002; 1500,0.001],
      tableVaporPressure=[10,186.62; 100,186.62; 500,186.62; 600,186.62; 634.85,186.62; 700,186.62;
                         800,186.62; 900,186.62; 1000,186.62; 1100,186.62; 1200,186.62; 1500,186.62]);

    annotation (Icon(coordinateSystem(preserveAspectRatio=false)), Diagram(
          coordinateSystem(preserveAspectRatio=false)));
  end FuelSaltLiFBeF2;

  model PointKineticMoltenSaltReactor "Component for point reactor simulation"
    // (c) 2016 Dr.Sdl
    // Basic point kinetic reactor class
    // some type definitions -------------------------------------------------------------
    type NeutronPopulation = Real(quantity = "neutrons", min = 0);
    type PrecurserConcentration = Real(quantity = "precursor concentration", min = 0);
    type Reactivity = Real(quantity = "reactivity in pcm");
    type DelayedNeutronFraction = Real(quantity = "delayed neutron fraction", min = 0);
    type DecayConstant = Real(quantity = "1/s", unit = "1/s", min = 0);
    type GenerationTime = Real(quantity = "s", unit = "s", min = 0);
    // -----------------------------------------------------------------------------------
    // parameters are taken from:
    // "An approach to the MSR dynamics and stability analysis", Claudia Guerrieri et al.,
    // Progress in Nuclear Energy 67 (2013) 56-73
    parameter Integer Nd = 6 "number of delayed neutron groups";
    parameter DelayedNeutronFraction beta = 303.6e-5
      "total fraction of delayed Neutrons";
    parameter DelayedNeutronFraction bet[Nd] = {23.9e-5, 59.1e-5, 62.6e-5, 115.5e-5, 31.6e-5, 10.9e-5}
      "fraction of delayed neutron precursors";
    parameter DecayConstant landa[Nd] = {1.272e-2, 31.74e-2, 0.116e-1, 1.33e-1, 1.4, 3.87}
      "decay constant of the jth precursor group";
    parameter DecayConstant lTransit = 1. / 4.5 "inverse core transit time";
    parameter GenerationTime Lamda = 0.36 / 1000 "mean neutron generation time";
    parameter Real alpha0=0 "reactor heat port heat transfer coefficient";
    parameter Real T_ref=Modelica.SIunits.Conversions.from_degC(650)
      "reactor heat port reference temperature";
    // -----------------------------------------------------------------------------------
    // Does it make sense to set the initial reactivity such that der(N)=0, i.e. steady state?
    // in a Modelica environment?
    parameter Reactivity rho0 = NullReactivity(beta, bet, landa, 6, lTransit)
      "initial reactivity";
    // -----------------------------------------------------------------------------------
    parameter Real EnergyPerFission = 200.0 * 1.6021766e-13
      "energy release per fission in Joules";
    // energy from MeV into joules
    // better to set the following parameters by an input connector in the future
    parameter Real NominalPower = 20.0 "nominal power of reactor in MW";
    // nominal power of reactor in MW
    parameter Real ReactorFuelTemperature = 700.0
      "average reactor fuel temperature";
    // reactor fuel temperature in K
    // -----------------------------------------------------------------------------------
  public
    Modelica.Blocks.Interfaces.RealInput rho
      "time dependent reactivity input in pcm"                                        annotation(Placement(visible = true, transformation(origin = {-133.6, 63.976}, extent = {{-20, -20}, {20, 20}}, rotation = 0), iconTransformation(origin = {-90, 70}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
    // variables are time dependent and determined by equation
    NeutronPopulation N;
    DelayedNeutronFraction C[Nd];
    // We need a heat port to transfer the energy generation of the reactor to a heated pipe element
    // N is the neutron population which we set proportional to the energy generation

    Modelica.Thermal.HeatTransfer.Interfaces.HeatPort_b port_b annotation(Placement(visible = true, transformation(origin = {10, 62.747}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {90, 70}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
    // -----------------------------------------------------------------------------------
  protected
    Modelica.Blocks.Interfaces.RealInput rho_internal;

  public
    Modelica.Blocks.Interfaces.RealOutput ReactorPowerOutput annotation (Placement(transformation(extent={{5,-45},{25,-25}}),
          iconTransformation(extent={{85,-45},{105,-25}})));
  initial equation
    //port_b.T = ReactorFuelTemperature;
    N = NominalPower * 1.0e6 / EnergyPerFission;

    // estimate for concentration at 20MW reactor power
    C[1] = 2.5e17;
    C[2] = 2.5e17;
    C[3] = 6.0e17;
    C[4] = 8.0e17;
    C[5] = 0.4e17;
    C[6] = 0.6e16;

     /*
  der(C[1]) = 0;
  der(C[2]) = 0;
  der(C[3]) = 0;
  der(C[4]) = 0;
  der(C[5]) = 0;
  der(C[6]) = 0;
  */
  equation

    der(N) = ((rho0+rho_internal)  * 1.0E-5 - beta) / Lamda * N + C[1] * landa[1] + C[2] * landa[2] + C[3] * landa[3] + C[4] * landa[4] + C[5] * landa[5] + C[6] * landa[6];
    //der(N)=0;

    // heat flow rate out of reactor
    port_b.Q_flow = -N*EnergyPerFission*(1 + alpha0*(port_b.T - T_ref));

    der(C[1]) = bet[1] / Lamda * N - landa[1] * C[1] - lTransit * C[1];
    der(C[2]) = bet[2] / Lamda * N - landa[2] * C[2] - lTransit * C[2];
    der(C[3]) = bet[3] / Lamda * N - landa[3] * C[3] - lTransit * C[3];
    der(C[4]) = bet[4] / Lamda * N - landa[4] * C[4] - lTransit * C[4];
    der(C[5]) = bet[5] / Lamda * N - landa[5] * C[5] - lTransit * C[5];
    der(C[6]) = bet[6] / Lamda * N - landa[6] * C[6] - lTransit * C[6];

    connect(rho,rho_internal);
    ReactorPowerOutput=N*EnergyPerFission*(1 + alpha0*(port_b.T - T_ref));
    annotation(Icon(coordinateSystem(extent = {{-100, -100}, {100, 100}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10, 10}), graphics={  Text(visible = true, origin = {-0.391, 86.277}, extent = {{-69.609, -16.598}, {69.609, 16.598}}, textString = "Point Reactor"), Rectangle(visible = true, origin = {-0, 13.839}, fillColor = {255, 255, 255}, extent = {{-76.952, -57.895}, {76.952, 57.895}}), Ellipse(visible = true, origin = {-34.35, 15.867},
              fillPattern =                                                                                                    FillPattern.Solid, extent = {{-13.203, -13.203}, {13.203, 13.203}}), Ellipse(visible = true, origin = {22.937, 48.797}, lineColor = {255, 0, 0}, fillColor = {255, 0, 0},
              fillPattern =                                                                                                    FillPattern.Solid, extent = {{-10.574, -10.574}, {10.574, 10.574}}), Ellipse(visible = true, origin = {39.426, -25.581}, lineColor = {255, 170, 0}, fillColor = {255, 170, 0},
              fillPattern =                                                                                                    FillPattern.Solid, extent = {{-10.574, -10.574}, {10.574, 10.574}}), Line(visible = true, origin = {-3.497, 32.308}, points = {{-14.098, -8.811}, {14.098, 8.811}}, thickness = 1), Line(visible = true, origin = {3.846, -6.168}, points = {{-20.853, 12.63}, {20.853, -12.63}}, thickness = 1), Ellipse(visible = true, origin = {38.853, 24.037}, lineColor = {0, 0, 255}, fillColor = {0, 0, 255},
              fillPattern =                                                                                                    FillPattern.Solid, extent = {{-5.24, -5.24}, {5.24, 5.24}}), Ellipse(visible = true, origin = {43.613, 3.558}, lineColor = {0, 0, 255}, fillColor = {0, 0, 255},
              fillPattern =                                                                                                    FillPattern.Solid, extent = {{-5.24, -5.24}, {5.24, 5.24}}), Text(visible = true, origin = {0, -143.142}, extent = {{-150, 40}, {150, 100}}, textString = "%name")}), Diagram(coordinateSystem(extent = {{-148.5, -105}, {148.5, 105}}, preserveAspectRatio = true, initialScale = 0.1, grid = {5, 5}), graphics={  Rectangle(visible=  true, origin=  {-56.952, 17.105}, fillColor=  {255, 255, 255}, extent=  {{-76.952, -57.895}, {76.952, 57.895}}), Text(visible=  true, origin=  {-55.391, 35}, extent=  {{-69.609, -16.598}, {69.609, 16.598}}, textString=  "Point Reactor")}));
  end PointKineticMoltenSaltReactor;

  function NullReactivity

    // calculate initial rho0 for steady state point reactor
    input Real beta;
    // beta fraction
    input Real bet[:];
    // partial beta fractions
    input Real landa[:];
    // decay constants in 1/s
    input Integer N;
    input Real lTransit;
    // inverse core transit time 1/s
    output Real y;
    // reactivity for point reactor steady state
  algorithm
    y := beta;
    for i in 1:N loop
      y := y - bet[i] * landa[i] / (landa[i] + lTransit) "initial reactivity";
    end for;
    y := y * 1E+5;
    // convert to pcm
    //print(y);

  end NullReactivity;

  model PointKineticMoltenSaltReactor_P1
    "Component for point reactor simulation"
    // (c) 2016 Dr.Sdl
    // Basic point kinetic reactor class
    // Adding more features to enable temperature feedback from fuel or coolant
    // some type definitions -------------------------------------------------------------
    type NeutronPopulation = Real(quantity = "neutrons", min = 0);
    type PrecurserConcentration = Real(quantity = "precursor concentration", min = 0);
    type Reactivity = Real(quantity = "reactivity in pcm");
    type DelayedNeutronFraction = Real(quantity = "delayed neutron fraction", min = 0);
    type DecayConstant = Real(quantity = "1/s", unit = "1/s", min = 0);
    type GenerationTime = Real(quantity = "s", unit = "s", min = 0);
    // -----------------------------------------------------------------------------------
    // parameters are taken from:
    // "An approach to the MSR dynamics and stability analysis", Claudia Guerrieri et al.,
    // Progress in Nuclear Energy 67 (2013) 56-73
    parameter Integer Nd = 6 "number of delayed neutron groups";
    parameter DelayedNeutronFraction beta = 303.6e-5
      "total fraction of delayed Neutrons";
    parameter DelayedNeutronFraction bet[Nd] = {23.9e-5, 59.1e-5, 62.6e-5, 115.5e-5, 31.6e-5, 10.9e-5}
      "fraction of delayed neutron precursors";
    parameter DecayConstant landa[Nd] = {1.272e-2, 31.74e-2, 0.116e-1, 1.33e-1, 1.4, 3.87}
      "decay constant of the jth precursor group";
    parameter DecayConstant lTransit = 1. / 4.5 "inverse core transit time";
    parameter GenerationTime Lamda = 0.36 / 1000 "mean neutron generation time";
    parameter Real alpha0=0 "reactor heat port heat transfer coefficient";
    parameter Real T_ref=Modelica.SIunits.Conversions.from_degC(650)
      "reactor heat port reference temperature";
    // -----------------------------------------------------------------------------------
    // Does it make sense to set the initial reactivity such that der(N)=0, i.e. steady state?
    // in a Modelica environment?
    parameter Reactivity rho0 = NullReactivity(beta, bet, landa, 6, lTransit)
      "initial reactivity";
    // -----------------------------------------------------------------------------------
    parameter Real EnergyPerFission = 200.0 * 1.6021766e-13
      "energy release per fission in Joules";
    // energy from MeV into joules
    // better to set the following parameters by an input connector in the future
    parameter Real NominalPower = 20.0/1000 "nominal power of reactor in MW";
    // nominal power of reactor in MW
    parameter Real ReactorFuelTemperature = 700.0
      "average reactor fuel temperature";
    // reactor fuel temperature in K
    // -----------------------------------------------------------------------------------
  public
    Modelica.Blocks.Interfaces.RealInput rho
      "time dependent reactivity input in pcm"                                        annotation(Placement(visible = true, transformation(origin = {-133.6, 63.976}, extent = {{-20, -20}, {20, 20}}, rotation = 0), iconTransformation(origin = {-90, 70}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
    // variables are time dependent and determined by equation
    NeutronPopulation N;
    DelayedNeutronFraction C[Nd];
    // We need a heat port to transfer the energy generation of the reactor to a heated pipe element
    // N is the neutron population which we set proportional to the energy generation

    Modelica.Thermal.HeatTransfer.Interfaces.HeatPort_b port_b annotation(Placement(visible = true, transformation(origin = {10, 62.747}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {90, 70}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
    // -----------------------------------------------------------------------------------
  protected
    Modelica.Blocks.Interfaces.RealInput rho_internal;
    Modelica.Blocks.Interfaces.RealInput rho_internal_control;

  public
    Modelica.Blocks.Interfaces.RealOutput ReactorPowerOutput annotation (Placement(transformation(extent={{5,-45},{25,-25}}),
          iconTransformation(extent={{85,-45},{105,-25}})));
    Modelica.Blocks.Interfaces.RealInput rho_control
      "port to add some more reactivity feedback"
      annotation (Placement(transformation(extent={{-154,-50},{-114,-10}}),
          iconTransformation(extent={{-100,-55},{-80,-35}})));
  initial equation
    //port_b.T = ReactorFuelTemperature;
    N = NominalPower * 1.0e6 / EnergyPerFission;

    // estimate for concentration at 20MW reactor power
    C[1] = 2.5e17/1000;
    C[2] = 2.5e17/1000;
    C[3] = 6.0e17/1000;
    C[4] = 8.0e17/1000;
    C[5] = 0.4e17/1000;
    C[6] = 0.6e16/1000;

     /*
  der(C[1]) = 0;
  der(C[2]) = 0;
  der(C[3]) = 0;
  der(C[4]) = 0;
  der(C[5]) = 0;
  der(C[6]) = 0;
  */
  equation

    der(N) = ((rho0+rho_internal+rho_internal_control)  * 1.0E-5 - beta) / Lamda * N + C[1] * landa[1] + C[2] * landa[2] + C[3] * landa[3] + C[4] * landa[4] + C[5] * landa[5] + C[6] * landa[6];
    //der(N)=0;

    // heat flow rate out of reactor
    port_b.Q_flow = -N*EnergyPerFission*(1 + alpha0*(port_b.T - T_ref));

    der(C[1]) = bet[1] / Lamda * N - landa[1] * C[1] - lTransit * C[1];
    der(C[2]) = bet[2] / Lamda * N - landa[2] * C[2] - lTransit * C[2];
    der(C[3]) = bet[3] / Lamda * N - landa[3] * C[3] - lTransit * C[3];
    der(C[4]) = bet[4] / Lamda * N - landa[4] * C[4] - lTransit * C[4];
    der(C[5]) = bet[5] / Lamda * N - landa[5] * C[5] - lTransit * C[5];
    der(C[6]) = bet[6] / Lamda * N - landa[6] * C[6] - lTransit * C[6];

    connect(rho,rho_internal);
    connect(rho_control,rho_internal_control);
    ReactorPowerOutput=N*EnergyPerFission*(1 + alpha0*(port_b.T - T_ref));
    annotation(Icon(coordinateSystem(extent = {{-100, -100}, {100, 100}}, preserveAspectRatio = true, initialScale = 0.1, grid = {10, 10}), graphics={  Text(visible = true, origin = {-0.391, 86.277}, extent = {{-69.609, -16.598}, {69.609, 16.598}}, textString = "Point Reactor"), Rectangle(visible = true, origin = {-0, 13.839}, fillColor = {255, 255, 255}, extent = {{-76.952, -57.895}, {76.952, 57.895}}), Ellipse(visible = true, origin = {-34.35, 15.867},
              fillPattern =                                                                                                    FillPattern.Solid, extent = {{-13.203, -13.203}, {13.203, 13.203}}), Ellipse(visible = true, origin = {22.937, 48.797}, lineColor = {255, 0, 0}, fillColor = {255, 0, 0},
              fillPattern =                                                                                                    FillPattern.Solid, extent = {{-10.574, -10.574}, {10.574, 10.574}}), Ellipse(visible = true, origin = {39.426, -25.581}, lineColor = {255, 170, 0}, fillColor = {255, 170, 0},
              fillPattern =                                                                                                    FillPattern.Solid, extent = {{-10.574, -10.574}, {10.574, 10.574}}), Line(visible = true, origin = {-3.497, 32.308}, points = {{-14.098, -8.811}, {14.098, 8.811}}, thickness = 1), Line(visible = true, origin = {3.846, -6.168}, points = {{-20.853, 12.63}, {20.853, -12.63}}, thickness = 1), Ellipse(visible = true, origin = {38.853, 24.037}, lineColor = {0, 0, 255}, fillColor = {0, 0, 255},
              fillPattern =                                                                                                    FillPattern.Solid, extent = {{-5.24, -5.24}, {5.24, 5.24}}), Ellipse(visible = true, origin = {43.613, 3.558}, lineColor = {0, 0, 255}, fillColor = {0, 0, 255},
              fillPattern =                                                                                                    FillPattern.Solid, extent = {{-5.24, -5.24}, {5.24, 5.24}}), Text(visible = true, origin = {0, -143.142}, extent = {{-150, 40}, {150, 100}}, textString = "%name")}), Diagram(coordinateSystem(extent = {{-148.5, -105}, {148.5, 105}}, preserveAspectRatio = true, initialScale = 0.1, grid = {5, 5}), graphics={  Rectangle(visible=  true, origin=  {-56.952, 17.105}, fillColor=  {255, 255, 255}, extent=  {{-76.952, -57.895}, {76.952, 57.895}}), Text(visible=  true, origin=  {-55.391, 35}, extent=  {{-69.609, -16.598}, {69.609, 16.598}}, textString=  "Point Reactor")}));
  end PointKineticMoltenSaltReactor_P1;

  package BOOTSTRAP

    model Bootstrap001
      "unit initialisation testing for molten salt fluid, open circuit"
        // (c) 2016 Dr.Sdl
        // First test of pump and "FuelSaltLiFBeF2" fluid in
        // an open circuit model pumping fluid from one tank into another tank
        // and heating the fluid at an intermediary pipe
        //

        replaceable package Medium =
           Nuclear.FuelSaltLiFBeF2 (final
              singleState=true, final enthalpyOfT=true);

       inner Modelica.Fluid.System system(
          m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        T_ambient=923.15,
        m_flow_start=0.5)   annotation (Placement(transformation(extent={{-140,120},
                {-120,140}},rotation=0)));

      Modelica.Fluid.Vessels.OpenTank tank(
        redeclare package Medium = Medium,
        nPorts=1,
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        ports(each p(start=110000)),
        height=4,
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        level_start=0.5,
        crossArea=3.00)
                  annotation (Placement(transformation(extent={{-120,26},{-100,46}},
              rotation=0)));
      Modelica.Fluid.Machines.ControlledPump pump(
        redeclare package Medium = Medium,
        N_nominal=1500,
        use_T_start=true,
        control_m_flow=false,
        allowFlowReversal=false,
        use_p_set=true,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        m_flow_start=system.m_flow_start,
        m_flow_nominal=system.m_flow_start,
        p_a_start=110000,
        p_b_start=130000,
        p_a_nominal=110000,
        p_b_nominal=130000)
        annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
      Modelica.Fluid.Valves.ValveIncompressible valve(
        redeclare package Medium = Medium,
        CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
        show_T=true,
        allowFlowReversal=false,
        m_flow_nominal=system.m_flow_start,
        dp_start=18000,
        dp_nominal=10000)
        annotation (Placement(transformation(extent={{20,-88},{0,-68}},rotation=0)));

      Modelica.Fluid.Pipes.DynamicPipe reactor(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        use_HeatTransfer=true,
        modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        nNodes=16,
        diameter=0.2,
        height_ab=0,
        length=3.9,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        p_a_start=130000) annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                                                                                         rotation=90,
            origin={50,82})));

    public
      Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow(redeclare package
          Medium =                                                                 Medium)
        annotation (Placement(transformation(extent={{-12,10},{8,30}}, rotation=0)));
      Modelica.Thermal.HeatTransfer.Sources.FixedHeatFlow burner(
        alpha=-0.5,
        Q_flow=5,
        T_ref=923.15)
        annotation (Placement(transformation(extent={{-34,72},{-14,92}},
                                                                       rotation=0)));
      Modelica.Blocks.Sources.Step handle(
        height=0.9,
        startTime=20,
        offset=0.1)   annotation (Placement(transformation(extent={{-30,-29},{-16,-15}},
                      rotation=0)));

    protected
      Modelica.Blocks.Interfaces.RealOutput m_flow
        annotation (Placement(transformation(extent={{-4,40},{8,52}},   rotation=
                0)));

    public
      Modelica.Blocks.Sources.Ramp PumpRamp(
        height=200000,
        duration=100,
        offset=130000,
        startTime=9) annotation (Placement(transformation(extent={{-80,54},{-60,74}})));
      Modelica.Fluid.Vessels.OpenTank tank1(
        redeclare package Medium = Medium,
        nPorts=1,
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        ports(each p(start=110000)),
        height=4,
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        level_start=0.5,
        crossArea=3.00)
                  annotation (Placement(transformation(extent={{-78,-62},{-58,-42}},
              rotation=0)));
    equation

      connect(sensor_m_flow.m_flow, m_flow)         annotation (Line(points={{-2,31},{-2,46},{2,46}},
                                                    color={0,0,127}));
      connect(sensor_m_flow.port_b, reactor.port_a)
        annotation (Line(points={{8,20},{8,20},{50,20},{50,72}},
                                                         color={0,127,255}));
      connect(handle.y, valve.opening)       annotation (Line(
          points={{-15.3,-22},{10,-22},{10,-70}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(pump.port_b, sensor_m_flow.port_a)
                                                annotation (Line(
          points={{-30,20},{-12,20}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(burner.port, reactor.heatPorts[1]) annotation (Line(
          points={{-14,82},{45.6,82},{45.6,79.1937}},
          color={191,0,0},
          smooth=Smooth.None));
      connect(tank.ports[1], pump.port_a)
        annotation (Line(points={{-110,26},{-110,20},{-50,20}},        color={0,127,255}));

      // connect(radiator.port_b, tank.ports[2]);
      connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
              {-35,28.2}}, color={0,0,127}));

      connect(reactor.port_b, valve.port_a) annotation (Line(points={{50,92},{74,92},
              {74,-78},{20,-78}}, color={0,127,255}));
      connect(tank1.ports[1], valve.port_b) annotation (Line(points={{-68,-62},{-22,
              -62},{-22,-78},{0,-78}}, color={0,127,255}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)));
    end Bootstrap001;

    model Bootstrap002
      "unit initialisation testing simplified reactor closed circuit for molten salt fluid"
        // (c) 2016 Dr.Sdl
        // Simplified layout of single fluid molten salt reactor circuit without reactor but
        // constant heat source and without heat exchanger and without hotleg and coldleg.
        //
        replaceable package Medium =
           Nuclear.FuelSaltLiFBeF2 (final
              singleState=true, final enthalpyOfT=true);

       inner Modelica.Fluid.System system(
          m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        T_ambient=923.15,
        m_flow_start=0.5)   annotation (Placement(transformation(extent={{-140,120},
                {-120,140}},rotation=0)));

      Modelica.Fluid.Vessels.OpenTank tank(
        redeclare package Medium = Medium,
        nPorts=2,
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        ports(each p(start=110000)),
        height=4,
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
           Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        level_start=0.5,
        crossArea=3.00)
                  annotation (Placement(transformation(extent={{-134,26},{-114,46}},
              rotation=0)));
      Modelica.Fluid.Machines.ControlledPump pump(
        redeclare package Medium = Medium,
        N_nominal=1500,
        use_T_start=true,
        control_m_flow=false,
        allowFlowReversal=false,
        use_p_set=true,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        m_flow_start=system.m_flow_start,
        m_flow_nominal=system.m_flow_start,
        p_a_start=110000,
        p_b_start=130000,
        p_a_nominal=110000,
        p_b_nominal=130000)
        annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
      Modelica.Fluid.Valves.ValveIncompressible valve(
        redeclare package Medium = Medium,
        CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
        show_T=true,
        allowFlowReversal=false,
        m_flow_nominal=system.m_flow_start,
        dp_start=18000,
        dp_nominal=10000)
        annotation (Placement(transformation(extent={{20,-88},{0,-68}},rotation=0)));

      Modelica.Fluid.Pipes.DynamicPipe reactor(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        use_HeatTransfer=true,
        modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        nNodes=16,
        diameter=0.2,
        height_ab=0,
        length=3.9,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        p_a_start=130000) annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                                                                                         rotation=90,
            origin={50,82})));

    public
      Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow(redeclare package
          Medium =                                                                 Medium)
        annotation (Placement(transformation(extent={{-12,10},{8,30}}, rotation=0)));
      Modelica.Thermal.HeatTransfer.Sources.FixedHeatFlow burner(
        alpha=-0.5,
        Q_flow=5,
        T_ref=923.15)
        annotation (Placement(transformation(extent={{-34,72},{-14,92}},
                                                                       rotation=0)));
      Modelica.Blocks.Sources.Step handle(
        height=0.9,
        startTime=20,
        offset=0.1)   annotation (Placement(transformation(extent={{-30,-29},{-16,-15}},
                      rotation=0)));

    protected
      Modelica.Blocks.Interfaces.RealOutput m_flow
        annotation (Placement(transformation(extent={{-4,40},{8,52}},   rotation=
                0)));

    public
      Modelica.Blocks.Sources.Ramp PumpRamp(
        height=200000,
        duration=100,
        offset=130000,
        startTime=9) annotation (Placement(transformation(extent={{-80,54},{-60,74}})));
    equation

      connect(sensor_m_flow.m_flow, m_flow)         annotation (Line(points={{-2,31},{-2,46},{2,46}},
                                                    color={0,0,127}));
      connect(sensor_m_flow.port_b, reactor.port_a)
        annotation (Line(points={{8,20},{8,20},{50,20},{50,72}},
                                                         color={0,127,255}));
      connect(handle.y, valve.opening)       annotation (Line(
          points={{-15.3,-22},{10,-22},{10,-70}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(pump.port_b, sensor_m_flow.port_a)
                                                annotation (Line(
          points={{-30,20},{-12,20}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(burner.port, reactor.heatPorts[1]) annotation (Line(
          points={{-14,82},{45.6,82},{45.6,79.1937}},
          color={191,0,0},
          smooth=Smooth.None));
      connect(tank.ports[1], pump.port_a)
        annotation (Line(points={{-126,26},{-126,20},{-50,20}},        color={0,127,255}));

      // connect(radiator.port_b, tank.ports[2]);
      connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
              {-35,28.2}}, color={0,0,127}));

      connect(reactor.port_b, valve.port_a) annotation (Line(points={{50,92},{74,92},
              {74,-78},{20,-78}}, color={0,127,255}));
      connect(valve.port_b, tank.ports[2]) annotation (Line(points={{0,-78},{-58,-78},
              {-122,-78},{-122,26}}, color={0,127,255}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)));
    end Bootstrap002;

    model Bootstrap003
      "unit initialisation testing reactor circuit layout without heat exchanger"
        // (c) 2016 Dr.Sdl
        // Bssic layout of single fluid molten salt reactor circuit without reactor but
        // constant heat source and without heat exchanger but with hotleg and coldleg.
        //
        replaceable package Medium =
           Nuclear.FuelSaltLiFBeF2 (final
              singleState=true, final enthalpyOfT=true);

       inner Modelica.Fluid.System system(
          m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        m_flow_start=0.5,
        T_ambient=923.15)   annotation (Placement(transformation(extent={{-140,120},
                {-120,140}},rotation=0)));

      Modelica.Fluid.Vessels.OpenTank tank(
        redeclare package Medium = Medium,
        nPorts=2,
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        ports(each p(start=110000)),
        height=4,
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
            Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        level_start=0.5,
        crossArea=3.00)
                  annotation (Placement(transformation(extent={{-120,26},{-100,46}},
              rotation=0)));
      Modelica.Fluid.Machines.ControlledPump pump(
        redeclare package Medium = Medium,
        N_nominal=1500,
        use_T_start=true,
        control_m_flow=false,
        allowFlowReversal=false,
        use_p_set=true,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        m_flow_start=system.m_flow_start,
        m_flow_nominal=system.m_flow_start,
        p_a_start=110000,
        p_b_start=130000,
        p_a_nominal=110000,
        p_b_nominal=130000)
        annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
      Modelica.Fluid.Valves.ValveIncompressible valve(
        redeclare package Medium = Medium,
        CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
        show_T=true,
        allowFlowReversal=false,
        m_flow_nominal=system.m_flow_start,
        dp_start=18000,
        dp_nominal=10000)
        annotation (Placement(transformation(extent={{112,-88},{92,-68}},
                                                                       rotation=0)));

      Modelica.Fluid.Pipes.DynamicPipe reactor(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        use_HeatTransfer=true,
        modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        nNodes=16,
        diameter=0.2,
        height_ab=0,
        length=3.9,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        p_a_start=130000) annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                                                                                         rotation=90,
            origin={50,82})));

      Modelica.Fluid.Pipes.DynamicPipe hotleg(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        diameter=0.2,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        length=10,
        p_a_start=130000) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=-90,
            origin={122,12})));

      Modelica.Fluid.Pipes.DynamicPipe coldleg(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        diameter=0.2,
        nNodes=1,
        modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        length=10,
        p_a_start=110000) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={-112,-20})));

    public
      Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow(redeclare package
          Medium =                                                                 Medium)
        annotation (Placement(transformation(extent={{-12,10},{8,30}}, rotation=0)));
      Modelica.Thermal.HeatTransfer.Sources.FixedHeatFlow burner(
        alpha=-0.5,
        Q_flow=5,
        T_ref=923.15)
        annotation (Placement(transformation(extent={{-34,72},{-14,92}},
                                                                       rotation=0)));
      Modelica.Fluid.Sensors.Temperature sensor_T_hotleg(redeclare package
          Medium =                                                                  Medium)
        annotation (Placement(transformation(extent={{40,108},{60,128}},
                                                                       rotation=0)));
      Modelica.Blocks.Sources.Step handle(
        height=0.9,
        startTime=20,
        offset=0.1)   annotation (Placement(transformation(extent={{74,-29},{88,-15}},
                      rotation=0)));

    protected
      Modelica.Blocks.Interfaces.RealOutput T_hotleg
        annotation (Placement(transformation(extent={{80,112},{92,124}}, rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput tankLevel
                                     annotation (Placement(transformation(extent={{-94,26},{-82,38}},
                                    rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput m_flow
        annotation (Placement(transformation(extent={{-4,40},{8,52}},   rotation=
                0)));

    public
      Modelica.Blocks.Sources.Ramp PumpRamp(
        height=200000,
        duration=100,
        offset=130000,
        startTime=9) annotation (Placement(transformation(extent={{-80,54},{-60,74}})));
    equation
    tankLevel = tank.level;
      connect(sensor_m_flow.m_flow, m_flow)         annotation (Line(points={{-2,31},{-2,46},{2,46}},
                                                    color={0,0,127}));
      connect(sensor_m_flow.port_b, reactor.port_a)
        annotation (Line(points={{8,20},{8,20},{50,20},{50,72}},
                                                         color={0,127,255}));
      connect(sensor_T_hotleg.T, T_hotleg)
        annotation (Line(points={{57,118},{82,118},{86,118}}, color={0,0,127}));
      connect(handle.y, valve.opening)       annotation (Line(
          points={{88.7,-22},{102,-22},{102,-70}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(pump.port_b, sensor_m_flow.port_a)
                                                annotation (Line(
          points={{-30,20},{-12,20}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(burner.port, reactor.heatPorts[1]) annotation (Line(
          points={{-14,82},{45.6,82},{45.6,79.1937}},
          color={191,0,0},
          smooth=Smooth.None));
      connect(sensor_T_hotleg.port, reactor.port_b) annotation (Line(
          points={{50,108},{50,92}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(reactor.port_b, hotleg.port_a) annotation (Line(
          points={{50,92},{50,104},{122,104},{122,22}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(hotleg.port_b, valve.port_a) annotation (Line(
          points={{122,2},{122,-78},{112,-78}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(tank.ports[1], pump.port_a)
        annotation (Line(points={{-112,26},{-112,20},{-50,20}},        color={0,127,255}));

      connect(coldleg.port_b, tank.ports[2])
        annotation (Line(points={{-112,-10},{-112,26},{-108,26}},
                                                              color={0,127,255}));

      // connect(radiator.port_b, tank.ports[2]);
      connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
              {-35,28.2}}, color={0,0,127}));

      connect(coldleg.port_a, valve.port_b) annotation (Line(points={{-112,-30},{-112,
              -30},{-112,-78},{92,-78}}, color={0,127,255}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)));
    end Bootstrap003;

    model Bootstrap004
      "unit initialisation testing reactor circuit layout with heat exchanger"
        // (c) 2016 Dr.Sdl
        // Basic layout of single fluid molten salt reactor circuit without reactor but
        // constant heat source and heat exchanger and coldleg and hotleg
        //

        replaceable package Medium =
           Nuclear.FuelSaltLiFBeF2 (final
              singleState=true, final enthalpyOfT=true);

       inner Modelica.Fluid.System system(
          m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        m_flow_start=0.05,
        T_ambient=923.15)   annotation (Placement(transformation(extent={{-140,120},
                {-120,140}},rotation=0)));

      Modelica.Fluid.Vessels.OpenTank tank(
        redeclare package Medium = Medium,
        nPorts=2,
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        ports(each p(start=110000)),
        height=4,
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
            Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        level_start=0.5,
        crossArea=3.00)
                  annotation (Placement(transformation(extent={{-120,26},{-100,46}},
              rotation=0)));
      Modelica.Fluid.Machines.ControlledPump pump(
        redeclare package Medium = Medium,
        N_nominal=1500,
        use_T_start=true,
        control_m_flow=false,
        allowFlowReversal=false,
        use_p_set=true,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        m_flow_start=system.m_flow_start,
        m_flow_nominal=system.m_flow_start,
        p_a_start=110000,
        p_b_start=130000,
        p_a_nominal=110000,
        p_b_nominal=130000)
        annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
      Modelica.Fluid.Valves.ValveIncompressible valve(
        redeclare package Medium = Medium,
        CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
        show_T=true,
        allowFlowReversal=false,
        m_flow_nominal=system.m_flow_start,
        dp_start=18000,
        dp_nominal=10000)
        annotation (Placement(transformation(extent={{102,-82},{82,-62}},
                                                                       rotation=0)));

      Modelica.Fluid.Pipes.DynamicPipe reactor(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        use_HeatTransfer=true,
        modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        nNodes=16,
        diameter=0.2,
        height_ab=0,
        length=3.9,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        p_a_start=130000) annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                                                                                         rotation=90,
            origin={50,82})));

      Modelica.Fluid.Examples.HeatExchanger.BaseClasses.BasicHX HEX(
        c_wall=500,
        s_wall=0.005,
        rho_wall=900,
        redeclare package Medium_1 =
            Medium,
        redeclare package Medium_2 =
            Medium,
        modelStructure_1=Modelica.Fluid.Types.ModelStructure.av_b,
        modelStructure_2=Modelica.Fluid.Types.ModelStructure.a_vb,
        redeclare model HeatTransfer_1 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.LocalPipeFlowHeatTransfer
            (alpha0=1000),
        redeclare model HeatTransfer_2 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
            (alpha0=2000),
        crossArea_1=3.14e-2,
        perimeter_1=0.628,
        T_start_1=Modelica.SIunits.Conversions.from_degC(650),
        T_start_2=Modelica.SIunits.Conversions.from_degC(650),
        k_wall=100,
        crossArea_2=3.14e-2,
        perimeter_2=0.628,
        nNodes=2,
        use_T_start=false,
        m_flow_start_2=system.m_flow_start,
        Twall_start=Modelica.SIunits.Conversions.from_degC(650),
        m_flow_start_1=0.05,
        length=0.001,
        area_h_1=0.628*0.001,
        area_h_2=0.628*0.001,
        dT=272.15,
        p_a_start1=105000,
        p_b_start1=100000,
        p_a_start2=127600,
        p_b_start2=117600)   annotation (Placement(transformation(extent={{-18,-86},
                {12,-56}}, rotation=0)));

      Modelica.Fluid.Sources.Boundary_pT HEXambient(
        nPorts=1,
        redeclare package Medium = Medium,
        T=Modelica.SIunits.Conversions.from_degC(650),
        p=100000) annotation (Placement(transformation(
            extent={{7,-7},{-7,7}},
            rotation=0,
            origin={46,-56})));

      Modelica.Fluid.Sources.MassFlowSource_T HEXmassFlowRate(
        nPorts=1,
        redeclare package Medium = Medium,
        use_T_in=false,
        use_X_in=false,
        use_m_flow_in=false,
        T=Modelica.SIunits.Conversions.from_degC(650),
        m_flow=0.05)
                  annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=0,
            origin={-40,-86})));

    public
      Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow(redeclare package
          Medium =                                                                 Medium)
        annotation (Placement(transformation(extent={{-12,10},{8,30}}, rotation=0)));
      Modelica.Thermal.HeatTransfer.Sources.FixedHeatFlow burner(
        alpha=-0.5,
        Q_flow=5000,
        T_ref=923.15)
        annotation (Placement(transformation(extent={{-34,72},{-14,92}},
                                                                       rotation=0)));
      Modelica.Blocks.Sources.Step handle(
        height=0.9,
        startTime=20,
        offset=0.9)   annotation (Placement(transformation(extent={{74,-29},{88,-15}},
                      rotation=0)));

    protected
      Modelica.Blocks.Interfaces.RealOutput m_flow
        annotation (Placement(transformation(extent={{-4,40},{8,52}},   rotation=
                0)));

    public
      Modelica.Blocks.Sources.Ramp PumpRamp(
        height=200000,
        duration=100,
        offset=130000,
        startTime=9) annotation (Placement(transformation(extent={{-80,54},{-60,74}})));
    equation

      connect(sensor_m_flow.m_flow, m_flow)         annotation (Line(points={{-2,31},{-2,46},{2,46}},
                                                    color={0,0,127}));
      connect(sensor_m_flow.port_b, reactor.port_a)
        annotation (Line(points={{8,20},{8,20},{50,20},{50,72}},
                                                         color={0,127,255}));
      connect(handle.y, valve.opening)       annotation (Line(
          points={{88.7,-22},{92,-22},{92,-64}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(pump.port_b, sensor_m_flow.port_a)
                                                annotation (Line(
          points={{-30,20},{-12,20}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(burner.port, reactor.heatPorts[1]) annotation (Line(
          points={{-14,82},{45.6,82},{45.6,79.1937}},
          color={191,0,0},
          smooth=Smooth.None));
      connect(tank.ports[1], pump.port_a)
        annotation (Line(points={{-112,26},{-112,20},{-50,20}},        color={0,127,255}));

      // connect(radiator.port_b, tank.ports[2]);
      connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
              {-35,28.2}}, color={0,0,127}));
      connect(valve.port_b, HEX.port_a2) annotation (Line(points={{82,-72},{58,-72},
              {58,-77.9},{13.5,-77.9}}, color={0,127,255}));
      connect(HEXmassFlowRate.ports[1], HEX.port_a1) annotation (Line(points={{-30,-86},
              {-22,-86},{-22,-71.3},{-19.5,-71.3}},      color={0,127,255}));
      connect(HEX.port_b1, HEXambient.ports[1]) annotation (Line(points={{13.5,-71.3},
              {13.5,-63.65},{39,-63.65},{39,-56}},        color={0,127,255}));

      connect(reactor.port_b, valve.port_a) annotation (Line(points={{50,92},{50,92},
              {50,100},{122,100},{122,-72},{102,-72}}, color={0,127,255}));
      connect(HEX.port_b2, tank.ports[2]) annotation (Line(points={{-19.5,-64.1},{-108,
              -64.1},{-108,26}}, color={0,127,255}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)));
    end Bootstrap004;

    model Bootstrap005
      "unit initialisation testing for simplified reactor circuit with heat exchanger and Glycol47"
        // (c) 2016 Dr.Sdl
        // In order to prepare the molten salt reactor circuit a test is run in a simplified
        // circuit with Glycol47 instead of FuelSaltLiFBeF2 fluid
        //

          /*
      replaceable package Medium =
      Modelica.Media.Water.StandardWater
      constrainedby Modelica.Media.Interfaces.PartialMedium;
      */

            replaceable package Medium =
            Modelica.Media.Incompressible.Examples.Glycol47 (final
              singleState=true, final enthalpyOfT=true);

            /*
       replaceable package Medium =
       Nuclear.FuelSaltLiFBeF2 (final 
       singleState=true, final enthalpyOfT=true);
       */
       inner Modelica.Fluid.System system(
          m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        m_flow_start=0.05,
        T_ambient=323.15)   annotation (Placement(transformation(extent={{-140,120},
                {-120,140}},rotation=0)));

      Modelica.Fluid.Vessels.OpenTank tank(
        redeclare package Medium = Medium,
        nPorts=2,
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        ports(each p(start=110000)),
        height=4,
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
            Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        level_start=0.5,
        crossArea=3.00,
        T_start=Modelica.SIunits.Conversions.from_degC(50))
                  annotation (Placement(transformation(extent={{-120,26},{-100,46}},
              rotation=0)));
      Modelica.Fluid.Machines.ControlledPump pump(
        redeclare package Medium = Medium,
        N_nominal=1500,
        use_T_start=true,
        control_m_flow=false,
        allowFlowReversal=false,
        use_p_set=true,
        m_flow_start=system.m_flow_start,
        m_flow_nominal=system.m_flow_start,
        T_start=Modelica.SIunits.Conversions.from_degC(50),
        p_a_start=110000,
        p_b_start=130000,
        p_a_nominal=110000,
        p_b_nominal=130000)
        annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
      Modelica.Fluid.Valves.ValveIncompressible valve(
        redeclare package Medium = Medium,
        CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
        show_T=true,
        allowFlowReversal=false,
        m_flow_nominal=system.m_flow_start,
        dp_start=18000,
        dp_nominal=10000)
        annotation (Placement(transformation(extent={{118,-92},{98,-72}},
                                                                       rotation=0)));

      Modelica.Fluid.Examples.HeatExchanger.BaseClasses.BasicHX HEX(
        c_wall=500,
        s_wall=0.005,
        rho_wall=900,
        redeclare package Medium_1 =
            Medium,
        redeclare package Medium_2 =
            Medium,
        modelStructure_1=Modelica.Fluid.Types.ModelStructure.av_b,
        modelStructure_2=Modelica.Fluid.Types.ModelStructure.a_vb,
        redeclare model HeatTransfer_1 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.LocalPipeFlowHeatTransfer
            (alpha0=1000),
        redeclare model HeatTransfer_2 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
            (alpha0=2000),
        crossArea_1=3.14e-2,
        perimeter_1=0.628,
        T_start_1=Modelica.SIunits.Conversions.from_degC(50),
        T_start_2=Modelica.SIunits.Conversions.from_degC(50),
        k_wall=100,
        crossArea_2=3.14e-2,
        perimeter_2=0.628,
        nNodes=2,
        use_T_start=false,
        m_flow_start_2=system.m_flow_start,
        Twall_start=Modelica.SIunits.Conversions.from_degC(50),
        length=1.001,
        area_h_1=0.628*1.001,
        area_h_2=0.628*1.001,
        m_flow_start_1=1,
        dT=272.15,
        p_a_start1=130000,
        p_b_start1=100000,
        p_a_start2=130000,
        p_b_start2=100000)   annotation (Placement(transformation(extent={{-18,-88},
                {12,-58}}, rotation=0)));

      Modelica.Fluid.Sources.Boundary_pT HEXambient(
        nPorts=1,
        redeclare package Medium = Medium,
        T=Modelica.SIunits.Conversions.from_degC(50),
        p=100000) annotation (Placement(transformation(
            extent={{7,-7},{-7,7}},
            rotation=0,
            origin={56,-64})));

      Modelica.Fluid.Sources.MassFlowSource_T HEXmassFlowRate(
        nPorts=1,
        redeclare package Medium = Medium,
        use_T_in=false,
        use_X_in=false,
        use_m_flow_in=false,
        T=Modelica.SIunits.Conversions.from_degC(50),
        m_flow=1) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=0,
            origin={-60,-90})));

    public
      Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow(redeclare package
          Medium =                                                                 Medium)
        annotation (Placement(transformation(extent={{-12,10},{8,30}}, rotation=0)));
      Modelica.Blocks.Sources.Step handle(
        height=0.9,
        startTime=20,
        offset=0.01)  annotation (Placement(transformation(extent={{74,-29},{88,-15}},
                      rotation=0)));

    protected
      Modelica.Blocks.Interfaces.RealOutput m_flow
        annotation (Placement(transformation(extent={{-4,40},{8,52}},   rotation=
                0)));

    public
      Modelica.Blocks.Sources.Ramp PumpRamp(
        height=200000,
        duration=100,
        offset=130000,
        startTime=9) annotation (Placement(transformation(extent={{-80,54},{-60,74}})));
    equation

      connect(sensor_m_flow.m_flow, m_flow)         annotation (Line(points={{-2,31},{-2,46},{2,46}},
                                                    color={0,0,127}));
      connect(handle.y, valve.opening)       annotation (Line(
          points={{88.7,-22},{108,-22},{108,-74}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(pump.port_b, sensor_m_flow.port_a)
                                                annotation (Line(
          points={{-30,20},{-12,20}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(tank.ports[1], pump.port_a)
        annotation (Line(points={{-112,26},{-112,20},{-50,20}},        color={0,127,255}));

      // connect(radiator.port_b, tank.ports[2]);
      connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
              {-35,28.2}}, color={0,0,127}));
      connect(valve.port_b, HEX.port_a2) annotation (Line(points={{98,-82},{58,
              -82},{58,-79.9},{13.5,-79.9}},
                                        color={0,127,255}));
      connect(HEXmassFlowRate.ports[1], HEX.port_a1) annotation (Line(points={{-50,-90},
              {-22,-90},{-22,-73.3},{-19.5,-73.3}},      color={0,127,255}));
      connect(HEX.port_b1, HEXambient.ports[1]) annotation (Line(points={{13.5,
              -73.3},{13.5,-63.65},{49,-63.65},{49,-64}}, color={0,127,255}));

      connect(HEX.port_b2, tank.ports[2]) annotation (Line(points={{-19.5,-66.1},
              {-108,-66.1},{-108,26}},
                                 color={0,127,255}));
      connect(sensor_m_flow.port_b, valve.port_a) annotation (Line(points={{8,20},{
              74,20},{140,20},{140,-82},{118,-82}},
                                                 color={0,127,255}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)));
    end Bootstrap005;

    model Bootstrap008 "unit initialisation testing point kinetic reactor"
        // (c) 2016 Dr.Sdl
        // First test of the "PointKineticMoltenSaltReactor" model
        //
       inner Modelica.Fluid.System system(
        m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        T_ambient=923.15,
        m_flow_start=0.5)   annotation (Placement(transformation(extent={{-140,120},
                {-120,140}},rotation=0)));

      PointKineticMoltenSaltReactor pointKineticReactor
        annotation (Placement(transformation(extent={{-68,12},{-48,32}})));

      Modelica.Blocks.Sources.Sine rhos(
        amplitude=50,
        freqHz=1,
        phase=0,
        offset=10,
        startTime=0) annotation (Placement(transformation(extent={{-122,48},{-102,68}})));
    protected
        Modelica.Blocks.Interfaces.RealOutput RPower
        annotation (Placement(transformation(extent={{-24,46},{-4,66}})));

    equation
      //pointKineticReactor.rho=0;
      //RPower=pointKineticReactor.N;
      //pointKineticReactor.rho=sin(time);

      connect(rhos.y, pointKineticReactor.rho) annotation (Line(points={{-101,58},{-86,58},{-78,58},{-78,29},{-67,29}}, color={0,0,127}));
      //RPower=pointKineticReactor.ReactorPowerOutput;

      connect(pointKineticReactor.ReactorPowerOutput, RPower)  annotation (Line(points={{-48.5,18.5},{-34,18.5},{-34,56},{-14,56}}, color={0,0,127}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)));
    end Bootstrap008;

    model Bootstrap006
      "unit initialisation testing reactor circuit with heat exchanger"
        // (c) 2016 Dr.Sdl
        // Very first attempt to construct a circuit with heat exchanger and with the
        // "FuelSaltLiFBeF2" fluid in order to find suitable initial parameters
        // for the planned molten salt reactor circuit
        //
        replaceable package Medium =
           Nuclear.FuelSaltLiFBeF2 (final
           singleState=true, final enthalpyOfT=true);

       inner Modelica.Fluid.System system(
          m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        m_flow_start=0.05,
        T_ambient=323.15,
        T_start=923.15)     annotation (Placement(transformation(extent={{-140,120},
                {-120,140}},rotation=0)));

      Modelica.Fluid.Vessels.OpenTank tank(
        redeclare package Medium = Medium,
        nPorts=2,
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        ports(each p(start=110000)),
        height=4,
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
            Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        crossArea=3.00,
        m_flow_nominal=0.005,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        level_start=0.5)
                  annotation (Placement(transformation(extent={{-126,32},{-106,52}},
              rotation=0)));
      Modelica.Fluid.Machines.ControlledPump pump(
        redeclare package Medium = Medium,
        N_nominal=1500,
        use_T_start=true,
        control_m_flow=false,
        allowFlowReversal=false,
        use_p_set=true,
        m_flow_start=system.m_flow_start,
        m_flow_nominal=system.m_flow_start,
        p_a_start=110000,
        p_b_start=130000,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        p_a_nominal=110000,
        p_b_nominal=130000)
        annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
      Modelica.Fluid.Valves.ValveIncompressible valve(
        redeclare package Medium = Medium,
        CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
        show_T=true,
        allowFlowReversal=false,
        m_flow_nominal=system.m_flow_start,
        dp_start=18000,
        dp_nominal=10000)
        annotation (Placement(transformation(extent={{116,-100},{96,-80}},
                                                                       rotation=0)));

    public
      Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow(redeclare package
          Medium =                                                                 Medium)
        annotation (Placement(transformation(extent={{-12,10},{8,30}}, rotation=0)));
      Modelica.Blocks.Sources.Step handle(
        height=0.9,
        startTime=20,
        offset=0.01)  annotation (Placement(transformation(extent={{74,-29},{88,-15}},
                      rotation=0)));

    protected
      Modelica.Blocks.Interfaces.RealOutput m_flow
        annotation (Placement(transformation(extent={{-4,40},{8,52}},   rotation=
                0)));

    public
      Modelica.Blocks.Sources.Ramp PumpRamp(
        height=200000,
        duration=100,
        offset=130000,
        startTime=9) annotation (Placement(transformation(extent={{-80,54},{-60,74}})));
      Modelica.Fluid.Examples.HeatExchanger.BaseClasses.BasicHX basicHX(
        crossArea_1=0.0314,
        crossArea_2=0.0314,
        perimeter_1=0.628,
        perimeter_2=0.628,
        s_wall=0.005,
        c_wall=500,
        rho_wall=900,
        m_flow_start_1=1,
        k_wall=100,
        m_flow_start_2=0.00076,
        redeclare package Medium_1 = Medium,
        redeclare package Medium_2 = Medium,
        length=10.1,
        area_h_1=0.628*10.1,
        area_h_2=0.628*10.1,
        Twall_start=323.15,
        dT=274.15,
        p_a_start1=110000,
        p_b_start1=110000,
        T_start_1=923.15,
        p_a_start2=120000,
        p_b_start2=120000,
        T_start_2=923.15)
        annotation (Placement(transformation(extent={{-18,-96},{2,-76}})));
      Modelica.Fluid.Sources.Boundary_pT boundary(
        redeclare package Medium = Medium,
        nPorts=1,
        p=100000,
        T=923.15) annotation (Placement(transformation(extent={{-14,-54},{6,-34}})));
      Modelica.Fluid.Sources.MassFlowSource_T boundary1(
        redeclare package Medium = Medium,
        nPorts=1,
        m_flow=1,
        T=923.15)
        annotation (Placement(transformation(extent={{-84,-124},{-64,-104}})));
    equation

      connect(sensor_m_flow.m_flow, m_flow)         annotation (Line(points={{-2,31},{-2,46},{2,46}},
                                                    color={0,0,127}));
      connect(handle.y, valve.opening)       annotation (Line(
          points={{88.7,-22},{106,-22},{106,-82}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(pump.port_b, sensor_m_flow.port_a)
                                                annotation (Line(
          points={{-30,20},{-12,20}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(tank.ports[1], pump.port_a)
        annotation (Line(points={{-118,32},{-118,20},{-50,20}},        color={0,127,255}));

      // connect(radiator.port_b, tank.ports[2]);
      connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
              {-35,28.2}}, color={0,0,127}));

      connect(sensor_m_flow.port_b, valve.port_a) annotation (Line(points={{8,20},{74,
              20},{140,20},{140,-90},{116,-90}}, color={0,127,255}));
      connect(valve.port_b, basicHX.port_a2)
        annotation (Line(points={{96,-90},{3,-90},{3,-90.6}}, color={0,127,255}));
      connect(boundary.ports[1], basicHX.port_b1) annotation (Line(points={{6,-44},{
              6,-44},{34,-44},{34,-86.2},{3,-86.2}},  color={0,127,255}));
      connect(boundary1.ports[1], basicHX.port_a1) annotation (Line(points={{-64,-114},
              {-26,-114},{-26,-86.2},{-19,-86.2}}, color={0,127,255}));
      connect(basicHX.port_b2, tank.ports[2]) annotation (Line(points={{-19,-81.4},{
              -114,-81.4},{-114,32}}, color={0,127,255}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)));
    end Bootstrap006;

    model Bootstrap007
      "unit initialisation testing for single molten salt circuit without reactor"
        // (c) 2016 Dr.Sdl
        // this model tests the planned primary circuit components and layout for the
        // "SingleFluidMoltenSaltReactor" with the presence of the "PointKineticMoltenSaltReactor"
        // model in order to find suitable initial parameters for pump, pipes etc. when
        // using the "FuelSaltLiFBeF2" fluid. Constant heat source instead of reactor used.
        //

        replaceable package Medium =
           Nuclear.FuelSaltLiFBeF2 (final
              singleState=true, final enthalpyOfT=true);

       inner Modelica.Fluid.System system(
          m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        T_ambient=923.15,
        m_flow_start=50,
        T_start=923.15)     annotation (Placement(transformation(extent={{-142,120},
                {-122,140}},rotation=0)));

      Modelica.Fluid.Vessels.OpenTank tank(
        redeclare package Medium = Medium,
        nPorts=2,
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        ports(each p(start=110000)),
        height=4,
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
            Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        level_start=0.5,
        crossArea=3.00,
        m_flow_nominal=0.005)
                  annotation (Placement(transformation(extent={{-120,26},{-100,46}},
              rotation=0)));
      Modelica.Fluid.Machines.ControlledPump pump(
        redeclare package Medium = Medium,
        N_nominal=1500,
        use_T_start=true,
        control_m_flow=false,
        allowFlowReversal=false,
        use_p_set=true,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        m_flow_start=system.m_flow_start,
        m_flow_nominal=system.m_flow_start,
        p_a_start=110000,
        p_b_start=130000,
        p_a_nominal=110000,
        p_b_nominal=130000)
        annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
      Modelica.Fluid.Valves.ValveIncompressible valve(
        redeclare package Medium = Medium,
        CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
        show_T=true,
        allowFlowReversal=false,
        m_flow_nominal=system.m_flow_start,
        dp_start=18000,
        dp_nominal=10000)
        annotation (Placement(transformation(extent={{102,-82},{82,-62}},
                                                                       rotation=0)));

      Modelica.Fluid.Pipes.DynamicPipe reactor(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        use_HeatTransfer=true,
        modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        nNodes=16,
        diameter=0.2,
        height_ab=0,
        length=3.9,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        p_a_start=130000) annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                                                                                         rotation=90,
            origin={50,82})));

      Modelica.Fluid.Pipes.DynamicPipe hotleg(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        diameter=0.2,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        length=10.01,
        p_a_start=130000) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=-90,
            origin={122,12})));

      Modelica.Fluid.Pipes.DynamicPipe coldleg(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        diameter=0.2,
        nNodes=1,
        modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        length=10.01,
        p_a_start=110000) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={-112,-20})));

      Modelica.Fluid.Examples.HeatExchanger.BaseClasses.BasicHX HEX(
        c_wall=500,
        s_wall=0.005,
        rho_wall=900,
        redeclare package Medium_1 =
            Medium,
        redeclare package Medium_2 =
            Medium,
        modelStructure_1=Modelica.Fluid.Types.ModelStructure.av_b,
        modelStructure_2=Modelica.Fluid.Types.ModelStructure.a_vb,
        crossArea_1=3.14e-2,
        perimeter_1=0.628,
        T_start_1=Modelica.SIunits.Conversions.from_degC(650),
        T_start_2=Modelica.SIunits.Conversions.from_degC(650),
        k_wall=100,
        crossArea_2=3.14e-2,
        perimeter_2=0.628,
        nNodes=2,
        use_T_start=false,
        Twall_start=Modelica.SIunits.Conversions.from_degC(650),
        m_flow_start_2=system.m_flow_start,
        length=10.0003,
        area_h_1=0.628*10.0003,
        area_h_2=0.628*10.0003,
        redeclare model HeatTransfer_1 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
            (alpha0=2000),
             redeclare model HeatTransfer_2 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
            (alpha0=2000),
        dT=1,
        p_a_start1=100000,
        p_b_start1=100000,
        m_flow_start_1=150,
        p_a_start2=117000,
        p_b_start2=117000)   annotation (Placement(transformation(extent={{-18,-86},
                {12,-56}}, rotation=0)));

      Modelica.Fluid.Sources.Boundary_pT HEXambient(
        nPorts=1,
        redeclare package Medium = Medium,
        T=Modelica.SIunits.Conversions.from_degC(650),
        p=100000) annotation (Placement(transformation(
            extent={{7,-7},{-7,7}},
            rotation=0,
            origin={46,-56})));

      Modelica.Fluid.Sources.MassFlowSource_T HEXmassFlowRate(
        nPorts=1,
        redeclare package Medium = Medium,
        use_T_in=false,
        use_X_in=false,
        use_m_flow_in=false,
        T=Modelica.SIunits.Conversions.from_degC(650),
        m_flow=150)
                  annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=0,
            origin={-42,-86})));

    public
      Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow(redeclare package
          Medium =                                                                 Medium)
        annotation (Placement(transformation(extent={{-12,10},{8,30}}, rotation=0)));
      Modelica.Thermal.HeatTransfer.Sources.FixedHeatFlow burner(
        alpha=-0.005,
        Q_flow(displayUnit="MW") = 20000000,
        T_ref=923.15)
        annotation (Placement(transformation(extent={{-36,72},{-16,92}},
                                                                       rotation=0)));
      Modelica.Fluid.Sensors.Temperature sensor_T_hotleg(redeclare package
          Medium =                                                                  Medium)
        annotation (Placement(transformation(extent={{40,108},{60,128}},
                                                                       rotation=0)));
      Modelica.Fluid.Sensors.Temperature sensor_T_coldleg(redeclare package
          Medium =                                                                   Medium)
        annotation (Placement(transformation(extent={{-42,-56},{-62,-36}}, rotation=0)));
      Modelica.Blocks.Sources.Step handle(
        height=0.9,
        startTime=20,
        offset=0.1)   annotation (Placement(transformation(extent={{74,-29},{88,-15}},
                      rotation=0)));

    protected
      Modelica.Blocks.Interfaces.RealOutput T_hotleg
        annotation (Placement(transformation(extent={{80,112},{92,124}}, rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput T_coldleg
        annotation (Placement(transformation(extent={{-66,-52},{-78,-40}}, rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput tankLevel
                                     annotation (Placement(transformation(extent={{-94,26},{-82,38}},
                                    rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput m_flow
        annotation (Placement(transformation(extent={{-4,40},{8,52}},   rotation=
                0)));

    public
      Modelica.Blocks.Sources.Ramp PumpRamp(
        duration=100,
        offset=130000,
        startTime=9,
        height=145000)
                     annotation (Placement(transformation(extent={{-80,54},{-60,74}})));
    equation
    tankLevel = tank.level;
      connect(sensor_m_flow.m_flow, m_flow)         annotation (Line(points={{-2,31},{-2,46},{2,46}},
                                                    color={0,0,127}));
      connect(sensor_m_flow.port_b, reactor.port_a)
        annotation (Line(points={{8,20},{8,20},{50,20},{50,72}},
                                                         color={0,127,255}));
      connect(sensor_T_hotleg.T, T_hotleg)
        annotation (Line(points={{57,118},{82,118},{86,118}}, color={0,0,127}));
      connect(handle.y, valve.opening)       annotation (Line(
          points={{88.7,-22},{92,-22},{92,-64}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(pump.port_b, sensor_m_flow.port_a)
                                                annotation (Line(
          points={{-30,20},{-12,20}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(sensor_T_coldleg.T, T_coldleg) annotation (Line(
          points={{-59,-46},{-72,-46}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(burner.port, reactor.heatPorts[1]) annotation (Line(
          points={{-16,82},{45.6,82},{45.6,79.1937}},
          color={191,0,0},
          smooth=Smooth.None));
      connect(sensor_T_hotleg.port, reactor.port_b) annotation (Line(
          points={{50,108},{50,92}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(reactor.port_b, hotleg.port_a) annotation (Line(
          points={{50,92},{50,104},{122,104},{122,22}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(hotleg.port_b, valve.port_a) annotation (Line(
          points={{122,2},{122,-72},{102,-72}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(tank.ports[1], pump.port_a)
        annotation (Line(points={{-112,26},{-112,20},{-50,20}},        color={0,127,255}));

      connect(coldleg.port_b, tank.ports[2])
        annotation (Line(points={{-112,-10},{-112,26},{-108,26}},
                                                              color={0,127,255}));

      // connect(radiator.port_b, tank.ports[2]);
      connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
              {-35,28.2}}, color={0,0,127}));
      connect(valve.port_b, HEX.port_a2) annotation (Line(points={{82,-72},{58,-72},
              {58,-77.9},{13.5,-77.9}}, color={0,127,255}));
      connect(HEX.port_b2, coldleg.port_a) annotation (Line(points={{-19.5,-64.1},{-112,
              -64.1},{-112,-30}},      color={0,127,255}));
      connect(sensor_T_coldleg.port, coldleg.port_a) annotation (Line(points={{-52,-56},
              {-52,-64},{-112,-64},{-112,-30}},        color={0,127,255}));
      connect(HEXmassFlowRate.ports[1], HEX.port_a1) annotation (Line(points={{-32,-86},
              {-22,-86},{-22,-71.3},{-19.5,-71.3}},      color={0,127,255}));
      connect(HEX.port_b1, HEXambient.ports[1]) annotation (Line(points={{13.5,-71.3},
              {13.5,-63.65},{39,-63.65},{39,-56}},        color={0,127,255}));

      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)));
    end Bootstrap007;

    model Bootstrap009
      "unit initialisation testing for point reactor component with heat port"
        // (c) 2016 Dr.Sdl
        // this model is a simple test to combine model "PointKineticMoltenSaltReactor" with other
        // components, such as the heat port of a pipe and a "sin" input block.
        //
          replaceable package Medium =
           Nuclear.FuelSaltLiFBeF2 (final
              singleState=true, final enthalpyOfT=true);

       inner Modelica.Fluid.System system(
        m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        T_ambient=923.15,
        m_flow_start=0.5)   annotation (Placement(transformation(extent={{-140,120},
                {-120,140}},rotation=0)));

      PointKineticMoltenSaltReactor pointKineticReactor
        annotation (Placement(transformation(extent={{-68,12},{-48,32}})));

      Modelica.Blocks.Sources.Sine rhos(
        amplitude=50,
        freqHz=1,
        phase=0,
        offset=10,
        startTime=0) annotation (Placement(transformation(extent={{-122,48},{-102,68}})));
    protected
        Modelica.Blocks.Interfaces.RealOutput RPower
        annotation (Placement(transformation(extent={{-16,-28},{4,-8}})));

    public
      Modelica.Thermal.HeatTransfer.Components.HeatCapacitor heatCapacitor(C=1000, T(start=923.15))
        annotation (Placement(transformation(extent={{-26,60},{-6,80}})));
      Modelica.Fluid.Pipes.DynamicPipe pipe(
        redeclare package Medium = Medium,
        length=10,
        diameter=0.2,
        m_flow_start=200,
        use_HeatTransfer=true)              annotation (Placement(transformation(extent={{66,36},{86,56}})));
      Modelica.Fluid.Sources.MassFlowSource_T FixedPump(
        nPorts=1,
        redeclare package Medium = Medium,
        m_flow=200,
        T=923.15) annotation (Placement(transformation(extent={{14,36},{34,56}})));
      Modelica.Fluid.Sources.FixedBoundary PressureBoundary(
        redeclare package Medium = Medium,
        nPorts=1,
        p=110000) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={138,46})));
    equation
      //pointKineticReactor.rho=0;
      //RPower=pointKineticReactor.N;
      //pointKineticReactor.rho=sin(time);

      connect(rhos.y, pointKineticReactor.rho) annotation (Line(points={{-101,58},{-86,58},{-78,58},{-78,29},{-67,29}}, color={0,0,127}));
      //RPower=pointKineticReactor.ReactorPowerOutput;

      connect(pointKineticReactor.ReactorPowerOutput, RPower)  annotation (Line(points={{-48.5,18.5},{-34,18.5},{-34,-18},{-6,-18}},color={0,0,127}));
      connect(pointKineticReactor.port_b, heatCapacitor.port) annotation (Line(points={{-49,29},{-16,29},{-16,60}}, color={191,0,0}));
      connect(FixedPump.ports[1], pipe.port_a) annotation (Line(points={{34,46},{52,46},{66,46}}, color={0,127,255}));
      connect(pipe.port_b, PressureBoundary.ports[1]) annotation (Line(points={{86,46},{128,46}}, color={0,127,255}));
      connect(pointKineticReactor.port_b, pipe.heatPorts[1])
        annotation (Line(points={{-49,29},{6,29},{6,92},{74.55,92},{74.55,50.4}}, color={191,0,0}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)));
    end Bootstrap009;

    model Bootstrap010
      "molten salt reactor circuit with fuel and coolant as single fluid"
        // (c) 2016 Dr.Sdl
        // Very first attempt to make the reactor fuel salt circuit work in combination with
        // the "PointKineticMoltenSaltReactor_P1" model. Since there is no fuel temperature feedback
        // and because the HEX secondary side temperature is forced constant only very little
        // power can be generated without crashing the system. Since the reactor is deeply
        // sub-critical at the beginning the inital guess for the reactor power inherent in
        // the "PointKineticMoltenSaltReactor_P1" model quickly decreases towards zero. By means
        // of block "ramp" the reactor is made swiftly critical and an excess reactivity of 20pcm
        // is reached at 30s. Without temperature feedback the reactor power is then exponentially
        // increasing. The primary circuit is also swiftly heating up because almost no heat
        // can be transferred in the heat exchanger yet.
        //
        replaceable package Medium =
           Nuclear.FuelSaltLiFBeF2 (final
              singleState=true, final enthalpyOfT=true);

       inner Modelica.Fluid.System system(
          m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        T_ambient=923.15,
        m_flow_start=50,
        T_start=923.15)     annotation (Placement(transformation(extent={{-142,120},
                {-122,140}},rotation=0)));

      Modelica.Fluid.Vessels.OpenTank tank(
        redeclare package Medium = Medium,
        nPorts=2,
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        ports(each p(start=110000)),
        height=4,
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
            Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        level_start=0.5,
        crossArea=3.00,
        m_flow_nominal=0.005)
                  annotation (Placement(transformation(extent={{-120,26},{-100,46}},
              rotation=0)));
      Modelica.Fluid.Machines.ControlledPump pump(
        redeclare package Medium = Medium,
        N_nominal=1500,
        use_T_start=true,
        control_m_flow=false,
        allowFlowReversal=false,
        use_p_set=true,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        m_flow_start=system.m_flow_start,
        m_flow_nominal=system.m_flow_start,
        p_a_start=110000,
        p_b_start=130000,
        p_a_nominal=110000,
        p_b_nominal=130000)
        annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
      Modelica.Fluid.Valves.ValveIncompressible valve(
        redeclare package Medium = Medium,
        CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
        show_T=true,
        allowFlowReversal=false,
        m_flow_nominal=system.m_flow_start,
        dp_start=18000,
        dp_nominal=10000)
        annotation (Placement(transformation(extent={{102,-82},{82,-62}},
                                                                       rotation=0)));

      Modelica.Fluid.Pipes.DynamicPipe reactor(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        use_HeatTransfer=true,
        modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        nNodes=16,
        diameter=0.2,
        height_ab=0,
        length=3.9,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        p_a_start=130000) annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                                                                                         rotation=90,
            origin={50,82})));

      Modelica.Fluid.Pipes.DynamicPipe hotleg(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        diameter=0.2,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        length=10.01,
        p_a_start=130000) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=-90,
            origin={122,12})));

      Modelica.Fluid.Pipes.DynamicPipe coldleg(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        diameter=0.2,
        nNodes=1,
        modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        length=10.01,
        p_a_start=110000) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={-112,-20})));

      Modelica.Fluid.Examples.HeatExchanger.BaseClasses.BasicHX HEX(
        c_wall=500,
        s_wall=0.005,
        rho_wall=900,
        redeclare package Medium_1 =
            Medium,
        redeclare package Medium_2 =
            Medium,
        modelStructure_1=Modelica.Fluid.Types.ModelStructure.av_b,
        modelStructure_2=Modelica.Fluid.Types.ModelStructure.a_vb,
        crossArea_1=3.14e-2,
        perimeter_1=0.628,
        T_start_1=Modelica.SIunits.Conversions.from_degC(650),
        T_start_2=Modelica.SIunits.Conversions.from_degC(650),
        k_wall=100,
        crossArea_2=3.14e-2,
        perimeter_2=0.628,
        nNodes=2,
        use_T_start=false,
        Twall_start=Modelica.SIunits.Conversions.from_degC(650),
        m_flow_start_2=system.m_flow_start,
        length=10.0003,
        area_h_1=0.628*10.0003,
        area_h_2=0.628*10.0003,
        redeclare model HeatTransfer_1 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
            (alpha0=2000),
             redeclare model HeatTransfer_2 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
            (alpha0=2000),
        dT=1,
        p_a_start1=100000,
        p_b_start1=100000,
        m_flow_start_1=150,
        p_a_start2=117000,
        p_b_start2=117000)   annotation (Placement(transformation(extent={{-18,-86},
                {12,-56}}, rotation=0)));

      Modelica.Fluid.Sources.Boundary_pT HEXambient(
        nPorts=1,
        redeclare package Medium = Medium,
        T=Modelica.SIunits.Conversions.from_degC(650),
        p=100000) annotation (Placement(transformation(
            extent={{7,-7},{-7,7}},
            rotation=0,
            origin={46,-56})));

      Modelica.Fluid.Sources.MassFlowSource_T HEXmassFlowRate(
        nPorts=1,
        redeclare package Medium = Medium,
        use_T_in=false,
        use_X_in=false,
        use_m_flow_in=false,
        T=Modelica.SIunits.Conversions.from_degC(650),
        m_flow=150)
                  annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=0,
            origin={-42,-86})));

    public
      Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow(redeclare package
          Medium =                                                                 Medium)
        annotation (Placement(transformation(extent={{-12,10},{8,30}}, rotation=0)));
      Modelica.Fluid.Sensors.Temperature sensor_T_hotleg(redeclare package
          Medium =                                                                  Medium)
        annotation (Placement(transformation(extent={{40,108},{60,128}},
                                                                       rotation=0)));
      Modelica.Fluid.Sensors.Temperature sensor_T_coldleg(redeclare package
          Medium =                                                                   Medium)
        annotation (Placement(transformation(extent={{-42,-56},{-62,-36}}, rotation=0)));
      Modelica.Blocks.Sources.Step handle(
        height=0.9,
        startTime=20,
        offset=0.1)   annotation (Placement(transformation(extent={{74,-29},{88,-15}},
                      rotation=0)));

    protected
      Modelica.Blocks.Interfaces.RealOutput T_hotleg
        annotation (Placement(transformation(extent={{80,112},{92,124}}, rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput T_coldleg
        annotation (Placement(transformation(extent={{-66,-52},{-78,-40}}, rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput tankLevel
                                     annotation (Placement(transformation(extent={{-94,26},{-82,38}},
                                    rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput m_flow
        annotation (Placement(transformation(extent={{-4,40},{8,52}},   rotation=
                0)));
      Modelica.Blocks.Interfaces.RealOutput r_power
      annotation (Placement(transformation(extent={{20,60},{30,70}})));

    public
      Modelica.Blocks.Sources.Ramp PumpRamp(
        duration=100,
        offset=130000,
        startTime=9,
        height=145000)
                     annotation (Placement(transformation(extent={{-80,54},{-60,74}})));

      Modelica.Thermal.HeatTransfer.Components.HeatCapacitor ReactorHeatStructure(C=100000, T(start=923.15))
        "heatCapacitor"
        annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={-6,124})));

      PointKineticMoltenSaltReactor_P1 pointKineticMoltenSaltReactor annotation (Placement(transformation(extent={{-14,82},{6,102}})));

      Modelica.Blocks.Math.Add add(k1=1.0, k2=1.0) annotation (Placement(transformation(extent={{-50,100},{-30,120}})));
      Modelica.Blocks.Sources.Ramp ramp(
        startTime=20,
        height=120,
        duration=10) annotation (Placement(transformation(extent={{-104,94},{-84,114}})));
      Modelica.Blocks.Sources.Constant rho0(k=-100) annotation (Placement(transformation(extent={{-104,124},{-84,144}})));
      Modelica.Blocks.Sources.Constant const(k=0) "NULL" annotation (Placement(transformation(extent={{-136,66},{-116,86}})));
    equation
    tankLevel = tank.level;
      connect(sensor_m_flow.m_flow, m_flow)         annotation (Line(points={{-2,31},{-2,46},{2,46}},
                                                    color={0,0,127}));
      connect(sensor_m_flow.port_b, reactor.port_a)
        annotation (Line(points={{8,20},{8,20},{50,20},{50,72}},
                                                         color={0,127,255}));
      connect(sensor_T_hotleg.T, T_hotleg)
        annotation (Line(points={{57,118},{82,118},{86,118}}, color={0,0,127}));
      connect(handle.y, valve.opening)       annotation (Line(
          points={{88.7,-22},{92,-22},{92,-64}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(pump.port_b, sensor_m_flow.port_a)
                                                annotation (Line(
          points={{-30,20},{-12,20}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(sensor_T_coldleg.T, T_coldleg) annotation (Line(
          points={{-59,-46},{-72,-46}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(sensor_T_hotleg.port, reactor.port_b) annotation (Line(
          points={{50,108},{50,92}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(reactor.port_b, hotleg.port_a) annotation (Line(
          points={{50,92},{50,104},{122,104},{122,22}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(hotleg.port_b, valve.port_a) annotation (Line(
          points={{122,2},{122,-72},{102,-72}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(tank.ports[1], pump.port_a)
        annotation (Line(points={{-112,26},{-112,20},{-50,20}},        color={0,127,255}));

      connect(coldleg.port_b, tank.ports[2])
        annotation (Line(points={{-112,-10},{-112,26},{-108,26}},
                                                              color={0,127,255}));

      // connect(radiator.port_b, tank.ports[2]);
      connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
              {-35,28.2}}, color={0,0,127}));
      connect(valve.port_b, HEX.port_a2) annotation (Line(points={{82,-72},{58,-72},
              {58,-77.9},{13.5,-77.9}}, color={0,127,255}));
      connect(HEX.port_b2, coldleg.port_a) annotation (Line(points={{-19.5,-64.1},{-112,
              -64.1},{-112,-30}},      color={0,127,255}));
      connect(sensor_T_coldleg.port, coldleg.port_a) annotation (Line(points={{-52,-56},
              {-52,-64},{-112,-64},{-112,-30}},        color={0,127,255}));
      connect(HEXmassFlowRate.ports[1], HEX.port_a1) annotation (Line(points={{-32,-86},
              {-22,-86},{-22,-71.3},{-19.5,-71.3}},      color={0,127,255}));
      connect(HEX.port_b1, HEXambient.ports[1]) annotation (Line(points={{13.5,-71.3},
              {13.5,-63.65},{39,-63.65},{39,-56}},        color={0,127,255}));

      connect(ReactorHeatStructure.port, reactor.heatPorts[2])
        annotation (Line(points={{4,124},{38,124},{38,79.5813},{45.6,79.5813}}, color={191,0,0}));

      connect(pointKineticMoltenSaltReactor.port_b, reactor.heatPorts[1])
        annotation (Line(points={{5,99},{26,99},{26,79.1937},{45.6,79.1937}}, color={191,0,0}));
      connect(pointKineticMoltenSaltReactor.ReactorPowerOutput, r_power)
        annotation (Line(points={{5.5,88.5},{10,88.5},{10,65},{25,65}}, color={0,0,127}));
      connect(add.y, pointKineticMoltenSaltReactor.rho)
        annotation (Line(points={{-29,110},{-26,110},{-26,99},{-13,99}}, color={0,0,127}));
      connect(ramp.y, add.u2) annotation (Line(points={{-83,104},{-52,104}}, color={0,0,127}));
      connect(rho0.y, add.u1) annotation (Line(points={{-83,134},{-83,134},{-60,134},{-60,116},{-52,116}}, color={0,0,127}));
      connect(const.y, pointKineticMoltenSaltReactor.rho_control)
        annotation (Line(points={{-115,76},{-94,76},{-94,86},{-58,86},{-58,87.5},{-13,87.5}}, color={0,0,127}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,240}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,240}},
            initialScale=0.1)));
    end Bootstrap010;

    model Bootstrap011
      "molten salt reactor circuit with fuel and coolant as single fluid"
        // (c) 2016 Dr.Sdl
        // This was the first step to make "SingleFluidMoltenSaltReactor" work with
        // active fuel salt temperature feedback and in order to find a suitable
        // set of parameters to initialize the system by a combination of
        // "Tfuel ramp", "HEXT ramp" and ordinary reactivity block "ramp"
        //
        replaceable package Medium =
           Nuclear.FuelSaltLiFBeF2 (final
              singleState=true, final enthalpyOfT=true);

       inner Modelica.Fluid.System system(
          m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        T_ambient=923.15,
        m_flow_start=50,
        T_start=923.15)     annotation (Placement(transformation(extent={{-142,120},
                {-122,140}},rotation=0)));

      Modelica.Fluid.Vessels.OpenTank tank(
        redeclare package Medium = Medium,
        nPorts=2,
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        ports(each p(start=110000)),
        height=4,
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
            Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        level_start=0.5,
        crossArea=3.00,
        m_flow_nominal=0.005)
                  annotation (Placement(transformation(extent={{-120,26},{-100,46}},
              rotation=0)));
      Modelica.Fluid.Machines.ControlledPump pump(
        redeclare package Medium = Medium,
        N_nominal=1500,
        use_T_start=true,
        control_m_flow=false,
        allowFlowReversal=false,
        use_p_set=true,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        m_flow_start=system.m_flow_start,
        m_flow_nominal=system.m_flow_start,
        p_a_start=110000,
        p_b_start=130000,
        p_a_nominal=110000,
        p_b_nominal=130000)
        annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
      Modelica.Fluid.Valves.ValveIncompressible valve(
        redeclare package Medium = Medium,
        CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
        show_T=true,
        allowFlowReversal=false,
        m_flow_nominal=system.m_flow_start,
        dp_start=18000,
        dp_nominal=10000)
        annotation (Placement(transformation(extent={{102,-82},{82,-62}},
                                                                       rotation=0)));

      Modelica.Fluid.Pipes.DynamicPipe reactor(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        use_HeatTransfer=true,
        modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        nNodes=16,
        diameter=0.2,
        height_ab=0,
        length=3.9,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        p_a_start=130000) annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                                                                                         rotation=90,
            origin={50,82})));

      Modelica.Fluid.Pipes.DynamicPipe hotleg(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        diameter=0.2,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        length=10.01,
        p_a_start=130000) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=-90,
            origin={122,12})));

      Modelica.Fluid.Pipes.DynamicPipe coldleg(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        diameter=0.2,
        nNodes=1,
        modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        length=10.01,
        p_a_start=110000) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={-112,-20})));

      Modelica.Fluid.Examples.HeatExchanger.BaseClasses.BasicHX HEX(
        c_wall=500,
        s_wall=0.005,
        rho_wall=900,
        redeclare package Medium_1 =
            Medium,
        redeclare package Medium_2 =
            Medium,
        modelStructure_1=Modelica.Fluid.Types.ModelStructure.av_b,
        modelStructure_2=Modelica.Fluid.Types.ModelStructure.a_vb,
        crossArea_1=3.14e-2,
        perimeter_1=0.628,
        T_start_1=Modelica.SIunits.Conversions.from_degC(610),
        T_start_2=Modelica.SIunits.Conversions.from_degC(650),
        k_wall=100,
        crossArea_2=3.14e-2,
        perimeter_2=0.628,
        nNodes=2,
        use_T_start=false,
        Twall_start=Modelica.SIunits.Conversions.from_degC(625),
        m_flow_start_2=system.m_flow_start,
        redeclare model HeatTransfer_1 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
            (alpha0=2000),
             redeclare model HeatTransfer_2 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
            (alpha0=2000),
        m_flow_start_1=150,
        length=11.0003,
        area_h_1=0.628*11.0003,
        area_h_2=0.628*11.0003,
        dT=1,
        p_a_start1=100000,
        p_b_start1=100000,
        p_a_start2=117000,
        p_b_start2=117000)   annotation (Placement(transformation(extent={{-18,-86},
                {12,-56}}, rotation=0)));

      Modelica.Fluid.Sources.Boundary_pT HEXambient(
        nPorts=1,
        redeclare package Medium = Medium,
        T=Modelica.SIunits.Conversions.from_degC(610),
        p=100000) annotation (Placement(transformation(
            extent={{7,-7},{-7,7}},
            rotation=0,
            origin={46,-56})));

      Modelica.Fluid.Sources.MassFlowSource_T HEXmassFlowRate(
        nPorts=1,
        redeclare package Medium = Medium,
        use_X_in=false,
        use_m_flow_in=false,
        T=Modelica.SIunits.Conversions.from_degC(610),
        m_flow=150,
        use_T_in=true)
                  annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=0,
            origin={-44,-86})));

    public
      Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow(redeclare package
          Medium =                                                                 Medium)
        annotation (Placement(transformation(extent={{-12,10},{8,30}}, rotation=0)));
      Modelica.Fluid.Sensors.Temperature sensor_T_hotleg(redeclare package
          Medium =                                                                  Medium)
        annotation (Placement(transformation(extent={{40,108},{60,128}},
                                                                       rotation=0)));
      Modelica.Fluid.Sensors.Temperature sensor_T_coldleg(redeclare package
          Medium =                                                                   Medium)
        annotation (Placement(transformation(extent={{-42,-56},{-62,-36}}, rotation=0)));
      Modelica.Blocks.Sources.Step handle(
        height=0.9,
        startTime=20,
        offset=0.1)   annotation (Placement(transformation(extent={{74,-29},{88,-15}},
                      rotation=0)));

    protected
      Modelica.Blocks.Interfaces.RealOutput T_hotleg
        annotation (Placement(transformation(extent={{80,112},{92,124}}, rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput T_coldleg
        annotation (Placement(transformation(extent={{-66,-52},{-78,-40}}, rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput tankLevel
                                     annotation (Placement(transformation(extent={{-94,26},{-82,38}},
                                    rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput m_flow
        annotation (Placement(transformation(extent={{-4,40},{8,52}},   rotation=
                0)));
      Modelica.Blocks.Interfaces.RealOutput r_power
      annotation (Placement(transformation(extent={{20,60},{30,70}})));

    public
      Modelica.Blocks.Sources.Ramp PumpRamp(
        duration=100,
        offset=130000,
        startTime=9,
        height=145000)
                     annotation (Placement(transformation(extent={{-80,54},{-60,74}})));

      Modelica.Thermal.HeatTransfer.Components.HeatCapacitor ReactorHeatStructure(C=100000, T(start=923.15))
        "heatCapacitor"
        annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={-6,124})));

      PointKineticMoltenSaltReactor_P1 pointKineticMoltenSaltReactor annotation (Placement(transformation(extent={{-14,82},{6,102}})));

      Modelica.Blocks.Sources.Ramp ramp(
        startTime=20,
        duration=100,
        height=2.2)  annotation (Placement(transformation(extent={{-104,94},{-84,114}})));
      Modelica.Blocks.Sources.Constant rho0(k=-2)   annotation (Placement(transformation(extent={{-104,124},{-84,144}})));
      Modelica.Blocks.Math.Add add1
        annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={-12,178})));
      Modelica.Blocks.Math.Gain Tfuel_rho_feedback(k=-9.5)
        "reactivity feedback in pcm/K"
        annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={-46,178})));
      Modelica.Blocks.Math.Add3 add3_1 annotation (Placement(transformation(extent={{-64,88},{-44,108}})));
      Modelica.Blocks.Sources.Ramp Tfuel_reference(
        offset=-890,
        duration=300,
        startTime=100,
        height=-11)    annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={30,162})));
      Modelica.Blocks.Sources.Ramp HEXT_ramp(
        duration=100,
        startTime=20,
        offset=883,
        height=-200) annotation (Placement(transformation(extent={{-104,-116},{-84,-96}})));
      Modelica.Blocks.Sources.Constant const(k=0) "NULL" annotation (Placement(transformation(extent={{-132,66},{-112,86}})));
    equation
    tankLevel = tank.level;
      connect(sensor_m_flow.m_flow, m_flow)         annotation (Line(points={{-2,31},{-2,46},{2,46}},
                                                    color={0,0,127}));
      connect(sensor_m_flow.port_b, reactor.port_a)
        annotation (Line(points={{8,20},{8,20},{50,20},{50,72}},
                                                         color={0,127,255}));
      connect(sensor_T_hotleg.T, T_hotleg)
        annotation (Line(points={{57,118},{82,118},{86,118}}, color={0,0,127}));
      connect(handle.y, valve.opening)       annotation (Line(
          points={{88.7,-22},{92,-22},{92,-64}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(pump.port_b, sensor_m_flow.port_a)
                                                annotation (Line(
          points={{-30,20},{-12,20}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(sensor_T_coldleg.T, T_coldleg) annotation (Line(
          points={{-59,-46},{-72,-46}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(sensor_T_hotleg.port, reactor.port_b) annotation (Line(
          points={{50,108},{50,92}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(reactor.port_b, hotleg.port_a) annotation (Line(
          points={{50,92},{50,104},{122,104},{122,22}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(hotleg.port_b, valve.port_a) annotation (Line(
          points={{122,2},{122,-72},{102,-72}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(tank.ports[1], pump.port_a)
        annotation (Line(points={{-112,26},{-112,20},{-50,20}},        color={0,127,255}));

      connect(coldleg.port_b, tank.ports[2])
        annotation (Line(points={{-112,-10},{-112,26},{-108,26}},
                                                              color={0,127,255}));

      // connect(radiator.port_b, tank.ports[2]);
      connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
              {-35,28.2}}, color={0,0,127}));
      connect(valve.port_b, HEX.port_a2) annotation (Line(points={{82,-72},{58,-72},
              {58,-77.9},{13.5,-77.9}}, color={0,127,255}));
      connect(HEX.port_b2, coldleg.port_a) annotation (Line(points={{-19.5,-64.1},{-112,
              -64.1},{-112,-30}},      color={0,127,255}));
      connect(sensor_T_coldleg.port, coldleg.port_a) annotation (Line(points={{-52,-56},
              {-52,-64},{-112,-64},{-112,-30}},        color={0,127,255}));
      connect(HEXmassFlowRate.ports[1], HEX.port_a1) annotation (Line(points={{-34,-86},{-22,-86},{-22,-71.3},{-19.5,-71.3}},
                                                         color={0,127,255}));
      connect(HEX.port_b1, HEXambient.ports[1]) annotation (Line(points={{13.5,-71.3},
              {13.5,-63.65},{39,-63.65},{39,-56}},        color={0,127,255}));

      connect(ReactorHeatStructure.port, reactor.heatPorts[2])
        annotation (Line(points={{4,124},{38,124},{38,79.5813},{45.6,79.5813}}, color={191,0,0}));

      connect(pointKineticMoltenSaltReactor.port_b, reactor.heatPorts[1])
        annotation (Line(points={{5,99},{26,99},{26,79.1937},{45.6,79.1937}}, color={191,0,0}));
      connect(pointKineticMoltenSaltReactor.ReactorPowerOutput, r_power)
        annotation (Line(points={{5.5,88.5},{10,88.5},{10,65},{25,65}}, color={0,0,127}));
      connect(sensor_T_hotleg.T, add1.u2) annotation (Line(points={{57,118},{58,118},{58,176},{58,184},{0,184}}, color={0,0,127}));
      connect(Tfuel_rho_feedback.u, add1.y) annotation (Line(points={{-34,178},{-32,178},{-23,178}}, color={0,0,127}));
      connect(ramp.y, add3_1.u3) annotation (Line(points={{-83,104},{-76,104},{-76,90},{-66,90}}, color={0,0,127}));
      connect(rho0.y, add3_1.u2) annotation (Line(points={{-83,134},{-74,134},{-74,98},{-66,98}}, color={0,0,127}));
      connect(Tfuel_rho_feedback.y, add3_1.u1) annotation (Line(points={{-57,178},{-70,178},{-70,106},{-66,106}}, color={0,0,127}));
      connect(pointKineticMoltenSaltReactor.rho, add3_1.y)
        annotation (Line(points={{-13,99},{-31.5,99},{-31.5,98},{-43,98}}, color={0,0,127}));
      connect(Tfuel_reference.y, add1.u1) annotation (Line(points={{19,162},{19,162},{14,162},{14,172},{0,172}}, color={0,0,127}));
      connect(HEXT_ramp.y, HEXmassFlowRate.T_in) annotation (Line(points={{-83,-106},{-78,-106},{-78,-82},{-56,-82}}, color={0,0,127}));
      connect(const.y, pointKineticMoltenSaltReactor.rho_control)
        annotation (Line(points={{-111,76},{-102,76},{-92,76},{-92,82},{-26,82},{-26,87.5},{-13,87.5}}, color={0,0,127}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,200}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,200}},
            initialScale=0.1)));
    end Bootstrap011;

    model Bootstrap012
      "molten salt reactor circuit with fuel and coolant as single fluid"
        // (c) 2016 Dr.Sdl
        // Same as SingleFluidMoltenSalt Reactor model to test subcooled fuel salt transient
        //
        replaceable package Medium =
           Nuclear.FuelSaltLiFBeF2 (final
              singleState=true, final enthalpyOfT=true);

       inner Modelica.Fluid.System system(
          m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        T_ambient=923.15,
        m_flow_start=50,
        T_start=923.15)     annotation (Placement(transformation(extent={{-142,120},
                {-122,140}},rotation=0)));

      Modelica.Fluid.Vessels.OpenTank tank(
        redeclare package Medium = Medium,
        nPorts=2,
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        ports(each p(start=110000)),
        height=4,
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
            Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        level_start=0.5,
        crossArea=3.00,
        m_flow_nominal=0.005)
                  annotation (Placement(transformation(extent={{-120,26},{-100,46}},
              rotation=0)));
      Modelica.Fluid.Machines.ControlledPump pump(
        redeclare package Medium = Medium,
        N_nominal=1500,
        use_T_start=true,
        control_m_flow=false,
        allowFlowReversal=false,
        use_p_set=true,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        m_flow_start=system.m_flow_start,
        m_flow_nominal=system.m_flow_start,
        p_a_start=110000,
        p_b_start=130000,
        p_a_nominal=110000,
        p_b_nominal=130000)
        annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
      Modelica.Fluid.Valves.ValveIncompressible valve(
        redeclare package Medium = Medium,
        CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
        show_T=true,
        allowFlowReversal=false,
        m_flow_nominal=system.m_flow_start,
        dp_start=18000,
        dp_nominal=10000)
        annotation (Placement(transformation(extent={{102,-82},{82,-62}},
                                                                       rotation=0)));

      Modelica.Fluid.Pipes.DynamicPipe reactor(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        use_HeatTransfer=true,
        modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        nNodes=16,
        diameter=0.2,
        height_ab=0,
        length=3.9,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        p_a_start=130000) annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                                                                                         rotation=90,
            origin={50,82})));

      Modelica.Fluid.Pipes.DynamicPipe hotleg(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        diameter=0.2,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        length=10.01,
        p_a_start=130000) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=-90,
            origin={122,12})));

      Modelica.Fluid.Pipes.DynamicPipe coldleg(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        diameter=0.2,
        nNodes=1,
        modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        length=10.01,
        p_a_start=110000) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={-112,-20})));

      Modelica.Fluid.Examples.HeatExchanger.BaseClasses.BasicHX HEX(
        c_wall=500,
        s_wall=0.005,
        rho_wall=900,
        redeclare package Medium_1 =
            Medium,
        redeclare package Medium_2 =
            Medium,
        modelStructure_1=Modelica.Fluid.Types.ModelStructure.av_b,
        modelStructure_2=Modelica.Fluid.Types.ModelStructure.a_vb,
        crossArea_1=3.14e-2,
        perimeter_1=0.628,
        T_start_1=Modelica.SIunits.Conversions.from_degC(610),
        T_start_2=Modelica.SIunits.Conversions.from_degC(650),
        k_wall=100,
        crossArea_2=3.14e-2,
        perimeter_2=0.628,
        nNodes=2,
        use_T_start=false,
        Twall_start=Modelica.SIunits.Conversions.from_degC(625),
        m_flow_start_2=system.m_flow_start,
        redeclare model HeatTransfer_1 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
            (alpha0=2000),
             redeclare model HeatTransfer_2 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
            (alpha0=2000),
        m_flow_start_1=150,
        length=11.0003,
        area_h_1=0.628*11.0003,
        area_h_2=0.628*11.0003,
        dT=1,
        p_a_start1=100000,
        p_b_start1=100000,
        p_a_start2=117000,
        p_b_start2=117000)   annotation (Placement(transformation(extent={{-18,-86},
                {12,-56}}, rotation=0)));

      Modelica.Fluid.Sources.Boundary_pT HEXambient(
        nPorts=1,
        redeclare package Medium = Medium,
        T=Modelica.SIunits.Conversions.from_degC(610),
        p=100000) annotation (Placement(transformation(
            extent={{7,-7},{-7,7}},
            rotation=0,
            origin={46,-56})));

      Modelica.Fluid.Sources.MassFlowSource_T HEXmassFlowRate(
        nPorts=1,
        redeclare package Medium = Medium,
        use_X_in=false,
        use_m_flow_in=false,
        T=Modelica.SIunits.Conversions.from_degC(610),
        m_flow=150,
        use_T_in=true)
                  annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=0,
            origin={-38,-86})));

    public
      Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow(redeclare package
          Medium =                                                                 Medium)
        annotation (Placement(transformation(extent={{-12,10},{8,30}}, rotation=0)));
      Modelica.Fluid.Sensors.Temperature sensor_T_hotleg(redeclare package
          Medium =                                                                  Medium)
        annotation (Placement(transformation(extent={{40,108},{60,128}},
                                                                       rotation=0)));
      Modelica.Fluid.Sensors.Temperature sensor_T_coldleg(redeclare package
          Medium =                                                                   Medium)
        annotation (Placement(transformation(extent={{-42,-56},{-62,-36}}, rotation=0)));
      Modelica.Blocks.Sources.Step handle(
        height=0.9,
        startTime=20,
        offset=0.1)   annotation (Placement(transformation(extent={{74,-29},{88,-15}},
                      rotation=0)));

    protected
      Modelica.Blocks.Interfaces.RealOutput T_hotleg
        annotation (Placement(transformation(extent={{80,112},{92,124}}, rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput T_coldleg
        annotation (Placement(transformation(extent={{-66,-52},{-78,-40}}, rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput tankLevel
                                     annotation (Placement(transformation(extent={{-94,26},{-82,38}},
                                    rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput m_flow
        annotation (Placement(transformation(extent={{-4,40},{8,52}},   rotation=
                0)));
      Modelica.Blocks.Interfaces.RealOutput r_power
      annotation (Placement(transformation(extent={{20,60},{30,70}})));

    public
      Modelica.Blocks.Sources.Ramp PumpRamp(
        duration=100,
        offset=130000,
        startTime=9,
        height=145000)
                     annotation (Placement(transformation(extent={{-80,54},{-60,74}})));

      Modelica.Thermal.HeatTransfer.Components.HeatCapacitor ReactorHeatStructure(C=100000, T(start=923.15))
        "heatCapacitor"
        annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={-6,124})));

      PointKineticMoltenSaltReactor_P1 pointKineticMoltenSaltReactor annotation (Placement(transformation(extent={{-14,82},{6,102}})));

      Modelica.Blocks.Sources.Ramp ramp(
        startTime=20,
        duration=100,
        height=2.2)  annotation (Placement(transformation(extent={{-104,94},{-84,114}})));
      Modelica.Blocks.Sources.Constant rho0(k=-2)   annotation (Placement(transformation(extent={{-104,124},{-84,144}})));
      Modelica.Blocks.Math.Add add1
        annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={-12,178})));
      Modelica.Blocks.Math.Gain Tfuel_rho_feedback(k=-9.5)
        "reactivity feedback in pcm/K"
        annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={-46,178})));
      Modelica.Blocks.Math.Add3 add3_1 annotation (Placement(transformation(extent={{-64,88},{-44,108}})));
      Modelica.Blocks.Sources.Ramp Tfuel_reference(
        offset=-890,
        duration=300,
        startTime=100,
        height=-11)    annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={30,162})));
      Modelica.Blocks.Sources.Ramp HEXT_ramp(
        duration=100,
        startTime=20,
        offset=883,
        height=-200) annotation (Placement(transformation(extent={{-140,-142},{-120,-122}})));
      Modelica.Blocks.Sources.Constant const(k=0) "NULL" annotation (Placement(transformation(extent={{-132,66},{-112,86}})));
      Modelica.Blocks.Math.Add add annotation (Placement(transformation(extent={{-80,-136},{-60,-116}})));
      Modelica.Blocks.Sources.Ramp subcool(
        height=-100,
        duration=10,
        startTime=700) "subcool transient ramp" annotation (Placement(transformation(extent={{-140,-102},{-120,-82}})));
    equation
    tankLevel = tank.level;
      connect(sensor_m_flow.m_flow, m_flow)         annotation (Line(points={{-2,31},{-2,46},{2,46}},
                                                    color={0,0,127}));
      connect(sensor_m_flow.port_b, reactor.port_a)
        annotation (Line(points={{8,20},{8,20},{50,20},{50,72}},
                                                         color={0,127,255}));
      connect(sensor_T_hotleg.T, T_hotleg)
        annotation (Line(points={{57,118},{82,118},{86,118}}, color={0,0,127}));
      connect(handle.y, valve.opening)       annotation (Line(
          points={{88.7,-22},{92,-22},{92,-64}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(pump.port_b, sensor_m_flow.port_a)
                                                annotation (Line(
          points={{-30,20},{-12,20}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(sensor_T_coldleg.T, T_coldleg) annotation (Line(
          points={{-59,-46},{-72,-46}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(sensor_T_hotleg.port, reactor.port_b) annotation (Line(
          points={{50,108},{50,92}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(reactor.port_b, hotleg.port_a) annotation (Line(
          points={{50,92},{50,104},{122,104},{122,22}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(hotleg.port_b, valve.port_a) annotation (Line(
          points={{122,2},{122,-72},{102,-72}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(tank.ports[1], pump.port_a)
        annotation (Line(points={{-112,26},{-112,20},{-50,20}},        color={0,127,255}));

      connect(coldleg.port_b, tank.ports[2])
        annotation (Line(points={{-112,-10},{-112,26},{-108,26}},
                                                              color={0,127,255}));

      // connect(radiator.port_b, tank.ports[2]);
      connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
              {-35,28.2}}, color={0,0,127}));
      connect(valve.port_b, HEX.port_a2) annotation (Line(points={{82,-72},{58,-72},
              {58,-77.9},{13.5,-77.9}}, color={0,127,255}));
      connect(HEX.port_b2, coldleg.port_a) annotation (Line(points={{-19.5,-64.1},{-112,
              -64.1},{-112,-30}},      color={0,127,255}));
      connect(sensor_T_coldleg.port, coldleg.port_a) annotation (Line(points={{-52,-56},
              {-52,-64},{-112,-64},{-112,-30}},        color={0,127,255}));
      connect(HEXmassFlowRate.ports[1], HEX.port_a1) annotation (Line(points={{-28,-86},{-22,-86},{-22,-71.3},{-19.5,-71.3}},
                                                         color={0,127,255}));
      connect(HEX.port_b1, HEXambient.ports[1]) annotation (Line(points={{13.5,-71.3},
              {13.5,-63.65},{39,-63.65},{39,-56}},        color={0,127,255}));

      connect(ReactorHeatStructure.port, reactor.heatPorts[2])
        annotation (Line(points={{4,124},{38,124},{38,79.5813},{45.6,79.5813}}, color={191,0,0}));

      connect(pointKineticMoltenSaltReactor.port_b, reactor.heatPorts[1])
        annotation (Line(points={{5,99},{26,99},{26,79.1937},{45.6,79.1937}}, color={191,0,0}));
      connect(pointKineticMoltenSaltReactor.ReactorPowerOutput, r_power)
        annotation (Line(points={{5.5,88.5},{10,88.5},{10,65},{25,65}}, color={0,0,127}));
      connect(sensor_T_hotleg.T, add1.u2) annotation (Line(points={{57,118},{58,118},{58,176},{58,184},{0,184}}, color={0,0,127}));
      connect(Tfuel_rho_feedback.u, add1.y) annotation (Line(points={{-34,178},{-32,178},{-23,178}}, color={0,0,127}));
      connect(ramp.y, add3_1.u3) annotation (Line(points={{-83,104},{-76,104},{-76,90},{-66,90}}, color={0,0,127}));
      connect(rho0.y, add3_1.u2) annotation (Line(points={{-83,134},{-74,134},{-74,98},{-66,98}}, color={0,0,127}));
      connect(Tfuel_rho_feedback.y, add3_1.u1) annotation (Line(points={{-57,178},{-70,178},{-70,106},{-66,106}}, color={0,0,127}));
      connect(pointKineticMoltenSaltReactor.rho, add3_1.y)
        annotation (Line(points={{-13,99},{-31.5,99},{-31.5,98},{-43,98}}, color={0,0,127}));
      connect(Tfuel_reference.y, add1.u1) annotation (Line(points={{19,162},{19,162},{14,162},{14,172},{0,172}}, color={0,0,127}));
      connect(const.y, pointKineticMoltenSaltReactor.rho_control)
        annotation (Line(points={{-111,76},{-102,76},{-92,76},{-92,82},{-26,82},{-26,87.5},{-13,87.5}}, color={0,0,127}));
      connect(HEXT_ramp.y, add.u2) annotation (Line(points={{-119,-132},{-82,-132}}, color={0,0,127}));
      connect(HEXmassFlowRate.T_in, add.y) annotation (Line(points={{-50,-82},{-54,-82},{-54,-126},{-59,-126}}, color={0,0,127}));
      connect(subcool.y, add.u1) annotation (Line(points={{-119,-92},{-90,-92},{-90,-120},{-82,-120}}, color={0,0,127}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,200}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,200}},
            initialScale=0.1)));
    end Bootstrap012;

    model Bootstrap013
      "molten salt reactor circuit with fuel and coolant as single fluid"
        // (c) 2016 Dr.Sdl
        // Same as SingleFluidMoltenSalt Reactor model to test reactivity pulse transient
        // Transient ramp at HEX is inactive
        //
        replaceable package Medium =
           Nuclear.FuelSaltLiFBeF2 (final
              singleState=true, final enthalpyOfT=true);

       inner Modelica.Fluid.System system(
          m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        T_ambient=923.15,
        m_flow_start=50,
        T_start=923.15)     annotation (Placement(transformation(extent={{-142,120},
                {-122,140}},rotation=0)));

      Modelica.Fluid.Vessels.OpenTank tank(
        redeclare package Medium = Medium,
        nPorts=2,
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        ports(each p(start=110000)),
        height=4,
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
            Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        level_start=0.5,
        crossArea=3.00,
        m_flow_nominal=0.005)
                  annotation (Placement(transformation(extent={{-120,26},{-100,46}},
              rotation=0)));
      Modelica.Fluid.Machines.ControlledPump pump(
        redeclare package Medium = Medium,
        N_nominal=1500,
        use_T_start=true,
        control_m_flow=false,
        allowFlowReversal=false,
        use_p_set=true,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        m_flow_start=system.m_flow_start,
        m_flow_nominal=system.m_flow_start,
        p_a_start=110000,
        p_b_start=130000,
        p_a_nominal=110000,
        p_b_nominal=130000)
        annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
      Modelica.Fluid.Valves.ValveIncompressible valve(
        redeclare package Medium = Medium,
        CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
        show_T=true,
        allowFlowReversal=false,
        m_flow_nominal=system.m_flow_start,
        dp_start=18000,
        dp_nominal=10000)
        annotation (Placement(transformation(extent={{102,-82},{82,-62}},
                                                                       rotation=0)));

      Modelica.Fluid.Pipes.DynamicPipe reactor(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        use_HeatTransfer=true,
        modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        nNodes=16,
        diameter=0.2,
        height_ab=0,
        length=3.9,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        p_a_start=130000) annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                                                                                         rotation=90,
            origin={50,82})));

      Modelica.Fluid.Pipes.DynamicPipe hotleg(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        diameter=0.2,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        length=10.01,
        p_a_start=130000) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=-90,
            origin={122,12})));

      Modelica.Fluid.Pipes.DynamicPipe coldleg(
        redeclare package Medium = Medium,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        diameter=0.2,
        nNodes=1,
        modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        length=10.01,
        p_a_start=110000) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={-112,-20})));

      Modelica.Fluid.Examples.HeatExchanger.BaseClasses.BasicHX HEX(
        c_wall=500,
        s_wall=0.005,
        rho_wall=900,
        redeclare package Medium_1 =
            Medium,
        redeclare package Medium_2 =
            Medium,
        modelStructure_1=Modelica.Fluid.Types.ModelStructure.av_b,
        modelStructure_2=Modelica.Fluid.Types.ModelStructure.a_vb,
        crossArea_1=3.14e-2,
        perimeter_1=0.628,
        T_start_1=Modelica.SIunits.Conversions.from_degC(610),
        T_start_2=Modelica.SIunits.Conversions.from_degC(650),
        k_wall=100,
        crossArea_2=3.14e-2,
        perimeter_2=0.628,
        nNodes=2,
        use_T_start=false,
        Twall_start=Modelica.SIunits.Conversions.from_degC(625),
        m_flow_start_2=system.m_flow_start,
        redeclare model HeatTransfer_1 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
            (alpha0=2000),
             redeclare model HeatTransfer_2 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
            (alpha0=2000),
        m_flow_start_1=150,
        length=11.0003,
        area_h_1=0.628*11.0003,
        area_h_2=0.628*11.0003,
        dT=1,
        p_a_start1=100000,
        p_b_start1=100000,
        p_a_start2=117000,
        p_b_start2=117000)   annotation (Placement(transformation(extent={{-18,-86},
                {12,-56}}, rotation=0)));

      Modelica.Fluid.Sources.Boundary_pT HEXambient(
        nPorts=1,
        redeclare package Medium = Medium,
        T=Modelica.SIunits.Conversions.from_degC(610),
        p=100000) annotation (Placement(transformation(
            extent={{7,-7},{-7,7}},
            rotation=0,
            origin={46,-56})));

      Modelica.Fluid.Sources.MassFlowSource_T HEXmassFlowRate(
        nPorts=1,
        redeclare package Medium = Medium,
        use_X_in=false,
        use_m_flow_in=false,
        T=Modelica.SIunits.Conversions.from_degC(610),
        m_flow=150,
        use_T_in=true)
                  annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=0,
            origin={-38,-86})));

    public
      Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow(redeclare package
          Medium =                                                                 Medium)
        annotation (Placement(transformation(extent={{-12,10},{8,30}}, rotation=0)));
      Modelica.Fluid.Sensors.Temperature sensor_T_hotleg(redeclare package
          Medium =                                                                  Medium)
        annotation (Placement(transformation(extent={{40,108},{60,128}},
                                                                       rotation=0)));
      Modelica.Fluid.Sensors.Temperature sensor_T_coldleg(redeclare package
          Medium =                                                                   Medium)
        annotation (Placement(transformation(extent={{-42,-56},{-62,-36}}, rotation=0)));
      Modelica.Blocks.Sources.Step handle(
        height=0.9,
        startTime=20,
        offset=0.1)   annotation (Placement(transformation(extent={{74,-29},{88,-15}},
                      rotation=0)));

    protected
      Modelica.Blocks.Interfaces.RealOutput T_hotleg
        annotation (Placement(transformation(extent={{80,112},{92,124}}, rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput T_coldleg
        annotation (Placement(transformation(extent={{-66,-52},{-78,-40}}, rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput tankLevel
                                     annotation (Placement(transformation(extent={{-94,26},{-82,38}},
                                    rotation=0)));
      Modelica.Blocks.Interfaces.RealOutput m_flow
        annotation (Placement(transformation(extent={{-4,40},{8,52}},   rotation=
                0)));
      Modelica.Blocks.Interfaces.RealOutput r_power
      annotation (Placement(transformation(extent={{20,60},{30,70}})));

    public
      Modelica.Blocks.Sources.Ramp PumpRamp(
        duration=100,
        offset=130000,
        startTime=9,
        height=145000)
                     annotation (Placement(transformation(extent={{-80,54},{-60,74}})));

      Modelica.Thermal.HeatTransfer.Components.HeatCapacitor ReactorHeatStructure(C=100000, T(start=923.15))
        "heatCapacitor"
        annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=90,
            origin={-6,124})));

      PointKineticMoltenSaltReactor_P1 pointKineticMoltenSaltReactor annotation (Placement(transformation(extent={{-16,84},{4,104}})));

      Modelica.Blocks.Sources.Ramp ramp(
        startTime=20,
        duration=100,
        height=2.2)  annotation (Placement(transformation(extent={{-104,94},{-84,114}})));
      Modelica.Blocks.Sources.Constant rho0(k=-2)   annotation (Placement(transformation(extent={{-104,124},{-84,144}})));
      Modelica.Blocks.Math.Add add1
        annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={-12,178})));
      Modelica.Blocks.Math.Gain Tfuel_rho_feedback(k=-9.5)
        "reactivity feedback in pcm/K"
        annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={-46,178})));
      Modelica.Blocks.Math.Add3 add3_1 annotation (Placement(transformation(extent={{-64,88},{-44,108}})));
      Modelica.Blocks.Sources.Ramp Tfuel_reference(
        offset=-890,
        duration=300,
        startTime=100,
        height=-11)    annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=180,
            origin={30,162})));
      Modelica.Blocks.Sources.Ramp HEXT_ramp(
        duration=100,
        startTime=20,
        offset=883,
        height=-200) annotation (Placement(transformation(extent={{-140,-142},{-120,-122}})));
      Modelica.Blocks.Math.Add add annotation (Placement(transformation(extent={{-80,-136},{-60,-116}})));
      Modelica.Blocks.Sources.Ramp subcool(
        duration=10,
        startTime=700,
        height=0) "subcool transient ramp" annotation (Placement(transformation(extent={{-140,-102},{-120,-82}})));
      Modelica.Blocks.Sources.Pulse pulse(
        width=95,
        period=5,
        nperiod=1,
        startTime=700,
        amplitude=40) "reactivity burst pulse" annotation (Placement(transformation(extent={{-140,58},{-120,78}})));
    equation
    tankLevel = tank.level;
      connect(sensor_m_flow.m_flow, m_flow)         annotation (Line(points={{-2,31},{-2,46},{2,46}},
                                                    color={0,0,127}));
      connect(sensor_m_flow.port_b, reactor.port_a)
        annotation (Line(points={{8,20},{8,20},{50,20},{50,72}},
                                                         color={0,127,255}));
      connect(sensor_T_hotleg.T, T_hotleg)
        annotation (Line(points={{57,118},{82,118},{86,118}}, color={0,0,127}));
      connect(handle.y, valve.opening)       annotation (Line(
          points={{88.7,-22},{92,-22},{92,-64}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(pump.port_b, sensor_m_flow.port_a)
                                                annotation (Line(
          points={{-30,20},{-12,20}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(sensor_T_coldleg.T, T_coldleg) annotation (Line(
          points={{-59,-46},{-72,-46}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(sensor_T_hotleg.port, reactor.port_b) annotation (Line(
          points={{50,108},{50,92}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(reactor.port_b, hotleg.port_a) annotation (Line(
          points={{50,92},{50,104},{122,104},{122,22}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(hotleg.port_b, valve.port_a) annotation (Line(
          points={{122,2},{122,-72},{102,-72}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(tank.ports[1], pump.port_a)
        annotation (Line(points={{-112,26},{-112,20},{-50,20}},        color={0,127,255}));

      connect(coldleg.port_b, tank.ports[2])
        annotation (Line(points={{-112,-10},{-112,26},{-108,26}},
                                                              color={0,127,255}));

      // connect(radiator.port_b, tank.ports[2]);
      connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
              {-35,28.2}}, color={0,0,127}));
      connect(valve.port_b, HEX.port_a2) annotation (Line(points={{82,-72},{58,-72},
              {58,-77.9},{13.5,-77.9}}, color={0,127,255}));
      connect(HEX.port_b2, coldleg.port_a) annotation (Line(points={{-19.5,-64.1},{-112,
              -64.1},{-112,-30}},      color={0,127,255}));
      connect(sensor_T_coldleg.port, coldleg.port_a) annotation (Line(points={{-52,-56},
              {-52,-64},{-112,-64},{-112,-30}},        color={0,127,255}));
      connect(HEXmassFlowRate.ports[1], HEX.port_a1) annotation (Line(points={{-28,-86},{-22,-86},{-22,-71.3},{-19.5,-71.3}},
                                                         color={0,127,255}));
      connect(HEX.port_b1, HEXambient.ports[1]) annotation (Line(points={{13.5,-71.3},
              {13.5,-63.65},{39,-63.65},{39,-56}},        color={0,127,255}));

      connect(ReactorHeatStructure.port, reactor.heatPorts[2])
        annotation (Line(points={{4,124},{38,124},{38,79.5813},{45.6,79.5813}}, color={191,0,0}));

      connect(pointKineticMoltenSaltReactor.port_b, reactor.heatPorts[1])
        annotation (Line(points={{3,101},{26,101},{26,79.1937},{45.6,79.1937}},
                                                                              color={191,0,0}));
      connect(pointKineticMoltenSaltReactor.ReactorPowerOutput, r_power)
        annotation (Line(points={{3.5,90.5},{10,90.5},{10,65},{25,65}}, color={0,0,127}));
      connect(sensor_T_hotleg.T, add1.u2) annotation (Line(points={{57,118},{58,118},{58,176},{58,184},{0,184}}, color={0,0,127}));
      connect(Tfuel_rho_feedback.u, add1.y) annotation (Line(points={{-34,178},{-32,178},{-23,178}}, color={0,0,127}));
      connect(ramp.y, add3_1.u3) annotation (Line(points={{-83,104},{-76,104},{-76,90},{-66,90}}, color={0,0,127}));
      connect(rho0.y, add3_1.u2) annotation (Line(points={{-83,134},{-74,134},{-74,98},{-66,98}}, color={0,0,127}));
      connect(Tfuel_rho_feedback.y, add3_1.u1) annotation (Line(points={{-57,178},{-70,178},{-70,106},{-66,106}}, color={0,0,127}));
      connect(pointKineticMoltenSaltReactor.rho, add3_1.y)
        annotation (Line(points={{-15,101},{-31.5,101},{-31.5,98},{-43,98}},
                                                                           color={0,0,127}));
      connect(Tfuel_reference.y, add1.u1) annotation (Line(points={{19,162},{19,162},{14,162},{14,172},{0,172}}, color={0,0,127}));
      connect(HEXT_ramp.y, add.u2) annotation (Line(points={{-119,-132},{-82,-132}}, color={0,0,127}));
      connect(HEXmassFlowRate.T_in, add.y) annotation (Line(points={{-50,-82},{-54,-82},{-54,-126},{-59,-126}}, color={0,0,127}));
      connect(subcool.y, add.u1) annotation (Line(points={{-119,-92},{-90,-92},{-90,-120},{-82,-120}}, color={0,0,127}));
      connect(pulse.y, pointKineticMoltenSaltReactor.rho_control)
        annotation (Line(points={{-119,68},{-102,68},{-102,82},{-30,82},{-30,89.5},{-15,89.5}}, color={0,0,127}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,200}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,200}},
            initialScale=0.1)));
    end Bootstrap013;

    model Bootstrap015
      "unit initialisation testing for simplified cooling circuit with heat exchanger and lead coolant"
        // (c) 2016 Dr.Sdl
        // In order to prepare the dual fluid molten salt reactor's
        // secondary circuit a test is run in a simplified
        // circuit with lead fluid
        //

        replaceable package Medium =
           Nuclear.CoolantLead (final
              singleState=true, final enthalpyOfT=true);

       inner Modelica.Fluid.System system(
          m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        T_ambient=973.15,
        m_flow_start=0.5,
        T_start=973.15)     annotation (Placement(transformation(extent={{-140,120},
                {-120,140}},rotation=0)));

      Modelica.Fluid.Vessels.OpenTank tank(
        redeclare package Medium = Medium,
        nPorts=2,
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        ports(each p(start=110000)),
        height=4,
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
            Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        crossArea=3.00,
        T_start=Modelica.SIunits.Conversions.from_degC(700),
        level_start=0.1,
        m_flow_nominal=0.5)
                  annotation (Placement(transformation(extent={{-120,26},{-100,46}},
              rotation=0)));
      Modelica.Fluid.Machines.ControlledPump pump(
        redeclare package Medium = Medium,
        N_nominal=1500,
        use_T_start=true,
        control_m_flow=false,
        allowFlowReversal=false,
        use_p_set=true,
        m_flow_start=system.m_flow_start,
        m_flow_nominal=system.m_flow_start,
        T_start=Modelica.SIunits.Conversions.from_degC(700),
        p_a_start=110000,
        p_a_nominal=110000,
        p_b_nominal=130000)
        annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
      Modelica.Fluid.Valves.ValveIncompressible valve(
        redeclare package Medium = Medium,
        CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
        show_T=true,
        allowFlowReversal=false,
        m_flow_nominal=system.m_flow_start,
        dp_start=10000,
        dp_nominal=10000)
        annotation (Placement(transformation(extent={{118,-92},{98,-72}},
                                                                       rotation=0)));

      Modelica.Fluid.Examples.HeatExchanger.BaseClasses.BasicHX HEX(
        c_wall=500,
        s_wall=0.005,
        rho_wall=900,
        redeclare package Medium_1 =
            Medium,
        redeclare package Medium_2 =
            Medium,
        crossArea_1=3.14e-2,
        perimeter_1=0.628,
        T_start_1=Modelica.SIunits.Conversions.from_degC(700),
        T_start_2=Modelica.SIunits.Conversions.from_degC(700),
        k_wall=100,
        crossArea_2=3.14e-2,
        perimeter_2=0.628,
        nNodes=2,
        use_T_start=false,
        m_flow_start_2=system.m_flow_start,
        area_h_1=0.628*1.001,
        area_h_2=0.628*1.001,
        m_flow_start_1=1,
        length=1.001,
        Twall_start=Modelica.SIunits.Conversions.from_degC(700),
        allowFlowReversal=false,
        modelStructure_1=Modelica.Fluid.Types.ModelStructure.av_vb,
        modelStructure_2=Modelica.Fluid.Types.ModelStructure.av_vb,
        dT=1,
        p_a_start1=100000,
        p_b_start1=100000,
        p_a_start2=130000,
        p_b_start2=110000)   annotation (Placement(transformation(extent={{-18,-88},
                {12,-58}}, rotation=0)));
                /*
    modelStructure_1=Modelica.Fluid.Types.ModelStructure.av_b,
    modelStructure_2=Modelica.Fluid.Types.ModelStructure.a_vb,
    redeclare model HeatTransfer_1 =
        Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.LocalPipeFlowHeatTransfer
        (alpha0=2000),
    redeclare model HeatTransfer_2 =
        Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
        (alpha0=2000),
        */

      Modelica.Fluid.Sources.Boundary_pT HEXambient(
        nPorts=1,
        redeclare package Medium = Medium,
        T=Modelica.SIunits.Conversions.from_degC(600),
        p=100000) annotation (Placement(transformation(
            extent={{7,-7},{-7,7}},
            rotation=0,
            origin={56,-64})));

      Modelica.Fluid.Sources.MassFlowSource_T HEXmassFlowRate(
        nPorts=1,
        redeclare package Medium = Medium,
        use_T_in=false,
        use_X_in=false,
        use_m_flow_in=false,
        m_flow=1,
        T=Modelica.SIunits.Conversions.from_degC(600))
                  annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=0,
            origin={-60,-90})));

    public
      Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow(redeclare package
          Medium =                                                                 Medium)
        annotation (Placement(transformation(extent={{-12,10},{8,30}}, rotation=0)));
      Modelica.Blocks.Sources.Step handle(
        height=0.9,
        startTime=20,
        offset=0.1)   annotation (Placement(transformation(extent={{74,-29},{88,-15}},
                      rotation=0)));

    protected
      Modelica.Blocks.Interfaces.RealOutput m_flow
        annotation (Placement(transformation(extent={{-4,40},{8,52}},   rotation=
                0)));

    public
      Modelica.Blocks.Sources.Ramp PumpRamp(
        duration=100,
        offset=130000,
        startTime=9,
        height=30000)
                     annotation (Placement(transformation(extent={{-80,54},{-60,74}})));
    equation

      connect(sensor_m_flow.m_flow, m_flow)         annotation (Line(points={{-2,31},{-2,46},{2,46}},
                                                    color={0,0,127}));
      connect(handle.y, valve.opening)       annotation (Line(
          points={{88.7,-22},{108,-22},{108,-74}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(pump.port_b, sensor_m_flow.port_a)
                                                annotation (Line(
          points={{-30,20},{-12,20}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(tank.ports[1], pump.port_a)
        annotation (Line(points={{-112,26},{-112,20},{-50,20}},        color={0,127,255}));

      // connect(radiator.port_b, tank.ports[2]);
      connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
              {-35,28.2}}, color={0,0,127}));
      connect(valve.port_b, HEX.port_a2) annotation (Line(points={{98,-82},{58,
              -82},{58,-79.9},{13.5,-79.9}},
                                        color={0,127,255}));
      connect(HEXmassFlowRate.ports[1], HEX.port_a1) annotation (Line(points={{-50,-90},
              {-22,-90},{-22,-73.3},{-19.5,-73.3}},      color={0,127,255}));
      connect(HEX.port_b1, HEXambient.ports[1]) annotation (Line(points={{13.5,
              -73.3},{13.5,-63.65},{49,-63.65},{49,-64}}, color={0,127,255}));

      connect(HEX.port_b2, tank.ports[2]) annotation (Line(points={{-19.5,-66.1},
              {-108,-66.1},{-108,26}},
                                 color={0,127,255}));
      connect(sensor_m_flow.port_b, valve.port_a) annotation (Line(points={{8,20},{
              74,20},{140,20},{140,-82},{118,-82}},
                                                 color={0,127,255}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)));
    end Bootstrap015;

    model Bootstrap014
      "unit initialisation testing simplified reactor closed circuit for lead coolant"
        // (c) 2016 Dr.Sdl
        // Simplified layout of dual fluid molten salt reactor secondary circuit without reactor but
        // constant heat source and without heat exchanger.
        //
        replaceable package Medium =
           Nuclear.CoolantLead (final
              singleState=true, final enthalpyOfT=true);

       inner Modelica.Fluid.System system(
          m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        m_flow_start=0.5,
        T_ambient=923.15,
        p_start=110000)     annotation (Placement(transformation(extent={{-140,122},
                {-120,142}},rotation=0)));

      Modelica.Fluid.Vessels.OpenTank tank(
           height=4,
        crossArea=3,
        redeclare package Medium = Medium,
        use_portsData=true,
        ports(each p(start=110000)),
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
                  Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        m_flow_nominal=0.5,
        level_start=0.1,
        nPorts=2,
        T_start=923.15)
                  annotation (Placement(transformation(extent={{-134,26},{-114,46}},
              rotation=0)));

        /* */

      Modelica.Fluid.Machines.ControlledPump pump(
        redeclare package Medium = Medium,
        control_m_flow=false,
        allowFlowReversal=false,
        use_p_set=true,
        T_start=Modelica.SIunits.Conversions.from_degC(650),
        m_flow_nominal=0.5,
        p_a_start=110000,
        p_a_nominal=110000,
        p_b_nominal=130000)
        annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
        //use_T_start=true,
        //m_flow_start=system.m_flow_start,

      Modelica.Fluid.Valves.ValveIncompressible valve(
        redeclare package Medium = Medium,
        CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
        show_T=true,
        allowFlowReversal=false,
        m_flow_nominal=system.m_flow_start,
        dp_start=20000,
        dp_nominal=10000)
        annotation (Placement(transformation(extent={{20,-88},{0,-68}},rotation=0)));

      Modelica.Fluid.Pipes.DynamicPipe reactor(
        redeclare package Medium = Medium,
        use_HeatTransfer=true,
        nNodes=16,
        diameter=0.2,
        length=3.9,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
         modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
        p_a_start=130000,
        p_b_start=110000) annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                                                                                         rotation=90,
            origin={54,82})));

      Modelica.Thermal.HeatTransfer.Sources.FixedHeatFlow burner(Q_flow=0.5, T_ref=923.15)
        annotation (Placement(transformation(extent={{-34,72},{-14,92}},
                                                                       rotation=0)));
        //alpha=-0.5,
      Modelica.Blocks.Sources.Step handle(
        startTime=20,
        height=0.1,
        offset=0.9)   annotation (Placement(transformation(extent={{-30,-29},{-16,-15}},
                      rotation=0)));

    public
      Modelica.Blocks.Sources.Ramp PumpRamp(
        height=30000,
        duration=100,
        startTime=9,
        offset=130000)
                     annotation (Placement(transformation(extent={{-80,54},{-60,74}})));

      Modelica.Fluid.Sensors.MassFlowRate massFlowRate
        annotation (Placement(transformation(extent={{12,10},{32,30}})));
    protected
      Modelica.Blocks.Interfaces.RealOutput m_flow
        annotation (Placement(transformation(extent={{28,40},{48,60}}),
            iconTransformation(extent={{28,40},{48,60}})));
    equation
      connect(handle.y, valve.opening)       annotation (Line(
          points={{-15.3,-22},{10,-22},{10,-70}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(burner.port, reactor.heatPorts[1]) annotation (Line(
          points={{-14,82},{49.6,82},{49.6,79.1937}},
          color={191,0,0},
          smooth=Smooth.None));
      connect(tank.ports[1], pump.port_a)
        annotation (Line(points={{-126,26},{-126,20},{-50,20}},        color={0,127,255}));

      // connect(radiator.port_b, tank.ports[2]);
      connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
              {-35,28.2}}, color={0,0,127}));

      connect(reactor.port_b, valve.port_a) annotation (Line(points={{54,92},{74,92},
              {74,-78},{20,-78}}, color={0,127,255}));
      connect(valve.port_b, tank.ports[2]) annotation (Line(points={{0,-78},{-66,-78},
              {-122,-78},{-122,26}}, color={0,127,255}));
      connect(pump.port_b, massFlowRate.port_a)
        annotation (Line(points={{-30,20},{12,20},{12,20}}, color={0,127,255}));
      connect(massFlowRate.port_b, reactor.port_a)
        annotation (Line(points={{32,20},{54,20},{54,72}}, color={0,127,255}));
      connect(massFlowRate.m_flow, m_flow)
        annotation (Line(points={{22,31},{22,50},{38,50}}, color={0,0,127}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{150,150}},
            initialScale=0.1)));
    end Bootstrap014;

    model Bootstrap016
      "unit initialisation testing simplified reactor open circuit for lead coolant"
        // (c) 2016 Dr.Sdl
        // Simplified layout of dual fluid molten salt reactor secondary circuit without reactor but
        // constant heat source and without heat exchanger.
        //
        replaceable package Medium =
           Nuclear.CoolantLead (final
              singleState=true, final enthalpyOfT=true);

       inner Modelica.Fluid.System system(
          m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        m_flow_start=0.5,
        T_ambient=923.15,
        p_start=100000)     annotation (Placement(transformation(extent={{-140,122},
                {-120,142}},rotation=0)));

      Modelica.Fluid.Pipes.DynamicPipe pipe(
        redeclare package Medium = Medium,
        length=3.9,
        diameter=0.2,
        nNodes=16,
        use_HeatTransfer=true,
        p_a_start=130000,
        p_b_start=130000)
        annotation (Placement(transformation(extent={{-20,10},{0,30}})));

      Modelica.Fluid.Vessels.OpenTank tank(
        height=4,
        crossArea=3,
        redeclare package Medium = Medium,
        use_portsData=true,
        ports(each p(start=110000)),
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        nPorts=1,
        m_flow_nominal=0.5,
        level_start=0.1,
        T_start=923.15)
        annotation (Placement(transformation(extent={{66,40},{106,80}})));

        /*
  Modelica.Fluid.Sources.Boundary_pT boundary1(nPorts=1,
    redeclare package Medium = Medium,
    T=923.15)
    annotation (Placement(transformation(extent={{-28,-54},{-8,-34}})));
    */
      Modelica.Fluid.Valves.ValveLinear valveLinear(
        redeclare package Medium = Medium,
        allowFlowReversal=false,
        m_flow_nominal=0.5,
        dp_start=20000,
        dp_nominal=10000)
        annotation (Placement(transformation(extent={{30,10},{50,30}})));
      Modelica.Blocks.Sources.Constant const(k=0.9)
        annotation (Placement(transformation(extent={{12,40},{32,60}})));
      Modelica.Thermal.HeatTransfer.Sources.FixedHeatFlow fixedHeatFlow(Q_flow=0.5,
          T_ref=923.15)
        annotation (Placement(transformation(extent={{-48,52},{-28,72}})));
      Modelica.Fluid.Machines.ControlledPump pump(
        m_flow_nominal=0.5,
        control_m_flow=false,
        use_p_set=true,
        allowFlowReversal=false,
        redeclare package Medium = Medium,
        p_a_start=110000,
        T_start=923.15,
        p_a_nominal=110000,
        p_b_nominal=130000)
        annotation (Placement(transformation(extent={{-68,10},{-48,30}})));
      Modelica.Blocks.Sources.Constant const1(k=130000)
        annotation (Placement(transformation(extent={{-92,48},{-72,68}})));
      Modelica.Fluid.Vessels.OpenTank tank1(
        height=4,
        crossArea=3,
        redeclare package Medium = Medium,
        use_portsData=true,
        ports(each p(start=110000)),
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        m_flow_nominal=0.5,
        level_start=0.1,
        nPorts=1,
        T_start=923.15)
        annotation (Placement(transformation(extent={{-156,34},{-116,74}})));
    equation

      connect(valveLinear.port_b, tank.ports[1])
        annotation (Line(points={{50,20},{86,20},{86,40}}, color={0,127,255}));
      connect(const.y, valveLinear.opening)
        annotation (Line(points={{33,50},{40,50},{40,28}},color={0,0,127}));
      connect(pipe.port_b, valveLinear.port_a)
        annotation (Line(points={{0,20},{30,20}}, color={0,127,255}));
      connect(fixedHeatFlow.port, pipe.heatPorts[1]) annotation (Line(points={{-28,62},
              {-28,60},{-12.8063,60},{-12.8063,24.4}},          color={191,0,0}));
      connect(pump.port_b, pipe.port_a)
        annotation (Line(points={{-48,20},{-48,20},{-20,20}}, color={0,127,255}));
      connect(const1.y, pump.p_set) annotation (Line(points={{-71,58},{-71,58},{-66,
              58},{-64,58},{-64,44},{-53,44},{-53,28.2}}, color={0,0,127}));
      connect(pump.port_a, tank1.ports[1]) annotation (Line(points={{-68,20},{-68,20},
              {-136,20},{-136,34}}, color={0,127,255}));
    end Bootstrap016;

    model Bootstrap017
      "unit initialisation testing for two simplified, independenz cooling circuits"
        // (c) 2016 Dr.Sdl
        // In order to prepare the dual fluid molten salt reactor's
        // first amd secondary circuit a test is run for two independent
        // circuits

        replaceable package Medium =
           Nuclear.CoolantLead (final
              singleState=true, final enthalpyOfT=true);

       replaceable package ColdMedium =
           Nuclear.CoolantLead (final
              singleState=true, final enthalpyOfT=true);

       inner Modelica.Fluid.System system(
          m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
        m_flow_start=0.5,
        T_ambient=773.15,
        T_start=773.15)     annotation (Placement(transformation(extent={{-140,120},
                {-120,140}},rotation=0)));

      Modelica.Fluid.Vessels.OpenTank tank(
        redeclare package Medium = Medium,
        nPorts=2,
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        ports(each p(start=110000)),
        height=4,
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
            Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        crossArea=3.00,
        T_start=Modelica.SIunits.Conversions.from_degC(700),
        level_start=0.1,
        m_flow_nominal=0.5)
                  annotation (Placement(transformation(extent={{-120,26},{-100,46}},
              rotation=0)));
      Modelica.Fluid.Machines.ControlledPump pump(
        redeclare package Medium = Medium,
        N_nominal=1500,
        use_T_start=true,
        control_m_flow=false,
        allowFlowReversal=false,
        use_p_set=true,
        m_flow_start=system.m_flow_start,
        m_flow_nominal=system.m_flow_start,
        T_start=Modelica.SIunits.Conversions.from_degC(700),
        p_a_start=110000,
        p_a_nominal=110000,
        p_b_nominal=130000)
        annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
      Modelica.Fluid.Valves.ValveIncompressible valve(
        redeclare package Medium = Medium,
        CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
        show_T=true,
        allowFlowReversal=false,
        m_flow_nominal=system.m_flow_start,
        dp_start=10000,
        dp_nominal=10000)
        annotation (Placement(transformation(extent={{118,-92},{98,-72}},
                                                                       rotation=0)));

      Modelica.Fluid.Examples.HeatExchanger.BaseClasses.BasicHX HEX(
        c_wall=500,
        s_wall=0.005,
        rho_wall=900,
        redeclare package Medium_1 =
            Medium,
        redeclare package Medium_2 =
            Medium,
        crossArea_1=3.14e-2,
        perimeter_1=0.628,
        T_start_1=Modelica.SIunits.Conversions.from_degC(700),
        T_start_2=Modelica.SIunits.Conversions.from_degC(700),
        k_wall=100,
        crossArea_2=3.14e-2,
        perimeter_2=0.628,
        nNodes=2,
        use_T_start=false,
        m_flow_start_2=system.m_flow_start,
        area_h_1=0.628*1.001,
        area_h_2=0.628*1.001,
        m_flow_start_1=1,
        length=1.001,
        Twall_start=Modelica.SIunits.Conversions.from_degC(700),
        allowFlowReversal=false,
        modelStructure_1=Modelica.Fluid.Types.ModelStructure.av_vb,
        modelStructure_2=Modelica.Fluid.Types.ModelStructure.av_vb,
        dT=1,
        p_a_start1=100000,
        p_b_start1=100000,
        p_a_start2=130000,
        p_b_start2=110000)   annotation (Placement(transformation(extent={{-18,-88},
                {12,-58}}, rotation=0)));

      Modelica.Fluid.Sources.Boundary_pT HEXambient(
        nPorts=1,
        redeclare package Medium = Medium,
        T=Modelica.SIunits.Conversions.from_degC(600),
        p=100000) annotation (Placement(transformation(
            extent={{7,-7},{-7,7}},
            rotation=0,
            origin={56,-64})));

      Modelica.Fluid.Sources.MassFlowSource_T HEXmassFlowRate(
        nPorts=1,
        redeclare package Medium = Medium,
        use_T_in=false,
        use_X_in=false,
        use_m_flow_in=false,
        m_flow=1,
        T=Modelica.SIunits.Conversions.from_degC(600))
                  annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=0,
            origin={-60,-90})));

    public
      Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow(redeclare package
          Medium =                                                                 Medium)
        annotation (Placement(transformation(extent={{-12,10},{8,30}}, rotation=0)));
      Modelica.Blocks.Sources.Step handle(
        height=0.9,
        startTime=20,
        offset=0.1)   annotation (Placement(transformation(extent={{74,-29},{88,-15}},
                      rotation=0)));

    protected
      Modelica.Blocks.Interfaces.RealOutput m_flow
        annotation (Placement(transformation(extent={{-4,40},{8,52}},   rotation=
                0)));

    public
      Modelica.Blocks.Sources.Ramp PumpRamp(
        duration=100,
        offset=130000,
        startTime=9,
        height=30000)
                     annotation (Placement(transformation(extent={{-80,54},{-60,74}})));

      Modelica.Fluid.Vessels.OpenTank tank2nd(
        nPorts=2,
        massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
        use_HeatTransfer=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
        ports(each p(start=110000)),
        height=4,
        portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
            Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
        crossArea=3.00,
        level_start=0.1,
        m_flow_nominal=0.5,
        redeclare package Medium = ColdMedium,
        T_start=Modelica.SIunits.Conversions.from_degC(500),
        T_ambient=773.15)
         annotation (Placement(transformation(extent={{266,190},{286,210}},
                           rotation=0)));

      Modelica.Fluid.Machines.ControlledPump Pump2nd(
        N_nominal=1500,
        use_T_start=true,
        control_m_flow=false,
        allowFlowReversal=false,
        use_p_set=true,
        m_flow_start=0.5,
        m_flow_nominal=0.5,
        redeclare package Medium = ColdMedium,
        T_start=Modelica.SIunits.Conversions.from_degC(500),
        p_a_start=110000,
        p_b_start=130000,
        p_a_nominal=110000,
        p_b_nominal=130000) annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=0,
            origin={348,164})));

      Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow2nd(redeclare package
          Medium =
            ColdMedium)
                    annotation (Placement(transformation(extent={{380,154},{400,174}},
              rotation=0)));

      Modelica.Fluid.Valves.ValveIncompressible valve2nd(
        CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
        show_T=true,
        allowFlowReversal=false,
        m_flow_nominal=0.5,
        redeclare package Medium = ColdMedium,
        dp_start=10000,
        dp_nominal=10000) annotation (Placement(transformation(extent={{400,48},{380,
                68}}, rotation=0)));

      Modelica.Fluid.Examples.HeatExchanger.BaseClasses.BasicHX HEX2nd(
        c_wall=500,
        s_wall=0.005,
        rho_wall=900,
        crossArea_1=3.14e-2,
        perimeter_1=0.628,
        T_start_1=Modelica.SIunits.Conversions.from_degC(350),
        T_start_2=Modelica.SIunits.Conversions.from_degC(500),
        k_wall=100,
        crossArea_2=3.14e-2,
        perimeter_2=0.628,
        nNodes=2,
        use_T_start=false,
        m_flow_start_2=0.5,
        m_flow_start_1=1,
        allowFlowReversal=false,
        modelStructure_1=Modelica.Fluid.Types.ModelStructure.av_vb,
        redeclare package Medium_2 = ColdMedium,
            redeclare model HeatTransfer_1 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
            (alpha0=2000),
             redeclare model HeatTransfer_2 =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
            (alpha0=2000),
        length=10.001,
        area_h_1=0.628*10.001,
        area_h_2=0.628*10.001,
        modelStructure_2=Modelica.Fluid.Types.ModelStructure.av_vb,
        Twall_start=Modelica.SIunits.Conversions.from_degC(300),
        redeclare package Medium_1 = Modelica.Media.Water.StandardWater,
        dT=-150,
        p_a_start1=100000,
        p_b_start1=100000,
        p_a_start2=112000,
        p_b_start2=112000) annotation (Placement(transformation(extent={{282,50},{312,
                80}}, rotation=0)));

      Modelica.Fluid.Sources.MassFlowSource_T HEXmassFlowRate2nd(
        nPorts=1,
        use_T_in=false,
        use_X_in=false,
        use_m_flow_in=false,
        m_flow=100,
        redeclare package Medium = Modelica.Media.Water.StandardWater,
        T=Modelica.SIunits.Conversions.from_degC(250)) annotation (Placement(
            transformation(
            extent={{-10,-10},{10,10}},
            rotation=0,
            origin={232,32})));

      Modelica.Fluid.Sources.Boundary_pT HEXambient2nd(
        nPorts=1,
        redeclare package Medium = Modelica.Media.Water.StandardWater,
        p=15800000,
        T=Modelica.SIunits.Conversions.from_degC(330))
         annotation (Placement(transformation(
            extent={{7,-7},{-7,7}},
            rotation=0,
            origin={370,88})));
      Modelica.Blocks.Sources.Step handle2nd(
        height=0.9,
        offset=0.1,
        startTime=10)
                    annotation (Placement(transformation(extent={{390,107},{404,121}},
              rotation=0)));
    public
      Modelica.Blocks.Sources.Ramp PumpRamp2nd(
        duration=100,
        offset=130000,
        startTime=9,
        height=30000)
        annotation (Placement(transformation(extent={{320,188},{340,208}})));

      Modelica.Fluid.Pipes.DynamicPipe reactor2nd(
        diameter=0.2,
        use_T_start=true,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        redeclare package Medium = ColdMedium,
        m_flow_start=0.5,
        useInnerPortProperties=false,
        length=3.9,
        allowFlowReversal=true,
        use_HeatTransfer=true,
        modelStructure=Modelica.Fluid.Types.ModelStructure.av_vb,
        p_a_start=130000,
        T_start=773.15,
        nNodes=16)        annotation (Placement(transformation(
            extent={{-10,-10},{10,10}},
            rotation=-90,
            origin={450,142})));
    /*
  Modelica.Thermal.HeatTransfer.Sources.FixedHeatFlow burner(Q_flow=0.05, T_ref=773.15)
    annotation (Placement(transformation(extent={{152,102},{172,122}},
                                                                   rotation=0)));
*/
      Modelica.Fluid.Pipes.DynamicPipe leg2nd(
        redeclare package Medium = ColdMedium,
        length=5,
        diameter=0.2,
        use_T_start=true,
        allowFlowReversal=false,
        m_flow_start=0.5,
        redeclare model FlowModel =
            Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
        redeclare model HeatTransfer =
            Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
        use_HeatTransfer=false,
        nNodes=2,
        modelStructure=Modelica.Fluid.Types.ModelStructure.av_vb,
        p_a_start=110000,
        T_start=773.15)
        annotation (Placement(transformation(extent={{236,156},{256,176}})));
      Modelica.Thermal.HeatTransfer.Sources.FixedHeatFlow fixedHeatFlow(Q_flow=1,
          T_ref=923.15)
        annotation (Placement(transformation(extent={{438,222},{458,242}})));
    equation

      connect(sensor_m_flow.m_flow, m_flow)         annotation (Line(points={{-2,31},{-2,46},{2,46}},
                                                    color={0,0,127}));
      connect(handle.y, valve.opening)       annotation (Line(
          points={{88.7,-22},{108,-22},{108,-74}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(pump.port_b, sensor_m_flow.port_a)
                                                annotation (Line(
          points={{-30,20},{-12,20}},
          color={0,127,255},
          smooth=Smooth.None));
      connect(tank.ports[1], pump.port_a)
        annotation (Line(points={{-112,26},{-112,20},{-50,20}},        color={0,127,255}));

      // connect(radiator.port_b, tank.ports[2]);
      connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
              {-35,28.2}}, color={0,0,127}));
      connect(valve.port_b, HEX.port_a2) annotation (Line(points={{98,-82},{58,
              -82},{58,-79.9},{13.5,-79.9}},
                                        color={0,127,255}));
      connect(HEXmassFlowRate.ports[1], HEX.port_a1) annotation (Line(points={{-50,-90},
              {-22,-90},{-22,-73.3},{-19.5,-73.3}},      color={0,127,255}));
      connect(HEX.port_b1, HEXambient.ports[1]) annotation (Line(points={{13.5,
              -73.3},{13.5,-63.65},{49,-63.65},{49,-64}}, color={0,127,255}));

      connect(HEX.port_b2, tank.ports[2]) annotation (Line(points={{-19.5,-66.1},
              {-108,-66.1},{-108,26}},
                                 color={0,127,255}));
      connect(sensor_m_flow.port_b, valve.port_a) annotation (Line(points={{8,20},{
              74,20},{140,20},{140,-82},{118,-82}},
                                                 color={0,127,255}));

      connect(tank2nd.ports[1],Pump2nd.port_a) annotation (Line(points={{274,190},{290,
              190},{290,164},{338,164}},     color={0,127,255}));
      connect(Pump2nd.port_b,sensor_m_flow2nd.port_a) annotation (Line(points={{358,164},
              {370,164},{380,164}},      color={0,127,255}));
      connect(valve2nd.port_b,HEX2nd.port_a2) annotation (Line(points={{380,58},{360,
              58},{360,58.1},{313.5,58.1}}, color={0,127,255}));
      connect(HEX2nd.port_b1,HEXambient2nd.ports[1]) annotation (Line(points={{313.5,
              64.7},{346,64.7},{346,88},{363,88}}, color={0,127,255}));
      connect(HEXmassFlowRate2nd.ports[1],HEX2nd.port_a1) annotation (Line(points={{242,32},
              {242,32},{278,32},{278,64.7},{280.5,64.7}},          color={0,127,255}));
      connect(handle2nd.y,valve2nd.opening) annotation (Line(points={{404.7,114},{410,
              114},{410,76},{390,76},{390,66}},color={0,0,127}));
      connect(PumpRamp2nd.y,Pump2nd.p_set) annotation (Line(points={{341,198},{353,
              198},{353,172.2}}, color={0,0,127}));

      connect(leg2nd.port_b, tank2nd.ports[2]) annotation (Line(points={{256,166},{256,
              164},{278,164},{278,190}}, color={0,127,255}));
      connect(HEX2nd.port_b2, leg2nd.port_a) annotation (Line(points={{280.5,71.9},{198,
              71.9},{198,166},{236,166}}, color={0,127,255}));
      connect(sensor_m_flow2nd.port_b, reactor2nd.port_a) annotation (Line(points={{400,
              164},{426,164},{450,164},{450,152}}, color={0,127,255}));
      connect(reactor2nd.port_b, valve2nd.port_a) annotation (Line(points={{450,132},
              {450,132},{450,54},{450,58},{400,58}}, color={0,127,255}));
      connect(fixedHeatFlow.port, reactor2nd.heatPorts[1]) annotation (Line(points={{458,232},
              {468,232},{468,230},{472,230},{472,144.806},{454.4,144.806}},
            color={191,0,0}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{480,280}},
            initialScale=0.1)),                                      Diagram(
            coordinateSystem(preserveAspectRatio=false,
            extent={{-150,-150},{480,280}},
            initialScale=0.1)));
    end Bootstrap017;
  end BOOTSTRAP;

  model DualFluidMoltenSaltReactor "dual fluid molten salt reactor circuit"
      // (c) 2016 Dr.Sdl
      // this model represents a simple dual fluid reactor and is based on the
      // "SingleFluidMoltenSaltReactor". An independent, secondary circuit with
      // lead as coolant has been added. The heat in the latter is removed by a
      // high pressure water boundary condition.
      //
      // The thermodynamic connection between fuel salt and lead is very simple:
      // The heat is conducted from the 8th node of "reactor" to the 8th node
      // of "reactor2n".
      // In a more advanced dual fluid reactor model one would connect each of
      // the 16 nodes individually and also would one split the reactor power
      // according to its axial profile. For the time being the full reactor power
      // is put into node 1 of the "reactor" pipe. For professional applications
      // one would need to develope a separate "duelfluidreactor" class element
      // consisting of a fuel-to-lead heat exchanger kind of model plus the point
      // kinetic reactor.
      //

      replaceable package Medium =
         Nuclear.FuelSaltLiFBeF2 (final
            singleState=true, final enthalpyOfT=true);

       replaceable package ColdMedium =
         Nuclear.CoolantLead (final
            singleState=true, final enthalpyOfT=true);

     inner Modelica.Fluid.System system(
        m_flow_small=1.0e-4, energyDynamics=Modelica.Fluid.Types.Dynamics.SteadyStateInitial,
      T_ambient=923.15,
      m_flow_start=50,
      T_start=923.15)     annotation (Placement(transformation(extent={{-452,
              170},{-432,190}},
                          rotation=0)));

    Modelica.Fluid.Vessels.OpenTank tank(
      redeclare package Medium = Medium,
      nPorts=2,
      massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
      use_HeatTransfer=true,
      redeclare model HeatTransfer =
          Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
      ports(each p(start=110000)),
      height=4,
      portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
          Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
      T_start=Modelica.SIunits.Conversions.from_degC(650),
      level_start=0.5,
      crossArea=3.00,
      m_flow_nominal=0.005)
                annotation (Placement(transformation(extent={{-120,26},{-100,46}},
            rotation=0)));
    Modelica.Fluid.Machines.ControlledPump pump(
      redeclare package Medium = Medium,
      N_nominal=1500,
      use_T_start=true,
      control_m_flow=false,
      allowFlowReversal=false,
      use_p_set=true,
      T_start=Modelica.SIunits.Conversions.from_degC(650),
      m_flow_start=system.m_flow_start,
      m_flow_nominal=system.m_flow_start,
      p_a_start=110000,
      p_b_start=130000,
      p_a_nominal=110000,
      p_b_nominal=130000)
      annotation (Placement(transformation(extent={{-50,10},{-30,30}}, rotation=0)));
    Modelica.Fluid.Valves.ValveIncompressible valve(
      redeclare package Medium = Medium,
      CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
      show_T=true,
      allowFlowReversal=false,
      m_flow_nominal=system.m_flow_start,
      dp_start=18000,
      dp_nominal=10000)
      annotation (Placement(transformation(extent={{102,-82},{82,-62}},
                                                                     rotation=0)));

    Modelica.Fluid.Pipes.DynamicPipe reactor(
      redeclare package Medium = Medium,
      use_T_start=true,
      redeclare model HeatTransfer =
          Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
      redeclare model FlowModel =
          Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
      use_HeatTransfer=true,
      modelStructure=Modelica.Fluid.Types.ModelStructure.a_v_b,
      nNodes=16,
      diameter=0.2,
      height_ab=0,
      length=3.9,
      T_start=Modelica.SIunits.Conversions.from_degC(650),
      p_a_start=130000) annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                                                                                       rotation=90,
          origin={50,84})));

    Modelica.Fluid.Pipes.DynamicPipe hotleg(
      redeclare package Medium = Medium,
      use_T_start=true,
      redeclare model HeatTransfer =
          Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
      redeclare model FlowModel =
          Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
      diameter=0.2,
      T_start=Modelica.SIunits.Conversions.from_degC(650),
      length=10.01,
      p_a_start=130000) annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=-90,
          origin={122,12})));

    Modelica.Fluid.Pipes.DynamicPipe hotfeed(
      redeclare package Medium = Medium,
      use_T_start=true,
      redeclare model HeatTransfer =
          Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
      redeclare model FlowModel =
          Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
      diameter=0.2,
      T_start=Modelica.SIunits.Conversions.from_degC(650),
      length=10.01,
      p_a_start=110000,
      nNodes=2,
      modelStructure=Modelica.Fluid.Types.ModelStructure.av_vb)
                        annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=90,
          origin={-112,-20})));

  public
    Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow(redeclare package Medium
        =                                                                        Medium)
      annotation (Placement(transformation(extent={{-12,10},{8,30}}, rotation=0)));
    Modelica.Fluid.Sensors.Temperature sensor_T_hotleg(redeclare package Medium
        =                                                                         Medium)
      annotation (Placement(transformation(extent={{40,108},{60,128}},
                                                                     rotation=0)));
    Modelica.Fluid.Sensors.Temperature sensor_T_coldleg(redeclare package
        Medium =                                                                   Medium)
      annotation (Placement(transformation(extent={{-42,-56},{-62,-36}}, rotation=0)));
    Modelica.Blocks.Sources.Step handle(
      height=0.9,
      offset=0.1,
      startTime=9)  annotation (Placement(transformation(extent={{74,-29},{88,-15}},
                    rotation=0)));

  protected
    Modelica.Blocks.Interfaces.RealOutput T_hotleg
      annotation (Placement(transformation(extent={{80,112},{92,124}}, rotation=0)));
    Modelica.Blocks.Interfaces.RealOutput T_coldleg
      annotation (Placement(transformation(extent={{-66,-52},{-78,-40}}, rotation=0)));
    Modelica.Blocks.Interfaces.RealOutput tankLevel
                                   annotation (Placement(transformation(extent={{-94,26},{-82,38}},
                                  rotation=0)));
    Modelica.Blocks.Interfaces.RealOutput m_flow
      annotation (Placement(transformation(extent={{-4,40},{8,52}},   rotation=
              0)));
    Modelica.Blocks.Interfaces.RealOutput r_power
    annotation (Placement(transformation(extent={{20,60},{30,70}})));

  public
    Modelica.Blocks.Sources.Ramp PumpRamp(
      duration=100,
      offset=130000,
      startTime=9,
      height=14500)
                   annotation (Placement(transformation(extent={{-80,54},{-60,74}})));

    Modelica.Thermal.HeatTransfer.Components.HeatCapacitor ReactorHeatStructure(C=100, T(
          start=923.15)) "heatCapacitor"
      annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=90,
          origin={-8,138})));

    PointKineticMoltenSaltReactor_P1 pointKineticMoltenSaltReactor annotation (Placement(transformation(extent={{-24,76},
              {-4,96}})));

    Modelica.Blocks.Sources.Ramp ramp(
      duration=100,
      startTime=1400,
      height=1)    annotation (Placement(transformation(extent={{-134,150},{
              -114,170}})));
    Modelica.Blocks.Sources.Constant rho0(k=-2)   annotation (Placement(transformation(extent={{-158,
              168},{-138,188}})));
    Modelica.Blocks.Math.Add add1
      annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=180,
          origin={-12,178})));
    Modelica.Blocks.Math.Gain Tfuel_rho_feedback(k=-9.5)
      "reactivity feedback in pcm/K"
      annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=180,
          origin={-46,178})));
    Modelica.Blocks.Math.Add3 add3_1 annotation (Placement(transformation(extent={{-64,132},
              {-44,152}})));
    Modelica.Blocks.Sources.Ramp Tfuel_reference(
      height=-11,
      offset=-1200,
      duration=100,
      startTime=6000)
                     annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=180,
          origin={30,162})));
    Modelica.Blocks.Sources.Pulse pulse(
      width=95,
      period=5,
      nperiod=1,
      amplitude=40,
      startTime=1400) "reactivity burst pulse"
                                             annotation (Placement(transformation(extent={{-140,58},{-120,78}})));

    Modelica.Fluid.Vessels.OpenTank tank2nd(
      nPorts=2,
      massDynamics=Modelica.Fluid.Types.Dynamics.FixedInitial,
      use_HeatTransfer=true,
      redeclare model HeatTransfer =
          Modelica.Fluid.Vessels.BaseClasses.HeatTransfer.IdealHeatTransfer (k=10),
      ports(each p(start=110000)),
      height=4,
      portsData={Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2),
          Modelica.Fluid.Vessels.BaseClasses.VesselPortsData(diameter=0.2)},
      crossArea=3.00,
      level_start=0.1,
      m_flow_nominal=0.5,
      redeclare package Medium = ColdMedium,
      T_start=Modelica.SIunits.Conversions.from_degC(500),
      T_ambient=773.15)
       annotation (Placement(transformation(extent={{-376,80},{-356,100}},
                         rotation=0)));

    Modelica.Fluid.Machines.ControlledPump Pump2nd(
      N_nominal=1500,
      use_T_start=true,
      control_m_flow=false,
      allowFlowReversal=false,
      use_p_set=true,
      m_flow_start=0.5,
      m_flow_nominal=0.5,
      redeclare package Medium = ColdMedium,
      T_start=Modelica.SIunits.Conversions.from_degC(500),
      p_a_start=110000,
      p_b_start=130000,
      p_a_nominal=110000,
      p_b_nominal=130000) annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=0,
          origin={-294,54})));

    Modelica.Fluid.Sensors.MassFlowRate sensor_m_flow2nd(redeclare package
        Medium =
          ColdMedium)
                  annotation (Placement(transformation(extent={{-262,44},{-242,64}},
            rotation=0)));

    Modelica.Fluid.Valves.ValveIncompressible valve2nd(
      CvData=Modelica.Fluid.Types.CvTypes.OpPoint,
      show_T=true,
      allowFlowReversal=false,
      m_flow_nominal=0.5,
      redeclare package Medium = ColdMedium,
      dp_start=10000,
      dp_nominal=10000) annotation (Placement(transformation(extent={{-242,-62},{-262,
              -42}},rotation=0)));

    Modelica.Fluid.Examples.HeatExchanger.BaseClasses.BasicHX HEX2nd(
      c_wall=500,
      s_wall=0.005,
      rho_wall=900,
      crossArea_1=3.14e-2,
      perimeter_1=0.628,
      T_start_1=Modelica.SIunits.Conversions.from_degC(350),
      T_start_2=Modelica.SIunits.Conversions.from_degC(500),
      k_wall=100,
      crossArea_2=3.14e-2,
      perimeter_2=0.628,
      nNodes=2,
      use_T_start=false,
      m_flow_start_2=0.5,
      m_flow_start_1=1,
      allowFlowReversal=false,
      modelStructure_1=Modelica.Fluid.Types.ModelStructure.av_vb,
      redeclare package Medium_2 = ColdMedium,
          redeclare model HeatTransfer_1 =
          Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
          (alpha0=2000),
           redeclare model HeatTransfer_2 =
          Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.ConstantFlowHeatTransfer
          (alpha0=2000),
      length=10.001,
      area_h_1=0.628*10.001,
      area_h_2=0.628*10.001,
      modelStructure_2=Modelica.Fluid.Types.ModelStructure.av_vb,
      Twall_start=Modelica.SIunits.Conversions.from_degC(300),
      redeclare package Medium_1 = Modelica.Media.Water.StandardWater,
      dT=-150,
      p_a_start1=100000,
      p_b_start1=100000,
      p_a_start2=112000,
      p_b_start2=112000) annotation (Placement(transformation(extent={{-360,-60},{-330,
              -30}},rotation=0)));

    Modelica.Fluid.Sources.MassFlowSource_T HEXmassFlowRate2nd(
      nPorts=2,
      use_X_in=false,
      redeclare package Medium = Modelica.Media.Water.StandardWater,
      T=Modelica.SIunits.Conversions.from_degC(300),
      m_flow=10,
      use_m_flow_in=false,
      use_T_in=true)                                 annotation (Placement(
          transformation(
          extent={{-10,-10},{10,10}},
          rotation=0,
          origin={-390,-78})));

    Modelica.Fluid.Sources.Boundary_pT HEXambient2nd(
      nPorts=1,
      redeclare package Medium = Modelica.Media.Water.StandardWater,
      p=15800000,
      T=Modelica.SIunits.Conversions.from_degC(330))
       annotation (Placement(transformation(
          extent={{7,-7},{-7,7}},
          rotation=0,
          origin={-272,-22})));

    Modelica.Blocks.Sources.Step handle2nd(
      height=0.9,
      offset=0.1,
      startTime=5)
                  annotation (Placement(transformation(extent={{-252,-3},{-238,11}},
            rotation=0)));

    Modelica.Blocks.Sources.Ramp PumpRamp2nd(
      duration=100,
      offset=130000,
      startTime=400,
      height=1)
      annotation (Placement(transformation(extent={{-322,80},{-302,100}})));

    Modelica.Fluid.Pipes.DynamicPipe reactor2nd(
      diameter=0.2,
      use_T_start=true,
      redeclare model HeatTransfer =
          Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
      redeclare model FlowModel =
          Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
      redeclare package Medium = ColdMedium,
      m_flow_start=0.5,
      useInnerPortProperties=false,
      length=3.9,
      allowFlowReversal=false,
      use_HeatTransfer=true,
      modelStructure=Modelica.Fluid.Types.ModelStructure.av_vb,
      p_a_start=130000,
      T_start=773.15,
      nNodes=16)        annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=-90,
          origin={-192,32})));

    Modelica.Fluid.Pipes.DynamicPipe leg2nd(
      redeclare package Medium = ColdMedium,
      length=5,
      diameter=0.2,
      use_T_start=true,
      allowFlowReversal=false,
      m_flow_start=0.5,
      redeclare model FlowModel =
          Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
      redeclare model HeatTransfer =
          Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
      use_HeatTransfer=false,
      nNodes=2,
      modelStructure=Modelica.Fluid.Types.ModelStructure.av_vb,
      p_a_start=110000,
      T_start=773.15)
      annotation (Placement(transformation(extent={{-406,46},{-386,66}})));

    Modelica.Thermal.HeatTransfer.Components.HeatCapacitor CoolantHeatStructure(C=100, T(
          start=773.15)) "heatCapacitor" annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=90,
          origin={-198,76})));
  protected
    Modelica.Blocks.Interfaces.RealOutput m_flow2nd annotation (Placement(
          transformation(extent={{-246,76},{-234,88}}, rotation=0)));
  public
    Modelica.Fluid.Sensors.Temperature sensor_T_2nd(redeclare package Medium =
          ColdMedium)
                  annotation (Placement(transformation(extent={{-180,-26},{-160,
              -6}},
            rotation=0)));
  protected
    Modelica.Blocks.Interfaces.RealOutput T_2nd annotation (Placement(
          transformation(extent={{-150,-36},{-138,-24}}, rotation=0)));
  public
    Modelica.Thermal.HeatTransfer.Components.ThermalConductor DFR01(G=1000000)
      annotation (Placement(transformation(extent={{-150,112},{-130,132}})));
    Modelica.Fluid.Pipes.DynamicPipe hotbend(
      redeclare package Medium = Medium,
      use_T_start=true,
      redeclare model HeatTransfer =
          Modelica.Fluid.Pipes.BaseClasses.HeatTransfer.IdealFlowHeatTransfer,
      redeclare model FlowModel =
          Modelica.Fluid.Pipes.BaseClasses.FlowModels.DetailedPipeFlow,
      diameter=0.2,
      T_start=Modelica.SIunits.Conversions.from_degC(650),
      length=10.01,
      p_a_start=130000) annotation (Placement(transformation(
          extent={{-10,-10},{10,10}},
          rotation=180,
          origin={0,-72})));
  public
    Modelica.Fluid.Sensors.Temperature sensor_Tcold_2nd(redeclare package
        Medium = ColdMedium) annotation (Placement(transformation(extent={{-426,
              -14},{-406,6}}, rotation=0)));
  protected
    Modelica.Blocks.Interfaces.RealOutput T_cold_2nd annotation (Placement(
          transformation(extent={{-386,-10},{-374,2}}, rotation=0)));
  public
    Modelica.Blocks.Sources.Ramp PumpRamp1(
      height=-100,
      duration=100,
      offset=573,
      startTime=300)
                   annotation (Placement(transformation(extent={{-444,-122},{
              -424,-102}})));
  public
    Modelica.Thermal.HeatTransfer.Components.ThermalConductor DFR02(G=1000000)
      annotation (Placement(transformation(extent={{-150,88},{-130,108}})));
  equation
  tankLevel = tank.level;
    connect(sensor_m_flow.m_flow, m_flow)         annotation (Line(points={{-2,31},{-2,46},{2,46}},
                                                  color={0,0,127}));
    connect(sensor_m_flow.port_b, reactor.port_a)
      annotation (Line(points={{8,20},{8,20},{50,20},{50,74}},
                                                       color={0,127,255}));
    connect(sensor_T_hotleg.T, T_hotleg)
      annotation (Line(points={{57,118},{82,118},{86,118}}, color={0,0,127}));
    connect(handle.y, valve.opening)       annotation (Line(
        points={{88.7,-22},{92,-22},{92,-64}},
        color={0,0,127},
        smooth=Smooth.None));
    connect(pump.port_b, sensor_m_flow.port_a)
                                              annotation (Line(
        points={{-30,20},{-12,20}},
        color={0,127,255},
        smooth=Smooth.None));
    connect(sensor_T_coldleg.T, T_coldleg) annotation (Line(
        points={{-59,-46},{-72,-46}},
        color={0,0,127},
        smooth=Smooth.None));
    connect(sensor_T_hotleg.port, reactor.port_b) annotation (Line(
        points={{50,108},{50,94}},
        color={0,127,255},
        smooth=Smooth.None));
    connect(reactor.port_b, hotleg.port_a) annotation (Line(
        points={{50,94},{50,104},{122,104},{122,22}},
        color={0,127,255},
        smooth=Smooth.None));
    connect(hotleg.port_b, valve.port_a) annotation (Line(
        points={{122,2},{122,-72},{102,-72}},
        color={0,127,255},
        smooth=Smooth.None));
    connect(tank.ports[1], pump.port_a)
      annotation (Line(points={{-112,26},{-112,20},{-50,20}},        color={0,127,255}));

    connect(hotfeed.port_b, tank.ports[2])
      annotation (Line(points={{-112,-10},{-112,26},{-108,26}},
                                                            color={0,127,255}));

    // connect(radiator.port_b, tank.ports[2]);
    connect(PumpRamp.y, pump.p_set) annotation (Line(points={{-59,64},{-36,64},{-36,58},{-36,28},{-36,28.2},
            {-35,28.2}}, color={0,0,127}));
    connect(sensor_T_coldleg.port,hotfeed. port_a) annotation (Line(points={{-52,-56},
            {-52,-64},{-112,-64},{-112,-30}},        color={0,127,255}));

    connect(ReactorHeatStructure.port, reactor.heatPorts[2])
      annotation (Line(points={{2,138},{38,138},{38,81.5813},{45.6,81.5813}}, color={191,0,0}));

    connect(pointKineticMoltenSaltReactor.port_b, reactor.heatPorts[1])
      annotation (Line(points={{-5,93},{26,93},{26,81.1937},{45.6,81.1937}},color={191,0,0}));
    connect(pointKineticMoltenSaltReactor.ReactorPowerOutput, r_power)
      annotation (Line(points={{-4.5,82.5},{10,82.5},{10,65},{25,65}},color={0,0,127}));
    connect(sensor_T_hotleg.T, add1.u2) annotation (Line(points={{57,118},{58,118},{58,176},{58,184},{0,184}}, color={0,0,127}));
    connect(Tfuel_rho_feedback.u, add1.y) annotation (Line(points={{-34,178},{-32,178},{-23,178}}, color={0,0,127}));
    connect(ramp.y, add3_1.u3) annotation (Line(points={{-113,160},{-76,160},{
            -76,134},{-66,134}},                                                                color={0,0,127}));
    connect(rho0.y, add3_1.u2) annotation (Line(points={{-137,178},{-74,178},{
            -74,142},{-66,142}},                                                                color={0,0,127}));
    connect(Tfuel_rho_feedback.y, add3_1.u1) annotation (Line(points={{-57,178},
            {-70,178},{-70,150},{-66,150}},                                                                     color={0,0,127}));
    connect(pointKineticMoltenSaltReactor.rho, add3_1.y)
      annotation (Line(points={{-23,93},{-31.5,93},{-31.5,142},{-43,142}},
                                                                         color={0,0,127}));
    connect(Tfuel_reference.y, add1.u1) annotation (Line(points={{19,162},{19,162},{14,162},{14,172},{0,172}}, color={0,0,127}));
    connect(pulse.y, pointKineticMoltenSaltReactor.rho_control)
      annotation (Line(points={{-119,68},{-102,68},{-102,82},{-30,82},{-30,81.5},
            {-23,81.5}},                                                                      color={0,0,127}));

    connect(tank2nd.ports[1],Pump2nd.port_a) annotation (Line(points={{-368,80},{-352,
            80},{-352,54},{-304,54}},      color={0,127,255}));
    connect(Pump2nd.port_b,sensor_m_flow2nd.port_a) annotation (Line(points={{-284,54},
            {-272,54},{-262,54}},      color={0,127,255}));
    connect(valve2nd.port_b,HEX2nd.port_a2) annotation (Line(points={{-262,-52},{-282,
            -52},{-282,-51.9},{-328.5,-51.9}},
                                          color={0,127,255}));
    connect(HEX2nd.port_b1,HEXambient2nd.ports[1]) annotation (Line(points={{-328.5,
            -45.3},{-296,-45.3},{-296,-22},{-279,-22}},
                                                 color={0,127,255}));
    connect(HEXmassFlowRate2nd.ports[1],HEX2nd.port_a1) annotation (Line(points={{-380,
            -77.5},{-380,-78},{-364,-78},{-364,-45.3},{-361.5,-45.3}},
                                                                 color={0,127,255}));
    connect(handle2nd.y,valve2nd.opening) annotation (Line(points={{-237.3,4},{-232,
            4},{-232,-34},{-252,-34},{-252,-44}},
                                             color={0,0,127}));
    connect(PumpRamp2nd.y,Pump2nd.p_set) annotation (Line(points={{-301,90},{
            -289,90},{-289,62.2}},
                               color={0,0,127}));
    connect(leg2nd.port_b,tank2nd. ports[2]) annotation (Line(points={{-386,56},{-386,
            54},{-364,54},{-364,80}},  color={0,127,255}));
    connect(HEX2nd.port_b2,leg2nd. port_a) annotation (Line(points={{-361.5,-38.1},
            {-444,-38.1},{-444,56},{-406,56}},
                                        color={0,127,255}));
    connect(sensor_m_flow2nd.port_b,reactor2nd. port_a) annotation (Line(points={{-242,54},
            {-216,54},{-192,54},{-192,42}},      color={0,127,255}));
    connect(reactor2nd.port_b,valve2nd. port_a) annotation (Line(points={{-192,22},
            {-192,22},{-192,-56},{-192,-52},{-242,-52}},
                                                   color={0,127,255}));

    connect(CoolantHeatStructure.port, reactor2nd.heatPorts[1]) annotation (
        Line(points={{-188,76},{-182,76},{-176,76},{-176,34.8062},{-187.6,
            34.8062}}, color={191,0,0}));
    connect(sensor_m_flow2nd.m_flow, m_flow2nd) annotation (Line(points={{-252,65},
            {-252,65},{-252,82},{-240,82}}, color={0,0,127}));
    connect(T_2nd, sensor_T_2nd.T) annotation (Line(points={{-144,-30},{-163,
            -30},{-163,-16}},
                         color={0,0,127}));
    connect(DFR01.port_b, reactor.heatPorts[16]) annotation (Line(points={{-130,
            122},{-54,122},{16,122},{16,87.0062},{45.6,87.0062}}, color={191,0,
            0}));
    connect(DFR01.port_a, reactor2nd.heatPorts[1]) annotation (Line(points={{-150,
            122},{-162,122},{-162,34.8062},{-187.6,34.8062}},      color={191,0,
            0}));
    connect(reactor2nd.port_b, sensor_T_2nd.port) annotation (Line(points={{
            -192,22},{-192,22},{-192,-26},{-170,-26}}, color={0,127,255}));
    connect(valve.port_b, hotbend.port_a) annotation (Line(points={{82,-72},{46,
            -72},{10,-72}}, color={0,127,255}));
    connect(hotbend.port_b, hotfeed.port_a) annotation (Line(points={{-10,-72},
            {-112,-72},{-112,-30}}, color={0,127,255}));
    connect(sensor_Tcold_2nd.port, leg2nd.port_a) annotation (Line(points={{
            -416,-14},{-416,-38},{-414,-38},{-444,-38.1},{-444,56},{-406,56}},
          color={0,127,255}));
    connect(sensor_Tcold_2nd.T, T_cold_2nd) annotation (Line(points={{-409,-4},
            {-380,-4},{-380,-4}}, color={0,0,127}));
    connect(PumpRamp1.y, HEXmassFlowRate2nd.T_in) annotation (Line(points={{
            -423,-112},{-418,-112},{-412,-112},{-412,-74},{-402,-74}}, color={0,
            0,127}));
    connect(DFR02.port_b, reactor.heatPorts[6]) annotation (Line(points={{-130,98},
            {-62,98},{14,98},{14,83.1312},{45.6,83.1312}},     color={191,0,0}));
    connect(DFR02.port_a, reactor2nd.heatPorts[6]) annotation (Line(points={{-150,98},
            {-154,98},{-154,32.8687},{-187.6,32.8687}},          color={191,0,0}));
    annotation (Icon(coordinateSystem(preserveAspectRatio=false,
          extent={{-460,-150},{180,200}},
          initialScale=0.1)),                                      Diagram(
          coordinateSystem(preserveAspectRatio=false,
          extent={{-460,-150},{180,200}},
          initialScale=0.1)));
  end DualFluidMoltenSaltReactor;

  model CoolantLead "lead coolant for dual fluid reactor"
    // (c) 2016 Dr.Sdl
    // data taken from "Handbook on Lead-bismuth Eutectic Alloy and Lead
    // Properties, Materials Compatibility, Thermal-hydraulics and Technologies"
    // 2007 Edition,
    // https://www.oecd-nea.org/science/reports/2007/nea6195-handbook.html
    //
    // Attention: for practical reasons the temperature range runs from 10 to 1500 Celsius
    // and the formulas are of course not valid for this full range and there can be
    // phase transitions in between. Therefore you should check if your fluid
    // state is in the range of validity as given by the literature.
    // You can narrow the range of validity with T_min and T_max.
    //
    // density in kg/m3
    // heat capacity in J/kg/K
    // thermal conductivity in W/m/K
    // dynamic viscosity in  kg/m/s
    // vapor pressure in Pa
    //
    extends Modelica.Media.Incompressible.TableBased(
      mediumName="Coolant Lead",
      T_min=Modelica.SIunits.Conversions.from_degC(10),
      T_max=Modelica.SIunits.Conversions.from_degC(1500),
      TinK=false,
      T0=273.15,
      tableDensity=[10,11028.8; 100,10921.3; 500,10443.6; 600,10324.1; 634.85,10282.5; 700,10204.7;
                    800,10085.2; 900,9965.8; 1000,9846.3; 1100,9726.9; 1200,9607.5; 1500,9249.1],
      tableHeatCapacity=[10,143.6; 100,148.3; 500,145.1; 600,143.5; 634.85,143.0; 700,142.1; 800,140.8;
                         900,139.7; 1000,138.4; 1100,138.2; 1200,137.7; 1500,137.4],
      tableConductivity=[10,12.3; 100,13.3; 500,17.7; 600,18.8; 634.85,19.2; 700,19.9; 800,21.0;
                         900,22.1; 1000,23.2; 1100,24.3; 1200,25.4; 1500,28.7],
      tableViscosity=[10,0.0198; 100,0.0080; 500,0.0018; 600,0.0015; 634.85,0.0015; 700,0.0014; 800,0.0012;
                      900,0.0011; 1000,0.0011; 1100,0.0010; 1200,0.0009; 1500,0.0008],
      tableVaporPressure=[10,0.01; 100,0.01; 500,0.01; 600,0.05; 634.85,0.14; 700,0.75;
                         800,6.32; 900,36.9; 1000,163.4; 1100,582.4; 1200,1746.6; 1500,22402.4]);

    annotation (Icon(coordinateSystem(preserveAspectRatio=false)), Diagram(
          coordinateSystem(preserveAspectRatio=false)));
  end CoolantLead;
  annotation (uses(Modelica(version="3.2.1")));
end Nuclear;
